#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
import CliParser
# pylint: disable-next=consider-using-from-import
import CliPlugin.TapAggIntfCli as TapAggIntfCli
# pylint: disable-next=consider-using-from-import
import CliPlugin.TapAggPmapCli as TapAggPmapCli
import CliPlugin.VlanCli as VlanCli # pylint: disable=consider-using-from-import
# pylint: disable-next=consider-using-from-import
import CliPlugin.IpAddrMatcher as IpAddrMatcher
from CliPlugin.MacAddr import macAddrMatcher
from CliPlugin.EthIntfCli import EthPhyAutoIntfType
from CliPlugin.MirroringTapAggConfigMode import matcherType, matcherTapagg, \
   nodeDzGre
from CliPlugin.LagIntfCli import LagAutoIntfType
from CliPlugin.RecircIntfCli import RecircAutoIntfType
from CliPlugin.SwitchIntfCli import SwitchAutoIntfType
from CliPlugin.InternalRecircIntfCli import InternalRecircAutoIntfType
import CliPlugin.IntfCli as IntfCli # pylint: disable=consider-using-from-import
import CliToken.Mac

nodeServicePolicy = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'service-policy',
      helpdesc='Service policy configuration' ) )
matcherDefault = CliMatcher.KeywordMatcher( 'default',
      helpdesc='Configure default tap destination' )
matcherTruncationTool = CliMatcher.KeywordMatcher( 'truncation',
      helpdesc='Configure if egress packet should be truncated' )
matcherTruncationTap = CliMatcher.KeywordMatcher( 'truncation',
      helpdesc='Configure if ingress packet should be truncated' )
matcherGroupName = CliMatcher.DynamicNameMatcher( TapAggIntfCli.getGroupList,
      'Group name' )
matcherDFNexthopGroup = CliCommand.guardedKeyword( 'nexthop-group',
      helpdesc='Set default nexthop-group',
      guard=TapAggIntfCli.guardDefaultNexthopGroupTapAgg )
matcherDzGrePolicyTag = CliMatcher.KeywordMatcher( 'policy',
      helpdesc='Policy VLAN tag' )
matcherDzGrePortTag = CliMatcher.KeywordMatcher( 'port',
      helpdesc='Port VLAN tag' )
nodeDot1Q = CliCommand.guardedKeyword( 'dot1q',
      helpdesc='Remove dot1q tag1',
      guard=TapAggIntfCli.dot1qRemoveGuard )
nodeTap = CliCommand.guardedKeyword( 'tap',
      helpdesc='Set tap mode characteristics of the interface',
      guard=TapAggIntfCli.tapaggGuard )
nodeTool = CliCommand.guardedKeyword( 'tool',
      helpdesc='Set tool mode characteristics of the interface',
      guard=TapAggIntfCli.tapaggGuard )
nodeTapagg = CliCommand.Node( matcher=matcherTapagg,
      guard=TapAggPmapCli.guardPMapTapAgg )
nodeVnBrEncap = CliCommand.guardedKeyword( 'encapsulation',
      helpdesc='Configure encapsulation parameters',
      guard=TapAggIntfCli.vnBrTagStripToolGuard )
nodeVnBrStrip = CliMatcher.KeywordMatcher( 'strip',
      helpdesc='Remove a tag in packet header' )

class AggDFNexthopGroupExprFactory( CliCommand.CliExpressionFactory ):
   def generate( self, name ):
      nexthopGroupName = name + "_nexthop-group"
      class AggDFNexthopGroupExpr( CliCommand.CliExpression ):
         # pylint: disable-next=consider-using-f-string
         expression = '%s { NHG }' % nexthopGroupName
         data = {
            nexthopGroupName : matcherDFNexthopGroup,
            'NHG' : TapAggIntfCli.matcherNexthopGroupName,
         }
         @staticmethod
         def adapter( mode, args, argsList ):
            if not args.pop( nexthopGroupName, None ):
               return
            nexthopGroups = args.pop( 'NHG', None )
            args[ name ] = nexthopGroups
      return AggDFNexthopGroupExpr

#--------------------------------------------------------------------------------
# [ no | default ] service-policy type tapagg input PMAPNAME
#--------------------------------------------------------------------------------
class ServicePolicyTypeTapaggInputPmapCmd( CliCommand.CliCommandClass ):
   syntax = 'service-policy type tapagg input PMAP_NAME'
   noOrDefaultSyntax = 'service-policy type tapagg input ...'
   data = {
      'service-policy' : nodeServicePolicy,
      'type' : matcherType,
      'tapagg' : nodeTapagg,
      'input' : 'Apply the policy map to ingress packets',
      'PMAP_NAME' : CliMatcher.DynamicNameMatcher(
         TapAggPmapCli.getPMapNameRuleTapAgg, 'TapAgg Policy Map Name' ),
   }

   handler = TapAggPmapCli.handleServicePolicy
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( ServicePolicyTypeTapaggInputPmapCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport tap default { group GROUP }
#--------------------------------------------------------------------------------
class SwitchportTapDefaultCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport tap default { group GROUP_NAME }'
   noOrDefaultSyntax = ( 'switchport tap default '
                         ' ( group | { group GROUP_NAME } )' )
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'tap' : nodeTap,
      'default' : matcherDefault,
      'group' : 'Set tap group for the interface',
      'GROUP_NAME' : matcherGroupName,
   }
   handler = TapAggIntfCli.handleTapGroups
   noOrDefaultHandler = TapAggIntfCli.handleNoTapGroups

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportTapDefaultCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport tap default { nexthop-group GROUP }
#--------------------------------------------------------------------------------
class SwitchportTapDefaultNexthopGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport tap default NEXTHOP_GROUPS'
   noOrDefaultSyntax = ( 'switchport tap default '
                         ' ( nexthop-group | NEXTHOP_GROUPS )' )
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'tap' : nodeTap,
      'default' : matcherDefault,
      'nexthop-group' : matcherDFNexthopGroup,
      'NEXTHOP_GROUPS' : AggDFNexthopGroupExprFactory(),
   }
   handler = TapAggIntfCli.handleTapNexthopGroups
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportTapDefaultNexthopGroupCmd )

#--------------------------------------------------------------------------------
# [ no | default ] mac timestamp ( before-fcs | replace-fcs | header )
#--------------------------------------------------------------------------------
class MacTimestampCmd( CliCommand.CliCommandClass ):
   syntax = 'mac timestamp ( before-fcs | replace-fcs | header )'
   noOrDefaultSyntax = 'mac timestamp ...'
   data = {
      'mac' : CliToken.Mac.macMatcherForConfigIf,
      'timestamp' : 'Configure interface timestamp properties',
      'before-fcs' : CliCommand.guardedKeyword( 'before-fcs',
         helpdesc='insert timestamp before fcs field',
         guard=TapAggIntfCli.timestampFooterSupportedGuard ),
      'replace-fcs' : CliCommand.guardedKeyword( 'replace-fcs',
         helpdesc='replace fcs field with timestamp',
         guard=TapAggIntfCli.timestampFooterSupportedGuard ),
      'header' : CliCommand.guardedKeyword( 'header',
         helpdesc='insert timestamp in Ethernet header',
         guard=TapAggIntfCli.timestampHeaderSupportedGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      timestampMode = None
      if 'before-fcs' in args:
         timestampMode = 'timestampModeBeforeFcs'
      elif 'replace-fcs' in args:
         timestampMode = 'timestampModeReplaceFcs'
      else:
         timestampMode = 'timestampModeHeader'

      if timestampMode == 'timestampModeHeader':
         mode.addWarning(
            "Time-stamping is only available on tool interfaces in "
            "Tap Aggregation mode" )
      mode.intf.setTimestampMode( timestampMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.intf.setTimestampMode( 'timestampModeDisabled' )

TapAggIntfCli.TimestampModelet.addCommandClass( MacTimestampCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport tool ( group set { GROUP } | { group GROUP } )
#--------------------------------------------------------------------------------

# The 'set' keyword is obsolete as such, but supported for backward
# compatibility reasons.
# We also support new syntax of group <name> group <name> group <name>
# and old syntax of 'space' separated list but for downgrade compatibility
# reason, we generate the old syntax in 'show running-config'

class SwitchportToolCmd( CliCommand.CliCommandClass ):
   syntax = ( 'switchport tool ( ( group set { GROUP_NAME } ) | '
              '( { group GROUP_NAME } ) )' )
   noOrDefaultSyntax = ( 'switchport tool ( ( group [ set ] ... ) | '
                         '( { group GROUP_NAME } ) )' )
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'tool' : nodeTool,
      'group' : 'Set tool group for the interface',
      'set' : 'Set tool groups for the interface',
      'GROUP_NAME' : matcherGroupName,
   }

   handler = TapAggIntfCli.handleToolGroups
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportToolCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport tap native vlan VLAN_ID
#--------------------------------------------------------------------------------
class SwitchportTapNativeVlanVlanIdCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport tap native vlan VLAN_ID'
   noOrDefaultSyntax = 'switchport tap native vlan ...'
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'tap' : nodeTap,
      'native' : 'Set native VLAN when interface is in tap mode',
      'vlan' : 'Configure VLAN parameters',
      'VLAN_ID' : VlanCli.vlanIdMatcher,
   }

   handler = TapAggIntfCli.handleTapNativeVlan
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportTapNativeVlanVlanIdCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport tap mpls pop [ all ]
#--------------------------------------------------------------------------------
class SwitchportTapMplsPopAllCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport tap mpls pop all'
   noOrDefaultSyntax = 'switchport tap mpls pop [ all ]'
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'tap' : nodeTap,
      'mpls' : CliCommand.guardedKeyword( 'mpls',
         helpdesc='Configure MPLS parameters',
         guard=TapAggIntfCli.mplsPopGuard ),
      'pop' : 'Pop MPLS labels',
      'all' : 'Pop all MPLS labels',
   }

   handler = TapAggIntfCli.handleTapMplsPop
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportTapMplsPopAllCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport tool mpls pop [ all ]
#--------------------------------------------------------------------------------
class SwitchportToolMplsPopAllCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport tool mpls pop all'
   noOrDefaultSyntax = 'switchport tool mpls pop [ all ]'
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'tool' : nodeTool,
      'mpls' : CliCommand.guardedKeyword( 'mpls',
         helpdesc='Configure MPLS parameters',
         guard=TapAggIntfCli.mplsPopToolGuard ),
      'pop' : 'Pop MPLS labels',
      'all' : 'Pop all MPLS labels',
   }

   handler = TapAggIntfCli.handleToolMplsPop
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportToolMplsPopAllCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport tool encapsulation vn-tag strip
#--------------------------------------------------------------------------------
class SwitchportToolVnTagStripCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport tool encapsulation vn-tag strip'
   noOrDefaultSyntax = 'switchport tool encapsulation vn-tag strip'
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'tool' : nodeTool,
      'encapsulation' : nodeVnBrEncap,
      'vn-tag' : 'VN-tag',
      'strip' : nodeVnBrStrip,
   }

   handler = TapAggIntfCli.handleToolVnStrip
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportToolVnTagStripCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport tool encapsulation dot1br strip
#--------------------------------------------------------------------------------
class SwitchportToolBrTagStripCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport tool encapsulation dot1br strip'
   noOrDefaultSyntax = 'switchport tool encapsulation dot1br strip'
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'tool' : nodeTool,
      'encapsulation' : nodeVnBrEncap,
      'dot1br' : '802.1 BR tag',
      'strip' : nodeVnBrStrip,
   }

   handler = TapAggIntfCli.handleToolBrStrip
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportToolBrTagStripCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport tool dot1q remove outer DOT_1Q_TAG
#--------------------------------------------------------------------------------
class SwitchportToolDot1QRemoveOuterCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport tool dot1q remove outer DOT_1Q_TAG'
   noOrDefaultSyntax = 'switchport tool dot1q remove outer ...'
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'tool' : nodeTool,
      'dot1q' : nodeDot1Q,
      'remove' : 'Remove tag',
      'outer' : 'Remove outer tag',
      'DOT_1Q_TAG' : TapAggIntfCli.matcherDot1QTag,
   }

   handler = TapAggIntfCli.handleDot1qRemoveVlan
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportToolDot1QRemoveOuterCmd )

#--------------------------------------------------------------------------------
# switchport tool identity ( ( dot1q [ source dzgre ( policy | port ) ] ) |
#                            ( qinq [ source dzgre ( ( policy inner port ) |
#                            ( port inner policy ) ) ] ) )
# [ no | default ] switchport tool identity [ [ dot1q | qinq ]
#                                             [ source dzgre ... ] ]
#--------------------------------------------------------------------------------
class SwitchportToolIdentityCmd( CliCommand.CliCommandClass ):
   syntax = ( 'switchport tool identity '
              '( ( dot1q [ source dzgre ( policy | port ) ] ) | '
              '( qinq [ source dzgre ( ( policy inner port ) | '
              '( port inner policy ) ) ] ) )' )
   noOrDefaultSyntax = ( 'switchport tool identity '
                         '[ [ dot1q | qinq ] [ source dzgre ... ] ]' )
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'tool' : nodeTool,
      'identity' : 'Configure ID tagging',
      'dot1q' : 'Single dot1q tag',
      'qinq' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'qinq',
         helpdesc='Double dot1q tags' ),
         guard=TapAggIntfCli.qinqIdentityTaggingGuard ),
      'source' : 'Configure tag source',
      'dzgre' : nodeDzGre,
      'inner' : 'Configure inner VLAN tag',
      'policy' : matcherDzGrePolicyTag,
      'port' : matcherDzGrePortTag,
   }

   handler = TapAggIntfCli.handleIdentityTagging
   noOrDefaultHandler = TapAggIntfCli.handleNoIdentityTagging

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportToolIdentityCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport tap encapsulation vxlan strip
#--------------------------------------------------------------------------------
class SwitchportTapEncapVxlanStripCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport tap encapsulation vxlan strip'
   noOrDefaultSyntax = syntax
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'tap' : nodeTap,
      'encapsulation' : CliCommand.guardedKeyword( 'encapsulation',
         helpdesc='Configure encapsulation parameters',
         guard=TapAggIntfCli.vxlanStripGuard ),
      'vxlan' : 'Configure VXLAN parameters',
      'strip' : 'Strip encapsulation header',
   }

   handler = TapAggIntfCli.handleTapVxlanStrip
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportTapEncapVxlanStripCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport tap encapsulation gre strip
#--------------------------------------------------------------------------------
class SwitchportTapEncapGreStripCmd( CliCommand.CliCommandClass ):
   syntax = ( 'switchport tap encapsulation gre '
              '[ destination DST_IP [ source SRC_IP ] ] '
              'strip | ( protocol PROTOCOL [ feature header length LENGTH ] '
              'strip [ re-encapsulation ethernet ] )' )
   noOrDefaultSyntax = ( 'switchport tap encapsulation gre '
                         '[ destination DST_IP [ source SRC_IP ] ] '
                         'strip | ( protocol strip | ( PROTOCOL '
                         '[ feature header length LENGTH ] strip '
                         '[ re-encapsulation ethernet ] ) )'
                       )
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'tap' : nodeTap,
      'encapsulation' : CliCommand.guardedKeyword( 'encapsulation',
         helpdesc='Configure encapsulation parameters',
         guard=TapAggIntfCli.greStripGuard ),
      'gre' : 'Configure GRE parameters',
      'destination' : 'Destination of tunnel packets',
      'DST_IP' : IpAddrMatcher.ipAddrMatcher,
      'source' : 'Source of tunnel packets',
      'SRC_IP' : IpAddrMatcher.ipAddrMatcher,
      'protocol' :'Protocol type in GRE header',
      'PROTOCOL' : CliMatcher.IntegerMatcher( 0, 65535,
         helpdesc='Protocol range', helpname='0x0000-0xFFFF' ),
      'feature' : 'Feature header options',
      'header' : 'Feature header options',
      'length' : 'Feature header length in bytes ',
      'LENGTH' : CliMatcher.IntegerMatcher( 1, 16,
         helpdesc='Header length in bytes' ),
      'strip' : 'Strip encapsulation header',
      're-encapsulation' : 'Extra header to prepend to the terminated packet',
      'ethernet' : 'Ethernet header',
   }

   handler = TapAggIntfCli.handleTapGreStrip
   noOrDefaultHandler = TapAggIntfCli.handleNoTapGreStrip
   adapter = TapAggIntfCli.tapTunnelIpStripAdapter

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportTapEncapGreStripCmd )

# --------------------------------------------------------------------------------
# [ no | default ] switchport tap encapsulation gtp version VERSION strip
# --------------------------------------------------------------------------------
class SwitchportTapEncapGtpStripCmd( CliCommand.CliCommandClass ):
   syntax = ( 'switchport tap encapsulation gtp version VERSION '
              '[ destination DST_IP [ source SRC_IP ] ] strip' )
   noOrDefaultSyntax = syntax
   data = {
      'switchport': VlanCli.switchportMatcher,
      'tap': nodeTap,
      'encapsulation': CliCommand.guardedKeyword( 'encapsulation',
         helpdesc='Configure encapsulation parameters',
         guard=TapAggIntfCli.gtpStripGuard ),
      'gtp': 'Configure GTP parameters',
      'version': 'The specific version of GTP',
      'VERSION': CliMatcher.IntegerMatcher( 1, 1, helpdesc='GTP version' ),
      'destination': 'Destination of tunnel packets',
      'DST_IP': IpAddrMatcher.ipAddrMatcher,
      'source': 'Source of tunnel packets',
      'SRC_IP': IpAddrMatcher.ipAddrMatcher,
      'strip': 'Strip encapsulation header'
   }

   handler = TapAggIntfCli.handleTapGtpStrip
   noOrDefaultHandler = TapAggIntfCli.handleNoTapGtpStrip
   adapter = TapAggIntfCli.tapTunnelIpStripAdapter

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportTapEncapGtpStripCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport tap default interface INTFS
#--------------------------------------------------------------------------------
intfRangeConfigMatcher = IntfCli.intfRangeWithSingleExpression(
   'INTFS', explicitIntfTypes=( EthPhyAutoIntfType,
                                LagAutoIntfType,
                                RecircAutoIntfType,
                                SwitchAutoIntfType,
                                InternalRecircAutoIntfType ) )

class SwitchportTapDefaultInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = ( 'switchport tap default interface INTFS' )
   noOrDefaultSyntax = ( 'switchport tap default interface [ INTFS ]' )
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'tap' : nodeTap,
      'default' : matcherDefault,
      'interface' : 'Set interfaces for the tap port',
      'INTFS' : intfRangeConfigMatcher,
   }

   handler = TapAggIntfCli.handleTapRawIntf
   noOrDefaultHandler = TapAggIntfCli.handleNoTapRawIntf

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportTapDefaultInterfaceCmd )

#--------------------------------------------------------------------------------
# switchport mode ( tap | tool | tap-tool )
#--------------------------------------------------------------------------------
class SwitchportModeCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport mode ( tap | tool | tap-tool )'
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'mode' : VlanCli.modeNode,
      'tap' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'tap',
         helpdesc='Set switchport mode to TAP unconditionally' ),
         guard=TapAggIntfCli.tapaggGuard ),
      'tool' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'tool',
         helpdesc='Set switchport mode to TOOL unconditionally' ),
         guard=TapAggIntfCli.tapaggGuard ),
      'tap-tool' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'tap-tool',
         helpdesc='Set switchport mode to both Tap and Tool' ),
         guard=TapAggIntfCli.tapToolPortGuard ),
   }
   handler = TapAggIntfCli.setSwitchportTapToolMode

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportModeCmd )

#--------------------------------------------------------------------------------
# switchport tap truncation
#--------------------------------------------------------------------------------
class SwitchportTapTruncationCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport tap truncation'
   data = {
      'switchport': VlanCli.switchportMatcher,
      'tap': nodeTap,
      'truncation': CliCommand.Node(
         matcher=matcherTruncationTap,
         guard=TapAggIntfCli.truncationNoSizePerIngressPortGuard ),
   }
   handler = TapAggIntfCli.handleTapTruncation
TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportTapTruncationCmd )

#--------------------------------------------------------------------------------
# switchport tap truncation SIZE
# [ no | default ] switchport tap truncation ...
#--------------------------------------------------------------------------------
class SwitchportTapTruncationSizeCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport tap truncation SIZE'
   noOrDefaultSyntax = 'switchport tap truncation ...'
   data = {
      'switchport': VlanCli.switchportMatcher,
      'tap': nodeTap,
      'truncation': CliCommand.Node(
         matcher=matcherTruncationTap,
         guard=TapAggIntfCli.ingressTruncationGuard ),
      'SIZE': CliCommand.Node(
         matcher=TapAggIntfCli.matcherTapTruncationSize,
         guard=TapAggIntfCli.truncationSizePerIngressPortGuard ),
   }
   handler = TapAggIntfCli.handleTapTruncation
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportTapTruncationSizeCmd )

#--------------------------------------------------------------------------------
# switchport tool truncation
#--------------------------------------------------------------------------------
class SwitchportToolTruncationCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport tool truncation'
   data = {
      'switchport': VlanCli.switchportMatcher,
      'tool': nodeTool,
      'truncation': CliCommand.Node(
         matcher=matcherTruncationTool,
         guard=TapAggIntfCli.truncationNoSizePerEgressPortGuard ),
   }
   handler = TapAggIntfCli.handleToolTruncation
TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportToolTruncationCmd )

#--------------------------------------------------------------------------------
# switchport tool truncation SIZE
# [ no | default ] switchport tool truncation ...
#--------------------------------------------------------------------------------
class SwitchportToolTruncationSizeCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport tool truncation SIZE'
   noOrDefaultSyntax = 'switchport tool truncation ...'
   data = {
      'switchport': VlanCli.switchportMatcher,
      'tool': nodeTool,
      'truncation': CliCommand.Node(
         matcher=matcherTruncationTool,
         guard=TapAggIntfCli.egressTruncationGuard ),
      'SIZE': CliCommand.Node(
         matcher=TapAggIntfCli.matcherToolTruncationSize,
         guard=TapAggIntfCli.truncationSizePerEgressPortGuard ),
   }
   handler = TapAggIntfCli.handleToolTruncation
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportToolTruncationSizeCmd )

#--------------------------------------------------------------------------------
# switchport tap header remove size SIZE [ preserve ethernet ]
# [ no | default ] switchport tap header remove size <trailing garbage>
#--------------------------------------------------------------------------------
class SwitchportTapHeaderRemoveCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport tap header remove size SIZE [ preserve ethernet ]'
   noOrDefaultSyntax = 'switchport tap header remove size ...'
   data = {
      'switchport': VlanCli.switchportMatcher,
      'tap': nodeTap,
      'header' : CliCommand.guardedKeyword( 'header',
         helpdesc='Configure header manipulation',
         guard=TapAggIntfCli.tapHeaderRemoveSupportedGuard ),
      'remove' : 'Remove a configurable number of bytes after the Ethernet header, '
                 'including any VLAN tags',
      'size' : 'Set the number of bytes to remove',
      'SIZE': CliMatcher.IntegerMatcher(
         2, 62, helpdesc='The size in bytes (must be even)' ),
      'preserve' : 'Preserve the original header',
      'ethernet' : 'Preserve the original Ethernet header, including VLAN tags '
                   'if any',
   }
   handler = TapAggIntfCli.handleTapHeaderRemove
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportTapHeaderRemoveCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport tap identity ( ( PORT_ID [ inner INNER_PORT_ID ] )
#                                            | PORT_EXT_ID )
#--------------------------------------------------------------------------------
class SwitchportTapIdentityPortIdCmd( CliCommand.CliCommandClass ):
   syntax = ( 'switchport tap identity ( ( PORT_ID '
              '[ inner INNER_PORT_ID ] ) | PORT_EXT_ID ) ' )
   noOrDefaultSyntax = 'switchport tap identity ...'
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'tap' : nodeTap,
      'identity' : 'Configure tap port ID tag',
      'PORT_ID' : CliMatcher.IntegerMatcher( 1, 4094,
         helpdesc='Tap port VLAN ID tag' ),
      'PORT_EXT_ID' : CliCommand.Node( matcher=CliMatcher.IntegerMatcher( 1, 65535,
         helpdesc='Tap port extended ID', priority=CliParser.PRIO_LOW ),
         guard=TapAggIntfCli.dzGreEncodeSupportedGuard ),
      'inner' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'inner',
         helpdesc='Configure inner tap port ID tag' ),
         guard=TapAggIntfCli.qinqIdentityTaggingGuard ),
      'INNER_PORT_ID' : CliMatcher.IntegerMatcher( 1, 4094,
         helpdesc='Inner tap port ID tag' ),
   }

   handler = TapAggIntfCli.handleTapIdentity
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportTapIdentityPortIdCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport tap mac-address dest DMAC [ src SMAC ]
#--------------------------------------------------------------------------------
class SwitchportTapMacAddressCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport tap mac-address dest DMAC [ src SMAC ]'
   noOrDefaultSyntax = 'switchport tap mac-address ...'
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'tap' : nodeTap,
      'mac-address' : TapAggIntfCli.matcherTapMacAddress,
      'dest' : TapAggIntfCli.matcherMacAddressDest,
      'DMAC' : macAddrMatcher,
      'src' : TapAggIntfCli.matcherMacAddressSrc,
      'SMAC' : macAddrMatcher,
   }

   handler = TapAggIntfCli.handleTapMacAddress
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportTapMacAddressCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport ( tap | tool ) allowed vlan ( all | none |
#                                                 [ add | remove | except ] VLANS )
#--------------------------------------------------------------------------------
class SwitchportTapToolAllowedVlanCmd( CliCommand.CliCommandClass ):
   syntax = ( 'switchport ( tap | tool ) allowed vlan '
              '( all | none | ( [ OP ] VLANS ) ) ' )
   noOrDefaultSyntax = 'switchport ( tap | tool ) allowed vlan ...'
   data = {
      'switchport' : VlanCli.switchportMatcher,
      'tap' : nodeTap,
      'tool' : nodeTool,
      'allowed' : 'Configure allowed VLANs when interface is in tap or tool mode',
      'vlan' : 'Configure VLAN parameters',
      'all' : 'All VLANs',
      'none' : 'No VLANs',
      'OP' : VlanCli.allowedVlanOpMatcher,
      'VLANS' : VlanCli.tapToolAllowedVlanIdRangeMatcher,
   }

   handler = VlanCli.handleSwitchportModeAllowedVlan
   noOrDefaultHandler = handler

TapAggIntfCli.TapAggModelet.addCommandClass( SwitchportTapToolAllowedVlanCmd )
