#!/usr/bin/env python3
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac

from CliModel import Bool
from CliModel import Int
from CliModel import Dict
from CliModel import Enum
from CliModel import List
from CliModel import Model
from CliModel import Str
from CliModel import Submodel
from ArnetModel import IpGenericAddress
import Arnet
import IntfModels
import textwrap
import Vlan
import collections
from collections import defaultdict
from TableOutput import TableFormatter, Headings, Format, createTable
from AclLib import getAclTypeDisplayName
from CliPlugin import IntfCli
from CliPlugin.AclCliModel import IpAclList, Ipv6AclList, MacAclList
import six

class AclDetails( Model ):
   ipAclList = Submodel( valueType=IpAclList, help='IPv4 access-lists',
                         optional=True )
   ipv6AclList = Submodel( valueType=Ipv6AclList, help='IPv6 access-lists',
                           optional=True )
   macAclList = Submodel( valueType=MacAclList, help='MAC access-lists',
                          optional=True )

   def getList( self, aclType ):
      if aclType == 'ip':
         return self.ipAclList
      elif aclType == 'ipv6':
         return self.ipv6AclList
      elif aclType == 'mac':
         return self.macAclList
      else:
         return None

class MirrorSessions( Model ):
   __revision__ = 3

   class MirrorSession( Model ):
      """ Represents a MirroringSession: """
      __revision__ = 3

      class Interface( Model ):
         # Used only by the sourceInactiveInterfaces and
         # destinationInterface
         name = IntfModels.Interface( help="Interface name" )
         operState = Enum( values=( 'unknown', 'active', 'inactive' ),
                           help="The interface's current operating status." )
         srcIpGenAddr = IpGenericAddress( help="Mirroring Tunnel source address",
                                          optional=True )
         dstIpGenAddr = IpGenericAddress(
            help="Mirroring Tunnel destination address", optional=True )
         ttl = Int( help="TTL of Mirroring Tunnel", optional=True )
         dscp = Int( help="DSCP of Mirroring Tunnel", optional=True )
         vrf = Str( help="VRF of Mirroring Tunnel", optional=True )
         tunnelType = Enum( values=( 'invalid', 'gre', 'vxlan' ),
                            help="Type of mirroring tunnel", optional=True )
         # Gre tunnel parameters
         greHdrProto = Int( help="protocol in GRE header", optional=True )
         key = Int( help="Key in GRE tunnel", optional=True )
         # Vxlan tunnel parameters
         vti = IntfModels.Interface( help="VXLAN tunnel interface", optional=True )
         vlan = Int( help="VLAN of VXLAN tunnel", optional=True )
         vni = Int( help="VNI of VXLAN tunnel", optional=True )
         forwardingDrop = Bool( help="mirror drop packets", optional=True )
         operStateReason = Str( help="Reason for the current operating status. "
                                "Only set if the operState is inactive" )

      class Vlan( Model ):
         vlan = Int( help="VLAN ID" )
         operState = Enum( values=( 'unknown', 'inactive' ),
                           help="The interface's current operating status." )
         operStateReason = Str( help="Reason the state is inactive. "
                                "Only set if the operState is inactive",
                                optional=True )

      sourceRxVlans = List( valueType=int,
                           help="VLANs mirroring their ingress traffic",
                           optional=True )
      sourceInactiveVlans = List( valueType=Vlan,
                                  help="Inactive VLANs configured to "
                                  "mirror traffic in this session", optional=True )
      sourceVlans = List( valueType=int,
                          help="All VLANs configured as mirroring sources "
                          "in this session", optional=True )
      sourceRxInterfaces = List( valueType=str,
                                 help="Interfaces mirroring their ingress traffic" )
      sourceTxInterfaces = List( valueType=str,
                                 help="Interfaces mirroring their egress traffic" )
      sourceBothInterfaces = List( valueType=str,
                                   help="Interfaces mirroring both their ingress "
                                   "and egress traffic" )
      sourceInactiveInterfaces = List( valueType=Interface,
                                       help="Inactive interfaces configured "
                                       "to mirror traffic in this session" )
      sourceUnknownInterfaces = List( valueType=str,
                                      help="Interfaces in an unknown state" )
      sourceInterfaces = List( valueType=str,
                               help="All Interfaces configured as mirroring "
                               "sources in this session" )

      targetInterfaces = List( valueType=Interface,
                               help="Interfaces to which traffic is forwarded",
                               optional=True )

      class TunnelIntfs( Model ):
         class MemberIntfs( Model ):
            memberIntfs = List( valueType=IntfModels.Interface,
                                help="Port-Channel member interfaces" )
         tunnelIntfs = Dict( keyType=IntfModels.Interface,
                             valueType=MemberIntfs,
                             help="tunnel next hop interface" )
      greTunnels = Dict( keyType=IntfModels.Interface,
                         valueType=TunnelIntfs,
                         help="Next hop interfaces selected for GRE "
                         "tunnel destination", optional=True )
      vxlanTunnels = Dict( keyType=IntfModels.Interface,
                         valueType=TunnelIntfs,
                         help="Next hop interfaces selected for VXLAN "
                         "tunnel destination", optional=True )
      aclType = Enum( values=Tac.Type( 'Acl::AclType' ).attributes,
                      help="ACL type", optional=True )

      class Acl( Model ):
         aclName = Str( help="ACL name" )
         aclPriority = Int( help="ACL priority", optional=True )
         aclType = Enum( values=Tac.Type( 'Acl::AclType' ).attributes,
                         help="ACL type", optional=True )
         aclDetails = Bool( help='Display the unsupported TCAM qualifiers' )
         aclUnsupportedQualifiers = List( valueType=str,
                                          help='Unsupported qualifiers',
                                          optional=True )

      class AclsWithMirrorAction( Model ):
         class Ipv4AclWithMirrorAction( Model ):
            name = Str( help="ACL name" )
            direction = Enum( values=( 'in', 'out' ),
                              help="Direction of access-list with mirror action" )
            acl = Submodel( valueType=IpAclList, help='IPv4 access lists' )
         class Ipv6AclWithMirrorAction( Model ):
            name = Str( help="ACL name" )
            direction = Enum( values=( 'in', 'out' ),
                              help="Direction of access-list with mirror action" )
            acl = Submodel( valueType=Ipv6AclList, help='IPv6 access lists' )
         class MacAclWithMirrorAction( Model ):
            name = Str( help="ACL name" )
            direction = Enum( values=( 'in', 'out' ),
                              help="Direction of access-list with mirror action" )
            acl = Submodel( valueType=MacAclList, help='MAC access lists' )
         ipAcls = List( valueType=Ipv4AclWithMirrorAction,
                        help='IPv4 access lists with mirror action',
                        optional=True )
         ipv6Acls = List( valueType=Ipv6AclWithMirrorAction,
                          help='IPv6 access lists with mirror action',
                          optional=True )
         macAcls = List( valueType=MacAclWithMirrorAction,
                         help='MAC access lists with mirror action',
                         optional=True )

      sourceAcls = Dict( keyType=str, valueType=Acl,
                      help="The ACL applied to the source interface",
                      optional=True )
      sourceAclsDetails = Dict( keyType=str, valueType=AclDetails,
                                help="Mapping from source ACL name to its details",
                                optional=True )
      sessionAcl = Str( help="ACL applied to the session." )
      sessionAclDetails = Submodel( valueType=AclDetails,
                                    help="Session access-list details",
                                    optional=True )
      aclsWithMirrorAction = Submodel( valueType=AclsWithMirrorAction,
                                       help="Access lists with mirror action"
                                            " rules",
                                       optional=True )
      truncate = Bool( help="Packet truncation enabled in the session." )
      truncationSize = Int( help="Packet truncation size in the session.",
                            optional=True )
      egressAclMirroredTruncation = Int( help="Set to 128 if metadata is enabled "
                                              "for the TX mirroring to GRE",
                                         optional=True )
      truncateStatus = Str( help="Status and reason for truncation in the session." )
      headerRemovalSize = Int( help="Header removal size in the session." )
      mirrorDeviceName = Str( help="The mirror-device name to use for "
                           "receiving packets mirrored to cpu.", optional=True )
      hwProgrammingStatusSupported = Bool( help="Whether platform code supports"
         " outputting hardware programming status. Can be used by CLI to indicate"
         " whether monitor sessions are programmed in hardware" )
      programmedInHw = Bool( help="Whether session is programmed in HW" )
      rxSampleRate = Int( help="Mirroring sampling rate.", optional=True )
      grePayloadType = Enum( values=( "payloadTypeNone",
                                      "payloadTypeFullPacket",
                                      "payloadTypeInnerPacket" ),
                              help="Payload type of mirrored packets in the "
                              "session.", optional=True )
      greMetadata = List( valueType=str,
                          help="Metadata elements in the GRE header." )

      greTimestampingSupported = Bool( help="Whether the hardware platform supports "
                                       "GRE timestamping. Can be used by CLI to "
                                       "determine whether or not to report "
                                       "timestamping enabled state." )
      greTimestampingEnabled = Bool( help="Whether timestamping is enabled for GRE "
                                     "mirror sessions", optional=True )
      txMirrorAclSupported = Bool( help="Whether tx Mirroring ACLs are supported "
                                        "for mirror sessions", optional=True )

      mirrorRateLimit = Int( help="Mirroring rate limit in bps.", optional=True )
      mirrorRateLimitChip = Enum(
            values=( "per-ingress-chip", "per-egress-chip" ),
            help="Rate limit on multi-chip.",
            optional=True )

      tcamProfileFeature = Enum(
         values=( "mirror-feature1", "mirror-feature2",
                  "mirror-feature3", "mirror-feature4" ),
         help="TCAM profile feature for mirroring ACL", optional=True )
      
      congestionMonitor = Bool( help="Mirroring on congestion" )
      cpuQueue = Str( help="Copp queue set for CPU port", optional=True )
      _displayInSummary = Bool( help="Show session in summary output" )

      def doNotDisplayInSummary( self ):
         self._displayInSummary = False

      def displayInSummary( self ):
         return self._displayInSummary

      def degrade( self, dictRepr, revision ):
         if revision == 1 and 'sourceAcls' in dictRepr:
            dictRepr[ 'sourceAcls' ] = { k : v[ 'aclName' ] for ( k, v )
                                       in six.iteritems( dictRepr[ 'sourceAcls' ] ) }
         # pylint: disable-next=consider-using-in
         if ( revision == 1 or revision == 2 ) and 'greTunnels' in dictRepr:
            tmp = collections.defaultdict( lambda: collections.defaultdict( list ) )
            for k, v in six.iteritems( dictRepr[ 'greTunnels' ] ) :
               for nh, intfs in six.iteritems( v[ 'tunnelIntfs' ] ) :
                  if intfs[ 'memberIntfs' ]:
                     tmp[ k ][ 'nextHopIntfs' ].extend( intfs[ 'memberIntfs' ] )
                  else:
                     tmp[ k ][ 'nextHopIntfs' ].append( nh )
            def default_to_normal( d ):
               if isinstance( d, collections.defaultdict ):
                  d = { k : default_to_normal( v ) for k, v in six.iteritems( d ) }
               return d

            dictRepr[ 'nextHopForTunnels' ] = default_to_normal( tmp )
         return dictRepr

   # In the following collection, the key (session name) can
   # correspond to a None value if that session doesn't exist. This
   # happens if a user asks for a non-existant session, i.e. "show
   # monitor session bad_session_name"
   sessions = Dict( valueType=MirrorSession,
                    help="Map from session name to the session details " )

   name = Str( help="The name of the specified session in the CLI command, "
               "if it was set", optional=True )
   _summary = Str( help="Summary of monitor session interfaces, unfiltered or"
                   "filtered by source or destination" )
   _detail = Str( help="Details of monitor sessions, show all details or only "
                  "session access-lists", optional=True )
   multiDestSessions = List( valueType=str,
                              help="List of sessions that have multiple active "
                              "destinations configured." )

   def render( self ):
      def shortenIntfName( intf ):
         if isinstance( intf, Tac.Type( 'Arnet::IntfId' ) ):
            pass
         elif isinstance( intf, str ):
            if intf.startswith( 'Vxlan' ):
               return intf 
            intf = Tac.Value( 'Arnet::IntfId', intf )
         # MirrorTunnel9998 used as dummy interface, so drop ID
         if intf.stringValue == 'MirrorTunnel9998':
            return 'MirrorTunnel'
         else:
            return intf.shortName

      def sessionHeader( sessionName ):
         sessHeader = ''
         sessHeader += '\n'
         sessHeader += 'Session %s \n' % sessionName
         sessHeader += '------------------------'
         sessHeader += '\n'
         return sessHeader

      def printIntfList( intfList, sourceAcls, sourceDirBoth=False,
                         txAclSupported=False ):
         # This function is hideous, but it existed in this form
         # originally, so that's what we'll use.
         intfLine = ''
         firstLine = True
         if intfList == []: # pylint: disable=too-many-nested-blocks
            print()
         else:
            for intf in intfList:
               intfLine += ", "
               if isinstance( intf, MirrorSessions.MirrorSession.Interface ):
                  intfLine += shortenIntfName( intf.name.stringValue ) + \
                        ' ( ' + intf.operStateReason + ' ) '
               else:
                  intfLine += shortenIntfName( intf )
                  if intf in sourceAcls:
                     aclType = getAclTypeDisplayName( sourceAcls[ intf ].aclType )
                     intfLine += '(%s ACL: %s' % (
                           aclType,
                           sourceAcls[ intf ].aclName )
                     if sourceDirBoth and not txAclSupported:
                        intfLine += ' ingress'
                     if sourceAcls[ intf ].aclPriority > 0:
                        intfLine += ' [ prio %d ]' % (
                              sourceAcls[ intf ].aclPriority )
                     if sourceAcls[ intf ].aclUnsupportedQualifiers:
                        if sourceAcls[ intf ].aclDetails:
                           qualifiers = sourceAcls[ intf ].aclUnsupportedQualifiers
                           intfLine += ' unsupported qualifiers: %s)' % \
                                       ', '.join( sorted( qualifiers ) )

                        else:
                           intfLine += ' with unsupported qualifiers)'
                     else:
                        intfLine += ')'
               if len( intfLine ) > 31 :
                  if not firstLine:
                     print( ' ' * 13, end=' ' )
                  intfLine = intfLine.lstrip( ',' )
                  print( intfLine )
                  firstLine = False
                  intfLine = ''
            if intfLine:
               if not firstLine:
                  print( ' ' * 13, end=' ' )
               intfLine = intfLine.lstrip( ',' )
               print( intfLine )

      def printVlanList( vlanList ):
         if not vlanList:
            print()
            return

         numInactiveVlans = sum( isinstance(
                   vlan, MirrorSessions.MirrorSession.Vlan ) for vlan in vlanList )
         if numInactiveVlans:
            # group by operStateReason
            assert numInactiveVlans == len( vlanList )
            for vlan in vlanList:
               print( 'VLAN %d ( %s )' % ( vlan.vlan, vlan.operStateReason ) )
         else:
            print( Vlan.vlanSetToCanonicalString( vlanList ) )

      def showSession( name, sessModel, multiDestSessions ):
         print( sessionHeader( name ) )
         if sessModel.hwProgrammingStatusSupported:
            status = 'No'
            if sessModel.programmedInHw:
               status = 'Yes'
            print( "Programmed in HW: %s" % status )
            print()
         print( 'Sources:' )
         print()
         if sessModel.sourceRxInterfaces:
            print( '%10s %2s' % ( 'Rx Only Interfaces:', ' ' ), end=' ' )
            printIntfList( sessModel.sourceRxInterfaces, sessModel.sourceAcls )
         if sessModel.sourceTxInterfaces:
            print( '%10s %2s' % ( 'Tx Only Interfaces:', ' ' ), end=' ' )
            printIntfList( sessModel.sourceTxInterfaces, sessModel.sourceAcls )
         if sessModel.sourceBothInterfaces:
            print( '%7s %5s' %  ( 'Both Interfaces:', ' ' ), end=' ' )
            printIntfList( sessModel.sourceBothInterfaces, sessModel.sourceAcls,
                  True, sessModel.txMirrorAclSupported )
         if sessModel.sourceInactiveInterfaces:
            print( '%11s %1s' % ( 'Inactive Interfaces:', ' ' ), end=' ' )
            printIntfList( sessModel.sourceInactiveInterfaces, sessModel.sourceAcls )
         if sessModel.sourceUnknownInterfaces:
            print( '%11s %1s' % ( 'Unknown:', ' ' ), end=' ' )
            printIntfList( sessModel.sourceUnknownInterfaces, sessModel.sourceAcls )
         if sessModel.sourceRxVlans:
            print( '%10s %2s' % ( 'Rx Only VLANs:', ' ' ), end=' ' )
            printVlanList( sessModel.sourceRxVlans )
         if sessModel.sourceInactiveVlans:
            print( '%10s %2s' % ( 'Inactive VLANs:', ' ' ), end=' ' )
            printVlanList( sessModel.sourceInactiveVlans )
         print()

         if not sessModel.targetInterfaces:
            print( 'No destination configured' )
         else:
            print( 'Destination Ports:' )
            print()
         tunnelTable = None
         # pylint: disable-next=too-many-nested-blocks
         for targetIntf in sessModel.targetInterfaces:
            shortName = shortenIntfName( targetIntf.name.stringValue )
            # active mirror tunnel interfaces
            if shortName.startswith( 'MirrorTunnel' ) and \
                  targetIntf.operState == "active":
               tunnelIntfsDict = None
               if not tunnelTable:
                  formatCol = Format( justify='center', padding=-3 )
                  formatCol.noTrailingSpaceIs( True )
                  if targetIntf.tunnelType == 'gre':
                     headings = [ '', 'status', 'source', 'dest',
                        'TTL', 'DSCP', 'key', 'proto', 'VRF', 'fwd-drop' ]
                     tunnelTable = createTable( headings, tableWidth=160 )
                     tunnelTable.formatColumns( *[ formatCol ] * len( headings ) ) 
                  elif targetIntf.tunnelType == 'vxlan':
                     headings = [ '', 'status', 'interface',
                        'source', 'dest', 'VLAN', 'VNI', 'TTL', 'DSCP', 'VRF' ]
                     tunnelTable = createTable( headings, tableWidth=160 )
                     tunnelTable.formatColumns( *[ formatCol ] * len( headings ) ) 
               if targetIntf.tunnelType == 'gre':
                  if targetIntf.forwardingDrop:
                     fwdDrop = "yes"
                  else:
                     fwdDrop = "no"
                  if targetIntf.key is not None:
                     keyFormat = "{0:X}"
                     key = "0x" + keyFormat.format( targetIntf.key )
                  else:
                     key = str( targetIntf.key ).lower()
                  tunnelTable.newRow( shortName, targetIntf.operState, 
                     targetIntf.srcIpGenAddr, targetIntf.dstIpGenAddr,
                     targetIntf.ttl, targetIntf.dscp, key, targetIntf.greHdrProto,
                     targetIntf.vrf, fwdDrop )
                  print( tunnelTable.output() )
                  tunnelIntfsDict = sessModel.greTunnels
               elif targetIntf.tunnelType == 'vxlan':
                  tunnelTable.newRow( shortName, targetIntf.operState,
                     targetIntf.vti.stringValue, targetIntf.srcIpGenAddr,
                     targetIntf.dstIpGenAddr, targetIntf.vlan, targetIntf.vni,
                    targetIntf.ttl, targetIntf.dscp, targetIntf.vrf )
                  print( tunnelTable.output() )
                  tunnelIntfsDict = sessModel.vxlanTunnels
               # render tunnel next hop interface
               if tunnelIntfsDict and shortName in tunnelIntfsDict:
                  nextHopIntfStr = "      next hop interfaces: "
                  count = 0
                  tunnelIntfs = tunnelIntfsDict[ shortName ].tunnelIntfs
                  sortedTunnelIntfs = sorted( tunnelIntfs.keys() )
                  numNHs = len( tunnelIntfs )
                  for tunnelIntf in sortedTunnelIntfs:
                     numNHs -= 1
                     if not tunnelIntfs[ tunnelIntf ].memberIntfs:
                        nextHopIntfStr += "%s" % shortenIntfName( tunnelIntf )
                        if numNHs != 0:
                           nextHopIntfStr += ","
                        if count % 8 == 7:
                           print( nextHopIntfStr )
                           nextHopIntfStr = "\t\t\t   "
                        else:
                           nextHopIntfStr += " "
                        count += 1
                     else:
                        nextHopIntfStr += "%s( " % shortenIntfName( tunnelIntf )
                        members = len( tunnelIntfs[ tunnelIntf ].memberIntfs )
                        for intf in tunnelIntfs[ tunnelIntf ].memberIntfs:
                           members -= 1
                           nextHopIntfStr += "%s" % shortenIntfName( intf )
                           if members == 0:
                              nextHopIntfStr += " )"
                              if numNHs != 0:
                                 nextHopIntfStr += ","
                           else:
                              nextHopIntfStr += ","
                           if count % 8 == 7:
                              print( nextHopIntfStr )
                              nextHopIntfStr = "\t\t\t   "
                           else:
                              nextHopIntfStr += " "
                           count += 1
                  if nextHopIntfStr != "":
                     print( nextHopIntfStr )
            else: # non-active or non MirrorTunnel intfs
               print( '    %s : ' % shortName, end=' ' )
               if targetIntf.operState == "inactive":
                  print( 'inactive (%s)' % targetIntf.operStateReason )
               elif targetIntf.operState == "active":
                  print( 'active', end=' ' )
                  if shortName == 'Cpu' and \
                        sessModel.mirrorDeviceName is not None:
                     mirrorCpuQueue = sessModel.cpuQueue
                     if mirrorCpuQueue is not None:
                        # display configured cpu queue of session
                        # when single destination
                        if len( sessModel.targetInterfaces ) == 1:
                           print( '(%s:%s)' %
                                 ( sessModel.mirrorDeviceName, mirrorCpuQueue ) )
                        else:
                           # multi dest session use cpu queue CoppSystemIpMcast
                           # this can be updated after fixing BUG989033
                           print( '(%s:copp-system-ipmc)' %
                                  ( sessModel.mirrorDeviceName ) )
                     else:
                        print( '(%s)' % ( sessModel.mirrorDeviceName ) )
               else:
                  print( 'unknown' )
               print()

         sessionAcl = sessModel.sessionAcl
         if sessionAcl != '':
            print()
            print( ( '%s access list: %s' % (
               getAclTypeDisplayName( sessModel.aclType ), sessionAcl ) ) )

         if sessModel.truncate:
            print()
            print( 'Truncation: %s' % sessModel.truncateStatus )
            if sessModel.truncationSize:
               msg = 'Nominal truncation size in bytes: %d' % \
                     sessModel.truncationSize
               print()
               print( msg )

         if sessModel.egressAclMirroredTruncation:
            msg = 'TX Mirrored packet truncation size in bytes: %d' % \
                  sessModel.egressAclMirroredTruncation
            print()
            print( msg )

         if sessModel.headerRemovalSize:
            msg = 'Header removal size in bytes: %d' % \
                  sessModel.headerRemovalSize
            if multiDestSessions:
               msg += '. Not Active*'
            print()
            print( msg )

         if sessModel.rxSampleRate:
            msg = 'Sampling Rate: %d' % sessModel.rxSampleRate
            if sessModel.sourceTxInterfaces or \
               sessModel.sourceBothInterfaces:
               msg += ' (Disabled because session has non-Rx sources)'
            print()
            print( msg )

         if sessModel.grePayloadType and \
            sessModel.grePayloadType != "payloadTypeNone":
            if sessModel.grePayloadType == 'payloadTypeFullPacket':
               grePayloadTypeCliName = 'full-packet'
            elif sessModel.grePayloadType == 'payloadTypeInnerPacket':
               grePayloadTypeCliName = 'inner-packet'
            print()
            print( 'GRE Payload Type: %s' % grePayloadTypeCliName )

         if sessModel.greMetadata:
            msg = 'GRE Metadata: %s' % ( " ".join( sessModel.greMetadata ) )
            print()
            print( msg )

         if sessModel.aclsWithMirrorAction:

            if sessModel.aclsWithMirrorAction.ipAcls:
               print()
               printAccessLists( self,
                                 'IPv4', sessModel.aclsWithMirrorAction.ipAcls )

            if sessModel.aclsWithMirrorAction.ipv6Acls:
               print()
               printAccessLists( self,
                                 'IPv6', sessModel.aclsWithMirrorAction.ipv6Acls )

            if sessModel.aclsWithMirrorAction.macAcls:
               print()
               printAccessLists( self,
                                 'MAC', sessModel.aclsWithMirrorAction.macAcls )

         if sessModel.greTimestampingSupported and \
            sessModel.greTimestampingEnabled is not None:
            greTimestampingModeCliName = \
               "enabled" if sessModel.greTimestampingEnabled else "disabled"
            msg = 'GRE Timestamp Mode: %s' % greTimestampingModeCliName
            print()
            print( msg )

         if sessModel.mirrorRateLimit:
            msg = 'Rate Limit %s : %d kbps' % ( sessModel.mirrorRateLimitChip,
                     sessModel.mirrorRateLimit // 1000 )
            print()
            print( msg )

         if sessModel.tcamProfileFeature:
            msg = ( 'Access-list TCAM mirror feature : %s' %
                    sessModel.tcamProfileFeature )
            print()
            print( msg )

         if sessModel.congestionMonitor:
            msg = 'Mirroring on congestion: enabled'
            print()
            print( msg )

      # Displays a summary table of the sessions. Output can be filtered to only
      # show sessions that contain certain sources or destinations. Sessions not
      # to be displayed will have displayInSummary == False
      def showSummary( self ):
         if not [ sess for sess in self.sessions
                  if self.sessions[ sess ].displayInSummary() ]:
            if self._summary == 'Source':
               print( 'No sessions with any of the given sources' )
               return
            elif self._summary == 'Destination':
               print( 'No sessions with any of the given destinations' )
               return
            else:
               print( 'No sessions created' )
               return
         hwProgrammingStatusSupported = False
         for session in six.itervalues( self.sessions ):
            if session.hwProgrammingStatusSupported:
               hwProgrammingStatusSupported = True
               break

         tf = TableFormatter()
         sessionFormat = Format( justify="left", wrap=True,
                                 maxWidth=16, minWidth=1 )
         hwStatusFormat = Format( justify="left", wrap=True,
                                  maxWidth=10, minWidth=1 )
         destinationFormat = Format( justify="left", wrap=True,
                                     maxWidth=16, minWidth=1 )
         sourceFormat = Format( justify="left", wrap=True,
                                maxWidth=24, minWidth=1 )
         formats = [ sessionFormat, hwStatusFormat, destinationFormat, sourceFormat,
                     sourceFormat ]
         heading = [ 'Session', 'HW Status', 'Destinations', 'Source Interfaces',
                     'Source VLANs' ]
         if not hwProgrammingStatusSupported:
            del formats[ 1:2 ]
            del heading[ 1:2 ]

         header = Headings( tuple( heading ) )
         header.doApplyHeaders( tf )
         tf.formatColumns( *formats )
         sharedFound = False

         for name in sorted( self.sessions ):
            if not self.sessions[ name ].displayInSummary():
               continue
            # Look through the other sessions to see if any of this session's
            # sources or destinations are shared. Mark the with a * if so
            otherSessions = [ self.sessions[ sess ] for sess in
                  self.sessions if sess != name ]
            otherTargets = []
            otherSourceIntfs = []
            otherSourceVlans = []
            sessionSourceIntfs = []
            sessionSourceVlans = []
            sessionTargets = []
            for session in otherSessions:
               otherTargets.extend( session.targetInterfaces )
               otherSourceIntfs.extend( session.sourceInterfaces )
               otherSourceVlans.extend( session.sourceVlans )

            for srcIntf in self.sessions[ name ].sourceInterfaces:
               intfName = shortenIntfName( srcIntf )
               if srcIntf in otherSourceIntfs:
                  sessionSourceIntfs.append( intfName + '*' )
                  sharedFound = True
               else:
                  sessionSourceIntfs.append( intfName )
            for srcVlan in self.sessions[ name ].sourceVlans:
               vlanName = 'VLAN %d' % srcVlan
               if srcVlan in otherSourceVlans:
                  sessionSourceVlans.append( vlanName + '*' )
                  sharedFound = True
               else:
                  sessionSourceVlans.append( vlanName )
            for targetIntf in self.sessions[ name ].targetInterfaces:
               intfName = shortenIntfName( targetIntf.name.stringValue )
               if targetIntf in otherTargets:
                  sessionTargets.append( intfName + '*' )
                  sharedFound = True
               else:
                  sessionTargets.append( intfName )

            srcIntfString = ', '.join( sessionSourceIntfs )
            srcVlanString = ', '.join( sessionSourceVlans )
            destString = ', '.join( sessionTargets )
            if hwProgrammingStatusSupported:
               status = 'Inactive'
               if self.sessions[ name ].programmedInHw:
                  status = 'Active'
               tf.newRow( name,
                          status,
                          destString,
                          srcIntfString,
                          srcVlanString )
            else:
               tf.newRow( name,
                          destString,
                          srcIntfString,
                          srcVlanString )
         print( tf.output() )
         if sharedFound:
            print( '* Shared between multiple sessions' )

      # Helper functions for printing ACLs

      # "directionIn" and "directionOut" matches the possible directions for ACLs
      # in MirroringModels.py. These are used to determine if an ACL is applied
      # ingress or egress so they can be displayed appropriately.
      directionIn = 'in'
      directionOut = 'out'

      def printAccessLists( self, aclType, sessionAcls ):
         # prints the access lists (ingress and egress) for a set of session ACLs
         ingressMsg = ', '.join( acl.name for acl in sessionAcls
                                 if acl.direction == directionIn )

         egressMsg = ', '.join( acl.name for acl in sessionAcls
                                if acl.direction == directionOut )

         if ingressMsg:
            print( 'Ingress %s access lists with mirror action: %s'
                   % ( aclType, ingressMsg ) )
         if egressMsg:
            print( 'Egress %s access lists with mirror action: %s'
                   % ( aclType, egressMsg ) )

      def printAccessListsWithRules( self, aclType, sessionAcls ):
         # prints the access lists along with its rules for a set of session ACLs
         ingressAcls = []
         egressAcls = []

         # store all ingress and egress ACLs into appropriate lists
         for acl in sessionAcls:
            if acl.direction == directionIn:
               ingressAcls.append( acl )
            elif acl.direction == directionOut:
               egressAcls.append( acl )

         if ingressAcls:
            print( 'Ingress %s access lists with mirror action:' % aclType )
            for ingressAcl in ingressAcls:
               print( ingressAcl.name )
               for aclList in ingressAcl.acl.aclList:
                  for rule in aclList.sequence:
                     rule.render( indent=0 )
               print()

         if egressAcls:
            print( 'Egress %s access lists with mirror action:' % aclType )
            for egressAcl in egressAcls:
               print( egressAcl.name )
               for aclList in egressAcl.acl.aclList:
                  for rule in aclList.sequence:
                     rule.render( indent=0 )
               print()

      def printSourceAclIntfs( srcAcls, name, aclType ):
         intfs = [ intf
                   for intf, acl in srcAcls.items()
                   if acl.aclName == name and acl.aclType == aclType ]
         intfs = Arnet.sortIntf( intfs )
         intfsStr = ', '.join( IntfCli.Intf.getShortname( intf ) for intf in intfs )
         aclIntfStr = '{} ({})'.format( name, intfsStr )
         minIntfsLen = 3 * len( 'Eth1/1, ' )
         indent = min( 80 - minIntfsLen, len( name ) + 2 ) * ' '
         print( textwrap.fill( aclIntfStr, width=80, subsequent_indent=indent ) )

      def printSourceAclsWithRules( srcAcls, srcAclsDetails ):
         aclTypes = defaultdict( list )
         for aclName, entry in six.iteritems( srcAclsDetails ):
            if entry.ipAclList:
               aclTypes[ 'ip' ].append( aclName )
            if entry.ipv6AclList:
               aclTypes[ 'ipv6' ].append( aclName )
            if entry.macAclList:
               aclTypes[ 'mac' ].append( aclName )

         # pylint: disable-next=too-many-nested-blocks
         for aType, acls in six.iteritems( aclTypes ):
            print( 'Ingress %s source interface access lists:' % (
               getAclTypeDisplayName( aType ) ) )
            for name in sorted( acls ):
               typedAclList = None
               entry = srcAclsDetails.get( name )
               if entry:
                  printSourceAclIntfs( srcAcls, name, aType )
                  typedAclList = entry.getList( aType )
                  if typedAclList and typedAclList.aclList:
                     for acl in typedAclList.aclList:
                        for rule in acl.sequence:
                           rule.render( indent=0 )
                        print()

      # Helper functions for printing ACLs END

      def showAccessListDetails( self ):
         for name, sessModel in self.sessions.items():
            typedAclList = None
            printSessionHeader = True
            if sessModel.sessionAcl and \
               sessModel.aclType and \
               sessModel.sessionAclDetails:
               typedAclList = sessModel.sessionAclDetails.getList(
                     sessModel.aclType )

            if typedAclList and typedAclList.aclList:
               print( sessionHeader( name ) )
               printSessionHeader = False
               for acl in typedAclList.aclList:
                  print( 'Ingress %s session access list %s:' % (
                     getAclTypeDisplayName( sessModel.aclType ),
                     sessModel.sessionAcl ) )
                  for rule in acl.sequence:
                     rule.render( indent=0 )
                  print()

            if sessModel.aclsWithMirrorAction:
               if printSessionHeader:
                  print( sessionHeader( name ) )
                  printSessionHeader = False
               if sessModel.aclsWithMirrorAction.ipAcls:
                  printAccessListsWithRules( self,
                        'IPv4', sessModel.aclsWithMirrorAction.ipAcls )

               if sessModel.aclsWithMirrorAction.ipv6Acls:
                  printAccessListsWithRules( self,
                        'IPv6', sessModel.aclsWithMirrorAction.ipv6Acls )

               if sessModel.aclsWithMirrorAction.macAcls:
                  printAccessListsWithRules( self,
                        'MAC', sessModel.aclsWithMirrorAction.macAcls )

            if sessModel.sourceAclsDetails:
               if printSessionHeader:
                  print( sessionHeader( name ) )
               printSourceAclsWithRules( sessModel.sourceAcls,
                                         sessModel.sourceAclsDetails )

      if self.sessions:
         if self._summary:
            showSummary( self )
         elif self._detail == "access-list":
            showAccessListDetails( self )
         else:
            for name in sorted( self.sessions ):
               showSession( name, self.sessions[ name ], self.multiDestSessions )
               print()

            hdrRemoval = any( sess.headerRemovalSize
                              for name, sess in six.iteritems( self.sessions ) )
            if self.multiDestSessions and hdrRemoval:
               msg = '* Header removal is not active due to monitor sessions '
               msg += 'having multiple destinations: %s' % (
                        ', '.join( self.multiDestSessions ) )
               print( msg )
               print()

      elif self.name is None:
         print( 'No sessions created' )
      else:
         # They asked for a specific session
         print( 'Session not created' )
      print()

