#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

# -------------------------------------------------------------------------------
# This module implements the configuration commands for port mirroring.
# All commands defined here begin by 'monitor session'
# -------------------------------------------------------------------------------

import Arnet
import BasicCli
import Cell
import CliCommand
import CliGlobal
import CliMatcher
import CliParser
from CliPlugin import AclCli
from CliPlugin import EthIntfCli
from CliPlugin import IntfCli
from CliPlugin.IpAddrMatcher import IpAddrMatcher
from CliPlugin.Ip6AddrMatcher import Ip6AddrMatcher
from CliPlugin.LagIntfCli import LagAutoIntfType
from CliPlugin.MaintenanceCliLib import isSubIntf
from CliPlugin import MirroringMonitorShow
from CliPlugin.RecircIntfCli import EthRecircIntf, RecircAutoIntfType
from CliPlugin.SwitchIntfCli import SwitchAutoIntfType
from CliPlugin.InternalRecircIntfCli import InternalRecircAutoIntfType
from CliPlugin import TechSupportCli
from CliPlugin.VlanIntfCli import VlanIntf
from CliPlugin import VrfCli
from CliPlugin import VlanCli
from CliPlugin.LagCliLib import recircGuard
from CliPlugin import VirtualIntfRule
from CliToken.Ip import ipMatcherForConfigIf
from CliToken.Ipv6 import ipv6MatcherForConfigIf
from CliToken.Mac import macMatcherForConfigIf
import ConfigMount
import inspect
from Intf.IntfRange import(
   IntfRangeMatcher, IntfList, GenericRangeIntfType )
from IntfRangePlugin.EthIntf import EthPhyRangeIntfType
from Vlan import VlanIdRangeList
from IpLibConsts import DEFAULT_VRF
import LazyMount
from CliMode.Mirroring import(
   NamedDestConfigMode, NamedDestVxlanTunnelConfigMode )
from CliParserCommon import MatchResult, noMatch, AlreadyHandledError
import MirroringCliLib
import MirroringLib
import Plugins
import Tac
import Tracing
from TypeFuture import TacLazyType
import re
import copy
import Logging
import six

__defaultTraceHandle__ = Tracing.Handle( 'MirroringMonitorCli' )
t0 = Tracing.trace0

MIRRORING_DEFAULT_GRE_KEY_THIS_SESSION = Logging.LogHandle(
              "MIRRORING_DEFAULT_GRE_KEY_THIS_SESSION",
              severity=Logging.logWarning,
              fmt="Default header key (0x%s) will be applied to GRE "
                  "mirroring session %s.",
              explanation="Since there are other GRE tunnel mirroring sessions "
                          "configured with the header key field, this session "
                          "will contain a default GRE header key value.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

MIRRORING_DEFAULT_GRE_KEY_OTHER_SESSIONS = Logging.LogHandle(
              "MIRRORING_DEFAULT_GRE_KEY_OTHER_SESSIONS",
              severity=Logging.logWarning,
              fmt="Default header key (0x%s) will be applied to existing GRE "
                  "mirroring sessions without a configured key.",
              explanation="The default GRE header key will be applied "
                          "to existing GRE tunnel mirroring sessions "
                          "which have no configured key.  "
                          "This is because session %s has been "
                          "configured with a GRE header key value.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

MIRRORING_GRE_KEY_REMOVED_ALL_SESSIONS = Logging.LogHandle(
              "MIRRORING_GRE_KEY_REMOVED_ALL_SESSIONS",
              severity=Logging.logWarning,
              fmt="Header key will be removed from all GRE mirroring sessions.",
              explanation="The GRE header key field will be removed from "
                          "all GRE tunnel mirroring sessions. This is "
                          "because all other sessions only contain "
                          "the default GRE header key value.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

# Tac types
Direction = TacLazyType( 'Mirroring::Direction' )
GrePayloadType = TacLazyType( 'Mirroring::GrePayloadType' )
SampleRateConstants = TacLazyType( 'Mirroring::SampleRateConstants' )
TruncationSize = TacLazyType( 'Mirroring::TruncationSize' )
VniVtiPair = TacLazyType( "Vxlan::VniVtiPair" )
TunnelMode = TacLazyType( "Mirroring::TunnelMode" )

# Global variables
gv = CliGlobal.CliGlobal( dict(
   # Sysdb mounts
   aclConfig=None,
   aclCounterConfig=None,
   aclCounterStatus=None,
   aclStatusDp=None,
   hwCapability=None,
   mirroringConfig=None,
   mirroringHwCapability=None,
   mirroringHwConfig=None,
   mirroringHwStatus=None,
   mirroringStatus=None,
   vlanStatusDir=None,
   bridgingHwCapabilities=None,
   # Global values
   directionTx=Direction.directionTx,
   directionRx=Direction.directionRx,
   directionBoth=Direction.directionBoth,
   defaultGreHdrProto=TacLazyType( "Mirroring::GreTunnelKey" ).defaultGreHdrProto,
   headerRemovalDefaultVal=0,
) )

# Constants
strToDirection = {
   'rx': gv.directionRx,
   'tx': gv.directionTx,
   'both': gv.directionBoth,
}
# truncation sizes supported on any platform (for token generation, no duplicates)
# Jericho/Jericho+: 128, 192, QumranAx: 256
truncationSizeList = ( 128, 192, 256 )

GreAutoIntfType = GenericRangeIntfType(
      lambda: ( 0, gv.hwCapability.maxTunnelIntfNum - 1 ),
      "Gre", "gre", "GRE tunnel interface" )
MirrorTunnelAutoIntfType = GenericRangeIntfType(
      lambda: ( 0, gv.hwCapability.maxTunnelIntfNum - 1 ),
      "MirrorTunnel", "MirrorTunnel", "Mirroring tunnel interface" )

# ----------------------------------------------------------------------------
# Helper functions
# ----------------------------------------------------------------------------

def isGreIntf( intfName ):
   return intfName.startswith( "MirrorTunnel" )

def isCpuIntf( intfName ):
   return intfName.startswith( "Cpu" )

def isVlan( intfName ):
   return Tac.Type( 'Arnet::VlanIntfId' ).isVlanIntfId( intfName )

def sessionContainsGreDest( sessionConfig ):
   return any( isGreIntf( intf ) for intf in sessionConfig.targetIntf )

def sessionContainsNamedDest( sessionName ):
   sessionConfig = gv.mirroringConfig.session.get( sessionName )
   return sessionConfig and sessionConfig.destinationName != ''

def isRateLimitEgressChipSession( session ):
   return session and session.mirrorRateLimitInBps > 0 and \
          session.mirrorRateLimitChip == 'per-egress-chip'

def rateLimitEgressChipSessionNum( cfg ):
   rateLimitEgressChipSessions = 0
   for session in cfg.session.values():
      if isRateLimitEgressChipSession( session ):
         rateLimitEgressChipSessions += 1
   return rateLimitEgressChipSessions

# Helper function to return the VLAN ID (U16) from intfName such as "Vlan1"
def getVlanId( intfName ):
   assert isVlan( intfName )
   return Tac.Type( 'Arnet::VlanIntfId' ).vlanId( intfName )

def checkInternalSession( mode, name ):
   # pylint: disable-next=consider-using-in
   if ( name == MirroringCliLib.MirroringConstants.lanzMirrorSession or
        name == MirroringCliLib.MirroringConstants.msaMirrorSession or
        name == MirroringCliLib.MirroringConstants.mirrorDestInvalidMirrorSession ):
      emitInternalSessionMsg( mode, name )

def vxlanRangeFn( mode, context=None ):
    # Allow up to 7 VTIs while loading startup-config as hwCapabilities
    # is not yet available.
   if mode and mode.session_.startupConfig() and not mode.session_.guardsEnabled():
      return ( 1, 7 )
   return ( 1, gv.bridgingHwCapabilities.vxlanMaxTunnelInterfacesSupported )

def headerRemovalRangeFn( mode, context=None ):
   if mode and mode.session_.startupConfig() and not mode.session_.guardsEnabled():
      return ( 1, 1 )
   return ( gv.mirroringHwCapability.headerRemovalMin,
            gv.mirroringHwCapability.headerRemovalMax )

def cpuQueueNameMatcher( mode ):
   coppSystemQueues = gv.mirroringHwCapability.mirrorCpuQueueInfo.mirrorCpuQueue
   # hidden keywords no description
   return dict.fromkeys( coppSystemQueues, "" )

# ----------------------------------------------------------------------------
# Guards
# ----------------------------------------------------------------------------
def txMirrorAclSupported( mode ):
   if mode.session_.startupConfig() or \
         gv.mirroringHwCapability.txMirrorAclSupported:
      return True
   return False

def mirroringSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.maxSupportedSessions > 0:
      return None
   return CliParser.guardNotThisPlatform

def mirroringGrePayloadSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.supportedGrePayloadTypes:
      return None
   return CliParser.guardNotThisPlatform

def mirroringGrePayloadTypeSupportedGuard( mode, token ):
   grePayloadType = None
   if token == 'inner-packet':
      grePayloadType = GrePayloadType.payloadTypeInnerPacket
   elif token == 'full-packet':
      grePayloadType = GrePayloadType.payloadTypeFullPacket
   if gv.mirroringHwCapability.supportedGrePayloadTypes.get( grePayloadType ):
      return None
   return CliParser.guardNotThisPlatform

def mirroringGreMetadataSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.greMetadataSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringGreMetadataElementSupportedGuard( mode, token ):
   for metadataColl in six.itervalues(
                              gv.mirroringHwCapability.greMetadataSupported ):
      if ( MirroringLib.tokenMetadataElementDict[ token ]
            in metadataColl.metadataElement.values() ):
         return None
   return CliParser.guardNotThisPlatform

def mirroringGreTimestampingSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.greTimestampingSupported:
      return None
   return CliParser.guardNotThisPlatform

def disablePortMirroringSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.disablePortMirroringSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringDefaultsSupportedGuard( mode, token ):
   # Add additional defaults to the following if statement as needed
   if ( mirroringGrePayloadSupportedGuard( mode, token ) is None
        or mirroringGreMetadataSupportedGuard( mode, token ) is None
        or mirroringGreTimestampingSupportedGuard( mode, token ) is None ):
      return None
   return CliParser.guardNotThisPlatform

def mirroringTruncationSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.truncationSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringTruncationSizeGuard( mode, token ):
   if ( gv.mirroringHwCapability.truncationSizesSupported
        and gv.mirroringHwCapability.truncationSizesSupported.options ):
      return None
   return CliParser.guardNotThisPlatform

def mirroringTruncationSizeValueGuard( mode, token ):
   size = int( token )
   if ( gv.mirroringHwCapability.truncationSizesSupported and size in
         gv.mirroringHwCapability.truncationSizesSupported.options ):
      return None
   return CliParser.guardNotThisPlatform

def mirroringHeaderRemovalSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.headerRemovalSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringDestLagSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.lagDestSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringMultiDestSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.multiDestSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringAclSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.aclSupported:
      return None
   # Some Strata products do not support filtered mirroring but support other
   # mirroring sub-features using TCAM configuration. ( Vlan/SubIntf mirroring )
   if gv.mirroringHwCapability.tcamProfileMapRequired:
      return None
   return CliParser.guardNotThisPlatform

def mirroringIp4AclSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.ip4AclSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringAclPrioritiesSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.aclPrioritiesSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringMacAclSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.macAclSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringIp6AclSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.ip6AclSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringCpuSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.cpuDestSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringCpuQueueSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.mirrorCpuQueueSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringDestinationGreSupported( mode, token ):
   if gv.mirroringHwCapability.destinationGreSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringGreProtocolSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.greProtocolSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringGreKeyIp6SupportedGuard( mode, token ):
   if gv.mirroringHwCapability.greKeyIp6Supported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringGreKeySupportedGuard( mode, token ):
   if gv.mirroringHwCapability.greKeySupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringForwardingDropSupported( mode, token ):
   if gv.mirroringHwCapability.forwardingDropSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringRxSamplingSupported( mode, token ):
   if gv.mirroringHwCapability.rxSamplingSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringRateLimitIngressGuard( mode, token ):
   if gv.mirroringHwCapability.rateLimitIngressChipSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringRateLimitEgressGuard( mode, token ):
   if gv.mirroringHwCapability.rateLimitEgressChipSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringTxGreMetadataPerSessionGuard( mode, token ):
   if gv.mirroringHwCapability.txGreMetadataPerSessionSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringRxVlanSourceGuard( mode, token ):
   if gv.mirroringHwCapability.rxVlanSourceSupported:
      return None
   return CliParser.guardNotThisPlatform

# The problem that IntfRangeConfigRule can't parse interface type which is not
# created at startup-config time has been fixed by change @368045,
# we should use IntfRangeConfigRule here to avoid using separate rules for
# singleton phy interface
def mirroringSubIntfSupportedGuard( mode, token ):
   if( gv.mirroringHwCapability.subIfAsSrcSupported
       or not mode.session_.isInteractive()
       or not mode.session_.guardsEnabled() ):
      return None
   return CliParser.guardNotThisPlatform

def vxlanSrcSupportedGuard( mode, token ):
   if gv.mirroringHwCapability.vxlanSourceSupported:
      return None
   return CliParser.guardNotThisPlatform

def namedMirrorDestSupportedGuard( mode, token ):
   # mirror-to-vxlan is the initial user of named destinations
   if gv.mirroringHwCapability.rxVxlanDestSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringProfileFeatureGuard( mode, token ):
   if gv.mirroringHwCapability.tcamProfileMapRequired:
      return None
   return CliParser.guardNotThisPlatform

def mirroringCongestionFeatureGuard( mode, token ):
   if gv.mirroringHwCapability.mirrorCongestionSupported:
      return None
   return CliParser.guardNotThisPlatform

def mirroringTxGuard( mode, token ):
   if gv.mirroringHwCapability.txMirrorSessionLimit\
      and gv.mirroringHwCapability.maxSupportedTxMirrorSessions == 0:
      return CliParser.guardNotThisPlatform
   return None

# ----------------------------------------------------------------------------
# Matchers
# ----------------------------------------------------------------------------

mirSubIntfSupported = [ mirroringSubIntfSupportedGuard ]

SubIntfRangeDetails = Tac.Type( "Interface::SubIntfIdConstants" )
SubExtendedMin = SubIntfRangeDetails.subIntfExtendedIdMin
SubExtendedMax = SubIntfRangeDetails.subIntfExtendedIdMax
ethPhyRangeIntfType = EthPhyRangeIntfType( intfSubLowerBound=SubExtendedMin,
      intfSubUpperBound=SubExtendedMax )
ethPhyRangeIntfType.subIntfGuard_ = mirSubIntfSupported

lagAutoIntfType = copy.deepcopy( LagAutoIntfType )
lagAutoIntfType.subIntfGuard_ = mirSubIntfSupported

srcIntfTypes = ( ethPhyRangeIntfType,
                 EthIntfCli.UnconnectedEthPhyAutoIntfType,
                 lagAutoIntfType, RecircAutoIntfType, SwitchAutoIntfType,
                 InternalRecircAutoIntfType )

dstIntfTypes = ( EthIntfCli.EthPhyAutoIntfType,
                 LagAutoIntfType,
                 RecircAutoIntfType,
                 SwitchAutoIntfType,
                 InternalRecircAutoIntfType )

dstIntfTypeGuards = { LagAutoIntfType: mirroringDestLagSupportedGuard,
                      RecircAutoIntfType: recircGuard }

excludeNames = \
   '^(?!detail$|default$|default-action$|summary$|destination$)' \
   '([A-Za-z0-9_:{}\\[\\]-]+)'
matcherSessionName = CliMatcher.DynamicNameMatcher(
      lambda mode: gv.mirroringConfig.session,
      'Sessions name', pattern=excludeNames )
matcherDestination = CliMatcher.KeywordMatcher( 'destination',
      helpdesc='Mirroring destination configuration commands' )
matcherNamedDestination = CliCommand.guardedKeyword( 'destination',
      helpdesc='Mirroring destination name',
      guard=namedMirrorDestSupportedGuard )
matcherCongestion = CliCommand.guardedKeyword( 'congestion',
      helpdesc='Mirroring on congestion configuration',
      guard=mirroringCongestionFeatureGuard )
# same as DynamicNameMatcher except the match function is overridding to use
# case-insensitive match to exclude names like PoRt-ChAnNeL1

class DestinationNameMatcher( CliMatcher.DynamicNameMatcher ):

   def __init__( self, namesFn, helpdesc, pattern, **kargs ):
      # pylint: disable-next=super-with-arguments
      super( DestinationNameMatcher, self ).__init__( namesFn,
                                                      helpdesc,
                                                      pattern=pattern,
                                                      **kargs )
      self.reI_ = re.compile( '(?:%s)$' % pattern, flags=re.IGNORECASE )

   def match( self, mode, context, token ):
      m = self.reI_.match( token )
      if m:
         return MatchResult( m if self.rawResult_ else token, token )
      return noMatch

# destination names cannot resemble interface names, including cpu and tunnel
allTags = [ intfType.tagShort for intfType in dstIntfTypes ] + \
          [ intfType.tagLong for intfType in dstIntfTypes ] + \
          [ 'cpu', 'tunnel', 'vlan', 'vl', 'management', 'ma', 'Loopback', 'lo' ]
allowedDestPattern = '^(?!'
for index, tag in enumerate( allTags ):
   intfTypeRe = r'((%s)(\d.*))$' % tag # pattern for an intf e.g Ethernet5 or po1999
   allowedDestPattern += intfTypeRe
   last = index == len( allTags ) - 1
   if not last:
      allowedDestPattern += '|'
allowedDestPattern += ')'
# these trailing chars are needed since PatternMatcher appends a $ to self.re_
trailingCharPattern = r'([A-Za-z_0-9:{}\[\]-]+)'
allowedDestPattern += trailingCharPattern
matcherDestinationName = DestinationNameMatcher(
   lambda mode: gv.mirroringConfig.destination,
   helpdesc='Destination name', pattern=allowedDestPattern )

matcherGre = CliMatcher.KeywordMatcher( 'gre',
      helpdesc='GRE keyword' )
matcherMonitor = CliMatcher.KeywordMatcher( 'monitor',
      helpdesc='Monitor configuration commands' )
matcherSource = CliMatcher.KeywordMatcher( 'source',
      helpdesc='Mirroring source configuration commands' )
matcherSrcIntfTypes = IntfRangeMatcher( explicitIntfTypes=srcIntfTypes )
matcherDstIntfTypes = IntfRangeMatcher( explicitIntfTypes=dstIntfTypes,
                                        intfTypeGuards=dstIntfTypeGuards )
matcherVxlan = VirtualIntfRule.VirtualIntfMatcher(
       'Vxlan', None, None,
       rangeFunc=vxlanRangeFn,
       guard=vxlanSrcSupportedGuard,
       helpdesc="VXLAN Tunnel Interface" )
vniMatcherForConfig = CliMatcher.KeywordMatcher( 'vni',
                       helpdesc='VXLAN Network Identifier configuration' )
# ----------------------------------------------------------------------------
# Nodes
# ----------------------------------------------------------------------------

nodeAccessGroup = CliCommand.Node( matcher=AclCli.accessGroupKwMatcher,
      guard=mirroringAclSupportedGuard )
nodeDefault = CliCommand.guardedKeyword( 'default',
      helpdesc='Default mirroring configuration commands',
      guard=mirroringDefaultsSupportedGuard )
nodeEncapsulation = CliCommand.guardedKeyword( 'encapsulation',
      helpdesc='Mirroring encapsulation configuration commands',
      guard=mirroringDestinationGreSupported )
nodeForwardingDrop = CliCommand.guardedKeyword( 'forwarding-drop',
      helpdesc='Mirror forwarding dropped packets',
      guard=mirroringForwardingDropSupported )
nodeHeader = CliCommand.guardedKeyword( 'header',
      helpdesc='Mirroring header configuration commands',
      guard=mirroringHeaderRemovalSupportedGuard )
nodeIp = CliCommand.Node( matcher=ipMatcherForConfigIf,
      alias='ACLTYPE',
      guard=mirroringIp4AclSupportedGuard )
nodeIpv6 = CliCommand.Node( matcher=ipv6MatcherForConfigIf,
      alias='ACLTYPE',
      guard=mirroringIp6AclSupportedGuard )
nodeMac = CliCommand.Node( matcher=macMatcherForConfigIf,
      alias='ACLTYPE',
      guard=mirroringMacAclSupportedGuard )
nodeMetadata = CliCommand.guardedKeyword( 'metadata',
      helpdesc='Mirroring GRE metadata format configuration',
      guard=mirroringGreMetadataSupportedGuard )
nodePayload = CliCommand.guardedKeyword( 'payload',
      helpdesc='Mirroring GRE payload type configuration commands',
      guard=mirroringGrePayloadSupportedGuard )
nodePriority = CliCommand.guardedKeyword( 'priority',
      helpdesc='Access list priority',
      guard=mirroringAclPrioritiesSupportedGuard )
nodeRemove = CliCommand.guardedKeyword( 'remove',
      helpdesc='Remove leading bytes from packet header',
      guard=mirroringHeaderRemovalSupportedGuard )
nodeSample = CliCommand.guardedKeyword( 'sample',
      helpdesc='Sampling Rate configuration commands',
      guard=mirroringRxSamplingSupported )
nodeSession = CliCommand.guardedKeyword( 'session',
      helpdesc='Monitor configuration commands',
      guard=mirroringSupportedGuard )
nodeTruncate = CliCommand.guardedKeyword( 'truncate',
      helpdesc='Mirroring truncate configuration commands',
      guard=mirroringTruncationSupportedGuard )
nodeTunnel = CliCommand.guardedKeyword( 'tunnel',
      helpdesc='Tunnel destination configuration',
      guard=mirroringDestinationGreSupported )
nodeVlan = CliCommand.guardedKeyword( 'vlan',
      helpdesc='VLAN as Mirroring session\'s source',
      guard=mirroringRxVlanSourceGuard )

nodeBoth = CliCommand.guardedKeyword( 'both',
         helpdesc='Configure mirroring in both transmit and receive directions',
         guard=mirroringTxGuard,
         alias="DIRECTION" )
nodeRx = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'rx',
         helpdesc='Configure mirroring only in receive direction' ),
      alias="DIRECTION" )
nodeTx = CliCommand.guardedKeyword( 'tx',
         helpdesc='Configure mirroring only in transmit direction',
         guard=mirroringTxGuard,
         alias="DIRECTION" )
nodeTcam = CliCommand.guardedKeyword(
   'tcam', helpdesc='TCAM configuration commands',
   guard=mirroringProfileFeatureGuard )
nodeTcamProfileFeature = CliMatcher.KeywordMatcher(
   'feature', helpdesc='TCAM profile feature configuration commands' )

# Match 'ACLTYPE access-group ACLNAME'
class AclTypeNameExpression( CliCommand.CliExpression ):
   expression = """( ip access-group IPACL )
                   | ( ipv6 access-group IPV6ACL )
                   | ( mac access-group MACACL )"""
   data = {
      'access-group': nodeAccessGroup,
      'ip': nodeIp,
      'IPACL': CliCommand.Node(
         matcher=AclCli.userIpAclNameMatcher, alias='ACLNAME' ),
      'ipv6': nodeIpv6,
      'IPV6ACL': CliCommand.Node(
         matcher=AclCli.userIp6AclNameMatcher, alias='ACLNAME' ),
      'mac': nodeMac,
      'MACACL': CliCommand.Node(
         matcher=AclCli.userMacAclNameMatcher, alias='ACLNAME' ),
   }

class MirroringModelet( CliParser.SingletonModelet ):
   pass

BasicCli.GlobalConfigMode.addModelet( MirroringModelet )

# ---------------------------------------------------------------------------
# Helper functions
# ----------------------------------------------------------------------------

def isMaxSessionsReached( mode, sessionToAdd, src=False, forwardingDrop=False ):
   return MirroringCliLib.isMaxSessionsReached( mode, sessionToAdd,
      gv.mirroringHwCapability, gv.mirroringConfig, src, forwardingDrop )

def hasValidSessionConfig( sessionCfg ):
   # Check if the session is in use or not
   return ( sessionCfg.srcIntf or sessionCfg.srcVxlanIntf or sessionCfg.targetIntf
            or sessionCfg.destinationName
            or sessionCfg.truncate or sessionCfg.sessionAcl
            or sessionCfg.headerRemovalSize != gv.headerRemovalDefaultVal
            or sessionCfg.mirrorTcamProfileFeature != "" )

def isValidSource( mode, sessionName, srcIntf, direction, deleteAcl=False ):
   # Source is invalid, if it is present as a source or a destination in any
   # other session or is present as a destination in the current session
   # Note that a single source can be shared between two sessions
   # as long as the direction being monitored does not overlap
   cfg = gv.mirroringConfig
   mySession = cfg.session.get( sessionName )
   mySessionAcl = mySession and mySession.sessionAcl
   for ( name, session ) in cfg.session.items():
      sessionSrcIntf = session.srcIntf.get( srcIntf )
      sessionValid = (
            name != sessionName
            and not mode.session_.startupConfig()
            and sessionSrcIntf
            and ( sessionSrcIntf.direction == direction
                  or gv.directionBoth in ( direction, sessionSrcIntf.direction ) ) )
      if ( sessionValid
           and not gv.mirroringHwCapability.multiSessionForSourceSupported ):
         return 'Duplicate'

      if srcIntf in session.targetIntf:
         if mode.session_.startupConfig() or \
               gv.mirroringHwCapability.twoWayDestinationSupported:
            if name == sessionName or direction != gv.directionRx:
               return 'Duplicate'
            else:
               pass
         else:
            return 'Duplicate'

      if ( sessionValid and gv.mirroringHwCapability.aclPrioritiesSupported ):
         if ( mySessionAcl
              and ( session.aclType != mySession.aclType
                    or ( sessionSrcIntf.srcAcl
                         and sessionSrcIntf.aclType != mySession.aclType ) ) ):
            # If my session ACL is configured, we may support same source in
            # multi-session in two cases:
            # 1) The session ACLs are of different types; or
            # 2) The mySession session ACL type is different from the source
            #    ACL type of the session being checked
            pass
         else:
            emitDuplicateSourceNoAclMsg( mode, srcIntf, name, deleteAcl )
            return 'Duplicate'

      if name == sessionName:
         for targetIntf in session.targetIntf:
            if direction == gv.directionRx or mode.session_.startupConfig():
               continue
            if ( targetIntf.startswith( "Cpu" )
                 and not gv.mirroringHwCapability.txCpuDestSupported ):
               return "CPU destination is only supported in RX direction"
            elif ( isGreIntf( targetIntf ) and not
                   getTxGreDestSupported( isSessionGreIp6( session ) ) ):
               return "Port mirroring to GRE destination is only supported in RX"\
                     " direction"

   # Verify number of TX monitor sessions is within platform limits.
   isNewTxSession = (
         direction in ( gv.directionTx, gv.directionBoth )
         and ( not mySession or not mySession.srcIntf
               or not any( intf.direction in ( gv.directionTx, gv.directionBoth )
                           for intf in mySession.srcIntf.values() ) ) )
   if isNewTxSession and gv.mirroringHwCapability.txMirrorSessionLimit:
      txMirrorSessions = 0
      for session in cfg.session.values():
         for intf in session.srcIntf.values():
            if intf.direction in ( gv.directionTx, gv.directionBoth ):
               txMirrorSessions += 1
               break
      if txMirrorSessions >= gv.mirroringHwCapability.maxSupportedTxMirrorSessions:
         if gv.mirroringHwCapability.maxSupportedTxMirrorSessions == 0:
            mode.addWarning( 'TX mirroring is not supported on this platform. '
                  'Configuring RX.' )
         else:
            return "maximum number of TX monitor sessions exceeded."

   return None

def isValidDestination( mode, sessionName, destIntf ):
   if not gv.mirroringHwCapability.subIfAsDestSupported and isSubIntf( destIntf ):
      return 'Sub-interface destination not supported'

   if sessionContainsNamedDest( sessionName ):
      return 'Destination type conflict'

   if isGreIntf( destIntf ):
      return None

   # Destination is invalid, if it is present as a source or a destination in any
   # other session, or is present as a source interface in the current session
   cfg = gv.mirroringConfig
   # cpu port can serve as destinations of multiple mirror sessions.
   if isCpuIntf( destIntf ):
      if sessionName not in cfg.session:
         return None
      sessionConfig = cfg.session[ sessionName ]

      if not gv.mirroringHwCapability.txCpuDestSupported:
         for intf in cfg.session[ sessionName ].srcIntf.values():
            if intf.direction in ( gv.directionTx, gv.directionBoth ):
               return 'CPU destination only supported in RX direction'

      if sessionContainsGreDest( sessionConfig ):
         return 'Interface type conflict'

      return None

   multiSessionForDestSupported = (
         mode.session_.startupConfig()
         or gv.mirroringHwCapability.multiSessionForDestSupported )
   # For ports other than cpu port, it cannot serve as the mirror destination
   # if it is already a source interface in a session or a destination in another
   # session
   for ( name, session ) in cfg.session.items():
      if destIntf in session.srcIntf:
         if mode.session_.startupConfig() or \
               gv.mirroringHwCapability.twoWayDestinationSupported:
            if ( name == sessionName
                 or session.srcIntf[ destIntf ].direction != gv.directionRx ):
               return 'Duplicate'
            else:
               pass
         else:
            return 'Duplicate'
      if ( not multiSessionForDestSupported
           and name != sessionName and destIntf in session.targetIntf ):
         return 'Duplicate'
      if name == sessionName and sessionContainsGreDest( session ):
         return 'Interface type conflict'

   return None

def isValidNamedDestination( mode, sessionName, destName ):
   sessionCfg = gv.mirroringConfig.session.get( sessionName )
   if not sessionCfg:
      return True
   if sessionCfg.destinationName == destName:
      return True
   # for sessions specifying a named destination, it cannot already have
   # non-named destinations (e.g Ethernet, Port-channel, CPU, GRE, etc)
   existingTargetIntfs = list( sessionCfg.targetIntf )
   if existingTargetIntfs:
      emitNamedDestWithOtherIntfNotSupportedMsg( mode, existingTargetIntfs[ 0 ] )
      return False
   return True

def isIngressOnlySession( sessionName ):
   sessionCfg = gv.mirroringConfig.session.get( sessionName )
   return ( not sessionCfg
            or all( x.direction == gv.directionRx
                    for x in sessionCfg.srcIntf.values() ) )

def isSamplingEnabled( sessionName ):
   sessionCfg = gv.mirroringConfig.session.get( sessionName )
   return ( sessionCfg
            and sessionCfg.rxSampleRate not in
             [ SampleRateConstants.sampleRateUndefined,
               SampleRateConstants.sampleRateMirrorAll ] )

def isSessionGreIp6( sessionCfg ):
   return ( isGreTunnelKeyIp6( sessionCfg.greTunnelKey ) or
            isGreTunnelKeyIp6( sessionCfg.dropGreTunnelKey ) )

def isGreTunnelKeyIp6( greTunnelKey ):
   return greTunnelKey and ( isIp6Addr( greTunnelKey.srcIpGenAddr ) or
                             isIp6Addr( greTunnelKey.dstIpGenAddr ) )

def isIpAddr( addr ):
   return False if addr is None else addr.af == 'ipv4'

def isIp6Addr( addr ):
   return False if addr is None else addr.af == 'ipv6'

def getTxGreDestSupported( ip6 ):
   return ( gv.mirroringHwCapability.txIp6GreDestSupported if ip6 else
            gv.mirroringHwCapability.txGreDestSupported )

# ----------------------------------------------------------------------------
# Mode message emit helpers
# ----------------------------------------------------------------------------

def emitNonExistentSessionMsg( mode, name ):
   errMsg = "Monitor session " + name + " does not exist."
   mode.addError( errMsg )

def emitNonExistentDestinationMsg( mode, name ):
   errMsg = "Monitor session named destination " + name + " does not exist."
   mode.addError( errMsg )

def emitNonExistentSrcIntfMsg( mode, srcIntf, sessionName ):
   errMsg = srcIntf + " is not a source interface in Session " + sessionName
   mode.addError( errMsg )

def emitAclNotMatchMsg( mode, aclName, origAcl ):
   if aclName:
      errMsg = "Access list " + aclName + " cannot be disapplied."
      if origAcl:
         errMsg += " The applied access list is " + origAcl
      else:
         errMsg += " No access list is currently applied."
   else:
      errMsg = "Please specify the access list to be disapplied.\
            The currently applied access list is " + origAcl
   mode.addWarning( errMsg )

def emitProfileMatchError( mode, sessName, feature ):
   errMsg = "Please specify another TCAM feature name." \
      " Feature " + feature + " is applied to session " + sessName
   mode.addError( errMsg )

def emitInternalSessionMsg( mode, name ):
   errMsg = "Monitor session " + name + " is reserved."
   mode.addErrorAndStop( errMsg )

def emitDirectionNotSupportedMsg( mode ):
   errMsg = "Port mirroring to GRE destination is supported in RX direction only"
   mode.addError( errMsg )

def emitGreWithOtherIntfNotSupportedMsg( mode, intfOrDestName ):
   msg = intfOrDestName
   if intfOrDestName.startswith( "MirrorTunnel" ):
      msg = "other tunnel destinations"
   errMsg = "destination cannot be GRE tunnel and " + msg
   mode.addError( errMsg )

def emitDuplicatePriorityNotSupportedMsg( mode, intfName, sessName, priority ):
   errMsg = ( "Source interface " + intfName + " with priority " +
              str( priority ) + " is already configured in monitor session " +
              sessName + ". Please specify a different priority." )
   mode.addError( errMsg )

def emitDuplicateAclNotSupportedMsg( mode, intfName, sessName, aclName ):
   errMsg = ( "Access list " + aclName + " is already applied to source interface " +
              intfName + " in monitor session " + sessName +
              ". Please apply a different access list." )
   mode.addError( errMsg )

def emitDuplicateSourceWithSessionAcl( mode ):
   errMsg = ( "Use of single source interface in multiple monitor sessions "
              "does not support interaction with session access lists." )
   mode.addError( errMsg )

def emitDuplicateSourceNoAclMsg( mode, intfName, sessName, delete=False ):
   errMsg = ( "Interface " + intfName + " is already configured as a source " +
              "interface in monitor session " + sessName + ". " )
   if delete:
      errMsg += ( "Please remove the source interface to disapply the "
                  "access list and priority from this session." )
   else:
      errMsg += ( "Please apply an access list and priority to all monitor sessions "
                  "that share a source interface." )
   mode.addError( errMsg )

def emitSessionAclAppliedMsg( aclName, emitFunc ):
   errMsg = ( "The session is applied with access list %s. "
              "Please remove the access list from the session." % aclName )
   if inspect.isroutine( emitFunc ):
      emitFunc( errMsg )

def emitTxAclNotSupportedMsg( aclName, emitFunc ):
   errMsg = ( "Egress (tx) mirroring is not supported with access list %s. "
              "Applying source to session without the access list." % aclName )
   if inspect.isroutine( emitFunc ):
      emitFunc( errMsg )

def emitDuplicateSourceNoPriorityMsg( mode, intfName, sessName, aclName ):
   errMsg = ( "Use of a single source interface in multiple monitor sessions "
              "requires a priority set on each access list." )
   mode.addError( errMsg )

def emitDuplicateSourceAclDestinationMsg( mode, intfName, sessName ):
   errMsg = ( "Interface " + intfName + " is already configured as a " +
              "destination interface in monitor session " + sessName + "." )
   mode.addError( errMsg )

def emitMaxSessionsReachedMsg( mode, name ):
   errMsg = ( "Could not create session '" + name +
              "'. Maximum mirroring sessions limit reached" )
   mode.addError( errMsg )

def emitMultiSessionNotApplicableMsg( mode, name, ip=False ):
   errMsgIp = ( "Mirror to IPv4 GRE tunnel session with GRE key cannot be " +
                "configured because " + name + ", a mirror to IPv6 GRE tunnel " +
                "session exists." )
   errMsgIp6 = ( "Mirror to IPv6 GRE tunnel session cannot be configured " +
                 "because " + name + ", a mirror to IPv4 GRE tunnel session " +
                 "with GRE key exists." )
   mode.addError( errMsgIp if ip else errMsgIp6 )

def emitNoMultiForwardingDropSessionMsg( mode ):
   errMsg = "forwarding-drop cannot be configured in multiple sessions"
   mode.addError( errMsg )

def emitFwdDropSubInterfaceSrcNotSupportedMsg( mode ):
   errMsg = "A forwarding-drop session cannot have a sub-interface as a source"
   mode.addError( errMsg )

def emitFwdDropVlanSrcNotSupportedMsg( mode ):
   errMsg = "A forwarding-drop session cannot have a VLAN as a source"
   mode.addError( errMsg )

def emitVlanSrcNotSupportedMsg( mode ):
   errMsg = "Setting VLAN as mirror source not supported"
   mode.addError( errMsg )

def emitNamedDestWithOtherIntfNotSupportedMsg( mode, intfName ):
   intfMsg = intfName
   if intfName.startswith( "MirrorTunnel" ):
      intfMsg = "other tunnel destinations"
   errMsg = "Named destination cannot be used with " + intfMsg
   mode.addError( errMsg )

# ----------------------------------------------------------------------------
# Handler functions
# ----------------------------------------------------------------------------

def getSessionConfig( mode, name, delete ):
   # returns: the sessionCfg, or raise AlreadyHandledError
   sessionCfg = gv.mirroringConfig.session.get( name )
   if delete and not sessionCfg:
      emitNonExistentSessionMsg( mode, name )
      raise CliParser.AlreadyHandledError()
   if sessionCfg and sessionCfg.secureMonitor != mode.session.secureMonitor():
      mode.addError( str( CliParser.guardNotPermitted ) )
      raise CliParser.AlreadyHandledError()
   return sessionCfg

def getDestinationConfig( destName ):
   return gv.mirroringConfig.destination.get( destName )

def setMirrorDestination( mode, name, intf, cpuQueue=None, delete=False ):
   cfg = gv.mirroringConfig
   sessionCfg = getSessionConfig( mode, name, delete )
   if delete:
      targetIntf = sessionCfg.targetIntf.get( intf )
      if targetIntf:
         if intf == 'Cpu' and sessionCfg.cpuQueue == cpuQueue:
            sessionCfg.cpuQueue = \
               gv.mirroringHwCapability.mirrorCpuQueueInfo.defaultCpuQueue
         else:
            del sessionCfg.targetIntf[ intf ]
            if not hasValidSessionConfig( sessionCfg ):
               del cfg.session[ name ]
   else:
      # display error if creating more than max sessions
      if isMaxSessionsReached( mode, name, False ):
         emitMaxSessionsReachedMsg( mode, name )
         return
      reason = isValidDestination( mode, name, intf )
      if reason:
         mode.addError( 'Mirror destination %s was rejected '
                        'with reason %s' % ( intf, reason ) )
         return

      if not sessionCfg:
         sessionCfg = cfg.session.newMember( name )
         sessionCfg.secureMonitor = mode.session.secureMonitor()
      sessionCfg.cpuQueue = cpuQueue if cpuQueue \
         else gv.mirroringHwCapability.mirrorCpuQueueInfo.defaultCpuQueue

      # If guards are not enabled or multi-dest is
      # supported add this to the target-intf coll.
      if ( gv.mirroringHwCapability.multiDestSupported
           or not mode.session_.guardsEnabled() ):
         if sessionCfg.targetIntf.get( intf ):
            return
         if ( gv.mirroringHwCapability.destinationsPerSessionLimit
              and len( sessionCfg.targetIntf ) >=
               gv.mirroringHwCapability.maxDestinationsPerSession ):
            mode.addError( 'Mirror destination %s rejected. '
               'Max destinations per session (%d) reached.' % ( intf,
                  gv.mirroringHwCapability.maxDestinationsPerSession ) )
            return

         if ( gv.mirroringHwCapability.multiDestinationSessionsLimit
              and len( sessionCfg.targetIntf ) >= 1 ):
            multiDestsSessions = 0
            for session in cfg.session.values():
               if session.name != name and len( session.targetIntf ) > 1:
                  multiDestsSessions += 1
            if ( multiDestsSessions >=
                 gv.mirroringHwCapability.maxMultiDestinationSessions ):
               mode.addError(
                  'Mirror destination %s rejected. Only %d session(s) '
                  'allowed to have multiple destinations.' % (
                     intf,
                     gv.mirroringHwCapability.maxMultiDestinationSessions ) )
               return

         # Display warning if there is already an interface configured and
         # there is a session with header removal configured
         hdrRemoval = any( sess.headerRemovalSize
                           for name, sess in six.iteritems( cfg.session ) )
         if sessionCfg.targetIntf and hdrRemoval:
            mode.addWarning( 'Header removal may stop working as this '
               'monitor session has multiple destinations configured.' )
         sessionCfg.targetIntf.newMember( intf )
      else:
         sessionCfg.targetIntf.clear()
         sessionCfg.targetIntf.newMember( intf )

def setMirrorDestinations( mode, name, intfList, cpuQueue=None, delete=False ):
   if isinstance( intfList, VlanIntf ):
      raise CliParser.InvalidInputError()
   checkInternalSession( mode, name )
   if intfList == 'cpu':
      setMirrorDestination( mode, name, 'Cpu', cpuQueue=cpuQueue, delete=delete )
      return
   if isinstance( intfList, EthIntfCli.EthPhyIntf ):
      if intfList.name.startswith( 'Management' ):
         raise CliParser.InvalidInputError()
      # Its a singleton object, not a list of intfs
      setMirrorDestination( mode, name, intfList.name, delete=delete )
   elif isinstance( intfList, EthRecircIntf ):
      setMirrorDestination( mode, name, intfList.name, delete=delete )
   elif isinstance( intfList, IntfList ):
      if len( set( intfList ) ) > 1:
         guardCode = mirroringMultiDestSupportedGuard( mode, None )
         if guardCode:
            raise AlreadyHandledError( msg=( "Multiple mirroring destinations are "
                                             "not supported on this platform" ),
                                       msgType=AlreadyHandledError.TYPE_ERROR )
         if gv.mirroringHwCapability.destinationsPerSessionLimit and not delete:
            sessionCfg = gv.mirroringConfig.session.get( name )
            if sessionCfg:
               desiredDests = sum( i not in sessionCfg.targetIntf for i in intfList )
               desiredDests += len( sessionCfg.targetIntf )
            else:
               desiredDests = len( set( intfList ) )
            if desiredDests > gv.mirroringHwCapability.maxDestinationsPerSession:
               mode.addError( 'Mirror destinations rejected. '
                  'Max destinations per session is %d.' % (
                     gv.mirroringHwCapability.maxDestinationsPerSession ) )
               return

         if gv.mirroringHwCapability.multiDestinationSessionsLimit and not delete:
            multiDestsSessions = 0
            for session in gv.mirroringConfig.session.values():
               if session.name != name and len( session.targetIntf ) > 1:
                  multiDestsSessions += 1
            if ( multiDestsSessions >=
                 gv.mirroringHwCapability.maxMultiDestinationSessions ):
               mode.addError(
                  'Mirror destination rejected. Only %d session(s) '
                  'allowed to have multiple destinations.' % (
                     gv.mirroringHwCapability.maxMultiDestinationSessions ) )
               return

      for intfName in intfList:
         setMirrorDestination( mode, name, intfName, delete=delete )
   else:
      raise CliParser.InvalidInputError()

# This function issues CLI warnings / syslogs related to the
# GRE header key field for certain chips.
#
# Refer to aid/8790 for additional background on Strata chip
# limitation behavior.
#
def maybeApplyDefaultGreKey( mode, name, addingThisSession, thisSessionHasKey ):
   # If the underlying hardware does not require "default GRE key"
   # processing, simply exit.
   if not gv.mirroringHwCapability.defaultGreKeySupported:
      return

   thereAreOtherGreSessions = False
   anotherGreSessionHasKey = False

   for session in gv.mirroringConfig.session.values():
      # Skip "this" session. We're only looking
      # at other GRE sessions (if any).
      if ( session.name != name ) and sessionContainsGreDest( session ):
         thereAreOtherGreSessions = True
         if session.greTunnelKey.keyConfigured:
            anotherGreSessionHasKey = True
            break

   keyFormat = "{0:X}"
   keyStr = keyFormat.format( gv.mirroringHwCapability.defaultGreKeyValue )
   # If there are no other mirror sessions with a
   # GRE tunnel destination, there is no need to
   # issue any warnings.
   if thereAreOtherGreSessions:
      if addingThisSession:
         if anotherGreSessionHasKey and not thisSessionHasKey:
            # Issue warning that default GRE key will be applied to the
            # session being added.
            mode.addWarning( 'Default header key (0x%s) will be applied to '
                             'GRE mirroring session %s.'
                             % ( keyStr, name ) )
            Logging.log( MIRRORING_DEFAULT_GRE_KEY_THIS_SESSION,
                         keyStr, name )
         elif not anotherGreSessionHasKey and thisSessionHasKey:
            mode.addWarning( 'Default header key (0x%s) will be applied to '
                             'existing GRE mirroring sessions without '
                             'a key.' %
                             keyStr )
            Logging.log( MIRRORING_DEFAULT_GRE_KEY_OTHER_SESSIONS,
                         keyStr )
      elif not anotherGreSessionHasKey and thisSessionHasKey:
         mode.addWarning( 'Header key field will be removed from all GRE '
                          'mirroring sessions.' )
         Logging.log( MIRRORING_GRE_KEY_REMOVED_ALL_SESSIONS )

def setMirrorDestinationGre( mode, name, greSrcAddr, greDstAddr,
                             forwardingDrop, ttl, dscp, greHdrProto,
                             vrf, keyConfigured, key, delete=False ):
   forwardingDrop = forwardingDrop is not None
   sessionCfg = getSessionConfig( mode, name, delete )

   # We use a dummy MirrorTunnel9998-9 for targetIntf for GRE tunnel dest.
   # When EOS supports GRE intf, cli would change to
   # monitor session <name> destination Gre<num>
   # and user will enter the greIntf<num>. For now, make Mirroring
   # agent happy by setting a dummy GreIntfId to targetIntf.
   # Mirroring agent allocated a correct GRE intfId and publishes
   # it to mirroring/hwconfig/session/<name>/targetIntf
   dummyTargetIntf = "MirrorTunnel9999" if forwardingDrop else "MirrorTunnel9998"
   if delete:
      greTunnelNone = MirroringLib.createGreTunnelKey( '0.0.0.0', '0.0.0.0',
                                                       128, 0, 0 )
      if forwardingDrop:
         if sessionCfg.dropGreTunnelKey is None:
            return
         greTunnelKey = sessionCfg.dropGreTunnelKey
         sessionCfg.dropGreTunnelKey = greTunnelNone
      else:
         if sessionCfg.greTunnelKey is None:
            return
         greTunnelKey = sessionCfg.greTunnelKey
         sessionCfg.greTunnelKey = greTunnelNone
      setMirrorDestination( mode, name, dummyTargetIntf, delete=True )
   else:
      ip = isIpAddr( greSrcAddr ) or isIpAddr( greDstAddr )
      # mirror to IPv4 GRE tunnel session with GRE key cannot coexist with
      # mirror to IPv6 GRE tunnel session
      for sessName, sessCfg in six.iteritems( gv.mirroringConfig.session ):
         if sessName == name:
            t0( "Skip checking of session", sessName, "as it is the same session" )
            continue
         greTunnelKey = sessCfg.greTunnelKey
         # only requires more checking if there is a mirror session with
         # GRE tunnel as destination
         if not gv.mirroringHwCapability.greDestIp4KeyIp6CoexistSupported and \
             greTunnelKey:
            # if current session is IPv4 and GRE key is involved,
            # check if the iterated session is a Mirror to IPv6 GRE session
            if ip and key != 0:
               if ( isIp6Addr( greTunnelKey.srcIpGenAddr ) or
                    isIp6Addr( greTunnelKey.dstIpGenAddr ) ):
                  emitMultiSessionNotApplicableMsg( mode, sessName, ip )
                  return
            # if current session is IPv6, check if the iterated session has GRE key
            elif not ip:
               if greTunnelKey.key != 0:
                  emitMultiSessionNotApplicableMsg( mode, sessName, ip )
                  return

         # only 1 forwarding drop session is allowed
         if forwardingDrop:
            if ( sessCfg.dropGreTunnelKey
                 and sessCfg.dropGreTunnelKey.forwardingDrop ):
               emitNoMultiForwardingDropSessionMsg( mode )
               return
      if isMaxSessionsReached( mode, name, False, forwardingDrop=forwardingDrop ):
         emitMaxSessionsReachedMsg( mode, name )
         return
      if not sessionCfg:
         sessionCfg = gv.mirroringConfig.session.newMember( name )
         sessionCfg.secureMonitor = mode.session.secureMonitor()

      ip6 = isIp6Addr( greSrcAddr ) or isIp6Addr( greDstAddr )
      txGreDestSupported = getTxGreDestSupported( ip6 )

      for srcIntf in sessionCfg.srcIntf.values():
         if ( not mode.session_.startupConfig() and
              srcIntf.direction != gv.directionRx and
              not txGreDestSupported ):
            emitDirectionNotSupportedMsg( mode, )
            return
         if forwardingDrop:
            if isSubIntf( srcIntf.name ):
               emitFwdDropSubInterfaceSrcNotSupportedMsg( mode )
               return
            elif isVlan( srcIntf.name ):
               emitFwdDropVlanSrcNotSupportedMsg( mode )
               return

      existingTargetIntfs = list( sessionCfg.targetIntf )
      if existingTargetIntfs and \
            not existingTargetIntfs[ 0 ].startswith( "MirrorTunnel" ):
         emitGreWithOtherIntfNotSupportedMsg( mode, existingTargetIntfs[ 0 ] )
         return
      if sessionCfg.destinationName:
         emitGreWithOtherIntfNotSupportedMsg( mode, sessionCfg.destinationName )
         return

      if gv.mirroringHwCapability.sessionsSharedGreTunnelSupported:
         sessionName = ''
      else:
         sessionName = name

      srcAddr = None if greSrcAddr is None else greSrcAddr.stringValue
      dstAddr = None if greDstAddr is None else greDstAddr.stringValue

      maybeApplyDefaultGreKey( mode, name, True, keyConfigured )
      greTunnelKey = MirroringLib.createGreTunnelKey( srcAddr, dstAddr,
                                         ttl, dscp,
                                         greHdrProto=greHdrProto,
                                         forwardingDrop=forwardingDrop,
                                         vrf=vrf, keyConfigured=keyConfigured,
                                         key=key, sessionName=sessionName )

      if forwardingDrop:
         greIntfId = gv.mirroringStatus.greTunnelKeyToIntfId.get(
            sessionCfg.dropGreTunnelKey )
         sessionCfg.dropGreTunnelKey = greTunnelKey
      else:
         greIntfId = gv.mirroringStatus.greTunnelKeyToIntfId.get(
            sessionCfg.greTunnelKey )
         sessionCfg.greTunnelKey = greTunnelKey

      if greIntfId != None: # pylint: disable=singleton-comparison
         del sessionCfg.targetIntf[ greIntfId ]
      sessionCfg.targetIntf.newMember( dummyTargetIntf )

def setMirrorSource( mode, name, intfName, direction=gv.directionBoth,
                     delete=False, vxlanIntf=False ):
   cfg = gv.mirroringConfig
   sessionCfg = getSessionConfig( mode, name, delete )
   if delete:
      if vxlanIntf:
         srcIntf = sessionCfg.srcVxlanIntf.get( intfName )
      else:
         srcIntf = sessionCfg.srcIntf.get( intfName )
      if srcIntf:
         # pylint: disable-next=consider-using-in
         if direction == gv.directionBoth or srcIntf.direction == direction:
            if vxlanIntf:
               del sessionCfg.srcVxlanIntf[ intfName ]
            else:
               del sessionCfg.srcIntf[ intfName ]
            if not hasValidSessionConfig( sessionCfg ):
               del cfg.session[ name ]
         else:
            if direction == gv.directionTx and srcIntf.direction == gv.directionBoth:
               setMirrorSource( mode, name, intfName, gv.directionRx,
                                delete=False, vxlanIntf=vxlanIntf )
            if direction == gv.directionRx and srcIntf.direction == gv.directionBoth:
               setMirrorSource( mode, name, intfName, gv.directionTx,
                                delete=False, vxlanIntf=vxlanIntf )
   else:
      # display error if creating more than max sessions
      if isMaxSessionsReached( mode, name, src=True ):
         emitMaxSessionsReachedMsg( mode, name )
         return
      if not vxlanIntf:
         reason = isValidSource( mode, name, intfName, direction )
         if reason:
            mode.addWarning( 'Mirror source ' + intfName +
                             ' was rejected with reason ' + reason )
            return
      if not sessionCfg:
         sessionCfg = cfg.session.newMember( name )
         sessionCfg.secureMonitor = mode.session.secureMonitor()
      if ( not vxlanIntf and isSubIntf( intfName ) and
           not gv.mirroringHwCapability.txSubIntfSourceSupported and
           not mode.session_.startupConfig() and
           direction != gv.directionRx ):
         mode.addError( 'Subintf source can only be used'
                        + ' with ingress mirroring (RX direction)' )
         raise CliParser.InvalidInputError()

      if ( sessionCfg.dropGreTunnelKey
            and sessionCfg.dropGreTunnelKey.forwardingDrop ):
         if isSubIntf( intfName ):
            emitFwdDropSubInterfaceSrcNotSupportedMsg( mode, )
            return
         elif isVlan( intfName ):
            emitFwdDropVlanSrcNotSupportedMsg( mode )
            return
      if ( not vxlanIntf ) and isVlan( intfName ):
         vlanId = getVlanId( intfName )
         if vlanId not in gv.vlanStatusDir.vlanStatus:
            mode.addWarning( "VLAN %d not configured" % vlanId )
      if ( direction != gv.directionRx
           and isSamplingEnabled( name )
           and isIngressOnlySession( name ) ):
         mode.addWarning( 'Sampling disabled due to non-RX source(s).' )

      # get srcIntf object if it exists
      if vxlanIntf:
         src = sessionCfg.srcVxlanIntf.get( intfName )
      else:
         src = sessionCfg.srcIntf.get( intfName )

      # tx mirroring with acl filtering is not supported on some platforms
      if direction == gv.directionTx and not txMirrorAclSupported( mode ):
         # since the new direction is tx the source cannot have an acl
         if src and src.srcAcl != '':
            src.resetAcl()
         # a session wide acl is not compatible with a tx source
         if sessionCfg.sessionAcl != '':
            emitTxAclNotSupportedMsg( sessionCfg.sessionAcl, mode.addError )
            return

      if gv.mirroringHwCapability.txMirrorSessionLimit and\
            gv.mirroringHwCapability.maxSupportedTxMirrorSessions == 0:
         # Due to the mirroringTxGuard placed on tx and both tokens, we cannot have
         # hit the condition above without having configured rx or both, in each
         # case, if we don't support tx mirroring, configure rx.
         direction = gv.directionRx

      # initialize the source (if needed) and set mirroring direction
      # keeps source acl if it still applies to new direction
      newSrc = False
      if not src:
         if vxlanIntf:
            src = sessionCfg.srcVxlanIntf.newMember( direction, intfName )
         else:
            src = sessionCfg.srcIntf.newMember( intfName, direction )
         newSrc = True
      else:
         src.direction = direction

      # copy the session acl settings to any new non-tx source
      if sessionCfg.sessionAcl != '':
         src.aclType = sessionCfg.aclType
         src.srcAcl = sessionCfg.sessionAcl
         if newSrc:
            # TBD: consider adding an aclPriority to the session, which could
            #      be inherited here.
            src.aclPriority = 0

def clearSessionVxlanDest( sessConfig ):
   sessConfig.vxlanTunnelKey = Tac.newInstance( "Mirroring::VxlanTunnelKey" )
   del sessConfig.targetIntf[ 'MirrorTunnel9997' ]

def updateSessionNamedDest( name, destName, delete=False ):
   cfg = gv.mirroringConfig
   sessConfig = cfg.session.get( name )
   if delete:
      clearSessionVxlanDest( sessConfig )
   else:
      destination = cfg.destination[ destName ]
      assert destination
      if destination.tunnelType == TunnelMode.tunnelVxlan:
         tunnelDest = destination.vxlanTunnelDestination
         sessConfig.vxlanTunnelKey = tunnelDest.vxlanTunnelKey
         # same idea as Gre9998
         sessConfig.targetIntf.newMember( "MirrorTunnel9997" )
      elif destination.tunnelType == TunnelMode.tunnelInvalid:
         clearSessionVxlanDest( sessConfig )
      else:
         assert False, "unsupported tunnelType"

# Destination object changed, re-evaluate all affected sessions
def updateNamedDestMirroringConfig( destName ):
   cfg = gv.mirroringConfig
   destination = getDestinationConfig( destName )
   for name, sessConfig in cfg.session.items():
      if destination:
         if name not in destination.session:
            continue
         updateSessionNamedDest( name, destName )
      elif sessConfig.destinationName == destName:
         updateSessionNamedDest( name, destName, delete=True )

def setMirrorNamedDestination( mode, name, destName, delete=False ):
   cfg = gv.mirroringConfig
   session = getSessionConfig( mode, name, delete )
   destination = getDestinationConfig( destName )
   if delete:
      if session.destinationName == destName:
         if destination:
            updateSessionNamedDest( name, destName, delete=True )
            destination.session.remove( name )
         session.destinationName = ''
         if not hasValidSessionConfig( session ):
            del cfg.session[ name ]
   else:
      if isMaxSessionsReached( mode, name, False ):
         emitMaxSessionsReachedMsg( mode, name )
         return
      if not isValidNamedDestination( mode, name, destName ):
         return
      if not session:
         session = cfg.session.newMember( name )
         session.secureMonitor = mode.session.secureMonitor()
      for dest in cfg.destination.values():
         dest.session.remove( name )
      session.destinationName = destName
      if destination:
         destination.session.add( name )
         updateSessionNamedDest( name, destName )

def setMirrorSources( mode, name, intfList, direction=gv.directionBoth,
                      delete=False, vxlanIntfs=False ):
   checkInternalSession( mode, name )
   if isinstance( intfList, VlanIntf ):
      raise CliParser.InvalidInputError()
   if isinstance( intfList, EthIntfCli.EthPhyIntf ):
      if intfList.name.startswith( 'Management' ):
         raise CliParser.InvalidInputError()
      # Its a singleton object, not a list of intfs
      setMirrorSource( mode, name, intfList.name, direction, delete )
   elif isinstance( intfList, IntfList ):
      for intfName in intfList:
         setMirrorSource( mode, name, intfName, direction, delete )
   elif isinstance( intfList, VlanIdRangeList ):
      for vlanRange in intfList.ranges:
         for vlan in range( vlanRange[ 0 ], vlanRange[ 1 ] + 1 ):
            setMirrorSource( mode, name, 'Vlan%d' % vlan, direction, delete )
   else:
      raise CliParser.InvalidInputError()

def setDefaultGrePayloadType( payloadType ):
   cfg = gv.mirroringConfig
   defaultSessionConfig = cfg.defaultSessionConfig
   defaultSessionConfig.grePayloadType = payloadType

def setDefaultGreMetadata( greMetadata ):
   cfg = gv.mirroringConfig
   defaultSessionConfig = cfg.defaultSessionConfig
   defaultSessionConfig.greMetadata.clear()
   for metadata, value in six.iteritems( greMetadata ):
      defaultSessionConfig.greMetadata[ metadata ] = value

def setDefaultGreTimestamping( timestampingEnabled ):
   cfg = gv.mirroringConfig
   defaultSessionConfig = cfg.defaultSessionConfig
   defaultSessionConfig.greTimestampingEnabled = timestampingEnabled

def setDefaultAction( disablePortMirroring, defaultAction ):
   if defaultAction == 'no-mirror':
      cfg = gv.mirroringConfig
      defaultSessionConfig = cfg.defaultSessionConfig
      defaultSessionConfig.portMirroringDisabled = disablePortMirroring

def isValidSourceAcl( mode, sessName, intfName, direction,
                      aclType, aclName, aclPriority ):
   cfg = gv.mirroringConfig
   for ( name, session ) in cfg.session.items():
      if intfName in session.targetIntf:
         if mode.session_.startupConfig() or \
               gv.mirroringHwCapability.twoWayDestinationSupported:
            if name == sessName or direction != gv.directionRx:
               emitDuplicateSourceAclDestinationMsg( mode, intfName, name )
               return False
            else:
               pass
         else:
            emitDuplicateSourceAclDestinationMsg( mode, intfName, name )
            return False

      if ( name == sessName
           or intfName not in session.srcIntf
           or mode.session_.startupConfig() ):
         continue

      # Check for duplicate source with session acl
      # we support same source in different session with different ACL
      # type.
      if session.sessionAcl and session.aclType == aclType:
         emitDuplicateSourceWithSessionAcl( mode )
         return False

      srcToCheck = session.srcIntf.get( intfName )
      if ( ( direction == gv.directionRx
             and srcToCheck.direction == gv.directionTx )
           or ( direction == gv.directionTx
                and srcToCheck.direction == gv.directionRx ) ):
         # If the duplicate source has strictly nonoverlapping direction
         # then we can still apply an ACL
         continue

      # Check for duplicate source without acl
      if not srcToCheck.srcAcl:
         emitDuplicateSourceNoAclMsg( mode, intfName, name )
         return False

      # Check for duplicate source without priority
      if ( ( not aclPriority or not srcToCheck.aclPriority )
           and aclType == srcToCheck.aclType ):
         emitDuplicateSourceNoPriorityMsg( mode, intfName, name, aclName )
         return False

      # Check for duplicate Acl
      if aclName == srcToCheck.srcAcl and aclType == srcToCheck.aclType:
         emitDuplicateAclNotSupportedMsg( mode, intfName, name, aclName )
         return False

      # Check for duplicate priority
      if ( aclPriority == srcToCheck.aclPriority
           and aclType == srcToCheck.aclType ):
         emitDuplicatePriorityNotSupportedMsg( mode, intfName, name, aclPriority )
         return False
   return True

def setSourceAcl( mode, name, intfName, direction,
                  aclType='ip', aclName='', aclPriority=0,
                  delete=False ):
   cfg = gv.mirroringConfig
   sessionCfg = getSessionConfig( mode, name, delete )
   if delete:
      srcIntf = sessionCfg.srcIntf.get( intfName )
      if not srcIntf:
         emitNonExistentSrcIntfMsg( mode, intfName, name )
         return
      if sessionCfg.sessionAcl != '':
         # source and session acl may differ if tx acl is not supported
         if srcIntf.direction != gv.directionTx or srcIntf.srcAcl != '' or \
            txMirrorAclSupported( mode ):
            assert sessionCfg.sessionAcl == srcIntf.srcAcl
         emitSessionAclAppliedMsg( sessionCfg.sessionAcl, mode.addWarning )
         return
      if srcIntf.srcAcl == aclName:
         # we should use actual configured srcIntf direction to validate
         cfgDirection = srcIntf.direction
         reason = isValidSource( mode, name, intfName, cfgDirection, delete )
      else:
         emitAclNotMatchMsg( mode, aclName, srcIntf.srcAcl )
         reason = 'Access list does not match'

      if reason:
         mode.addWarning( 'Removing mirror source access list on '
                          + intfName + ' was rejected with reason ' + reason )
      else:
         # Reset aclType to default
         srcIntf.resetAcl()
   else:
      if aclName == '':
         return

      if isMaxSessionsReached( mode, name, src=True ):
         emitMaxSessionsReachedMsg( mode, name )
         return
      if gv.mirroringHwCapability.aclPrioritiesSupported:
         status = isValidSourceAcl( mode, name, intfName, direction,
                                    aclType, aclName, aclPriority )
         if status is False:
            return
      # This won't catch everything but just add a bit more check
      aclConfig = gv.aclConfig.config[ aclType ].acl.get( aclName )
      if aclConfig and aclConfig.secureMonitor != mode.session.secureMonitor():
         mode.addError( str( CliParser.guardNotPermitted ) )
         return

      if not sessionCfg:
         sessionCfg = cfg.session.newMember( name )
         sessionCfg.secureMonitor = mode.session.secureMonitor()

      src = sessionCfg.srcIntf.get( intfName )
      if not src:
         for targetIntf in sessionCfg.targetIntf:
            if direction == gv.directionRx or mode.session_.startupConfig():
               continue
            if ( isGreIntf( targetIntf ) and not
                getTxGreDestSupported( isSessionGreIp6( sessionCfg ) ) ):
               emitDirectionNotSupportedMsg( mode )
               return

         src = sessionCfg.srcIntf.newMember( intfName, direction )
         if sessionCfg.sessionAcl:
            src.aclType = sessionCfg.aclType
            src.srcAcl = sessionCfg.sessionAcl
            # TBD: consider adding an aclPriority to the session, which could
            # be inherited here.
            src.aclPriority = 0
      if src.direction == gv.directionTx and not txMirrorAclSupported( mode ):
         mode.addWarning(
               "Access list cannot be applied to ports whose egress traffic are"
               " mirrored." )
         return

      if not ( mode.session_.startupConfig()
          or gv.mirroringHwCapability.multipleAclTypesPerSessionSupported ) and \
          src.srcAcl and src.aclType != aclType:
         mode.addWarning(
               "Source %s in Session %s is applied with access list %s with type %s "
               "instead of access list %s with type %s. "
               % ( intfName, name, src.srcAcl, src.aclType, aclName, aclType ) +
               "Only one access list type per session is supported." )
         return

      if src.srcAcl != '' and ( src.srcAcl != aclName or src.aclType != aclType ):
         mode.addWarning(
               "Source %s in Session %s was applied with access list %s of acl "
               "type %s, please remove this access list first before applying "
               "another access list."
               % ( intfName, name, src.srcAcl, src.aclType ) )
         return

      if not ( mode.session_.startupConfig()
               or gv.mirroringHwCapability.multipleAclTypesPerSessionSupported ):
         for srcIntfName in sessionCfg.srcIntf:
            srcIntf = sessionCfg.srcIntf[ srcIntfName ]
            if srcIntf.srcAcl != "" and srcIntf.aclType != aclType:
               mode.addError(
                  "Source %s in Session %s was applied with access list type %s. "
                  % ( srcIntfName, name, srcIntf.aclType ) +
                  "Only one access list type per session is supported." )
               return

      if aclName not in gv.aclConfig.config[ aclType ].acl:
         mode.addWarning(
            "Access list %s not configured. Assigning anyway." % aclName )

      src.aclType = aclType
      src.srcAcl = aclName
      src.aclPriority = aclPriority

def setSourcesAcl( mode, name, intfList, aclName, direction, aclType='ip',
                   aclPriority=0, delete=False ):
   checkInternalSession( mode, name )
   for intfName in intfList:
      setSourceAcl( mode, name, intfName, direction, aclType,
                    aclName, aclPriority, delete )

def setMirrorTruncate( mode, name, delete=False, size=None ):
   checkInternalSession( mode, name )
   cfg = gv.mirroringConfig
   sessionCfg = getSessionConfig( mode, name, delete )
   if delete:
      sessionCfg.truncate = False
      sessionCfg.truncationSize = TruncationSize.null
      if not hasValidSessionConfig( sessionCfg ):
         del cfg.session[ name ]
   else:
      if not sessionCfg:
         sessionCfg = cfg.session.newMember( name )
         sessionCfg.secureMonitor = mode.session.secureMonitor()
      sessionCfg.truncate = True
      if size:
         sessionCfg.truncationSize = size
      else:
         sessionCfg.truncationSize = TruncationSize.null

def setMirrorHeaderRemoval( mode, name, size=gv.headerRemovalDefaultVal,
                            delete=False ):
   checkInternalSession( mode, name )
   cfg = gv.mirroringConfig
   sessionCfg = getSessionConfig( mode, name, delete )
   if delete:
      sessionCfg.headerRemovalSize = gv.headerRemovalDefaultVal
      if not hasValidSessionConfig( sessionCfg ):
         del cfg.session[ name ]
   else:
      multiDestSessions = [ key for key, status in
                            six.iteritems( gv.mirroringStatus.sessionStatus )
                            if status.multiDestActive ]

      if multiDestSessions:
         mode.addWarning( 'Header removal will not work as there '
            'are monitor sessions with multiple active destinations: %s.' %
            ( ', '.join( multiDestSessions ) ) )
      if not sessionCfg:
         sessionCfg = cfg.session.newMember( name )
         sessionCfg.secureMonitor = mode.session.secureMonitor()
      sessionCfg.headerRemovalSize = size

def setMirrorSampleRate( mode, name, rate, delete=False ):
   cfg = gv.mirroringConfig
   sessionCfg = getSessionConfig( mode, name, delete )
   if not sessionCfg:
      sessionCfg = cfg.session.newMember( name )
      sessionCfg.secureMonitor = mode.session.secureMonitor()
   sessionCfg.rxSampleRate = rate
   if ( rate != SampleRateConstants.sampleRateMirrorAll
        and not isIngressOnlySession( name ) ):
      mode.addWarning( 'Sampling disabled due to non-RX source(s).' )

def setSessionProfile( mode, name, feature, delete=False ):
   cfg = gv.mirroringConfig
   sessionCfg = getSessionConfig( mode, name, delete=delete )
   if not sessionCfg:
      sessionCfg = cfg.session.newMember( name )
      sessionCfg.secureMonitor = mode.session.secureMonitor()
   if feature != '':
      for session in cfg.session.values():
         # If any other session is configured with same
         # tcam feature then emit error
         if session.mirrorTcamProfileFeature == feature and session.name != name:
            emitProfileMatchError( mode, session.name, feature )
            return
   sessionCfg.mirrorTcamProfileFeature = feature
   # Remove session if delete=True and session is not valid
   if delete:
      if not hasValidSessionConfig( sessionCfg ):
         del cfg.session[ name ]

def setSessionCongestion( mode, name, delete=False ):
   checkInternalSession( mode, name )
   cfg = gv.mirroringConfig
   sessionCfg = getSessionConfig( mode, name, delete )
   if delete:
      if sessionCfg:
         sessionCfg.congestionMonitor = False
   else:
      if not sessionCfg:
         sessionCfg = cfg.session.newMember( name )
      sessionCfg.congestionMonitor = True

def setSessionAcl( mode, name, aclName="", aclType="ip", delete=False ):
   checkInternalSession( mode, name )
   cfg = gv.mirroringConfig
   sessionCfg = getSessionConfig( mode, name, delete )
   if delete:
      if sessionCfg.sessionAcl != aclName:
         emitAclNotMatchMsg( mode, aclName, sessionCfg.sessionAcl )
         return
      intfNames = Arnet.sortIntf( sessionCfg.srcIntf )
      for intfName in intfNames:
         srcIntf = sessionCfg.srcIntf[ intfName ]
         # source and session acl may differ if tx acl is not supported
         if srcIntf.direction != gv.directionTx or srcIntf.srcAcl != '' or \
            txMirrorAclSupported( mode ):
            assert srcIntf.srcAcl == sessionCfg.sessionAcl
         # Reset aclType to default
         srcIntf.resetAcl()
      intfNames = ( sessionCfg.srcVxlanIntf )
      for intfName in intfNames:
         srcIntf = sessionCfg.srcVxlanIntf[ intfName ]
         # source and session acl may differ if tx acl is not supported
         if srcIntf.direction != gv.directionTx or srcIntf.srcAcl != '' or \
            txMirrorAclSupported( mode ):
            assert srcIntf.srcAcl == sessionCfg.sessionAcl
         # Reset aclType to default
         srcIntf.resetAcl()
      sessionCfg.sessionAcl = ''
      if not hasValidSessionConfig( sessionCfg ):
         del cfg.session[ name ]
      return

   if not aclName:
      return
   if isMaxSessionsReached( mode, name, src=True ):
      emitMaxSessionsReachedMsg( mode, name )
      return
   # This won't catch everything but just add a bit more check
   aclConfig = gv.aclConfig.config[ aclType ].acl.get( aclName )
   if aclConfig and aclConfig.secureMonitor != mode.session.secureMonitor():
      mode.addError( str( CliParser.guardNotPermitted ) )
      return

   # if the acl is not configured, warn the user of this first
   if aclName not in gv.aclConfig.config[ aclType ].acl:
      mode.addWarning(
         "Access list %s not configured. Assigning anyway." % aclName )

   if not sessionCfg:
      sessionCfg = cfg.session.newMember( name )
      sessionCfg.secureMonitor = mode.session.secureMonitor()
      sessionCfg.sessionAcl = ''
   else:
      # if the session is already applied with the same acl, we do not
      # do anything.
      if sessionCfg.sessionAcl == aclName and sessionCfg.aclType == aclType:
         return

      if sessionCfg.secureMonitor != mode.session.secureMonitor():
         mode.addError( str( CliParser.guardNotPermitted ) )
         return

      # if the session is already applied with another acl, the user
      # need to remove the acl first
      if sessionCfg.sessionAcl != '':
         mode.addWarning(
            "Session %s was applied with %s access list %s, "
            "please remove this access list first before "
            "applying another access list."
            % ( name, sessionCfg.aclType, sessionCfg.sessionAcl ) )
         return

      # if any source in the session is already applied with an acl, the user
      # need to remove the acl first
      intfNames = Arnet.sortIntf( sessionCfg.srcIntf )
      for intfName in intfNames:
         srcIntf = sessionCfg.srcIntf[ intfName ]
         if srcIntf.srcAcl != '':
            mode.addWarning(
               "Source %s in the session was applied with access list %s, "
               "please remove the access list first before applying an "
               "access list to the session."
               % ( intfName, srcIntf.srcAcl ) )
            return
         if srcIntf.direction == gv.directionTx:
            mode.addWarning(
                  "Access list cannot be applied because at least one "
                  "source port has egress traffic mirroring." )
            return

   sessionCfg.aclType = aclType
   sessionCfg.sessionAcl = aclName
   # Update the srcAcl in each src in the mirror session
   intfNames = Arnet.sortIntf( sessionCfg.srcIntf )
   for intfName in intfNames:
      srcIntf = sessionCfg.srcIntf[ intfName ]
      srcIntf.aclType = aclType
      srcIntf.srcAcl = aclName
   intfNames = ( sessionCfg.srcVxlanIntf )
   for intfName in intfNames:
      srcIntf = sessionCfg.srcVxlanIntf[ intfName ]
      srcIntf.aclType = aclType
      srcIntf.srcAcl = aclName

def doShowMirrorSessions( mode, args ):
   return MirroringMonitorShow.doShowMirrorSessions( gv, mode, args )

# ----------------------------------------------------------------------------
# CLI hander functions
# ---------------------------------------------------------------------------

def destinationGreCommand( mode, name, greSrcAddr, greDstAddr,
                           forwardingDrop=None, ttlDscpProtoVrfKey=None ):
   ttl = 128
   dscp = 0
   greHdrProto = gv.defaultGreHdrProto
   vrf = DEFAULT_VRF
   key = 0
   keyConfigured = False
   for param in ttlDscpProtoVrfKey:
      if param[ 0 ] == 'ttl':
         ttl = param[ 1 ]
      elif param[ 0 ] == 'dscp':
         dscp = param[ 1 ]
      elif param[ 0 ] == 'protocol':
         greHdrProto = param[ 1 ]
      elif param[ 0 ] == 'VRF':
         vrf = param[ 1 ]
      elif param[ 0 ] == 'key' or param[ 0 ] == 'KEY_KW':
         key = param[ 1 ]
         keyConfigured = True

   if greSrcAddr == greDstAddr:
      errMsg = 'source and destination cannot be same'
      mode.addError( errMsg )
      return
   if ( not mode.session_.startupConfig() and
        ( not gv.mirroringHwCapability.mirrorToIp6GreSupported
          and ( greSrcAddr.af == 'ipv6' or greDstAddr.af == 'ipv6' ) ) ):
      raise AlreadyHandledError( msg='IPv6 is not supported on this platform',
                                 msgType=AlreadyHandledError.TYPE_ERROR )

   def isInvalidGreAddress( ip ):
      return ip.isMulticast or ip.isLinkLocal or ip.isBroadcast

   if isInvalidGreAddress( greSrcAddr ):
      errMsg = 'Invalid source IP address'
      mode.addError( errMsg )
      return

   if isInvalidGreAddress( greDstAddr ):
      errMsg = 'Invalid destination IP address'
      mode.addError( errMsg )
      return

   setMirrorDestinationGre( mode, name, greSrcAddr, greDstAddr,
                            forwardingDrop, ttl, dscp,
                            greHdrProto, vrf, keyConfigured, key )

def noDestinationGreCommand( mode, name, greSrcAddr=None, greDstAddr=None,
                             forwardingDrop=None, ttlDscpProtoVrfKey=None ):
   maybeApplyDefaultGreKey( mode, name, False, False )
   setMirrorDestinationGre( mode, name, greSrcAddr, greDstAddr,
                            forwardingDrop, ttl=128, dscp=0,
                            greHdrProto=gv.defaultGreHdrProto,
                            vrf=DEFAULT_VRF, keyConfigured=False, key=0,
                            delete=True )

def defaultGreMetadataCommand( mode, args ):
   upper2Byte = args[ 'UPPER2B' ]
   lower2Byte = args[ 'LOWER2B' ]
   greMetadata = { MirroringLib.tokenMetadataElementDict[ upper2Byte ]: True,
                   MirroringLib.tokenMetadataElementDict[ lower2Byte ]: True }
   setDefaultGreMetadata( greMetadata )

def noDefaultGreMetadataCommand( mode, args ):
   setDefaultGreMetadata( {} )

def rateLimitCommand( mode, name, rateLimitChip, rateLimit, rateLimitUnit,
                      delete=False ):
   mirrorRateLimitInBps = 0
   if rateLimitUnit == 'bps':
      mirrorRateLimitInBps = rateLimit
   elif rateLimitUnit == 'kbps':
      mirrorRateLimitInBps = rateLimit * 1000
   elif rateLimitUnit == 'mbps':
      mirrorRateLimitInBps = rateLimit * 1000 * 1000
   else:
      mode.addWarning( 'Invalid rate-limit configuration.' )
      return
   # The Cli will limit the input range to 18 kpbs and 300 gbps
   # These values are from AradPolicerConstants minRateKbps and maxRateKbps
   if rateLimit > 0:
      if mirrorRateLimitInBps < 18000 or mirrorRateLimitInBps > 300 * 1e9:
         msg = 'rate-limit value should be between 18kbps and 300gbps.'
         mode.addWarning( msg )
         return

   cfg = gv.mirroringConfig
   sessionCfg = getSessionConfig( mode, name, delete=delete )

   if not sessionCfg:
      sessionCfg = cfg.session.newMember( name )
      sessionCfg.secureMonitor = mode.session.secureMonitor()
   sessionCfg.mirrorRateLimitChip = rateLimitChip
   sessionCfg.mirrorRateLimitInBps = mirrorRateLimitInBps

   if not delete and rateLimitChip == 'per-egress-chip' and \
      gv.mirroringHwCapability.rateLimitEgressChipSupported:
      numLimit = gv.mirroringHwCapability.maxRateLimitEgressChipSessions
      if rateLimitEgressChipSessionNum( cfg ) > numLimit:
         msg = ( 'rate-limit per-egress-chip function can only work on maximum %d '
                 'mirroring sessions due to hardware limitations.' ) % numLimit
         mode.addWarning( msg )

def noRateLimitCommand( mode, name, rateLimitChip ):
   rateLimitCommand( mode, name, rateLimitChip, 0, 'bps',
                     delete=True )

def noSessionCommand( mode, args ):
   name = args[ 'NAME' ]
   checkInternalSession( mode, name )
   cfg = getSessionConfig( mode, name, True )
   if cfg:
      session = gv.mirroringConfig.session[ name ]
      if session.greTunnelKey is not None:
         maybeApplyDefaultGreKey( mode, name, False,
                                  session.greTunnelKey.keyConfigured )
      dest = getDestinationConfig( session.destinationName )
      if dest:
         dest.session.remove( name )
      del gv.mirroringConfig.session[ name ]

# ---------------------------------------------------------------------------
# CLI commands classes
# ---------------------------------------------------------------------------

# --------------------------------------------------------------------------------
# ( no | default ) monitor session NAME
# --------------------------------------------------------------------------------
class NoSessionCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'monitor session NAME'
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'NAME': matcherSessionName,
   }

   noOrDefaultHandler = noSessionCommand

MirroringModelet.addCommandClass( NoSessionCmd )

# --------------------------------------------------------------------------------
# [ no | default ] monitor session NAME ( mac | ip | ipv6 ) access-group ACLNAME
# --------------------------------------------------------------------------------
def sessionAclCommand( mode, args ):
   delete = CliCommand.isNoOrDefaultCmd( args )
   name = args[ 'NAME' ]
   aclName = args[ 'ACLNAME' ]
   aclType = args[ 'ACLTYPE' ]
   setSessionAcl( mode, name, aclName, aclType=aclType, delete=delete )

class NameAclCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor session NAME ACL'
   noOrDefaultSyntax = syntax
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'NAME': matcherSessionName,
      'ACL': AclTypeNameExpression,
   }

   handler = sessionAclCommand
   noOrDefaultHandler = handler

MirroringModelet.addCommandClass( NameAclCmd )

# --------------------------------------------------------------------------------
# monitor session NAME [ forwarding-drop ] destination tunnel mode gre
#     source GRESRCADDR destination GREDSTADDR
#     [ { ( ( ttl TTLRANGE ) | ( dscp DSCPRANGE )
#           | ( protocol PROTORANGE ) | VRF ) } ]
# ( no | default ) monitor session NAME [ forwarding-drop ] destination tunnel
#     mode gre ...
# --------------------------------------------------------------------------------

class NameDestinationTunnelModeGreCmd( CliCommand.CliCommandClass ):
   syntax = """monitor session NAME [ forwarding-drop ] destination tunnel mode gre
      source ( ( GRE_SRC_IPADDR GRE_DEST GRE_DST_IPADDR [ { ( key KEYRANGE )
                                                          | ( ttl TTLRANGE )
                                                          | ( dscp DSCPRANGE )
                                                          | ( protocol PROTORANGE )
                                                          | VRF } ] )
             | ( GRE_SRC_IP6ADDR GRE_DEST GRE_DST_IP6ADDR [ { ( KEY_KW KEYRANGE )
                                                            | ( ttl TTLRANGE )
                                                            | ( dscp DSCPRANGE )
                                                            | ( protocol PROTORANGE )
                                                            | VRF } ] ) )"""
   noOrDefaultSyntax = """
      monitor session NAME [ forwarding-drop ] destination tunnel mode gre ..."""
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'NAME': matcherSessionName,
      'forwarding-drop': nodeForwardingDrop,
      'destination': matcherDestination,
      'tunnel': nodeTunnel,
      'mode': 'Tunnel mode selection',
      'gre': 'Generic Routing Encapsulation',
      'source': 'GRE source configuration',
      'GRE_SRC_IPADDR': IpAddrMatcher(
         helpdesc='Source IP address of GRE tunnel' ),
      'GRE_SRC_IP6ADDR': Ip6AddrMatcher(
         helpdesc='Source IPv6 address of GRE tunnel' ),
      'GRE_DEST': CliMatcher.KeywordMatcher(
         'destination', helpdesc='GRE destination configuration' ),
      'GRE_DST_IPADDR': IpAddrMatcher(
         helpdesc='Destination IP address of GRE tunnel' ),
      'GRE_DST_IP6ADDR': Ip6AddrMatcher(
         helpdesc='Destination IPv6 address of GRE tunnel' ),
      # SetRule: ttl, dscp, protocol and vrf can be matched in any order
      # but each only once
      # i.e. ( ttl, vrf, dscp ) valid
      #      ( ttl, vrf, ttl ) invalid
      'ttl': CliCommand.singleKeyword( 'ttl', helpdesc='TTL of the GRE tunnel' ),
      'TTLRANGE': CliMatcher.IntegerMatcher( 1, 255, helpdesc='TTL range' ),
      'dscp': CliCommand.singleKeyword( 'dscp', helpdesc='DSCP of the GRE tunnel' ),
      'DSCPRANGE': CliMatcher.IntegerMatcher( 0, 63, helpdesc='DSCP range' ),
      'protocol': CliCommand.singleKeyword(
         'protocol', helpdesc='Protocol type in GRE header',
         guard=mirroringGreProtocolSupportedGuard ),
      'PROTORANGE': CliMatcher.IntegerMatcher( 0, 65535,
         helpdesc='Protocol range', helpname='0x0000-0xFFFF' ),
      'VRF': VrfCli.VrfExprFactory( helpdesc='VRF name of the GRE tunnel',
                                    inclDefaultVrf=True, maxMatches=1 ),
      'key': CliCommand.singleKeyword( 'key', helpdesc='GRE key',
                                       guard=mirroringGreKeySupportedGuard ),
      'KEY_KW': CliCommand.singleKeyword( 'key', helpdesc='GRE key',
                                          guard=mirroringGreKeyIp6SupportedGuard ),
      'KEYRANGE': CliMatcher.IntegerMatcher( 0, 2**32 - 1, helpdesc='key value' ),
   }

   @staticmethod
   def handler( mode, args ):
      name = args[ 'NAME' ]
      forwardingDrop = args.get( 'forwarding-drop' )
      greSrcAddr = args.get( 'GRE_SRC_IPADDR' ) or args.get( 'GRE_SRC_IP6ADDR' )
      greDstAddr = args.get( 'GRE_DST_IPADDR' ) or args.get( 'GRE_DST_IP6ADDR' )
      greSrcIpGenAddr = Arnet.IpGenAddr( str( greSrcAddr ) )
      greDstIpGenAddr = Arnet.IpGenAddr( str( greDstAddr ) )

      ttlDscpProtoVrfKey = []
      params = (
         ( 'ttl', 'TTLRANGE' ),
         ( 'dscp', 'DSCPRANGE' ),
         ( 'protocol', 'PROTORANGE' ),
         ( 'VRF', 'VRF' ),
         ( 'key', 'KEYRANGE' ),
         ( 'KEY_KW', 'KEYRANGE' ),
      )
      for paramName, valName in params:
         if paramName in args:
            val = args[ valName ]
            if isinstance( val, list ):
               val = val[ 0 ]
            ttlDscpProtoVrfKey.append( ( paramName, val ) )
      destinationGreCommand(
            mode, name, greSrcIpGenAddr, greDstIpGenAddr,
            forwardingDrop, ttlDscpProtoVrfKey )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      name = args[ 'NAME' ]
      forwardingDrop = args.get( 'forwarding-drop' )
      noDestinationGreCommand( mode, name, forwardingDrop=forwardingDrop )

MirroringModelet.addCommandClass( NameDestinationTunnelModeGreCmd )

# --------------------------------------------------------------------------------
# [ no | default ] monitor session NAME destination ( CPU | ETH | LAG | DEST )
# --------------------------------------------------------------------------------

def noDestinationCommand( mode, args ):
   destinationCommand( mode, args, delete=True )

def destinationCommand( mode, args, delete=False ):
   name = args.get( 'NAME' )
   intf = args.get( 'INTF' )
   dest = args.get( 'DEST' )
   cpuQueue = args.get( 'QUEUE' )
   if dest:
      setMirrorNamedDestination( mode, name, dest, delete=delete )
   else:
      setMirrorDestinations( mode, name, intf, cpuQueue=cpuQueue, delete=delete )

class NameDestinationCmd( CliCommand.CliCommandClass ):
   syntax = """monitor session NAME destination 
      ( ( cpu [ QUEUE ] ) | INTFS | DEST )"""
   noOrDefaultSyntax = syntax
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'NAME': matcherSessionName,
      'destination': matcherDestination,
      'cpu': CliCommand.guardedKeyword(
         'cpu', helpdesc='CPU port(s)',
         guard=mirroringCpuSupportedGuard,
         alias="INTF" ),
      'QUEUE': CliCommand.Node(
         matcher=CliMatcher.DynamicKeywordMatcher( cpuQueueNameMatcher,
                                                   alwaysMatchInStartupConfig=True ),
         guard=mirroringCpuQueueSupportedGuard ),
      'INTFS': CliCommand.Node( matcher=matcherDstIntfTypes, alias="INTF" ),
      'DEST': CliCommand.Node( matcherDestinationName,
                  guard=namedMirrorDestSupportedGuard )
   }

   handler = destinationCommand
   noOrDefaultHandler = noDestinationCommand

MirroringModelet.addCommandClass( NameDestinationCmd )

# --------------------------------------------------------------------------------
# monitor session NAME header remove size SIZE
# ( no | default ) monitor session NAME header remove ...
# --------------------------------------------------------------------------------

def headerRemovalCommand( mode, args ):
   delete = CliCommand.isNoOrDefaultCmd( args )
   name = args[ 'NAME' ]
   size = args.get( 'SIZE', gv.headerRemovalDefaultVal )
   setMirrorHeaderRemoval( mode, name, size=size, delete=delete )

class NameHeaderRemoveCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor session NAME header remove size SIZE'
   noOrDefaultSyntax = 'monitor session NAME header remove ...'
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'NAME': matcherSessionName,
      'header': nodeHeader,
      'remove': nodeRemove,
      'size': 'Set the size of header to be removed',
      'SIZE': CliMatcher.DynamicIntegerMatcher(
         headerRemovalRangeFn,
         helpdesc='Mirroring header removal size in bytes' ),
   }

   handler = headerRemovalCommand
   noOrDefaultHandler = handler

MirroringModelet.addCommandClass( NameHeaderRemoveCmd )

# --------------------------------------------------------------------------------
# monitor session NAME rate-limit ( per-ingress-chip | per-egress-chip )
#     LIMIT ( bps | kbps | mbps )
# ( no | default ) monitor session NAME rate-limit
#     ( per-ingress-chip | per-egress-chip ) ...
# --------------------------------------------------------------------------------

class NameRateLimitCmd( CliCommand.CliCommandClass ):
   syntax = """monitor session NAME rate-limit
               ( per-ingress-chip | per-egress-chip ) LIMIT UNIT"""
   noOrDefaultSyntax = """monitor session NAME rate-limit
                          ( per-ingress-chip | per-egress-chip ) ..."""
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'NAME': matcherSessionName,
      'rate-limit': 'Mirroring rate limit commands',
      'per-ingress-chip': CliCommand.guardedKeyword(
         'per-ingress-chip', helpdesc='Rate limit at ingress chip',
         guard=mirroringRateLimitIngressGuard ),
      'per-egress-chip': CliCommand.guardedKeyword(
          'per-egress-chip', helpdesc='Rate limit at egress chip',
          guard=mirroringRateLimitEgressGuard ),
      'LIMIT': CliMatcher.IntegerMatcher( 1, 10000000000,
         helpdesc='Mirroring rate limit' ),
      'UNIT': CliMatcher.EnumMatcher( {
         'bps': 'Rate unit bps',
         'kbps': 'Rate unit kbps',
         'mbps': 'Rate unit mbps',
      } ),
   }

   @staticmethod
   def handler( mode, args ):
      name = args[ 'NAME' ]
      rateLimitChip = (
            'per-ingress-chip' if 'per-ingress-chip' in args else 'per-egress-chip' )
      rateLimit = args[ 'LIMIT' ]
      rateLimitUnit = args[ 'UNIT' ]
      rateLimitCommand(
            mode, name, rateLimitChip, rateLimit, rateLimitUnit )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      name = args[ 'NAME' ]
      rateLimitChip = (
            'per-ingress-chip' if 'per-ingress-chip' in args else 'per-egress-chip' )
      noRateLimitCommand( mode, name, rateLimitChip )

MirroringModelet.addCommandClass( NameRateLimitCmd )

# --------------------------------------------------------------------------------
# monitor session NAME sample SAMPLERATE
# ( no | default ) monitor session NAME sample ...
# --------------------------------------------------------------------------------

def sampleRateCommand( mode, args ):
   name = args[ 'NAME' ]
   rate = args.get( 'SAMPLERATE', SampleRateConstants.sampleRateMirrorAll )
   setMirrorSampleRate( mode, name, rate=rate,
                        delete=CliCommand.isNoOrDefaultCmd( args ) )

class NameSampleSampleCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor session NAME sample SAMPLERATE'
   noOrDefaultSyntax = 'monitor session NAME sample ...'
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'NAME': matcherSessionName,
      'sample': nodeSample,
      'SAMPLERATE': CliMatcher.IntegerMatcher( 1, 0xFFFFFFFE,
         helpdesc='1/<rate> is the packet sampling probability' ),
   }

   handler = sampleRateCommand
   noOrDefaultHandler = handler

MirroringModelet.addCommandClass( NameSampleSampleCmd )

# --------------------------------------------------------------------------------
# [ no | default ] monitor session NAME source INTFS [ rx | tx | both ]
# --------------------------------------------------------------------------------

def setMirrorSourcesCommand( mode, args ):
   delete = CliCommand.isNoOrDefaultCmd( args )
   name = args[ 'NAME' ]
   intfList = args[ 'INTFS' ]
   direction = args.get( 'DIRECTION', 'both' )
   direction = strToDirection[ direction ]
   setMirrorSources(
         mode, name, intfList, direction=direction, delete=delete )

class NameSourceIntfsRxTxBothCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor session NAME source INTFS [ rx | tx | both ]'
   noOrDefaultSyntax = syntax
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'NAME': matcherSessionName,
      'source': matcherSource,
      'INTFS': matcherSrcIntfTypes,
      'rx': nodeRx,
      'tx': nodeTx,
      'both': nodeBoth,
   }

   handler = setMirrorSourcesCommand
   noOrDefaultHandler = handler

MirroringModelet.addCommandClass( NameSourceIntfsRxTxBothCmd )

# --------------------------------------------------------------------------------
# [ no | default ] monitor session NAME source VXLANINTFS vni VNINUM
#                                                 [ rx | tx | both ]
# --------------------------------------------------------------------------------

def setVxlanMirrorSourcesCommand( mode, args ):
   delete = CliCommand.isNoOrDefaultCmd( args )
   name = args[ 'NAME' ]
   vtiName = args[ 'VXLANINTF' ]
   vni = args.get( 'VNINUM' )
   direction = args.get( 'DIRECTION', 'both' )
   direction = strToDirection[ direction ]
   vtiIntfId = Tac.Value( "Arnet::IntfId", vtiName )
   vniExt = Tac.Value( "Vxlan::VniExt", vni )
   vniVti = Tac.Value( "Vxlan::VniVtiPair", vniExt, vtiIntfId )
   setMirrorSource( mode, name, vniVti, direction, delete=delete,
                       vxlanIntf=True )

# To match VNI, Ideally we should use the matcher function in
# VxlanCli or VxlanController.
# However using these functions is causing dependency cycles.
# Please see bug729214.
class NameVxlanSourceIntfsRxTxBothCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor session NAME source VXLANINTF vni VNINUM [ rx | tx | both ]'
   noOrDefaultSyntax = syntax
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'NAME': matcherSessionName,
      'source': matcherSource,
      'VXLANINTF': matcherVxlan,
      'vni': vniMatcherForConfig,
      'VNINUM': CliMatcher.IntegerMatcher( 1, ( 2 ** 32 ) - 2,
                helpdesc='Virtual Network Identifier' ),
      'rx': nodeRx,
      'tx': nodeTx,
      'both': nodeBoth,
   }
   handler = setVxlanMirrorSourcesCommand
   noOrDefaultHandler = handler

MirroringModelet.addCommandClass( NameVxlanSourceIntfsRxTxBothCmd )

# --------------------------------------------------------------------------------
# [ no | default ] monitor session NAME source vlan VLANS rx
# --------------------------------------------------------------------------------

def setMirrorVlanSourcesCommand( mode, args ):
   delete = CliCommand.isNoOrDefaultCmd( args )
   name = args[ 'NAME' ]
   intfList = args[ 'VLANS' ]
   direction = args.get( 'DIRECTION', 'rx' )
   direction = strToDirection[ direction ]
   setMirrorSources(
         mode, name, intfList, direction=direction, delete=delete )

class NameSourceVlansRxCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor session NAME source vlan VLANS [ rx ]'
   noOrDefaultSyntax = syntax
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'NAME': matcherSessionName,
      'source': matcherSource,
      'vlan': nodeVlan,
      'VLANS': VlanCli.vlanIdRangeMatcher,
      'rx': nodeRx,
   }

   handler = setMirrorVlanSourcesCommand
   noOrDefaultHandler = handler

MirroringModelet.addCommandClass( NameSourceVlansRxCmd )

# --------------------------------------------------------------------------------
# [ no | default ] monitor session NAME source INTFS [ rx | both ]
#     ( ip | ipv6 | mac ) access-group ACLNAME [ priority PRIORITYVALUE ] ]
# --------------------------------------------------------------------------------

def setSourceAclCommand( mode, args ):
   delete = CliCommand.isNoOrDefaultCmd( args )
   name = args[ 'NAME' ]
   intfList = args[ 'INTFS' ]
   aclName = args[ 'ACLNAME' ]
   aclType = args[ 'ACLTYPE' ]
   priority = args.get( 'PRIORITYVALUE', 0 )
   direction = args.get( 'DIRECTION', 'both' )
   direction = strToDirection[ direction ]
   setSourcesAcl( mode, name, intfList, aclName, direction,
                               aclType=aclType, aclPriority=priority,
                               delete=delete )

class NameSourceIntfsAclCmd( CliCommand.CliCommandClass ):
   syntax = """monitor session NAME source INTFS [ rx | tx | both ]
               ACL [ priority PRIORITYVALUE ]"""
   noOrDefaultSyntax = syntax
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'NAME': matcherSessionName,
      'source': matcherSource,
      'INTFS': matcherSrcIntfTypes,
      'rx': nodeRx,
      'tx': nodeTx,
      'both': nodeBoth,
      'ACL': AclTypeNameExpression,
      'priority': nodePriority,
      'PRIORITYVALUE': CliMatcher.IntegerMatcher( 0, 100,
         helpdesc='ACL priority value' ),
   }

   handler = setSourceAclCommand
   noOrDefaultHandler = handler

MirroringModelet.addCommandClass( NameSourceIntfsAclCmd )

# --------------------------------------------------------------------------------
# [ no | default ] monitor session NAME truncate [ size SIZE ]
# --------------------------------------------------------------------------------

def truncateCommand( mode, args ):
   delete = CliCommand.isNoOrDefaultCmd( args )
   name = args[ 'NAME' ]
   size = args.get( 'SIZE' )
   if size is not None:
      size = int( size )
   setMirrorTruncate( mode, name, delete=delete, size=size )

class NameTruncateCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor session NAME truncate [ size SIZE ]'
   noOrDefaultSyntax = 'monitor session NAME truncate ...'
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'NAME': matcherSessionName,
      'truncate': nodeTruncate,
      'size': CliCommand.guardedKeyword( 'size',
         helpdesc='Mirroring truncation size configuration commands',
         guard=mirroringTruncationSizeGuard ),
      'SIZE': CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( {
            str( size ): 'Nominal mirroring truncation size in bytes'
            for size in truncationSizeList
         } ), guard=mirroringTruncationSizeValueGuard )
   }

   handler = truncateCommand
   noOrDefaultHandler = handler

MirroringModelet.addCommandClass( NameTruncateCmd )

# --------------------------------------------------------------------------------
# monitor session default encapsulation gre metadata
#     ( ingress-vlan | egress-port ) ingress-port
# ( no | default ) monitor session default encapsulation gre metadata ...
# --------------------------------------------------------------------------------

# BUG433081: The current implementation is specific to Sand platform. Ideally, the
# syntax should be derived from mirroringHwCapability.greMetadataSupported
class DefaultEncapsulationGreMetadataCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor session default encapsulation gre metadata UPPER2B LOWER2B'
   noOrDefaultSyntax = 'monitor session default encapsulation gre metadata ...'
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'default': nodeDefault,
      'encapsulation': nodeEncapsulation,
      'gre': matcherGre,
      'metadata': nodeMetadata,
      'UPPER2B': CliCommand.Node( matcher=CliMatcher.EnumMatcher( {
            'ingress-vlan': 'Ingress port VLAN ID',
            'egress-port': 'Egress port ID',
         } ), guard=mirroringGreMetadataElementSupportedGuard ),
      'LOWER2B': CliCommand.Node( matcher=CliMatcher.EnumMatcher( {
            'ingress-port': 'Ingress port ID',
         } ), guard=mirroringGreMetadataElementSupportedGuard ),
   }

   handler = defaultGreMetadataCommand
   noOrDefaultHandler = noDefaultGreMetadataCommand

MirroringModelet.addCommandClass( DefaultEncapsulationGreMetadataCmd )

# --------------------------------------------------------------------------------
# monitor session default encapsulation gre payload ( full-packet | inner-packet )
# ( no | default ) monitor session default encapsulation gre payload ...
# --------------------------------------------------------------------------------

class DefaultEncapsulationGrePayloadCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor session default encapsulation gre payload TYPE'
   noOrDefaultSyntax = 'monitor session default encapsulation gre payload ...'
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'default': nodeDefault,
      'encapsulation': nodeEncapsulation,
      'gre': matcherGre,
      'payload': nodePayload,
      'TYPE': CliCommand.Node( matcher=CliMatcher.EnumMatcher( {
            'full-packet': 'Mirror full packet',
            'inner-packet': 'Mirror packet without terminated headers',
         } ), guard=mirroringGrePayloadTypeSupportedGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      payloadType = {
            'full-packet': GrePayloadType.payloadTypeFullPacket,
            'inner-packet': GrePayloadType.payloadTypeInnerPacket,
      }[ args[ 'TYPE' ] ]
      setDefaultGrePayloadType( payloadType )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setDefaultGrePayloadType( GrePayloadType.payloadTypeHwDefault )

MirroringModelet.addCommandClass( DefaultEncapsulationGrePayloadCmd )

# --------------------------------------------------------------------------------
# monitor session default encapsulation gre timestamp
# ( no | default ) monitor session default encapsulation gre timestamp ...
# --------------------------------------------------------------------------------

class DefaultEncapsulationGreTimestampCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor session default encapsulation gre timestamp'
   noOrDefaultSyntax = syntax + ' ...'
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'default': nodeDefault,
      'encapsulation': nodeEncapsulation,
      'gre': matcherGre,
      'timestamp': CliCommand.guardedKeyword( 'timestamp',
         helpdesc='Add timestamps in the GRE payload of mirrored packets',
         guard=mirroringGreTimestampingSupportedGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      setDefaultGreTimestamping( timestampingEnabled=True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setDefaultGreTimestamping( timestampingEnabled=False )

MirroringModelet.addCommandClass( DefaultEncapsulationGreTimestampCmd )

# --------------------------------------------------------------------------------
# Named Destination (used for tunneled mirror destinations) definitions
# --------------------------------------------------------------------------------

class NamedDestContext( object ): # pylint: disable=useless-object-inheritance

   def __init__( self, destName ):
      self.destName = destName

   def childMode( self ):
      return NamedDestConfigMode

   def vxlanTunnelConfigMode( self ):
      return NamedDestVxlanTunnelConfigMode

   def updateNamedDestConfig( self, delete=False ):
      cfg = gv.mirroringConfig
      if delete:
         del cfg.destination[ self.destName ]
      else:
         destination = cfg.destination.newMember( self.destName )
         for session in cfg.session.values():
            if session.destinationName == self.destName:
               destination.session.add( session.name )
      updateNamedDestMirroringConfig( self.destName )

   def updateNamedDestTunnelType( self, tunnelType ):
      cfg = gv.mirroringConfig
      destination = cfg.destination[ self.destName ]
      destination.tunnelType = tunnelType
      if tunnelType == TunnelMode.tunnelVxlan:
         destination.vxlanTunnelDestination = ( 'vxlan', )
      elif tunnelType == TunnelMode.tunnelInvalid:
         destination.vxlanTunnelDestination = None
      else:
         assert False, 'unsupported tunnelType'
      updateNamedDestMirroringConfig( self.destName )

   def updateNamedDestVti( self, vti ):
      cfg = gv.mirroringConfig
      destination = cfg.destination[ self.destName ]
      assert destination.tunnelType == TunnelMode.tunnelVxlan
      if vti:
         vti = Tac.Value( "Arnet::IntfId", vti )
      else:
         vti = Tac.Value( "Arnet::IntfId" )
      oldKey = destination.vxlanTunnelDestination.vxlanTunnelKey
      destination.vxlanTunnelDestination.vxlanTunnelKey = \
         MirroringLib.updateVxlanTunnelKey( oldKey, vti=vti )
      updateNamedDestMirroringConfig( self.destName )

   def updateNamedDestVlan( self, vlanId ):
      cfg = gv.mirroringConfig
      destination = cfg.destination[ self.destName ]
      assert destination.tunnelType == TunnelMode.tunnelVxlan
      oldKey = destination.vxlanTunnelDestination.vxlanTunnelKey
      destination.vxlanTunnelDestination.vxlanTunnelKey = \
         MirroringLib.updateVxlanTunnelKey( oldKey, vlan=vlanId )
      updateNamedDestMirroringConfig( self.destName )

   def updateNamedDestDestVtep( self, mode, destVtep, ttl=None, dscp=None ):
      def invalidDestVtepAddr( ip ):
         return ip.isMulticast or ip.isLinkLocal or ip.isBroadcast
      cfg = gv.mirroringConfig
      destination = cfg.destination[ self.destName ]
      assert destination.tunnelType == TunnelMode.tunnelVxlan
      if destVtep:
         destVtep = Tac.Value( "Arnet::IpGenAddr", destVtep )
      else:
         destVtep = Tac.Value( "Arnet::IpGenAddr" )
      if invalidDestVtepAddr( destVtep ):
         errMsg = 'Invalid remote VTEP IP address'
         mode.addError( errMsg )
         return
      vxlanTunnelKey = Tac.Type( "Mirroring::VxlanTunnelKey" )
      defaultTtl = vxlanTunnelKey.defaultVxlanTtl
      defaultDscp = vxlanTunnelKey.defaultVxlanDscp
      updatedFields = { 'remoteVtepIp': destVtep,
                        'ttl': defaultTtl, 'dscp': defaultDscp }
      if ttl:
         updatedFields[ 'ttl' ] = ttl
      if dscp:
         updatedFields[ 'dscp' ] = dscp
      oldKey = destination.vxlanTunnelDestination.vxlanTunnelKey
      destination.vxlanTunnelDestination.vxlanTunnelKey = \
            MirroringLib.updateVxlanTunnelKey( oldKey, **updatedFields )
      updateNamedDestMirroringConfig( self.destName )

matcherVti = VirtualIntfRule.VirtualIntfMatcher(
       'Vxlan', None, None,
       rangeFunc=vxlanRangeFn,
       helpdesc="VXLAN Tunnel Interface" )

# --------------------------------------------------------------------------------
# monitor session destination DEST
#    tunnel mode vxlan
#       vxlan interface <vti>
#       vlan <vlan>
#       [ vtep <vtepAddr> ] [ dscp <dscp> ] [ ttl <ttl> ]
# --------------------------------------------------------------------------------
class VxlanTunnelDestVxlanIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan interface VTI'
   noOrDefaultSyntax = 'vxlan interface ...'
   data = {
      'vxlan': 'Configure VXLAN Tunnel Interface',
      'interface': 'VXLAN Tunnel Interface',
      'VTI': matcherVti
   }

   @classmethod
   def handler( cls, mode, args ):
      context = mode.context
      vti = args[ 'VTI' ]
      context.updateNamedDestVti( vti )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      context = mode.context
      context.updateNamedDestVti( Tac.newInstance( "Arnet::IntfId" ) )

class VxlanTunnelDestVlanCmd( CliCommand.CliCommandClass ):
   syntax = 'vlan VLAN_ID'
   noOrDefaultSyntax = 'vlan ...'
   data = {
      'vlan': 'Configure VXLAN VLAN',
      'VLAN_ID': VlanCli.vlanIdMatcher
   }

   @classmethod
   def handler( cls, mode, args ):
      context = mode.context
      vlan = args[ 'VLAN_ID' ]
      vlanId = Tac.Value( "Bridging::VlanIdOrNone", vlan.id_ )
      context.updateNamedDestVlan( vlanId )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      context = mode.context
      vlanId = Tac.Value( "Bridging::VlanIdOrNone", 0 )
      context.updateNamedDestVlan( vlanId )

class VxlanTunnelDestVtepParamCmd( CliCommand.CliCommandClass ):
   syntax = 'vtep DEST_VTEP [ { ( ttl TTL ) | ( dscp DSCP ) } ]'
   noOrDefaultSyntax = 'vtep ...'
   data = {
      'vtep': 'Configure VXLAN mirror destination VTEP address',
      'DEST_VTEP': IpAddrMatcher(
         helpdesc='VXLAN mirror destination remote VTEP IPv4 address' ),
      'ttl': CliCommand.singleKeyword( 'ttl',
         helpdesc='TTL of the vxlan tunnel' ),
      'TTL': CliMatcher.IntegerMatcher( 1, 255, helpdesc='TTL range' ),
      'dscp': CliCommand.singleKeyword( 'dscp',
         helpdesc='DSCP of the vxlan tunnel' ),
      'DSCP': CliMatcher.IntegerMatcher( 0, 63, helpdesc='DSCP range' ),
   }

   @classmethod
   def handler( cls, mode, args ):
      context = mode.context
      vtep = args[ 'DEST_VTEP' ]
      ttl = args.get( 'TTL', [ None ] )[ 0 ]
      dscp = args.get( 'DSCP', [ None ] )[ 0 ]
      context.updateNamedDestDestVtep( mode, vtep, ttl=ttl, dscp=dscp )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      context = mode.context
      context.updateNamedDestDestVtep( mode, Tac.Value( "Arnet::IpGenAddr" ) )

# --------------------------------------------------------------------------------
# monitor session destination DEST
#    [ no | default ] tunnel mode TUNNEL_TYPE
# --------------------------------------------------------------------------------
class NamedDestTunnelModeCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel mode TUNNEL_TYPE'
   noOrDefaultSyntax = 'tunnel mode ...'
   data = {
      'tunnel': 'Configure mirror destination tunnel',
      'mode': 'Tunnel mode',
      'TUNNEL_TYPE': CliMatcher.EnumMatcher( {
         'vxlan': 'Mirror traffic to a VXLAN tunnel'
      } ),
   }

   @classmethod
   def handler( cls, mode, args ):
      context = mode.context
      if context.destName not in gv.mirroringConfig.destination:
         emitNonExistentDestinationMsg( mode, context.destName )
         return
      if args[ 'TUNNEL_TYPE' ] == 'vxlan':
         tunnelType = TunnelMode.tunnelVxlan
         tunMode = context.vxlanTunnelConfigMode()
      else:
         assert False, "unknown tunnel mode"
      childMode = mode.childMode( tunMode, context=context, tunnelType=tunnelType )
      mode.session_.gotoChildMode( childMode )
      context.updateNamedDestTunnelType( tunnelType )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      context = mode.context
      if context.destName not in gv.mirroringConfig.destination:
         emitNonExistentDestinationMsg( mode, context.destName )
         return
      context.updateNamedDestTunnelType( TunnelMode.tunnelInvalid )

# --------------------------------------------------------------------------------
# monitor session destination DEST
# [ no | default ] monitor session destination DEST
# --------------------------------------------------------------------------------

class NamedDestConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor session destination DEST_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'destination': matcherNamedDestination,
      'DEST_NAME': matcherDestinationName
   }

   @staticmethod
   def _context( destName ):
      return NamedDestContext( destName )

   @classmethod
   def handler( cls, mode, args ):
      destName = args[ 'DEST_NAME' ]
      maxNamedDestSupported = gv.mirroringHwCapability.maxNamedDestinations
      if len( gv.mirroringConfig.destination ) >= maxNamedDestSupported and \
         destName not in gv.mirroringConfig.destination:
         mode.addError( 'Number of destinations cannot exceed limit %d' %
                        maxNamedDestSupported )
         return
      context = cls._context( destName )
      childMode = mode.childMode( context.childMode(), context=context )
      mode.session_.gotoChildMode( childMode )
      context.updateNamedDestConfig()

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      destName = args[ 'DEST_NAME' ]
      context = cls._context( destName )
      context.updateNamedDestConfig( delete=True )

MirroringModelet.addCommandClass( NamedDestConfigCmd )
NamedDestConfigMode.addCommandClass( NamedDestTunnelModeCmd )
NamedDestVxlanTunnelConfigMode.addCommandClass( VxlanTunnelDestVxlanIntfCmd )
NamedDestVxlanTunnelConfigMode.addCommandClass( VxlanTunnelDestVlanCmd )
NamedDestVxlanTunnelConfigMode.addCommandClass( VxlanTunnelDestVtepParamCmd )
#NamedDestVxlanTunnelConfigMode.addCommandClass( VxlanTunnelDestDscpDefaultCmd )
#NamedDestVxlanTunnelConfigMode.addCommandClass( VxlanTunnelDestTtlDefaultCmd )

# --------------------------------------------------------------------------------
# [ no | default ] monitor session default-action ACTION
# --------------------------------------------------------------------------------

def defaultActionCommand( mode, args ):
   disablePortMirroring = not CliCommand.isNoOrDefaultCmd( args )
   defaultAction = args[ 'ACTION' ]
   setDefaultAction( disablePortMirroring, defaultAction )

class DefaultActionNoMirrorCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor session default-action ( no-mirror )'
   noOrDefaultSyntax = syntax
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'default-action': CliCommand.guardedKeyword( 'default-action',
         helpdesc='Set the default action when no access lists are programmed',
         guard=disablePortMirroringSupportedGuard ),
      # Actions
      'no-mirror': CliCommand.guardedKeyword(
         'no-mirror', helpdesc='No mirroring when no access list is configured',
         guard=disablePortMirroringSupportedGuard,
         alias="ACTION" ),
   }

   handler = defaultActionCommand
   noOrDefaultHandler = handler

MirroringModelet.addCommandClass( DefaultActionNoMirrorCmd )

# --------------------------------------------------------------------------------
# monitor session NAME encapsulation gre metadata tx
# ( no | default ) monitor session NAME encapsulation gre metadata tx
# --------------------------------------------------------------------------------

def sessionGreMetadataCommand( mode, args ):
   disableMetadata = not CliCommand.isNoOrDefaultCmd( args )
   name = args[ 'NAME' ]
   sessionCfg = getSessionConfig( mode, name, True )
   sessionCfg.txGreMetadataEnabled = disableMetadata

class SessionEncapsulationMetadataCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor session NAME encapsulation gre metadata tx'
   noOrDefaultSyntax = syntax + "..."
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'NAME': matcherSessionName,
      'encapsulation': nodeEncapsulation,
      'gre': matcherGre,
      'metadata': nodeMetadata,
      'tx': CliCommand.guardedKeyword(
         'tx', helpdesc='Add metadata in tx mirrored packet. The packet will be'
         ' truncated to 128 bytes', guard=mirroringTxGreMetadataPerSessionGuard )
   }

   handler = sessionGreMetadataCommand
   noOrDefaultHandler = sessionGreMetadataCommand

MirroringModelet.addCommandClass( SessionEncapsulationMetadataCmd )

# --------------------------------------------------------------------------------
# [ no | default ] monitor session NAME access-group tcam feature
#    mirror-feature{1|2|3|4}
# --------------------------------------------------------------------------------
def sessionProfileCommand( mode, args ):
   delete = CliCommand.isNoOrDefaultCmd( args )
   name = args[ 'NAME' ]
   featureName = args.get( 'FEATURENAME', '' )
   setSessionProfile( mode, name, featureName, delete=delete )

class NameProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor session NAME access-group tcam feature FEATURENAME'
   noOrDefaultSyntax = syntax.replace( 'FEATURENAME', '...' )
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'NAME': matcherSessionName,
      'access-group': nodeAccessGroup,
      'tcam': nodeTcam,
      'feature': nodeTcamProfileFeature,
      'FEATURENAME': CliMatcher.EnumMatcher( {
         'mirror-feature1': 'Mirror TCAM profile feature 1',
         'mirror-feature2': 'Mirror TCAM profile feature 2',
         'mirror-feature3': 'Mirror TCAM profile feature 3',
         'mirror-feature4': 'Mirror TCAM profile feature 4',
         }
      ),
   }

   handler = sessionProfileCommand
   noOrDefaultHandler = handler

MirroringModelet.addCommandClass( NameProfileCmd )

# --------------------------------------------------------------------------------
# [ no | default ] monitor session NAME congestion
# --------------------------------------------------------------------------------
def sessionCongestionCommand( mode, args ):
   delete = CliCommand.isNoOrDefaultCmd( args )
   name = args[ 'NAME' ]
   setSessionCongestion( mode, name, delete )

class NameCongestionCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor session NAME congestion'
   noOrDefaultSyntax = syntax
   data = {
      'monitor': matcherMonitor,
      'session': nodeSession,
      'NAME': matcherSessionName,
      'congestion': matcherCongestion,
   }

   handler = sessionCongestionCommand
   noOrDefaultHandler = handler

MirroringModelet.addCommandClass( NameCongestionCmd )

# ----------------------------------------------------------
# Register Mirroring show commands into "show tech-support".
# ----------------------------------------------------------
# Timestamps are made up to maintain historical order within show tech-support
TechSupportCli.registerShowTechSupportCmd(
   '2010-01-01 00:12:00',
   cmds=[ 'show monitor session', 'show tap aggregation groups' ] )

def mirroringQuietDataLinkExplanation( intfName, mode ):
   for sessionStatus in gv.mirroringStatus.sessionStatus.values():
      operStatus = sessionStatus.targetOperStatus.get( intfName )
      if operStatus and operStatus.state == 'operStateActive':
         return ( 'monitoring', 'Mirroring destination', 'mirroring configuration' )
   return ( None, None, None )

IntfCli.quietDataLinkExplanationHook.addExtension(
      mirroringQuietDataLinkExplanation )

@Plugins.plugin( provides=( 'mirroring/config', ) )
def Plugin( em ):
   gv.mirroringConfig = ConfigMount.mount( em, "mirroring/config",
                                               "Mirroring::Config", "w" )
   gv.mirroringStatus = LazyMount.mount( em, "mirroring/status",
                                             "Mirroring::Status", "r" )
   gv.mirroringHwStatus = LazyMount.mount( em, "mirroring/hwstatus",
                                               "Mirroring::HwStatus", "r" )
   gv.mirroringHwCapability = LazyMount.mount( em, "mirroring/hwcapability",
                                                   "Mirroring::HwCapability", "r" )
   gv.hwCapability = LazyMount.mount( em, "routing/hardware/status",
                                          "Routing::Hardware::Status", "r" )
   gv.aclStatusDp = LazyMount.mount( em,
                                     "cell/%d/acl/status/dp" % Cell.cellId(),
                                     "Tac::Dir", "ri" )
   gv.aclConfig = LazyMount.mount( em, "acl/config/cli",
                                       "Acl::Input::Config", "r" )
   gv.mirroringHwConfig = LazyMount.mount( em, "mirroring/hwconfig",
                                               "Mirroring::Config", "r" )
   gv.vlanStatusDir = LazyMount.mount( em, "bridging/vlan/status",
                                           "Bridging::VlanStatusDir", "r" )
   gv.aclCounterConfig = LazyMount.mount( em, "mirroring/acl/counter/config",
                                          "Mirroring::AclCounterConfig", "w" )
   gv.aclCounterStatus = LazyMount.mount( em, "mirroring/acl/counter/status",
                                          "Mirroring::AclCounterStatus", "r" )
   gv.bridgingHwCapabilities = LazyMount.mount( em, 'bridging/hwcapabilities',
                                        'Bridging::HwCapabilities', 'r' )

   def mirrorNameFn( mode ):
      return gv.mirroringConfig.session

   def validateMirror( mode, sessionName ):
      config = gv.mirroringConfig.session.get( sessionName )
      if config and mode.session.secureMonitor() != config.secureMonitor:
         return str( CliParser.guardNotPermitted )
      return None

   AclCli.registerMirroringCallback( mirrorNameFn,
                                     validateMirror )
