#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-from-import

import BasicCli
import CliCommand
import CliMatcher
import ShowCommand
import CliPlugin.MirroringMonitorCli as MirroringMonitorCli
import CliPlugin.TapAggIntfCli as TapAggIntfCli
import CliPlugin.TapAggPmapCli as TapAggPmapCli
import CliPlugin.MirroringClassMapModeTapAgg as MirroringClassMapModeTapAgg
import CliPlugin.LagIntfCli as LagIntfCli
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.MirroringTapAggConfigMode as MirroringTapAggConfigMode
import CliPlugin.VlanCli as VlanCli
from CliPlugin.MirroringModels import MirrorSessions
from CliPlugin.PolicyMapModel import PolicyMapAllModel, ClassMapAllModel
from CliPlugin.TapAggModels import IntfCounterModel, ToolToTapMap, TapToToolMap, \
   TapAggGroups, TapAggNexthopGroups, TunnelCounterModel
from CliPlugin.MirroringModelets import nodeTapagg
from CliToken.Monitor import monitorMatcherForShow
from Intf.IntfRange import IntfRangeMatcher

matcherAggregation = CliMatcher.KeywordMatcher( 'aggregation',
   helpdesc='Show tap aggregation' )
matcherConfigured = CliMatcher.KeywordMatcher( 'configured',
   helpdesc='Show configured rather than active' )
matcherDetail = CliMatcher.KeywordMatcher( 'detail',
   helpdesc='More comprehensive output' )
matcherDetailMirroring = CliMatcher.KeywordMatcher( 'detail',
   helpdesc='Detailed mirroring session information' )
matcherNexthopGroups = CliCommand.guardedKeyword( 'nexthop-groups',
   helpdesc='Show tap aggregation nexthop groups',
   guard=TapAggIntfCli.guardDefaultNexthopGroupTapAgg )
matcherGroups = CliMatcher.KeywordMatcher( 'groups',
   helpdesc='Show tap aggregation groups' )
matcherMap = CliMatcher.KeywordMatcher( 'map',
   helpdesc='Show mappings between taps and tools' )
matcherPmapSummary = CliMatcher.KeywordMatcher( 'summary',
   helpdesc='Policy map summary' )
matcherSummaryMirroring = CliMatcher.KeywordMatcher( 'summary',
   helpdesc='Summary of mirroring session interfaces' )
matcherTap = CliMatcher.KeywordMatcher( 'tap',
   helpdesc='Show tap aggregation groups' )
matcherTapPort = CliMatcher.KeywordMatcher( 'tap',
   helpdesc='Show interface TAP information' )
matcherCounters = CliCommand.guardedKeyword( 'counters',
   helpdesc='Show tap aggregation counters',
   guard=TapAggIntfCli.counterAccessGuard )
matcherTunnel = CliCommand.guardedKeyword( 'tunnel',
   helpdesc='Count of tunnel packets',
   guard=TapAggIntfCli.tapTunnelCounterAccessGuard )

#--------------------------------------------------------------------------------
# show class-map type tapagg [ CMAPNAME ]
#--------------------------------------------------------------------------------
class ClassMapTypeTapaggCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show class-map type tapagg [ CMAPNAME ]'
   data = {
      'class-map': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'class-map',
         helpdesc='Show Class Map' ), guard=TapAggPmapCli.guardPMapTapAgg ),
      'type': MirroringTapAggConfigMode.matcherType,
      'tapagg': nodeTapagg,
      'CMAPNAME': CliMatcher.DynamicNameMatcher( TapAggPmapCli.getCMapNameRuleTapAgg,
         'Class Map Name' ),
   }
   handler = MirroringClassMapModeTapAgg.showClassMap
   cliModel = ClassMapAllModel

BasicCli.addShowCommandClass( ClassMapTypeTapaggCmd )

#--------------------------------------------------------------------------------
# show policy-map type tapagg [ PMAPNAME ] [ summary ]
#--------------------------------------------------------------------------------
class PolicyMapTypeTapaggCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show policy-map type tapagg [ PMAPNAME ] [ summary ]'
   data = {
      'policy-map': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'policy-map', helpdesc='Show Policy Map' ),
         guard=TapAggPmapCli.guardPMapTapAgg ),
      'type': MirroringTapAggConfigMode.matcherType,
      'tapagg': nodeTapagg,
      'summary': matcherPmapSummary,
      'PMAPNAME': CliMatcher.DynamicNameMatcher(
         TapAggPmapCli.getPMapNameRuleTapAgg, 'TapAgg Policy Map Name' ),
   }
   handler = TapAggPmapCli.PolicyMapModeTapAgg.showPolicyMap
   cliModel = PolicyMapAllModel

BasicCli.addShowCommandClass( PolicyMapTypeTapaggCmd )

#--------------------------------------------------------------------------------
# show tap aggregation counters default-forwarding
#--------------------------------------------------------------------------------
class TapAggregationCountersDefaultForwardingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show tap aggregation counters default-forwarding'
   data = {
      'tap': matcherTap,
      'aggregation': matcherAggregation,
      'counters': matcherCounters,
      'default-forwarding': 'Count of packets forwarded '
                            'using the default forwarding action',
   }
   handler = TapAggIntfCli.doShowAllTapAggCounters
   cliModel = IntfCounterModel

BasicCli.addShowCommandClass( TapAggregationCountersDefaultForwardingCmd )

#--------------------------------------------------------------------------------
# show tap aggregation counters tunnel termination gre
#--------------------------------------------------------------------------------
class TapAggregationCountersGreTunnelCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show tap aggregation counters tunnel termination gre'
   data = {
      'tap': matcherTap,
      'aggregation': matcherAggregation,
      'counters': matcherCounters,
      'tunnel': matcherTunnel,
      'termination': 'Tunnel termination',
      'gre': 'GRE tunnel termination',
   }
   handler = TapAggIntfCli.doShowAllTapTunnelCounters
   cliModel = TunnelCounterModel

BasicCli.addShowCommandClass( TapAggregationCountersGreTunnelCmd )

#--------------------------------------------------------------------------------
# show tap aggregation groups map TAPPORT [ configured ]
#--------------------------------------------------------------------------------
class TapAggregationGroupsMapTapCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show tap aggregation groups map TAPPORT [ configured ]'
   data = {
      'tap': matcherTap,
      'aggregation': matcherAggregation,
      'groups': matcherGroups,
      'map': matcherMap,
      'TAPPORT': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'tap',
         helpdesc='Show interface TAP information' ),
         guard=TapAggIntfCli.tapaggGuard ),
      'configured': matcherConfigured,
   }
   handler = TapAggIntfCli.doShowTapAggMap
   cliModel = TapToToolMap

BasicCli.addShowCommandClass( TapAggregationGroupsMapTapCmd )

#--------------------------------------------------------------------------------
# show tap aggregation groups map TOOLPORT [ configured ]
#--------------------------------------------------------------------------------
class TapAggregationGroupsMapToolCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show tap aggregation groups map TOOLPORT [ configured ]'
   data = {
      'tap': matcherTap,
      'aggregation': matcherAggregation,
      'groups': matcherGroups,
      'map': matcherMap,
      'TOOLPORT': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'tool',
         helpdesc='Show interface TOOL information' ),
         guard=TapAggIntfCli.tapaggGuard ),
      'configured': matcherConfigured,
   }
   handler = TapAggIntfCli.doShowTapAggMap
   cliModel = ToolToTapMap

BasicCli.addShowCommandClass( TapAggregationGroupsMapToolCmd )

#--------------------------------------------------------------------------------
# show monitor session [ SESSNAME ]
#              [ ( detail [ access-list ] )
#              | ( summary [ ( source ( SRCINTF | ( vlan VLANS ) ) )
#                          | ( destination ( cpu
#                                          | ETHINTF
#                                          | LAGINTF
#                                          | GREINTF
#                                          | MIRRORTUNNELINTF ) ) ] ) ]
#--------------------------------------------------------------------------------
class MonitorSessionCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show monitor session [ SESSNAME ]'
              '[ ( detail [ access-list ] )'
              '| ( summary [ ( source ( SRCINTF | ( vlan VLANS ) ) )'
              '            | ( destination ( cpu'
              '                            | ETHINTF'
              '                            | LAGINTF'
              '                            | GREINTF'
              '                            | MIRRORTUNNELINTF ) ) ] ) ]' )

   data = {
      'monitor': monitorMatcherForShow,
      'session': 'Mirroring session information',
      'SESSNAME': MirroringMonitorCli.matcherSessionName,
      'detail': matcherDetailMirroring,
      'access-list': 'Detailed session access-list information',
      'summary': matcherSummaryMirroring,
      'source': 'Mirroring source configuration commands',
      'SRCINTF': MirroringMonitorCli.matcherSrcIntfTypes,
      'vlan': MirroringMonitorCli.nodeVlan,
      'VLANS': VlanCli.vlanIdRangeMatcher,
      'destination': 'Mirroring destination configuration commands',
      'cpu': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'cpu',
         helpdesc='CPU port(s)' ),
         guard=MirroringMonitorCli.mirroringCpuSupportedGuard ),
      'ETHINTF': IntfRangeMatcher(
         explicitIntfTypes=( EthIntfCli.EthPhyAutoIntfType, ) ),
      'LAGINTF': CliCommand.Node( matcher=IntfRangeMatcher(
         explicitIntfTypes=( LagIntfCli.LagAutoIntfType, ) ),
         guard=MirroringMonitorCli.mirroringDestLagSupportedGuard ),
      'GREINTF': CliCommand.Node( matcher=IntfRangeMatcher(
         explicitIntfTypes=( MirroringMonitorCli.GreAutoIntfType, ) ),
         guard=MirroringMonitorCli.mirroringDestinationGreSupported,
         hidden=True ),
      'MIRRORTUNNELINTF': CliCommand.Node( matcher=IntfRangeMatcher(
         explicitIntfTypes=( MirroringMonitorCli.MirrorTunnelAutoIntfType, ) ),
         guard=MirroringMonitorCli.mirroringDestinationGreSupported )
   }
   handler = MirroringMonitorCli.doShowMirrorSessions
   cliModel = MirrorSessions

BasicCli.addShowCommandClass( MonitorSessionCmd )

#--------------------------------------------------------------------------------
# show tap aggregation groups [ { GROUPNAME } ] [ detail ]
#--------------------------------------------------------------------------------
class TapAggregationGroupsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show tap aggregation groups [ { GROUPNAME } ] [ detail ]'
   data = {
      'tap': matcherTap,
      'aggregation': matcherAggregation,
      'groups': matcherGroups,
      'detail': matcherDetail,
      'GROUPNAME': TapAggIntfCli.matcherTapAggGroupNameDynamic,
   }
   handler = TapAggIntfCli.doShowTapAggGroups
   cliModel = TapAggGroups

BasicCli.addShowCommandClass( TapAggregationGroupsCmd )

#--------------------------------------------------------------------------------
# show tap aggregation nexthop-groups [ { NHGNAME } ]
#--------------------------------------------------------------------------------
class TapAggregationNexthopGroupsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show tap aggregation nexthop-groups [ { NHGNAME } ]'
   data = {
      'tap': matcherTap,
      'aggregation': matcherAggregation,
      'nexthop-groups': matcherNexthopGroups,
      'NHGNAME': TapAggIntfCli.matcherNexthopGroupName,
   }
   handler = TapAggIntfCli.doShowTapAggNexthopGroups
   cliModel = TapAggNexthopGroups

BasicCli.addShowCommandClass( TapAggregationNexthopGroupsCmd )
