#!/usr/bin/env python3
# Copyright (c) 2009-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=relative-beyond-top-level

from __future__ import absolute_import, division, print_function
import re
import six
from six.moves import filter
from six.moves import zip
import Arnet
import CliParser
from CliPlugin.BridgingCliModel import UnicastMacAddressTable
import CliPlugin.IntfModel as IntfModel # pylint: disable=consider-using-from-import
import CliPlugin.LagCliLib as LagCliLib # pylint: disable=consider-using-from-import
from CliPlugin.MlagModel import MlagLacpSysId
import CliPlugin.VlanCli as VlanCli # pylint: disable=consider-using-from-import
import Ethernet
import LazyMount
from .MlagModel import Mlag, negStatusStrToEnum, Interfaces, ConfigSanity, \
    ParamValue, InterfaceValue, GlobalFeatureParam, IntfFeatureParam, SubInterfaces
from MlagMountHelper import MlagStatusLazyMounter
from MlagShared import heartbeatTimeout, NEG_STATUS_CONNECTED
import SmashLazyMount
from StpCliUtil import parseVidToMstiMap, inverseVlanMap
import Tac
from Vlan import vlanSetToCanonicalString
import Toggles.MlagToggleLib
from Toggles.StpToggleLib import toggleStpPriorityConfigSanityEnabled

mlagConfig = None
mlagStatus = None
mlagHwStatus = None
mlagHostTable = None
mlagProtoStatus = None
ethIntfConfig = None
ethIntfStatus = None
subIntfConfig = None
subIntfStatus = None
lagIntfStatusDir = None
configCheckDir = None
bridgingStatus = None
aclRequestToPeer = None
aclRequestFromPeer = None
aclResponseToPeer = None
aclResponseFromPeer = None
aclTimeoutStatus = None

# mlagConfigSanityConsistencyCheckOverride allows customized mlag config
# consistency check function to be installed to override default behavior. The
# override is key'ed by config sanity check paramKey with custom function being
# the value. The function will take paramEntry and return True (consistent) or
# False (inconsistent)
mlagConfigSanityConsistencyCheckOverride = {}

# Function to split the mstConfigSpec string-representation into its individual parts
# The mstConfigSpec comes across the wire looking like this:

# Value('Stp::MstConfigSpec', ** {'regionId': 'Bill', 'configRevision': 0,
# 'vidToMstiMap': 00 01 00 01})

# We need each of these values to be separated, so we do some string parsing
# to split each into its respective attribute/value pair. Note that using Python's
# json parser does not work because the json parser expects attributes/values to be
# double-quoted, and what we have here is single-quoted.
def extractMstConfigSpecAttrs( mstConfigSpecStr ):
   if not mstConfigSpecStr:
      return ( '', '', '' )
   regionKey = "'regionId': '"
   regionStart = mstConfigSpecStr.find( regionKey ) + len( regionKey )
   regionEnd = mstConfigSpecStr.find( "', '", regionStart )
   regionStr = mstConfigSpecStr[ regionStart:regionEnd ]
   # To remove escapes from string
   regionStr = six.ensure_binary( regionStr ).decode( 'unicode_escape' )

   revisionKey = " 'configRevision': "
   revisionStart = mstConfigSpecStr.find( revisionKey ) + len( revisionKey )
   revisionEnd = mstConfigSpecStr.find( ",", revisionStart )
   revisionStr = mstConfigSpecStr[ revisionStart:revisionEnd ]
   revisionStr = six.ensure_binary( revisionStr ).decode( 'unicode_escape' )

   # If one peer is running EOS version Tokyo and the other is running any version
   # newer than Tokyo, there will be a format mismatch for the vidToMstiMap.
   # Previously, vidToMstiMap was a string, so we can detect the old format by
   # checking for a single-quote. We will handle this format mismatch by ignoring
   # differences to vidToMstiMap
   if " 'vidToMstiMap': '" in mstConfigSpecStr:
      return ( regionStr, revisionStr, '' )
   mapKey = " 'vidToMstiMap': "
   mapStart = mstConfigSpecStr.find( mapKey ) + len( mapKey )
   # In case another attribute is added to mstConfigSpec, we look for a
   # curly-brace or comma to end the vidToMstiMap
   mapEnd = re.search( "(?=}|,)", mstConfigSpecStr[ mapStart: ] ).end()
   mapStr = mstConfigSpecStr[ mapStart:mapStart + mapEnd ]
   return ( regionStr, revisionStr, mapStr )

def mstConfigSpecCheck( entry ):
   ( localRegion, localRev, localMap ) = extractMstConfigSpecAttrs( entry.localVal )
   ( peerRegion, peerRev, peerMap ) = extractMstConfigSpecAttrs( entry.peerVal )
   return ( localRegion == peerRegion ) and ( localRev == peerRev ) and \
          ( localMap == peerMap )

def lacpPortIdRangeCheck( entry ):
   complete = bool( entry.localVal ) and bool( entry.peerVal )
   if complete:
      # both local and peer values are known. go ahead and check for overlap
      default = Tac.Value( "Lacp::PortIdRange" )
      local = Tac.Value( "Lacp::PortIdRange" )
      peer = Tac.Value( "Lacp::PortIdRange" )
      local.stringValue = entry.localVal
      peer.stringValue = entry.peerVal
      return( not local.overlap( peer ) or
              ( local == default and peer == default ) )
   else:
      # incomplete entry; still consistent if both localVal and peerVal are ""
      # or None
      return not bool( entry.localVal ) and not bool( entry.peerVal )

mlagConfigSanityConsistencyCheckOverride[ 'lacp port-id range' ] = \
      lacpPortIdRangeCheck
mlagConfigSanityConsistencyCheckOverride[ 'mst-config-spec' ] = \
      mstConfigSpecCheck

def mlagSupportedGuard( mode, token ):
   if mlagHwStatus.mlagSupported:
      return None
   return CliParser.guardNotThisPlatform

#-------------------------------------------------------------------------------
# Hook for 'show lacp sys-id' to display MLAG sys-id
#-------------------------------------------------------------------------------
def showLacpSysIdForMlag( lc, brief ):
   mlagSystemId = mlagStatus.systemId
   if mlagSystemId == '00:00:00:00:00:00':
      return None
   return MlagLacpSysId( systemId=mlagSystemId,
                         priority=lc.priority,
                         _brief=brief, _internal=False )

LagCliLib.showLacpSysIdHook.addExtension( showLacpSysIdForMlag )

#-------------------------------------------------------------------------------
# Hook for 'show lacp internal' to display MLAG sys-id
#-------------------------------------------------------------------------------
# pylint: disable-next=inconsistent-return-statements
def showLacpInternalSysIdForMlag( lc ):
   mlagSystemId = mlagStatus.systemId
   if mlagSystemId == '00:00:00:00:00:00':
      return
   return MlagLacpSysId( systemId=mlagSystemId,
                         priority=lc.priority,
                         _brief=True, _internal=True )

LagCliLib.showLacpInternalSysIdHook.addExtension( showLacpInternalSysIdForMlag )

#-------------------------------------------------------------------------------
# Hook for 'show lacp' commands to annotate port-channels part of an MLAG
#-------------------------------------------------------------------------------
def showLacpAnnotateForMlag():
   msg = "Only local interfaces for MLAGs are displayed. " \
       "Connect to the peer to see the state for peer interfaces."
   def isMlagPortChannel( portChannel ):
      if portChannel not in mlagStatus.intfStatus:
         return False
      if mlagStatus.intfStatus[ portChannel ].status == 'linkDisabled':
         return False
      return True

   return ( msg, isMlagPortChannel )

LagCliLib.registerLacpAnnotateHook( showLacpAnnotateForMlag )

# Summarize based on port category. Takes Mlag::Status and returns
# counts of Mlag::IntfStatus categorized.
def portCategorySummary( status ):
   disabledPorts = 0
   configuredPorts = 0
   inactivePorts = 0
   activePartialLocalUpPorts = 0
   activePartialRemoteUpPorts = 0
   activeFullPorts = 0
   if status.mlagState not in ('primary', 'secondary'):
      disabledPorts = len( status.intfStatus ) 
   else:
      for port in six.itervalues( status.intfStatus ):
         if port.status == 'linkConfigured':
            configuredPorts += 1
         elif port.status == 'linkActive':
            activeFullPorts += 1
         elif port.status == 'linkInactive':
            if port.localLinkStatus == 'linkUp':
               activePartialLocalUpPorts += 1
            elif port.peerLinkStatus == 'linkUp':
               activePartialRemoteUpPorts += 1
            else:
               inactivePorts += 1
   return ( disabledPorts, configuredPorts, inactivePorts,
            activePartialLocalUpPorts, activePartialRemoteUpPorts,
            activeFullPorts )

def showMlag( mode, args ):
   # Force mounting of the LazyMount._Proxy objects before grabbing
   # the activity lock, or a deadlock will result between the blocking
   # mount and the activity thread.
   # pylint: disable-msg=W0212
   mlagConfig._mount()
   mlagStatus._mount()
   mlagHwStatus._mount()
   mlagProtoStatus._mount()
   configCheckDir._mount()
   return doShowMlag( mode, args )

@Tac.withActivityLock
def doShowMlag( mode, args ):
   mlag = Mlag()
   cfg = mlagConfig
   hwst = mlagHwStatus
   configSanity = configCheckDir
   if cfg.domainId:
      mlag.domainId = cfg.domainId
   mlag.localInterface = cfg.localIntfId
   if cfg.peerAddress != '0.0.0.0':
      mlag.peerAddress = cfg.peerAddress
   mlag.peerLink = cfg.peerLinkIntfId
   if cfg.heartbeatPeerAddress.address != '0.0.0.0':
      mlag.heartbeatPeerAddress = cfg.heartbeatPeerAddress.address.stringValue
      if cfg.heartbeatPeerAddress.vrf:
         mlag.heartbeatPeerVrf = cfg.heartbeatPeerAddress.vrf
   reloadDelay = cfg.reloadDelay
   reloadDelayConfigured = ( cfg.reloadDelay.reloadDelayType == 
                             "reloadDelayConfigured" )
   mlag.reloadDelay = -1 if reloadDelay.delay == Tac.endOfTime \
         else int( reloadDelay.delay ) if reloadDelayConfigured \
         else int( hwst.reloadDelay )
   if cfg.reloadDelayNonMlag.reloadDelayType == "reloadDelayConfigured":
      mlag.reloadDelayNonMlag = -1 if cfg.reloadDelayNonMlag.delay == Tac.endOfTime \
       else int( cfg.reloadDelayNonMlag.delay )
   else:
      mlag.reloadDelayNonMlag = mlag.reloadDelay

   status = mlagStatus
   if status.mlagState in ( 'primary', 'secondary' ):
      mlag.state = 'active'
   else:
      mlag.state = status.mlagState

   def _configSanityValue( cs ):
      for pluginEntry in six.itervalues( cs.configPluginDir ):
         for paramKey, paramEntry in six.iteritems( pluginEntry.configGlobalDiff ):
            func = mlagConfigSanityConsistencyCheckOverride.get( paramKey,
                  lambda paramEntry: paramEntry.localVal == paramEntry.peerVal )
            inconsistent = not func( paramEntry )
            if inconsistent:
               return 'inconsistent'
         for intfTable in six.itervalues( pluginEntry.configIntfDir ):
            for intfEntry in six.itervalues( intfTable.configIntfDiff ):
               if intfEntry.localVal != intfEntry.peerVal:
                  return 'inconsistent'
      return 'consistent'

   if mlag.state == 'active':
      mlag.configSanity = _configSanityValue( configSanity )
   else:
      mlag.configSanity = None

   if status.negotiationStatus:
      mlag.negStatus = negStatusStrToEnum( status.negotiationStatus )
   def _intfOperStatusEnum( s ):
      return IntfModel.intfOperStatusToEnum( s.operStatus ) if s else None
   if status.peerLinkIntf:
      mlag.peerLinkStatus = _intfOperStatusEnum( status.peerLinkIntf )
   if status.localInterface:
      mlag.localIntfStatus = _intfOperStatusEnum( status.localInterface )
   if status.systemId != '00:00:00:00:00:00':
      mlag.systemId = status.systemId
   ( disabledPorts, configuredPorts, inactivePorts,
     activePartialLocalUpPorts, activePartialRemoteUpPorts,
     activeFullPorts ) = portCategorySummary( status )
   mlag.mlagPorts["Disabled"] = disabledPorts
   mlag.mlagPorts["Configured"] = configuredPorts
   mlag.mlagPorts["Inactive"] = inactivePorts
   mlag.mlagPorts["Active-partial"] = \
       activePartialLocalUpPorts + activePartialRemoteUpPorts
   mlag.mlagPorts["Active-full"] = activeFullPorts

   proto = mlagProtoStatus
   mlag.portsErrdisabled = status.mlagState != 'primary' \
       and proto.portsErrdisabled \
       and proto.portsErrdisabledWhen != Tac.endOfTime
   # portsErrdisabled when set to True in mlagState which is not primary
   # requires a portsErrdisabledWhen which is not endOfTime
   if mlag.portsErrdisabled:
      mlag.portsErrdisabledTime = proto.portsErrdisabledWhen
   dpState = status.dualPrimaryDetectionState
   if dpState == 'dpConfigured':
      mlag.dualPrimaryDetectionState = 'configured'
   elif dpState == 'dpRunning':
      mlag.dualPrimaryDetectionState = 'running'
   elif dpState == 'dpDetected':
      mlag.dualPrimaryDetectionState = 'detected'
   elif dpState == 'dpDisabled':
      mlag.dualPrimaryDetectionState = 'disabled'
   mlag.dualPrimaryPortsErrdisabled = status.dualPrimaryIntfErrdisable
   if cfg.dualPrimaryDetectionDelay:
      mlag.dualPrimaryMlagRecoveryDelay = int( cfg.dualPrimaryMlagRecoveryDelay )
      mlag.dualPrimaryNonMlagRecoveryDelay = \
                                          int( cfg.dualPrimaryNonMlagRecoveryDelay )
   if mlag.dualPrimaryPortsErrdisabled and \
         status.dualPrimaryIntfErrdisableWhen != Tac.endOfTime:
      mlag.dualPrimaryPortsErrdisabledTime = status.dualPrimaryIntfErrdisableWhen

   if 'detail' not in args:
      return mlag

   mlag.detail = Mlag.Detail()
   d = mlag.detail
   d.mlagState = status.mlagState
   d.peerMlagState = status.peerMlagState
   d.stateChanges = proto.stateChanges
   if proto.lastStateChangeTime:
      d.lastStateChangeTime = proto.lastStateChangeTime
   d.mlagHwReady = mlagHwStatus.mlagHwReady

   def getFailoverCauseListFromFailoverBitmask( fcBitmask ):
      '''
      This function converts the failoverCause attribute ( which is stored as a
      bitmask ) to a list of actual failover causes.
      '''
      fcMask = Tac.Value( 'Mlag::FailoverCauseBitMask' )
      fcName = Tac.Value( 'Mlag::FailoverCauseName' )
      bitToCauseDict = dict( # pylint: disable=consider-using-dict-comprehension
         [ ( fcMask.__getattribute__( mask ), fcName.__getattribute__( name ) )
           for mask, name in zip ( fcMask.attributes, fcName.attributes ) ] )
      failoverCauseList = []
      # If no bit is set in Bitmask, then failover cause is simply Unknown.
      if fcBitmask == fcMask.unknown:
         failoverCauseList.append( fcName.unknown )
         return failoverCauseList

      for bit, cause in bitToCauseDict.items():
         # If "bit" is set in fcBitmask, append the corresponding cause into the list
         if ( fcBitmask & bit ): # pylint: disable=superfluous-parens
            failoverCauseList.append( cause )
      return failoverCauseList

   d.failover = status.failover
   d.failoverCauseList = getFailoverCauseListFromFailoverBitmask(
      status.failoverCause )
   d.failoverInitiated = status.failoverInitiated
   if proto.lastFailoverChangeTime:
      d.lastFailoverChangeTime = proto.lastFailoverChangeTime
   d.secondaryFromFailover = proto.secondaryFromFailover
   d.primaryPriority = cfg.primaryPriority
   # pylint: disable-next=protected-access
   d._primaryPriorityDefault = cfg.primaryPriorityDefault
   d.peerPrimaryPriority = proto.peerPrimaryPriority
   d.peerMacAddress = status.peerMacAddr
   d.peerMacRoutingSupported = mlagHwStatus.mlagPeerMacRoutingSupported 
   d.peerPortsErrdisabled = \
       status.mlagState == 'primary' and proto.portsErrdisabled
   d.lacpStandby = cfg.lacpStandby
   d.heartbeatInterval = cfg.heartbeatInterval
   d.effectiveHeartbeatInterval = proto.effectiveHeartbeatInterval
   d.heartbeatTimeout = heartbeatTimeout( cfg, d.effectiveHeartbeatInterval )
   if proto.lastHeartbeatTimeout:
      d.lastHeartbeatTimeout = proto.lastHeartbeatTimeout
   d.heartbeatTimeoutsSinceReboot = proto.heartbeatTimeoutsSinceReboot
   d.udpHeartbeatAlive = proto.udpHeartbeatAlive
   d.udpHeartbeatsReceived = proto.udpHeartbeatsReceived
   d.udpHeartbeatsSent = proto.udpHeartbeatsSent
   if proto.peerMonotonicOffsetValid:
      d.peerMonotonicClockOffset = proto.peerMonotonicOffset
   d.enabled = proto.running
   d.mountChanges = proto.mountChanges
   if cfg.dualPrimaryDetectionDelay:
      d.dualPrimaryDetectionDelay = int( cfg.dualPrimaryDetectionDelay )
      if cfg.dualPrimaryAction == 'dualPrimaryActionNone':
         d.dualPrimaryAction = 'none'
      elif cfg.dualPrimaryAction == 'dualPrimaryActionErrdisableAllInterfaces':
         d.dualPrimaryAction = 'errdisableAllInterfaces'
   d.fastMacRedirectionEnabled = status.fastMacRedirection
   d.mlagIntfEgressAclInterlockInactiveReason = \
      status.mlagIntfEgressAclInterlockInactiveReason
   if not mlagHwStatus.egressFilterInterlockConfigurable:
      d.mlagIntfEgressAclInterlockState = 'unsupported'
   elif status.mlagIntfEgressAclInterlockEnabled:
      d.mlagIntfEgressAclInterlockState = 'active'
   elif cfg.mlagIntfEgressAclInterlockConfigured:
      d.mlagIntfEgressAclInterlockState = 'inactive'
   else:
      d.mlagIntfEgressAclInterlockState = 'unconfigured'
   return mlag

#-------------------------------------------------------------------------------
# show mlag interfaces [ number|range ] [ states <states> ] [ detail ]
# show mlag interfaces [ number|range ] members
# show mlag interfaces filter egress
# show mlag subinterfaces
# show mlag interfaces ignored
#-------------------------------------------------------------------------------
def mlagIntfs( mlagList ):
   good = []
   bad = []
   for ( intfId, mlagId ) in mlagConfig.intfConfig.items():
      if mlagList is not None:
         # An mlag range was specified. Either skip this mlag or
         # remove it from the list.
         if int( mlagId ) not in mlagList: # pylint: disable=no-else-continue
            continue
         else:
            mlagList.remove( int( mlagId ) )
      intf = Interfaces.MlagInterface()
      intf.localInterface = intfId
      good.append( ( mlagId, intf ) )
   # Assemble the MLAGs that don't exist but were requested.
   if  mlagList is not None:
      for mid in mlagList:
         intf = Interfaces.MlagInterface()
         bad.append( ( str( mid ), intf ) )
   return good, bad

def showMlagInt( mode, args ):
   mlagList = list( args[ 'MLAG_IDS' ].values() ) if 'MLAG_IDS' in args else None
   filterStates = []
   if 'states' in args:
      filterStates = args[ 'STATES' ]
   if filterStates:
      filterStates = modifiedFilterStates( filterStates )
   intfs = Interfaces()
   goodIntfs, badIntfs = mlagIntfs( mlagList )
   for mlagId, intf in goodIntfs:
      intfStatus = mlagStatus.intfStatus.get( intf.localInterface )
      statusDetail = mlagStatusStrDetail( intfStatus )
      if filterStates and statusDetail not in filterStates:
         continue
      intf.members = None
      localConfig = ethIntfConfig.intfConfig.get( intf.localInterface )
      if localConfig and localConfig.description:
         intf.localInterfaceDescription = localConfig.description
      intf.localInterfaceStatus = _intfStatus( intfStatus, 'localLinkStatus' )
      if intfStatus:
         intf.peerInterface = intfStatus.peerIntfName
      intf.peerInterfaceStatus = _intfStatus( intfStatus, 'peerLinkStatus' )
      if intf.localInterface in mlagStatus.ignoredIntfConfig:
         intf.status = 'disabled-ignored'
      else:
         intf.status = mlagStatusStr( intfStatus )
      if 'detail' in args:
         idet = Interfaces.MlagInterface.Detail()
         if localConfig:
            idet.localInterfaceEnabled = 'ena' if localConfig.enabled else 'dis'
         if intfStatus:
            idet.peerInterfaceEnabled = \
                { 'unknownEnabledState' : None, \
                     'enabled' : 'ena', \
                     'shutdown' : 'dis' }[ intfStatus.peerIntfEnabledState ]
            idet.lastChangeTime = intfStatus.lastChangeTime
            idet.changeCount = intfStatus.changeCount
         intf.detail = idet
      intfs.interfaces[ mlagId ] = intf
   for mlagId, intf in badIntfs:
      intf.members = None
      intfs.interfaces[ mlagId ] = intf
   return intfs

def showMlagSubIntf( mode, args ):
   subIntfs = SubInterfaces()
   mlagList = list( args[ 'MLAG_IDS' ].values() ) if 'MLAG_IDS' in args else None
   goodIntfs, badIntfs = mlagIntfs( mlagList )
   for mlagId, parentIntf in goodIntfs:
      subIntfMapping = mlagStatus.mlagIntfToSubIntf.get( parentIntf.localInterface )
      if not subIntfMapping:
         continue
      intfMap = SubInterfaces.MlagToSubIntf()
      for subIntfId in subIntfMapping.subIntf.keys():
         mlagSubIntf = SubInterfaces.MlagToSubIntf.MlagSubIntf()
         localConfig = subIntfConfig.intfConfig.get( subIntfId )
         if localConfig and localConfig.description:
            mlagSubIntf.desc = localConfig.description
         else:
            mlagSubIntf.desc = None
         intfStatus = subIntfStatus.intfStatus.get( subIntfId )
         status = _intfStatus( intfStatus, 'linkStatus' )
         mlagSubIntf.localStatus = status
         intfMap.subInterfaces[ subIntfId ] = mlagSubIntf
      subIntfs.mlags[ int( mlagId ) ] = intfMap

   for mlagId, _ in badIntfs:
      subIntfs.badInterfaces.append( int( mlagId ) )
   
   return subIntfs

def _intfStatus( linkStatus, attr ):
   if not linkStatus:
      return 'unknown'
   else:
      return getattr( linkStatus, attr ).removeprefix( 'link' ).lower()

def showMlagIntIgnored( mode, args ):
   intfs = Interfaces()
   mlagList = list( args[ 'MLAG_IDS' ].values() ) if 'MLAG_IDS' in args else None
   goodIntfs, badIntfs = mlagIntfs( mlagList )

   def _ignoreReasonStr( ignoreReason ):
      if ignoreReason == 'noSwitchport':
         return 'not-a-switchport'
      else:
         return 'n/a'

   for mlagId, intf in goodIntfs:
      ignoreReason = mlagStatus.ignoredIntfConfig.get( intf.localInterface )
      if ignoreReason:
         intf.ignoreReason = _ignoreReasonStr( ignoreReason )
         intf.members = None
         intfs.interfaces[ mlagId ] = intf

   for mlagId, intf in badIntfs:
      intf.members = None
      intfs.interfaces[ mlagId ] = intf
   
   return intfs

def mlagStatusStr( intfStatus ):
   string = mlagStatusStrDetail( intfStatus )
   if string == 'active-partial-local-up':
      return 'active-partial'
   elif string == 'active-partial-remote-up':
      return 'active-partial'
   else:
      return string

def mlagStatusStrDetail( intfStatus ):
   if not intfStatus:
      return "n/a"
   if intfStatus.status == 'linkActive':
      return 'active-full'
   if intfStatus.status == 'linkInactive':
      if intfStatus.localLinkStatus == 'linkUp':
         return 'active-partial-local-up'
      elif intfStatus.peerLinkStatus == 'linkUp':
         return 'active-partial-remote-up'
      else:
         return 'inactive'
   return intfStatus.status[ 4: ].lower()

def modifiedFilterStates( filterStates ):
   if 'active-partial' in filterStates:
      filterStates.remove( 'active-partial' )
      filterStates.append( 'active-partial-local-up' )
      filterStates.append( 'active-partial-remote-up' )
   return filterStates

def showMlagMembers( mode, args ):
   mlagList = list( args[ 'MLAG_IDS' ].values() ) if 'MLAG_IDS' in args else None
   elisd = lagIntfStatusDir
   intfs = Interfaces()
   goodIntfs, badIntfs = mlagIntfs( mlagList )
   for mlagId, intf in goodIntfs:
      elis = elisd.intfStatus.get( intf.localInterface )
      if( elis and elis.member ):
         for am in Arnet.sortIntf( elis.member ):
            intf.members.append( am )
      intfs.interfaces[ mlagId ] = intf
   for mlagId, intf in badIntfs:
      intfs.interfaces[ mlagId ] = intf
   return intfs

def aclProtocolStateStr( intfId ):
   reqToPeer = aclRequestToPeer.installRequest
   reqFromPeer = aclRequestFromPeer.installRequest
   respToPeer = aclResponseToPeer.installResponse
   respFromPeer = aclResponseFromPeer.installResponse
   if intfId not in reqToPeer:
      localState = "n/a"
   elif intfId not in respFromPeer or reqToPeer[ intfId ] != respFromPeer[ intfId ]:
      localState = "pending"
      if intfId in aclTimeoutStatus.aclIntfTimeoutStatus and \
         aclTimeoutStatus.aclIntfTimeoutStatus[ intfId ].inTimeout:
         localState = "timeout"
   else:
      assert reqToPeer[ intfId ] == respFromPeer[ intfId ]
      localState = "installed"

   if intfId not in reqFromPeer:
      peerState = "n/a"
   elif intfId not in respToPeer or reqFromPeer[ intfId ] != respToPeer[ intfId ]:
      peerState = "pending"
   else:
      assert reqFromPeer[ intfId ] == respToPeer[ intfId ]
      peerState = "installed"
   return localState, peerState

def showMlagFilters( mode, args ):
   mlagList = list( args[ 'MLAG_IDS' ].values() ) if 'MLAG_IDS' in args else None
   intfs = Interfaces()
   if not mlagStatus.mlagIntfEgressAclInterlockEnabled:
      intfs.interfaces = dict() # pylint: disable=use-dict-literal
      intfs.mlagIntfEgressAclInterlockEnabled = False
      return intfs
   goodIntfs, badIntfs = mlagIntfs( mlagList )
   for mlagId, intf in goodIntfs:
      intfId = intf.localInterface
      intf.localAclProtocolState, intf.peerAclProtocolState = \
         aclProtocolStateStr( intfId )
      if intfId in aclTimeoutStatus.aclIntfTimeoutStatus:
         intf.timeoutCount = \
            aclTimeoutStatus.aclIntfTimeoutStatus[ intfId ].timeoutCount
         intf.lastTimeoutTime = \
            aclTimeoutStatus.aclIntfTimeoutStatus[ intfId ].lastTimeoutTime
      intf.members = None
      intfs.interfaces[ mlagId ] = intf
   for mlagId, intf in badIntfs:
      intf.members = None
      intfs.interfaces[ mlagId ] = intf
   return intfs

#-------------------------------------------------------------------------------
# show mac-address-table mlag-peer ...
#-------------------------------------------------------------------------------
def decodeIntf( intf ):
   # In remote Mac entries, the 'intf' field encodes the tunnel interface and the
   # remote VTEP IP address.
   pos = intf.find( ':' )
   if pos == -1:
      return None

   return intf[ : pos ]

def decodePeerMacAddr( addr ):
   if addr.entryType in [ 'peerConfiguredRemoteMac', 'peerLearnedRemoteMac',
                          'peerReceivedRemoteMac', 'peerEvpnRemoteMac' ]:
      _intf = decodeIntf( addr.intf )
   else:
      _intf = addr.intf
   moves = None
   lastMove = None
   if addr.entryType in [ 'peerStaticMac', 'peerConfiguredRemoteMac',
                          'peerAuthenticatedMac' ]:
      typ = 'static'
   elif addr.entryType in [ 'peerDynamicMac', 'peerLearnedRemoteMac',
                            'peerReceivedRemoteMac' ]:
      if not _intf:
         return ( None, None, None, None )
      typ = 'dynamic'
      moves = addr.moves
      lastMove = addr.lastMoveTime
   else:
      raise ValueError( "invalid address type: " + repr( addr ) )

   return ( _intf, typ, moves, lastMove)

def showMacTable( mode, args ):
   addrTypeFilter = args.get( 'ENTRY_TYPE' )
   macAddr = args.get( 'MACADDR' )
   vlanId = args.get( 'VLANID' )
   intfNames = set( args.get( 'INTFS', [] ) )

   if macAddr:
      macAddr = Ethernet.convertMacAddrToCanonical( macAddr )

   def _filterMacAddr( macEntry ): # pylint: disable=inconsistent-return-statements
      if macEntry.entryType in [ 'peerConfiguredRemoteMac', 'peerLearnedRemoteMac',
                                 'peerReceivedRemoteMac', 'peerEvpnRemoteMac' ]:
         _intf = decodeIntf( macEntry.intf )
         if not _intf:
            return False
      else:
         _intf = macEntry.intf
      if macAddr:
         addrStr = Ethernet.convertMacAddrToCanonical(
            macEntry.hostKey.address )
         if addrStr != macAddr:
            return False
      if vlanId:
         if VlanCli.Vlan( macEntry.hostKey.vlanId ) != vlanId:
            return False
      if intfNames and \
             _intf not in intfNames:
         return False
      if addrTypeFilter is not None:
         if addrTypeFilter == 'static':
            if macEntry.entryType in [ 'peerStaticMac', 'peerConfiguredRemoteMac',
                                       'peerAuthenticatedMac' ]:
               return True
         elif addrTypeFilter == 'unicast':
            return Ethernet.isUnicast( macEntry.hostKey.address )
         else:
            if macEntry.entryType in [ 'peerDynamicMac', 'peerLearnedRemoteMac' ]:
               return True
      else:
         return True

   hostTable = mlagHostTable
   addresses = list( filter( _filterMacAddr, hostTable.hostEntry.values() ) )
   table = UnicastMacAddressTable()
   for addr in sorted( addresses, key=lambda x: ( x.hostKey.vlanId,
                                                  x.hostKey.address ) ):

      ( _intf, typ, moves, lastMove) = decodePeerMacAddr( addr )
      if  _intf is None:
         continue

      table.tableEntries.append(
         UnicastMacAddressTable.UnicastTableEntry(
            vlanId=addr.hostKey.vlanId,
            macAddress=addr.hostKey.address,
            entryType=typ,
            interface=_intf,
            lastMove=lastMove,
            moves=moves ) )
   return table

#-------------------------------------------------------------------------------
# show mlag config-sanity [all]
#-------------------------------------------------------------------------------
def showMlagConfSanity( mode, args ):
   allConfigSanity = 'all' in args
   featureConfigDiff = ConfigSanity()
   dataTable = configCheckDir

   featureConfigDiff.mlagActive = mlagStatus.mlagState in ( 'primary', 'secondary' )
   featureConfigDiff.mlagConnected = \
      mlagStatus.negotiationStatus == NEG_STATUS_CONNECTED
   
   if not featureConfigDiff.mlagActive or not featureConfigDiff.mlagConnected:
      return featureConfigDiff

   if allConfigSanity:
      featureConfigDiff.detail = True

   if not dataTable.configPluginDir:
      return featureConfigDiff

   def mstiMapToDict( vidToMstiMapStr ):
      if vidToMstiMapStr:
         vidToMstiMapBytes = bytearray.fromhex( vidToMstiMapStr )
         vDict = parseVidToMstiMap( vidToMstiMapBytes )
         return inverseVlanMap( vDict, False )
      else:
         return {}

   def mstiModelUpdate( allConfigSanity, modelToUpdate, valueIn ):
      localMap = mstiMapToDict( valueIn.localValue )
      peerMap = mstiMapToDict( valueIn.peerValue )
      instKeys = set( localMap )
      instKeys.update( peerMap )
      allInstId = list( instKeys )
      allInstId.sort()
      # If neither dict has any instances, we are in 'default' state
      if not allInstId and allConfigSanity:
         modelToUpdate.globalParameters[ "mst-instance" ] = valueIn
         return
      for instId in allInstId:
         paramValue = ParamValue()
         localVlans = None
         peerVlans = None
         if instId in localMap: # pylint: disable=consider-using-get
            localVlans = localMap[ instId ]
         if instId in peerMap: # pylint: disable=consider-using-get
            peerVlans = peerMap[ instId ]
         if localVlans != peerVlans or allConfigSanity:
            paramValue.localValue = vlanSetToCanonicalString( localVlans )
            paramValue.peerValue = vlanSetToCanonicalString( peerVlans )
            # pylint: disable-next=consider-using-f-string
            curParamName = "mst-instance%d" % instId
            modelToUpdate.globalParameters[ curParamName ] = paramValue

   def mstConfigSpecToAttrs( paramEntry, allConfigSanity ):
      pvRegion = ParamValue()
      pvRevision = ParamValue()
      pvMap = ParamValue()
      ( pvRegion.localValue, pvRevision.localValue, pvMap.localValue ) = \
                                    extractMstConfigSpecAttrs( paramEntry.localVal )
      ( pvRegion.peerValue, pvRevision.peerValue, pvMap.peerValue ) = \
                                    extractMstConfigSpecAttrs( paramEntry.peerVal )
      if pvRegion.localValue == pvRegion.peerValue and not allConfigSanity:
         pvRegion = None
      if pvRevision.localValue == pvRevision.peerValue and not allConfigSanity:
         pvRevision = None
      # The vidToMstiMap gets checked for differences in mstiModelUpdate,
      # so it does not need to be checked here
      return ( pvRegion, pvRevision, pvMap )

   # A valid return value indicates that the modelToUpdate was modified
   def createModelParamValue( allConfigSanity, paramDiffs, 
                              modelToUpdate, paramTable=None ):
      paramValue = None # used a sentinel
      for paramKey, paramEntry in six.iteritems( paramDiffs ):
         func = mlagConfigSanityConsistencyCheckOverride.get( paramKey, 
                  lambda paramEntry: paramEntry.localVal == paramEntry.peerVal )
         inconsistent = not func( paramEntry )
         if allConfigSanity or inconsistent: 
            paramValue = ParamValue()
            if paramKey == "mst-config-spec": # pylint: disable=no-else-continue
               ( pvRegionId, pvConfigRevision, pvVidToMstiMap ) = \
                                 mstConfigSpecToAttrs( paramEntry, allConfigSanity )
               if pvRegionId:
                  modelToUpdate.globalParameters[ 'mst-region-name' ] = pvRegionId
               if pvConfigRevision:
                  modelToUpdate.globalParameters[ 'mst-revision-number' ] = \
                                                                     pvConfigRevision
               if pvVidToMstiMap:
                  mstiModelUpdate( allConfigSanity, modelToUpdate, pvVidToMstiMap )
               continue
            else:
               paramValue.localValue = \
                   re.sub( "[']", "", paramEntry.localVal ).rstrip('.') \
                   if paramEntry.localVal else None
               paramValue.peerValue = \
                   re.sub( "[']", "", paramEntry.peerVal ).rstrip('.') \
                   if paramEntry.peerVal else None
            if isinstance( modelToUpdate, InterfaceValue ):
               if paramTable.paramName.startswith( 'vlan-Tpid' ):
                  if paramValue.localValue is not None:
                     # pylint: disable-next=consider-using-f-string
                     paramValue.localValue = "0x{:x}".format(
                        int( paramValue.localValue ) )
                  if paramValue.peerValue is not None:
                     # pylint: disable-next=consider-using-f-string
                     paramValue.peerValue = "0x{:x}".format(
                        int( paramValue.peerValue ) )
               modelToUpdate.interface[ paramEntry.intfName ] = paramValue
            else:
               if paramEntry.paramName == 'hello-time':
                  if paramValue.localValue is not None:
                     # pylint: disable-next=consider-using-f-string
                     paramValue.localValue = "{:2.3f}".format (
                        float( paramValue.localValue ) / 1000 )
                  if paramValue.peerValue is not None:
                     # pylint: disable-next=consider-using-f-string
                     paramValue.peerValue = "{:2.3f}".format (
                        float( paramValue.peerValue ) / 1000 )
               if toggleStpPriorityConfigSanityEnabled() and \
                  paramEntry.paramName.startswith( 'dummyAttr') :
                  continue
               if toggleStpPriorityConfigSanityEnabled() and \
                  paramEntry.paramName == 'priority Cist':
                  # Replace 'Cist' with 'Mst0' to keep consistency with other outputs
                  modelToUpdate.globalParameters[ 'priority Mst0' ] = paramValue
               else:
                  modelToUpdate.globalParameters[ paramEntry.paramName ] = paramValue
      return paramValue

   globalFeatureParamDict = {}
   for pluginEntry in six.itervalues( dataTable.configPluginDir ):
      # update global param model
      globalFeatureParams = globalFeatureParamDict.setdefault( 
            pluginEntry.pluginName, GlobalFeatureParam() )
      paramValue = createModelParamValue( \
         allConfigSanity, pluginEntry.configGlobalDiff, globalFeatureParams )
      if paramValue: # at least one global param in diff for this feature
         if pluginEntry.pluginName in featureConfigDiff.globalConfiguration:
            featureConfigDiff.globalConfiguration[ pluginEntry.pluginName ]. \
               globalParameters.update( globalFeatureParams.globalParameters )
         else:
            featureConfigDiff.globalConfiguration[ pluginEntry.pluginName ] = \
               globalFeatureParams
      # update interface param model
      paramValue = None # sentinel
      intfFeatureParams = IntfFeatureParam()
      for intfTable in six.itervalues( pluginEntry.configIntfDir ):
         intfValue = InterfaceValue()
         paramValue = createModelParamValue( \
            allConfigSanity, intfTable.configIntfDiff, intfValue, intfTable )
         if paramValue: # atleast one interface param in diff
            intfFeatureParams.interfaceParameters[ intfTable.paramName ] = \
                intfValue
      if intfFeatureParams.interfaceParameters:
         # at least one parameter in interface diff for this feature
         if pluginEntry.pluginName in featureConfigDiff.interfaceConfiguration:
            featureConfigDiff.interfaceConfiguration[ pluginEntry.pluginName ]. \
               interfaceParameters.update( intfFeatureParams.interfaceParameters )
         else:
            featureConfigDiff.interfaceConfiguration[ pluginEntry.pluginName ] = \
               intfFeatureParams

   return featureConfigDiff

#-------------------------------------------------------------------------------
# show mac-address-table mlag-peer diff ...
#-------------------------------------------------------------------------------

UNSPECIFIED_INTERFACE = Tac.Value( "Arnet::IntfId" )

def  decodeLocalMacAddr( localEntry ):
   moves = None
   lastMove = None
   port = localEntry.getRawAttribute( "intf" )
   entryType = localEntry.entryType
   if entryType in ( 'configuredStaticMac', 'peerStaticMac', \
                     'configuredRemoteMac', 'peerConfiguredRemoteMac',
                     'authenticatedMac', 'peerAuthenticatedMac' ):
      entryType = 'static'
   elif entryType in ( 'learnedDynamicMac', 'peerDynamicMac', \
                       'learnedRemoteMac', 'receivedRemoteMac', \
                       'peerLearnedRemoteMac', 'peerReceivedRemoteMac',
                       'configuredDynamicMac', 'softwareLearnedDynamicMac' ):
      if port == UNSPECIFIED_INTERFACE:
         return ( None, None, None, None )
      entryType = 'dynamic'
      moves = localEntry.moves
      lastMove = localEntry.lastMoveTime + Tac.utcNow() - Tac.now()

   else:
      raise ValueError( "invalid localEntryess type: "
                        + repr( localEntry ) )

   return ( port, entryType, moves, lastMove )


#return true if two entries match. Otherwise, false
def compareEntry( localEntry, peerEntry ):
   
   if localEntry is None or peerEntry is None:
      return False
   else:
      if peerEntry.entryType in [ 'peerConfiguredRemoteMac', 
                           'peerLearnedRemoteMac',
                           'peerReceivedRemoteMac' ]:
         peerIntf = decodeIntf( peerEntry.intf )
      else:
         peerIntf = peerEntry.intf
 
      localIntf = localEntry.intf
      if ( localIntf != peerIntf or 
            localEntry.entryType != peerEntry.entryType ):
         return False
   return True
   
def showMacTableOnlyInPeer( mode, args ):
   table = UnicastMacAddressTable()

   for peerEntry in sorted( mlagHostTable.hostEntry.values(), 
                              key=lambda x: ( x.hostKey.vlanId,
                                            x.hostKey.address ) ):
      localKey = Tac.Value( 'Bridging::HostKey', 
            peerEntry.vlanId, peerEntry.address)

      localEntry = bridgingStatus.smashFdbStatus.get( localKey )

      match = compareEntry( localEntry, peerEntry )

      if not match:
         (intf, typ, moves, lastMove) = decodePeerMacAddr( peerEntry )
         table.tableEntries.append(
            UnicastMacAddressTable.UnicastTableEntry(
               vlanId=peerEntry.vlanId,
               macAddress=peerEntry.address,
               entryType=typ,
               interface=intf,
               lastMove=lastMove,
               moves=moves ) )
 
   return table

def showMacTableOnlyInLocal( mode, args ):
   table = UnicastMacAddressTable()
   for localEntry in sorted( six.itervalues( bridgingStatus.smashFdbStatus ),
                              key=lambda x: ( x.key.fid,
                                            x.key.addr ) ):

      if localEntry.entryType[:4] != "peer":
         continue

      key = Tac.Value( 'Mlag::HostEntryKey', localEntry.key.fid,
                     localEntry.key.addr )
      peerEntry = mlagHostTable.hostEntry.get( key )
      
      match = compareEntry( localEntry, peerEntry)
      
      if not match:
         (intf, typ, moves, lastMove) = decodeLocalMacAddr( localEntry )
         
         table.tableEntries.append(
            UnicastMacAddressTable.UnicastTableEntry(
               vlanId=localEntry.key.fid,
               macAddress=localEntry.key.addr,
               entryType=typ,
               interface=intf,
               lastMove=lastMove,
               moves=moves ) )
 
   return table
   
#-------------------------------------------------------------------------------
# Mounts
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global mlagConfig, mlagStatus, mlagProtoStatus, mlagHostTable, \
       lagIntfStatusDir, mlagHwStatus, configCheckDir
   global ethIntfConfig, ethIntfStatus
   global subIntfConfig, subIntfStatus
   global bridgingStatus
   global aclRequestToPeer, aclRequestFromPeer, aclResponseToPeer, \
       aclResponseFromPeer, aclTimeoutStatus
   # LazyMount mlag/config, Mlag::Config and its dependent paths
   if Toggles.MlagToggleLib.toggleMlagL2SubinterfacesEnabled():
      mlagConfig = LazyMount.mount( entityManager, 'mlag/input/config/cli',
                                    'Mlag::Config', 'r' )
   else:
      mlagConfig = LazyMount.mount( entityManager, 'mlag/config',
                                    'Mlag::Config', 'r' )

   # LazyMount mlag/status, Mlag::Status and its dependent paths
   mlagStatus = MlagStatusLazyMounter( entityManager )
   mlagHwStatus = LazyMount.mount( entityManager, "mlag/hardware/status",
         "Mlag::Hardware::Status", "r" )
   mlagProtoStatus = LazyMount.mount( entityManager, "mlag/proto",
                                      "Mlag::ProtoStatus", "rO" )
   mlagHostTable = LazyMount.mount( entityManager, "mlag/hostTable",
                                    "Mlag::HostTable", "rO" )
   ethIntfConfig = LazyMount.mount( entityManager, "interface/config/eth/intf",
                                    "Interface::EthIntfConfigDir", "r" )
   ethIntfStatus = LazyMount.mount( entityManager, "interface/status/eth/intf",
                                    "Interface::EthIntfStatusDir", "r" )
   subIntfConfig = LazyMount.mount( entityManager, "interface/config/subintf",
                                    "Interface::SubIntfConfigDir", "r" )
   subIntfStatus = LazyMount.mount( entityManager, "interface/status/subintf",
                                    "Interface::SubIntfStatusDir", "r" )
   lagIntfStatusDir = LazyMount.mount( entityManager, "interface/status/eth/lag",
                                       "Interface::EthLagIntfStatusDir", "r" )
   configCheckDir = LazyMount.mount( entityManager, "mlag/configCheck", 
                                        "Mlag::ConfigCheck::ConfigCheckDir", "rO" )
   bridgingStatus = SmashLazyMount.mount( entityManager, "bridging/status",
                                          "Smash::Bridging::Status",
                                          SmashLazyMount.mountInfo( 'reader' ) )
   aclRequestToPeer = LazyMount.mount( entityManager,
                                       "mlag/aclInstall/request/toPeer",
                                       "Mlag::AclInstallRequest", "r" )
   aclRequestFromPeer = LazyMount.mount( entityManager,
                                         "mlag/aclInstall/request/fromPeer",
                                         "Mlag::AclInstallRequest", "r" )
   aclResponseToPeer = LazyMount.mount( entityManager,
                                        "mlag/aclInstall/response/toPeer",
                                        "Mlag::AclInstallResponse", "r" )
   aclResponseFromPeer = LazyMount.mount( entityManager,
                                        "mlag/aclInstall/response/fromPeer",
                                        "Mlag::AclInstallResponse", "r" )
   aclTimeoutStatus = LazyMount.mount( entityManager, "mlag/aclInstall/timeout",
                                       "Mlag::AclTimeoutStatus", "r" )

