# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import BasicCli
import CliParser
import CliCommand
import CliMatcher
import LazyMount
from CliPlugin import IgmpSnoopingCliLib
from CliPlugin import IntfCli
from CliPlugin import Ip6AddrMatcher
from CliPlugin import MlagIgmpSnoopingCli # pylint: disable=unused-import
from CliPlugin.IgmpSnoopingCli import (
      IgmpSnoopingMrouter,
      clearSnoopingCounters,
      doSetSnoopingVlanMember,
      doShowGmpSnoopingMrouter,
      getVlansOrWarn,
      interfaceMatcher,
      intfRangeConfigMatcher,
      isSnoopingAgentRunning,
      memberMatcher,
      setSnoopingVlanMrouter,
      showClientSnoopingsGroupsHandler,
      syncCounterCheckpointDir,
      vlanIdMatcher,
      mlagConnectionStatePrintingTable,
      mlagMountStatePrintingTable,
      switchPortMatcher,
      reportFloodingMatcher,
)
from CliPlugin.VlanCli import VlanConfigMode, vlanSetMatcher
from CliMode.MldSnooping import MldSnoopingMode, MldSnoopingVlanMode
from CliToken.Clear import clearKwNode
import ConfigMount
import FileUrl
import Url
import Tac
import ShowCommand
import Toggles.GmpToggleLib
from IgmpSnoopingModel import GmpMembershipInfo, GmpMembershipDetailInfo
from IgmpSnoopingModel import GmpMembershipCountInfo
# Add explicit dependency to load MlagIgmpSnoopingCli plugin. MlagIgmpSnoopingCli
# adds an IgmpSnoopingCliHelper extension to provide apis required to find peer-link
# intf
from MldSnoopingModel import ( MldSnoopingCounters, MldSnoopingCountersInterface,
                               MldSnoopingInfo, MldSnoopingVlanInfo,
                               MldSnoopingQuerier, MldSnoopingQuerierDetail,
                               MldSnoopingVlanQuerier,
                               MldSnoopingQuerierCounters, MldSnoopingPacketCounters,
                               MldSnoopingMembership, VlanMembership,
                               GroupMembership, SourceMembership,
                               MldSnoopingVlanIntfs, MldSnoopingReportFlooding )
from MldSnooping import agentName

# All globals are dictionaries with index as entityManager. This is to support Cli
# with multiple entity managers in a cohab test environment
mldSnoopingConfigMap = {}
bridgingHwCapabilitiesMap = {}
forwardingStatusMap = {}
protocolStatusMap = {}
querierStatusMap = {}
mldSnoopingCounterConfigMap = {}
counterDirMap = {}
counterCheckpointDirMap = {}
bridgingInputCliMap = {}
peerQuerierStatusMap = {}

mldMatcher = CliMatcher.KeywordMatcher( 'mld',
                  helpdesc='Multicast Listener Discovery commands' )

def mldSnoopingSupportedGuard( mode, token ):
   bridgingHwCapabilities = bridgingHwCapabilitiesMap[ mode.entityManager ]
   if bridgingHwCapabilities.mldSnoopingSupported:
      return None
   return CliParser.guardNotThisPlatform

mldVersionTable = {
   'igmpVersionUnknown': 'unknown',
   'mldVersion1': 'v1',
   'mldVersion2': 'v2'
}

#-------------------------------------------------------------------------------
# The "config-mld-snooping" mode.
#-------------------------------------------------------------------------------
class MldSnoopingConfigMode( MldSnoopingMode, BasicCli.ConfigModeBase ):
   name = 'mld snooping configuration'

   def __init__( self, parent, session ):
      MldSnoopingMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-------------------------------------------------------------------------------
# The "[no|default] mld snooping" command, in "config" mode.
#-------------------------------------------------------------------------------
class EnterMldSnoopingConfigMode( CliCommand.CliCommandClass ):
   syntax = 'mld snooping'
   noOrDefaultSyntax = syntax
   data = {
            'mld': 'Multicast Listener Discovery commands',
            'snooping': CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher(
                  'snooping',
                  helpdesc='Configure snooping parameters' ),
               guard=mldSnoopingSupportedGuard )
          }

   @staticmethod
   def handler( mode, args ):
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      if not mldSnoopingConfig.mldSnoopingConfigured:
         mldSnoopingConfig.mldSnoopingConfigured = True
         mldSnoopingConfig.vlanDefaultEnabled = True
      childMode = mode.childMode( MldSnoopingConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      mldSnoopingConfig.mldSnoopingConfigured = False
      mldSnoopingConfig.vlanDefaultEnabled = False
      mldSnoopingConfig.vlanConfig.clear()

#-------------------------------------------------------------------------------
# The "[no|default] disabled" command, in "config-mld-snooping" mode.
#-------------------------------------------------------------------------------
class MldSnoopingDisabled( CliCommand.CliCommandClass ):
   syntax = 'disabled'
   noOrDefaultSyntax = syntax
   data = { 'disabled': 'Disable MLD snooping' }

   @staticmethod
   def handler( mode, args ):
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      mldSnoopingConfig.vlanDefaultEnabled = False

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      mldSnoopingConfig.reset()

#-------------------------------------------------------------------------------
# The "config-mld-snooping-vlan" mode.
#-------------------------------------------------------------------------------
class MldSnoopingVlanConfigMode( MldSnoopingVlanMode, BasicCli.ConfigModeBase ):
   name = 'vlan configuration'

   def __init__( self, parent, session, vlans ):
      # VlanConfigMode is not CLI parser cache-safe as it modifies the individual
      # VLANs of the VLAN argument, and thus also makes MldSnoopingVlanConfigMode not
      # cache-safe. Avoid this by making a copy.
      self.vlans = vlans.copy()
      MldSnoopingVlanMode.__init__( self, str( self.vlans ) )
      self.vlanConfigMode = VlanConfigMode( parent, session, self.vlans )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def showActive( self ):
      runningConfigUrl = FileUrl.localRunningConfig( *Url.urlArgsFromMode( self ) )
      self.vlanConfigMode.printShowActiveForVlanRange( runningConfigUrl,
                                                       configmode='mld snooping',
                                                       prefix='   ' )

   def showActiveAll( self, showDetail ):
      runningConfigAllUrl = FileUrl.localRunningConfigAll(
         *Url.urlArgsFromMode( self ), showDetail=showDetail )
      self.vlanConfigMode.printShowActiveForVlanRange( runningConfigAllUrl,
                                                       configmode='mld snooping',
                                                       prefix='   ' )


#-------------------------------------------------------------------------------
# The "vlan <vlan range>" command, in "config-mld-snooping" mode.
#-------------------------------------------------------------------------------
class EnterMldSnoopingVlanConfigMode( CliCommand.CliCommandClass ):
   syntax = 'vlan VLANSET'
   noOrDefaultSyntax = syntax
   data = {
            'vlan': 'Specify VLAN/VLAN range',
            'VLANSET': vlanSetMatcher
          }

   @staticmethod
   def handler( mode, args ):
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      vlanSet = args[ 'VLANSET' ]
      vlanConfig = mldSnoopingConfig.vlanConfig
      for vlanId in vlanSet:
         vlan = vlanConfig.get( vlanId )
         if not vlan:
            vlan = mldSnoopingConfig.newVlanConfig( vlanId )
            vlan.enabled = 'vlanStateDefault'
         if not vlan.vlanQuerierConfig:
            vlan.vlanQuerierConfig = ()

      childMode = mode.childMode( MldSnoopingVlanConfigMode, vlans=vlanSet )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      vlanSet = args[ 'VLANSET' ]
      vlanConfig = mldSnoopingConfig.vlanConfig
      for vlanId in vlanSet:
         del vlanConfig[ vlanId ]

def setMldSnoopingVlans_( mode, enabled ):
   mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
   for vlanId in mode.vlans:
      mldSnoopingConfig.vlanConfig[ vlanId ].enabled = enabled

def disableMldSnoopingVlans( mode, args ):
   setMldSnoopingVlans_( mode, 'vlanStateDisabled' )

def enableMldSnoopingVlans( mode, args ):
   setMldSnoopingVlans_( mode, 'vlanStateDefault' )

#-------------------------------------------------------------------------------
# The "[no|default] disabled" command, in "config-mld-snooping-vlan" mode.
#-------------------------------------------------------------------------------
class MldSnoopingVlanDisabled( CliCommand.CliCommandClass ):
   syntax = 'disabled'
   noOrDefaultSyntax = syntax
   data = { 'disabled': 'Disable mld snooping on vlans' }
   handler = disableMldSnoopingVlans
   noOrDefaultHandler = enableMldSnoopingVlans

#-------------------------------------------------------------------------------
# The "[no|default] member <IPV6ADDR> interface <intf>" command,
# in "config-mld-snooping-vlan" mode.
#-------------------------------------------------------------------------------
class VlanMemberGroupCommand( CliCommand.CliCommandClass ):
   syntax = 'member IPV6ADDR interface INTF'
   noOrDefaultSyntax = syntax
   data = {
      'member': memberMatcher,
      'IPV6ADDR': Ip6AddrMatcher.ip6AddrMatcher,
      'interface': interfaceMatcher,
      'INTF': intfRangeConfigMatcher
   }

   @staticmethod
   def handler( mode, args ):
      ipv6Addr = args[ 'IPV6ADDR' ]
      intfs = args[ 'INTF' ]
      no = CliCommand.isNoOrDefaultCmd( args )
      vlanIds = mode.vlans
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      for vlanId in vlanIds:
         doSetSnoopingVlanMember( mode, mldSnoopingConfig, vlanId, ipv6Addr, intfs,
                                  no=no )

   noOrDefaultHandler = handler

#-------------------------------------------------------------------------------
# The "[no|default] multicast-router interface <intf>" command,
# in "config-mld-snooping-vlan" mode.
#-------------------------------------------------------------------------------
class VlanMulticastRouterCommand( CliCommand.CliCommandClass ):
   syntax = "multicast-router interface INTF"
   noOrDefaultSyntax = syntax
   data = {
      "multicast-router": "Multicast router related configuration",
      "interface": interfaceMatcher,
      "INTF": intfRangeConfigMatcher
   }

   @staticmethod
   def handler( mode, args ):
      intfs = args[ 'INTF' ]
      no = CliCommand.isNoOrDefaultCmd( args )
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      setSnoopingVlanMrouter( mode, mldSnoopingConfig, mode.vlans, intfs, no=no )

   noOrDefaultHandler = handler

def getVlanMldConfigs( mode, mldSnoopingConfig ):
   return { vlan: mldSnoopingConfig.newVlanConfig( vlan ) for vlan in mode.vlans }

# -------------------------------------------------------------------------------
# The "[no|default] report-flooding switch-port <intf>" command
# in "config-mld-snooping-vlan" mode.
# -------------------------------------------------------------------------------
class VlanReportFloodingSwitchPortIntf( CliCommand.CliCommandClass ):
   syntax = "report-flooding switch-port INTF"
   noOrDefaultSyntax = syntax
   data = {
      "report-flooding": reportFloodingMatcher,
      "switch-port": switchPortMatcher,
      "INTF": intfRangeConfigMatcher
   }

   @staticmethod
   def handler( mode, args ):
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      vlanConfigs = getVlanMldConfigs( mode, mldSnoopingConfig )
      interfaces = IntfCli.Intf.getAll( mode, args[ 'INTF' ], config=True )
      for vlanConfig in vlanConfigs.values():
         vlanConfig.reportFloodingConfig = 'vlanReportFloodingEnabled'
         for interface in interfaces:
            intfName = interface.name
            interface.create()
            vlanConfig.switchPort[ intfName ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      vlanConfigs = getVlanMldConfigs( mode, mldSnoopingConfig )
      interfaces = IntfCli.Intf.getAll( mode, args[ 'INTF' ], config=True )
      for vlanConfig in vlanConfigs.values():
         for interface in interfaces:
            intfName = interface.name
            del vlanConfig.switchPort[ intfName ]
            if not vlanConfig.switchPort and \
               not vlanConfig.reportFloodingAllSwitchPorts:
               vlanConfig.reportFloodingConfig = 'vlanReportFloodingDefault'

if Toggles.GmpToggleLib.toggleMldReportFloodingEnabled():
   MldSnoopingVlanConfigMode.addCommandClass( VlanReportFloodingSwitchPortIntf )

# -------------------------------------------------------------------------------
# The "[no|default] report-flooding switch-port all" command
# in "config-mld-snooping-vlan" mode.
# -------------------------------------------------------------------------------
class VlanReportFloodingSwitchPortAll( CliCommand.CliCommandClass ):
   syntax = "report-flooding switch-port all"
   noOrDefaultSyntax = syntax
   data = {
      "report-flooding": reportFloodingMatcher,
      "switch-port": switchPortMatcher,
      "all": "Configure flooding to all switch-ports of the VLAN's topology"
   }

   @staticmethod
   def handler( mode, args ):
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      vlanConfigs = getVlanMldConfigs( mode, mldSnoopingConfig )
      for vlanConfig in vlanConfigs.values():
         vlanConfig.reportFloodingAllSwitchPorts = True
         vlanConfig.reportFloodingConfig = 'vlanReportFloodingEnabled'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      vlanConfigs = getVlanMldConfigs( mode, mldSnoopingConfig )
      for vlanConfig in vlanConfigs.values():
         vlanConfig.reportFloodingAllSwitchPorts = False
         if not vlanConfig.switchPort:
            vlanConfig.reportFloodingConfig = 'vlanReportFloodingDefault'

if Toggles.GmpToggleLib.toggleMldReportFloodingEnabled():
   MldSnoopingVlanConfigMode.addCommandClass( VlanReportFloodingSwitchPortAll )

# matches CliPlugin.IgmpSnoopingCli.matcherQuerier, it just begins with MLD
matcherMldQuerier = CliMatcher.KeywordMatcher( "querier",
   helpdesc="MLD querier related status and configuration" )

# -------------------------------------------------------------------------------
# The "[no|default] querier" command in "config-mld-snooping-vlan" mode.
# -------------------------------------------------------------------------------
class MldL2Querier( CliCommand.CliCommandClass ):
   syntax = "querier"
   noOrDefaultSyntax = syntax
   data = { "querier": matcherMldQuerier }

   @staticmethod
   def handler( mode, args ):
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      vlanConfigs = getVlanMldConfigs( mode, mldSnoopingConfig )
      querierState = ( 'vlanQuerierDefault' if CliCommand.isNoOrDefaultCmd( args )
                       else 'vlanQuerierEnabled' )
      for config in vlanConfigs.values():
         config.vlanQuerierConfig.enabledConfigured = querierState

   noOrDefaultHandler = handler


MldSnoopingVlanConfigMode.addCommandClass( MldL2Querier )

# -------------------------------------------------------------------------------
# The "[no|default] querier interval <1-31744> [seconds]" command,
# in "config-mld-snooping-vlan" mode.
# -------------------------------------------------------------------------------

class MldQuerierIntervalCommand( CliCommand.CliCommandClass ):
   syntax = "querier interval NUMSEC [ seconds ]"
   noOrDefaultSyntax = "querier interval ..."
   data = {
      "querier": matcherMldQuerier,
      "interval": "Configure interval that multicast listeners are queried",
      "NUMSEC": CliMatcher.IntegerMatcher(
         1,
         31744,
         helpdesc='Number of seconds' ),
      "seconds": "Seconds"
   }

   @staticmethod
   def handler( mode, args ):
      queryInterval = args.get( 'NUMSEC', 0.0 )
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      vlanConfigs = getVlanMldConfigs( mode, mldSnoopingConfig )
      for config in vlanConfigs.values():
         config.vlanQuerierConfig.queryIntervalConfigured = queryInterval

   noOrDefaultHandler = handler

# -------------------------------------------------------------------------------
# The "[no|default] querier max response interval <1-8387> [seconds]" command,
# in "config-mld-snooping-vlan" mode.
# -------------------------------------------------------------------------------

class MldQuerierResponseIntervalCommand( CliCommand.CliCommandClass ):
   syntax = "querier max response interval INTERVAL [ seconds ]"
   noOrDefaultSyntax = "querier max response interval ..."
   data = {
      "querier": matcherMldQuerier,
      "max": "Maximum time to wait for multicast listeners to respond.",
      "response": "Maximum time to wait for multicast listeners to respond.",
      "interval": "Configure the maximum query response time "
                  "for multicast listeners",
      "INTERVAL": CliMatcher.IntegerMatcher(
         1,
         8387,
         helpdesc='Number of seconds' ),
      "seconds": "Seconds"
   }

   @staticmethod
   def handler( mode, args ):
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      queryResponseInterval = args.get( 'INTERVAL', 0.0 )
      vlanConfigs = getVlanMldConfigs( mode, mldSnoopingConfig )
      for config in vlanConfigs.values():
         config.vlanQuerierConfig.queryResponseIntervalConfigured = \
            queryResponseInterval

   noOrDefaultHandler = handler

matcherLastMemberQuery = CliMatcher.KeywordMatcher( "query",
   helpdesc="MLD last member configuration" )
# -------------------------------------------------------------------------------
# The "querier last member query count <1-3>" command,
# in "config-mld-snooping-vlan" mode.
# -------------------------------------------------------------------------------

class MldQuerierMemberQueryCountCommand( CliCommand.CliCommandClass ):
   syntax = "querier last member query count QUERYCOUNT"
   noOrDefaultSyntax = "querier last member query count ..."
   data = {
      "querier": matcherMldQuerier,
      "last": "Configure parameters for when there are no more listeners",
      "member": "Configure parameters for when there are no more listeners",
      "query": matcherLastMemberQuery,
      "count": "Configure number of Multicast Address Specific Queries sent"
               " before MLD state is cleared",
      "QUERYCOUNT": CliMatcher.IntegerMatcher(
         1,
         3,
         helpdesc='Last member query count' ),
   }

   @staticmethod
   def handler( mode, args ):
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      lastMemberQueryCount = args.get( 'QUERYCOUNT', 0 )
      vlanConfigs = getVlanMldConfigs( mode, mldSnoopingConfig )
      for config in vlanConfigs.values():
         config.vlanQuerierConfig.lastMemberQueryCountConfigured = \
            lastMemberQueryCount

   noOrDefaultHandler = handler

# -------------------------------------------------------------------------------
# The "querier last member query interval <1-25> [seconds]" command,
# in "config-mld-snooping-vlan" mode.
# -------------------------------------------------------------------------------
class MldQuerierLastMemberIntervalCommand( CliCommand.CliCommandClass ):
   syntax = "querier last member query interval INTERVAL [seconds]"
   noOrDefaultSyntax = "querier last member query interval ..."
   data = {
      "querier": matcherMldQuerier,
      "last": "Configure parameters for when there are no more listeners",
      "member": "Configure parameters for when there are no more listeners",
      "query": matcherLastMemberQuery,
      "interval": "Configure time between last member queries",
      "INTERVAL": CliMatcher.IntegerMatcher(
         1,
         25,
         helpdesc='Number of seconds' ),
      "seconds": "Seconds",
   }

   @staticmethod
   def handler( mode, args ):
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      lastMemberQueryInterval = args.get( 'INTERVAL', 0.0 )
      vlanConfigs = getVlanMldConfigs( mode, mldSnoopingConfig )
      for config in vlanConfigs.values():
         config.vlanQuerierConfig.lastMemberQueryIntervalConfigured = \
               lastMemberQueryInterval

   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( EnterMldSnoopingConfigMode )
MldSnoopingConfigMode.addCommandClass( MldSnoopingDisabled )
MldSnoopingConfigMode.addCommandClass( EnterMldSnoopingVlanConfigMode )
MldSnoopingVlanConfigMode.addCommandClass( MldSnoopingVlanDisabled )
MldSnoopingVlanConfigMode.addCommandClass( VlanMemberGroupCommand )
MldSnoopingVlanConfigMode.addCommandClass( VlanMulticastRouterCommand )
MldSnoopingVlanConfigMode.addCommandClass( MldQuerierIntervalCommand )
MldSnoopingVlanConfigMode.addCommandClass( MldQuerierMemberQueryCountCommand )
MldSnoopingVlanConfigMode.addCommandClass( MldQuerierLastMemberIntervalCommand )
MldSnoopingVlanConfigMode.addCommandClass( MldQuerierResponseIntervalCommand )

detailHelpStr = 'More comprehensive output'

# -------------------------------------------------------------------------------
# "show mld snooping mrouter"
# -------------------------------------------------------------------------------
class ShowMldSnoopingMrouter( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mld snooping mrouter [ detail ] [ vlan VLANID ]'
   data = {
      'mld': mldMatcher,
      'snooping': 'Show snooping information',
      'mrouter': 'MLD multicast router information',
      'detail': detailHelpStr,
      'vlan': 'Specify VLAN',
      'VLANID': vlanIdMatcher,
   }
   cliModel = IgmpSnoopingMrouter

   @staticmethod
   def handler( mode, args ):
      vlanId = args.get( 'VLANID' )
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      protocolStatus = protocolStatusMap[ mode.entityManager ]
      forwardingStatus = forwardingStatusMap[ mode.entityManager ]
      return doShowGmpSnoopingMrouter( mode, vlanId=vlanId,
            detail='detail' in args, sysdbNodes=( None,
                                                  mldSnoopingConfig.vlanConfig,
                                                  protocolStatus.contextStatus,
                                                  forwardingStatus.vlanStatus ) )

BasicCli.addShowCommandClass( ShowMldSnoopingMrouter )

def updatePortCounters( mode, args ):
   if not isSnoopingAgentRunning( mode.entityManager.sysname(), agentName() ):
      mode.addError( '%s agent is not running, displaying stale counters'
                     % agentName() )
      return

   mldSnoopingCounterConfig = mldSnoopingCounterConfigMap[ mode.entityManager ]
   mldSnoopingCounterConfig.counterUpdateRequestTime = Tac.now() # monotonic time
   forwardingStatus = forwardingStatusMap[ mode.entityManager ]

   def countersUpdated():
      return forwardingStatus.counterUpdateTime >= \
             mldSnoopingCounterConfig.counterUpdateRequestTime
   try:
      Tac.waitFor( countersUpdated, description="counter update", maxDelay=0.2,
                   sleep=True, timeout=30.0 )
   except Tac.Timeout:
      mode.addWarning( "Displaying stale counters" )

def doShowMldSnoopingCounter( mode, args ):
   ''' show multicast counter entries. '''
   counterDir = counterDirMap[ mode.entityManager ]
   counterCheckpointDir = counterCheckpointDirMap[ mode.entityManager ]
   syncCounterCheckpointDir( counterCheckpointDir, counterDir )
   intfs = counterCheckpointDir.intfCounter

   if not intfs:
      return MldSnoopingCounters()
   errorSpecific = "errors" in args
   c = Tac.newInstance( "Bridging::IgmpSnooping::IgmpContextIntfCounter", "" )
   interfaces = { }
   for intf in Arnet.sortIntf( intfs ):
      counter = counterDir.intfCounter.get( intf )
      counterCheckpoint = counterCheckpointDir.intfCounter.get( intf )
      if counter is None or counterCheckpoint is None:
         continue
      # print difference since last checkpointed counter
      c.doCopy( counter )
      c.doSubtract( counterCheckpoint )
      interfaceCounters = MldSnoopingCountersInterface()
      interfaceCounters.shortPacketsReceived = c.inPktTooShort
      interfaceCounters.nonIpPacketsReceived = c.inPktNonIp
      interfaceCounters.badChecksumIpPacketsReceived = c.inPktBadIpChecksum
      interfaceCounters.unknownIpPacketsReceived = c.inPktUnknown
      interfaceCounters.badChecksumPimPacketsReceived = c.inPktBadPimChecksum

      interfaceCounters.badChecksumIcmpV6PacketsReceived = c.inPktBadIcmpV6Checksum
      interfaceCounters.badMldQueryReceived = c.inPktBadMldQuery
      interfaceCounters.badMldV2ReportReceived = c.inPktBadMldV2Report
      if not errorSpecific:
         interfaceCounters.pimPacketsReceived = c.inPktPim
         interfaceCounters.otherPacketsSent = c.outPktOther

         interfaceCounters.mldV1QueryReceived = c.inPktMldV1Query
         interfaceCounters.mldV2QueryReceived = c.inPktMldV2Query
         interfaceCounters.mldV2ReportReceived = c.inPktMldV2Report
         interfaceCounters.otherIcmpPacketsReceived = c.inPktIcmpV6Other
         interfaceCounters.mldQuerySend = c.outPktGmpQuery
         interfaceCounters.mldReportSend = c.outPktGmpReport

      interfaces[ intf ] = interfaceCounters
   return MldSnoopingCounters( interfaces=interfaces,
         _errorSpecific=errorSpecific )

#-------------------------------------------------------------------------------
# "show mld snooping counters"
#-------------------------------------------------------------------------------
class ShowMldSnoopingCounters( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mld snooping counters [ errors ]'
   data = {
      'mld': mldMatcher,
      'snooping': 'Show snooping information',
      'counters': 'MLD counter information',
      'errors': 'Error counters',
   }
   cliModel = MldSnoopingCounters
   prepareFunction = updatePortCounters
   handler = doShowMldSnoopingCounter

BasicCli.addShowCommandClass( ShowMldSnoopingCounters )

def doShowMldSnoopingQuerier( mode, args ):
   # pylint: disable-msg=W0612
   vlanId = args.get( 'VLANID' )
   protocolStatus = protocolStatusMap[ mode.entityManager ]
   contextStatuses = protocolStatus.contextStatus
   peerQuerierStatus = peerQuerierStatusMap[ mode.entityManager ]

   vlans = {}
   if vlanId:
      contextStatus = contextStatuses.get( vlanId )
      if contextStatus:
         vlanInfo = showVlanQuerier( contextStatus, peerQuerierStatus, vlanId )
         if vlanInfo:
            vlans[ vlanId ] = vlanInfo
      return MldSnoopingQuerier( vlans=vlans, _vlanSpecific=True )
   else:
      vlanIds = list( contextStatuses )
      vlanIds.sort( key=int )
      for vlan in vlanIds:
         contextStatus = contextStatuses.get( vlan )
         if contextStatus:
            vlanInfo = showVlanQuerier( contextStatus, peerQuerierStatus, vlan )
            if vlanInfo:
               vlans[ vlan ] = vlanInfo
      return MldSnoopingQuerier( vlans=vlans )

def showVlanQuerier( status, peerQuerierStatus, vlanId, config=None, vqs=None ):
   querier = status.querier
   if querier is None:
      return None
   querierIntf = querier.intf
   helper = IgmpSnoopingCliLib.IgmpSnoopingCliHelper()
   if querierIntf is None:
      return None
   elif helper.isPeerIntf( querierIntf ): # pylint: disable-msg=no-member
      # Peer-link is local querier intf. Check if the address matches peer's querier
      # address
      if status.vlanId in peerQuerierStatus.vlanQuerierStatus:
         peerVlanQuerier = peerQuerierStatus.vlanQuerierStatus[ status.vlanId ]
         if querier.addr == peerVlanQuerier.addr:
            # Local querier address is same as peer's querier address. Check if there
            # is a corresponding local mlag intf
            localMlagIntf = peerVlanQuerier.localMlagIntfId
            querierIntf = localMlagIntf if localMlagIntf else querierIntf

   vlanQuerierInfo = MldSnoopingVlanQuerier()
   vlanQuerierInfo.querierAddress = str( querier.addr )
   vlanQuerierInfo.mldVersion = mldVersionTable[ status.querierVersion ]
   vlanQuerierInfo.querierInterface = querierIntf
   vlanQuerierInfo.queryResponseInterval = status.queryResponseInterval

   if config:
      vlanQuerierInfo.adminState = getVlanQuerierAdminState( config, vlanId )
   if vqs and vqs.gmpQuerierStatus:
      vlanQuerierInfo.operationalState = vqs.gmpQuerierStatus.state
      vlanQuerierInfo.queryInterval = vqs.gmpQuerierStatus.queryInterval
      vlanQuerierInfo.listenerTimeout = vqs.gmpQuerierStatus.groupMembershipInterval

   return vlanQuerierInfo

def getVlanQuerierAdminState( config, vlanId ):
   vc = config.vlanConfig.get( vlanId )
   if vc and vc.vlanQuerierConfig:
      enabled = vc.vlanQuerierConfig.enabledConfigured == 'vlanQuerierEnabled'
   else:
      enabled = False
   return 'enabled' if enabled else 'disabled'

# -------------------------------------------------------------------------------
# "show mld snooping report-flooding [ vlan VLANID ]"
# -------------------------------------------------------------------------------
def doShowMldSnoopingReportFlooding( mode, args ):
   inputVlanId = args.get( 'VLANID' )
   protocolStatus = protocolStatusMap[ mode.entityManager ]
   contextStatuses = protocolStatus.contextStatus
   vlans = {}
   vlanIds = {}
   if inputVlanId:
      vlanIds = [ inputVlanId ]
   else:
      vlanIds = list( contextStatuses )
   for vlanId in vlanIds:
      contextStatus = contextStatuses.get( vlanId )
      if contextStatus.reportFloodingEnabled:
         vlanIntfs = MldSnoopingVlanIntfs()
         vlanIntfs.interfaces.extend( contextStatus.switchPort )
         vlans[ vlanId ] = vlanIntfs
   return MldSnoopingReportFlooding( vlans=vlans )

class ShowMldSnoopingReportFlooding( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mld snooping report-flooding [ vlan VLANID ]'
   data = {
      'mld': mldMatcher,
      'snooping': 'Show snooping information',
      'report-flooding': 'Report-flooding information',
      'vlan': 'Specify VLAN',
      'VLANID': vlanIdMatcher,
   }
   cliModel = MldSnoopingReportFlooding
   handler = doShowMldSnoopingReportFlooding

if Toggles.GmpToggleLib.toggleMldReportFloodingEnabled():
   BasicCli.addShowCommandClass( ShowMldSnoopingReportFlooding )

#-------------------------------------------------------------------------------
# "show mld snooping querier [ vlan VLANID ]"
#-------------------------------------------------------------------------------
class ShowMldSnoopingQuerier( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mld snooping querier [ vlan VLANID ]'
   data = {
      'mld': mldMatcher,
      'snooping': 'Show snooping information',
      'querier': 'MLD querier information',
      'vlan': 'Specify VLAN',
      'VLANID': vlanIdMatcher,
   }
   cliModel = MldSnoopingQuerier
   handler = doShowMldSnoopingQuerier

BasicCli.addShowCommandClass( ShowMldSnoopingQuerier )

#-------------------------------------------------------------------------------
# "show mld snooping querier [ vlan VLANID ] detail"
#-------------------------------------------------------------------------------
class ShowMldSnoopingQuerierDetail( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mld snooping querier [ vlan VLANID ] detail'
   data = {
      'mld': mldMatcher,
      'snooping': 'Show snooping information',
      'querier': 'MLD querier information',
      'vlan': 'Specify VLAN',
      'VLANID': vlanIdMatcher,
      'detail': detailHelpStr,
   }
   cliModel = MldSnoopingQuerierDetail

   @staticmethod
   def handler( mode, args ):
      vlanId = args.get( 'VLANID' )
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      protocolStatus = protocolStatusMap[ mode.entityManager ]
      contextStatuses = protocolStatus.contextStatus
      querierStatuses = querierStatusMap[ mode.entityManager ]
      querierStatus = querierStatuses.querierStatus
      peerQuerierStatus = peerQuerierStatusMap[ mode.entityManager ]

      vlans = {}
      if vlanId:
         contextStatus = contextStatuses.get( vlanId )
         if contextStatus:
            querierName = str( vlanId )
            vlanInfo = showVlanQuerier( contextStatus, peerQuerierStatus,
                                        vlanId, mldSnoopingConfig,
                                        querierStatus.get( querierName ) )
            if vlanInfo:
               vlans[ vlanId ] = vlanInfo
      else:
         vlanIds = list( contextStatuses )
         vlanIds.sort( key=int )
         for vlan in vlanIds:
            contextStatus = contextStatuses.get( vlan )
            if contextStatus:
               querierName = str( vlan )
               vlanInfo = showVlanQuerier( contextStatus, peerQuerierStatus,
                                           vlan, mldSnoopingConfig,
                                           querierStatus.get( querierName ) )
               if vlanInfo:
                  vlans[ vlan ] = vlanInfo

      return MldSnoopingQuerierDetail( vlans=vlans )


BasicCli.addShowCommandClass( ShowMldSnoopingQuerierDetail )

def showQuerierTxCountersHelper( txCounters, gqs ):
   txCounters.v1GeneralQueries += gqs.v1QueriesSent
   txCounters.v1GSQueries += gqs.mldV1GSQueriesSent
   txCounters.v2GeneralQueries += gqs.v2QueriesSent
   txCounters.v2GSQueries += gqs.mldV2GSQueriesSent
   txCounters.v2GSSQueries += gqs.mldV2GSSQueriesSent

def showQuerierRxCountersHelper( rxCounters, gqs ):
   rxCounters.v1GeneralQueries += gqs.v1QueriesReceived
   rxCounters.v1GSQueries += gqs.mldV1GSQueriesReceived
   rxCounters.v2GeneralQueries += gqs.v2QueriesReceived
   rxCounters.v2GSQueries += gqs.mldV2GSQueriesReceived
   rxCounters.v2GSSQueries += gqs.mldV2GSSQueriesReceived
   rxCounters.v1Reports += gqs.v1ReportReceived
   rxCounters.v2Reports += gqs.v2ReportReceived
   rxCounters.v1Dones += gqs.mldV1DonesReceived
   rxCounters.errorPackets += gqs.errorPacketReceived
   rxCounters.otherPackets += gqs.otherPacketReceived

#-------------------------------------------------------------------------------
# "show mld snooping querier counters"
#-------------------------------------------------------------------------------
class ShowMldSnoopingQuerierCounters( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mld snooping querier counters [ vlan VLANID ]'
   data = {
      'mld': mldMatcher,
      'snooping': 'Show snooping information',
      'querier': 'MLD querier information',
      'counters': 'Show MLD L2 querier packet counters',
      'vlan': 'Filter VLAN',
      'VLANID': vlanIdMatcher,
   }
   cliModel = MldSnoopingQuerierCounters

   @staticmethod
   def handler( mode, args ):
      vlanId = args.get( 'VLANID' )
      protocolStatus = protocolStatusMap[ mode.entityManager ]
      contextStatuses = protocolStatus.contextStatus
      querierStatuses = querierStatusMap[ mode.entityManager ]
      querierStatus = querierStatuses.querierStatus

      tx = MldSnoopingPacketCounters()
      rx = MldSnoopingPacketCounters()
      if vlanId and vlanId in contextStatuses:
         vlans = [ vlanId ]
      elif vlanId:
         vlans = []
      else:
         vlans = contextStatuses

      for vlan in vlans:
         querierName = str( vlan )
         vqs = querierStatus.get( querierName )
         if vqs and vqs.gmpQuerierStatus:
            gqs = vqs.gmpQuerierStatus
            showQuerierTxCountersHelper( tx, gqs )
            showQuerierRxCountersHelper( rx, gqs )

      return MldSnoopingQuerierCounters( txCounters=tx, rxCounters=rx )


BasicCli.addShowCommandClass( ShowMldSnoopingQuerierCounters )

def getVlanMembership( vqs, grpAddr=None ):
   if not vqs or not vqs.gmpQuerierStatus:
      return None
   groupMemberships = {}
   if grpAddr:
      # Get the group if it exists
      group = Arnet.IpGenAddr( grpAddr.stringValue )
      memberGroup = vqs.gmpQuerierStatus.group.get( group )
      if memberGroup:
         groupInfo = getGroupMembership( memberGroup )
         if groupInfo:
            groupMemberships[ grpAddr ] = groupInfo
      return VlanMembership( groupMemberships=groupMemberships )

   for group in sorted( vqs.gmpQuerierStatus.group ):
      # Try to get the group again, use get function in case it's gone
      memberGroup = vqs.gmpQuerierStatus.group.get( group )
      if memberGroup is None:
         continue
      groupInfo = getGroupMembership( memberGroup )
      if groupInfo:
         groupMemberships[ group.v6Addr ] = groupInfo

   return VlanMembership( groupMemberships=groupMemberships )

def getGroupMembership( group ):
   filterMode = group.filterMode
   mldVersion = mldVersionTable[ group.compatibilityMode ]
   sourceMemberships = {}
   for source in sorted( group.source ):
      # Try to get the source again, use get function in case it's gone
      memberSource = group.source.get( source )
      if memberSource is None:
         continue
      sourceInfo = getSourceMembership( memberSource, mldVersion )
      if sourceInfo:
         sourceMemberships[ source.v6Addr ] = sourceInfo

   return GroupMembership( filterMode=filterMode, mldVersion=mldVersion,
                           sourceMemberships=sourceMemberships )

def getSourceMembership( source, mldVersion ):
   return SourceMembership( mldVersion=mldVersion )

#-------------------------------------------------------------------------------
# "show mld snooping querier membership"
#-------------------------------------------------------------------------------
class ShowMldSnoopingQuerierMembership( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mld snooping querier membership [ vlan VLANID ] [ group GRPADDR ]'
   data = {
      'mld': mldMatcher,
      'snooping': 'Show snooping information',
      'querier': 'MLD querier information',
      'membership': 'Show MLD L2 querier membership',
      'vlan': 'Specify VLAN',
      'VLANID': vlanIdMatcher,
      'group': 'Specify group',
      'GRPADDR': Ip6AddrMatcher.ip6AddrMatcher,
   }
   cliModel = MldSnoopingMembership

   @staticmethod
   def handler( mode, args ):
      vlanId = args.get( 'VLANID' )
      groupAddr = args.get( 'GRPADDR' )
      protocolStatus = protocolStatusMap[ mode.entityManager ]
      contextStatuses = protocolStatus.contextStatus
      querierStatuses = querierStatusMap[ mode.entityManager ]
      querierStatus = querierStatuses.querierStatus

      if groupAddr:
         membership = MldSnoopingMembership( _groupSpecific=True )
      else:
         membership = MldSnoopingMembership()

      if vlanId:
         contextStatus = contextStatuses.get( vlanId )
         if contextStatus:
            querierName = str( vlanId )
            membershipInfo = getVlanMembership( querierStatus.get( querierName ),
                                                grpAddr=groupAddr )
            if membershipInfo:
               membership.vlanMemberships[ vlanId ] = membershipInfo
      else:
         for vlanId in sorted( contextStatuses ):
            contextStatus = contextStatuses.get( vlanId )
            if contextStatus:
               querierName = str( vlanId )
               membershipInfo = getVlanMembership( querierStatus.get( querierName ),
                                                   grpAddr=groupAddr )
               if membershipInfo:
                  membership.vlanMemberships[ vlanId ] = membershipInfo

      return membership


BasicCli.addShowCommandClass( ShowMldSnoopingQuerierMembership )

def getVlanSnoopingState( config, vlanId ):
   vlan = config.vlanConfig.get( vlanId )
   if vlan and vlan.enabled == 'vlanStateDefault':
      return 'enabled' if config.vlanDefaultEnabled else 'disabled'
   return 'disabled'

def doShowMldSnooping( mode, args ):
   ''' the function shows singleton attributes for the global settings,
       and per-vlan settings. Only the active vlans are shown here, as per
       industry-standard.
   '''
   mldSnooping = MldSnoopingInfo()
   mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
   forwardingStatus = forwardingStatusMap[ mode.entityManager ]
   protocolStatus = protocolStatusMap[ mode.entityManager ]
   if not mldSnoopingConfig.mldSnoopingConfigured:
      return mldSnooping

   vlanIds = getVlansOrWarn( mode, args, protocolStatus )
   if not vlanIds:
      return mldSnooping

   mldSnooping.mldSnoopingState = \
         'enabled' if mldSnoopingConfig.vlanDefaultEnabled else 'disabled'

   mldSnooping.robustness = mldSnoopingConfig.robustness
   mldSnooping.mlagConnectionState = \
      mlagConnectionStatePrintingTable[ protocolStatus.mlagConnectionState ]
   mldSnooping.mlagMountState = \
      mlagMountStatePrintingTable[ protocolStatus.mlagMountState ]
   for vlanId in vlanIds:
      mldSnooping.vlans[ vlanId ] = \
            doShowMldSnoopingVlan( mldSnoopingConfig, forwardingStatus,
                                   protocolStatus, vlanId )
   return mldSnooping

def doShowMldSnoopingVlan( config, forwardingStatus, protocolStatus, vlanId ):
   ''' show singleton attributes for a given vlan. '''

   vlans = MldSnoopingVlanInfo()
   vlans.mldSnoopingState = getVlanSnoopingState( config, vlanId )
   vlanConfig = config.vlanConfig.get( vlanId )

   if vlanConfig and ( vlanConfig.maxGroups < vlanConfig.unlimitedGroups ):
      vlans.maxGroups = vlanConfig.maxGroups

   contextStatus = protocolStatus.contextStatus.get( vlanId )
   if contextStatus and contextStatus.groupLimitExceeded:
      vlans.groupsOverrun = True

   statuses = forwardingStatus.vlanStatus
   statusVlan = statuses.get( vlanId )
   if contextStatus:
      vlans.pruningActive = contextStatus.confident
      vlans.evpnProxyActive = contextStatus.igmpProxyEnabled
   if statusVlan:
      vlans.floodingTraffic = statusVlan.useVlanFloodset
   return vlans

#-------------------------------------------------------------------------------
# "show mld snooping [vlan VLAN_ID]"
#-------------------------------------------------------------------------------
class ShowMldSnooping( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mld snooping [ vlan VLAN_ID ] '
   data = {
      'mld': mldMatcher,
      'snooping': 'Show snooping information',
      'vlan': 'Specify VLAN',
      'VLAN_ID': vlanIdMatcher,
   }
   cliModel = MldSnoopingInfo
   handler = doShowMldSnooping

BasicCli.addShowCommandClass( ShowMldSnooping )

#-------------------------------------------------------------------------------
# clear mld snooping counters [intfs] in enable mode
#-------------------------------------------------------------------------------
class ClearMldSnoopingCounters( CliCommand.CliCommandClass ):
   syntax = 'clear mld snooping counters [INTF]'
   data = {
            'clear': clearKwNode,
            'mld': 'Multicast Listener Discovery commands',
            'snooping': 'Snooping related',
            'counters': 'Counters',
            'INTF': intfRangeConfigMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      intfs = args.get( 'INTF' )
      interfaces = []
      if intfs:
         interfaces = IntfCli.Intf.getAll( mode, intfs, config=True )
         if not interfaces:
            return
      counterCheckpointDir = counterCheckpointDirMap[ mode.entityManager ]
      counterDir = counterDirMap[ mode.entityManager ]
      clearSnoopingCounters( mode, counterCheckpointDir, counterDir, igmp=False,
                             intfs=interfaces )

BasicCli.EnableMode.addCommandClass( ClearMldSnoopingCounters )

# -----------------------------------------------------------------------------------
# Display MembershipJoinStatus from different clients - user, local, mlag.
# If no filter is provided, display the merged state.
#
# show ip igmp snooping groups [ local | mlag | user | evpn ] \
#                              [ vlan <vlanid> ] [ A.B.C.D ] [ detail ] [ count ]
#
# Example of output:
# IGMP Snooping Group Membership
# EX: Filter mode Exclude
# IN: Filter mode Include
# VLAN  Group            Members
#----  ---------------  ----------------------------------------
# 10    225.0.0.1        Et1, Et2, Et3, Et4
#                       10.1.1.1(IR), 10.1.1.2(IR), 10.1.1.3(IR)
# 10    225.0.0.2        Et1, Et2, Et3, Et4, 10.1.1.1(IR),
#                       10.1.1.2(IR), 10.1.1.3(IR)
#10    *                Et3, Et4

clientMatcher = CliMatcher.EnumMatcher(
      { 'local': 'Show groups learnt locally via MLD',
        'mlag': 'Show groups learnt via MLAG peer',
        'user': 'Show groups configured by user',
        'evpn': 'Show groups learnt via EVPN' } )

vlanMatcher = CliMatcher.KeywordMatcher( 'vlan', helpdesc='Specify VLAN' )
showSnoopingMatcher = CliMatcher.KeywordMatcher( 'snooping',
                         helpdesc='Show snooping information' )
showGroupsMatcher = CliMatcher.KeywordMatcher( 'groups',
                       helpdesc='MLD group information' )

class ShowClientSnoopingGroups( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show mld snooping groups [ CLIENT ] [ vlan VLANID ] [ GRPADDR ]' )
   data = {
         'mld': mldMatcher,
         'snooping': showSnoopingMatcher,
         'groups': showGroupsMatcher,
         'CLIENT': clientMatcher,
         'vlan': vlanMatcher,
         'VLANID': vlanIdMatcher,
         'GRPADDR': Ip6AddrMatcher.ip6AddrMatcher,
         }
   cliModel = GmpMembershipInfo

   @staticmethod
   def handler( mode, args ):
      return showClientSnoopingsGroupsHandler( mode, agentName(), args,
                                               'showIpIgmpSnoopingGroups',
                                               GmpMembershipInfo )

class ShowClientSnoopingGroupsDetail( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show mld snooping groups [ CLIENT ] [ vlan VLANID ] [ GRPADDR ] '
              'detail' )
   data = {
         'mld': mldMatcher,
         'snooping': showSnoopingMatcher,
         'groups': showGroupsMatcher,
         'CLIENT': clientMatcher,
         'vlan': vlanMatcher,
         'VLANID': vlanIdMatcher,
         'GRPADDR': Ip6AddrMatcher.ip6AddrMatcher,
         'detail': detailHelpStr,
         }
   cliModel = GmpMembershipDetailInfo

   @staticmethod
   def handler( mode, args ):
      return showClientSnoopingsGroupsHandler( mode, agentName(), args,
                                               'showIpIgmpSnoopingGroups',
                                               GmpMembershipDetailInfo )

class ShowClientSnoopingGroupsCount( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show mld snooping groups [ CLIENT ] [ vlan VLANID ] count' )
   data = {
         'mld': mldMatcher,
         'snooping': showSnoopingMatcher,
         'groups': showGroupsMatcher,
         'CLIENT': clientMatcher,
         'vlan': vlanMatcher,
         'VLANID': vlanIdMatcher,
         'count': 'Show membership count',
         }
   cliModel = GmpMembershipCountInfo

   @staticmethod
   def handler( mode, args ):
      return showClientSnoopingsGroupsHandler( mode, agentName(), args,
                                               'showIpIgmpSnoopingGroupsCount',
                                               GmpMembershipCountInfo )

BasicCli.addShowCommandClass( ShowClientSnoopingGroups )
BasicCli.addShowCommandClass( ShowClientSnoopingGroupsDetail )
BasicCli.addShowCommandClass( ShowClientSnoopingGroupsCount )

def Plugin( entityManager ):
   mldSnoopingConfigMap[ entityManager ] = ConfigMount.mount(
      entityManager,
      "bridging/mldsnooping/config",
      "Bridging::IgmpSnooping::Config", "w" )
   bridgingHwCapabilitiesMap[ entityManager ] = LazyMount.mount(
      entityManager,
      "bridging/hwcapabilities",
      "Bridging::HwCapabilities", "r" )
   forwardingStatusMap[ entityManager ] = LazyMount.mount(
      entityManager,
      "bridging/mldsnooping/forwarding/status",
      "Bridging::IgmpSnooping::Status", "r" )
   protocolStatusMap[ entityManager ] = LazyMount.mount(
      entityManager,
      "bridging/mldsnooping/protocol/status",
      "Bridging::IgmpSnooping::IgmpProtocolStatus", "r" )
   querierStatusMap[ entityManager ] = LazyMount.mount(
      entityManager,
      "bridging/mldsnooping/querier/status",
      "Igmp::QuerierStatusDir", "r" )
   counterDirMap[ entityManager ] = LazyMount.mount(
      entityManager,
      'bridging/mldsnooping/counterDir',
      'Bridging::IgmpSnooping::IgmpCounterDir', 'r' )
   counterCheckpointDirMap[ entityManager ] = LazyMount.mount(
      entityManager,
      'bridging/mldsnooping/counterCheckpointDir',
      'Bridging::IgmpSnooping::IgmpCounterDir', 'w' )
   mldSnoopingCounterConfigMap[ entityManager ] = LazyMount.mount(
      entityManager,
      "bridging/mldsnooping/counterConfig",
      "Bridging::IgmpSnooping::CounterConfig", "w" )
   bridgingInputCliMap[ entityManager ] = LazyMount.mount(
      entityManager,
      'bridging/input/config/cli',
      'Bridging::Input::CliConfig', 'r' )
   peerQuerierStatusMap[ entityManager ] = LazyMount.mount(
      entityManager,
      'bridging/mldsnooping/mlag/peerQuerierStatus',
      'Bridging::IgmpSnooping::PeerQuerierStatus', 'r' )
