# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
from CliMode.Models import (
      ModelsMode,
      ModulesMode,
      MgmtModelsMode,
      ProviderAFTMode,
      ProviderHttpCommandsMode,
      ProviderSmashMode,
      ProviderSysdbMode,
      ProviderIPFIXMode,
      ProviderSflowMode,
      ProviderIsisMode,
      ProviderOspfMode,
      ProviderBgpMode,
      ProviderBgpRibMode,
      ProviderMacsecMode,
      ProviderConfigurationMode,
)
import CliSession
import ConfigMount
import GitLib
import LazyMount
import Tac
import Toggles.RoutingLibToggleLib as RoutingLibToggle
from IpLibConsts import DEFAULT_VRF

octaConfig = None
gnmiConfig = None
cliConfig = None
runningConfigDiff = None

# Constants for config diff feature.
MAX_DIFF_SIZE = 128 * 1024
MAX_AGG_DIFF_SIZE = 800 * 1024
MAX_ENTITY_DIFF_SIZE = 30 * 1024

# ------------------------------------------------------
# Management models config commands
# ------------------------------------------------------

class MgmtModelsConfigMode( MgmtModelsMode, BasicCli.ConfigModeBase ):
   """CLI configuration mode 'management api models'."""

   name = "Models configuration"

   def __init__( self, parent, session ):
      self.config_ = octaConfig

      MgmtModelsMode.__init__( self, "api-models" )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoMgmtModelsConfigMode( mode, args ):
   childMode = mode.childMode( MgmtModelsConfigMode )
   mode.session_.gotoChildMode( childMode )

def noMgmtModelsConfigMode( mode, args ):
   """Resets Models configuration to default."""
   noShowDefaultLeafs( mode, None )
   noProviderAFTConfigMode( mode, None )
   noProviderSmashConfigMode( mode, None )
   noProviderSysdbConfigMode( mode, None )
   noProviderIPFIXConfigMode( mode, None )
   noProviderSflowConfigMode( mode, None )
   noProviderHttpCommandsConfigMode( mode, None )
   noProviderIsisConfigMode( mode, None )
   if RoutingLibToggle.toggleOspf2LsdbExportEnabled():
      noProviderOspfConfigMode( mode, None )
   noProviderBgpConfigMode( mode, None )
   noProviderMacsecConfigMode( mode, None )
   noModelsConfigMode( mode, None )
   noModulesConfigMode( mode, None )
   noProviderConfigurationConfigMode( mode, None )

# ------------------------------------------------------
# provider aft
# ------------------------------------------------------

def setOctaAttrsAndMayBeWarning( mode, attrList, value ):
   for attr in attrList:
      setattr( octaConfig.aftOptions, attr, value )
   warnIfAgentRunning( mode, 'aft' )

class ProviderAFTConfigMode( ProviderAFTMode, BasicCli.ConfigModeBase ):
   """CLI configuration submode 'provider aft'."""

   name = 'Provider for AFT'

   def __init__( self, parent, session ):
      self.config_ = octaConfig
      ProviderAFTMode.__init__( self, "aft" )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoProviderAFTConfigMode( mode, args ):
   childMode = mode.childMode( ProviderAFTConfigMode )
   mode.session_.gotoChildMode( childMode )

def noProviderAFTConfigMode( mode, args ):
   setOctaAttrsAndMayBeWarning( mode,
         [ 'ipv4Unicast', 'ipv6Unicast', 'mpls', 'routeSummary' ], False )

def aftIpv4Unicast( mode, args ):
   setOctaAttrsAndMayBeWarning( mode, [ 'ipv4Unicast' ], True )

def noAftIpv4Unicast( mode, args ):
   setOctaAttrsAndMayBeWarning( mode, [ 'ipv4Unicast' ], False )

def aftIpv6Unicast( mode, args ):
   setOctaAttrsAndMayBeWarning( mode, [ 'ipv6Unicast' ], True )

def noAftIpv6Unicast( mode, args ):
   setOctaAttrsAndMayBeWarning( mode, [ 'ipv6Unicast' ], False )

def aftMpls( mode, args ):
   setOctaAttrsAndMayBeWarning( mode, [ 'mpls' ], True )

def noAftMpls( mode, args ):
   setOctaAttrsAndMayBeWarning( mode, [ 'mpls' ], False )

def aftRouteSummary( mode, args ):
   setOctaAttrsAndMayBeWarning( mode, [ 'routeSummary' ], True )

def noAftRouteSummary( mode, args ):
   setOctaAttrsAndMayBeWarning( mode, [ 'routeSummary' ], False )

# ------------------------------------------------------
# provider smash
# ------------------------------------------------------

restartWarning = 'The Octa agent needs to be restarted ' \
                      'in order to have path changes take effect. ' \
                      'This can be done by running \'agent Octa terminate\'.'
class ProviderSmashConfigMode( ProviderSmashMode, BasicCli.ConfigModeBase ):
   """CLI configuration submode 'provider smash'."""

   name = 'Provider for Smash'

   def __init__( self, parent, session ):
      self.config_ = octaConfig
      self.smashIncludes = { path for path in octaConfig.smashIncludes.values() 
            if path }
      self.smashExcludes = { path for path in octaConfig.smashExcludes.values() 
            if path }

      ProviderSmashMode.__init__( self, "smash" )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      """Commit onExit"""
      octaConfig.smashIncludes.clear()
      for i, path in enumerate( sorted( self.smashIncludes ) ):
         octaConfig.smashIncludes[ i ] = path

      octaConfig.smashExcludes.clear()
      for i, path in enumerate( sorted( self.smashExcludes ) ):
         octaConfig.smashExcludes[ i ] = path

def getSmashPathFromInput( smashPath ):
   smashPath = smashPath.strip( '/' )
   if smashPath.startswith( "Smash/" ):
      smashPath = smashPath[ 6: ]
   return smashPath

def gotoProviderSmashConfigMode( mode, args ):
   childMode = mode.childMode( ProviderSmashConfigMode )
   mode.session_.gotoChildMode( childMode )

def noProviderSmashConfigMode( mode, args ):
   childMode = mode.childMode( ProviderSmashConfigMode )
   if not childMode.smashIncludes and not childMode.smashExcludes:
      return
   octaConfig.smashIncludes.clear()
   octaConfig.smashExcludes.clear()
   if running():
      mode.addWarning( restartWarning )

def setSmashPath( mode, args ):
   smashPath = getSmashPathFromInput( args.get( 'SMASHPATH', "" ) )
   if 'disabled' in args:
      if smashPath in mode.smashExcludes:
         return
      mode.smashIncludes.discard( smashPath )
      mode.smashExcludes.add( smashPath )
   else:
      if smashPath in mode.smashIncludes:
         return
      mode.smashExcludes.discard( smashPath )
      mode.smashIncludes.add( smashPath )
   if running():
      mode.addWarning( restartWarning )

def noSmashPath( mode, args ):
   smashPath = getSmashPathFromInput( args.get( 'SMASHPATH', "" ) )
   if 'disabled' in args:
      if smashPath not in mode.smashExcludes:
         return
      mode.smashExcludes.discard( smashPath )
   else:
      if smashPath not in mode.smashIncludes:
         return
      mode.smashIncludes.discard( smashPath )
   if running():
      mode.addWarning( restartWarning )

# ------------------------------------------------------
# provider sysdb
# ------------------------------------------------------

class ProviderSysdbConfigMode( ProviderSysdbMode, BasicCli.ConfigModeBase ):
   """CLI configuration submode 'provider sysdb'."""

   name = 'Provider for Sysdb'

   def __init__( self, parent, session ):
      self.config_ = octaConfig
      self.sysdbExcludes = {}

      ProviderSysdbMode.__init__( self, "sysdb" )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      """Commit onExit"""
      if self.sysdbExcludes:
         for key, val in self.sysdbExcludes.items():
            octaConfig.sysdbExcludes[ key ] = val
      else:
         octaConfig.sysdbExcludes.clear()

def getSysdbPathFromInput( sysdbPath ):
   sysdbPath = sysdbPath.strip( '/' )
   if sysdbPath.startswith( "Sysdb/" ):
      sysdbPath = sysdbPath[ 6: ]
   return sysdbPath

def gotoProviderSysdbConfigMode( mode, args ):
   childMode = mode.childMode( ProviderSysdbConfigMode )
   mode.session_.gotoChildMode( childMode )

def noProviderSysdbConfigMode( mode, args ):
   if not octaConfig.sysdbExcludes:
      return
   octaConfig.sysdbExcludes.clear()
   if running():
      mode.addWarning( restartWarning )

def setSysdbPath( mode, args ):
   sysdbPath = getSysdbPathFromInput( args.get( 'SYSDBPATH', "" ) )
   if sysdbPath in mode.sysdbExcludes:
      return
   myEnum = Tac.Type( "Octa::SysdbExcludesStatus" )
   mode.sysdbExcludes[ sysdbPath ] = myEnum.unknown
   if running():
      mode.addWarning( restartWarning )

def noSysdbPath( mode, args ):
   sysdbPath = getSysdbPathFromInput( args.get( 'SYSDBPATH', "" ) )
   if not sysdbPath in mode.sysdbExcludes:
      return
   mode.sysdbExcludes.pop( sysdbPath, None )
   if running():
      mode.addWarning( restartWarning )

# ------------------------------------------------------
# provider IPFIX and provider sFlow
# ------------------------------------------------------

def maybeAddWarning( mode, condition, message ):
   if condition and running():
      mode.addWarning( message )

# ------------------------------------------------------
# provider IPFIX
# ------------------------------------------------------

class IPFIXDefaults:
   enableCollector = False
   address = "127.0.0.1"
   port = 4739
   domain = "default"
   flowTableSize = 16384
   flowLifetime = 300

ipfixRestartWarning = 'The Octa agent needs to be restarted ' \
                      'in order to have IPFIX changes take effect. ' \
                      'This can be done by running \'agent Octa terminate\'.'

class ProviderIPFIXConfigMode( ProviderIPFIXMode, BasicCli.ConfigModeBase ):
   """CLI configuration submode 'provider ipfix'."""

   name = 'Provider for IPFIX'
   defaults = IPFIXDefaults()

   def __init__( self, parent, session ):
      self.config_ = octaConfig
      ProviderIPFIXMode.__init__( self, "ipfix" )
      BasicCli.ConfigModeBase.__init__( self, parent, session ) 

def gotoProviderIPFIXConfigMode( mode, args ):
   childMode = mode.childMode( ProviderIPFIXConfigMode )
   # if octaConfig.ecoOptions doesn't exist yet, create it
   if octaConfig.ecoOptions is None:
      octaConfig.ecoOptions = ( 'ECOOptions', )
   wasEnabled = octaConfig.ecoOptions.ipfixEnableCollector
   octaConfig.ecoOptions.ipfixEnableCollector = True
   maybeAddWarning( mode, not wasEnabled, ipfixRestartWarning )
   mode.session_.gotoChildMode( childMode )

def noProviderIPFIXConfigMode( mode, args ):
   if octaConfig.ecoOptions is None:
      return
   wasEnabled = octaConfig.ecoOptions.ipfixEnableCollector
   octaConfig.ecoOptions.ipfixEnableCollector = False
   octaConfig.ecoOptions.ipfixAddress = IPFIXDefaults.address
   octaConfig.ecoOptions.ipfixPort = IPFIXDefaults.port
   octaConfig.ecoOptions.ipfixDomain = IPFIXDefaults.domain
   octaConfig.ecoOptions.ipfixFlowTableSize = IPFIXDefaults.flowTableSize
   octaConfig.ecoOptions.ipfixFlowLifetime = IPFIXDefaults.flowLifetime
   maybeAddWarning( mode, wasEnabled, ipfixRestartWarning )

def _setIPFIXListener( mode, address, port ):
   previousAddress = octaConfig.ecoOptions.ipfixAddress
   previousPort = octaConfig.ecoOptions.ipfixPort
   octaConfig.ecoOptions.ipfixAddress = address
   octaConfig.ecoOptions.ipfixPort = port
   maybeAddWarning( mode, previousAddress != address or previousPort != port,
                    ipfixRestartWarning )

def setIPFIXListener( mode, args ):
   # [default] listener <addr> [ port <port> ]
   address = args.get( 'ADDRESS', IPFIXDefaults.address )
   port = args.get( 'PORT', IPFIXDefaults.port )
   _setIPFIXListener( mode, address, port )

def defaultIPFIXListenerHandler( mode, args ):
   _setIPFIXListener( mode, IPFIXDefaults.address, IPFIXDefaults.port )

def _setIPFIXDomain( mode, domain ):
   previousDomain = octaConfig.ecoOptions.ipfixDomain 
   octaConfig.ecoOptions.ipfixDomain = domain
   maybeAddWarning( mode, previousDomain != domain, ipfixRestartWarning)

def setIPFIXDomain( mode, args ):
   # [default] domain NAME
   _setIPFIXDomain( mode, args.get( 'NAME', IPFIXDefaults.domain ) )

def defaultIPFIXDomainHandler( mode, args ):
   _setIPFIXDomain( mode, IPFIXDefaults.domain )

def _setIPFIXFlow( mode, size, lifetime ):
   previousSize = octaConfig.ecoOptions.ipfixFlowTableSize
   previousLifetime = octaConfig.ecoOptions.ipfixFlowLifetime
   if size is not None:
      octaConfig.ecoOptions.ipfixFlowTableSize = size
   if lifetime is not None:
      octaConfig.ecoOptions.ipfixFlowLifetime = lifetime
   maybeAddWarning( mode, previousSize != size or previousLifetime != lifetime,
                    ipfixRestartWarning )

def setIPFIXFlow( mode, args ):
   # [default] flow [ table size SIZE ] [ lifetime N seconds ]
   _setIPFIXFlow( mode, args.get( 'SIZE' ), args.get( 'LIFETIME' ) )

def defaultIPFIXFlowHandler( mode, args ):
   _setIPFIXFlow( mode, IPFIXDefaults.flowTableSize, IPFIXDefaults.flowLifetime )

# ------------------------------------------------------
# provider sFlow
# ------------------------------------------------------

class SflowDefaults:
   enableCollector = False
   address = "127.0.0.1"
   port = 6343
   domain = "default"
   flowTableSize = 16384
   flowLifetime = 300

sflowRestartWarning = 'The Octa agent needs to be restarted ' \
                      'in order to have sFlow changes take effect. ' \
                      'This can be done by running \'agent Octa terminate\'.'

class ProviderSflowConfigMode( ProviderSflowMode, BasicCli.ConfigModeBase ):
   """CLI configuration submode 'provider sflow'."""

   name = 'Provider for sFlow'
   defaults = SflowDefaults()

   def __init__( self, parent, session ):
      self.config_ = octaConfig
      ProviderSflowMode.__init__( self, "sflow" )
      BasicCli.ConfigModeBase.__init__( self, parent, session ) 

def gotoProviderSflowConfigMode( mode, args ):
   childMode = mode.childMode( ProviderSflowConfigMode )
   # if octaConfig.ecoOptions doesn't exist yet, create it
   if octaConfig.ecoOptions is None:
      octaConfig.ecoOptions = ( 'ECOOptions', )
   wasEnabled = octaConfig.ecoOptions.sflowEnableCollector
   octaConfig.ecoOptions.sflowEnableCollector = True
   maybeAddWarning( mode, not wasEnabled, sflowRestartWarning )
   mode.session_.gotoChildMode( childMode )

def noProviderSflowConfigMode( mode, args ):
   if octaConfig.ecoOptions is None:
      return
   wasEnabled = octaConfig.ecoOptions.sflowEnableCollector
   octaConfig.ecoOptions.sflowEnableCollector = False
   octaConfig.ecoOptions.sflowAddress = SflowDefaults.address
   octaConfig.ecoOptions.sflowPort = SflowDefaults.port
   octaConfig.ecoOptions.sflowDomain = SflowDefaults.domain
   octaConfig.ecoOptions.sflowFlowTableSize = SflowDefaults.flowTableSize
   octaConfig.ecoOptions.sflowFlowLifetime = SflowDefaults.flowLifetime
   maybeAddWarning( mode, wasEnabled, sflowRestartWarning )

def _setSflowListener( mode, address, port ):
   previousAddress = octaConfig.ecoOptions.sflowAddress
   previousPort = octaConfig.ecoOptions.sflowPort
   octaConfig.ecoOptions.sflowAddress = address
   octaConfig.ecoOptions.sflowPort = port
   maybeAddWarning( mode, previousAddress != address or previousPort != port,
                    sflowRestartWarning )

def setSflowListener( mode, args ):
   # [default] listener <addr> [ port <port> ]
   address = args.get( 'ADDRESS', SflowDefaults.address )
   port = args.get( 'PORT', SflowDefaults.port )
   _setSflowListener( mode, address, port )

def defaultSflowListenerHandler( mode, args ):
   _setSflowListener( mode, SflowDefaults.address, SflowDefaults.port )

def _setSflowDomain( mode, domain ):
   previousDomain = octaConfig.ecoOptions.sflowDomain 
   octaConfig.ecoOptions.sflowDomain = domain
   maybeAddWarning( mode, previousDomain != domain, sflowRestartWarning)

def setSflowDomain( mode, args ):
   # [default] domain NAME
   _setSflowDomain( mode, args.get( 'NAME', SflowDefaults.domain ) )

def defaultSflowDomainHandler( mode, args ):
   _setSflowDomain( mode, SflowDefaults.domain )

def _setSflowSamples( mode, size, lifetime ):
   previousSize = octaConfig.ecoOptions.sflowFlowTableSize
   previousLifetime = octaConfig.ecoOptions.sflowFlowLifetime
   if size is not None:
      octaConfig.ecoOptions.sflowFlowTableSize = size
   if lifetime is not None:
      octaConfig.ecoOptions.sflowFlowLifetime = lifetime
   maybeAddWarning( mode, previousSize != size or previousLifetime != lifetime,
                    sflowRestartWarning )

def setSflowSamples( mode, args ):
   # [default] samples [ table size SIZE ] [ lifetime N seconds ]
   _setSflowSamples( mode, args.get( 'SIZE' ), args.get( 'LIFETIME' ) )

def defaultSflowSamplesHandler( mode, args ):
   _setSflowSamples( mode, SflowDefaults.flowTableSize, SflowDefaults.flowLifetime )
 
# ------------------------------------------------------
# provider http-commands
# ------------------------------------------------------

class ProviderHttpCommandsConfigMode( ProviderHttpCommandsMode,
      BasicCli.ConfigModeBase ):
   """CLI configuration submode 'provider http-commands'."""

   name = 'Provider for Http commands'

   def __init__( self, parent, session ):
      self.config_ = octaConfig
      ProviderHttpCommandsMode.__init__( self, "http-commands" )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoProviderHttpCommandsConfigMode( mode, args ):
   childMode = mode.childMode( ProviderHttpCommandsConfigMode )
   mode.session_.gotoChildMode( childMode )

def noProviderHttpCommandsConfigMode( mode, args ):
   noOpenConfigSandCountersEapiInterval( mode, args )

def noOpenConfigSandCountersEapiInterval( mode, args ):
   octaConfig.httpCommands.sandCounters.cpuCountersEapiInterval = 0

# ------------------------------------------------------
# provider isis
# ------------------------------------------------------
class ProviderIsisConfigMode( ProviderIsisMode, BasicCli.ConfigModeBase ):
   """CLI configuration submode 'provider isis'."""

   name = 'Provider for IS-IS'

   def __init__( self, parent, session ):
      self.config_ = octaConfig
      ProviderIsisMode.__init__( self, "isis" )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoProviderIsisConfigMode( mode, args ):
   childMode = mode.childMode( ProviderIsisConfigMode )
   mode.session_.gotoChildMode( childMode )

def noProviderIsisConfigMode( mode, args ):
   noIsisLinkStateDatabase( mode, args )

def isisLinkStateDatabase( mode, args ):
   octaConfig.isisOptions.linkStateDatabase = True

def noIsisLinkStateDatabase( mode, args ):
   octaConfig.isisOptions.linkStateDatabase = False

# ------------------------------------------------------
# provider ospf
# ------------------------------------------------------
class ProviderOspfConfigMode( ProviderOspfMode, BasicCli.ConfigModeBase ):
   """CLI configuration submode 'provider ospf'."""

   name = 'Provider for OSPFv2'

   def __init__( self, parent, session ):
      self.config_ = octaConfig
      ProviderOspfMode.__init__( self, "ospf" )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoProviderOspfConfigMode( mode, args ):
   childMode = mode.childMode( ProviderOspfConfigMode )
   mode.session_.gotoChildMode( childMode )

def noProviderOspfConfigMode( mode, args ):
   if octaConfig.ospfOptions:
      octaConfig.ospfOptions.linkStateDatabaseForVrf.clear()

def ospfLinkStateDatabase( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   octaConfig.ospfOptions.linkStateDatabaseForVrf.add( vrfName )

def noOspfLinkStateDatabase( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   if vrfName in octaConfig.ospfOptions.linkStateDatabaseForVrf:
      octaConfig.ospfOptions.linkStateDatabaseForVrf.remove( vrfName )

# ------------------------------------------------------
# provider bgp
# ------------------------------------------------------
class ProviderBgpConfigMode( ProviderBgpMode, BasicCli.ConfigModeBase ):
   """CLI configuration submode 'provider bgp'."""

   name = 'Provider for BGP'

   def __init__( self, parent, session ):
      ProviderBgpMode.__init__( self, "bgp" )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class ProviderBgpRibConfigMode( ProviderBgpRibMode, BasicCli.ConfigModeBase ):
   """CLI configuration submode 'bgp-rib'."""

   name = 'BGP Rib'

   def __init__( self, parent, session ):
      self.config_ = octaConfig
      ProviderBgpRibMode.__init__( self, "bgp-rib" )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoProviderBgpConfigMode( mode, args ):
   childMode = mode.childMode( ProviderBgpConfigMode )
   mode.session_.gotoChildMode( childMode )

def gotoProviderBgpRibConfigMode( mode, args ):
   childMode = mode.childMode( ProviderBgpRibConfigMode )
   mode.session_.gotoChildMode( childMode )

def noProviderBgpConfigMode( mode, args ):
   noProviderBgpRibConfigMode( mode, args )

def noProviderBgpRibConfigMode( mode, args ):
   noBgpRibIpv4Unicast( mode, args )
   noBgpRibIpv6Unicast( mode, args )

def bgpRibIpv4Unicast( mode, args ):
   octaConfig.bgpRibIpv4Unicast = True

def noBgpRibIpv4Unicast( mode, args ):
   octaConfig.bgpRibIpv4Unicast = False

def bgpRibIpv6Unicast( mode, args ):
   octaConfig.bgpRibIpv6Unicast = True

def noBgpRibIpv6Unicast( mode, args ):
   octaConfig.bgpRibIpv6Unicast = False

# ------------------------------------------------------
# provider macsec
# ------------------------------------------------------
class ProviderMacsecConfigMode( ProviderMacsecMode, BasicCli.ConfigModeBase ):
   """CLI configuration submode 'provider macsec'."""

   name = 'Provider for Macsec'

   def __init__( self, parent, session ):
      self.config_ = octaConfig
      ProviderMacsecMode.__init__( self, "macsec" )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoProviderMacsecConfigMode( mode, args ):
   childMode = mode.childMode( ProviderMacsecConfigMode )
   mode.session_.gotoChildMode( childMode )

def noProviderMacsecConfigMode( mode, args ):
   octaConfig.macsecOptions = Tac.Value( "Octa::MacsecOptions" )

def _setMacsecOptions( mka, interfaces ):
   octaConfig.macsecOptions = Tac.Value( "Octa::MacsecOptions", mka=mka,
                                         interfaces=interfaces )

def setMacsecOptions( mode, args ):
   mka = octaConfig.macsecOptions.mka
   interfaces = octaConfig.macsecOptions.interfaces
   if 'mka' in args:
      mka = True
   elif 'interfaces' in args:
      interfaces = True
   _setMacsecOptions( mka, interfaces )

def resetMacsecOptions( mode, args ):
   mka = octaConfig.macsecOptions.mka
   interfaces = octaConfig.macsecOptions.interfaces
   if 'mka' in args:
      mka = False
   elif 'interfaces' in args:
      interfaces = False
   _setMacsecOptions( mka, interfaces )

# ------------------------------------------------------
# provider configuration
# ------------------------------------------------------
class ProviderConfigurationConfigMode( ProviderConfigurationMode,
                                       BasicCli.ConfigModeBase ):
   """CLI configuration submode 'provider configuration'."""

   name = 'Provider for Configuration'

   def __init__( self, parent, session ):
      self.config_ = gnmiConfig
      ProviderConfigurationMode.__init__( self, "configuration" )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoProviderConfigurationConfigMode( mode, args ):
   childMode = mode.childMode( ProviderConfigurationConfigMode )
   mode.session_.gotoChildMode( childMode )

def noProviderConfigurationConfigMode( mode, args ):
   gnmiConfig.saveSessionDiffs = False
   gnmiConfig.maxHistorySize = gnmiConfig.defaultMaxHistorySize

def configurationUpdates( mode, args ):
   gnmiConfig.saveSessionDiffs = True

def noConfigurationUpdates( mode, args ):
   gnmiConfig.saveSessionDiffs = False

def setHistoryLimit( mode, args ):
   gnmiConfig.maxHistorySize = args[ "HISTORY_LIMIT" ]

def noSetHistoryLimit( mode, args ):
   gnmiConfig.maxHistorySize = gnmiConfig.defaultMaxHistorySize
   
# ------------------------------------------------------
# models
# ------------------------------------------------------

class ModelsConfigMode( ModelsMode,
      BasicCli.ConfigModeBase ):
   """CLI configuration submode 'models'."""

   name = 'Configure YANG paths'

   def __init__( self, parent, session ):
      self.config_ = octaConfig
      ModelsMode.__init__( self, 'models' )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoModelsConfigMode( mode, args ):
   childMode = mode.childMode( ModelsConfigMode )
   mode.session_.gotoChildMode( childMode )

def noModelsConfigMode( mode, args ):
   octaConfig.yangPaths.clear()

def addModelPath( mode, args ):
   path = normalizePath( args[ 'PATH' ] )
   octaConfig.yangPaths[ path ] = 'disabled' not in args

def noModelPath( mode, args ):
   path = normalizePath( args[ 'PATH' ] )
   del octaConfig.yangPaths[ path ]

def normalizePath( path ):
   # Remove quotes
   path = path.replace( '"', '' )
   path = path.replace( "'", "" )
   # Remove trailing slashes
   path = path.rstrip( '/' )
   # Always have leading slash
   if not path.startswith( '/' ):
      path = '/' + path
   return path

# ------------------------------------------------------
# leaf default-value visible
# ------------------------------------------------------

def showDefaultLeafs( mode, args ):
   octaConfig.defaultLeafVisible = True

def noShowDefaultLeafs( mode, args ):
   octaConfig.defaultLeafVisible = False

# ------------------------------------------------------
# modules
# ------------------------------------------------------

supportedModuleGroups = { 'arista-segmentation' }

class ModulesConfigMode( ModulesMode,
      BasicCli.ConfigModeBase ):
   """CLI configuration submode 'modules'."""

   name = 'Configure YANG modules'

   def __init__( self, parent, session ):
      self.config_ = octaConfig
      ModulesMode.__init__( self, 'modules' )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoModulesConfigMode( mode, args ):
   childMode = mode.childMode( ModulesConfigMode )
   mode.session_.gotoChildMode( childMode )

def noModulesConfigMode( mode, args ):
   octaConfig.exclusiveModuleGroups.clear()

def addModuleGroup( mode, args ):
   moduleGroup = args[ 'MODULE_GROUP' ]
   octaConfig.exclusiveModuleGroups.add( moduleGroup )
   if moduleGroup not in supportedModuleGroups:
      mode.addWarning( f'Exclusive module group "{moduleGroup}" is not supported' )
   warnIfAgentRunning( mode, 'module' )

def noModuleGroup( mode, args ):
   moduleGroup = args[ 'MODULE_GROUP' ]
   if moduleGroup not in octaConfig.exclusiveModuleGroups:
      return
   octaConfig.exclusiveModuleGroups.remove( moduleGroup )
   warnIfAgentRunning( mode, 'module' )

def running():
   return gnmiConfig.enabled and octaConfig.enabled

agentRestartWarning = 'The %s agent needs to be restarted ' \
                      'in order to have %s changes take effect. ' \
                      'This can be done by running \'agent %s terminate\'.'

def warnIfAgentRunning( mode, configName ):
   agentName = ""
   if gnmiConfig.enabled:
      agentName = "OpenConfig"
      if octaConfig.enabled:
         agentName = "Octa"
   if agentName:
      mode.addWarning( agentRestartWarning % ( agentName, configName, agentName ) )

# ------------------------------------------------------
# postCommitHander for OC config diff telemetry
# ------------------------------------------------------
def getSessionConfigDiffFromGit( mode, sessionName ):
   commit = GitLib.getGitCommits(
         mode.entityManager.sysname(),
         trailerKeys=(
            'Session-name',
            'Commit-type',
            'Timestamp-ns' ),
         maxNumEntries=1 )
   if not commit:
      return None

   if ( commit[ 0 ][ 'trailers' ][ 'Session-name' ] != sessionName or
         commit[ 0 ][ 'trailers' ][ 'Commit-type' ] != 'post-commit' ):
      return None

   # get the diff of the commit
   sessionDiff = GitLib.gitDiff( mode.entityManager.sysname(),
         commit[ 0 ][ 'commitHash' ] )
   if not sessionDiff:
      # nothing changed or unable to compute diff
      return None

   return commit[ 0 ][ 'trailers' ][ 'Timestamp-ns' ], commit[ 0 ][ 'author' ], \
          sessionDiff

def maybeSaveConfigDiff( mode, sessionName ):
   if not gnmiConfig.saveSessionDiffs:
      # We are not enabling config diffs; remove all diffs.
      runningConfigDiff.diff.clear()
      return

   # See if we need to remove entries from runningConfigDiff.diff
   # due to maxHistorySize
   if len( runningConfigDiff.diff ) > gnmiConfig.maxHistorySize:
      numRemove = len( runningConfigDiff.diff ) - gnmiConfig.maxHistorySize
      # runningConfigDiff.diff is an ordered collection.
      # Remove the first numRemoves entries from this.
      for ts in list( runningConfigDiff.diff )[ : numRemove ]:
         del runningConfigDiff.diff[ ts ]

   if cliConfig.maxCheckpoints <= 1:
      # we can't compute any diffs since no checkpoints are being created
      return

   result = getSessionConfigDiffFromGit( mode, sessionName )
   if result is None:
      return
   timestamp, username, sessionDiff = result

   # Now we know that we indeed have a diff that needs to be
   # stored in runningConfigDiff.diff.
   if len( runningConfigDiff.diff ) == gnmiConfig.maxHistorySize:
      # runningConfigDiff.diff is an ordered collection.
      # Removing the first entry will do.
      ts = list( runningConfigDiff.diff )[ 0 ]
      del runningConfigDiff.diff[ ts ]

   # Size of the diff.
   sessionDiffSize = len( sessionDiff )
   displayedDiffSize = sessionDiffSize

   # Limit the diff string to 128KB.
   if sessionDiffSize > MAX_DIFF_SIZE:
      sessionDiff = f'** Truncated to 128KB **\n{sessionDiff[ : MAX_DIFF_SIZE ]}'
      displayedDiffSize = MAX_DIFF_SIZE

   # Truncate old entries based on aggregate config diff size.
   # Part 1: Determine the aggregate config diff size.
   aggregateDiffSize = \
         sum( len( d.changes ) for d in runningConfigDiff.diff.values() )
   aggregateDiffSize += displayedDiffSize

   # Truncate old entries based on aggregate config diff size.
   # Part 2: Remove old entries until aggregate size is <= 800KB.
   # runningConfigDiff.diff is an ordered collection so the
   # loop should start with the oldest and move to newest.
   for ts, diffData in list( runningConfigDiff.diff.items() ):
      if aggregateDiffSize <= MAX_AGG_DIFF_SIZE:
         break
      aggregateDiffSize -= len( diffData.changes )
      del runningConfigDiff.diff[ ts ]

   # We have an internal attrLog limit of ~32KB for nominals/ordinals and
   # for initial construction of entities.  Individual changes to a Tac::String
   # type attribute in an entity has an attrLog limit of ~10MB.
   # runningConfigDiff.diff is a collection of entities.  So, if the
   # config diff size exceeds 30KB (2KB slack for metadata), we construct
   # the entity with empty string as the config diff attribute and then
   # later individually update the attribute to the actual diff string.
   # This has a small drawback (when diff size exceeds 30KB) :
   # - Clients subscribing to the path might see an empty diff string initially
   #   in the first message and then see the actual diff string in a separate
   #   message.  AirStream might coalesce this into a single message (coalescing
   #   is not guaranteed).
   if len( sessionDiff ) > MAX_ENTITY_DIFF_SIZE:
      configDiff = runningConfigDiff.diff.newMember( int( timestamp ),
            username, sessionName, sessionDiffSize, "" )
      configDiff.changes = sessionDiff
   else:
      runningConfigDiff.diff.newMember( int( timestamp ),
            username, sessionName, sessionDiffSize, sessionDiff )

def Plugin( entityManager ):
   global octaConfig, gnmiConfig, cliConfig, runningConfigDiff

   octaConfig = ConfigMount.mount( entityManager, "mgmt/octa/config",
                                   "Octa::Config", "w" )
   gnmiConfig = ConfigMount.mount( entityManager, "mgmt/gnmi/config",
                                   "Gnmi::Config", "w" )
   cliConfig = LazyMount.mount( entityManager,
                                "cli/session/input/config",
                                "Cli::Session::CliConfig", "r" )
   runningConfigDiff = LazyMount.mount( entityManager,
                                        "openconfig/session/history/status",
                                        "OpenConfig::SessionHistoryStatus", "w" )

   CliSession.registerPostCommitHandler( maybeSaveConfigDiff )
