# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ArnetModel
import CliModel
import Tac
import re
import datetime
from TableOutput import createTable, Format

statusEnumToString = {
   "active" : "Active",
   "standby" : "Standby",
   "disabled" : "Disabled",
   "ok" : "Ok",
   "failed" : "Failed",
   "unknown" : "Unknown",
   "poweringOn" : "Powering On",
   "poweredOff" : "Powered Off",
   "upgradingFpga" : "Upgrading Fpga",
   "disabledUntilSystemUpgrade" : "System Software Upgrade Required",
}

_powerOffShortReasonToReasonEnumMap = {
      "userConfig" : "user configuration",
      "ejectorOpen" : "ejector open",
      "overheat" : "critical system temperature",
      "tempSensorGroupFailure" : "card temp sensor group failure",
      "insufficientFans" : "insufficient fans",
      "incompatibleFan" : "incompatible fan",
      "powerOffNotSupported" : "power off not supported",
      "SSU" : "SSU",
      "slotUnsupported" : "slot unsupported",
      "insufficientPower" : "insufficient power",
      "cardModelIncompatible" : "card model incompatible",
}

class Module( CliModel.Model ):
   portCount = CliModel.Int( help="Number of ports" )
   typeDescription = CliModel.Str( help="Verbose description of the Module's type" )
   modelName = CliModel.Str( help="Module's model string" )
   serialNumber = CliModel.Str( help="Serial number" )
   minMacAddress = ArnetModel.MacAddress(
         help="Allocated Module's MAC address range. Inclusive lower bound" )
   maxMacAddress = ArnetModel.MacAddress(
         help="Allocated Module's MAC address range. Inclusive upper bound" )
   numMacAddresses = CliModel.Int( help="Number of MAC addresses in range" )
   hardwareRev = CliModel.Str( help="Hardware Revision" )
   softwareRev = CliModel.Str( help="Software Revision" )
   status = CliModel.Enum(
         help="Module's status. For all Modules: ok-Module operating properly. " + \
              "failed-Module in a failed state. unknown-Unable to determine "    + \
              "Module's state. poweringOn-Module is powering on. "               + \
              "disabledUntilSystemUpgrade-Module is disabled until system "      + \
              "software upgrade. "                                               + \
              "poweredOff-Module is powered off. For supervisor Modules: "       + \
              "active-Supervisor is currently active. standby-Supervisor is in " + \
              "standby mode. disabled-Supervisor is disabled.",
         values=( "ok", "failed", "unknown", "poweringOn",
                  "poweredOff", "active", "standby", "disabled", "upgradingFpga",
                  "disabledUntilSystemUpgrade", )
         )
   uptime = CliModel.Float( help="UTC timestamp of module power-on completion",
                            optional=True )
   powerOffReason = CliModel.Enum( values=( "userConfig", "ejectorOpen",
                                            "overheat", "tempSensorGroupFailure",
                                            "insufficientFans", "incompatibleFan",
                                            "powerOffNotSupported", "SSU",
                                            "slotUnsupported", "unknown",
                                            "insufficientPower",
                                            "cardModelIncompatible", ),
      help='Reason for being powered off', optional=True )

class Modules( CliModel.Model ):
   modules = CliModel.Dict( help="Mapping between Module's name and its data",
                            valueType=Module,
                            keyType=str )
   redundancyProtocol = CliModel.Enum( help="informs current redundancy protocol",
                                       values=( 'unknownProtocol', 'sso', 'rpr',
                                                'simplex' ) )
   redundancyMode = CliModel.Enum( help="informs current redundancy mode",
                                   values=('active', 'standby', 'disabled',
                                           'initializing', 'switchover',
                                           'unknown' ) )

   def render( self ):
      modulesNotAvailable = ( self.redundancyMode == "standby" \
                              and self.redundancyProtocol == "rpr" ) or \
                            ( self.redundancyMode == "disabled" \
                              and self.redundancyProtocol == "simplex" )
      if modulesNotAvailable:
         print( "No module information is available. Please run this command from" +
                " the active supervisor." )
         return

      def sortKey( module ):
         label, tag = re.search( r"([a-zA-Z]*)(\d+)", module[ 0 ] ).groups()
         return ( label, int( tag ) )

      def makeTable( header ):
         t = createTable( header )
         f = Format( justify="left" )
         f.noPadLeftIs( True )
         f.padLimitIs( True )
         t.formatColumns( *[ f ] * len( header ) )
         return t

      def printTable( table ):
         # Strip whitespace from the end of the lines, because that's how it was
         # before and it's what some parsers expect. Also remove the last newline.
         print( '\n'.join( map( str.rstrip, table.output().split( '\n' )[ :-1 ] ) ) )

      table1 = makeTable( ( "Module", "Ports", "Card Type", "Model", "Serial No." ) )
      table2 = makeTable( ( "Module", "MAC addresses (count)", "Hw", "Sw" ) )
      table3 = makeTable( ( "Module", "Status", "Uptime", "Power off reason" ) )

      for card, m in sorted( self.modules.items(), key=sortKey ):
         table1.newRow( card, m.portCount, m.typeDescription, m.modelName,
                        m.serialNumber )

         if ( m.minMacAddress.stringValue != "00:00:00:00:00:00" and
              m.maxMacAddress.stringValue != "00:00:00:00:00:00" ):
            # pylint: disable-next=consider-using-f-string
            macAddrStr = "%s - %s (%d)" % ( m.minMacAddress, m.maxMacAddress,
                                            m.numMacAddresses )
         else:
            # Card has no associated mac addresses
            macAddrStr = ""
         table2.newRow( card, macAddrStr, m.hardwareRev, m.softwareRev )

         if m.uptime is not None:
            uptime = Tac.utcNow() - m.uptime
            td = datetime.timedelta( seconds=int( uptime ) )
            uptime = str( td )
         else:
            uptime = "Offline"

         powerOffReason = _powerOffShortReasonToReasonEnumMap.get( m.powerOffReason,
                                                                   "N/A" )
         table3.newRow( card, statusEnumToString[ m.status ],
                        uptime, powerOffReason )

      printTable( table1 )
      print()
      printTable( table2 )
      print()
      printTable( table3 )

