# Copyright (c) 2009 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import CliToken.Terminal
import CliToken.Reset
import ConfigMount
from CliModel import Model
from CliModel import Int
import ShowCommand
import TerminalUtil

cliConfig = None

#-----------------------------------------------------------------------------------
# [ no | default ] terminal length <numLines>
# For enable/unpriv mode
#-----------------------------------------------------------------------------------
lengthKwMatcher = CliMatcher.KeywordMatcher(
   'length', helpdesc='Configure the pagination length' )
numLinesMatcher = CliMatcher.IntegerMatcher(
   0, 32767,
   helpdesc='Terminal pager lines, disable the pager with value 0' )

def setSessionTerminalLength( mode, numLines=-1 ):
   mode.session_.terminalCtx_.terminalLengthOverrideIs( numLines )
   if not mode.session_.startupConfig():
      if numLines == -1:
         print( "Pagination automatic." )
      elif numLines == 0:
         print( "Pagination disabled." )
      else:
         # pylint: disable-next=consider-using-f-string
         print( "Pagination set to %d lines." % numLines )

class SessionTerminalLength( CliCommand.CliCommandClass ):
   syntax = '''terminal length <numLines>'''
   noOrDefaultSyntax = '''terminal length ...'''
   data = {
            'terminal': CliToken.Terminal.terminalKwForExec,
            'length': lengthKwMatcher,
            '<numLines>': numLinesMatcher
          }

   @staticmethod
   def handler( mode, args ):
      setSessionTerminalLength( mode, args[ '<numLines>' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setSessionTerminalLength( mode )

BasicCli.ExecMode.addCommandClass( SessionTerminalLength )

#-----------------------------------------------------------------------------------
# [ no | default ] terminal length <numLines>
# for configure mode
#-----------------------------------------------------------------------------------
def setGlobalTerminalLength( mode, args ):
   numLines = args.get( '<numLines>', -1 )
   cliConfig.terminalLengthOverride = numLines
   # Update terminal length either immediately or if in a cli session,
   # after commit.
   mode.session_.maybeCallConfigSessionOnCommitHandler(
      "terminal-length",
      lambda m, onSessionCommit: setSessionTerminalLength( m, numLines ) )

class ConfigTerminalLength( CliCommand.CliCommandClass ):
   syntax = '''terminal length <numLines>'''
   noOrDefaultSyntax = '''terminal length ...'''
   data = {
            'terminal': CliToken.Terminal.terminalKwForConfig,
            'length': lengthKwMatcher,
            '<numLines>': numLinesMatcher
          }
   handler = setGlobalTerminalLength
   noOrDefaultHandler = setGlobalTerminalLength

BasicCli.GlobalConfigMode.addCommandClass( ConfigTerminalLength )

#-----------------------------------------------------------------------------------
# [ no | default ] terminal width <numCols>
# For enable/unpriv mode
#-----------------------------------------------------------------------------------
widthKwMatcher = CliMatcher.KeywordMatcher(
   'width', helpdesc='Configure the terminal width' )
numColsMatcher = CliMatcher.IntegerMatcher( TerminalUtil.minTerminalWidth,
                                            TerminalUtil.maxTerminalWidth,
                                            helpdesc='Number of columns on screen' )

def setSessionTerminalWidth( mode, numCols=-1 ):
   mode.session_.terminalCtx_.terminalWidthOverrideIs( numCols )
   if numCols == -1:
      print( "Width automatic." )
   else:
      # pylint: disable-next=consider-using-f-string
      print( "Width set to %d columns." % numCols )

class SessionTerminalWidth( CliCommand.CliCommandClass ):
   syntax = '''terminal width <numCols>'''
   noOrDefaultSyntax = '''terminal width ...'''
   data = {
            'terminal': CliToken.Terminal.terminalKwForExec,
            'width': widthKwMatcher,
            '<numCols>': numColsMatcher
          }

   @staticmethod
   def handler( mode, args ):
      setSessionTerminalWidth( mode, args[ '<numCols>' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setSessionTerminalWidth( mode )

BasicCli.ExecMode.addCommandClass( SessionTerminalWidth )

#-----------------------------------------------------------------------------------
# [ no | default ] terminal width <numCols>
# for configure mode
#-----------------------------------------------------------------------------------
def setGlobalTerminalWidth( mode, args ):
   numCols = args.get( '<numCols>', -1 )
   cliConfig.terminalWidthOverride = numCols
   # Update terminal width either immediately or if in a cli session,
   # after commit.
   mode.session_.maybeCallConfigSessionOnCommitHandler(
      "terminal-width",
      lambda m, onSessionCommit: setSessionTerminalWidth( m, numCols ) )

class ConfigTerminalWidth( CliCommand.CliCommandClass ):
   syntax = '''terminal width <numCols>'''
   noOrDefaultSyntax = '''terminal width ...'''
   data = {
            'terminal': CliToken.Terminal.terminalKwForConfig,
            'width': widthKwMatcher,
            '<numCols>': numColsMatcher
          }

   handler = setGlobalTerminalWidth
   noOrDefaultHandler = setGlobalTerminalWidth

BasicCli.GlobalConfigMode.addCommandClass( ConfigTerminalWidth )

#-----------------------------------------------------------------------------------
# terminal reset
#-----------------------------------------------------------------------------------
resetKwMatcher = CliMatcher.KeywordMatcher( 'reset', helpdesc='Reset terminal' )

def terminalReset( mode, args ):
   import Tac # pylint: disable=import-outside-toplevel
   Tac.run( [ 'reset' ] )

class TerminalReset( CliCommand.CliCommandClass ):
   syntax = '''terminal reset'''
   data = {
            'terminal': CliToken.Terminal.terminalKwForExec,
            'reset': resetKwMatcher
          }
   handler = terminalReset

BasicCli.ExecMode.addCommandClass( TerminalReset )

#-----------------------------------------------------------------------------------
# reset
#-----------------------------------------------------------------------------------
class Reset( CliCommand.CliCommandClass ):
   syntax = '''reset'''
   data = { 'reset': CliToken.Reset.resetKwApi }
   handler = terminalReset
   hidden = True

BasicCli.ExecMode.addCommandClass( Reset )

#-----------------------------------------------------------------------------------
# [ no | default ] terminal dont-ask
#-----------------------------------------------------------------------------------
class TerminalDontAsk( CliCommand.CliCommandClass ):
   syntax = '''terminal dont-ask'''
   noOrDefaultSyntax = syntax
   data = {
            'terminal': CliToken.Terminal.terminalKwForExec,
            'dont-ask': 'Do not prompt for user confirmation'
          }

   @staticmethod
   def handler( mode, args ):
      mode.session.commandConfirmation_ = False

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.session.commandConfirmation_ = True

BasicCli.ExecMode.addCommandClass( TerminalDontAsk )

#-----------------------------------------------------------------------------------
# [ no | default ] terminal context help
#-----------------------------------------------------------------------------------
class TerminalContextHelp( CliCommand.CliCommandClass ):
   syntax = '''terminal context-help [ disabled ]'''
   data = {
            'terminal': CliToken.Terminal.terminalKwForExec,
            'context-help': 'Setup context help (?) and tab completion (tab)',
            'disabled': 'Disable context help (?) and tab completion (tab)'
          }

   @staticmethod
   def handler( mode, args ):
      enabled = 'disabled' not in args
      mode.session_.cliInput.keyBindingsEnabledIs( enabled )

BasicCli.ExecMode.addCommandClass( TerminalContextHelp )

#----------------------------------------------------------------------------------
# [ no | default ] terminal confirmation disabled
#----------------------------------------------------------------------------------
class TerminalConfirmation( CliCommand.CliCommandClass ):
   syntax = 'terminal confirmation disabled'
   noOrDefaultSyntax = 'terminal confirmation disabled'
   data = {
            'terminal': CliToken.Terminal.terminalKwForConfig,
            'confirmation': 'User confirmation of commands',
            'disabled': 'Do not prompt for user confirmation'
          }

   @staticmethod
   def handler( mode, args ):
      cliConfig.commandConfirmation = False

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cliConfig.commandConfirmation = True

BasicCli.GlobalConfigMode.addCommandClass( TerminalConfirmation )

#-----------------------------------------------------------------------------------
# "show terminal" that shows both length and width
#-----------------------------------------------------------------------------------
class TerminalDetails( Model ):
   length = Int( help="Length of the terminal in rows" )
   width = Int( help="Width of the terminal in columns" )

   def render( self ):
      # pylint: disable-next=bad-string-format-type,consider-using-f-string
      print( "Length: %d rows, Width: %d columns" % ( self.length, self.width ) )

class ShowTerminal( ShowCommand.ShowCliCommandClass ):
   syntax = 'show terminal'
   data = { 'terminal': 'Display terminal length/width' }
   cliModel = TerminalDetails

   @staticmethod
   def handler( mode, args ):
      numRows, numCols = TerminalUtil.ioctl_GWINSZ()[ 0 : 2 ]
      return TerminalDetails( length=numRows, width=numCols )

BasicCli.addShowCommandClass( ShowTerminal )

#-----------------------------------------------------------------------------------
# "show terminal" that shows both length and width
#-----------------------------------------------------------------------------------
class SessionExpandAliases( CliCommand.CliCommandClass ):
   syntax = 'terminal alias disabled'
   noOrDefaultSyntax = syntax
   data = {
            'terminal': CliToken.Terminal.terminalKwForExec,
            'alias': 'Configure aliases options',
            'disabled': 'Disable aliases for this session'
          }

   @staticmethod
   def handler( mode, args ):
      mode.session.expandAliases = False

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.session.expandAliases = True

BasicCli.ExecMode.addCommandClass( SessionExpandAliases )

#-----------------------------------------------------------------------------------
# Plugin func
#-----------------------------------------------------------------------------------
def Plugin( entityManager ):
   global cliConfig

   cliConfig = ConfigMount.mount( entityManager, "cli/config", "Cli::Config", "w" )
