# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
from CliPlugin import MplsModel
from CliPlugin.IntfCli import Intf
import CliPlugin.Ip6AddrMatcher
import CliPlugin.IpAddrMatcher
import CliPlugin.IpGenAddrMatcher
from CliPlugin.MplsCli import (
      impNullTunnelNode,
      intfValMatcher,
      labelsAdapter,
      labelStackKeywordNode,
      labelStackValNode,
      metricMatcher,
      metricValMatcher,
      mplsForShowNode,
      mplsNodeForConfig,
      _nexthopMatcher,
      _nexthopOnIntfMatcher,
      outLabelValMatcher,
      payloadTypeMatcher,
      payloadTypeNoBypassMatcher,
      popMatcher,
      staticMatcher,
      staticTunnelMatcher,
      swapLabelMatcher,
      topLabelMatcher,
      topLabelValMatcher,
      tunnelNameValMatcher,
      tunnelNode,
)
from CliPlugin.TunnelCli import (
      tunnelIndexMatcher,
      TunnelTableIdentifier,
      readMountTunnelTable,
)
from CliPlugin.BridgingCli import (
      MacAddrTableExprForConfig,
      staticKwMatcher as macStaticKwMatcher,
      matcherVlan as matcherVlanKw,
      MacAddr,
)
from CliPlugin.VlanCli import vlanIdMatcher
from CliToken.Bridging import nodeMacAddressTableDebugHidden
import ConfigMount
import FibUtils
import LazyMount
from MplsLib import tunTypeEnumDict
from MplsTypeLib import tunnelTypes
import ShowCommand
import SharedMem
import Smash
import Tac
from SrTePolicyCommonLib import srTePolicyStatusPath

MplsLabel = Tac.Type( 'Arnet::MplsLabel' )
VlanId = Tac.Type( 'Bridging::VlanId' )
FecType = Tac.Type( "Smash::Fib::AdjType" )

bgpLuLfibInput = None
debugLfib = None
lfibInfo = None
routing6Status = None
routingStatus = None
staticTunnelConfig = None
systemTunnelFib = None
dsfTunnelFib = None
gueTunnelFib = None
tunnelFib = None
srTePolicyStatus = None
srteForwardingStatus = None
rsvpLerTunnelTable = None
lfibVskOverrideConfig = None
lfibViaSetStatus = None

debugNodeHidden = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'debug', helpdesc='' ),
      hidden=True )
labelValMatcher = CliMatcher.IntegerMatcher(
      MplsLabel.unassignedMin, MplsLabel.max,
      helpdesc="MPLS label" )
vlanValMatcher = CliMatcher.IntegerMatcher(
      VlanId.min, VlanId.max,
      helpdesc="Identifier for a Virtual LAN" )
trueMatcher = CliMatcher.KeywordMatcher( 'true',
      helpdesc='Enable control-word' )
falseMatcher = CliMatcher.KeywordMatcher( 'false',
      helpdesc='Disable control-word' )
controlWordPresentMatcher = CliMatcher.KeywordMatcher( 'control-word-present',
      helpdesc='Enable/disable control-word' )
tunnelTypeValMatcher = CliMatcher.EnumMatcher( tunnelTypes )

#-----------------------------------------------------------
# [ no | default ] mpls debug ip-lookup-via LABEL VRF ( ipv4 | ipv6 )
#-----------------------------------------------------------
class MplsDebugIpLookupVia( CliCommand.CliCommandClass ):
   syntax = 'mpls debug ip-lookup-via LABEL VRF ( ipv4 | ipv6 )'
   noOrDefaultSyntax = 'mpls debug ip-lookup-via LABEL ...'
   data = {
      'mpls': mplsNodeForConfig,
      'debug': debugNodeHidden,
      'ip-lookup-via': 'IpLookupVia related (label, VRF, and address family)',
      'LABEL': labelValMatcher,
      'VRF': CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
                                        helpname='WORD',
                                        helpdesc='Name of VRF' ),
      'ipv4': 'IPv4 related',
      'ipv6': 'IPv6 related',
   }
   handler = "MplsDebugCliHandler.MplsDebugIpLookupVia_handler"
   noOrDefaultHandler = "MplsDebugCliHandler.MplsDebugIpLookupVia_noOrDefaultHandler"

#-----------------------------------------------------------
# [ no | default ] mpls debug vlan-via LABEL VLAN
#                  control-word-present ( true | false )
#-----------------------------------------------------------
class MplsDebugVlanVia( CliCommand.CliCommandClass ):
   syntax = 'mpls debug vlan-via LABEL VLAN control-word-present ( true | false )'
   noOrDefaultSyntax = 'mpls debug vlan-via LABEL ...'
   data = {
      'mpls': mplsNodeForConfig,
      'debug': debugNodeHidden,
      'vlan-via': 'VlanVia related (label, vlanId, and control word)',
      'LABEL': labelValMatcher,
      'VLAN': vlanValMatcher,
      'control-word-present': controlWordPresentMatcher,
      'true': trueMatcher,
      'false': falseMatcher
   }
   handler = "MplsDebugCliHandler.MplsDebugVlanVia_handler"
   noOrDefaultHandler = "MplsDebugCliHandler.MplsDebugVlanVia_noOrDefaultHandler"

#-----------------------------------------------------------
# [ no | default ] mpls debug vlan-flood-via LABEL VLAN
#                  control-word-present ( true | false )
#-----------------------------------------------------------
class MplsDebugVlanFloodVia( CliCommand.CliCommandClass ):
   syntax = 'mpls debug vlan-flood-via LABEL VLAN control-word-present (true|false)'
   noOrDefaultSyntax = 'mpls debug vlan-flood-via LABEL ...'

   data = {
      'mpls': mplsNodeForConfig,
      'debug': debugNodeHidden,
      'vlan-flood-via': 'VlanFloodVia related (label, vlanId, and control word)',
      'LABEL': labelValMatcher,
      'VLAN': vlanValMatcher,
      'control-word-present': controlWordPresentMatcher,
      'true': trueMatcher,
      'false': falseMatcher
   }
   handler = "MplsDebugCliHandler.MplsDebugVlanFloodVia_handler"
   noOrDefaultHandler = \
      "MplsDebugCliHandler.MplsDebugVlanFloodVia_noOrDefaultHandler"

# -------------------------------------------------------------
# [ no | default ] mpls debug es-filter-via LABEL INTF
# -------------------------------------------------------------
class MplsDebugEthernetSegmentFilterVia( CliCommand.CliCommandClass ):
   syntax = 'mpls debug es-filter-via LABEL INTF'
   noOrDefaultSyntax = 'mpls debug es-filter-via LABEL ...'
   data = {
      'mpls': mplsNodeForConfig,
      'debug': debugNodeHidden,
      'es-filter-via' : 'EthernetSegmentFilterVia related (label and interface)',
      'LABEL': labelValMatcher,
      'INTF': Intf.matcher,
   }
   handler = "MplsDebugCliHandler.MplsDebugEthernetSegmentFilterVia_handler"
   noOrDefaultHandler = \
      "MplsDebugCliHandler.MplsDebugEthernetSegmentFilterVia_noOrDefaultHandler"

BasicCli.GlobalConfigMode.addCommandClass( MplsDebugIpLookupVia )
BasicCli.GlobalConfigMode.addCommandClass( MplsDebugVlanVia )
BasicCli.GlobalConfigMode.addCommandClass( MplsDebugVlanFloodVia )
BasicCli.GlobalConfigMode.addCommandClass( MplsDebugEthernetSegmentFilterVia )

#-------------------------------------------------------------------------
# [no|default] mac address-table debug static <mac_addr> vlan <vlan_id> label
# <label_value> tunnel <tunnel_type> <tunnel_index>  (hidden from user)
#-------------------------------------------------------------------------
def getTunnelIntfId( tunnelType, tunnelIndex ):
   tunTypeEnum = tunTypeEnumDict[ tunnelType ]
   tunnelId = Tac.Type( 'Tunnel::TunnelTable::TunnelId' ).convertToTunnelValue(
                 tunTypeEnum, tunnelIndex )
   tunnelIntfId = Tac.Type( 'Arnet::DynamicTunnelIntfId' ).tunnelIdToIntfId(
                     tunnelId )
   return tunnelIntfId

tunnelTypeMatcher = CliMatcher.EnumMatcher( {
   'static': 'Static tunnel',
   'ldp': 'LDP tunnel',
   'segment-routing': 'Segment Routing tunnel',
} )

class MacAddrDebugStaticTunnel( CliCommand.CliCommandClass ):
   syntax = ( 'MAC_ADDR_TABLE debug static MACADDR vlan VLAN_ID label LABEL '
              'tunnel TUN_TYPE TUN_IDX' )
   noOrDefaultSyntax = syntax
   data = {
      'MAC_ADDR_TABLE': MacAddrTableExprForConfig,
      'debug': nodeMacAddressTableDebugHidden,
      'static': macStaticKwMatcher,
      'MACADDR': MacAddr.MacAddrMatcher(),
      'vlan': matcherVlanKw,
      'VLAN_ID': vlanIdMatcher,
      'label': 'Specify the MPLS label to push',
      'LABEL': labelValMatcher,
      'INTF': Intf.matcher,
      'tunnel': 'Tunnel interface',
      'TUN_TYPE': tunnelTypeMatcher,
      'TUN_IDX': tunnelIndexMatcher,
   }
   hidden = True
   handler = "MplsDebugCliHandler.MacAddrDebugStaticTunnel_handler"
   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( MacAddrDebugStaticTunnel )

#------------------------------------------
#  [ no | default ] mpls static top-label LABEL debug [ backup ]
#                   ( ( tunnel-type TTYPE tunnel-index TINDEX ) |
#                   ( ( intf intfAddr ) | nhaddr )
#                   [ ( swap-label OUT_LABEL)
#                   | ( pop [ payload-type PTYPE ] ) ]
#                   [ metric METRIC ]
# ( Hidden )
#------------------------------------------
class MplsStaticTopLabelTunnel( CliCommand.CliCommandClass ):
   syntax = ( 'mpls static top-label LABEL debug [ backup ]'
              '( ( tunnel-type TTYPE tunnel-index TINDEX ) |'
              '( ( INTF ADDR_ON_INTF ) | ADDR ) )'
              '[ ( swap-label OUT_LABEL) '
              '| ( pop [ payload-type PTYPE ] ) ] '
              '[ metric METRIC ]' )
   noOrDefaultSyntax = syntax
   data = {
      'mpls': mplsNodeForConfig,
      'static': staticMatcher,
      'top-label': topLabelMatcher,
      'LABEL': topLabelValMatcher,
      'debug': debugNodeHidden,
      'backup': 'backup static route via',
      'tunnel-type': 'Tunnel type',
      'TTYPE': tunnelTypeValMatcher,
      'tunnel-index': 'Tunnel index',
      'TINDEX': tunnelIndexMatcher,
      'INTF': intfValMatcher,
      'ADDR_ON_INTF': _nexthopOnIntfMatcher,
      'ADDR': _nexthopMatcher,
      'swap-label': swapLabelMatcher,
      'OUT_LABEL': outLabelValMatcher,
      'pop': popMatcher,
      'payload-type': payloadTypeMatcher,
      'PTYPE': payloadTypeNoBypassMatcher,
      'metric': metricMatcher,
      'METRIC': metricValMatcher,
   }
   handler = "MplsDebugCliHandler.MplsStaticTopLabelTunnel_handler"
   noOrDefaultHandler = \
      "MplsDebugCliHandler.MplsStaticTopLabelTunnel_noOrDefaultHandler"

BasicCli.GlobalConfigMode.addCommandClass( MplsStaticTopLabelTunnel )

#----------------------------------------------------
# show mpls lfib input bgp labeled-unicast
#----------------------------------------------------
class ShowMplsLfibInputBgpLuCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls lfib input bgp labeled-unicast'
   data = {
      'mpls': mplsForShowNode,
      'lfib': 'Show MPLS LFIB',
      'input' : 'Show MPLS LFIB inputs',
      'bgp' : 'Show BGP LFIB inputs',
      'labeled-unicast' : 'Show BGP labeled-unicast LFIB inputs',
   }
   hidden = True
   cliModel = MplsModel.MplsRoutes
   handler = "MplsDebugCliHandler.ShowMplsLfibInputBgpLuCmd_handler"

BasicCli.addShowCommandClass( ShowMplsLfibInputBgpLuCmd )

#------------------------------------------
# [ no | default ] mpls static top-label LABEL debug
#                  fec-type FTYPE fec-index FINDEX
#                  ( ( pop [ payload-type PTYPE ] )
#                  | ( forward ) )
# ( Hidden )
#------------------------------------------
def getFecTypeTokens():
   tokens = {}
   shortNameToCompleteName = {}
   completeNameToShortName = {}
   for adj in FecType.attributes:
      if adj.endswith( 'Adj' ):
         trimmedAdj = adj.lower().replace( "adj", "" )
         shortNameToCompleteName[ trimmedAdj ] = adj
         completeNameToShortName[ adj ] = trimmedAdj
         # pylint: disable-next=consider-using-f-string
         tokens[ trimmedAdj ] = 'Match %s fec type' % trimmedAdj

   return tokens, shortNameToCompleteName, completeNameToShortName

fecTypes, shortFecTypeToCompleteType, _ = getFecTypeTokens()

class MplsStaticTopLabelFec( CliCommand.CliCommandClass ):
   syntax = ( 'mpls static top-label LABEL debug '
              'fec-type FTYPE fec-index FINDEX '
              '( ( pop [ payload-type PTYPE ] ) '
              '| ( forward ) )' )
   noOrDefaultSyntax = 'mpls static top-label LABEL debug ...'
   data = {
      'mpls': mplsNodeForConfig,
      'static': staticMatcher,
      'top-label': topLabelMatcher,
      'LABEL': topLabelValMatcher,
      'debug': debugNodeHidden,
      'fec-type': 'FEC type',
      'FTYPE': CliMatcher.EnumMatcher( fecTypes ),
      'fec-index': 'FEC index',
      'FINDEX': CliMatcher.IntegerMatcher( 1, 2 ** 56 - 1, helpdesc='FEC index' ),
      'pop': popMatcher,
      'payload-type': payloadTypeMatcher,
      'PTYPE': payloadTypeNoBypassMatcher,
      'forward': 'Forward the top label',
   }
   handler = "MplsDebugCliHandler.MplsStaticTopLabelFec_handler"
   noOrDefaultHandler = \
      "MplsDebugCliHandler.MplsStaticTopLabelFec_noOrDefaultHandler"

BasicCli.GlobalConfigMode.addCommandClass( MplsStaticTopLabelFec )

#-------------------------------------------------------------------------
# The "[no] mpls debug [ backup ] tunnel static <name> <endpoint> <nexthop>
# <interface> label-stack <label-stack>" command
# Note: This command shares syntax with MplsStaticTunnel in MplsCli.py. If
# updating it here, please update the non-debug command accordingly.
#-------------------------------------------------------------------------
class MplsDebugStaticTunnel( CliCommand.CliCommandClass ):
   syntax = '''mpls debug [ backup ] tunnel static NAME
               ( ( ( ( ( TEP_V4_ADDR | TEP_V4_PREFIX ) NEXTHOP_V4 ) |
                     ( ( TEP_V6_ADDR | TEP_V6_PREFIX ) NEXTHOP_V6 ) )
                   INTF ) |
                 ( ( resolving R_TEP ) |
                   ( resolving-tunnel RT_TEP tunnel-type RT_TYPE
                     tunnel-index RT_INDEX ) ) )
               ( ( label-stack { LABELS } ) | imp-null-tunnel )
            '''

   noOrDefaultSyntax = '''mpls debug [ backup ] tunnel static NAME ...'''

   data = {
      'mpls': mplsNodeForConfig,
      'debug': debugNodeHidden,
      'backup': 'Backup static tunnel via',
      'tunnel': tunnelNode,
      'static': staticTunnelMatcher,
      'NAME': tunnelNameValMatcher,
      'TEP_V4_ADDR': CliPlugin.IpAddrMatcher.ipAddrMatcher,
      'TEP_V4_PREFIX': CliPlugin.IpAddrMatcher.ipPrefixExpr(
         'IP address',
         "Subnet's mask value",
         'IP address with mask length',
         overlap=CliPlugin.IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO
      ),
      'NEXTHOP_V4': CliPlugin.IpAddrMatcher.ipAddrMatcher,
      # FIXME BUG200408: Un-hide IPv6 endpoints and nexthop in "mpls static tunnel"
      'TEP_V6_ADDR': CliCommand.Node(
         matcher=CliPlugin.Ip6AddrMatcher.ip6AddrMatcher,
         hidden=True ),
      'TEP_V6_PREFIX': CliCommand.Node(
         matcher=CliPlugin.Ip6AddrMatcher.ip6PrefixMatcher,
         hidden=True ),
      'NEXTHOP_V6': CliCommand.Node(
         matcher=CliPlugin.Ip6AddrMatcher.ip6AddrMatcher,
         hidden=True ),
      'INTF': intfValMatcher,
      'resolving': 'Static tunnel over resolving prefix',
      'R_TEP': CliPlugin.IpGenAddrMatcher.IpGenPrefixMatcher( 'Resolving prefix' ),
      'resolving-tunnel': 'Static tunnel over tunnel',
      'RT_TEP': CliPlugin.IpGenAddrMatcher.IpGenPrefixMatcher(
         'Resolving tunnel prefix' ),
      'tunnel-type': 'Tunnel type',
      'RT_TYPE': tunnelTypeValMatcher,
      'tunnel-index': 'Tunnel index',
      'RT_INDEX': tunnelIndexMatcher,
      'label-stack': labelStackKeywordNode,
      'LABELS': labelStackValNode,
      'imp-null-tunnel': impNullTunnelNode,
   }

   adapter = labelsAdapter
   handler = "MplsDebugCliHandler.MplsDebugStaticTunnel_handler"
   noOrDefaultHandler = \
      "MplsDebugCliHandler.MplsDebugStaticTunnel_noOrDefaultHandler"

BasicCli.GlobalConfigMode.addCommandClass( MplsDebugStaticTunnel )

def Plugin( entityManager ):
   global bgpLuLfibInput
   global debugLfib
   global lfibInfo
   global routing6Status
   global routingStatus
   global staticTunnelConfig
   global systemTunnelFib
   global dsfTunnelFib
   global gueTunnelFib
   global tunnelFib
   global srTePolicyStatus
   global srteForwardingStatus
   global lfibVskOverrideConfig
   global lfibViaSetStatus
   global rsvpLerTunnelTable
   # Mpls::LfibSysdbStatus is essentially a Sysdb-compatible version of
   # Mpls::LfibStatus. (Currently, we only care about the ipLookupVia field in it.)
   # This way, we can write to the debugLfib in CLI, and then have a SyncSm to
   # copy the relevant info from this debugLfib into a Mpls::LfibStatus. This
   # Mpls::LfibStatus will not be persistently stored, but will just be passed
   # into the LfibSmashMergeSm downstream. This way, we can add and remove
   # entries in the debugLfib and have changes propagate through the system.
   # We're only interested in the IpLookupVia for now. See aid/3901.
   smashEm = SharedMem.entityManager( sysdbEm=entityManager )
   readerInfo = Smash.mountInfo( 'reader' )
   bgpLuLfibInput = smashEm.doMount( "mpls/protoLfibInputDir/bgpLu",
                                     "Mpls::LfibStatus", readerInfo )
   debugLfib = ConfigMount.mount( entityManager, 'mpls/debug/lfib',
                                  'Mpls::LfibSysdbStatus', 'w' )
   lfibInfo = LazyMount.mount( entityManager, 'routing/mpls/lfibInfo',
                               'Mpls::LfibInfo', 'r' )
   routing6Status = smashEm.doMount( 'routing6/status', 'Smash::Fib6::RouteStatus',
                                     FibUtils.routeStatusInfo( 'reader' ) )
   routingStatus = smashEm.doMount( 'routing/status', 'Smash::Fib::RouteStatus',
                                    FibUtils.routeStatusInfo( 'reader' ) )
   staticTunnelConfig = ConfigMount.mount( entityManager, "tunnel/static/config",
                                          "Tunnel::Static::Config", "w" )
   systemTunnelFib = smashEm.doMount(
      'tunnel/tunnelFib', 'Tunnel::TunnelFib::TunnelFib', readerInfo )
   dsfTunnelFib = smashEm.doMount(
      'dsf/tunnelFib', "Tunnel::TunnelFib::TunnelFib", readerInfo )
   gueTunnelFib = smashEm.doMount(
      "tunnel/gueTunnelFib", "Tunnel::TunnelFib::TunnelFib", readerInfo )
   tunnelFib = Tac.newInstance( "Tunnel::TunnelFib::UnifiedTunnelFibView",
                                systemTunnelFib, dsfTunnelFib, gueTunnelFib )
   srTePolicyStatus = smashEm.doMount( srTePolicyStatusPath(),
                                       'SrTePolicy::PolicyStatus', readerInfo )
   srteForwardingStatus = smashEm.doMount( "forwarding/srte/status",
                                       "Smash::Fib::ForwardingStatus", readerInfo )
   rsvpLerTunnelTable = readMountTunnelTable(
            TunnelTableIdentifier.rsvpLerTunnelTable, entityManager )
   lfibVskOverrideConfig = smashEm.doMount( "te/cbf/vskoverride/mpls/config",
                                            "Mpls::Override::LfibVskOverrideConfig",
                                            readerInfo )
   lfibViaSetStatus = smashEm.doMount( "mpls/cbf/overrideLfib",
                                       "Mpls::LfibViaSetStatus",
                                       readerInfo )
