#!/usr/bin/env python3
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
from CliModel import (
   Bool,
   Dict,
   Enum,
   Int,
   Model,
   Str,
)
from CliPlugin import MplsModel
from TableOutput import createTable, Format
from TypeFuture import TacLazyType

MplsLabel = TacLazyType( 'Arnet::MplsLabel' )
MplsRouteHelper = TacLazyType( 'Mpls::MplsRouteHelper' )

def printt( val, indent=0 ):
   print( ' ' * 3 * indent + val )

class MplsLfibCounterLabelEntry( Model ):
   totalPackets = Int( help="Total number of received packets on label" )
   totalBytes = Int( help="Total number of received bytes on label" )
   unprogrammed = Bool( help="Route is unprogrammed in hardware" )
   source = Enum( values=MplsModel.lfibSources, help="LFIB source", optional=True )
   sourceDetails = Str( help="Details from the source for the label", optional=True )
   failedCounterAlloc = Bool( help="Counter resources for the label could not be "
                                   "allocated", optional=True )

class MplsLfibCounters( Model ):
   counters = Dict( keyType=int, valueType=MplsLfibCounterLabelEntry,
                    help="A dictionary of ingress MPLS LFIB counters keyed by "
                         "their top label" )
   multiLabelCounters = Dict( valueType=MplsLfibCounterLabelEntry,
                              help="A dictionary of ingress multi-label MPLS LFIB "
                                   "counters keyed by their top labels (ordered "
                                   "top-most to bottom-most)",
                              optional=True )

   def render( self ):
      if not self.counters and not self.multiLabelCounters:
         return

      capiStrToSourceMap = None
      # Get a reverse mapping of CAPI source strings to enum values
      capiStrToSourceMap = MplsModel.lfibSourceCapiStrToEnum()
      MplsModel.printLfibSourcesHeader()
      headers = [ 'Label', 'Packets', 'Bytes', 'Source', 'Source Details' ]
      f1 = Format( justify='left', minWidth=10 )
      f1.padLimitIs( True )
      f2 = Format( justify='right', minWidth=15 )
      f2.padLimitIs( True )
      f3 = Format( justify='right', minWidth=10 )
      f3.padLimitIs( True )
      outputTable = createTable( headers )
      for counterDict in [ self.counters, self.multiLabelCounters ]:
         for counterKey, counter in sorted( counterDict.items() ):
            if counterKey == MplsLabel.null:
               continue
            counterFlags = []
            totalBytes = counter.totalBytes
            totalPackets = counter.totalPackets
            if counter.failedCounterAlloc:
               counterFlags.append( "N" )
               totalBytes = "-"
               totalPackets = "-"
            if counter.unprogrammed:
               counterFlags.append( "U" )
            labelStr = str( counterKey )
            if counterFlags:
               # pylint: disable-next=consider-using-f-string
               labelStr = "{} [{}]".format( labelStr, ",".join( counterFlags ) )
            # Get the source enum value and use it to get the source code
            source = capiStrToSourceMap.get( counter.source )
            sourceCode = MplsRouteHelper.lfibSourceToSourceCode( source ) \
                     if source else '-'
            details = counter.sourceDetails or '-'
            outputTable.newRow( labelStr, totalPackets, totalBytes, sourceCode,
                                details )


      # Now print out labels that are not being counted.
      counter = self.counters.get( MplsLabel.null )

      if counter:
         outputTable.newRow( "Other", counter.totalPackets, counter.totalBytes,
                             "-", "Aggregated counts for no-resource labels" )

      outputTable.formatColumns( f1, f2, f2, f3, f2 )
      # TODO: only print the N flag legend on platforms that support iditifying
      # labels that failed counter resource allocation
      # if counter and counter.hasAttr( 'failedCounterAlloc' ):
      #   printt( "N - No counter resource available" )
      printt( "N - No counter resource available" )
      printt( "U - Unprogrammed" )
      printt( outputTable.output() )

class MplsLfibCountersSummary( Model ):
   totalMplsLabels = \
         Int( help="Total unique MPLS label stacks in the LFIB" )
   countableMplsLabels = \
         Int( help="Total unique MPLS label stacks being individually counted",
                   optional=True )
   noResourceMplsLabels = \
         Int( help="Total unique MPLS label stacks unable to be individually"
                   "counted due to lack of availble counter resources",
                   optional=True )
   agreggateNoResourceBytes = \
         Int( help="Total aggregated bytes for all MPLS LFIB label stacks which are "
                    "unable to be individually counted due to lack of availble "
                    "counter resources" )
   agreggateNoResourcePkts = \
         Int( help="Total aggregated packets for all MPLS LFIB label stacks which "
                    "are unable to be individually counted due to lack of availble "
                    "counter resources" )

   def render( self ):
      print( f"Total MPLS labels: {self.totalMplsLabels}" )
      if self.countableMplsLabels is not None:
         print( f"Countable MPLS labels: {self.countableMplsLabels}" )
      if self.noResourceMplsLabels is not None:
         print( f"Non-countable MPLS labels: {self.noResourceMplsLabels}" )
      # pylint: disable-next=consider-using-f-string
      print( "Non-countable labels: Aggregate packets: {}, Aggregate bytes: {}".
             format( self.agreggateNoResourcePkts, self.agreggateNoResourceBytes ) )
