#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
from CliPlugin.BgpVpnCli import ( ExtCommValuesAndExactExpression,
      LargeCommValuesAndExactExpression,
      IpPrefixShowExpression,
      Ipv6PrefixShowExpression,
      RdCliExpression,
      NexthopExpression,
      )
from CliPlugin import TechSupportCli
from CliPlugin.ArBgpCli import ShowBgpDebugPolicyBase
from CliPlugin.Ip6AddrMatcher import ip6PrefixMatcher
from CliPlugin.IpAddrMatcher import ipPrefixMatcher
from CliPlugin.MplsVpnCli import tokenVpnV4, tokenVpnV6, mplsVpnAfExpression
from CliPlugin.RoutingBgpShowCli import summaryVrfModel
from CliPlugin.RouteDistinguisher import RdDistinguisherMatcher
from CliPlugin.RoutingBgpCli import (
      V4V6PeerKeyCliExpression )
from CliToken.RoutingBgpShowCliTokens import (
      bgpAfterShow,
      CommunityValuesAndExactRule,
      detail as kwDetail,
      internal as kwInternal,
      neighbors,
      RouteTypeMaybeFilteredMatcher,
      summary,
      )
import CliToken.RoutingBgp as bgpTokens
import LazyMount
import ShowCommand

bgpConfig = None

#-------------------------------------------------------------------------------
# "show bgp ( vpn-ipv4 | vpn-ipv6 ) summary"
#-------------------------------------------------------------------------------
class MplsVpnSummaryCommand( ShowCommand.ShowCliCommandClass ):
   syntax = "show bgp AF summary"
   data = {
         "bgp" : bgpAfterShow,
         "AF" : mplsVpnAfExpression,
         "summary" : summary,
   }
   cliModel = summaryVrfModel
   handler = "MplsVpnShowCliHandler.handlerMplsVpnSummaryCommand"

BasicCli.addShowCommandClass( MplsVpnSummaryCommand )

#-------------------------------------------------------------------------------
# "show bgp vpn-ipv4 [ community <communities> [ exact ] ]
#                    [ extcommunity <extcommunities> [ exact ] ]
#                    [ largecommunity <largecommunities> [ exact ] ]
#                    [ prefix <ipv4-prefix> [ longer-prefixes ] ]
#                    [ rd <rd> ]
#                    [ next-hop ( <v4addr> | <v6addr> ) ]
#                    [ detail ]"
#-------------------------------------------------------------------------------

class ShowBgpVpnV4Cmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( "show bgp vpn-ipv4 [ COMM ] [ EXT_COMM ] [ LARGE_COMM ]"
              " [ IP_PREFIX ] [ RD ] [ NEXT_HOP ] [ detail ]" )
   data = {
         "bgp" : bgpAfterShow,
         "vpn-ipv4" : tokenVpnV4,
         "COMM" : CommunityValuesAndExactRule,
         "EXT_COMM" : ExtCommValuesAndExactExpression,
         "LARGE_COMM" : LargeCommValuesAndExactExpression,
         "IP_PREFIX" : IpPrefixShowExpression,
         "RD" : RdCliExpression,
         "NEXT_HOP" : NexthopExpression,
         "detail" : kwDetail
   }
   cliModel = "MplsVpnCliModels.mplsVpnRoutesVrfModel"
   handler = "MplsVpnShowCliHandler.handlerVpnV4"

BasicCli.addShowCommandClass( ShowBgpVpnV4Cmd )

#-------------------------------------------------------------------------------
# "show bgp vpn-ipv4 <ipv4-prefix> rd <rd> internal"
#-------------------------------------------------------------------------------
# Internal command for specific ipv4 prefix & RD.
class ShowBgpVpnV4InternalCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show bgp vpn-ipv4 PREFIX RD internal"
   data = {
         "bgp" : bgpAfterShow,
         "vpn-ipv4" : tokenVpnV4,
         "PREFIX" : ipPrefixMatcher,
         "RD" : RdCliExpression,
         "internal" : kwInternal
   }
   cliModel = "MplsVpnCliModels.mplsVpnRoutesVrfModel"
   handler = "MplsVpnShowCliHandler.handlerVpnV4"

BasicCli.addShowCommandClass( ShowBgpVpnV4InternalCmd )

#-------------------------------------------------------------------------------
# "show bgp vpn-ipv6 [ community <communities> [ exact ] ]
#                    [ extcommunity <extcommunities> [ exact ] ]
#                    [ largecommunity <larecommunities> [ exact ] ]
#                    [ prefix <ipv6-prefix> [ longer-prefixes ] ]
#                    [ rd <rd> ]
#                    [ next-hop ( <v4addr> | <v6addr> ) ]
#                    [ detail ]"
#-------------------------------------------------------------------------------
class ShowBgpVpnV6Cmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( "show bgp vpn-ipv6 [ COMM ] [ EXT_COMM ] [ LARGE_COMM ]"
              " [ IPv6_PREFIX ] [ RD ] [ NEXT_HOP ] [ detail ]" )
   data = {
         "bgp" : bgpAfterShow,
         "vpn-ipv6" : tokenVpnV6,
         "COMM" : CommunityValuesAndExactRule,
         "EXT_COMM" : ExtCommValuesAndExactExpression,
         "LARGE_COMM" : LargeCommValuesAndExactExpression,
         "IPv6_PREFIX" : Ipv6PrefixShowExpression,
         "RD" : RdCliExpression,
         "NEXT_HOP" : NexthopExpression,
         "detail" : kwDetail
   }
   cliModel = "MplsVpnCliModels.mplsVpnRoutesVrfModel"
   handler = "MplsVpnShowCliHandler.handlerVpnV6"

BasicCli.addShowCommandClass( ShowBgpVpnV6Cmd )

#-------------------------------------------------------------------------------
# "show bgp vpn-ipv6 <ipv6-prefix> rd <rd> internal"
#-------------------------------------------------------------------------------
# Internal command for specific ipv6 prefix & RD.
class ShowBgpVpnV6InternalCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show bgp vpn-ipv6 PREFIX RD internal"
   data = {
         "bgp" : bgpAfterShow,
         "vpn-ipv6" : tokenVpnV6,
         "PREFIX" : ip6PrefixMatcher,
         "RD" : RdCliExpression,
         "internal" : kwInternal
   }
   cliModel = "MplsVpnCliModels.mplsVpnRoutesVrfModel"
   handler = "MplsVpnShowCliHandler.handlerVpnV6"

BasicCli.addShowCommandClass( ShowBgpVpnV6InternalCmd )

#-------------------------------------------------------------------------------
# "show bgp neighbors <ip> vpn-ipv4
#                    ( routes | received-routes | advertised-routes )
#                    [ community <communities> [ exact ] ]
#                    [ extcommunity <extcommunities> [ exact ] ]
#                    [ largecommunity <largecommunities> [ exact ] ]
#                    [ <ipv4-prefix> [ longer-prefixes ] ]
#                    [ rd <rd> ]
#                    [ next-hop ( <v4addr> | <v6addr> ) ]
#                    [ detail ]"
#-------------------------------------------------------------------------------
# Show BGP vpn-ipv4 routes filtered by neighbors IP, and optionally filter by
# communities, extcommunities, largecommunities, ipv4-prefix, RD, and/or next-hop.
class ShowBgpNbrVpnV4Cmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( "show bgp neighbors PEER_ADDR vpn-ipv4 ROUTE_TYPE"
              " [ COMM ] [ EXT_COMM ] [ LARGE_COMM ]"
              " [ IP_PREFIX ] [ RD ] [ NEXT_HOP ] [ detail ]" )
   data = {
         "bgp" : bgpAfterShow,
         "neighbors" : neighbors,
         "PEER_ADDR" : V4V6PeerKeyCliExpression,
         "vpn-ipv4" : tokenVpnV4,
         "ROUTE_TYPE" : RouteTypeMaybeFilteredMatcher,
         "COMM" : CommunityValuesAndExactRule,
         "EXT_COMM" : ExtCommValuesAndExactExpression,
         "LARGE_COMM" : LargeCommValuesAndExactExpression,
         "IP_PREFIX" : IpPrefixShowExpression,
         "RD" : RdCliExpression,
         "NEXT_HOP" : NexthopExpression,
         "detail" : kwDetail
   }
   cliModel = "MplsVpnCliModels.mplsVpnRoutesVrfModel"
   handler = "MplsVpnShowCliHandler.handlerVpnV4"

BasicCli.addShowCommandClass( ShowBgpNbrVpnV4Cmd )


#-------------------------------------------------------------------------------
# "show bgp neighbors <ip> vpn-ipv4
#                    ( routes | received-routes | advertised-routes )
#                    <ipv4-prefix> rd <rd> internal"
#-------------------------------------------------------------------------------
# Internal command for specific ipv4 prefix & RD.
class ShowBgpNbrVpnV4InternalCmd( ShowCommand.ShowCliCommandClass ):
   syntax = (
      "show bgp neighbors PEER_ADDR vpn-ipv4 ROUTE_TYPE PREFIX RD internal" )
   data = {
         "bgp" : bgpAfterShow,
         "neighbors" : neighbors,
         "PEER_ADDR" : V4V6PeerKeyCliExpression,
         "vpn-ipv4" : tokenVpnV4,
         "ROUTE_TYPE" : RouteTypeMaybeFilteredMatcher,
         "PREFIX" : ipPrefixMatcher,
         "RD" : RdCliExpression,
         "internal" : kwInternal
   }
   cliModel = "MplsVpnCliModels.mplsVpnRoutesVrfModel"
   handler = "MplsVpnShowCliHandler.handlerVpnV4"

BasicCli.addShowCommandClass( ShowBgpNbrVpnV4InternalCmd )

#-------------------------------------------------------------------------------
# "show bgp neighbors <ip> vpn-ipv6
#                    ( routes | received-routes | advertised-routes )
#                    [ community <communities> [ exact ] ]
#                    [ extcommunity <extcommunities> [ exact ] ]
#                    [ largecommunity <largecommunities> [ exact ] ]
#                    [ <ipv6-prefix> [ longer-prefixes ] ]
#                    [ rd <rd> ]
#                    [ next-hop ( <v4addr> | <v6addr> ) ]
#                    [ detail ]"
#-------------------------------------------------------------------------------
# Show BGP vpn-ipv6 routes filtered by neighbors IP, and optionally filter by
# communities, extcommunities, largecommunities, ipv6-prefix, RD, and/or next-hop.
class ShowBgpNbrVpnV6Cmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( "show bgp neighbors PEER_ADDR vpn-ipv6 ROUTE_TYPE"
              " [ COMM ] [ EXT_COMM ] [ LARGE_COMM ]"
              " [ IPv6_PREFIX ] [ RD ] [ NEXT_HOP ] [ detail ]" )
   data = {
         "bgp" : bgpAfterShow,
         "neighbors" : neighbors,
         "PEER_ADDR" : V4V6PeerKeyCliExpression,
         "vpn-ipv6" : tokenVpnV6,
         "ROUTE_TYPE" : RouteTypeMaybeFilteredMatcher,
         "COMM" : CommunityValuesAndExactRule,
         "EXT_COMM" : ExtCommValuesAndExactExpression,
         "LARGE_COMM" : LargeCommValuesAndExactExpression,
         "IPv6_PREFIX" : Ipv6PrefixShowExpression,
         "RD" : RdCliExpression,
         "NEXT_HOP" : NexthopExpression,
         "detail" : kwDetail
   }
   cliModel = "MplsVpnCliModels.mplsVpnRoutesVrfModel"
   handler = "MplsVpnShowCliHandler.handlerVpnV6"

BasicCli.addShowCommandClass( ShowBgpNbrVpnV6Cmd )

#-------------------------------------------------------------------------------
# "show bgp neighbors <ip> vpn-ipv6
#                    ( routes | received-routes | advertised-routes )
#                    <ipv6-prefix> rd <rd> internal"
#-------------------------------------------------------------------------------
# Internal command for specific ipv6 prefix & RD.
class ShowBgpNbrVpnV6InternalCmd( ShowCommand.ShowCliCommandClass ):
   syntax = (
      "show bgp neighbors PEER_ADDR vpn-ipv6 ROUTE_TYPE PREFIX RD internal" )
   data = {
         "bgp" : bgpAfterShow,
         "neighbors" : neighbors,
         "PEER_ADDR" : V4V6PeerKeyCliExpression,
         "vpn-ipv6" : tokenVpnV6,
         "ROUTE_TYPE" : RouteTypeMaybeFilteredMatcher,
         "PREFIX" : ip6PrefixMatcher,
         "RD" : RdCliExpression,
         "internal" : kwInternal
   }
   cliModel = "MplsVpnCliModels.mplsVpnRoutesVrfModel"
   handler = "MplsVpnShowCliHandler.handlerVpnV6"

BasicCli.addShowCommandClass( ShowBgpNbrVpnV6InternalCmd )

class ShowBgpDebugPolicyMplsVpnBase( ShowBgpDebugPolicyBase ):
   data = ShowBgpDebugPolicyBase.data.copy()
   data.update( {
      "vpn-ipv4" : tokenVpnV4,
      "vpn-ipv6" : tokenVpnV6,
      'rd' : bgpTokens.rd,
      'RD' : RdDistinguisherMatcher( 'BGP route distinguisher' ),
   } )
   handler = "MplsVpnShowCliHandler.policyDebugMplsVpnHandler"

# -------------------------------------------------------------------------------
# show bgp debug policy import ( vpn-route filter-rcf ( vpn-ipv4 | vpn-ipv6 ) |
#    ( ipv4 | ipv6 ) unicast [ vrf <vrf name> ] [ rcf <function name>() ] <prefix>
#    rd RD )
# -------------------------------------------------------------------------------
class ShowBgpDebugPolicyImport( ShowBgpDebugPolicyMplsVpnBase ):
   syntax = '''show bgp debug policy import
            ( ( ( ( vpn-route filter-rcf vpn-ipv4 )
                | ( ipv4 SAFI ) )
              [ VRF ] [ rcf RCF_FUNC_NAME ] PREFIX )
            | ( ( ( vpn-route filter-rcf vpn-ipv6 )
                | ( ipv6 SAFI ) )
              [ VRF ] [ rcf RCF_FUNC_NAME ] PREFIX6 ) )
            rd RD'''

   data = ShowBgpDebugPolicyMplsVpnBase.data.copy()
   data.update( {
      'import' : 'Debug import policy application',
      'vpn-route' : bgpTokens.rcfVpnRoute,
      'filter-rcf' : bgpTokens.rcfFilter,
   } )

BasicCli.addShowCommandClass( ShowBgpDebugPolicyImport )

# A VRF is not specified in the MPLS VPN debug policy commands for inbound/outbound
# below because MPLS VPN routes only exist in the default VRF.
# -------------------------------------------------------------------------------
# show bgp debug policy inbound neighbor ( ADDR | ADDR6 | all )
# ( vpn-ipv4 | vpn-ipv6 ) [ rcf RCF_FUNC_NAME() ] ( PREFIX | PREFIX6 ) rd RD
# -------------------------------------------------------------------------------

class ShowBgpDebugPolicyMplsVpnInbound( ShowBgpDebugPolicyMplsVpnBase ):
   data = ShowBgpDebugPolicyMplsVpnBase.data.copy()
   syntax = "show bgp debug policy " \
            "inbound neighbor ( ADDR | ADDR6 | all ) ( vpn-ipv4 | vpn-ipv6 ) " \
            "[ rcf RCF_FUNC_NAME() ] ( PREFIX | PREFIX6 ) rd RD"

BasicCli.addShowCommandClass( ShowBgpDebugPolicyMplsVpnInbound )

# -------------------------------------------------------------------------------
# show bgp debug policy outbound neighbor ( ADDR | ADDR6 )
# ( vpn-ipv4 | vpn-ipv6 ) [ rcf RCF_FUNC_NAME() ] ( PREFIX | PREFIX6 ) rd RD
# -------------------------------------------------------------------------------

class ShowBgpDebugPolicyMplsVpnOutbound( ShowBgpDebugPolicyMplsVpnBase ):
   data = ShowBgpDebugPolicyMplsVpnBase.data.copy()
   syntax = "show bgp debug policy " \
            "outbound neighbor ( ADDR | ADDR6 ) ( vpn-ipv4 | vpn-ipv6 ) " \
            "[ rcf RCF_FUNC_NAME() ] ( PREFIX | PREFIX6 ) rd RD"

BasicCli.addShowCommandClass( ShowBgpDebugPolicyMplsVpnOutbound )

# --------------------------------------------------------------------------------
# show bgp debug policy export ( vrf-route filter-rcf ( ( ipv4 | ipv6 ) unicast ) |
#    ( vpn-ipv4 | vpn-ipv6 ) ) [ vrf <vrf name> ] [ rcf <function name>() ] <prefix>
# --------------------------------------------------------------------------------
class ShowBgpDebugPolicyMplsVpnExport( ShowBgpDebugPolicyBase ):
   syntax = '''show bgp debug policy export
            ( ( ( ( vrf-route filter-rcf ipv4 SAFI )
                | ( vpn-ipv4 ) )
              [ VRF ] [ rcf RCF_FUNC_NAME ] PREFIX )
            | ( ( ( vrf-route filter-rcf ipv6 SAFI )
                | ( vpn-ipv6 ) )
              [ VRF ] [ rcf RCF_FUNC_NAME ] PREFIX6 ) )
            '''

   data = ShowBgpDebugPolicyBase.data.copy()
   data.update( {
      'export' : 'Debug export policy application',
      'vrf-route' : bgpTokens.rcfVrfRoute,
      'filter-rcf' : bgpTokens.rcfFilter,
      'vpn-ipv4' : 'MPLS L3 VPN IPv4 Unicast address family',
      'vpn-ipv6' : 'MPLS L3 VPN IPv6 Unicast address family',
   } )
   handler = "MplsVpnShowCliHandler.policyDebugMplsVpnHandler"

BasicCli.addShowCommandClass( ShowBgpDebugPolicyMplsVpnExport )

#-------------------------------------------------------------------------------
# register mpls vpn commands for show tech-support
#
# Not registering the "show bgp vpn-ipvX" commands as the output could be
# overwhelming in scaled setups.
#-------------------------------------------------------------------------------
def showCommandsGuardMplsVpn():
   return bgpConfig.asNumber != 0

TechSupportCli.registerShowTechSupportCmd(
   '2019-09-19 10:51:00',
   cmds=[ 'show bgp vpn-ipv4 summary',
          'show bgp vpn-ipv6 summary' ],
   cmdsGuard=showCommandsGuardMplsVpn,
   summaryCmds=[ 'show bgp vpn-ipv4 summary',
                 'show bgp vpn-ipv6 summary' ],
   summaryCmdsGuard=showCommandsGuardMplsVpn,
)

def Plugin( entityManager ):
   global bgpConfig

   bgpConfig = LazyMount.mount( entityManager, "routing/bgp/config",
                                "Routing::Bgp::Config", "r" )
