# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetModel import IpGenericAddress
from CliModel import Model, Dict, Enum, Bool
import TableOutput
from TypeFuture import TacLazyType
from functools import cmp_to_key

IpGenAddr = TacLazyType( 'Arnet::IpGenAddr' )

def roleToUI( role ):
   if role == 'decap':
      return ''
   else:
      return 'BUD-Role'

class DecapExtraInfo( Model ):
   role = Enum( help='Decap role status', values=( 'decap', 'bud' ) )

class MrouteVxlanDecapGroup( Model ):
   __revision__ = 2
   sources = Dict( help="Mapping of source to more information of decap group",
             keyType=IpGenericAddress, valueType=DecapExtraInfo )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         # sources was a list of IpGenericAddress in revision 1
         sourceList = list( dictRepr[ 'sources' ] )
         dictRepr[ 'sources' ] = sourceList
      return dictRepr

class MrouteVxlanDecapInterface( Model ):
   __revision__ = 2
   groups = Dict( help="A mapping of group to it's sources",
         keyType=IpGenericAddress,
         valueType=MrouteVxlanDecapGroup )

class MrouteVxlanDecapJoinedInterfaces( Model ):
   __revision__ = 2
   interfaces = Dict( help="A mapping of interface to joined groups", keyType=str,
         valueType=MrouteVxlanDecapInterface )
   detail_ = Bool( help="Display information in detail", optional=True )

   def render( self ):
      def compareAddress( addr1, addr2 ):
         if addr1 == addr2:
            return 0
         elif addr1 < addr2:
            return -1
         return 1

      def groupSort( addr1, addr2 ):
         return compareAddress( IpGenAddr( addr1 ), IpGenAddr( addr2 ) )

      if self.interfaces:
         if self.detail_:
            headers = ( "Interface", "Group", "Source", "Decap Info" )
         else:
            headers = ( "Interface", "Group", "Source" )
         formatLeft = TableOutput.Format( justify="left" )
         formatLeft.noPadLeftIs( True )

         table = TableOutput.createTable( headers )
         table.formatColumns( formatLeft, formatLeft, formatLeft, formatLeft )
         for interface, groups in sorted( self.interfaces.items() ):
            for group in sorted( groups.groups, key=cmp_to_key( groupSort ) ):
               sources = groups.groups[ group ].sources
               for source in sorted( sources,
                                     key=lambda x: IpGenAddr( x ).sortKey ):
                  if self.detail_:
                     role = sources[ source ].role
                     table.newRow( interface, group, source, roleToUI( role ) )
                  else:
                     table.newRow( interface, group, source )
         print( table.output() )
