# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCliModes
import CliCommand
# pylint: disable-next=consider-using-from-import
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.MsdpCli as MsdpCli # pylint: disable=consider-using-from-import
from CliPlugin.RouterMulticastCliLib import AddressFamily
from CliToken.Clear import clearKwNode
from CliToken.Ip import ipMatcherForClear
from IpLibConsts import DEFAULT_VRF
import Tac

#--------------------------------------------------------------------------------
# clear ip msdp [ VRF ] sa-cache [ GADDR ]
#--------------------------------------------------------------------------------
class ClearIpMsdpSaCacheCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ip msdp [ VRF ] sa-cache [ GADDR ]'
   data = {
      'clear': clearKwNode,
      'ip': ipMatcherForClear,
      'msdp': 'MSDP Protocol information',
      'VRF': MsdpCli.vrfExprFactory,
      'sa-cache': 'Clear SA Cache',
      'GADDR': IpAddrMatcher.IpAddrMatcher(
         helpdesc='Multicast Group Address to clear from SA Cache' ),
   }

   @staticmethod
   def handler( mode, args ):
      vrfName = args.get( 'VRF' )
      grpAddr = args.get( 'GADDR' )

      af = AddressFamily.ipv4
      if not vrfName:
         vrfName = DEFAULT_VRF
      elif not MsdpCli.vrfExists( af, vrfName ):
         # pylint: disable-next=consider-using-f-string
         mode.addError( "Invalid vrf name %s " % vrfName )
         return
      _vrfClearConfig = MsdpCli.clearConfig( af, vrfName )

      if grpAddr:
         error = IpAddrMatcher.validateMulticastIpAddr( grpAddr )
         if error:
            # pylint: disable-next=consider-using-f-string
            mode.addError( "Invalid Multicast Group: %s" % grpAddr ) 
         _vrfClearConfig.gAddrToClear = grpAddr
      else:
         _vrfClearConfig.gAddrToClear = "0.0.0.0"
      _vrfClearConfig.clearGrpRequestTime = Tac.now()
      try:
         Tac.waitFor( 
            lambda: 
               MsdpCli.msdpStatus.vrfStatus[ vrfName ].lastGrpClearTime >= \
               _vrfClearConfig.clearGrpRequestTime,
            description='SA Cache to get cleared.',
            warnAfter=None, sleep=True, maxDelay=0.5, timeout=5 )
      except Tac.Timeout:
         mode.addError( "SA Cache may not have been cleared yet." ) 

BasicCliModes.EnableMode.addCommandClass( ClearIpMsdpSaCacheCmd )
