#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Arnet import IpGenAddr
import Tac
import LazyMount
import BasicCli
import ShowCommand
import CliMatcher
from CliPlugin import MulticastTunnelModel, MulticastConfigCheckModel
from CliPlugin.IpAddrMatcher import IpAddrMatcher, validateMulticastIpAddr
import CliPlugin.TechSupportCli
from CliPlugin.VrfCli import VrfExprFactory
import CliPlugin.VxlanIntfMcastVpnCli
from CliPlugin.VxlanCli import VtiNameToId, getVtiConfig, matcherVlan
from CliPlugin.VlanCli import vlanIdMatcher
import CliToken.McastVpn
import CliToken.McastCommon

#
# show multicast ipv4|ipv6 evpn encap [ flood ]
# show multicast ipv4|ipv6 evpn decap received [ flood ]
# show multicast ipv4 evpn encap mapping vrf <vrfName> <overlay-group>

_ipTunnelVlanGroupStatuses = {}
bridgingHwCapabilities = None
ipStatus = None
vtiConfigDir = None
ipTunnelGroupConfig = None

_ipTunnelStatusTypes = \
   Tac.Type( "Routing::Multicast::IpTunnelGroupStatusType" ).attributes

floodMatcher = \
   CliMatcher.KeywordMatcher( 'flood', helpdesc='VLAN flooding information' )

receivedMatcher = \
   CliMatcher.KeywordMatcher( 'received', helpdesc='Received decapsulation' )

matcherConfigSanity = CliMatcher.KeywordMatcher( 'config-sanity',
   helpdesc='Show hints of potential Mcastvpn config problems' )

def showMulticastEvpn( mode, args ):
   af = args[ "AF" ]
   encap = "encap" in args
   statusType = "irbEncapStatus" if encap else "irbDecapStatus"
   vlanGroupStatus = _ipTunnelVlanGroupStatuses[ af ][ statusType ]
   model = MulticastTunnelModel.MulticastTunnelVlans()
   model.initFromTacModel( vlanGroupStatus, encap=encap )

   return model

class ShowMulticastEvpn( ShowCommand.ShowCliCommandClass ):
   syntax = 'show multicast AF evpn encap'

   data = {
      'AF': CliToken.McastCommon.afMatcherForShow,
      'multicast': CliToken.McastCommon.multicastMatcherForShow,
      'evpn': CliToken.McastVpn.evpnForShow,
      'encap': CliToken.McastVpn.encapForShow,
   }
   cliModel = MulticastTunnelModel.MulticastTunnelVlans

   handler = showMulticastEvpn

BasicCli.addShowCommandClass( ShowMulticastEvpn )

vtiNameToId = VtiNameToId()

def multicastConfigVlanVrfSetModel( vtiConfig, model, vlans, vrfs ):
   for vlan in vlans:
      vni = vtiConfig.vlanToVniMap.get( vlan )
      if vni is None:
         if model.evpnVlanVniSet is None:
            model.evpnVlanVniSet = MulticastConfigCheckModel.EvpnVlanVniCheck()
         model.evpnVlanVniSet.vlans[ vlan ] = vni
   for vrf in vrfs:
      vni = vtiConfig.vrfToVniMap.get( vrf )
      if vni is None:
         if model.vrfSet is None:
            model.vrfSet = MulticastConfigCheckModel.VrfCheck()
         model.vrfSet.vrfs[ vrf ] = vtiConfig.vrfToVniMap.get( vrf )

   return model

def showMultiCastIpv4EvpnConfigSanityVlan( mode, args ):
   model = MulticastConfigCheckModel.McastVpnVlanConfigCheck()
   ipTunnelGroupIntfConfig = ipTunnelGroupConfig.intfConfig.get( 'Vxlan1' )

   if ipTunnelGroupIntfConfig is None:
      return model

   vtiConfig = getVtiConfig( 'Vxlan1' )
   vlan = args.get( 'VLAN_ID' )
   vrf = args.get( 'VRF' )

   if vlan is None and vrf is None:
      vlans = ipTunnelGroupIntfConfig.vlanToMulticastGroup
      vrfs = ipTunnelGroupIntfConfig.vrfToMulticastGroup
      return multicastConfigVlanVrfSetModel( vtiConfig, model, vlans, vrfs )

   if vlan:
      if vlan.id in ipTunnelGroupIntfConfig.vlanToMulticastGroup:
         multicastConfigVlanVrfSetModel( vtiConfig, model, ( vlan.id, ), () )

   if vrf:
      if vrf in ipTunnelGroupIntfConfig.vrfToMulticastGroup:
         multicastConfigVlanVrfSetModel( vtiConfig, model, (), ( vrf, ) )

   return model

class ShowMultiCastIpv4EvpnConfigSanityVlan( ShowCommand.ShowCliCommandClass ):
   syntax = 'show multicast ipv4 [ VRF ] evpn config-sanity [ vlan VLAN_ID ]'
   data = {
      'multicast': CliToken.McastCommon.multicastMatcherForShow,
      'ipv4': 'Details related to IPv4',
      'VRF': VrfExprFactory( helpdesc='VRF name' ),
      'evpn': CliToken.McastVpn.evpnForShow,
      'config-sanity': matcherConfigSanity,
      'vlan': matcherVlan,
      'VLAN_ID': vlanIdMatcher,

   }
   cliModel = MulticastConfigCheckModel.McastVpnVlanConfigCheck
   handler = showMultiCastIpv4EvpnConfigSanityVlan
   hidden = True

BasicCli.addShowCommandClass( ShowMultiCastIpv4EvpnConfigSanityVlan )

class ShowMulticastEvpnDecap( ShowCommand.ShowCliCommandClass ):
   syntax = 'show multicast AF evpn decap received'

   data = {
      'AF': CliToken.McastCommon.afMatcherForShow,
      'multicast': CliToken.McastCommon.multicastMatcherForShow,
      'evpn': CliToken.McastVpn.evpnForShow,
      'decap': CliToken.McastVpn.decapForShow,
      'received': receivedMatcher,
   }
   cliModel = MulticastTunnelModel.MulticastTunnelVlans

   handler = showMulticastEvpn

BasicCli.addShowCommandClass( ShowMulticastEvpnDecap )

def showMulticastEvpnFlood( mode, args ):
   af = args[ "AF" ]
   statusType = "irbEncapStatus" if "encap" in args else "irbDecapStatus"
   vlanGroupStatus = _ipTunnelVlanGroupStatuses[ af ][ statusType ]
   model = MulticastTunnelModel.MulticastFloodVlans()
   model.initFromTacModel( vlanGroupStatus )

   return model

class ShowMulticastEvpnFlood( ShowCommand.ShowCliCommandClass ):
   syntax = 'show multicast AF evpn encap flood'

   data = {
      'AF': CliToken.McastCommon.afMatcherForShow,
      'multicast': CliToken.McastCommon.multicastMatcherForShow,
      'evpn': CliToken.McastVpn.evpnForShow,
      'encap': CliToken.McastVpn.encapForShow,
      'flood': floodMatcher,
   }
   cliModel = MulticastTunnelModel.MulticastFloodVlans

   handler = showMulticastEvpnFlood

BasicCli.addShowCommandClass( ShowMulticastEvpnFlood )

class ShowMulticastEvpnDecapFlood( ShowCommand.ShowCliCommandClass ):
   syntax = 'show multicast AF evpn decap received flood'

   data = {
      'AF': CliToken.McastCommon.afMatcherForShow,
      'multicast': CliToken.McastCommon.multicastMatcherForShow,
      'evpn': CliToken.McastVpn.evpnForShow,
      'decap': CliToken.McastVpn.decapForShow,
      'received': receivedMatcher,
      'flood': floodMatcher,
   }
   cliModel = MulticastTunnelModel.MulticastFloodVlans

   handler = showMulticastEvpnFlood

BasicCli.addShowCommandClass( ShowMulticastEvpnDecapFlood )

def underlayMulticastSupported():
   return bridgingHwCapabilities.vxlanUnderlayMcastSupported

def floodingOverUnderlayMulticastSupported():
   return bridgingHwCapabilities.vxlanFloodOverUnderlayMcastSupported

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2020-05-07 14:02:34',
   cmds=[ 'show multicast ipv4 evpn encap',
          'show multicast ipv4 evpn decap received' ],
   cmdsGuard=underlayMulticastSupported )

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2020-05-07 14:39:17',
   cmds=[ 'show multicast ipv4 evpn encap flood',
          'show multicast ipv4 evpn decap received flood' ],
   cmdsGuard=floodingOverUnderlayMulticastSupported )

def showMulticastEvpnEncapMapping( mode, args ):
   overlayGrp = args[ 'OVERLAY_GRP' ]
   if validateMulticastIpAddr( overlayGrp, False ):
      mode.addError( 'Overlay group must be a valid multicast address' )
      return None

   # Multicast config is allowed only in Vxlan1 vxlan interface
   tunnelGrpIntfConfig = \
      CliPlugin.VxlanIntfMcastVpnCli.ipTunnelGroupConfig.intfConfig.get( 'Vxlan1' )
   if tunnelGrpIntfConfig is None:
      # No vxlan intf config
      return MulticastTunnelModel.MulticastEncapMappingVlans( vlans={} )

   vrfName = args[ 'VRF' ]
   vrfIntfStatus = ipStatus.vrfIpIntfStatus.get( vrfName )
   if vrfIntfStatus is None:
      # Vrf is not present
      return MulticastTunnelModel.MulticastEncapMappingVlans( vlans={} )

   af = 'ipv4'
   vlanGroupStatus = _ipTunnelVlanGroupStatuses[ af ][ 'irbEncapStatus' ]
   encapVlans = {}
   vlanIntfIdType = Tac.Type( 'Arnet::VlanIntfId' )
   ovgType = Tac.Type( 'Routing::Multicast::OverlayVlanGroup' )
   for intfId in vrfIntfStatus.ipIntfStatus:
      if not vlanIntfIdType.isVlanIntfId( intfId ):
         # Not a vlan intf
         continue
      vlanId = vlanIntfIdType.vlanId( intfId )
      # Check if default encap group is present for the vlan
      wcOvg = ovgType.anyGroup( vlanId, af )
      if wcOvg not in vlanGroupStatus.vlanToMulticastGroup:
         # Default encap group mapping is not present. Default group mapping is
         # mandatory to generate any specific SPMSI mapping for the overlay group.
         # Default encap group mapping may not be populated either because of missing
         # default group config or if vlan does not have a VNI
         continue

      ovg = ovgType( vlanId, IpGenAddr( overlayGrp ) )
      overlayToEncapGrp = vlanGroupStatus.vlanToMulticastGroup.get( ovg )
      if overlayToEncapGrp is not None:
         # We have already seen this overlay group for the vlan and computed its
         # corresponding encap group
         encapGrps = overlayToEncapGrp.multicastGroup.keys()
         if encapGrps:
            # There will ever be only one encap group for a given overlay vlan group
            encapVlans[ vlanId ] = str( encapGrps[ 0 ].group )
            continue

      # We haven't seen this overlay group yet or we haven't computed the encap
      # group yet, go ahead and explicitly compute the expected encap group
      mapping = Tac.newInstance( 'Multicast::Vpn::VlanToGroupMapping', vlanId )
      vlanToMcGrpConfig = tunnelGrpIntfConfig.vlanToMulticastGroup.get( vlanId )
      if vlanToMcGrpConfig is not None:
         mapping.vlanTunnelConfigEntry = vlanToMcGrpConfig.tunnelConfigEntry
      vrfToMcGrpConfig = tunnelGrpIntfConfig.vrfToMulticastGroup.get( vrfName )
      if vrfToMcGrpConfig is not None:
         mapping.vrfTunnelConfigEntry = vrfToMcGrpConfig.tunnelConfigEntry
      overlayGrpAddr = Tac.Value( 'Routing::Multicast::OverlayAddr', overlayGrp )
      encapVlans[ vlanId ] = str( mapping.underlayGroup( overlayGrpAddr ) )

   return MulticastTunnelModel.MulticastEncapMappingVlans( vlans=encapVlans )

class ShowMulticastEvpnEncapMapping( ShowCommand.ShowCliCommandClass ):
   syntax = 'show multicast ipv4 evpn encap mapping VRF OVERLAY_GRP'

   data = {
      'ipv4': 'Details related to IPv4',
      'multicast': CliToken.McastCommon.multicastMatcherForShow,
      'evpn': CliToken.McastVpn.evpnForShow,
      'encap': CliToken.McastVpn.encapForShow,
      'mapping': 'Predict mapping of overlay group to encap group',
      'VRF': VrfExprFactory( helpdesc='VRF name' ),
      'OVERLAY_GRP': IpAddrMatcher( helpdesc='Overlay IPv4 group' ),
   }
   cliModel = MulticastTunnelModel.MulticastEncapMappingVlans

   handler = showMulticastEvpnEncapMapping

BasicCli.addShowCommandClass( ShowMulticastEvpnEncapMapping )

def Plugin( entityManager ):

   global bridgingHwCapabilities
   global ipStatus
   global ipTunnelGroupConfig

   for af in [ "ipv4", "ipv6" ]:
      _ipTunnelVlanGroupStatuses[ af ] = {}
      tacType = "Routing::Multicast::IpTunnelVlanGroupStatus"
      for statusType in _ipTunnelStatusTypes:
         path = Tac.Type( tacType ).mountPath( af, statusType )
         _ipTunnelVlanGroupStatuses[ af ][ statusType ] = \
            LazyMount.mount( entityManager, path, tacType, 'r' )

   bridgingHwCapabilities = LazyMount.mount( entityManager,
                                             "bridging/hwcapabilities",
                                             "Bridging::HwCapabilities", "r" )
   ipStatus = LazyMount.mount( entityManager, "ip/status", "Ip::Status", "r" )

   typeName = 'Routing::Multicast::IpTunnelGroupConfig'
   ipTunnelGroupConfig = LazyMount.mount( entityManager,
                                            Tac.Type( typeName ).mountPath,
                                            typeName,
                                            "w" )
