# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetModel import IpGenericAddress
from CliModel import Dict, Int, List, Model, Submodel
from IntfModels import Interface
import TableOutput

class MvpnVrfPmsis( Model ):
   pmsis = List( valueType=str, help='List of PMSIs for MVPN VRF' )

class MvpnVrfs( Model ):
   vrfs = Dict( help='Mapping of VRF to PMSIs', keyType=str,
                valueType=MvpnVrfPmsis )

   def render( self ):
      if self.vrfs:
         headers = ( 'VRF', 'PMSI' )
         formatLeft = TableOutput.Format( justify='left' )
         formatLeft.padLimitIs( True )
         table = TableOutput.createTable( headers )
         table.formatColumns( formatLeft, formatLeft )
         for vrf, vrfPmsis in sorted( self.vrfs.items() ):
            # Handler is expected to populate just one element in pmsis list
            table.newRow( vrf, vrfPmsis.pmsis[ 0 ] )
         print( table.output() )

class PmsiMldpOpaqueValueRd( Model ):
   rds = Dict( keyType=str, valueType=Interface,
               help='Dictionary of pmsiIntfIds indexed by route distinguisher' )

class PmsiMldpOpaqueValueGroup( Model ):
   groups = Dict( keyType=IpGenericAddress, valueType=PmsiMldpOpaqueValueRd,
                  help='Dictionary of OpaqueValueRd models indexed by group addr' )

class PmsiMldpOpaqueValue( Model ):
   genericLspIds = Dict( keyType=int, valueType=Interface, optional=True,
                         help='Dictionary of pmsiIntfIds indexed by LSP ID' )
   transitV4Src = Dict( keyType=IpGenericAddress, optional=True,
                        valueType=PmsiMldpOpaqueValueGroup,
                        help='Dictionary of OpaqueValueGroup models indexed by '
                             'source address' )
   invalidOpaqueValues = Dict( keyType=str, valueType=Interface, optional=True,
                               help='Dictionary of pmsiIntfIds indexed by invalid '
                                    'opaque values' )

class PmsiMldpOpaqueValueModel( Model ):
   opaqueValues = Submodel( valueType=PmsiMldpOpaqueValue,
                            help='MLDP P2MP tunnel opaque value information' )

class PmsiMldpTunnelDecap( Model ):
   rootAddrs = Dict( keyType=IpGenericAddress, valueType=PmsiMldpOpaqueValueModel,
                     help='Dictionary of OpaqueValue models indexed by root addr' )

class PmsiRsvpTunnelDecapExtTunnelId( Model ):
   extTunnelIds = Dict( keyType=IpGenericAddress, valueType=Interface,
                        help='Dictionary of pmsiIntfIds indexed by ext tunnel ID' )

class PmsiRsvpTunnelDecapTunnelId( Model ):
   tunnelIds = Dict( keyType=int, valueType=PmsiRsvpTunnelDecapExtTunnelId,
                     help='Dictionary of ext tunnel ID models indexed by tunnel ID' )

class PmsiRsvpTunnelDecap( Model ):
   p2mpIds = Dict( keyType=int, valueType=PmsiRsvpTunnelDecapTunnelId,
                   help='Dictionary of RSVP tunnel ID models indexed by P2MP ID' )

class PmsiTunnelDecapModel( Model ):
   vrfId = Int( optional=True, help="VRF ID" )
   mldpP2mp = Submodel( optional=True, valueType=PmsiMldpTunnelDecap,
                        help='MLDP P2MP tunnel decapsulation information' )
   rsvpTeP2mpLsp = Submodel( optional=True, valueType=PmsiRsvpTunnelDecap,
                             help='RSVP-TE P2MP tunnel decapsulation information' )

class PmsiTunnelDecapStatus( Model ):
   vrfs = Dict( keyType=str, valueType=PmsiTunnelDecapModel,
                help='PMSI tunnel decapsulation information indexed by VRF name' )

   def render( self ):
      headers = ( 'PMSI Tunnel', 'PMSI IntfId' )
      formatLeft = TableOutput.Format( justify='left' )
      formatLeft.padLimitIs( True )
      if not self.vrfs:
         return

      # pylint: disable=too-many-nested-blocks
      for vrfName, tunnelDecap in sorted( self.vrfs.items() ):
         if tunnelDecap.mldpP2mp and tunnelDecap.mldpP2mp.rootAddrs:
            print( f'VRF: {vrfName}, VRF ID: {tunnelDecap.vrfId}, '
                   'PMSI tunnel type: MLDP P2MP LSP' )
            mldpTable = TableOutput.createTable( headers )
            mldpTable.formatColumns( formatLeft, formatLeft )
            rootAddrs = tunnelDecap.mldpP2mp.rootAddrs
            for rootAddr, decapEntry in sorted( rootAddrs.items() ):
               genericLspIds = decapEntry.opaqueValues.genericLspIds
               if genericLspIds is not None:
                  for lsp, intfId in sorted( genericLspIds.items() ):
                     tunnel = f'Root Address: {rootAddr}, LSP ID: {lsp}'
                     mldpTable.newRow( tunnel, intfId )

               transitV4Src = decapEntry.opaqueValues.transitV4Src
               if transitV4Src is not None:
                  for source, groupModel in sorted( transitV4Src.items() ):
                     for group, rdModel in sorted( groupModel.groups.items() ):
                        for rd, intfId in sorted( rdModel.rds.items() ):
                           tunnel = ( f'Root Address: {rootAddr}, Source: {source}, '
                                      f'Group: {group}, RD: {rd}' )
                           mldpTable.newRow( tunnel, intfId )

               invalidValues = decapEntry.opaqueValues.invalidOpaqueValues
               if invalidValues is not None:
                  for opaqueValue, intfId in sorted( invalidValues.items() ):
                     tunnel = f'Root Address: {rootAddr}, {opaqueValue}'
                     mldpTable.newRow( tunnel, intfId )

            print( mldpTable.output() )

         if tunnelDecap.rsvpTeP2mpLsp and tunnelDecap.rsvpTeP2mpLsp.p2mpIds:
            print( f'VRF: {vrfName}, VRF ID: {tunnelDecap.vrfId}, '
                   'PMSI tunnel type: RSVP-TE P2MP LSP' )
            rsvpTable = TableOutput.createTable( headers )
            rsvpTable.formatColumns( formatLeft, formatLeft )
            p2mpIds = tunnelDecap.rsvpTeP2mpLsp.p2mpIds
            for p2mpId, tunnelIdModel in sorted( p2mpIds.items() ):
               tunnelIds = tunnelIdModel.tunnelIds
               for tunnelId, extTunnelIdModel in sorted( tunnelIds.items() ):
                  extTunnelIds = extTunnelIdModel.extTunnelIds
                  for extTunnelId, intfId in sorted( extTunnelIds.items() ):
                     tunnel = ( f'P2MP ID: {p2mpId}, Tunnel ID: {tunnelId}, '
                                f'Extended Tunnel ID: {extTunnelId}' )
                     rsvpTable.newRow( tunnel, intfId )
            print( rsvpTable.output() )
