# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
# pylint: disable=no-else-continue
# pylint: disable=inconsistent-return-statements
# pylint: disable=consider-using-in

import itertools
import os
import re
import subprocess
import sys
import json
from functools import partial
from ipaddress import IPv4Network
from itertools import chain, tee
from socket import IPPROTO_TCP, IPPROTO_UDP
from io import StringIO

import AgentCommandRequest
import AgentDirectory
import Arnet
import BasicCli
import Cell
import CliParser
import ConfigMount
import Intf.IntfRange as IntfRange # pylint: disable=consider-using-from-import
import LazyMount
import NatAgent
import SharedMem
import Shark
import SharkLazyMount
import Smash
import Tac
from Arnet.NsLib import DEFAULT_NS, runMaybeInNetNs
from ArnetModel import IpAddrAndMask, IpAddrAndPort
from ArPyUtils import naturalOrderKey
from CliCommand import CliExpression, Node, guardedKeyword
from CliMatcher import (
   DynamicNameMatcher,
   EnumMatcher,
   IntegerMatcher,
   KeywordMatcher,
   PatternMatcher,
)
from CliMode.NatMode import (
   NatFlowActionMode,
   NatFlowMatchMode,
   NatFlowMode,
   NatFlowPolicyMode,
   NatPoolMode,
   NatPortOnlyPoolMode,
   NatProfileMode,
   NatSyncMode,
)
from CliMode.NatServiceList import NatServiceListMode
from CliParserCommon import PRIO_NORMAL
from CliPlugin import (
   EthIntfCli,
   IntfCli,
   IpAddrMatcher,
   IraIpIntfCli,
   IraVrfCli,
   LagIntfCli,
   MacAddr,
   SwitchIntfCli,
   TechSupportCli,
   VlanIntfCli,
)
from CliPlugin.AclCli import aclUnconfReferencesHook
from CliPlugin.NatModels import (
   STATUS_STATE,
   NatAclList,
   NatAddrOnlyTranslations,
   NatConnectionLimit,
   NatDynamicCountersModel,
   NatDynamicCountersProfileModel,
   NatDynamicCountersProtoModel,
   NatFlowAction,
   NatFlowActionCmdModel,
   NatFlowActionPolicy,
   NatFlowAddress,
   NatFlowIntf,
   NatFlowMatch,
   NatFlowMatchActions,
   NatFlowMatchCmdModel,
   NatFlowMatchInfo,
   NatFlowMatchPolicy,
   NatFlowPolicyInfo,
   NatFlowProfileCmdModel,
   NatFlowProfilePolicy,
   NatHostConnectionLimit,
   NatIpTranslations,
   NatLoggingModel,
   NatPoolConnectionLimit,
   NatPoolModel,
   NatProfileModel,
   NatServiceListCommandModel,
   NatServiceListModel,
   NatSyncFwdToPeer,
   NatSynchronizationAdvertisedTranslations,
   NatSynchronizationModel,
   NatSynchronizationPeer,
   NatTranslationConnectionDropped,
   NatTranslationFlows,
   NatTranslationRate,
   NatTranslationRates,
   NatTranslationRatesAll,
   NatTranslationSummary,
   NatTranslationDynamicSummary,
   NatVrfCounters,
   TwiceNatIpTranslations,
   Version,
   VrfNatIpTranslations,
   ipProtoMap,
)
from CliPlugin.ShowConfigConsistencyCliModels import (
   ConfigConsistencyMultiTableModelBuilder,
)
from CliPlugin.VrfCli import (
   ALL_VRF_NAME,
   DEFAULT_VRF,
   CliVrfMapper,
   VrfExprFactory,
   vrfExists,
   vrfMap,
)
from CliPrint import CliPrint
from ConfigConsistencyChecker import (
   configConsistencyCategory,
   configConsistencyCheck,
   configConsistencyCluster,
)
from IpUtils import Mask
from TypeFuture import TacLazyType
from IptablesHelper import RT
from Toggles.NatToggleLib import toggleNatConnLimitNewCodeEnabled

natConfig = None
natStatus = None
routingHwRouteStatus = None
vxlanNatStatus = None
dynamicConnectionStatus = None
natDropRulesStatus = None
dynamicConnectionHwStatus = None
natSyncConfig = None
natSyncStatus = None
natHwCapabilities = None
natHwStatus = None
natHwFlowStatus = None
aclConfig = None
allVrfStatusLocal = None
mcastTranslationMode = None
platformWarnings = None
advertisedStatus = smashedAdvertisedStatus = advertisedStatusSmashReader = None
discoveredStatus = None
vrfIdMapStatus = None
sfeLauncherConfig = None

mlib = CliPrint().lib

modelet = IraIpIntfCli.RoutingProtocolIntfConfigModelet

AclAction = Tac.Type( 'Acl::Action' )

DefaultValue = TacLazyType( "Ip::Nat::DefaultValue" )

kernelProgramState = Tac.Type( "Ip::Nat::NatKernelProgrammed" )

McastTranslationType = TacLazyType( 'Ip::Nat::McastTranslationType' )

NatVrfIntfId = TacLazyType( "Arnet::NatVrfIntfId" )

reprogramWarningStr = ( "Existing NAT rules will be reprogrammed"
                        " immediately. Traffic may be disrupted" )

def sfe():
   # pylint: disable=consider-using-ternary
   return ( ( sfeLauncherConfig and 'Sfe' in sfeLauncherConfig ) or
         os.getenv( "SIMULATION_SFE" ) )

def getNatIntfId( intfId ):
   if NatVrfIntfId.isNatVrfProfileIntfId( intfId ):
      return NatVrfIntfId.toIntfId( intfId )
   else:
      return intfId

def getSourceVrf( intfId, vrfName ):
   if NatVrfIntfId.isNatVrfProfileIntfId( intfId ):
      srcVrfId = NatVrfIntfId.vrfId( intfId )
      if srcVrfId not in vrfIdMapStatus.vrfIdToName:
         return None
      sourceVrf = vrfIdMapStatus.vrfIdToName[ srcVrfId ].vrfName
      # Return sourceVrf as None if it is same as the profile VRF.
      return sourceVrf if sourceVrf != vrfName else None
   else:
      return None

def _natSupported():
   return bool( natHwCapabilities.natSupported )

def natSupported( mode, token ):
   if _natSupported():
      return None
   else:
      return CliParser.guardNotThisPlatform

def natKernelSupported( mode, token ):
   if _natSupported() and not natHwCapabilities.natDpdkProcessingSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def natDpdkSupported( mode, token ):
   if _natSupported() and natHwCapabilities.natDpdkProcessingSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def natRandomAddressSelectionSupported( mode, token ):
   if _natSupported() and natHwCapabilities.natRandomAddressSelectionSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def natL4PortSupported( mode, token ):
   if natHwCapabilities.natSupported and natHwCapabilities.natL4PortSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def natProtocolSupported( mode, token ):
   if natHwCapabilities.natSupported and natHwCapabilities.natProtocolSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def natGroupSupported( mode, token ):
   if natHwCapabilities.natSupported and natHwCapabilities.natGroupSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def natDynamicTwiceSupported( mode, token ):
   if natHwCapabilities.natSupported and natHwCapabilities.natGroupSupported and \
      natHwCapabilities.natDynamicTwiceSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def natAclSupported( mode, token ):
   if natHwCapabilities.natSupported and natHwCapabilities.natAclSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def natServiceListSupported( mode, token ):
   if natHwCapabilities.natSupported and natHwCapabilities.natServiceListSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def natAddrOnlySupported( mode, token ):
   if natHwCapabilities.natSupported and natHwCapabilities.natAddrOnlySupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def natAddrOnlyKernelSupported( mode, token ):
   if natHwCapabilities.natSupported and natHwCapabilities.natAddrOnlySupported and \
      not natHwCapabilities.natDpdkProcessingSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def natAddrOnlyFlowSupported( mode, token ):
   if natHwCapabilities.natSupported and natHwCapabilities.natAddrOnlySupported and \
      natHwCapabilities.natDpdkProcessingSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def natCountersSupported( mode, token ):
   if natHwCapabilities.natSupported and natHwCapabilities.natCountersSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def dynamicNatCountersSupported( mode, token ):
   if natHwCapabilities.natSupported and \
         natHwCapabilities.natDynamicCountersSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def natFragmentCommandSupported( mode, token ):
   if ( natHwCapabilities.natSupported and
        natHwCapabilities.natFragmentCommandSupported ):
      return None
   else:
      return CliParser.guardNotThisPlatform

def natVrfLeakCommandSupported( mode, token ):
   if ( natHwCapabilities.natSupported and
        natHwCapabilities.natVrfLeakCommandSupported ):
      return None
   else:
      return CliParser.guardNotThisPlatform

def natLoggingSupported( mode, token ):
   if natHwCapabilities.natSupported and natHwCapabilities.natLoggingSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def natFlowSupported( mode, token ):
   if natHwCapabilities.natSupported and natHwCapabilities.natFlowSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def natImmediateTcpEstablishSupported( mode, token ):
   if ( natHwCapabilities.natSupported and
        natHwCapabilities.natImmediateTcpEstablishSupported ):
      return None
   else:
      return CliParser.guardNotThisPlatform

def natSyncFwdTrappedPktSupported( mode, token ):
   if ( natHwCapabilities.natSupported and
        natHwCapabilities.natProfileConfigSupported and
        natHwCapabilities.natSyncFwdTrappedPktSupported ):
      return None
   else:
      return CliParser.guardNotThisPlatform

def checkTunnelIntfSupport( mode ):
   intfName = mode.intf.name if hasattr( mode, 'intf' ) else ''
   if 'Tunnel' in intfName and not natHwCapabilities.natTunnelIntfSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def natIntfStaticSupported( mode, token ):
   if natHwCapabilities.natSupported and natHwCapabilities.natIntfStaticSupported:
      return checkTunnelIntfSupport( mode )
   else:
      return CliParser.guardNotThisPlatform

def natIntfSnatIngressStaticSupported( mode, token ):
   if ( natHwCapabilities.natSupported and
        natHwCapabilities.natIntfStaticSupported and
        natHwCapabilities.natIngressSourceStaticSupported ):
      return checkTunnelIntfSupport( mode )
   else:
      return CliParser.guardNotThisPlatform

# Enable the egress SNAT keyword when egress mcast NAT is supported or
# ingress SNAT is supported to allow specifying the default direction
def natIntfSnatEgressStaticSupported( mode, token ):
   if ( natHwCapabilities.natSupported and
        ( natHwCapabilities.natEgressMcastStaticSupported or
          ( natHwCapabilities.natIntfStaticSupported and
            natHwCapabilities.natIngressSourceStaticSupported ) ) ):
      return None
   else:
      return CliParser.guardNotThisPlatform

def natIntfDnatEgressStaticSupported( mode, token ):
   if ( natHwCapabilities.natSupported and
        ( natHwCapabilities.natEgressDestinationStaticSupported or
          natHwCapabilities.natEgressMcastStaticSupported ) ):
      return checkTunnelIntfSupport( mode )
   else:
      return CliParser.guardNotThisPlatform

def natStaticSupported( mode, token ):
   if natHwCapabilities.natSupported and \
      ( natHwCapabilities.natIntfStaticSupported or \
        natHwCapabilities.natEgressMcastStaticSupported ):
      return checkTunnelIntfSupport( mode )
   else:
      return CliParser.guardNotThisPlatform

def natIntfDynamicSupported( mode, token ):
   if natHwCapabilities.natSupported and natHwCapabilities.natIntfDynamicSupported:
      return checkTunnelIntfSupport( mode )
   else:
      return CliParser.guardNotThisPlatform

def natConnectionLimitSupported( mode, token ):
   if natHwCapabilities.natSupported and \
      natHwCapabilities.natIntfDynamicSupported and \
      natHwCapabilities.natConnectionLimitSupported:
      return checkTunnelIntfSupport( mode )
   else:
      return CliParser.guardNotThisPlatform

def natIpIntfConfigSupported( mode, token ):
   if natHwCapabilities.natSupported and \
      natHwCapabilities.natIntfConfigSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def _natProfileConfigSupported():
   return bool( natHwCapabilities.natSupported and \
                natHwCapabilities.natProfileConfigSupported )

def natIpProfileConfigSupported( mode, token ):
   if _natProfileConfigSupported():
      return None
   else:
      return CliParser.guardNotThisPlatform

def natIpServiceProfileConfigSupported( mode, token ):
   if _natProfileConfigSupported():
      return checkTunnelIntfSupport( mode )
   else:
      return CliParser.guardNotThisPlatform

def natVrfHopStaticSupported( mode, token ):
   if natHwCapabilities.natSupported and natHwCapabilities.natVrfHopStaticSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def natOutsideVrfSupported( mode, token ):
   if natHwCapabilities.natSupported and natHwCapabilities.natOutsideVrfSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def natInVrfSupported( mode, token ):
   if natHwCapabilities.natSupported and natHwCapabilities.natInVrfSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def natLimitedMulticastTranslations( mode, token ):
   if ( natHwCapabilities.natSupported and
        natHwCapabilities.natLimitedMulticastTranslations ):
      return None
   else:
      return CliParser.guardNotThisPlatform

def natUpnpPortMapSupported( mode, token ):
   if ( natHwCapabilities.natSupported and
        natHwCapabilities.natUpnpPortMapSupported ):
      return None
   else:
      return CliParser.guardNotThisPlatform

def natPortOnlySupported( mode, token ):
   if natHwCapabilities.natPortOnlySupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def natSwPrefixTranslationSupported( mode, token ):
   if natHwCapabilities.natSwPrefixTranslationSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def getNatIntfConfig( intfName, profileMode ):
   intfConfig = natConfig.intfConfig.get( intfName )
   if not intfConfig:
      return None
   else:
      # This may happen if an intf name is used as profile name
      if intfConfig.profile != profileMode:
         return None
   return intfConfig

def getOrCreateNatIntfConfig( intfName, profileMode, mode=None, vrfName=None ):
   intfConfig = natConfig.intfConfig.get( intfName )
   if not intfConfig:
      intfConfig = natConfig.intfConfig.newMember( intfName, profileMode )
   else:
      # This may happen if an intf name is used as profile name
      if intfConfig.profile != profileMode:
         return None
   # intfConfig.vrfName priority
   # 1. vrfName specified ( via profile creating CLI )
   # 2. existing intfConfig.vrfName
   # 3. current nat profile CLI sub-mode's vrfName
   #    ( most of the time intfConfig and natProfileMode have the same lifetime,
   #      unless we add and remove the same nat config in the middle of creating
   #      a nat profile )
   # 4. DEFAULT_VRF by default
   if profileMode:
      newVrf = None
      if vrfName:
         newVrf = vrfName
      elif not intfConfig.vrfName:
         newVrf = getattr( mode, 'vrfName', None ) or DEFAULT_VRF
      oldVrf = intfConfig.vrfName
      if oldVrf and newVrf and oldVrf != newVrf:
         # reinitiate intf config. Delete all NAT configurations, but keep
         # the information w.r.t which interfaces are associated with the profile
         # pylint: disable-next=unnecessary-comprehension
         profileIntfs = [ intf for intf in intfConfig.profileIntf ]
         _, warning = vrfChangeHook( intfName, oldVrf, newVrf, False )
         if mode:
            mode.addWarning( warning )
         intfConfig = natConfig.intfConfig.newMember( intfName, profileMode )
         for intf in profileIntfs:
            intfConfig.profileIntf[ intf ] = True
      if newVrf:
         intfConfig.vrfName = newVrf
   return intfConfig

# Returns the intf or profile name based on the mode. The second parameter
# returned is a boolean, True for profile mode, False for intf mode
def getIntfName( mode ):
   if hasattr( mode, 'intf' ):
      return ( mode.intf.name, False )
   else:
      return ( mode.profileName, True )

def deleteNatIntfConfig( intfName ):
   intfConfig = natConfig.intfConfig.get( intfName )
   if not intfConfig:
      return
   if( intfConfig.staticNat or
       intfConfig.dynamicNat or
       intfConfig.twiceNat or
       intfConfig.profileIntf or
       intfConfig.flowPolicy ):
      return
   del natConfig.intfConfig[ intfName ]

def getOrCreateNatVrfConfig( vrfName ):
   vrfConfig = natConfig.vrfConfig.get( vrfName )
   if not vrfConfig:
      vrfConfig = natConfig.vrfConfig.newMember( vrfName )
   return vrfConfig

matcherNat = KeywordMatcher( 'nat', helpdesc='Network Address Translation (NAT)' )
matcherTranslation = KeywordMatcher( 'translation', helpdesc='NAT translation rule' )
matcherSource = KeywordMatcher( 'source',
      helpdesc='Translation of source addresses' )
matcherDestination = KeywordMatcher( 'destination',
      helpdesc='Translation of destination addresses' )
matcherTarget = EnumMatcher(
      { m.helpname_ : m.helpdesc_ for m in [ matcherSource, matcherDestination ] } )
matcherStatic = KeywordMatcher( 'static', helpdesc='Static NAT' )
matcherProto = EnumMatcher( { 'tcp': 'TCP', 'udp': 'UDP' } )
matcherInterface = KeywordMatcher( 'interface', helpdesc='Name of the interface' )
matcherProfile = KeywordMatcher( 'profile', helpdesc='NAT profile' )
matcherProfileName = DynamicNameMatcher( lambda mode: natConfig.intfConfig,
                                         pattern='[A-Za-z0-9_-]+',
                                         helpdesc='NAT profile name' )
vrfExprFactory = VrfExprFactory(
      helpdesc='VRF',
      inclAllVrf=True )
matcherAddress = KeywordMatcher(
      'address', helpdesc='NAT translation for an address' )
matcherIpAddr = IpAddrMatcher.IpAddrMatcher( helpdesc='Inside host IP address' )
matcherIpLocal = IpAddrMatcher.IpAddrMatcher( helpdesc='Original address in packet' )
matcherIpGlobal = IpAddrMatcher.IpAddrMatcher( helpdesc='Translated address' )
matcherSrcPort = IntegerMatcher( 1, 65535, helpdesc='Inside host port' )
nameMatcher = partial( PatternMatcher, pattern='[A-Za-z0-9_-]+', helpname='WORD' )
matcherPoolName = DynamicNameMatcher( lambda mode: natConfig.poolConfig,
                                      helpdesc='NAT pool name' )
matcherPrefixValue = IntegerMatcher( 16, 32,
      helpdesc='Length of the prefix in bits' )
matcherServiceListName = nameMatcher( helpdesc='NAT service-list name' )
matcherFlowPolicyName = nameMatcher( helpdesc='NAT flow policy name' )
matcherFlowMatchName = nameMatcher( helpdesc='NAT flow match name' )
matcherFlowActionName = nameMatcher( helpdesc='NAT flow action name' )
matcherIpPrefix = IpAddrMatcher.IpPrefixMatcher( helpdesc='Prefix to match on',
                                                 overlap=IpAddrMatcher.\
                                                       PREFIX_OVERLAP_REJECT )
matcherMacAddr = MacAddr.macAddrMatcher

EthIntfType = EthIntfCli.EthPhyAutoIntfType
PortChannelType = LagIntfCli.LagAutoIntfType
SwitchIntfType = SwitchIntfCli.SwitchAutoIntfType
VlanIntfType = VlanIntfCli.VlanAutoIntfType
matcherFlowIntf = IntfRange.IntfRangeMatcher( explicitIntfTypes=[ EthIntfType,
                                                                  PortChannelType,
                                                                  SwitchIntfType,
                                                                  VlanIntfType ] )

nodeNat = Node( matcher=matcherNat, guard=natSupported )
nodeCounters = guardedKeyword( 'counters', 'NAT counter information',
                               natKernelSupported )
nodeLogging = guardedKeyword( 'logging', 'NAT event logging',
                              natLoggingSupported )
nodeFlow = guardedKeyword( 'flow', 'NAT flow', natFlowSupported )
nodePool = guardedKeyword( 'pool', 'NAT Pool many-to-many translation',
                           natIntfDynamicSupported )
nodeNatForHardware = guardedKeyword( 'nat',
      'Hardware parameters related to NAT', natSupported )

nodeServiceList = guardedKeyword( 'service-list', 'NAT service-list',
                                  natServiceListSupported )

class NatAclExpression( CliExpression ):
   expression = "( access-list | acl )"
   data = {
            'access-list' : guardedKeyword(
               'access-list', 'IP ACL for use as NAT filter', natAclSupported ),
            'acl' : guardedKeyword(
               'acl', 'IP ACL for use as NAT filter', natAclSupported,
               priority=PRIO_NORMAL, hidden=True ),
          }

#------------------------------------------------------------------------------------
# (config)# ip nat profile <name> [vrf <name>]
#------------------------------------------------------------------------------------

def vrfChangeHook( intfName, oldVrf, newVrf, vrfDelete ):
   intfConfig = natConfig.intfConfig.get( intfName )
   if not intfConfig:
      return ( True, None )
   del natConfig.intfConfig[ intfName ]

   intfId = natStatus.intfStatusKey.get( intfName )
   if not intfId:
      return ( True, None )
   Tac.waitFor( lambda: intfId not in natStatus.intfStatusKey,
                description="%s's NAT status to be cleaned" % intfName,
                maxDelay=0.5, timeout=60, sleep=True )
   warnMessage = "NAT configurations on %s removed because of VRF changed" % intfName

   return ( True, warnMessage )

class NatProfileConfigMode( NatProfileMode, BasicCli.ConfigModeBase ):
   name = 'NAT profile configuration'

   def __init__( self, parent, session, profileName, vrfName ):
      NatProfileMode.__init__( self, ( profileName, vrfName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      intfConfig = getOrCreateNatIntfConfig( profileName, True, self, vrfName )
      if not intfConfig:
         return
      assert intfConfig.vrfName
      self.vrfName = intfConfig.vrfName
      if not vrfExists( self.vrfName ) and not self.session_.startupConfig():
         # Do not generate the warning while processing startup-config
         self.addWarning( "VRF %s doesn't exist. Config doesn't take effect"
                          " until the VRF is configured" % self.vrfName )
      intfConfig.vrfName = self.vrfName

def gotoIpNatProfileConfigMode( mode, args ):
   profileName = args[ 'PROFILENAME' ]
   vrfName = args.get( 'VRF' )
   childMode = mode.childMode( NatProfileConfigMode,
                               profileName=profileName, vrfName=vrfName )
   mode.session_.gotoChildMode( childMode )

def deleteIpNatProfileConfig( mode, args ):
   profileName = args[ 'PROFILENAME' ]
   intfConfig = getNatIntfConfig( profileName, True )
   if not intfConfig:
      return

   for rule in intfConfig.staticNat:
      if intfConfig.staticNat[ rule ].comment:
         natConfig.commentCount -= 1
   intfConfig.staticNat.clear()

   for rule in intfConfig.dynamicNat:
      if intfConfig.dynamicNat[ rule ].comment:
         natConfig.commentCount -= 1
   intfConfig.dynamicNat.clear()

   for rule in intfConfig.twiceNat:
      if intfConfig.twiceNat[ rule ].comment:
         natConfig.commentCount -= 1
   intfConfig.twiceNat.clear()

   # The intf list associated with the profile is not cleared. As long as at least
   # one intf is using the profile, it is not deleted
   if not intfConfig.profileIntf:
      del natConfig.intfConfig[ profileName ]

#------------------------------------------------------------------------------------
# (config-if)# ip nat service-profile <name>
#------------------------------------------------------------------------------------

def setIntfNatProfile( mode, args ):
   profileName = args[ 'PROFILENAME' ]
   intfName = mode.intf.name
   intfId = Tac.Value( "Arnet::IntfId", intfName )
   currentProfile = natConfig.profileConfig.get( intfId )
   if currentProfile and currentProfile != profileName:
      # If the intf is configured in a different profile, remove it first
      currentIntfConfig = getNatIntfConfig( currentProfile, True )
      assert currentIntfConfig != None # pylint: disable=singleton-comparison
      del currentIntfConfig.profileIntf[ intfId ]
      deleteNatIntfConfig( currentProfile )

   intfConfig = getOrCreateNatIntfConfig( profileName, True, mode )
   if not intfConfig:
      return
   ipIntf = IraIpIntfCli.IpIntf( mode.intf, mode.sysdbRoot, createIfMissing=True )
   if intfConfig.vrfName != ipIntf.getStatus().vrf and \
         not mode.session_.startupConfig():
      # Do not generate the warning while processing startup-config
      mode.addWarning( "NAT service profile not attached to the interface until "
                       "profile's VRF (currently %s) is the same as interface's VRF "
                       "(currently %s)" %
                       ( intfConfig.vrfName, ipIntf.getStatus().vrf ) )
   intfConfig.profileIntf[ intfId ] = True
   natConfig.profileConfig[ intfId ] = profileName

def clearIntfNatProfile( mode, args ):
   intfName = mode.intf.name
   intfId = Tac.Value( "Arnet::IntfId", intfName )
   profileName = natConfig.profileConfig.get( intfId )
   if profileName is not None:
      del natConfig.profileConfig[ intfId ]
      intfConfig = getNatIntfConfig( profileName, True )
      assert intfConfig != None # pylint: disable=singleton-comparison
      del intfConfig.profileIntf[ intfId ]
      deleteNatIntfConfig( profileName )

#------------------------------------------------------------------------------------
# show ip nat profile [<name>]
#------------------------------------------------------------------------------------

def cmdShowIpNatProfile( mode, profileName, model ):
   intfConfig = getNatIntfConfig( profileName, True )
   if not intfConfig:
      mode.addError( "Nat profile %s not configured\n" % profileName )
      return

   intfId = natStatus.intfStatusKey.get( profileName )
   if not intfId:
      return
   intfStatus = natStatus.intfStatus.get( intfId )
   if not intfStatus:
      return
   profileId = intfStatus.profileOrVlanId()

   p = NatProfileModel.NatProfile()
   p.profileName = intfConfig.name
   p.profileId = profileId
   p.vrfName = intfStatus.vrfName
   p.intfList = {}
   for intfId in intfConfig.profileIntf:
      vlanId = intfStatus.validIntf.get( intfId )
      if vlanId is None:
         vlanId = 0
      p.intfList[ intfId ] = vlanId

   # If NAT with VxLAN is enabled, add dynamicVlans to the output
   if natHwCapabilities.natWithVxlanSupported:
      for vlanId in vxlanNatStatus.dynVlanNatProfile:
         vlanIntfId = Tac.Value( "Arnet::IntfId", f"Vlan{vlanId}" )
         p.intfList[ vlanIntfId ] = vlanId

   model.profiles.append( p )

def cmdShowIpNatProfiles( mode, args ):
   profileName = args.get( 'PROFILE' )
   model = NatProfileModel()
   if profileName != None: # pylint: disable=singleton-comparison
      cmdShowIpNatProfile( mode, profileName, model )
   else:
      for profileName in natConfig.intfConfig:
         cmdShowIpNatProfile( mode, profileName, model )
   return model

#------------------------------------------------------------------------------------
# (config-if)# ip nat source|destination static local-ip <addr> [port <value>]
#     [acl <name>] global-ip <addr> [port <value>] [protocol tcp|udp] [group <id>]
#------------------------------------------------------------------------------------
def checkOneIsAnyAndOtherNot( target, targetAddr, other, otherAddr ):
   if targetAddr.address != '0.0.0.0' and targetAddr.mask != 0:
      raise ValueError( "%s address must be 'any'" % target )

   if otherAddr.address == '0.0.0.0' or otherAddr.mask == 0:
      raise ValueError( "%s address cannot be 'any'" % other )

   return "(%s, %s)" % ( targetAddr.stringValue, otherAddr.stringValue )

def checkDestinationRule( rule, target=None, isStaticNat=True ):
   if isStaticNat and rule.action != AclAction.permit:
      raise ValueError( "action must be 'permit'" )

   if rule.filter.proto != 0 and not natHwCapabilities.natAclProtocolSupported:
      raise ValueError( "protocol match rule not allowed" )

   source = rule.filter.source
   destination = rule.filter.destination
   if target is None:
      return "(%s, %s)" % ( source.stringValue, destination.stringValue )

   if target == 'source':
      return checkOneIsAnyAndOtherNot( target, source, 'destination', destination )
   else:
      return checkOneIsAnyAndOtherNot( target, destination, 'source', source )

def stopIfStaticNatAclIsInvalid( mode, aclName, target ):
   if not aclName:
      return

   acl = aclConfig.config[ 'ip' ].acl[ aclName ].currCfg
   for seq, uid in acl.ruleBySequence.items():
      try:
         checkDestinationRule( acl.ipRuleById[ uid ], target )
      except ValueError as e:
         mode.addErrorAndStop(
            "Acl {acl} contains rules invalid as static nat destination filter.\n"
            "Try 'show ip nat acl {acl}' for more information.\n"
            "Error: Seq no {seq}: {e} ".format( acl=aclName, seq=seq, e=e ) )

def stopIfUnicastMulticastAcl( mode, aclName, target ):
   if not aclName:
      return

   acl = aclConfig.config[ 'ip' ].acl[ aclName ].currCfg
   unicastIpUsed = False
   multicastIpUsed = False
   for uid in acl.ruleBySequence.values():
      rule = acl.ipRuleById[ uid ]
      filterIpWithMask = ( rule.filter.destination if target == 'source' else
                           rule.filter.source )
      filterIpAddr = Arnet.IpAddr( filterIpWithMask.address )
      if filterIpAddr.isMulticast: # pylint: disable-msg=simplifiable-if-statement
         multicastIpUsed = True
      else:
         unicastIpUsed = True
      if multicastIpUsed and unicastIpUsed:
         break

   if multicastIpUsed:
      if target == 'destination':
         mode.addErrorAndStop( "Access list cannot contain a filter where "
                               "source address is a multicast address" )
      elif unicastIpUsed:
         mode.addErrorAndStop( "Access list cannot contain a combination "
                               "of unicast and multicast filters" )

def checkStaticNatLimitedMulticastWarning( mode, target, localIp, globalIp,
                                          globalPort ):
   # Show warning if we're programming a multicast rule with
   # a currently unsupported translation type.
   # We can only show the warning for destination rules since
   # we don't know yet if a given source rule is multicast or not.
   if target == 'destination':
      if ( Arnet.IpAddr( localIp ).isMulticast or
           Arnet.IpAddr( globalIp ).isMulticast ):
         if globalPort:
            translationType = McastTranslationType.destinationPortTranslation
            translationName = "destination-port"
         else:
            translationType = McastTranslationType.destinationTranslation
            translationName = "destination"
         checkMulticastTranslation( mode, translationType, translationName )


def checkMulticastTranslation( mode, translationType, translationName ):
   # Check if the translationType is in the translationMode collection
   # or if we're using the default mode types (dest, source, twice)
   if ( translationType not in mcastTranslationMode.translationType and
        not ( not mcastTranslationMode.translationType and
              ( translationType ==
                McastTranslationType.destinationTranslation or
                translationType ==
                McastTranslationType.twiceTranslation ) ) ):
      mode.addWarning( "Unable to program Multicast translation of "
                       "type %s.\nConfig doesn't take effect "
                       "until %s translations are enabled." %
                       ( translationName, translationName ) )


def directionWithUnicastDestination( mode, target, localIp, globalIp, ingress,
                                     twice ):
   # Block config if we're programming an ingress/egress unicast rule.
   # We can only block destination rules since we don't know yet if
   # a given source rule is multicast or not until routes are configured.
   if not ( Arnet.IpAddr( localIp ).isMulticast or
            Arnet.IpAddr( globalIp ).isMulticast ):
      if target == 'destination':
         if ingress:
            if twice:
               # Unicast destination ingress twice NAT
               mode.addError( "Only multicast is supported for twice destination "
                              "ingress NAT" )
               return True
         else:
            if not twice:
               # Unicast destination egress NAT
               mode.addError( "Only multicast is supported for destination "
                              "egress NAT" )
               return True
   return False

def checkTwiceNatProtocolType( mode, protocol, twiceNatConfig ):
   if twiceNatConfig.protocol != protocol:
      mode.addError( "Protocol %s does not match destination Nat protocol "
                     "%s already configured for this group" %
                     ( ipProtoMap.get( protocol ),
                       ipProtoMap.get( twiceNatConfig.protocol ) ) )
      return False
   return True

def checkTwiceNatAsymmetricPortConfig( mode, target, localPort, globalPort,
                                       dynTwiceFound, twiceNatConfig ):
   if not isTwiceNatAsymmetricPortConfigSupported():
      if target == 'source':
         if twiceNatConfig.dstIpAndPortOld.port and not globalPort:
            mode.addError( "L4 port (%d) configured in Destination NAT but "
                           "missing in Source NAT for this group" %
                           twiceNatConfig.dstIpAndPortOld.port )
            return False
         if( globalPort and not twiceNatConfig.dstIpAndPortOld.port and
             not dynTwiceFound ):
            mode.addError( "L4 port (%d) configured in Source NAT but missing "
                           "in Destination NAT for this group" % globalPort )
            return False
      if target == 'destination':
         if twiceNatConfig.srcIpAndPortOld.port and not globalPort :
            mode.addError( "L4 port (%d) configured in Source NAT but missing "
                           "in Destination NAT for this group" %
                           twiceNatConfig.srcIpAndPortOld.port )
            return False
         if( globalPort and not twiceNatConfig.srcIpAndPortOld.port and
             not dynTwiceFound ):
            mode.addError( "L4 port (%d) configured in Destination NAT but "
                           "missing in Source NAT for this group" %
                           globalPort )
            return False
   return True

def checkTwiceNatAsymmetricSamePort( mode, target, localPort, globalPort,
                                     dynTwiceFound, twiceNatConfig ):
   if ( isTwiceNatAsymmetricPortConfigSupported() and
        natHwCapabilities.natTwiceAsymmetricSamePortRequired ) :
      errorMessage = ( "For Twice NAT rules where the port is specified in "
                       "only one direction, the original and translated "
                       "ports must be the same" )
      if target == 'source':
         if twiceNatConfig.dstIpAndPortOld.port and not globalPort:
            # Condition where the L4 ports are specified
            # in the destination rule but not in the source rule
            if ( twiceNatConfig.dstIpAndPortOld.port !=
                 twiceNatConfig.dstIpAndPortNew.port ):
               # If a port translation is configured and
               # the platform only supports asymmetric qualification
               # CLI should error out since platform does not support
               # asymmetric port translation
               mode.addError( errorMessage )
               return False
         if( not twiceNatConfig.dstIpAndPortOld.port and
             not dynTwiceFound and globalPort ):
            # Condition where the L4 ports are specified
            # in the source rule but not the in destination rule
            if globalPort != localPort:
               # If a port translation is configured for just this rule and
               # the platform only supports asymmetric qualification
               # CLI should error out
               mode.addError( errorMessage )
               return False
      if target == 'destination':
         if twiceNatConfig.srcIpAndPortOld.port and not globalPort:
            # Condition where the L4 ports are specified
            # in the source rule but not the in destination rule
            if ( twiceNatConfig.srcIpAndPortOld.port !=
                 twiceNatConfig.srcIpAndPortNew.port ) :
               # If a port translation is configured and
               # the platform only supports asymmetric qualification
               # CLI should error out
               mode.addError( errorMessage )
               return False
         if( not twiceNatConfig.srcIpAndPortOld.port and
             not dynTwiceFound and globalPort ):
            # Condition where the L4 ports are specified
            # in the destination rule but not in the source rule
            if globalPort != localPort:
               # If a port translation is configured for just this rule and
               # the platform only supports asymmetric qualification
               # CLI should error out
               mode.addError( errorMessage )
               return False
   return True

def checkTwiceNatDirection( mode, ingress, twiceNatConfig ):
   if twiceNatConfig.ingress != ingress:
      mode.addError( "%s does not match Destination NAT direction "
                     "already configured for this group" %
                     ( "Ingress" if ingress else "Egress" ) )
      return False
   return True

def checkTwiceNatLimitedMulticastWarning( mode, target, localPort, globalPort,
                                          localIp, globalIp, twiceNatConfig ):
   isMulticast = False
   sourcePortTranslation = False
   destPortTranslation = False
   if target == 'source':
      if ( Arnet.IpAddr( twiceNatConfig.dstIpAndPortOld.ip ).isMulticast or
           Arnet.IpAddr( twiceNatConfig.dstIpAndPortNew.ip ).isMulticast ):
         isMulticast = True
         sourcePortTranslation = ( localPort != 0 and localPort != globalPort )
         destPortTranslation = ( twiceNatConfig.dstIpAndPortNew.port != 0 and
                                 twiceNatConfig.dstIpAndPortNew.port !=
                                 twiceNatConfig.dstIpAndPortOld.port )
   if target == 'destination':
      if ( Arnet.IpAddr( localIp ).isMulticast or
           Arnet.IpAddr( globalIp ).isMulticast ):
         isMulticast = True
         destPortTranslation = ( localPort != 0 and localPort != globalPort )
         sourcePortTranslation = ( twiceNatConfig.srcIpAndPortNew.port != 0 and
                                   twiceNatConfig.srcIpAndPortNew.port !=
                                   twiceNatConfig.srcIpAndPortOld.port )
   # Set the multicast translation type to twice port
   # if there are any port translations in the configuraiton.
   # Otherwise set the multicast translation type to twiceTranslation
   if isMulticast:
      if sourcePortTranslation or destPortTranslation:
         translationType = McastTranslationType.twicePortTranslation
         translationName = "twice-port"
      else:
         translationType = McastTranslationType.twiceTranslation
         translationName = "twice"
      checkMulticastTranslation( mode, translationType, translationName )


def twiceNatGroupValidation( mode, target, localPort, globalPort, localIp, globalIp,
                             protocol, ingress, dynTwiceFound, twiceNatConfig ):
   if not checkTwiceNatProtocolType( mode, protocol, twiceNatConfig ) :
      return False
   if not checkTwiceNatAsymmetricPortConfig( mode, target, localPort, globalPort,
                                             dynTwiceFound, twiceNatConfig ):
      return False
   if not checkTwiceNatAsymmetricSamePort( mode, target, localPort, globalPort,
                                           dynTwiceFound, twiceNatConfig ):
      return False
   if not checkTwiceNatDirection( mode, ingress, twiceNatConfig ):
      return False
   if natHwCapabilities.natLimitedMulticastTranslations:
      checkTwiceNatLimitedMulticastWarning( mode, target, localPort,
                                            globalPort, localIp, globalIp,
                                            twiceNatConfig )
   return True

def cmdTwiceNat( mode, group, target, localIp, globalIp,
                 localPort, globalPort, protocol, ingress, comment='' ):

   if ( localPort > 0 and globalPort == 0 ) or ( localPort == 0 and globalPort > 0 ):
      mode.addError( "The original and translated ports must be either both 0 or "
                     "both non 0" )
      return False

   if natHwCapabilities.natDirectionMulticastOnly:
      if directionWithUnicastDestination( mode, target, localIp, globalIp,
                                          ingress, True ):
         return False

   # Must create the intfConfig after verifying the rule is valid, otherwise
   # we will create an empty intfConfig
   intfName, profileMode = getIntfName( mode )
   intfConfig = getOrCreateNatIntfConfig( intfName, profileMode, mode )
   if not intfConfig:
      return

   # If dynamic Twice NAT is supported, make sure that there isn't any
   # conflicting configuration present
   dynTwiceFound = False
   if natHwCapabilities.natDynamicTwiceSupported:
      for dynNat in intfConfig.dynamicNat.values():
         if dynNat.dynamicKey.group == group:
            if target == dynNat.target:
               mode.addError( "A dynamic %s entry for the group is "
                              "already configured" % target )
               return
            else:
               if( not isTwiceNatAsymmetricPortConfigSupported() and
                   not localPort ):
                  mode.addError( "Twice NAT with asymmetric port configuration "
                                 "is not allowed in the current profile" )
                  return
               dynTwiceFound = True

   # If group is already present then add config to existing group.
   # Make sure there are no conflicts.
   if group in intfConfig.twiceNat:
      twiceNatConfig = intfConfig.twiceNat[ group ]

      if target == 'source':
         if twiceNatConfig.dstIpAndPortOld.ip != '0.0.0.0':
            if not twiceNatGroupValidation( mode, target, localPort, globalPort,
                                            localIp, globalIp, protocol, ingress,
                                            dynTwiceFound, twiceNatConfig ):
               return False

      if target == 'destination':
         if twiceNatConfig.srcIpAndPortOld.ip != '0.0.0.0':
            if not twiceNatGroupValidation( mode, target, localPort, globalPort,
                                            localIp, globalIp, protocol, ingress,
                                            dynTwiceFound, twiceNatConfig ):
               return False
   else:
      twiceNatConfig = intfConfig.twiceNat.newMember( group )

   localAddress = Tac.Value( "Arnet::IpAndPort", localIp, localPort )
   globalAddress = Tac.Value( "Arnet::IpAndPort", globalIp, globalPort )

   # Make sure that this configuration doesn't exists already in a different group
   if target == 'source':
      if twiceNatConfig.dstIpAndPortOld.ip != '0.0.0.0':
         for tGrp, tCfg in intfConfig.twiceNat.items():
            if tGrp != group and tCfg.srcIpAndPortOld == localAddress and \
               tCfg.dstIpAndPortOld == twiceNatConfig.dstIpAndPortOld:
               if not protocol or not tCfg.protocol or protocol == tCfg.protocol:
                  mode.addError( "Twice nat rules already configured in group %d" %
                                 tGrp )
                  return False
   else:
      if twiceNatConfig.srcIpAndPortOld.ip != '0.0.0.0':
         for tGrp, tCfg in intfConfig.twiceNat.items():
            if tGrp != group and tCfg.dstIpAndPortOld == localAddress and \
               tCfg.srcIpAndPortOld == twiceNatConfig.srcIpAndPortOld:
               if not protocol or not tCfg.protocol or protocol == tCfg.protocol:
                  mode.addError( "Twice nat rules already configured in group %d" %
                                 tGrp )
                  return False

   if natHwCapabilities.natDirectionMulticastOnly:
      if directionWithUnicastDestination( mode, target, localIp, globalIp,
                                          ingress, True ):
         return False

   if target == 'source':
      twiceNatConfig.srcIpAndPortOld = localAddress
      twiceNatConfig.srcIpAndPortNew = globalAddress
   else:
      twiceNatConfig.dstIpAndPortOld = localAddress
      twiceNatConfig.dstIpAndPortNew = globalAddress
   twiceNatConfig.protocol = protocol
   twiceNatConfig.ingress = ingress

   if comment and not twiceNatConfig.comment:
      natConfig.commentCount += 1
   if not comment and twiceNatConfig.comment:
      natConfig.commentCount -= 1
   twiceNatConfig.comment = comment

   if( twiceNatConfig.srcIpAndPortOld.ip != '0.0.0.0' or
       twiceNatConfig.dstIpAndPortOld.ip != '0.0.0.0' ):
      twiceNatConfig.version += 1
   return True

def cmdNoTwiceNat( mode, group, target, localIp, localPort ):

   intfName, profileMode = getIntfName( mode )
   intfConfig = getNatIntfConfig( intfName, profileMode )
   if not intfConfig:
      return

   if group not in intfConfig.twiceNat:
      return

   defaultAddress = Tac.Value( "Arnet::IpAndPort", '0.0.0.0', 0 )
   localAddress = Tac.Value( "Arnet::IpAndPort", localIp, localPort )
   twiceNatConfig = intfConfig.twiceNat[ group ]
   warnStr = '(%s, %s) does not exist' % ( localIp, group )
   if target == 'source':
      if( localAddress != defaultAddress and
          localAddress != twiceNatConfig.srcIpAndPortOld ):
         mode.addWarning( warnStr )
         return
      twiceNatConfig.srcIpAndPortOld = defaultAddress
      twiceNatConfig.srcIpAndPortNew = defaultAddress
   else:
      if( localAddress != defaultAddress and
          localAddress != twiceNatConfig.dstIpAndPortOld ):
         mode.addWarning( warnStr )
         return
      twiceNatConfig.dstIpAndPortOld = defaultAddress
      twiceNatConfig.dstIpAndPortNew = defaultAddress

   if( twiceNatConfig.srcIpAndPortOld == defaultAddress and
       twiceNatConfig.dstIpAndPortOld == defaultAddress ):
      if twiceNatConfig.comment:
         natConfig.commentCount -= 1
      del intfConfig.twiceNat[ group ]
   else:
      twiceNatConfig.version += 1

def isNatEgressMcastStaticSupported():
   # Assume that egress mcast is supported if hwCapabilities
   # have not been set yet.
   # This happens during initial config load at dut startup
   return ( natHwCapabilities.natEgressMcastStaticSupported or
            natHwCapabilities.version == 0 )

def isNatEgressMcastStaticAclSupported():
   # Assume that egress mcast is supported if hwCapabilities
   # have not been set yet.
   # This happens during initial config load at dut startup
   return ( natHwCapabilities.natEgressMcastStaticAclSupported or
            natHwCapabilities.version == 0 )

def isNatIngressTwiceConfigSupported():
   # Assume that ingress twice mcast is supported if hwCapabilities
   # have not been set yet.
   # This happens during initial config load at dut startup
   return ( natHwCapabilities.natIngressTwiceConfigSupported or
            natHwCapabilities.version == 0 )

def isTwiceNatAsymmetricPortConfigSupported():
   # Assume that twice nat asymmetric port config is supported
   # if hwCapabilities have not been set yet.
   # This happens during initial config load at dut startup
   return ( natHwCapabilities.natTwiceAsymmetricPortConfigSupported or
            natHwCapabilities.version == 0 )

def isDnatMcast( target, egress, localIp, globalIp=None ):

   def isMcastAddr( addr ):
      if not addr:
         return False
      if not IpAddrMatcher.isValidIpAddr( addr ):
         return False
      c = IpAddrMatcher.splitIpAddrToInts( addr )
      return c[ 0 ] >= 224 and c[ 0 ] <= 239

   return ( target == 'destination' and egress and
           ( isMcastAddr( localIp ) or isMcastAddr( globalIp ) ) )

def isSnatMcast( target, egress ):
   # Egress keyword check which is specific for platforms that
   # support NatEgressMcastStatic
   return isNatEgressMcastStaticSupported() and target == 'source' and egress

def cmdStaticNat( mode, args ):
   target = 'source' if 'source' in args else 'destination'
   localIp = args[ 'LOCALIP' ]
   globalIp = args[ 'GLOBALIP' ]
   aclName = args.get( 'ACLNAME', '' )
   localPort = args.get( 'LOCALPORT', 0 )
   globalPort = args.get( 'GLOBALPORT', 0 )
   protocol = args.get( 'PROTO' )
   protocol = { 'tcp' : IPPROTO_TCP, 'udp' : IPPROTO_UDP }.get( protocol, 0 )
   prefixLen = args.get( 'PREFIXLEN', 0 )
   group = args.get( 'GROUP', 0 )
   egress = 'SNAT_EGRESS' in args or 'DNAT_EGRESS' in args
   ingress = 'SNAT_INGRESS' in args or 'DNAT_INGRESS' in args
   comment = args.get( 'COMMENT', '' )

   dnatEgressMcast = isDnatMcast( target, egress, localIp, globalIp )
   snatEgressMcast = isSnatMcast( target, egress )
   if dnatEgressMcast:
      if IpAddrMatcher.validateMulticastIpAddr( globalIp ) or \
         IpAddrMatcher.validateMulticastIpAddr( localIp ):
         mode.addError( "Both local and global address must be "
                        "configured as multicast." )
         return

      if aclName and not isNatEgressMcastStaticAclSupported():
         mode.addError( "Acl not allowed for static egress multicast NAT" )
         return

      if localPort and protocol != IPPROTO_UDP:
         mode.addError( "Only UDP is supported when a port is configured" )
         return

      if not localPort and protocol:
         mode.addError( "The protocol can be specified only if a port is "
                        "configured" )
         return

   natTable = ( 'egress' if ( dnatEgressMcast or snatEgressMcast )
                else 'defaultImpl' )

   if target =='source' and natTable == 'egress':
      if aclName:
         mode.addError( "Acl not allowed for source egress multicast NAT" )
         return
      if localPort:
         if protocol != IPPROTO_UDP:
            mode.addError( "Only UDP is supported when a port is configured for "
                           "source egress multicast NAT" )
            return
      else:
         if protocol == IPPROTO_TCP:
            mode.addError( "TCP is not supported for source egress multicast NAT" )
            return
         elif protocol == IPPROTO_UDP:
            mode.addError( "UDP is only supported when a port is configured for "
                           "source egress multicast NAT" )
            return

   if group and ( ingress or egress ):
      if not isNatIngressTwiceConfigSupported():
         mode.addError( 'No ingress or egress can be specified if group is used' )
         return
   
   # /32 is the same as not specifying 'prefix-length' for static NAT so remove it
   if prefixLen == 32:
      prefixLen = 0

   # Discard any host bits before performing other checks
   if natHwCapabilities.natSwPrefixTranslationSupported and prefixLen:
      localIp = Arnet.Subnet( localIp, prefixLen ).toValue().address
      globalIp = Arnet.Subnet( globalIp, prefixLen ).toValue().address

   if ( IpAddrMatcher.isMartianIpAddr( globalIp ) or
        IpAddrMatcher.isMartianIpAddr( localIp ) ):
      mode.addError( "Not a valid host address" )
      return

   if globalIp == '0.0.0.0' or localIp == '0.0.0.0':
      mode.addError( "IP address 0.0.0.0 not supported in NAT configuration" )
      return

   if globalPort != 0 and localPort == 0:
      mode.addError( "The translated port is allowed only if the original "
                     "port is specified" )
      return

   if not natHwCapabilities.natUnicastMulticastAclSupported and target == 'source':
      if Arnet.IpAddr( localIp ).isMulticast or Arnet.IpAddr( globalIp ).isMulticast:
         mode.addError( "Multicast addresses are not supported for source NAT" )
         return

   if group:
      if aclName:
         mode.addError( "Acl not allowed in Twice Nat config. Please remove "
                        "and try again" )
         return
      cmdTwiceNat( mode, group, target, localIp, globalIp, localPort, globalPort,
                   protocol, ingress, comment )
      return

   if aclName:
      if aclName in aclConfig.config[ 'ip' ].acl:
         stopIfStaticNatAclIsInvalid( mode, aclName, target )
         if not natHwCapabilities.natUnicastMulticastAclSupported:
            stopIfUnicastMulticastAcl( mode, aclName, target )
      elif not mode.session_.startupConfig():
         # Do not generate the warning while processing startup-config
         mode.addWarning( "Acl %s not configured. Assigning anyway." % aclName )

   ingressDir = ingress or ( target == 'destination' and not egress )
   localAddress = Tac.Value( "Arnet::IpAndPort", localIp, localPort )
   globalAddress = Tac.Value( "Arnet::IpAndPort", globalIp, globalPort )
   staticKey = Tac.Value( "Ip::Nat::StaticKey", localAddress, aclName, natTable )
   staticNatConfig = Tac.Value( "Ip::Nat::NatStaticConfig", staticKey,
                                globalAddress, target, protocol, prefixLen,
                                ingressDir, comment )

   if natHwCapabilities.natDirectionMulticastOnly:
      if directionWithUnicastDestination( mode, target, localIp, globalIp,
                                          ingressDir, False ):
         return

   if natHwCapabilities.natLimitedMulticastTranslations:
      checkStaticNatLimitedMulticastWarning( mode, target, localIp, globalIp,
                                             globalPort )

   # Must create the intfConfig after verifying the rule is valid, otherwise
   # we will create an empty intfConfig
   intfName, profileMode = getIntfName( mode )
   intfConfig = getOrCreateNatIntfConfig( intfName, profileMode, mode )
   if not intfConfig:
      return

   if not natHwCapabilities.natStaticAclOverloadSupported or \
         natHwCapabilities.natSwPrefixTranslationSupported:
      newPrefix = prefixLen if prefixLen else 32
      newSubnet = Arnet.Subnet( localIp, newPrefix )
      for staticNat in intfConfig.staticNat.values():
         # Check for overlapping subnets in other NatStaticConfigs
         if natHwCapabilities.natSwPrefixTranslationSupported:
            # Skip further check if staticKeys match as staticNat will be overwritten
            if staticKey == staticNat.staticKey:
               continue

            currPrefix = staticNat.prefixLen if staticNat.prefixLen else 32
            currKey = staticNat.staticKey
            currSubnet = Arnet.Subnet( currKey.localAddress.ip,
                                       currPrefix )
            if newSubnet.overlapsWith( currSubnet ) and \
                  ( localPort == currKey.localAddress.port ) and \
                  ( aclName == currKey.acl ) and ( natTable == currKey.table ):
               msg = "Subnets overlap. Please delete existing NAT rule '%s/%s'" % \
                      ( currKey.localAddress.ip, currPrefix )
               msg += " before adding this one."
               mode.addError( msg )
               return

         if not natHwCapabilities.natStaticAclOverloadSupported:
            if staticNat.staticKey.localAddress == localAddress and \
                  staticNat.globalAddress == globalAddress and \
                  staticNat.staticKey.acl != aclName:
               msg = "Acl %s is already attached to the NAT rule." % \
                     staticNat.staticKey.acl
               msg += " Please delete that first to use new Acl %s." % aclName
               mode.addError( msg )
               return

   # remove (if present)
   if staticKey in intfConfig.staticNat and \
         intfConfig.staticNat[ staticKey ].comment:
      natConfig.commentCount -= 1
   del intfConfig.staticNat[ staticKey ]

   # remove rule in reverse direction if both aren't supported at once
   if not isNatEgressMcastStaticSupported():
      natTableReverse = 'egress' if natTable == 'defaultImpl' else 'defaultImpl'
      staticKeyReverse = Tac.Value( "Ip::Nat::StaticKey", localAddress, aclName,
                                    natTableReverse )

      if staticKeyReverse in intfConfig.staticNat and \
            intfConfig.staticNat[ staticKeyReverse ].comment:
         natConfig.commentCount -= 1
      del intfConfig.staticNat[ staticKeyReverse ]

   # add
   intfConfig.staticNat.addMember( staticNatConfig )
   if comment:
      natConfig.commentCount += 1

def cmdNoStaticNat( mode, args ):
   target = 'source' if 'source' in args else 'destination'
   localIp = args[ 'LOCALIP' ]
   aclName = args.get( 'ACLNAME', '' )
   localPort = args.get( 'LOCALPORT', 0 )
   group = args.get( 'GROUP' )
   egress = 'SNAT_EGRESS' in args or 'DNAT_EGRESS' in args

   intfName, profileMode = getIntfName( mode )
   intfConfig = getNatIntfConfig( intfName, profileMode )
   if not intfConfig:
      return

   if group:
      cmdNoTwiceNat( mode, group, target, localIp, localPort )
   else:
      localAddress = Tac.Value( "Arnet::IpAndPort", localIp, localPort )
      dnatEgressMcast = isDnatMcast( target, egress, localIp )
      snatEgressMcast = isSnatMcast( target, egress )

      natTable = ( 'egress' if ( dnatEgressMcast or snatEgressMcast )
                   else 'defaultImpl' )
      staticKey = Tac.Value( "Ip::Nat::StaticKey", localAddress, aclName, natTable )
      if staticKey in intfConfig.staticNat:
         if staticKey in intfConfig.staticNat and \
               intfConfig.staticNat[ staticKey ].comment:
            natConfig.commentCount -= 1
         del intfConfig.staticNat[ staticKey ]
      else:
         warnStr = 'ip %s' % localIp
         if localPort:
            warnStr += ', port %s' % localPort
         if aclName:
            warnStr += ', acl %s' % aclName
         warnStr = 'Static nat rule (%s) does not exist' % warnStr
         mode.addWarning( warnStr )

   deleteNatIntfConfig( intfName )

#------------------------------------------------------------------------------------
# (config)# ip nat <source|destination> static <local-ip> <global-ip>
#                  inside-vrf <vrf>, in global config mode
# (config-vrf)# ip nat <source|destination> static <local-ip> <global-ip>
#                  inside-vrf <vrf>, in vrf definition mode
#------------------------------------------------------------------------------------

def natVrfCleanup( vrfName ):
   # In case of vrf deletion, go through every nat vrf rule configured
   # and delete any rule where the outsideVrf is the one being deleted
   for vrf, config in natConfig.vrfConfig.items():
      for addr, value in config.staticNat.items():
         if value.outsideVrf == vrfName:
            del config.staticNat[ addr ]
            if len( config.staticNat ) == 0:
               del natConfig.vrfConfig[ vrf ]

def cmdStaticNatVrfPolicer( mode, args ):
   insideVrf = args[ 'INSIDE_VRF' ]
   rate = args[ 'RATE' ]
   burst = args[ 'BURST' ]
   if insideVrf == 'default':
      mode.addError( "Not supported for default vrf" )
      return
   if not vrfExists( insideVrf ):
      mode.addWarning( "Vrf %s does not exists. Assigning anyway." % insideVrf )
   insideVrfConfig = getOrCreateNatVrfConfig( insideVrf )
   if not insideVrfConfig:
      mode.addError( "Unable to create config" )
      return
   insideVrfConfig.sourcePolicer = Tac.Value( "Ip::Nat::VrfPolicer", rate, burst )

def cmdNoStaticNatVrfPolicer( mode, args ):
   insideVrf = args[ 'INSIDE_VRF' ]
   vrfConfig = natConfig.vrfConfig.get( insideVrf )
   if not vrfConfig:
      return
   vrfConfig.sourcePolicer = Tac.Value( "Ip::Nat::VrfPolicer" )

def cmdStaticNatVrf( mode, args ):
   target = args[ 'TARGET' ]
   localIp = args[ 'LOCALIP' ]
   globalIp = args[ 'GLOBALIP' ]
   insideVrf = args[ 'INSIDE_VRF' ]
   outsideVrf = getattr( mode, 'vrfName', DEFAULT_VRF )
   if not vrfExists( insideVrf ):
      mode.addWarning( "Vrf %s does not exists. Assigning anyway." % insideVrf )

   insideVrfConfig = getOrCreateNatVrfConfig( insideVrf )
   if not insideVrfConfig:
      return

   outsideVrfConfig = getOrCreateNatVrfConfig( outsideVrf )
   if not outsideVrfConfig:
      return

   localAddress = Tac.Value( "Arnet::IpAddr", stringValue=localIp )
   if localAddress.isMulticast or \
      localAddress.stringValue == Tac.Value( "Arnet::IpAddr" ).ipAddrZero or \
      localAddress.stringValue == Tac.Value( "Arnet::IpAddr" ).ipAddrBroadcast:
      mode.addError( "Local ip address value not allowed" )
      return

   globalAddress = Tac.Value( "Arnet::IpAddr", stringValue=globalIp )
   if globalAddress.isMulticast or \
      globalAddress.stringValue == Tac.Value( "Arnet::IpAddr" ).ipAddrZero or \
      globalAddress.stringValue == Tac.Value( "Arnet::IpAddr" ).ipAddrBroadcast:
      mode.addError( "Global ip address value not allowed" )
      return

   if target == 'source':
      srcAddrOld = localAddress
      srcAddrNew = globalAddress
   else:
      srcAddrOld = globalAddress
      srcAddrNew = localAddress

   # If either the local or global address is already configured in any
   # nat rule (with the same vrf), delete the existing one
   if srcAddrOld in insideVrfConfig.staticNat:
      rule = insideVrfConfig.staticNat[ srcAddrOld ]
      del insideVrfConfig.staticNat[ srcAddrOld ]
      outsideVrfConfigOld = natConfig.vrfConfig.get( rule.outsideVrf )
      if outsideVrfConfigOld and \
         rule.srcAddressNew in outsideVrfConfigOld.staticNatRev:
         del outsideVrfConfigOld.staticNatRev[ rule.srcAddressNew ]

   if srcAddrNew in outsideVrfConfig.staticNatRev:
      rule = outsideVrfConfig.staticNatRev[ srcAddrNew ]
      del outsideVrfConfig.staticNatRev[ srcAddrNew ]
      # The outsideVrf in the rev mapping contains the inside vrf
      insideVrfConfigOld = natConfig.vrfConfig.get( rule.outsideVrf )
      if insideVrfConfigOld and rule.srcAddressOld in insideVrfConfigOld.staticNat:
         del insideVrfConfigOld.staticNat[ rule.srcAddressOld ]

   staticNatConfigVrf = Tac.Value( "Ip::Nat::NatVrfStaticConfig", srcAddrOld,
                                   srcAddrNew, target, outsideVrf )
   insideVrfConfig.staticNat.addMember( staticNatConfigVrf )

   # In the reverse mapping, I'm using the outsideVrf field to store the insideVrf
   staticNatConfigVrfRev = Tac.Value( "Ip::Nat::NatVrfStaticConfig", srcAddrOld,
                                      srcAddrNew, target, insideVrf )
   outsideVrfConfig.staticNatRev.addMember( staticNatConfigVrfRev )

def cmdNoStaticNatVrf( mode, args ):
   target = args[ 'TARGET' ]
   localIp = args[ 'LOCALIP' ]
   globalIp = args[ 'GLOBALIP' ]
   insideVrf = args[ 'INSIDE_VRF' ]

   insideVrfConfig = natConfig.vrfConfig.get( insideVrf )
   if not insideVrfConfig:
      mode.addWarning( "Static nat rule for address %s not found in vrf %s" %
                       ( localIp, insideVrf ) )
      return

   srcIpOld = localIp if target == 'source' else globalIp
   srcAddressOld = Tac.Value( "Arnet::IpAddr", stringValue=srcIpOld )
   if srcAddressOld in insideVrfConfig.staticNat:
      rule = insideVrfConfig.staticNat[ srcAddressOld ]
      del insideVrfConfig.staticNat[ srcAddressOld ]

      outsideVrfConfig = natConfig.vrfConfig.get( rule.outsideVrf )
      if outsideVrfConfig and rule.srcAddressNew in outsideVrfConfig.staticNatRev:
         del outsideVrfConfig.staticNatRev[ rule.srcAddressNew ]

      if len( insideVrfConfig.staticNat ) == 0 and \
         len( insideVrfConfig.staticNatRev ) == 0:
         del natConfig.vrfConfig[ insideVrf ]
      if len( outsideVrfConfig.staticNat ) == 0 and \
         len( outsideVrfConfig.staticNatRev ) == 0:
         del natConfig.vrfConfig[ rule.outsideVrf ]

   else:
      mode.addWarning( "Static nat rule for address %s not found in vrf %s" %
                       ( localIp, insideVrf ) )

#------------------------------------------------------------------------------------
# (config-if)# ip nat source|destination dynamic acl <name> overload
# (config-if)# ip nat source|destination dynamic acl <name> pool <name>
#       [address-only|full-cone|group <grp>] [service-list <name>] [priority <value>]
# Note: [address-only|full-cone] options for source configuration only
#------------------------------------------------------------------------------------

# Pool AND overload does not make sense as the kernel automatically
# overloads many-to-many connections.
def configDynamicNat( mode, args ):
   target = args.get( 'TARGET' ) or args.get( 'source' ) or args.get( 'destination' )
   aclName = args[ 'ACLNAME' ]
   poolName = args.get( 'POOLNAME', '' )
   overload = 'overload' in args
   fullCone = 'full-cone' in args
   addrOnly = 'address-only' in args
   group = args.get( 'GROUP', 0 )
   serviceListName = args.get( 'SERVICELISTNAME', '' )
   priority = args.get( 'PRIORITY', 0 )
   comment = args.get( 'COMMENT', '' )

   if aclName not in aclConfig.config[ 'ip' ].acl:
      if not mode.session_.startupConfig():
         # Do not generate the warning while processing startup-config
         mode.addWarning( "Acl %s not configured. Assigning anyway." % aclName )
   else:
      if group:
         acl = aclConfig.config[ 'ip' ].acl[ aclName ].currCfg
         for uid in acl.ruleBySequence.values():
            filterSrc = acl.ipRuleById[ uid ].filter.source
            filterDst = acl.ipRuleById[ uid ].filter.destination
            if target == 'source':
               if filterDst.address != '0.0.0.0' or filterDst.mask:
                  mode.addError( "Dynamic Twice NAT access list %s contains invalid "
                                 "rules with filters on destination address" )
                  return
            else:
               if filterSrc.address != '0.0.0.0' or filterSrc.mask:
                  mode.addError( "Dynamic Twice NAT access list %s contains invalid "
                                 "rules with filters on source address" )
                  return

   dynamicKey = Tac.Value( "Ip::Nat::DynamicKey", aclName, group )
   dynamicNat = Tac.Value( "Ip::Nat::NatDynamicConfig",
                           dynamicKey, target, poolName, overload, fullCone,
                           addrOnly, serviceListName, priority, comment )

   # Must create the intfConfig after verifying the rule is valid, otherwise
   # we will create an empty intfConfig
   intfName, profileMode = getIntfName( mode )
   intfConfig = getOrCreateNatIntfConfig( intfName, profileMode, mode )
   if not intfConfig:
      return
   if group:
      # If a group is specified, make sure that a full (i.e. source and destination)
      # static twice entry for the same group is not specified already
      if group in intfConfig.twiceNat:
         twiceNatConfig = intfConfig.twiceNat[ group ]

         if target == 'source':
            if twiceNatConfig.srcIpAndPortOld.ip != '0.0.0.0':
               mode.addError( "A static source entry for the group is " \
                              "already configured" )
               return
            # The check on natStatus and natSupported fixes a problem with system
            # restart where the configuration is rejected if the Nat agent is not
            # initialized when the configuration is processed
            if natStatus and natStatus.natSupported and \
               twiceNatConfig.dstIpAndPortOld.ip != '0.0.0.0' and \
               twiceNatConfig.dstIpAndPortOld.port == 0 and \
               not isTwiceNatAsymmetricPortConfigSupported():
               mode.addError( "Twice NAT with asymmetric port configuration " \
                              "is not allowed in the current profile" )
               return
         elif target == 'destination':
            if twiceNatConfig.dstIpAndPortOld.ip != '0.0.0.0':
               mode.addError( "A static destination entry for the group is " \
                              "already configured" )
               return
            if natStatus and natStatus.natSupported and \
               twiceNatConfig.srcIpAndPortOld.ip != '0.0.0.0' and \
               twiceNatConfig.srcIpAndPortOld.port == 0 and \
               not isTwiceNatAsymmetricPortConfigSupported():
               mode.addError( "Twice NAT with asymmetric port configuration " \
                              "is not allowed in the current profile" )
               return

      # Make also sure that we don't have already a dynamic entry associated to the
      # same group (with a different acl)
      for key in intfConfig.dynamicNat:
         if key.group == group:
            if intfConfig.dynamicNat[ key ].comment:
               natConfig.commentCount -= 1
            del intfConfig.dynamicNat[ key ]

      # All dynamic twice rules sharing the same access list must have the
      # same priority value
      for intf in natConfig.intfConfig.values():
         for key, value in intf.dynamicNat.items():
            if group and key.group and key.acl == aclName:
               if value.priority != priority:
                  mode.addError( "All dynamic twice NAT rules sharing the same " \
                                 "access list must be configured with the same " \
                                 "priority value" )
                  return

   if dynamicKey in intfConfig.dynamicNat:
      if intfConfig.dynamicNat[ dynamicKey ].comment:
         natConfig.commentCount -= 1
      del intfConfig.dynamicNat[ dynamicKey ]
   intfConfig.dynamicNat.addMember( dynamicNat )
   intfConfig.dynamicNatInitialized = True
   if comment:
      natConfig.commentCount += 1

def configNoDynamicNat( mode, args ):
   aclName = args[ 'ACLNAME' ]
   group = args.get( 'GROUP', 0 )
   dynamicKey = Tac.Value( "Ip::Nat::DynamicKey", aclName, group )
   intfName, profileMode = getIntfName( mode )
   intfConfig = getNatIntfConfig( intfName, profileMode )
   if intfConfig:
      if dynamicKey in intfConfig.dynamicNat and \
            intfConfig.dynamicNat[ dynamicKey ].comment:
         natConfig.commentCount -= 1
      del intfConfig.dynamicNat[ dynamicKey ]
      deleteNatIntfConfig( intfName )

#------------------------------------------------------------------------------------
# (config)# ip nat synchronization
#    description         Peer Description
#    peer-address        IP address of NAT sync peer
#    local-interface     Vlan interface for accepting connections from NAT sync peer
#    shutdown            Disable the NAT sync feature
#    port-range          Specify the port range
#    expiry-interval     Interval in seconds to keep peer's entries
#------------------------------------------------------------------------------------

class NatSyncConfigMode( NatSyncMode, BasicCli.ConfigModeBase ):
   name = 'NAT synchronization configuration'

   def __init__( self, parent, session ):
      NatSyncMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoIpNatSyncConfigMode( mode, args ):
   childMode = mode.childMode( NatSyncConfigMode )
   mode.session_.gotoChildMode( childMode )

def removeIpNatSyncConfigMode( mode, args ):
   removeNatPeerExpiryInterval( mode, args )
   removeNatPeerDesc( mode, args )
   removeNatPeerSysname( mode, args )
   removeNatPeerAddress( mode, args )
   removeNatPeerLocalIntf( mode, args )
   removeNatPeerShutdown( mode, args )
   removeNatPortRange( mode, args )
   configNatPortRangeSplit( mode, args )

def configNatPeerExpiryInterval( mode, args ):
   natSyncConfig.peerExpiryPeriod = args[ 'INTERVAL' ]

def removeNatPeerExpiryInterval( mode, args ):
   natSyncConfig.peerExpiryPeriod = \
         natSyncConfig.peerExpiryPeriodDefault

def configNatPeerDesc( mode, args ):
   natSyncConfig.description = args[ 'DESCRIPTION' ]

def removeNatPeerDesc( mode, args ):
   natSyncConfig.description = ""

def configNatPeerSysname( mode, args ):
   natSyncConfig.sysname = args[ 'SYSNAME' ]

def removeNatPeerSysname( mode, args ):
   natSyncConfig.sysname = natSyncConfig.defaultSysname

def configNatPeerAddress( mode, args ):
   natSyncConfig.remote = args[ 'PEERIP' ]

def removeNatPeerAddress( mode, args ):
   natSyncConfig.remote = '0.0.0.0'

def configNatPeerLocalIntf( mode, args ):
   natSyncConfig.localIntfId = args[ 'INTF' ].name

def removeNatPeerLocalIntf( mode, args ):
   natSyncConfig.localIntfId = ""

def configNatPeerShutdown( mode, args ):
   natSyncConfig.shutdown = True

def removeNatPeerShutdown( mode, args ):
   natSyncConfig.shutdown = False

def checkPoolPortRangeAndNatSyncPortRange( mode, poolName, poolStartPort,
                                           poolEndPort, syncStartPort, syncEndPort ):
   if poolStartPort != 0 and poolEndPort != 0 and \
      syncStartPort != 0 and syncEndPort != 0:
      if poolStartPort < syncStartPort or \
           poolEndPort > syncEndPort:
         mode.addError( "Port range for pool %s (%d, %d) does not lie " \
                         "within nat peer sync port range (%d, %d)" % \
                         ( poolName, poolStartPort, poolEndPort,
                           syncStartPort, syncEndPort ) )
         return False

   return True


def configNatPortRange( mode, args ):
   lowerPortRange = args[ 'LOWER' ]
   upperPortRange = args.get( 'UPPER' )
   if upperPortRange is not None:
      if upperPortRange < lowerPortRange:
         mode.addError( 'Upper port range must be greater '
               'or equal to lower port range' )
         return
   else:
      upperPortRange = lowerPortRange

   # make sure that port range specified is ok
   # for the port range configured for nat pools
   for pool in natConfig.poolConfig:
      for pr in natConfig.poolConfig[ pool ].poolRange:
         if not checkPoolPortRangeAndNatSyncPortRange( mode, pool, pr.startPort,
                                                       pr.endPort, lowerPortRange,
                                                       upperPortRange ):
            return

   natSyncConfig.portRange = Tac.newInstance( 'Ip::Nat::Sync::PortRange',
         lowerPortRange, upperPortRange )

def removeNatPortRange( mode, args ):
   natSyncConfig.portRange = Tac.newInstance( 'Ip::Nat::Sync::PortRange', 0, 0 )

def configNatPortRangeSplit( mode, args ):
   natSyncConfig.portRangeSplitEnabled = 'disabled' not in args

#------------------------------------------------------------------------------------
# show ip nat synchronization
#------------------------------------------------------------------------------------
def toUtc( timestamp ):
   utcTime = timestamp + Tac.utcNow() - Tac.now() if timestamp else timestamp
   return utcTime

def cmdShowIpNatSynchronization( mode, args ):
   
   syncInfo = NatSynchronizationModel()
   status = natSyncStatus
   if not status:
      return syncInfo

   natVersionPeer = status.natVersionPeer
   syncInfo.peerIpAddr = status.remote
   syncInfo.connPort = status.connPort
   syncInfo.connSrcIp = status.connSrc
   syncInfo.connState = status.state
   syncInfo.kernelDevName = status.kernelDevName
   syncInfo.localIntf = status.localIntfId
   syncInfo.connEstablishedTime = toUtc( status.fsmEstablishedTime
      ) if status.state == STATUS_STATE.connected else 0.
   syncInfo.connAttempts = status.retry
   syncInfo.shutdown = status.shutdown
   syncInfo.statusMountState = status.statusMountState
   syncInfo.natVerCompatible = status.verCompatible
   syncInfo.portRangeMode = status.portRangeMode
   syncInfo.versionMountState = status.versionMountState
   syncInfo.natVersion = Version()
   if syncInfo.versionMountState == 'mountMounted' and natVersionPeer:
      syncInfo.natVersion.minVersion = natVersionPeer.minVersion
      syncInfo.natVersion.maxVersion = natVersionPeer.maxVersion
   else:
      syncInfo.natVersion.minVersion = 0
      syncInfo.natVersion.maxVersion = 0

   for intfName, intf in natConfig.intfConfig.items():
      if intf.profile and intf.fwdMissTranslationInfo:
         peerInfo = NatSyncFwdToPeer()
         peerInfo.peerIpAddr = intf.fwdMissTranslationInfo.fwdIp
         peerInfo.peerIntf = intf.fwdMissTranslationInfo.fwdIntfId
         peerInfo.peerResolved = False
         for intfStatus in natHwStatus.intfStatus.values():
            if intfStatus.name == intfName:
               peerInfo.peerResolved = intfStatus.fwdMissTranslationPeerResolved
               break
         syncInfo.fwdToPeer[ intfName ] = peerInfo

   return syncInfo

#------------------------------------------------------------------------------------
# show ip nat synchronization peer
#------------------------------------------------------------------------------------
def cmdShowIpNatSynchronizationPeer( mode, args ):
   syncPeer = NatSynchronizationPeer()
   status = natSyncStatus

   if not status:
      return syncPeer

   natVersionPeer = status.natVersionPeer

   syncPeer.connPort = status.connPort
   syncPeer.connSrcIp = status.connSrc
   syncPeer.connEstablishedTime = toUtc( status.fsmEstablishedTime
      ) if status.state == STATUS_STATE.connected else 0.
   syncPeer.kernelDevName = status.kernelDevName
   syncPeer.localIntf = status.localIntfId
   syncPeer.natVersion = Version()
   syncPeer.peerIpAddr = status.remote
   syncPeer.connAttempts = status.retry
   syncPeer.shutdown = status.shutdown
   syncPeer.connState = status.state
   syncPeer.statusMountState = status.statusMountState
   syncPeer.natVerCompatible = status.verCompatible
   syncPeer.versionMountState = status.versionMountState
   syncPeer.portRangeMode = status.portRangeMode

   if syncPeer.versionMountState == 'mountMounted' and natVersionPeer:
      syncPeer.natVersion.minVersion = natVersionPeer.minVersion
      syncPeer.natVersion.maxVersion = natVersionPeer.maxVersion
   else:
      syncPeer.natVersion.minVersion = 0
      syncPeer.natVersion.maxVersion = 0

   return syncPeer

#------------------------------------------------------------------------------------
# show ip nat synchronization advertised-translations
#------------------------------------------------------------------------------------
def instantiateAdvertisedStatusReader():
   global advertisedStatus
   global advertisedStatusSmashReader

   LazyMount.force( natStatus )

   advertisedStatus = Tac.newInstance( 'Ip::Nat::Sync::AdvertisedStatus',
         'advertisedStatus' )
   advertisedStatusSmashReader = Tac.newInstance (
         'Ip::Nat::Sync::Smash::NatAdvertiseReaderSm',
         natStatus, smashedAdvertisedStatus, advertisedStatus )

def waitForAdvertisedStatusReader( mode ):
   if advertisedStatusSmashReader is None:
      instantiateAdvertisedStatusReader()
   try:
      Tac.waitFor( lambda: advertisedStatusSmashReader.initialized,
         timeout=60, warnAfter=None,
         sleep=not Tac.activityManager.inExecTime.isZero, maxDelay=0.1,
         description='advertised status to be read' )
   except Tac.Timeout:
      mode.addError( "command timed out" )
      return False
   return True

def createAdverTranslation( srcIp, dstIp, newIp, target,
      natType, intf, proto, pkt, pktReply, tcpState, lastActivityTime, detail,
      vrfName, sourceVrf=None ):
   TCPSTATES = ( "NONE", "SYN_SENT", "SYN_RECV", "ESTABLISHED", \
      "FIN_WAIT", "CLOSE_WAIT", "LAST_ACK", "TIME_WAIT", "CLOSE", "LISTEN", \
      "MAX", "IGNORE", "RETRANS", "UNACK", "TIMEOUT_MAX" )
   transDetails = None
   if detail:
      if proto and proto in ipProtoMap:
         protocol = ipProtoMap[ proto ]
      else:
         protocol = None
      transDetails = \
            NatSynchronizationAdvertisedTranslations.Translation.Details(
               protocolNumber=proto,
               protocol=protocol,
               packetCount=pkt,
               packetReplyCount=pktReply,
               lastActivityTime=\
               toUtc( lastActivityTime ),
               tcpState=TCPSTATES[ tcpState ],
               vrfName=vrfName,
               sourceVrfName=sourceVrf )

   translation = NatSynchronizationAdvertisedTranslations.Translation(
      interface=intf,
      sourceAddress=srcIp,
      destinationAddress=dstIp,
      natType=natType,
      target=target,
      details=transDetails,
      globalAddress=newIp )

   yield translation

def doShowAdverTranslation( connTuple, connEntry, intfName,
                            detail, ipAddr, port, target, vrfName, sourceVrf=None ):
   if ipAddr and ipAddr != connTuple.srcAddr.ip:
      return

   if port and port != connTuple.srcAddr.port:
      return

   if target and target != connEntry.target():
      return

   srcAddress = IpAddrAndPort( ip=connTuple.srcAddr.ip,
                               port=connTuple.srcAddr.port )
   dstAddress = IpAddrAndPort( ip=connTuple.dstAddr.ip,
                               port=connTuple.dstAddr.port )
   globalAddress = IpAddrAndPort( ip=connEntry.globalAddress.ip,
                                  port=connEntry.globalAddress.port )

   if connEntry.fullCone():
      natType = 'FC'
   elif connEntry.addrOnly():
      natType = 'AO'
   elif connEntry.twiceNat():
      natType = 'D-TW'
   else:
      natType = 'DYN'

   tgt = 'SRC' if connEntry.target() == 'source' else 'DST'
   proto = connTuple.proto
   pktCount = 0
   pktCountReply = 0
   tcpState = 3 if connEntry.established() else 0
   lastActivityTime = Tac.now()
   assert type( tcpState ) == int # pylint: disable=unidiomatic-typecheck
   yield from createAdverTranslation( srcAddress,
                                      dstAddress, globalAddress, tgt,
                                      natType, intfName, proto, pktCount,
                                      pktCountReply, tcpState,
                                      lastActivityTime, detail, vrfName,
                                      sourceVrf=sourceVrf )

def _cmdShowIpNatAdverTrans( mode, args, detail, status ):
   intfs = [ str( intf ) for intf in args.get( 'INTF', [] ) ]
   target = None
   ipAddr = None
   port = None

   for natIntf, intfStatus in sorted( status.intfStatus.items(),
         key=lambda k: naturalOrderKey( k[ 1 ].name ) ):
      intfName = intfStatus.name
      if intfs and intfName not in intfs:
         continue

      for connEntry in intfStatus.natAdvertisedConnection.values():
         sourceVrf = getSourceVrf( natIntf, intfStatus.vrfName )
         yield from doShowAdverTranslation( connEntry.connTuple, connEntry,
                                            intfName,
                                            detail, ipAddr, port, target,
                                            intfStatus.vrfName,
                                            sourceVrf=sourceVrf )

def cmdShowIpNatSynchronizationAdvertisedTranslations( mode, args ):
   detail = 'detail' in args
   waitForAdvertisedStatusReader( mode )
   translations = _cmdShowIpNatAdverTrans( mode, args, detail, advertisedStatus )
   return NatSynchronizationAdvertisedTranslations( translations=translations,
                                                    detailedView=detail )

#------------------------------------------------------------------------------------
# show ip nat synchronization discovered-translations
#------------------------------------------------------------------------------------

def _cmdShowIpNatSynchronizationDiscoveredTranslations( detail, intfs ):
   for key, conn in discoveredStatus.peerConnection.items():
      if key.reset:
         continue
      natIntf = getNatIntfId( key.dynConnKey.natIntf )
      intfStatus = natStatus.intfStatus.get( natIntf )
      if intfStatus is None:
         continue
      intfName = intfStatus.name
      if intfs and intfName not in intfs:
         continue
      sourceVrf = getSourceVrf( key.dynConnKey.natIntf, intfStatus.vrfName )
      yield from doShowAdverTranslation( key.dynConnKey.connTuple, conn, intfName,
                                         detail,
                                         None, None, None, intfStatus.vrfName,
                                         sourceVrf=sourceVrf )

def cmdShowIpNatSynchronizationDiscoveredTranslations( mode, args ):
   detail = 'detail' in args
   intfs = [ str( intf ) for intf in args.get( 'INTF', [] ) ]
   translations = _cmdShowIpNatSynchronizationDiscoveredTranslations( detail, intfs )
   return NatSynchronizationAdvertisedTranslations( translations=translations,
                                                    detailedView=detail )

#------------------------------------------------------------------------------------
# show ip nat acl
#------------------------------------------------------------------------------------
def populateNatAclModel( aclName ):
   aclModel = NatAclList.NatAcl( aclName=aclName,
                                 aclRules=[],
                                 interfaces=[],
                                 profiles=[] )
   isStaticNat = False
   aclModel.aclInfo = {}
   for intfName, intfConfig in sorted( natConfig.intfConfig.items() ):
      isProfile = intfConfig.profile
      found = False
      for dynKey in intfConfig.dynamicNat:
         if aclName != dynKey.acl:
            continue
         found = True
         if isProfile:
            aclModel.profiles.append( intfName )
         else:
            aclModel.interfaces.append( intfName )

         intfId = natStatus.intfStatusKey.get( intfName )
         if not intfId:
            continue
         intfStatus = natStatus.intfStatus.get( intfId )
         if intfStatus:
            trapRules = intfStatus.dynamicTrapRule.get( dynKey )
            if trapRules:
               trapInfo = trapRules.trapInfo
               aclInfo = NatAclList.NatAcl.NatAclInfo( fullCone=trapInfo.fullCone,
                                                       addrOnly=trapInfo.addrOnly,
                                                       poolId=trapInfo.poolId )
               aclModel.aclInfo[ intfName ] = aclInfo
         break

      if found:
         continue

      for staticNat in intfConfig.staticNat.values():
         if aclName != staticNat.staticKey.acl:
            continue
         isStaticNat = True
         if isProfile:
            aclModel.profiles.append( intfName )
         else:
            aclModel.interfaces.append( intfName )
         break

   if not aclConfig.config[ 'ip' ].acl or \
      not ( aclModel.interfaces or aclModel.profiles ):
      return None

   acl = aclConfig.config[ 'ip' ].acl[ aclName ]
   for seq, uid in acl.currCfg.ruleBySequence.items():
      rule = acl.currCfg.ipRuleById[ uid ]
      try:
         description = checkDestinationRule( rule, isStaticNat=isStaticNat )
         error = False
      except ValueError as valueError:
         error = True
         description = str( valueError )
      ruleModel = NatAclList.NatAcl.AclRule( seqNo=seq,
                                             valid=not error,
                                             description=description )
      aclModel.aclRules.append( ruleModel )

   return aclModel

def cmdShowIpNatAcl( mode, args ):
   aclName = args.get( 'ACL_NAME' )
   interface = args.get( 'INTF' )
   if aclName and aclName not in aclConfig.config[ 'ip' ].acl:
      mode.addError( "ACL '%s' not configured" % aclName )
      return

   acls = NatAclList()
   if aclName:
      acls.aclList.append( populateNatAclModel( aclName ) )
   else:
      for acl in aclConfig.config[ 'ip' ].acl:
         model = populateNatAclModel( acl )
         if interface:
            if model and str( interface ) in model.interfaces:
               model.interfaces = []
            else:
               model = None

         if model:
            acls.aclList.append( model )

   if interface and not acls.aclList:
      mode.addError( "Interface %s: no Nat ACL used on this interface" %
                        str( interface ) )
      return

   return acls

#---------------------------------------------------------------------------------
# show ip nat translation
#---------------------------------------------------------------------------------

def doShowTranslation( src, dst, newSrc, newDst, target, natType, intf, vrf, proto,
                       pkt, pktReply, detail, twiceGroup=None, direction=None,
                       flags=None, sourceVrf=None, comment='', origin='Local' ):

   transDetails = None
   if twiceGroup:
      # Used only for twice entries, which have a separate cli model for
      # compatibility reasons
      if detail:
         if proto and proto in ipProtoMap:
            protocolStr = ipProtoMap[ proto ]
         else:
            protocolStr = None
         transDetails = TwiceNatIpTranslations.Translation.Details(
            interface=intf, vrfName=vrf, group=twiceGroup, protocolNumber=proto,
            protocol=protocolStr, packetCount=pkt, packetReplyCount=pktReply )
      translation = TwiceNatIpTranslations.Translation(
         srcIpAndPortOld=src, dstIpAndPortOld=dst, srcIpAndPortNew=newSrc,
         dstIpAndPortNew=newDst, details=transDetails, comment=comment )
   else:
      if detail:
         established = None
         if flags is not None:
            established = flags.established
         transDetails = NatIpTranslations.Translation.Details(
               packetCount=pkt, packetReplyCount=pktReply, vrfName=vrf,
               established=established, sourceVrfName=sourceVrf, origin=origin )
      translation = NatIpTranslations.Translation(
         interface=intf, srcAddress=src, dstAddress=dst, newSrcAddress=newSrc,
         newDstAddress=newDst, protocol=proto, target=target, natType=natType,
         details=transDetails, direction=direction, comment=comment )
   yield translation

def doShowStaticTranslation( staticKey, connEntry, intfName, vrf,
                             target=None, ipAddr=None, port=None, hardware=False,
                             connCount=None, detail=None ):
   localAddress = connEntry.staticKey.localAddress

   if target and target != connEntry.target:
      return

   if ipAddr and ipAddr != localAddress.ip:
      return

   if port and port != localAddress.port:
      return

   if connEntry.protocol and connEntry.protocol != IPPROTO_TCP and \
      connEntry.protocol != IPPROTO_UDP:
      return

   natType = 'E-ST' if staticKey.table == 'egress' else 'ST'
   direction = 'ING' if connEntry.ingress else 'EGR'

   if hardware:
      packetCount = connCount.pktCount
      packetReplyCount = connCount.pktCountReply
      localAddress = IpAddrAndPort( ip=connEntry.staticKey.localAddress.ip,
                                    port=connEntry.staticKey.localAddress.port )
   else:
      packetCount = 0
      packetReplyCount = 0
      localAddress = IpAddrAndPort( ip=connEntry.staticKey.localAddress.ip,
                                    port=connEntry.staticKey.localAddress.port )

   globalAddress = IpAddrAndPort( ip=connEntry.globalAddress.ip,
                                  port=connEntry.globalAddress.port )

   if connEntry.target == 'source':
      srcAddress = localAddress
      dstAddress = None
      newSrc = globalAddress
      newDst = None
      tgt = 'SRC'
   else:
      srcAddress = None
      dstAddress = localAddress
      newSrc = None
      newDst = globalAddress
      tgt = 'DST'

   proto = connEntry.protocol

   try:
      comment = natConfig.intfConfig[ intfName ].staticNat[ staticKey ].comment
   except ( KeyError, AttributeError ):
      comment = ''

   if len( connEntry.filterAddress ):
      filterAddresses = [ a.filterAddr for a in connEntry.filterAddress ]
      for filterAddress in filterAddresses:
         if connEntry.target == 'source':
            dstAddress = IpAddrAndMask( ip=filterAddress.address,
                                        mask=filterAddress.mask )
         else:
            srcAddress = IpAddrAndMask( ip=filterAddress.address,
                                        mask=filterAddress.mask )
         yield from doShowTranslation( srcAddress, dstAddress, newSrc,
                                       newDst, tgt, natType, intfName, vrf,
                                       proto, packetCount, packetReplyCount,
                                       detail, direction=direction,
                                       comment=comment )
   else:
      yield from doShowTranslation( srcAddress, dstAddress, newSrc, newDst,
                                    tgt, natType, intfName, vrf, proto,
                                    packetCount, packetReplyCount, detail,
                                    direction=direction, comment=comment )

def doShowTwiceTranslation( group, connEntry, intfName, vrf, intfStatus=None,
                            ipAddr=None, port=None, hardware=False, connCount=None,
                            hwKey=None, detail=None, oldTwiceModel=False,
                            natType=None, flags=None, connMark=None, 
                            origin='Local' ):
   if hardware:
      src = hwKey.srcAddr
      dst = hwKey.dstAddr
      try:
         curEntry = connEntry.entry[ hwKey ]
      except ( KeyError, AttributeError ):
         # If connEntry is modified during iteration, hwKey may not exist anymore
         return
      srcNew = curEntry.src
      dstNew = curEntry.dst
      if src.port and not srcNew.port:
         src = IpAddrAndPort( ip=src.ip, port=0 )
      elif srcNew.port and not src.port:
         srcNew = IpAddrAndPort( ip=srcNew.ip, port=0 )
      if dst.port and not dstNew.port:
         dst = IpAddrAndPort( ip=dst.ip, port=0 )
      elif dstNew.port and not dst.port:
         dstNew = IpAddrAndPort( ip=dstNew.ip, port=0 )
      natType = "D-TW" if connEntry.dynamic else "TW"
   else:
      src = connEntry.srcIpAndPortOld
      dst = connEntry.dstIpAndPortOld
      srcNew = connEntry.srcIpAndPortNew
      dstNew = connEntry.dstIpAndPortNew

   if src.ip == '0.0.0.0' or dst.ip == '0.0.0.0':
      return

   if connEntry.protocol and connEntry.protocol != IPPROTO_TCP and \
      connEntry.protocol != IPPROTO_UDP:
      return

   if ipAddr and ipAddr != src.ip and ipAddr != dst.ip:
      return

   if port and port != src.port and port != dst.port:
      return

   srcAddr = IpAddrAndPort( ip=src.ip, port=src.port )
   dstAddr = IpAddrAndPort( ip=dst.ip, port=dst.port )
   srcAddrNew = IpAddrAndPort( ip=srcNew.ip, port=srcNew.port )
   dstAddrNew = IpAddrAndPort( ip=dstNew.ip, port=dstNew.port )

   proto = connEntry.protocol

   # retrieve the comments on the rules that generated this connEntry
   try:
      comment = natConfig.intfConfig[ intfName ].twiceNat[ group ].comment
   except ( KeyError, AttributeError ):
      comment = ''
   if natType == "D-TW" and connMark:
      try:
         connMarkRuleKey = natStatus.connMarkRule[ connMark ].cmrKey
         dynamicKey = Tac.Value( "Ip::Nat::DynamicKey",
                                 connMarkRuleKey.aclName,
                                 connMarkRuleKey.group )
         dynamicRule = natConfig.intfConfig[ intfName ].dynamicNat.get( dynamicKey )
         # if dynamic twice and both rules have comments, concatenate the comments
         if comment and dynamicRule.comment:
            comment += '; ' + dynamicRule.comment
         # if dynamic twice and only the dynamic rule has comment, use just this one
         elif dynamicRule.comment:
            comment = dynamicRule.comment
      except ( KeyError, AttributeError, TypeError ):
         pass

   if isNatIngressTwiceConfigSupported():
      direction = 'ING' if connEntry.ingress else 'EGR'
   else:
      direction = None

   if hardware:
      packetCount = connCount.pktCount
      packetReplyCount = connCount.pktCountReply
   else:
      packetCount = 0
      packetReplyCount = 0

   twiceGroup = 0 if not oldTwiceModel else group
   yield from doShowTranslation( srcAddr, dstAddr, srcAddrNew, dstAddrNew,
                                 None, natType, intfName, vrf, proto,
                                 packetCount, packetReplyCount, detail,
                                 twiceGroup=twiceGroup,
                                 direction=direction, flags=flags,
                                 origin=origin, comment=comment )

def doShowDynamicTranslation( connEntry, intfName, vrf,
                              target=None, ipAddr=None, port=None, hardware=False,
                              connCount=None, detail=None, flags=None,
                              sourceVrf=None, origin='Local' ):

   connTarget = connEntry.target()
   if target and target != connTarget:
      return

   connTuple = connEntry.key.connTuple
   if ipAddr and ipAddr != connTuple.srcAddr.ip and ipAddr != connTuple.dstAddr.ip:
      return

   if port and port != connTuple.srcAddr.port and port != connTuple.dstAddr.port:
      return

   srcAddress = IpAddrAndPort( ip=connTuple.srcAddr.ip,
                               port=connTuple.srcAddr.port )
   dstAddress = IpAddrAndPort( ip=connTuple.dstAddr.ip,
                               port=connTuple.dstAddr.port )

   if hardware:
      pktCount = connCount.pktCount
      pktCountReply = connCount.pktCountReply
   else:
      pktCount = 0
      pktCountReply = 0

   if connTarget == 'source':
      tgt = 'SRC'
      newSrc = IpAddrAndPort( ip=connEntry.globalAddress.ip,
                              port=connEntry.globalAddress.port )
      newDst = None
   else:
      tgt = 'DST'
      newSrc = None
      newDst = IpAddrAndPort( ip=connEntry.globalAddress.ip,
                              port=connEntry.globalAddress.port )

   if connEntry.fullCone():
      natType = 'FC'
   elif connEntry.addrOnly():
      natType = 'AO'
   elif connEntry.twiceNat():
      natType = 'D-TW'
   else:
      natType = 'DYN'
   proto = connTuple.proto

   # retrieve the comment on the rule that generated this connEntry
   comment = ''
   if intfName and not hardware:
      connMarkRule = natStatus.connMarkRule.get( connEntry.connMark, None )
      if connMarkRule:
         connMarkRuleKey = connMarkRule.cmrKey
         dynamicKey = Tac.Value( "Ip::Nat::DynamicKey", connMarkRuleKey.aclName,
                                 connMarkRuleKey.group )
         intfConfig = natConfig.intfConfig.get( intfName, None )
         if intfConfig:
            dynamicNat = intfConfig.dynamicNat.get( dynamicKey )
            if dynamicNat:
               comment = dynamicNat.comment

   yield from doShowTranslation( srcAddress, dstAddress, newSrc, newDst,
                                 tgt, natType, intfName, vrf, proto,
                                 pktCount, pktCountReply, detail,
                                 flags=flags, sourceVrf=sourceVrf,
                                 origin=origin, comment=comment )

def getSyncOrigin( connKey ):
   # check if the translation was made locally or discovered from peer
   origin = 'Local'
   peerConnKey = Tac.Value( "Ip::Nat::Sync::PeerConnectionKey",
                            False, connKey )
   if peerConnKey in discoveredStatus.peerConnection:
      origin = 'Peer'
   return origin

def doShowKernelTranslations( natIntf, natType, target, ipAddr,
                              port, vrfName, detail ):

   ns = DEFAULT_NS if vrfName == DEFAULT_VRF else \
        allVrfStatusLocal.vrf[ vrfName ].networkNamespace
   output = runMaybeInNetNs( ns, [ 'cat', '/proc/net/nf_conntrack' ], asRoot=True,
                             stdout=subprocess.PIPE, stderr=subprocess.STDOUT )
   lines = output.split( '\n' )
   connre = re.compile( r'(\S+)=(\S+)?' )
   for line in lines:
      if not line:
         continue
      try:
         flow = re.split( 'src=', line )
         flowHeader = flow[ 0 ].split()
         flowForward = 'src=' + flow[ 1 ]
         flowForwardKeys = dict( connre.findall( flowForward ) )
         flowReverse = 'src=' + flow[ 2 ]
         flowReverseKeys = dict( connre.findall( flowReverse ) )
      except ( IndexError, KeyError ):
         continue

      if flowHeader[ 0 ] != 'ipv4':
         continue

      proto = int( flowHeader[ 3 ] )

      src = flowForwardKeys.get( 'src', '0.0.0.0' )
      srcPort = flowForwardKeys.get( 'sport', 0 )
      dst = flowForwardKeys.get( 'dst', '0.0.0.0' )
      dstPort = flowForwardKeys.get( 'dport', 0 )
      newSrc = flowReverseKeys.get( 'src', '0.0.0.0' )
      newSrcPort = flowReverseKeys.get( 'sport', 0 )
      newDst = flowReverseKeys.get( 'dst', '0.0.0.0' )
      newDstPort = flowReverseKeys.get( 'dport', 0 )

      srcAddress = Tac.Value( "Arnet::IpAndPort", src, int( srcPort ) )
      dstAddress = Tac.Value( "Arnet::IpAndPort", dst, int( dstPort ) )
      fwdTuple = Tac.Value( "Arnet::ConnTuple", srcAddress, dstAddress, proto )

      srcAddress = Tac.Value( "Arnet::IpAndPort", newSrc, int( newSrcPort ) )
      dstAddress = Tac.Value( "Arnet::IpAndPort", newDst, int( newDstPort ) )
      revTuple = Tac.Value( "Arnet::ConnTuple", srcAddress, dstAddress, proto )

      mark = flowReverseKeys.get( 'mark', 0 )

      natIntfConnMark = natStatus.connMarkIntfStatus.get( int( mark ), None )
      if natIntfConnMark:
         intfName = natStatus.intfConfigKey.get( natIntfConnMark, None )
      else:
         intfName = None

      if natIntf and intfName and intfName != natIntf:
         continue

      if fwdTuple.srcAddr.ip == revTuple.dstAddr.ip and \
         fwdTuple.dstAddr.ip == revTuple.srcAddr.ip:
         continue
      elif fwdTuple.dstAddr.ip == revTuple.srcAddr.ip:
         globalAddress = revTuple.dstAddr
         tgt = 'source'
      elif fwdTuple.srcAddr.ip == revTuple.dstAddr.ip:
         globalAddress = revTuple.srcAddr
         tgt = 'destination'
      elif mark:
         # Both src and dst address change, this must be a dynamic twice entry
         group = 0
         twiceConn = Tac.Value(
            "Ip::Nat::NatTwiceConnection", group,
            fwdTuple.srcAddr, fwdTuple.dstAddr,
            revTuple.dstAddr, revTuple.srcAddr,
            proto, 0, False )
         yield from doShowTwiceTranslation(
               group, twiceConn, intfName, vrfName,
               ipAddr=ipAddr, port=port, detail=detail,
               natType="D-TW", connMark=mark )
         continue
      else:
         continue

      connKey = Tac.Value( "Ip::Nat::DynamicConnectionKey",
                           Tac.Value( "Arnet::IntfId", "" ), fwdTuple )
      connEntry = Tac.Value( "Ip::Nat::DynamicConnection",
                             connKey, globalAddress, int( mark ) )
      connEntry.targetIs( tgt )

      pkts = flowForwardKeys.get( 'packets', 0 )
      replyPkts = flowReverseKeys.get( 'packets', 0 )
      connCount = Tac.Value( "Ip::Nat::NatCounter",
                             int( pkts ), int( replyPkts ) )

      # check if the translation was made locally or discovered from peer
      origin = getSyncOrigin( connKey )

      yield from doShowDynamicTranslation(
            connEntry, intfName, vrfName, target=target, ipAddr=ipAddr,
            port=port, connCount=connCount, detail=detail, origin=origin )

def doShowHardwareTranslations( natIntf, natType, target, ipAddr,
                                port, vrfName, detail, upnp ):

   connCountDefault = Tac.Value( "Ip::Nat::NatCounter", 0, 0 )

   for intfId, hwIntfStatus in sorted( natHwStatus.intfStatus.items(),
         key=lambda k: naturalOrderKey( k[ 1 ].name ) ):
      intfName = hwIntfStatus.name
      if natIntf and intfName != natIntf:
         continue

      intfStatus = natStatus.intfStatus.get( intfId )
      if intfStatus is None:
         continue

      if vrfName and vrfName != ALL_VRF_NAME and intfStatus.vrfName != vrfName:
         continue

      if not natType or natType == 'static':
         for staticKey, connEntry in hwIntfStatus.staticConnectionHw.items():
            connCount = hwIntfStatus.staticConnectionHwCount.get( staticKey,
                                                                connCountDefault )
            yield from doShowStaticTranslation(
                  staticKey, connEntry, intfName, intfStatus.vrfName,
                  target=target, ipAddr=ipAddr, port=port, hardware=True,
                  connCount=connCount, detail=detail )

      if not natType or natType in [ 'twice', 'dynamic-twice' ] :
         oldTwiceModel = natType == 'twice'
         for group, connEntry in hwIntfStatus.twiceNatConnectionHw.items():
            twiceNatHwCount = hwIntfStatus.twiceNatConnectionHwCount.get( group )
            if natType == 'twice' and connEntry.dynamic:
               continue
            if natType == 'dynamic-twice' and not connEntry.dynamic:
               continue
            for hwKey in connEntry.entry:
               connCount = connCountDefault
               if twiceNatHwCount:
                  connCount = twiceNatHwCount.count.get( hwKey, connCountDefault )
               yield from doShowTwiceTranslation(
                     group, connEntry, intfName, intfStatus.vrfName, intfStatus,
                     ipAddr=ipAddr, port=port, hardware=True,
                     connCount=connCount, hwKey=hwKey, detail=detail,
                     oldTwiceModel=oldTwiceModel )

      if not natType or natType in [ 'dynamic', 'full-cone', 'address-only' ] :
         for connKey, connEntry in \
             dynamicConnectionHwStatus.dynamicConnectionHw.items():
            if upnp and not connEntry.upnpIgd():
               continue
            if getNatIntfId( connKey.natIntf ) != intfStatus.natIntf:
               continue
            if natType == 'full-cone' and not connEntry.fullCone():
               return
            if natType == 'address-only' and not connEntry.addrOnly():
               return
            connCount = hwIntfStatus.dynamicConnectionHwCount.get(
               connKey.connTuple, connCountDefault )
            sourceVrf = getSourceVrf( connKey.natIntf, intfStatus.vrfName )
            
            origin = getSyncOrigin( connKey )

            yield from doShowDynamicTranslation(
                  connEntry, intfName, intfStatus.vrfName,
                  target=target, ipAddr=ipAddr, port=port, hardware=True,
                  connCount=connCount, detail=detail, sourceVrf=sourceVrf,
                  origin=origin )

def getDynamicTwiceGroup( connEntry ):
   if not connEntry.twiceNat():
      return None
   connMark = connEntry.connMark
   if not connMark:
      return None
   connMarkRule = natStatus.connMarkRule.get( connMark )
   if not connMarkRule:
      return None
   return connMarkRule.cmrKey.group

def doShowSoftwareTranslations( natIntf, natType, target, ipAddr,
                                port, vrfName, detail, pending, upnp ):
   # 'pending' applies only when there is async programming of nat rules to hardware.
   if pending and not natHwCapabilities.natAsyncHwProgrammingSupported:
      return
# pylint: disable-msg=too-many-nested-blocks, R1702
   for intfId, intfStatus in sorted( natStatus.intfStatus.items(),
         key=lambda k: naturalOrderKey( k[ 1 ].name ) ):
      hwIntfStatus = natHwStatus.intfStatus.get( intfId )

      intfName = intfStatus.name
      if natIntf and intfName != natIntf:
         continue

      if vrfName and vrfName != ALL_VRF_NAME and intfStatus.vrfName != vrfName:
         continue

      if not natType or natType == 'static':
         for staticKey, connEntry in intfStatus.staticConnection.items():
            if natHwCapabilities.natSoftwareSupported:
               if staticKey not in intfStatus.kernelStatus.staticConnStatus or \
                     intfStatus.kernelStatus.staticConnStatus[ staticKey ] != \
                     kernelProgramState.success:
                  continue
            if pending and hwIntfStatus and \
               staticKey in hwIntfStatus.staticConnectionHw:
               continue
            yield from doShowStaticTranslation(
                  staticKey, connEntry, intfName, intfStatus.vrfName,
                  target=target, ipAddr=ipAddr, port=port, detail=detail )

      if not natType or natType == 'twice':
         oldTwiceModel = natType == 'twice'
         for group, connEntry in intfStatus.twiceNatConnection.items():
            if natHwCapabilities.natSoftwareSupported:
               if group not in intfStatus.kernelStatus.twiceNatConnStatus or \
                     intfStatus.kernelStatus.twiceNatConnStatus[ group ] != \
                     kernelProgramState.success:
                  continue
            if connEntry.srcIpAndPortOld.ip == '0.0.0.0' or \
               connEntry.dstIpAndPortOld.ip == '0.0.0.0':
               continue
            if pending and hwIntfStatus and \
               group in hwIntfStatus.twiceNatConnectionHw:
               continue
            yield from doShowTwiceTranslation(
                  group, connEntry, intfName, intfStatus.vrfName, intfStatus,
                  ipAddr=ipAddr, port=port, detail=detail,
                  oldTwiceModel=oldTwiceModel, natType='TW' )

      if not natType or natType in [ 'dynamic', 'full-cone', 'address-only',
                                     'dynamic-twice' ] :
         for connKey, connEntry in \
             dynamicConnectionStatus.dynamicConnection.items():
            if getNatIntfId( connKey.natIntf ) != intfStatus.natIntf:
               continue
            if upnp and not connEntry.upnpIgd():
               continue

            origin = getSyncOrigin( connKey )

            if connEntry.twiceNat():
               if natType in [ 'full-cone', 'address-only' ]:
                  continue
               group = getDynamicTwiceGroup( connEntry )
               found = False
               if group and group in intfStatus.twiceNatConnection:
                  if pending and hwIntfStatus and \
                     group in hwIntfStatus.twiceNatConnectionHw:
                     continue
                  twiceConn = intfStatus.twiceNatConnection[ group ]
                  connTuple = connKey.connTuple
                  if connEntry.target() == 'source':
                     if connTuple.dstAddr.ip == twiceConn.dstIpAndPortOld.ip:
                        if not twiceConn.dstIpAndPortOld.port or \
                           connTuple.dstAddr.port == twiceConn.dstIpAndPortOld.port:
                           found = True
                           twiceConn = Tac.Value(
                              "Ip::Nat::NatTwiceConnection", group,
                              connTuple.srcAddr, twiceConn.dstIpAndPortOld,
                              connEntry.globalAddress, twiceConn.dstIpAndPortNew,
                              connTuple.proto, twiceConn.vlanId, False )
                  else:
                     if connTuple.srcAddr.ip == twiceConn.srcIpAndPortOld.ip:
                        if not twiceConn.srcIpAndPortOld.port or \
                           connTuple.srcAddr.port == twiceConn.srcIpAndPortOld.port:
                           found = True
                           twiceConn = Tac.Value(
                              "Ip::Nat::NatTwiceConnection", group,
                              twiceConn.srcIpAndPortOld, connTuple.dstAddr,
                              twiceConn.srcIpAndPortNew, connEntry.globalAddress,
                              connTuple.proto, twiceConn.vlanId, False )
                  if found:
                     yield from doShowTwiceTranslation(
                           group, twiceConn, intfName, intfStatus.vrfName,
                           intfStatus, ipAddr=ipAddr, port=port,
                           detail=detail, natType='D-TW', flags=connEntry.flags,
                           connMark=connEntry.connMark, origin=origin )
               if not found:
                  yield from doShowDynamicTranslation(
                        connEntry, intfName, intfStatus.vrfName,
                        target=target, ipAddr=ipAddr, port=port, detail=detail,
                        flags=connEntry.flags, origin=origin )
            else:
               if natType == 'dynamic-twice':
                  continue
               elif natType == 'full-cone' and not connEntry.fullCone():
                  continue
               elif natType == 'address-only' and not connEntry.addrOnly():
                  continue
               if pending and \
                  connKey in dynamicConnectionHwStatus.dynamicConnectionHw:
                  continue
               sourceVrf = getSourceVrf( connKey.natIntf, intfStatus.vrfName )
               yield from doShowDynamicTranslation(
                     connEntry, intfName, intfStatus.vrfName,
                     target=target, ipAddr=ipAddr, port=port, detail=detail,
                     flags=connEntry.flags, sourceVrf=sourceVrf, origin=origin )

def _cmdShowIpNatTranslations( mode, args ):
   natIntf = str( args.get( 'INTF' ) or args.get( 'PROFILE', '' ) )
   vrfName = args.get( 'VRF' )
   if not natIntf:
      vrfName = vrfMap.lookupCliModeVrf( mode, vrfName )

   natType = args.get( 'DYNAMIC_TYPE', '' ) + args.get( 'STATIC_TYPE', '' ) + \
             args.get( 'DYNAMIC_TWICE', '' )
   tableType = args.get( 'TABLE', '' )
   hardware = tableType == 'hardware'
   kernel = tableType == 'kernel'
   pending = tableType == 'pending'
   target = args.get( 'TARGET' )
   ipAddr = args.get( 'ADDRESS' )
   port = args.get( 'PORT' )
   detail = 'detail' in args
   upnp = 'upnp' in args

   if hardware:
      yield from doShowHardwareTranslations(
            natIntf, natType, target, ipAddr, port, vrfName, detail, upnp )
   elif kernel:
      # Kernel not used for static and twice NAT
      if natType in [ 'static', 'twice' ]:
         return

      if vrfName == ALL_VRF_NAME or not vrfName:
         for vrf in chain( allVrfStatusLocal.vrf, [ DEFAULT_VRF ] ):
            yield from doShowKernelTranslations(
                  natIntf, natType, target, ipAddr, port, vrf, detail )
      else:
         if vrfName == DEFAULT_VRF or vrfName in allVrfStatusLocal.vrf:
            yield from doShowKernelTranslations(
                  natIntf, natType, target, ipAddr, port, vrfName, detail )
   else:
      yield from doShowSoftwareTranslations(
            natIntf, natType, target, ipAddr, port, vrfName, detail, pending, upnp )

def cmdShowIpNatTranslations( mode, args ):
   detail = 'detail' in args
   translations = _cmdShowIpNatTranslations( mode, args )

   dynamicNatTypes = { 'FC', 'AO', 'D-TW', 'DYN' }

   for warning in platformWarnings.showCmdWarning.values():
      if warning.showWarning:
         if warning.requiresDynamicRules:
            translations, translationsCopy = tee( translations )

            foundDynamicRule = any( t.natType in dynamicNatTypes
                                    for t in translationsCopy )
            if not foundDynamicRule:
               continue

         mode.addWarning( warning.warningStr )

   # if any translations have comments, include a comments column
   hasComments = ( natConfig.commentCount > 0 )

   return NatIpTranslations( translations=translations, detailedView=detail,
                             hasComments=hasComments )

def cmdShowIpNatTranslationsTwice( mode, args ):
   # Tweak the args in order to use the same code as in cmdShowIpNatTranslations
   args[ 'STATIC_TYPE' ] = 'twice'

   detail = 'detail' in args
   tableType = args.get( 'TABLE', '' )
   pending = tableType == 'pending'
   translations = _cmdShowIpNatTranslations( mode, args )
   hasComments = ( natConfig.commentCount > 0 )
   return TwiceNatIpTranslations( translations=translations, detailedView=detail,
                                  pendingView=pending, hasComments=hasComments )

#--------------------------------------------------------------------------------
# show ip nat translation address-only mapping
#--------------------------------------------------------------------------------

def cmdShowIpNatTranslationsAddrOnly( mode, args ):
   if not AgentDirectory.agent( mode.sysname, 'Nat' ):
      mode.addError( "Nat Agent not running." )
      return
   source = args.get( 'ADDRESS' )
   intf = args.get( 'INTF' ) or args.get( 'PROFILE' )
   vrfName = args.get( 'VRF' )
   if not intf:
      vrfName = vrfMap.lookupCliModeVrf( mode, vrfName )

   outputFormat = mlib.stringToOutputFormat( mode.session_.outputFormat_ )
   printer = mlib.initPrinter( sys.stdout.fileno(), outputFormat )

   mlib.startRender( printer )
   opFormat = "%-21.21s %-21.21s %-21.21s %-3s %-4s %-20s %-10s %-5s\n"

   mlib.addFrills( printer, opFormat, 8,
             "Source IP", "Destination IP", "Translated IP", "TGT",
              "Type", "Interface/Profile", "VRF", "Proto" )
   line = "-" * 112
   mlib.addFrills( printer, "%s", 1, line )
   mlib.startDict( printer, "addrOnlyTranslations" )
   command = "showAddrOnlyTranslations"
   if source:
      command += " srcIp " + source
   if intf:
      command += " interface " + str( intf )
   else:
      assert vrfName
      command += " vrfName " + vrfName

   AgentCommandRequest.runCliPrintSocketCommand( mode.entityManager,
                                                 NatAgent.agentName(),
                                                 "NatCliWithFormat",
                                                 command, mode=mode )
   mlib.endDict( printer )
   mlib.endRender( printer )
   mlib.deinitPrinter( printer )
   sys.stdout.write( '\n' )
   return NatAddrOnlyTranslations

#--------------------------------------------------------------------------------
# show ip nat translation address-only | full-cone flows
#--------------------------------------------------------------------------------

def showIpNatTranslationFlows( connMapping, ipAddr, globalAddr, sourceVrf=None ):
   for connKey, conn in connMapping.conn.items():
      target = conn.target
      established = conn.established

      # Verify address
      if ipAddr and ipAddr not in [ connKey.srcAddr.ip, connKey.dstAddr.ip,
                                    globalAddr ]:
         continue

      flow = NatTranslationFlows.IntfTranslationFlows.TranslationFlows.\
             TranslationFlow()
      flow.target = "SRC" if target == 'source' else "DST"
      flow.source = IpAddrAndPort( ip=connKey.srcAddr.ip,
                                   port=connKey.srcAddr.port )
      flow.destination = IpAddrAndPort( ip=connKey.dstAddr.ip,
                                        port=connKey.dstAddr.port )
      flow.protocol = ipProtoMap.get( connKey.proto, "-" )
      flow.established = established
      if sourceVrf:
         flow.sourceVrfName = sourceVrf

      yield flow

def cmdShowIpNatTranslationsFlows( mode, args ):
   addrOnly = args.get( 'ADDR_ONLY' )
   fullCone = args.get( 'FULL_CONE' )
   assert fullCone or addrOnly

   natIntf = str( args.get( 'INTF' ) or args.get( 'PROFILE', '' ) )
   vrfName = args.get( 'VRF' )
   if not natIntf:
      vrfName = vrfMap.lookupCliModeVrf( mode, vrfName )

   ipAddr = args.get( 'ADDRESS' )

   dynamicConnectionMappingStatus = SharkLazyMount.mount(
         mode.entityManager,
         "nat/connMapStatus",
         "Ip::Nat::DynamicConnectionMappingStatus",
         Shark.mountInfo( 'shadow' ),
         True )

   translationFlows = NatTranslationFlows()

   for intf, intfMapping in dynamicConnectionMappingStatus.intfMapping.items():
      # Verify natIntf
      intfName = natStatus.intfConfigKey.get( getNatIntfId( intf ) )
      if not intfName:
         continue
      if natIntf and intfName != natIntf:
         continue

      # Verify intfStatus
      intfId = natStatus.intfStatusKey.get( intfName )
      if not intfId:
         continue
      intfStatus = natStatus.intfStatus.get( intfId )
      if not intfStatus:
         continue

      # Verify vrfName
      if vrfName and vrfName != ALL_VRF_NAME and intfStatus.vrfName != vrfName:
         continue

      intfFlows = translationFlows.intfTranslationFlows.get( intfName,
            NatTranslationFlows.IntfTranslationFlows() )
      intfFlows.vrfName = intfStatus.vrfName

      for connMapping in intfMapping.connMapping.values():
         # Verify map type
         if( ( connMapping.mapType == 'mappingAddrOnly' and fullCone ) or
             ( connMapping.mapType == 'mappingFullCone' and addrOnly ) ):
            continue

         globalAddr = connMapping.globalAddress
         trFlows = NatTranslationFlows.IntfTranslationFlows.TranslationFlows()
         trFlows.globalAddr = IpAddrAndPort( ip=globalAddr.ip,
                                             port=globalAddr.port )
         sourceVrf = getSourceVrf( intfMapping.intf, intfStatus.vrfName )
         trFlows.flows = showIpNatTranslationFlows( connMapping, ipAddr,
               globalAddr.ip, sourceVrf=sourceVrf )
         if trFlows.flows: # pylint: disable=using-constant-test
            intfFlows.translationFlows.append( trFlows )

      if intfFlows.translationFlows:
         translationFlows.intfTranslationFlows[ intfName ] = intfFlows

   return translationFlows

#--------------------------------------------------------------------------------
# show ip nat translation vrf (VRF hopping only)
#--------------------------------------------------------------------------------

def doShowNatVrfStaticTranslation( translList, insideVrf, entry, hardware, detail ):

   if entry.target == 'source':
      srcAddress = entry.srcAddressOld
      dstAddress = None
      translAddress = entry.srcAddressNew
      target = 'SRC'
   else:
      srcAddress = None
      dstAddress = entry.srcAddressNew
      translAddress = entry.srcAddressOld
      target = 'DST'
   outsideVrf = entry.outsideVrf

   translDetails = None
   if detail:
      packetCount = 0
      packetReplyCount = 0
      translDetails = VrfNatIpTranslations.Translation.Details(
         outsideVrf=outsideVrf,
         packetCount=packetCount,
         packetReplyCount=packetReplyCount )

   translation = VrfNatIpTranslations.Translation(
      sourceAddress=srcAddress, destinationAddress=dstAddress,
      globalAddress=translAddress,
      insideVrf=insideVrf,
      target=target, details=translDetails )
   translList.append( translation )

def cmdShowIpVrfNatTranslations( mode, args ):
   insideVrf = args.get( 'INSIDE_VRF' )
   outsideVrf = args.get( 'OUTSIDE_VRF' )
   target = args.get( 'TARGET' )
   ipAddr = args.get( 'ADDRESS' )
   tableType = args.get( 'TABLE' )
   hardware = tableType == 'hardware'
   pending = tableType == 'pending'
   detail = 'detail' in args

   natTranslationsList = []
   statusColl = natHwStatus if hardware else natStatus
   for vrf, status in statusColl.vrfStatus.items():
      if insideVrf and insideVrf != vrf:
         continue
      for srcAddressOld, entry in status.staticNat.items():
         if outsideVrf and outsideVrf != entry.outsideVrf:
            continue
         if target and target != entry.target:
            continue
         if ipAddr and ipAddr != entry.srcAddressOld and \
                       ipAddr != entry.srcAddressNew:
            continue
         if pending:
            hwStatus = natHwStatus.vrfStatus.get( vrf )
            if hwStatus:
               if srcAddressOld in hwStatus.staticNat:
                  continue
         doShowNatVrfStaticTranslation( natTranslationsList, vrf,
                                        entry, hardware, detail )
   return VrfNatIpTranslations( translations=natTranslationsList,
                                detailedView=detail,
                                pendingView=pending )

#------------------------------------------------------------------------------------
# clear ip nat flow translations
#------------------------------------------------------------------------------------
def cmdClearIpNatTranslations( mode, args ):
   intf = args[ 'INTF' ][ 0 ] if 'INTF' in args else None
   profile = args[ 'PROFILENAME' ][ 0 ] if 'PROFILENAME' in args else None
   vrfName = args[ 'VRF' ][ 0 ] if 'VRF' in args else None
   proto = args.get( 'PROTO' )
   srcIpAddr = args[ 'SRCIPADDR' ][ 0 ] if 'SRCIPADDR' in args else None
   srcPort = args[ 'SRCPORT' ][ 0 ] if 'SRCPORT' in args else None
   destIpAddr = args[ 'DESTIPADDR' ][ 0 ] if 'DESTIPADDR' in args else None
   destPort = args[ 'DESTPORT' ][ 0 ] if 'DESTPORT' in args else None

   if not AgentDirectory.agent( mode.sysname, 'Nat' ):
      mode.addError( "Nat Agent not running." )
      return

   command = "clearNatTranslations"
   if intf is not None:
      command = command + " Interface " + str( intf )
   elif profile is not None:
      command = command + " Profile " + str( profile )
   else:
      # agent will infer vrfName from Interface or Profile.
      # if none of interface, profile or vrfName is specified from CLI,
      # use the VRF current CLI mode is in
      vrfName = vrfMap.lookupCliModeVrf( mode, vrfName )
      command = command + " vrfName " + vrfName

   if proto is not None:
      command = command + " Protocol " + proto
   if srcIpAddr:
      command = command + " srcIpAddr " + srcIpAddr
   if destIpAddr:
      command = command + " destIpAddr " + destIpAddr
   if srcPort:
      command = command + " srcPort " + str( srcPort )
   if destPort:
      command = command + " destPort " + str( destPort )

   # In dpdk mode the handler for the command is in the Sfe agent
   if natHwCapabilities and natHwCapabilities.natDpdkProcessingSupported:
      dirName = "natDpdk"
      commandType = "NatDpdk"
   else:
      dirName = NatAgent.agentName()
      commandType = "Nat"

   AgentCommandRequest.runSocketCommand( mode.entityManager, dirName, commandType,
                                         command, timeout=60 )

def cmdShowIpNatDynamicCounters( mode, args ):
   profileArg = args.get( 'PROFILE' )
   countersProfiles = {}
   for intfName in natHwStatus.intfStatus:
      profileName = natHwStatus.intfStatus[ intfName ].name
      if profileArg and profileName != profileArg:
         continue
      counters = natHwStatus.intfStatus[ intfName ].dynamicSummaryCounters
      counterProfile = NatDynamicCountersProfileModel()
      counterProfile.tcp = NatDynamicCountersProtoModel()
      counterProfile.udp = NatDynamicCountersProtoModel()
      counterProfile.other = NatDynamicCountersProtoModel()
      
      keyType = "Ip::Nat::NatDynamicSummaryCounterKey"
      counterProfile.tcp.source = counters[ 
            Tac.Value( keyType, IPPROTO_TCP, True, False ) ].pkts
      counterProfile.tcp.destination = counters[ 
            Tac.Value( keyType, IPPROTO_TCP, False, True ) ].pkts
      counterProfile.tcp.hairpin = counters[ 
            Tac.Value( keyType, IPPROTO_TCP, True, True ) ].pkts
      counterProfile.udp.source = counters[ 
            Tac.Value( keyType, IPPROTO_UDP, True, False ) ].pkts
      counterProfile.udp.destination = counters[ 
            Tac.Value( keyType, IPPROTO_UDP, False, True ) ].pkts
      counterProfile.udp.hairpin = counters[ 
            Tac.Value( keyType, IPPROTO_UDP, True, True ) ].pkts
      counterProfile.other.source = counters[ 
            Tac.Value( keyType, 0, True, False ) ].pkts
      counterProfile.other.destination = counters[ 
            Tac.Value( keyType, 0, False, True ) ].pkts
      counterProfile.other.hairpin = counters[ 
            Tac.Value( keyType, 0, True, True ) ].pkts

      countersProfiles[ profileName ] = counterProfile

   output = NatDynamicCountersModel()
   output.profiles = countersProfiles
   return output
      
#-------------------------------------------------------------------------------
# Destroy method of NatIntf class is called by Intf class when an interface
# object is deleted. Intf class will create a new instance of NatIntf
# and call destroy method on it.
# ------------------------------------------------------------------------------
class NatIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      name = self.intf_.name
      intfId = Tac.Value( "Arnet::IntfId", name )
      profileName = natConfig.profileConfig.get( intfId )
      if profileName is not None:
         del natConfig.profileConfig[ intfId ]
         intfConfig = natConfig.intfConfig.get( profileName )
         assert intfConfig != None # pylint: disable=singleton-comparison
         del intfConfig.profileIntf[ intfId ]
         deleteNatIntfConfig( profileName )
      if not natHwCapabilities.natProfileConfigSupported:
         del natConfig.intfConfig[ name ]

#------------------------------------------------------------------------------------
# Show Tech Commands
#------------------------------------------------------------------------------------
def _natRunning():
   # Taken from runnability criteria in SysdbPlugin
   return natConfig.intfConfig or natStatus.intfStatus

# Timestamps are made up to maintain historical order within show tech-support
TechSupportCli.registerShowTechSupportCmd(
   '2016-12-19 09:35:00',
   cmds=[ "show ip nat translation vrf all detail",
          "show ip nat translation hardware vrf all detail",
          "show ip nat translation kernel vrf all detail",
          "show ip nat translation pending vrf all detail",
          "show ip nat translation summary",
          "show ip nat access-list",
          "show ip nat counters vrf all",
          "show ip nat pool synchronization",
          "bash sudo conntrack -L",
          "show ip nat translation max-entries",
          "show ip nat synchronization",
          "show ip nat synchronization advertised-translations",
          "show ip nat synchronization discovered-translations",
        ],
   cmdsGuard=lambda: _natSupported() and _natRunning() )

TechSupportCli.registerShowTechSupportCmd(
   '2023-09-20 15:21:45',
   cmds = [ "show configuration consistency nat" ],
   cmdsGuard=lambda: ( _natSupported() and
                       _natRunning() and
                       natHwCapabilities.natConfigConsistencySupported ) )

#----------------------------------------------------------------------------------
# "[no|default] ip nat kernel buffer size", in "config" mode
#----------------------------------------------------------------------------------
def cmdNatIpKernelBufferSize( mode, args ):
   bufSize = args.get( 'BUFSIZE', 0 )
   natConfig.settings.natNetlinkSocketBufSize = bufSize

#----------------------------------------------------------------------------------
# "[no|default]
# ip nat translation udp-timeout|tcp-timeout [unestablished]|icmp-timeout",
# in "config" mode
#----------------------------------------------------------------------------------
def cmdIpNatTranslationTimeout( mode, args ):
   timeout = args[ 'TIMEOUT' ]
   if 'tcp-timeout' in args:
      if 'unestablished' in args:
         natConfig.timeoutConfig.unestablishedTcpTimeout = timeout
      else:
         natConfig.timeoutConfig.tcpTimeout = timeout
   elif 'udp-timeout' in args:
      natConfig.timeoutConfig.udpTimeout = timeout
   else:
      natConfig.timeoutConfig.icmpTimeout = timeout

def cmdNoIpNatTranslationTimeout( mode, args ):
   if 'tcp-timeout' in args:
      if 'unestablished' in args:
         natConfig.timeoutConfig.unestablishedTcpTimeout = \
            DefaultValue.unestablishedTcpTimeoutDefault 
      else:
         natConfig.timeoutConfig.tcpTimeout = DefaultValue.tcpTimeoutDefault
   elif 'udp-timeout' in args:
      natConfig.timeoutConfig.udpTimeout = DefaultValue.udpTimeoutDefault
   else:
      natConfig.timeoutConfig.icmpTimeout = DefaultValue.icmpTimeoutDefault

#----------------------------------------------------------------------------------
# nat flow commands
#----------------------------------------------------------------------------------
class NatFlowConfigMode( NatFlowMode, BasicCli.ConfigModeBase ):
   '''Configuration mode for nat flow'''
   name = 'nat-flow'

   def __init__( self, parent, session ):
      NatFlowMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoIpNatFlowConfigMode( mode, args ):
   childMode = mode.childMode( NatFlowConfigMode )
   mode.session_.gotoChildMode( childMode )

class NatFlowPolicyConfigMode( NatFlowPolicyMode,
                               BasicCli.ConfigModeBase ):
   ''' Configuration mode for nat flow policy '''
   name = 'nat flow policy configuration'

   def __init__( self, parent, session, natFlowPolicyName ):
      NatFlowPolicyMode.__init__( self, ( type( parent ),
                                          natFlowPolicyName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.policyName = natFlowPolicyName
      self.policy = natConfig.natFlow.get( self.policyName )
      if not self.policy:
         self.policy = natConfig.newNatFlow( self.policyName )

def gotoIpNatFlowPolicyConfigMode( mode, args ):
   natFlowPolicyName = args[ 'FLOW_POLICY_NAME' ]
   childMode = mode.childMode( NatFlowPolicyConfigMode,
                               natFlowPolicyName=natFlowPolicyName )
   mode.session_.gotoChildMode( childMode )

def deleteIpNatFlowPolicyConfig( mode, args ):
   natFlowPolicyName = args.get( 'FLOW_POLICY_NAME' )
   if natFlowPolicyName:
      del natConfig.natFlow[ natFlowPolicyName ]
   else:
      natConfig.natFlow.clear()

class NatFlowMatchConfigMode( NatFlowMatchMode,
                              BasicCli.ConfigModeBase ):
   ''' Configuration mode for nat flow match '''
   name = 'nat flow match configuration'

   def __init__( self, parent, session, natFlowMatchName):
      self.policyName = parent.policyName
      self.policy = natConfig.natFlow.get( self.policyName )
      params = parent, self.policyName, natFlowMatchName
      NatFlowMatchMode.__init__( self, params )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

      self.matchName = natFlowMatchName
      self.match = self.policy.match.get( self.matchName )
      if not self.match:
         self.match = self.policy.newMatch( self.matchName )

def gotoIpNatFlowMatchConfigMode( mode, args ):
   natFlowMatchName = args[ 'MATCH_NAME' ]
   childMode = mode.childMode( NatFlowMatchConfigMode,
                               natFlowMatchName=natFlowMatchName )
   mode.session_.gotoChildMode( childMode )

def deleteIpNatFlowMatchConfig( mode, args ):
   matchName = args.get( 'MATCH_NAME' )
   if matchName:
      del mode.policy.match[ matchName ]
   else:
      mode.policy.match.clear()

def configNatFlowMatchSourcePrefix( mode, args ):
   try:
      prefix = Arnet.Prefix( args[ 'PREFIX' ] )
   except ValueError:
      mode.addError( 'invalid prefix' )
      return
   match = mode.match
   tmpFlow = Tac.nonConst( match.flow )
   tmpFlow.src = prefix
   match.flow = Tac.const( tmpFlow )

def deleteNatFlowMatchSourcePrefix( mode, args ):
   match = mode.match

   # warn user if the new match is incompatible with any of its actions
   nullIpAndPort = Tac.Value( 'Arnet::IpAndPort' )
   for action in match.action.values():
      if action.addr.src != nullIpAndPort:
         mode.addWarning( 'source address required in the policy match for source '
                          'address translation' )
         break # only need to warn once

   tmpFlow = Tac.nonConst( match.flow )
   tmpFlow.src = Arnet.Prefix( '0.0.0.0/0' )
   match.flow = Tac.const( tmpFlow )

def configNatFlowMatchDestinationPrefix( mode, args ):
   try:
      prefix = Arnet.Prefix( args[ 'PREFIX' ] )
   except ValueError:
      mode.addError( 'invalid prefix' )
      return
   match = mode.match

   tmpFlow = Tac.nonConst( match.flow )
   tmpFlow.dst = prefix
   match.flow = Tac.const( tmpFlow )

   # warn user if the new match is incompatible with any of its actions
   matchDip = Arnet.IpAddr( prefix.address )
   for action in match.action.values():
      if not natFlowCheckMac( mode, action, matchDip ):
         break # only need to warn once

def deleteNatFlowMatchDestinationPrefix( mode, args ):
   match = mode.match

   tmpFlow = Tac.nonConst( match.flow )
   tmpFlow.dst = Arnet.Prefix( '0.0.0.0/0' )
   match.flow = Tac.const( tmpFlow )

   # warn user if the new match is incompatible with any of its actions
   nullIpAndPort = Tac.Value( 'Arnet::IpAndPort' )
   for action in match.action.values():
      if action.addr.dst != nullIpAndPort:
         mode.addWarning( 'destination address required in the policy match for '
                          'destination address translation' )
         break # only need to warn once
   matchDip = Arnet.IpAddr( match.flow.dst.address )
   for action in match.action.values():
      if not natFlowCheckMac( mode, action, matchDip ):
         break # only need to warn once

def configNatFlowMatchPort( mode, args ):
   proto = args[ 'PROTOCOL' ].lower()
   # get the protocol number from the string representation via a reverse-lookup
   # of the number->name mapping in ipProtoMap
   protoNum = { name.lower(): num for num, name in ipProtoMap.items() }.get( proto )
   if not protoNum:
      mode.addError( 'invalid L4 protocol' )
   lowerSrcPort = args.get( 'SRC_PORT', 0 )
   upperSrcPort = args.get( 'UPPER_SRC_PORT', lowerSrcPort )
   lowerDstPort = args.get( 'DST_PORT', 0 )
   upperDstPort = args.get( 'UPPER_DST_PORT', lowerDstPort )
   if upperSrcPort > lowerSrcPort and upperDstPort > lowerDstPort:
      mode.addError( "cannot program port range for both source and destination" )
      return
   if upperSrcPort < lowerSrcPort or upperDstPort < lowerDstPort:
      mode.addError( "upper port must be greater than or equal to lower port" )
      return

   match = mode.match
   tmpFlow = Tac.nonConst( match.flow )
   tmpFlow.protocol = protoNum
   tmpFlow.srcPortStart = lowerSrcPort
   tmpFlow.srcPortEnd = upperSrcPort
   tmpFlow.dstPortStart = lowerDstPort
   tmpFlow.dstPortEnd = upperDstPort
   match.flow = Tac.const( tmpFlow )

   # check if new match is incompatible with any of its actions
   if lowerSrcPort == 0:
      for action in match.action.values():
         if action.addr.src.port != 0:
            mode.addWarning( 'source port required in the policy match for source '
                             'port translation' )
            break
   if lowerDstPort == 0:
      for action in match.action.values():
         if action.addr.dst.port != 0:
            mode.addWarning( 'destination port required in the policy match for'
                             ' destination port translation' )
            break

def deleteNatFlowMatchPort( mode, args ):
   match = mode.match
   tmpFlow = Tac.nonConst( match.flow )
   notSrc = 'source' not in args
   notDst = 'destination' not in args

   if notSrc:
      tmpFlow.dstPortStart = 0
      tmpFlow.dstPortEnd = 0
   if notDst:
      tmpFlow.srcPortStart = 0
      tmpFlow.srcPortEnd = 0
   if notSrc and notDst:
      tmpFlow.protocol = 0

   match.flow = Tac.const( tmpFlow )

def configNatFlowMatchPriority( mode, args ):
   # if a priority is specified, use it. else, assume it's a 'no' command
   priority = args.get( 'PRIORITY', 0 )
   match = mode.match
   tmpFlow = Tac.nonConst( match.flow )
   tmpFlow.priority = priority
   match.flow = Tac.const( tmpFlow )

class NatFlowActionConfigMode( NatFlowActionMode,
                               BasicCli.ConfigModeBase ):
   ''' Configuration mode for nat flow action '''
   name = 'nat flow action configuration'

   def __init__( self, parent,  session, natFlowActionName ):
      self.policyName = parent.policyName
      self.policy = parent.policy
      self.matchName = parent.matchName
      self.match = parent.match
      params = parent, self.policyName, self.matchName, natFlowActionName
      NatFlowActionMode.__init__( self, params )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

      self.actionName = natFlowActionName
      self.action = self.match.action.get( self.actionName )
      if not self.action:
         self.action = self.match.newAction( self.actionName )

def gotoIpNatFlowActionConfigMode( mode, args ):
   natFlowActionName = args[ 'ACTION_NAME' ]
   childMode = mode.childMode( NatFlowActionConfigMode,
                               natFlowActionName=natFlowActionName )
   mode.session_.gotoChildMode( childMode )

def deleteIpNatFlowActionConfig( mode, args ):
   natFlowActionName = args.get( 'ACTION_NAME' )
   if natFlowActionName:
      del mode.match.action[ natFlowActionName ]
   else:
      mode.match.action.clear()

def natFlowCheckMac( mode, action, matchDip ):
   ''' this helper function prints a warning to the user if they configure a match +
       action pair that are incompatible due to the presence or absence of a MAC
       address specified in the action's outputs.
       if the action NATs the dip to multicast, OR
          the action does not NAT the dip but the match matches a mcast dip,
             then no MAC is allowed in the action's outputs.
          in this case, if the action contains an output that specifies a MAC,
          warn the user of the issue
       if the action NATs the dip to unicast, OR
          the action does not NAT the dip but the match matches a unicast dip, OR
          if neither the action nor match specify a dip,
             then a MAC is required in the output.
          ( note that Arnet.IpAddr.isUnicast returns True for 0.0.0.0, so we have to
            specify "unicast and not zero" to check if it is unicast )
          in this case, if the action contains an output that does not specify a
          MAC, warn the user of the issue

       this function returns False if it prints a warning and True if no issues are
       found, which helps for functions that call this on a set of actions but only
       want to print one warning
   '''
   nullEthAddr = Arnet.EthAddr( '0.0.0' )
   actionDip = Arnet.IpAddr( action.addr.dst.ip )
   if actionDip.isMulticast or ( actionDip.isZero and matchDip.isMulticast ):
      for intf in action.intf.values():
         if Arnet.EthAddr( intf.mac ) != nullEthAddr:
            mode.addWarning(
                  'MAC address not allowed for outgoing multicast packets' )
            return False
   if ( actionDip.isUnicast and not actionDip.isZero ) or \
      ( actionDip.isZero and ( matchDip.isZero or matchDip.isUnicast ) ):
      for intf in action.intf.values():
         if Arnet.EthAddr( intf.mac ) == nullEthAddr:
            mode.addWarning( 'MAC address required for unicast or unspecified '
                             'outgoing packets' )
            return False
   return True

def configNatFlowActionNat( mode, args ):
   ipStr = args[ 'ADDRESS' ]
   ip = Arnet.IpAddr( ipStr )
   port = args.get( 'PORT', 0 )
   addrAndPort = Tac.Value( 'Arnet::IpAndPort', ip, port )

   action = mode.action
   flow = mode.match.flow
   if 'source' in args:
      if flow.src == Arnet.Prefix( '0.0.0.0/0' ):
         mode.addWarning( 'source address required in the policy match for source '
                          'address translation' )
      if port != 0 and flow.srcPortStart == 0:
         mode.addWarning( 'source port required in the policy match for source '
                          'port translation' )
      portRange = flow.srcPortEnd - flow.srcPortStart
      if port != 0 and port + portRange >= 2 **16:
         mode.addError(
            'invalid translated port range: upper bound must be lower than 65536' )
         return
      dst = action.addr.dst
      action.addr = Tac.Value( 'Ip::Nat::NatFlowAddress',
                               addrAndPort, dst )
   elif 'destination' in args:
      if flow.dst == Arnet.Prefix( '0.0.0.0/0' ):
         mode.addWarning( 'destination address required in the policy match for '
                          'destination address translation' )
      if port != 0 and flow.dstPortStart == 0:
         mode.addWarning( 'destination port required in the policy match for '
                          'destination port translation' )
      portRange = flow.dstPortEnd - flow.dstPortStart
      if port != 0 and port + portRange >= 2 **16:
         mode.addError(
            'invalid translated port range: upper bound must be lower than 65536' )
         return

      src = action.addr.src
      action.addr = Tac.Value( 'Ip::Nat::NatFlowAddress',
                               src, addrAndPort )

   action.version += 1

   # warn user if new NAT rule results in incompatible match+action pair
   matchDip = Arnet.IpAddr( flow.dst.address )
   natFlowCheckMac( mode, action, matchDip )

def deleteNatFlowActionNat( mode, args ):
   action = mode.action
   nullAddr = Tac.Value( 'Arnet::IpAndPort', '0.0.0.0', 0 )

   if 'source' in args:
      dst = action.addr.dst
      action.addr = Tac.Value( 'Ip::Nat::NatFlowAddress',
                               nullAddr, dst )
   elif 'destination' in args:
      src = action.addr.src
      action.addr = Tac.Value( 'Ip::Nat::NatFlowAddress',
                               src, nullAddr )
   else:
      action.addr = Tac.Value( 'Ip::Nat::NatFlowAddress',
                               nullAddr, nullAddr )

   action.version += 1

   # warn user if deleting NAT rule results in incompatible match+action pair
   matchDip = Arnet.IpAddr( mode.match.flow.dst.address )
   natFlowCheckMac( mode, action, matchDip )


def configNatFlowActionOutput( mode, args ):
   intfs = args[ 'INTFS' ]
   macStr = args.get( 'MAC', '0.0.0' )
   mac = Arnet.EthAddr( macStr )
   vlanId = args.get( 'VLANID', 0 )

   match = mode.match
   action = mode.action

   # if vlan is specified but the interface is an svi, reject the command
   # pylint: disable-next=protected-access
   if vlanId and intfs.type_.tagLong == 'Vlan':
      mode.addErrorAndStop( 'VLAN not allowed for SVI interfaces' )

   # write the values to sysdb
   for intf in intfs:
      action.intf[ intf ] = Tac.Value( 'Ip::Nat::NatFlowNextHop', mac, vlanId )
   action.version += 1

   # warn user if mac config is incompatible with match+action config
   matchDip = Arnet.IpAddr( match.flow.dst.address )
   natFlowCheckMac( mode, action, matchDip )

def deleteNatFlowActionOutput( mode, args ):
   if intfs := args.get( 'INTFS' ):
      for intf in intfs:
         del mode.action.intf[ intf ]
   else:
      mode.action.intf.clear()

   mode.action.version += 1

def configNatFlowPolicyIntf( mode, args ):
   intfName, profileMode = getIntfName( mode )
   intfConfig = getOrCreateNatIntfConfig( intfName, profileMode, mode )
   if not intfConfig:
      return
   policyName = args[ 'POLICYNAME' ]
   intfConfig.flowPolicy[ policyName ] = True

def clearNatFlowPolicyIntf( mode, args ):
   intfName, profileMode = getIntfName( mode )
   intfConfig = getNatIntfConfig( intfName, profileMode )
   if not intfConfig:
      return
   policyName = args[ 'POLICYNAME' ]
   if policyName in intfConfig.flowPolicy:
      del intfConfig.flowPolicy[ policyName ]
      deleteNatIntfConfig( intfName )

def configTranslationMissForwardIntf( mode, args ):
   intfName, profileMode = getIntfName( mode )
   intfConfig = getOrCreateNatIntfConfig( intfName, profileMode, mode )
   if not intfConfig:
      return
   fwdIntfName = args[ 'INTF' ]
   fwdIpAddress = args[ 'IP' ]
   fwdIntfId = Tac.Value( "Arnet::IntfId", fwdIntfName.name )
   fwdMissTranslationInfo = Tac.Value( "Ip::Nat::FwdMissTranslationInfo" )
   fwdMissTranslationInfo.fwdIntfId = fwdIntfId 
   fwdMissTranslationInfo.fwdIp = fwdIpAddress
   intfConfig.fwdMissTranslationInfo = fwdMissTranslationInfo

def clearTranslationMissForwardIntf( mode, args ):
   intfName, profileMode = getIntfName( mode )
   intfConfig = getNatIntfConfig( intfName, profileMode )
   if not intfConfig:
      return
   fwdMissTranslationInfo = Tac.Value( "Ip::Nat::FwdMissTranslationInfo" )
   # Reset to default values
   intfConfig.fwdMissTranslationInfo = fwdMissTranslationInfo
#------------------------------------------------------------------------------------
# "show ip nat flow match"
#------------------------------------------------------------------------------------

def getNatFlowCollections( config, hardware ):
   if config:
      return ( natConfig, natConfig.intfConfig )
   elif hardware:
      return( natHwFlowStatus, natHwStatus.intfStatus )
   else:
      return( natStatus, natStatus.intfStatus )

def showIpNatFlowPolicyMatch( model, policyName, config, hardware ):
   p = NatFlowMatchPolicy()
   ( flowColl, _ ) = getNatFlowCollections( config, hardware )
   policy = flowColl.natFlow.get( policyName )
   if not policy:
      return
   for matchName, match in policy.match.items():
      m = NatFlowMatch()

      f = NatFlowAddress()
      f.srcAddr = IpAddrAndMask( ip=match.flow.src.address,
                                 mask=match.flow.src.len )
      f.dstAddr = IpAddrAndMask( ip=match.flow.dst.address,
                                 mask=match.flow.dst.len )
      f.protocol = match.flow.protocol
      f.srcPortStart = match.flow.srcPortStart
      f.srcPortEnd = match.flow.srcPortEnd
      f.dstPortStart = match.flow.dstPortStart
      f.dstPortEnd = match.flow.dstPortEnd
      f.priority = match.flow.priority
      m.flow = f

      if hardware:
         m.ucGroup = match.groupIdUc
         m.mcGroup = match.groupIdMc

      p.matches[ matchName ] = m

   model.policies[ policy.name ] = p

def cmdShowIpNatFlowMatch( mode, args ) :
   model = NatFlowMatchCmdModel()
   policyName = args.get( "POLICY" )
   config = 'config' in args
   hardware = 'hardware' in args
   model.hardware = hardware

   if policyName:
      if policyName not in natStatus.natFlow:
         mode.addError( "Policy %s does not exist" % ( policyName ) )
         return
      showIpNatFlowPolicyMatch( model, policyName, config, hardware )
   else:
      for policy in natStatus.natFlow:
         showIpNatFlowPolicyMatch( model, policy, config, hardware )

   return model

#------------------------------------------------------------------------------------
# "show ip nat flow action"
#------------------------------------------------------------------------------------

def showIpNatFlowPolicyAction( model, policyName, config ):
   p = NatFlowActionPolicy()
   ( flowColl, _ ) = getNatFlowCollections( config, None )
   policy = flowColl.natFlow.get( policyName )
   if not policy:
      return
   for matchName, match in policy.match.items():
      m = NatFlowMatchActions()
      srcPrefixLen = match.flow.src.len
      dstPrefixLen = match.flow.dst.len
      for actionName, action in match.action.items():
         a = NatFlowAction()
         # mask the bits that aren't translated
         natSrc = Arnet.AddrWithMask( "{}/{}".format( action.addr.src.ip,
                                                      srcPrefixLen )
                                    ).subnet.address
         natDst = Arnet.AddrWithMask( "{}/{}".format( action.addr.dst.ip,
                                                      dstPrefixLen )
                                    ).subnet.address
         a.natSrc = IpAddrAndPort( ip=natSrc, port=action.addr.src.port )
         a.natDst = IpAddrAndPort( ip=natDst, port=action.addr.dst.port )

         for intfName, intf in action.intf.items():
            i = NatFlowIntf()
            i.mac = Arnet.EthAddr( intf.mac )
            i.vlanId = intf.vlan
            a.intfs[ intfName ] = i

         m.actions[ actionName ] = a

      p.matches[ matchName ] = m

   model.policies[ policy.name ] = p

def cmdShowIpNatFlowAction( mode, args ) :
   model = NatFlowActionCmdModel()
   policyName = args.get( "POLICY" )
   config = 'config' in args
   hardware = 'hardware' in args

   # The hardware cli is implemented in the agent because the nat flow port hw
   # status is not mounted in sysdb
   if hardware:
      cmd = "showNatFlowActionHw " + args.get( "POLICY", "" )
      AgentCommandRequest.runSocketCommand( mode.entityManager, "NatFlowMcast",
                                            "NatFlowMcast", cmd, timeout=60 )
   else:
      if policyName:
         if policyName not in natStatus.natFlow:
            mode.addError( "Policy %s does not exist" % ( policyName ) )
            return
         showIpNatFlowPolicyAction( model, policyName, config )
      else:
         for policy in natStatus.natFlow:
            showIpNatFlowPolicyAction( model, policy, config )

   return model

#------------------------------------------------------------------------------------
# "show ip nat flow profile"
#------------------------------------------------------------------------------------

def showIpNatFlowPolicyProfile( model, policyName, config, hardware ):
   ( flowColl, flowIntfColl ) = getNatFlowCollections( config, hardware )
   policy = flowColl.natFlow.get( policyName )
   if not policy:
      return
   for intfKey, intfValue in flowIntfColl.items():
      if policyName in intfValue.flowPolicy:
         intfName = natStatus.intfConfigKey[ intfKey ] if not config else intfKey
         info = NatFlowPolicyInfo()
         if hardware:
            policyStatus = intfValue.flowPolicy[ policyName ]
            for match, groups in policyStatus.matchStatus.items():
               matchInfo = NatFlowMatchInfo()
               matchInfo.groupUc = groups.groupUc
               matchInfo.groupMc = groups.groupMc
               info.match[ match ] = matchInfo

         p = model.profiles.get( intfName, NatFlowProfilePolicy() )
         p.policies[ policy.name ] = info
         model.profiles[ intfName ] = p

def cmdShowIpNatFlowProfile( mode, args ) :
   model = NatFlowProfileCmdModel()
   policyName = args.get( "POLICY" )
   config = 'config' in args
   hardware = 'hardware' in args
   model.hardware = hardware

   if policyName:
      if policyName not in natStatus.natFlow:
         mode.addError( "Policy %s does not exist" % ( policyName ) )
         return
      showIpNatFlowPolicyProfile( model, policyName, config, hardware )
   else:
      for policy in natStatus.natFlow:
         showIpNatFlowPolicyProfile( model, policy, config, hardware )
      if config:
         for intfKey, intfValue in natConfig.intfConfig.items():
            intfName = natStatus.intfConfigKey[ intfKey ] if not config else intfKey
            for policy in intfValue.flowPolicy:
               if policy not in natConfig.natFlow:
                  p = model.profiles.get( intfName, NatFlowProfilePolicy() )
                  p.policies[ policy ] = NatFlowPolicyInfo()
                  model.profiles[ intfName ] = p

   return model

#----------------------------------------------------------------------------------
# "[no|default] ip nat pool <pool name>" command, in "config" mode
#----------------------------------------------------------------------------------
def gotoIpNatPoolConfigMode( mode, args ):
   poolName = args[ 'POOLNAME' ]
   portOnly = 'port-only' in args
   netmask = args.get( 'PREFIXLEN' ) or args.get( 'MASK' )
   if not portOnly:
      if type( netmask ) == int: # pylint: disable=unidiomatic-typecheck
         prefixLen = netmask
      else:
         num = Arnet.IpAddress( netmask ).value
         try:
            prefixLen = Mask.ipMaskToPrefixLen()[ num ]
            if prefixLen < 16:
               mode.addError( "Invalid prefix length: %d, netmask should have " \
                              "prefix length of 16 or more" % prefixLen )
               return
         except KeyError:
            mode.addError( "Invalid netmask: %s" % netmask )
            return

   if portOnly:
      childMode = mode.childMode( NatPortOnlyPoolConfigMode, poolName=poolName )
   else:
      childMode = mode.childMode( NatPoolConfigMode, poolName=poolName, \
            prefixLen=prefixLen )
   mode.session_.gotoChildMode( childMode )

def deleteIpNatPoolMode( mode, args ):
   poolName = args[ 'POOLNAME' ]
   del natConfig.poolConfig[ poolName ]

def createIpNatPoolConfig( mode, args ):
   poolName = args[ 'POOLNAME' ]
   startIp = args[ 'STARTIP' ]
   startIpValue = Arnet.IpAddress( startIp ).value
   endIp = args[ 'ENDIP' ]
   endIpValue = Arnet.IpAddress( endIp ).value
   netmask = args.get( 'PREFIXLEN' ) or args.get( 'MASK' )
   if startIpValue > endIpValue:
      mode.addError( "end IP must be greater than or equal to start IP" )
      return

   startPort = args.get( 'STARTPORT', 0 )
   endPort = args.get( 'ENDPORT', 0 )

   # endport must be >= startPort
   if endPort < startPort:
      mode.addError( "end port must be greater than or equal to start port." )
      return

   if not checkPoolPortRangeAndNatSyncPortRange( mode, poolName, startPort, endPort,
                                                 natSyncConfig.portRange.startPort,
                                                 natSyncConfig.portRange.endPort ):
      return

   if type( netmask ) == int: # pylint: disable=unidiomatic-typecheck
      length = netmask
   else:
      num = Arnet.IpAddress( netmask ).value
      try:
         length = Mask.ipMaskToPrefixLen()[ num ]
      except KeyError:
         mode.addError( "Invalid netmask: %s" % netmask )
         return

   if poolName in natConfig.poolConfig:
      mode.addError( "Pool %s already exists." % poolName )
      return

   if Arnet.Subnet( startIp, length ).toNum() != \
      Arnet.Subnet( endIp, length ).toNum():
      mode.addError( "StartIp and EndIp do not lie in same subnet." )
      return

   poolId = natConfig.poolIdGen + 1
   poolConfig = natConfig.newPoolConfig( poolName, length, poolId )
   natConfig.poolIdGen = poolId
   pool = Tac.Value( "Ip::Nat::PoolRange", startIp=startIp, endIp=endIp,
                     startPort=startPort, endPort=endPort )
   poolConfig.poolRange[ pool ] = True

class NatPoolConfigMode( NatPoolMode, BasicCli.ConfigModeBase ):
   name = 'NAT pool configuration'

   def __init__( self, parent, session, poolName, prefixLen ):
      param = ( poolName, prefixLen )
      NatPoolMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.poolName = poolName
      self.prefixLen = prefixLen
      self.poolConfig = natConfig.poolConfig.get( self.poolName, None )

      if not self.poolConfig:
         poolId = natConfig.poolIdGen + 1
         self.poolConfig = natConfig.newPoolConfig( self.poolName, self.prefixLen,
                                                    poolId )
         natConfig.poolIdGen = poolId
         return

      if self.poolConfig.portOnly:
         # Existing portOnly pool being converted into IP pool. Remove all ranges.
         self.poolConfig.prefixLen = self.prefixLen
         self.poolConfig.poolRange.clear()
         self.poolConfig.portOnly = False
         return

      if self.poolConfig.prefixLen == self.prefixLen:
         return

      for pool in self.poolConfig.poolRange:
         if Arnet.Subnet( pool.startIp, self.prefixLen ).toNum() != \
            Arnet.Subnet( pool.endIp, self.prefixLen ).toNum():
            self.addError(
               "Existing pool ranges invalid with " \
               "prefix length %s" % ( self.prefixLen ) )
            return

      # Update the new prefix length
      self.poolConfig.prefixLen = self.prefixLen

   def addPoolRange( self, args ):
      startIp = args[ 'STARTIP' ]
      startIpValue = Arnet.IpAddress( startIp ).value
      endIp = args[ 'ENDIP' ]
      endIpValue = Arnet.IpAddress( endIp ).value
      if startIpValue > endIpValue:
         self.addError( "end IP must be greater than or equal to start IP" )
         return

      startPort = args.get( 'STARTPORT', 0 )
      endPort = args.get( 'ENDPORT', 0 )

      # endport must be >= startPort
      if endPort < startPort:
         self.addError( "end port must be greater than or equal to start port." )
         return

      if not checkPoolPortRangeAndNatSyncPortRange( self, self.poolName,
                                                   startPort, endPort,
                                                   natSyncConfig.portRange.startPort,
                                                   natSyncConfig.portRange.endPort ):
         return

      if Arnet.Subnet( startIp, self.prefixLen ).toNum() != \
         Arnet.Subnet( endIp, self.prefixLen ).toNum():
         self.addError( "Start and End Ip addresses do not lie in same subnet." )
         return

      for pool in self.poolConfig.poolRange:
         if Arnet.Subnet( pool.startIp, self.prefixLen ).toNum() != \
            Arnet.Subnet( pool.endIp, self.prefixLen ).toNum():
            self.addError(
               "Existing pool ranges invalid with " \
               "prefix length %s" % ( self.prefixLen ) )
            return

      pool = Tac.Value( "Ip::Nat::PoolRange", startIp=startIp, endIp=endIp,
                        startPort=startPort, endPort=endPort )
      self.poolConfig.poolRange[ pool ] = True
      natConfig.poolConfig[ self.poolName ].prefixLen = self.prefixLen

   def noPoolRange( self, args ):
      startIp = args[ 'STARTIP' ]
      endIp = args[ 'ENDIP' ]
      startPort = args.get( 'STARTPORT', 0 )
      endPort = args.get( 'ENDPORT', 0 )
      pool = Tac.Value( "Ip::Nat::PoolRange", startIp=startIp, endIp=endIp,
                        startPort=startPort, endPort=endPort )
      del self.poolConfig.poolRange[ pool ]

   def doSetIpNatPoolUtilizationThreshold( self, args ):
      pool = self.poolName
      poolConfig = natConfig.poolConfig.get( pool )
      if poolConfig:
         if '__no__' in args:
            poolConfig.poolUtilizationLogThreshold = 0
         elif '__default__' in args:
            poolConfig.poolUtilizationLogThreshold = \
               DefaultValue.poolUtilizationLogThresholdDefault
         else:
            threshold = args.get( 'VALUE' )
            poolConfig.poolUtilizationLogThreshold = threshold

class NatPortOnlyPoolConfigMode( NatPortOnlyPoolMode, BasicCli.ConfigModeBase ):
   name = 'NAT port-only pool configuration'

   def __init__( self, parent, session, poolName ):
      param = poolName
      NatPortOnlyPoolMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.poolName = poolName
      self.poolConfig = natConfig.poolConfig.get( self.poolName, None )

      if not self.poolConfig:
         poolId = natConfig.poolIdGen + 1
         self.poolConfig = natConfig.newPoolConfig( self.poolName, 0, poolId )
         self.poolConfig.portOnly = True
         natConfig.poolIdGen = poolId
         return

      if not self.poolConfig.portOnly:
         # Existing IP pool being converted into portOnly pool. Remove all ranges.
         self.poolConfig.prefixLen = 0
         self.poolConfig.poolRange.clear()

      self.poolConfig.portOnly = True

   def addPoolRange( self, args ):
      startPort = args[ 'STARTPORT' ]
      endPort = args[ 'ENDPORT' ]

      # endport must be >= startPort
      if endPort < startPort:
         self.addError( "end port must be greater than or equal to start port." )
         return

      if not checkPoolPortRangeAndNatSyncPortRange( self, self.poolName,
                                                   startPort, endPort,
                                                   natSyncConfig.portRange.startPort,
                                                   natSyncConfig.portRange.endPort ):
         return

      pool = Tac.Value( "Ip::Nat::PoolRange", startIp=Arnet.IpAddress( 0 ),
            endIp=Arnet.IpAddress( 0 ), startPort=startPort, endPort=endPort )
      self.poolConfig.poolRange[ pool ] = True
      natConfig.poolConfig[ self.poolName ].prefixLen = 0
      natConfig.poolConfig[ self.poolName ].portOnly = True

   def noPoolRange( self, args ):
      startPort = args[ 'STARTPORT' ]
      endPort = args[ 'ENDPORT' ]
      pool = Tac.Value( "Ip::Nat::PoolRange", startIp=Arnet.IpAddress( 0 ),
            endIp=Arnet.IpAddress( 0 ), startPort=startPort, endPort=endPort )
      del self.poolConfig.poolRange[ pool ]

#------------------------------------------------------------------------------------
# show ip nat pool [ synchronization ]
#------------------------------------------------------------------------------------
def computeSplitRange( poolRange ):
   emptyPortRange = Tac.newInstance( 'Ip::Nat::Sync::PortRange', 0, 0 )
   splitMode = natSyncStatus.portRangeMode
   modes = Tac.Type( 'Ip::Nat::Sync::PortRangeMode' )
   # if the pool has a configured range, split it.
   # if not, see if nat sync has a manually configured range and use that.
   # if not, split the entire 64k port range.
   if poolRange.startPort and poolRange.endPort:
      startPort = poolRange.startPort
      endPort = poolRange.endPort
   elif natSyncConfig.portRange != emptyPortRange:
      startPort = natSyncConfig.portRange.startPort
      endPort = natSyncConfig.portRange.endPort
   else:
      startPort = 0
      endPort = emptyPortRange.maxEndPort

   lowerSplitEnd = ( startPort + endPort ) // 2
   upperSplitStart = lowerSplitEnd + 1
   if splitMode == modes.portRangeLower:
      ( localStart, localEnd ) = ( startPort, lowerSplitEnd )
   elif splitMode == modes.portRangeUpper:
      ( localStart, localEnd ) = ( upperSplitStart, endPort )
   elif splitMode == modes.portRangeFull:
      ( localStart, localEnd ) = ( startPort, endPort )

   return (localStart, localEnd )


def cmdShowIpNatPool( mode, args ):
   poolName = args.get( 'POOL' )
   poolConfigDir = natConfig.poolConfig
   split = 'synchronization' in args

   if not poolConfigDir:
      mode.addError( "No pools have been configured yet" )
      return

   if poolName and poolName not in poolConfigDir:
      mode.addError( "Pool %s doesn't exist" % ( poolName ) )
      return

   poolModel = NatPoolModel()
   poolModel.poolName = poolName
   poolModel.natPools = {}
   poolModel.showSplit = split
   if poolName:
      poolConfigDir = { poolName : natConfig.poolConfig[ poolName ] }
   for name, pool in poolConfigDir.items():
      natRanges = []
      for poolRange in sorted( pool.poolRange ):
         if split:
            ( localStartPort, localEndPort ) = computeSplitRange( poolRange )
         else:
            localStartPort = localEndPort = 0
         natRanges.append(
            NatPoolModel.NatPool.NatRange(
               startIp=poolRange.startIp, endIp=poolRange.endIp,
               startPort=poolRange.startPort, endPort=poolRange.endPort,
               localStartPort=localStartPort,
               localEndPort=localEndPort ) )
      prefixLen = pool.prefixLen
      poolId = pool.poolId
      portOnly = pool.portOnly
      poolModel.natPools[ name ] = NatPoolModel.NatPool( natRanges=natRanges,
                                                         prefixLen=prefixLen,
                                                         poolId=poolId,
                                                         portOnly=portOnly )

   return poolModel

#----------------------------------------------------------------------------------
# "[no|default] ip nat service-list <service-list-name>" command, in "config" mode
#----------------------------------------------------------------------------------
class NatServiceListService:
   protos = [ 'ftp' ]

   @staticmethod
   def addService( serviceListConfig, args ):
      service = args[ 'SERVICETYPE' ]
      if service not in NatServiceListService.protos:
         err = "Unknown service type '%s'" % service
      else:
         serviceListConfig.serviceFtp = True
         err = None
      return err

   @staticmethod
   def noService( serviceListConfig, args ):
      service = args[ 'SERVICETYPE' ]
      if service not in NatServiceListService.protos:
         err = "Unknown service type '%s'" % service
      else:
         serviceListConfig.serviceFtp = False
         err = None
      return err

   @staticmethod
   def showService( serviceListConfig ):
      cmdList = []
      if serviceListConfig.serviceFtp:
         cmd = NatServiceListCommandModel()
         cmd.protocol = 'ftp'
         cmdList.append( cmd )
      return cmdList

class NatServiceListConfigMode( NatServiceListMode, BasicCli.ConfigModeBase ):
   name = 'NAT service list configuration'

   def __init__( self, parent, session, serviceListName ):
      NatServiceListMode.__init__( self, serviceListName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.serviceListName = serviceListName
      self.serviceListConfig = natConfig.serviceListConfig.get(
                                 self.serviceListName, None )

      if not self.serviceListConfig:
         self.serviceListConfig = natConfig.newServiceListConfig(
                                       self.serviceListName )
         return

   def addService( self, args ):
      err = NatServiceListService.addService( self.serviceListConfig, args )
      if err:
         self.addError( err )

   def noService( self, args ):
      err = NatServiceListService.noService( self.serviceListConfig, args )
      if err:
         self.addError( err )

def gotoIpNatServiceListConfigMode( mode, args ):
   serviceListName = args[ 'NAME' ]
   if serviceListName not in natConfig.serviceListConfig:
      natConfig.newServiceListConfig( serviceListName )

   childMode = mode.childMode( NatServiceListConfigMode,
                               serviceListName=serviceListName )
   mode.session_.gotoChildMode( childMode )

def deleteIpNatServiceListMode( mode, args ):
   serviceListName = args[ 'NAME' ]
   if serviceListName in natConfig.serviceListConfig:
      del natConfig.serviceListConfig[ serviceListName ]

#------------------------------------------------------------------------------------
# show ip nat service-list
#------------------------------------------------------------------------------------
def showIpNatServiceList( serviceListConfig, model ):
   s = NatServiceListModel.NatServiceList()
   s.serviceListName = serviceListConfig.name
   s.serviceCmdList = NatServiceListService.showService( serviceListConfig )
   model.serviceLists.append( s )

def cmdShowIpNatServiceList( mode, args ):
   serviceListName = args.get( "SERVICE_LIST" )
   model = NatServiceListModel()
   if len(natConfig.serviceListConfig):
      if serviceListName:
         if serviceListName not in natConfig.serviceListConfig:
            mode.addError( "Service-list %s doesn't exist" % ( serviceListName ) )
            return
         serviceListConfig = natConfig.serviceListConfig[ serviceListName ]
         showIpNatServiceList( serviceListConfig, model )
      else:
         for serviceListConfig in natConfig.serviceListConfig.values():
            showIpNatServiceList( serviceListConfig, model )
   return model

#----------------------------------------------------------------------------------
# "[no|default] ip nat translation max-entries <limit>
#     [ host | <ipaddress> | pool <poolname> | symmetric | full-cone ]"
# in "config" mode
#----------------------------------------------------------------------------------
def cmdIpNatTranslationMaxEntries( mode, args ):
   maxEntries = args[ 'MAXENTRIES' ]
   addr = args.get( 'IPADDR' )
   if 'host' in args:
      # Set per Host connection limit
      natConfig.settings.allHostsConnLimit = maxEntries
   elif addr:
      addrWithMark = Tac.Value( "Arnet::IpAddrWithMask", addr, 32 )
      ipAddr = addrWithMark.address
      hc = Tac.Value( "Ip::Nat::HostConnectionConfig", ipAddr )
      hc.connLimit = maxEntries
      natConfig.hostConnectionConfig.addMember( hc )
   elif 'pool' in args:
      poolName = args.get('POOLNAME')
      pc = Tac.Value( "Ip::Nat::PoolConnectionConfig", poolName )
      pc.connLimit = maxEntries
      natConfig.poolConnectionConfig.addMember( pc )
   elif 'symmetric' in args:
      natConfig.settings.symmetricConnLimit = maxEntries
   elif 'full-cone' in args:
      natConfig.settings.fullConeConnLimit = maxEntries
   else:
      # no options specified. set global connection limit
      natConfig.settings.connLimit = maxEntries

def cmdNoIpNatTranslationMaxEntries( mode, args ):
   addr = args.get( 'IPADDR' )
   if 'host' in args:
      natConfig.settings.allHostsConnLimit = DefaultValue.connLimitDefault
   elif addr:
      addrWithMark = Tac.Value( "Arnet::IpAddrWithMask", addr, 32 )
      ipAddr = addrWithMark.address
      if ipAddr in natConfig.hostConnectionConfig:
         del natConfig.hostConnectionConfig[ addr ]
   elif 'pool' in args:
      poolName = args.get('POOLNAME')
      if poolName in natConfig.poolConnectionConfig:
         del natConfig.poolConnectionConfig[ poolName ]
   elif 'symmetric' in args:
      natConfig.settings.symmetricConnLimit = DefaultValue.connLimitDefault
   elif 'full-cone' in args:
      natConfig.settings.fullConeConnLimit = DefaultValue.connLimitDefault
   else:
      natConfig.settings.connLimit = DefaultValue.connLimitDefault

#----------------------------------------------------------------------------------
# "[no|default] ip nat translation low-mark <%>
#     [ host | pool | symmetric | full-cone ]", in "config" mode
#----------------------------------------------------------------------------------
def cmdIpNatTranslationLowMark( mode, args ):
   lowMarkPerc = args[ 'PERCENT' ]
   settings = natConfig.settings

   if 'host' in args:
      settings.allHostsLowMarkPerc = lowMarkPerc
   elif 'pool' in args:
      settings.allPoolsLowMarkPerc = lowMarkPerc
   elif 'symmetric' in args:
      settings.symmetricConnLimitLowMarkPerc = lowMarkPerc
   elif 'full-cone' in args:
      settings.fullConeConnLimitLowMarkPerc = lowMarkPerc
   else:
      settings.connLimitLowMarkPerc = lowMarkPerc

def cmdNoIpNatTranslationLowMark( mode, args ):
   settings = natConfig.settings
   if 'host' in args:
      settings.allHostsLowMarkPerc = DefaultValue.connLimitLowMarkPercDefault
   elif 'pool' in args:
      settings.allPoolsLowMarkPerc = DefaultValue.connLimitLowMarkPercDefault
   elif 'full-cone' in args:
      settings.fullConeConnLimitLowMarkPerc =\
      DefaultValue.connLimitLowMarkPercDefault
   elif 'symmetric' in args:
      settings.symmetricConnLimitLowMarkPerc =\
      DefaultValue.connLimitLowMarkPercDefault
   else:
      settings.connLimitLowMarkPerc = DefaultValue.connLimitLowMarkPercDefault

#------------------------------------------------------------------------------------
# [no|default] ip nat tcp flow creation trigger <syn | syn-ack>
#------------------------------------------------------------------------------------

def setTcpFlowCreationTrigger( mode, args ):
   value = args.get( 'TRIGGER', 'syn-ack' )
   natConfig.settings.tcpSynAckRequired = value == 'syn-ack'

#----------------------------------------------------------------------------------
# [no|default] ip nat translation address selection algorithm ( hash | random )
#----------------------------------------------------------------------------------

def setIpNatTranslationAlgorithm( mode, args ):
   natConfig.settings.natAddressSelectionHash = 'hash' in args

def setIpNatTranslationAlgorithmDefault( mode, args ):
   natConfig.settings.natAddressSelectionHash = True

#------------------------------------------------------------------------------------
# [no|default] ip nat translation address selection hash field source-ip
#------------------------------------------------------------------------------------

def setIpNatTranslationHash( mode, args ):
   natConfig.settings.sourceIpOnlyHash = True

def setIpNatTranslationHashDefault( mode, args ):
   natConfig.settings.sourceIpOnlyHash = False

#------------------------------------------------------------------------------------
# [no|default] ip nat translation address selection any
#------------------------------------------------------------------------------------

def setIpNatTranslationAddressSelection( mode, args ):
   natConfig.settings.ipAddressTranslationAny = True

def setIpNatTranslationAddressSelectionDefault( mode, args ):
   natConfig.settings.ipAddressTranslationAny = False

#------------------------------------------------------------------------------------
# show ip nat translation max-entries [<ADDR> | pool <POOL> ]
#------------------------------------------------------------------------------------
def cmdShowIpNatMaxEntries( mode, args ):
   if not AgentDirectory.agent( mode.sysname, 'Nat' ):
      mode.addError( "Nat Agent not running." )
      return

   command = "showIpNatMaxEntries"
   agent = NatAgent.agentName() 
   commandType = "NatCliWithFormat" 

   # In dpdk mode the handler for the command is in the Sfe agent
   if not toggleNatConnLimitNewCodeEnabled() and \
      natHwCapabilities and natHwCapabilities.natDpdkProcessingSupported:
      agent = "Sfe"
      commandType = "SfeCliWithFormat"

   AgentCommandRequest.runCliPrintSocketCommand(
      mode.entityManager, agent, commandType, command, mode=mode )
   return NatConnectionLimit

def cmdShowIpNatMaxEntriesHost( mode, args ):
   ipAddr = args[ 'IPADDR' ]

   if not AgentDirectory.agent( mode.sysname, 'Nat' ):
      mode.addError( "Nat Agent not running." )
      return

   command = "showIpNatMaxEntriesHost ipAddr " + ipAddr
   agent = NatAgent.agentName() 
   commandType = "NatCliWithFormat" 

   if not toggleNatConnLimitNewCodeEnabled() and \
      natHwCapabilities and natHwCapabilities.natDpdkProcessingSupported:
      agent = "Sfe"
      commandType = "SfeCliWithFormat"

   AgentCommandRequest.runCliPrintSocketCommand(
      mode.entityManager, agent, commandType, command, mode=mode )
   return NatHostConnectionLimit

def cmdShowIpNatMaxEntriesPool( mode, args ):
   poolName = args[ 'POOL' ]

   if not AgentDirectory.agent( mode.sysname, 'Nat' ):
      mode.addError( "Nat Agent not running." )
      return

   command = "showIpNatMaxEntriesPool poolName " + poolName
   agent = NatAgent.agentName()
   commandType = "NatCliWithFormat"

   if not toggleNatConnLimitNewCodeEnabled() and \
      natHwCapabilities and natHwCapabilities.natDpdkProcessingSupported:
      agent = "Sfe"
      commandType = "SfeCliWithFormat"

   AgentCommandRequest.runCliPrintSocketCommand(
      mode.entityManager, agent, commandType, command, mode=mode )
   return NatPoolConnectionLimit

#------------------------------------------------------------------------------------
# show ip nat translation rates
#------------------------------------------------------------------------------------

def collectTranslationRates( translationType, collection, ratesAll ):

   if 0 not in collection:
      return

   rates = NatTranslationRates()
   index = collection.get( 0 ).adds
   rateCountSize = Tac.Value( "Ip::Nat::SmashDefaultValues" ).rateCountSize

   adds = 0
   dels = 0
   for i in range( rateCountSize ):
      if not index:
         index = rateCountSize
      rateCount = collection.get( index )
      if not rateCount:
         break

      adds += rateCount.adds
      dels += rateCount.dels

      # Display rates for 5, 30, 60 and 300 seconds
      if i in ( 0, 5, 11, 59 ):
         rate = NatTranslationRate()
         rate.timeLen = ( i + 1 ) * 5
         rate.numAdd = adds
         rate.numDel = dels
         rates.rate.append( rate )

      index -= 1

   if rates:
      ratesAll.rates[ translationType ] = rates

def cmdShowIpNatTranslationRates( mode, args ):
   if not AgentDirectory.agent( mode.sysname, 'Nat' ):
      mode.addError( "Nat Agent not running." )
      return

   rates = NatTranslationRatesAll()

   if natHwCapabilities.natDpdkProcessingSupported:
      collectTranslationRates( "NAT DPDK", dynamicConnectionStatus.rateCount, rates )
   collectTranslationRates( "NAT Ale",
                            dynamicConnectionHwStatus.rateCountAle, rates )
   collectTranslationRates( "NAT hardware",
                            dynamicConnectionHwStatus.rateCountHw, rates )

   return rates

#----------------------------------------------------------------------------------
# "[no|default] ip nat translation counters
#----------------------------------------------------------------------------------
def cmdIpNatTranslationCounters( mode, args ):
   natConfig.settings.counterEnabled = True

def cmdNoIpNatTranslationCounters( mode, args ):
   natConfig.settings.counterEnabled = False

#----------------------------------------------------------------------------------
# "[ no | default ] ip nat logging"
#----------------------------------------------------------------------------------
def cmdIpNatLogging( mode, args ):
   natConfig.settings.loggingEnabled = True

def cmdNoIpNatLogging( mode, args ):
   natConfig.settings.loggingEnabled = False

#------------------------------------------------------------------------------------
# "show ip nat logging"
#------------------------------------------------------------------------------------
def cmdShowIpNatLogging( mode, args ):
   logging = NatLoggingModel()
   logging.enabled = natConfig.settings.loggingEnabled
   return logging

#------------------------------------------------------------------------------------
# [no | default] hardware nat static access-list resource sharing
#------------------------------------------------------------------------------------
def cmdHardwareNatStaticAclResourceSharing( mode, args ):
   natConfig.settings.staticAclSharingEnabled = True

def cmdNoHardwareNatStaticAclResourceSharing( mode, args ):
   natConfig.settings.staticAclSharingEnabled = False


#------------------------------------------------------------------------------------
# [ no | default ] hardware nat [ static | dynamic ] fragment [ disabled ]
#------------------------------------------------------------------------------------

def setNewHardwareNatFragmentState( mode, args, staticState, dynamicState ):
   newFragmentConfig = Tac.Value( "Ip::Nat::NatFragmentConfig",
                                  staticState, dynamicState )
   if newFragmentConfig != natConfig.fragmentConfig:
      mode.addWarning( reprogramWarningStr )
      natConfig.fragmentConfig = newFragmentConfig

def cmdHardwareNatFragment( mode, args ):
   newState = 'disabled' not in args

   if 'dynamic' not in args:
      staticState = 'fragmentEnabled' if newState else 'fragmentDisabled'
   else:
      staticState = natConfig.fragmentConfig.staticFragment

   if 'static' not in args:
      dynamicState = 'fragmentEnabled' if newState else 'fragmentDisabled'
   else:
      dynamicState = natConfig.fragmentConfig.dynamicFragment

   setNewHardwareNatFragmentState( mode, args, staticState, dynamicState )

def cmdNoHardwareNatFragment( mode, args ):
   setNewHardwareNatFragmentState( mode, args, 'fragmentDefault', 'fragmentDefault' )

#------------------------------------------------------------------------------------
# [ no | default ] hardware nat vrf leak
#------------------------------------------------------------------------------------
def cmdHardwareNatVrfLeak( mode, args ):
   natConfig.settings.vrfLeakEnabled = True

def cmdNoHardwareNatVrfLeak( mode, args ):
   natConfig.settings.vrfLeakEnabled = False

#------------------------------------------------------------------------------------
# show ip nat counters [vrf <name>]
#------------------------------------------------------------------------------------
def cmdShowIpNatCounters( mode, args ):
   vrfName = args.get( 'VRF' )
   vrfName = vrfMap.lookupCliModeVrf( mode, vrfName )
   if not ( vrfName == ALL_VRF_NAME or vrfExists( vrfName ) ):
      mode.addError( "VRF %s does not exist." % vrfName )
      return None

   if vrfName == ALL_VRF_NAME:
      vrfs = natStatus.vrfCounters
   else:
      vrfs = [ vrfName ]

   vrfCounters = NatVrfCounters()
   for vrf in vrfs:
      counters = NatVrfCounters.NatCounters()
      if vrf not in natStatus.vrfCounters:
         cntr = Tac.newInstance( "Ip::Nat::VrfCounters", "" )
      else:
         cntr = natStatus.vrfCounters[ vrf ]
      counters.newConnectionEvents = cntr.newConnEvent
      counters.updateConnectionEvents = cntr.updateConnEvent
      counters.deleteConnectionEvents = cntr.deleteConnEvent
      counters.updateConnectionErrors = cntr.updateConnError
      counters.deleteConnectionErrors = cntr.deleteConnError
      counters.kernelSyncRequests = cntr.kernelSyncRequest
      counters.kernelSyncRequestErrors = cntr.kernelSyncRequestError
      counters.kernelSyncInProgress = cntr.kernelSyncInProgress
      counters.iptablesWriteErrors = cntr.iptablesWriteError
      counters.netlinkSyncEagainErrors = cntr.nlSyncEagainError
      counters.netlinkSyncNoBufferErrors = cntr.nlSyncNoBufferError
      counters.netlinkSyncReceiveErrors = cntr.nlSyncReceiveError
      counters.netlinkSyncMsgErrors = cntr.nlSyncMsgError
      counters.netlinkSyncCreateTxErrors = cntr.nlSyncCreateTxError
      counters.netlinkSyncUpdateTxErrors = cntr.nlSyncUpdateTxError
      counters.netlinkSyncDeleteTxErrors = cntr.nlSyncDeleteTxError
      counters.netlinkReceiveErrors = cntr.nlReceiveError
      counters.netlinkNoBufferErrors = cntr.nlNoBufferError
      counters.netlinkTruncatedMessages = cntr.nlTruncatedMsg
      counters.netlinkSeqNumMismatches = cntr.nlSeqNumMismatch
      counters.netlinkSeqNumRangeMismatches = cntr.nlSeqNumRangeMismatch
      counters.netlinkBadMessageLengthErrors = cntr.nlBadMsgLen
      counters.netlinkNoSpaceErrors = cntr.nlNoSpaceError
      counters.netlinkDumpDoneErrors = cntr.nlDumpDoneError
      counters.netlinkDumpIgnoreTcpConnections = cntr.nlDumpIgnoreTcpConn
      counters.enobufDumpCnt = cntr.enobufDumpCnt
      counters.enobufMsgCnt = cntr.enobufMsgCnt
      counters.eagainDumpCnt = cntr.eagainDumpCnt
      counters.eagainRecvCnt = cntr.eagainRecvCnt
      counters.maxMsgCntInRecvBuf = cntr.maxMsgCntInRecvBuf
      counters.netlinkMsgErrors = cntr.nlMsgErr
      counters.netlinkSyncFCConflictErrors = cntr.nlSyncFCConflictError

      if cntr.nlAdd5s != 0xFFFFFFFF:
         counters.nlAdd5s = cntr.nlAdd5s // 5
         counters.nlDel5s = cntr.nlDel5s // 5
      if cntr.nlAdd30s != 0xFFFFFFFF:
         counters.nlAdd30s = cntr.nlAdd30s // 30
         counters.nlDel30s = cntr.nlDel30s // 30
      if cntr.nlAdd60s != 0xFFFFFFFF:
         counters.nlAdd60s = cntr.nlAdd60s // 60
         counters.nlDel60s = cntr.nlDel60s // 60
      if cntr.nlAdd300s != 0xFFFFFFFF:
         counters.nlAdd300s = cntr.nlAdd300s // 300
         counters.nlDel300s = cntr.nlDel300s // 300

      vrfCounters.vrfs[ vrf ] = counters

   return vrfCounters

#------------------------------------------------------------------------------------
# clear ip nat counters [vrf <name>]
#------------------------------------------------------------------------------------
def cmdClearIpNatCounters( mode, args ):
   if not AgentDirectory.agent( mode.sysname, 'Nat' ):
      mode.addError( "Nat Agent not running." )
      return

   vrfName = args.get( 'VRF' )
   vrfName = vrfMap.lookupCliModeVrf( mode, vrfName )
   if not ( vrfName == ALL_VRF_NAME or vrfExists( vrfName ) ):
      mode.addError( "VRF %s does not exist." % vrfName )
      return

   cmd = "clearCountersRequest vrf " + vrfName
   AgentCommandRequest.runSocketCommand( mode.entityManager,
                                         NatAgent.agentName(), "Nat",
                                         cmd, timeout=60 )

#------------------------------------------------------------------------------------
# show ip nat translation summary
#------------------------------------------------------------------------------------

def doShowKernelTranslationSummary( ipAddr ):
   # CAPI team prefers the NAT Summary to be modelled as a Dict. Kernel entries
   # with no interfaces should not occur ideally. They may only occur if there
   # is a software bug or race conditions. Such entries can be determined using
   # "show ip nat translation kernel" command also. So skip them here
   natKernelSummary = {}

   cmdStr = 'sudo conntrack -L'
   if ipAddr:
      cmdStr += '| grep %s' % ipAddr

   # Run one kernel query for every intfStatus (i.e. for every interface if in
   # interface config mode, or for every profile in profile config mode)
   for intfStatusKey, intfStatus in natStatus.intfStatus.items():
      # It's possible for the intfStatus to exist while the intfConfig has already
      # been cleaned up. In this case, skip over this intf.
      if intfStatusKey not in natStatus.intfConfigKey:
         continue
      intfName = natStatus.intfConfigKey[ intfStatusKey ]
      vrfName = intfStatus.vrfName
      if vrfName != DEFAULT_VRF:
         vrfStatus = allVrfStatusLocal.vrf.get( vrfName )
         if not vrfStatus:
            continue
         ns = vrfStatus.networkNamespace
         cmdStatus = ( "sudo ip netns exec %s " % ns ) + cmdStr
      else:
         cmdStatus = cmdStr

      # Actually, run 2 queries for each intfStatus: one for dynamic entries and
      # one for dynamic twice entries
      for dynamicTwice in [ False, True ]:
         firstMark = True
         for mark in intfStatus.ruleConnMark.values():
            markRule = natStatus.connMarkRule.get( mark )
            if not markRule:
               continue
            dynamicTwiceValid = ( markRule.cmrKey.group != 0 )
            if dynamicTwiceValid != dynamicTwice:
               continue
            if firstMark:
               cmd = cmdStatus + ' | grep "mark=%d' % mark
               firstMark = False
            else:
               cmd += r"\|mark=%d" % mark
         if firstMark:
            continue

         cmd += '" | wc -l'

         # pylint: disable-next=consider-using-with
         sub = subprocess.Popen( cmd, shell=True,
                                 stdout=subprocess.PIPE, stderr=subprocess.STDOUT )
         try:
            intfConnCnt = int( sub.communicate()[ 0 ].split( b'\n' )[ 1 ] )
         except ( IndexError, KeyError ):
            continue

         if intfConnCnt:
            currCount = natKernelSummary.get( intfName, [ 0, 0 ] )
            if dynamicTwice:
               currCount[ 1 ] += intfConnCnt
            else:
               currCount[ 0 ] += intfConnCnt
            natKernelSummary[ intfName ] = currCount

   return natKernelSummary

def cmdShowIpNatTranslationDynamicSummary( mode, args ):
   transDynSummaryDict = NatTranslationDynamicSummary()

   for natIntf in dynamicConnectionStatus.counter:
      # Get the name of the profile or interface corresponding 
      # to this NAT intf
      intfStatus = natStatus.intfStatus.get( natIntf )

      if intfStatus is None:
         continue

      intfName = intfStatus.name

      swCounter = dynamicConnectionStatus.counter.get( natIntf )
      hwCounter = dynamicConnectionHwStatus.counter.get( natIntf )
      if swCounter is None or hwCounter is None:
         continue

      dictEntry = NatTranslationDynamicSummary.DynamicTranslationSummary()

      dictEntry.numDynSwConnections = swCounter.count
      dictEntry.numDynHwConnections = hwCounter.count
      dictEntry.numFcSwConnections = swCounter.fullConeCount
      dictEntry.numFcHwConnections = hwCounter.fullConeCount
      dictEntry.numSymSwConnections = swCounter.symmetricCount
      dictEntry.numSymHwConnections = hwCounter.symmetricCount

      transDynSummaryDict.interfaces[ intfName ] = dictEntry

   return transDynSummaryDict

def cmdShowIpNatTranslationSummary( mode, args ):
# pylint: disable-msg=too-many-nested-blocks, R1702
   ipAddr = args.get( 'ADDRESS' )
   # tableType == 'hardware'
   transSummaryDict = NatTranslationSummary()

   # Dynamic connections hw (smash table)
   dynConnHw = {}

   if ipAddr:
      for connKey in dynamicConnectionHwStatus.dynamicConnectionHw:
         natIntf = getNatIntfId( connKey.natIntf )
         if ipAddr:
            if connKey.connTuple.srcAddr.ip == ipAddr:
               dynConnHw[ natIntf ] = dynConnHw.get( natIntf, 0 ) + 1
         else:
            dynConnHw[ natIntf ] = dynConnHw.get( natIntf, 0 ) + 1
   else:
      for natIntf in dynamicConnectionHwStatus.counter:
         counter = dynamicConnectionHwStatus.counter.get( natIntf )
         dynConnHw[ natIntf ] = counter.count if counter else 0

   # Dynamic connections (smash table)
   dynConn = {}
   dynTwiceConn = {}
   if ipAddr:
      for connKey, connEntry in \
          dynamicConnectionStatus.dynamicConnection.items():
         natIntf = getNatIntfId( connKey.natIntf )
         if not ipAddr or connKey.connTuple.srcAddr.ip == ipAddr:
            if connEntry.twiceNat():
               dynTwiceConn[ natIntf ] = dynTwiceConn.get( natIntf, 0 ) + 1
            else:
               dynConn[ natIntf ] = dynConn.get( natIntf, 0 ) + 1
   else:
      for natIntf in dynamicConnectionStatus.counter:
         counter = dynamicConnectionStatus.counter.get( natIntf )
         if counter is None:
            continue
         if counter.twiceCount:
            dynTwiceConn[ natIntf ] = counter.twiceCount
         dynConn[ natIntf ] = counter.count - counter.twiceCount

   # Hw connections
   for intfId in natHwStatus.intfStatus:
      intfStatusHw = natHwStatus.intfStatus.get( intfId )
      if intfStatusHw is None:
         continue

      intfName = intfStatusHw.name
      if ipAddr:
         numStaticHwConnections = sum( 1 for connEntry in \
                                       intfStatusHw.staticConnectionHw.values() \
                                       if connEntry.staticKey.localAddress.ip \
                                          == ipAddr )
         numTwiceHwConnections = 0
         numDynTwiceHwConnections = 0
         for twiceConn in intfStatusHw.twiceNatConnectionHw.values():
            for addrOld in twiceConn.entry:
               if addrOld.srcAddr.ip == ipAddr:
                  if twiceConn.dynamic:
                     numDynTwiceHwConnections += 1
                  else:
                     numTwiceHwConnections += 1
      else:
         numStaticHwConnections = len( intfStatusHw.staticConnectionHw )
         numTwiceHwConnections = 0
         numDynTwiceHwConnections = 0
         for twiceConn in intfStatusHw.twiceNatConnectionHw.values():
            if twiceConn.dynamic:
               numDynTwiceHwConnections += len( twiceConn.entry )
            else:
               numTwiceHwConnections += len( twiceConn.entry )
      numDynHwConnections = dynConnHw.get( intfStatusHw.natIntf, 0 )

      # If all counts are 0, don't return this interface. Ideally this should
      # only happen when address filter is specified in show command
      if ipAddr and ( numStaticHwConnections + numDynHwConnections +
                      numTwiceHwConnections + numDynTwiceHwConnections ) == 0:
         continue

      intfSummary = transSummaryDict.translations.setdefault(
                        intfName, NatTranslationSummary.TranslationSummary() )
      intfSummary.numStaticHwConnections += numStaticHwConnections
      intfSummary.numDynHwConnections += numDynHwConnections
      intfSummary.numTwiceHwConnections += numTwiceHwConnections
      intfSummary.numDynTwiceHwConnections += numDynTwiceHwConnections

   # Sw (Sysdb) connections
   for intfId in natStatus.intfStatus:
      intfStatus = natStatus.intfStatus.get( intfId )
      if intfStatus is None:
         continue

      intfName = intfStatus.name
      numTwiceSwConnections = 0
      if ipAddr:
         numStaticSwConnections = sum( 1 for connEntry in \
                                       intfStatus.staticConnection.values() \
                                   if connEntry.staticKey.localAddress.ip == ipAddr )
         for connEntry in intfStatus.twiceNatConnection.values():
            if connEntry.srcIpAndPortOld.ip == ipAddr and \
               connEntry.dstIpAndPortOld.ip != '0.0.0.0':
               numTwiceSwConnections += 1
      else:
         numStaticSwConnections = len( intfStatus.staticConnection )
         for connEntry in intfStatus.twiceNatConnection.values():
            if connEntry.srcIpAndPortOld.ip != '0.0.0.0' and \
               connEntry.dstIpAndPortOld.ip != '0.0.0.0':
               numTwiceSwConnections = len ( intfStatus.twiceNatConnection )
      numDynamicSwConnections = dynConn.get( intfStatus.natIntf, 0 )
      numDynamicTwiceSwConnections = dynTwiceConn.get( intfStatus.natIntf, 0 )

      # If all counts are 0, don't return this interface. Ideally this should
      # only happen when address filter is specified in show command
      if ( numStaticSwConnections + numTwiceSwConnections + \
           numDynamicSwConnections + numDynamicTwiceSwConnections ) == 0:
         continue

      intfSummary = transSummaryDict.translations.setdefault(
                        intfName, NatTranslationSummary.TranslationSummary() )
      intfSummary.numStaticSwConnections += numStaticSwConnections
      intfSummary.numTwiceSwConnections += numTwiceSwConnections
      intfSummary.numDynSwConnections += numDynamicSwConnections
      intfSummary.numDynTwiceSwConnections += numDynamicTwiceSwConnections

   # tableType == 'kernel'
   trans = doShowKernelTranslationSummary( ipAddr )

   for intName in trans: # pylint: disable=consider-using-dict-items
      intfSummary = transSummaryDict.translations.setdefault(
                        intName, NatTranslationSummary.TranslationSummary() )
      intfSummary.numKernelConnections = trans[ intName ][ 0 ]
      intfSummary.numDynTwiceKernelConnections = trans[ intName ][ 1 ]

   return transSummaryDict

def getVrfNameFromHwId_( vrfId ):
   for r in routingHwRouteStatus.vrfStatus.values():
      if r.vrfId == vrfId:
         return r.vrfName
   return ""

def getPoolNameFromId_( poolId ):
   for poolName, pool in natConfig.poolConfig.items():
      if pool.poolId == poolId:
         return poolName

   return ""

def cmdShowIpNatTranslationConnectionDropped( mode, args ):
   vrf = args.get( 'VRF' )
   model = NatTranslationConnectionDropped()

   def populateDpdkHostCounters_( vrf, model, counters ):
      table = NatTranslationConnectionDropped.DropTable()
      table.vrfName = vrf

      for hc in counters[ 'host_counter' ]:
         # get vrf name
         counterVrf = getVrfNameFromHwId_( hc[ 'vrf_id' ] )
         if vrf != counterVrf:
            continue

         for r in natDropRulesStatus.dropRulesHost:
            counter = NatTranslationConnectionDropped.DropTable.DropCounter()

            ipIntfStatus = natStatus.intfStatus.get( r.natIntf )
            if not ipIntfStatus:
               continue

            if ipIntfStatus.vrfName != vrf:
               continue

            if r.host != hc[ 'host' ]:
               continue

            if r.target == 'source':
               counter.srcIp = IpAddrAndMask( ip=r.host, mask=32 )
               counter.dstIp = IpAddrAndMask( ip='0.0.0.0', mask=0 )
            else:
               counter.srcIp = IpAddrAndMask( ip='0.0.0.0', mask=0 )
               counter.dstIp = IpAddrAndMask( ip=r.host, mask=32 )
            counter.counter = int( hc[ 'value' ] )
            table.counters.append( counter )

      model.vrfs[ vrf ] = table

   def populateDpdkPoolCounters_( vrf, model, counters ):
      for pc in counters[ 'pool_counter' ]:
         # get vrf name
         vrfName = getVrfNameFromHwId_( pc[ 'vrf_id' ] )
         if vrf != vrfName:
            continue

         poolName = getPoolNameFromId_( int( pc[ 'pool_id' ] ) )
         if not poolName:
            continue

         counter = NatTranslationConnectionDropped.PoolDropCounter()
         counter.poolName = poolName
         counter.vrfName = vrfName
         counter.counter = int( pc[ 'value' ] )
         model.pools.append( counter )

   def populateDpdkGlobalCounters_( model, counters ):
      rules = natDropRulesStatus.dropRulesBool

      def addCounter_( model, rules, ruleType, typeName, cnt ):
         counter = NatTranslationConnectionDropped.DropRuleCounter()
         if ruleType in rules:
            counter.counter = int( cnt )
            model.rules[ typeName ] = counter

      addCounter_( model, rules, 
         Tac.Value('Ip::Nat::NatDropRule', ** {'type': 'dropRules'}),
         "Global",
         counters[ 'drop_rule_pkts' ] )

      addCounter_( model, rules, 
         Tac.Value('Ip::Nat::NatDropRule', ** {'type': 'dropRulesFullCone'}),
         "Full Cone",
         counters[ 'drop_rule_full_cone_pkts' ] )

      addCounter_( model, rules, 
         Tac.Value('Ip::Nat::NatDropRule', ** {'type': 'dropRulesSymmetric'}),
         "Symmetric",
         counters[ 'drop_rule_symmetric_pkts' ] )
   

   def populateKernelCounters_( vrf, model ):
      # indexes of fields in iptables rows
      # 'pkts bytes target prot opt in out source destination'
      indexPkts = 0
      indexDeviceIn = 5
      indexDeviceOut = 6
      indexSrc = 7
      indexDst = 8

      table = NatTranslationConnectionDropped.DropTable()
      table.vrfName = vrf


      ns = DEFAULT_NS if vrf == DEFAULT_VRF else \
          allVrfStatusLocal.vrf[ vrf ].networkNamespace
      iptablesOutput = runMaybeInNetNs( ns,
            [ "iptables", "-t", "filter", "-L",
               RT( "{{forwardchain}}" ), "-v", "-n", "-x" ],
            asRoot=True,
            stdout=subprocess.PIPE, stderr=subprocess.STDOUT )
      iptablesOutput = iptablesOutput.split( '\n' )
      # filter DROP rows
      dropOutput = [ x.split() for x in iptablesOutput
            if re.search( r'^(\s*)(\d+)(\s+)(\d+)(\s+)DROP', x ) ]

      def _compareDevices( rule, record ):
         if not rule.inToOut:
            return rule.deviceName == record[ indexDeviceIn ]
         return rule.deviceName == record[ indexDeviceOut ]

      def _compareIpv4( ipTyped, ipStr ):
         # ipStr (that comes from iptabes output) doesn't have mask,
         # compare address itlsef
         if ipStr.find( "/" ) == -1:
            return ipTyped.address == ipStr
         return ipTyped.stringValue == ipStr

      def _compareHostDropRules( rule, record ):
         shortName = Tac.Value( "Arnet::IntfId", rule.natIntf ).shortName.lower().\
            replace( '/', '_' )
         deviceName = record[ indexDeviceIn ] if rule.target == 'destination' \
            else record[ indexDeviceOut ]
         deviceName = deviceName.replace( 'vlan', 'vl' )
         if shortName != deviceName:
            return False

         iptableAddr = record[ indexDst ] if rule.target == 'destination' \
            else record[ indexSrc ]
         iptableHost = iptableAddr.split( '/' )[ 0 ]
         return iptableHost == rule.host

      vrfRules = natStatus.vrfKernelRules.get( vrf )
      if not vrfRules:
         return model
      for r in vrfRules.natKernelDropRule:
         if not r.deny:
            continue
         dropData = [ x for x in dropOutput if _compareDevices( r, x ) and
               _compareIpv4( r.destination, x[ indexDst ] ) and
               _compareIpv4( r.source, x[ indexSrc ] ) ]

         for rec in dropData:
            counter = NatTranslationConnectionDropped.DropTable.DropCounter()
            counter.srcIp = IpAddrAndMask( ip=r.source.address,
                  mask=r.source.mask )
            counter.dstIp = IpAddrAndMask( ip=r.destination.address,
                  mask=r.destination.mask )
            counter.counter = int( rec[ indexPkts ] )
            table.counters.append( counter )

      # for each NAT host drop rule, find matching rows in the iptables output
      for r in natDropRulesStatus.dropRulesHost:
         dropData = [ x for x in dropOutput if _compareHostDropRules( r, x ) ]
         for rec in dropData:
            counter = NatTranslationConnectionDropped.DropTable.DropCounter()
            if r.target == 'source':
               counter.srcIp = IpAddrAndMask( ip=r.host, mask=32 )
               counter.dstIp = IpAddrAndMask( ip='0.0.0.0', mask=0 )
            else:
               counter.srcIp = IpAddrAndMask( ip='0.0.0.0', mask=0 )
               counter.dstIp = IpAddrAndMask( ip=r.host, mask=32 )
            counter.counter = int( rec[ indexPkts ] )
            table.counters.append( counter )

      model.vrfs[ table.vrfName ] = table

   if natHwCapabilities.natDpdkProcessingSupported:
      buff = StringIO()
      AgentCommandRequest.runSocketCommand( mode.entityManager, "Sfe",
            "sfe", "NATConnLimitCnt", timeout=30, keepalive=True, stringBuff=buff )
      counters = json.loads( buff.getvalue() )

   if not vrf:
      # Show all vrfs. Default vrf first, then the others
      if natHwCapabilities.natDpdkProcessingSupported:
         populateDpdkGlobalCounters_( model, counters )
         populateDpdkHostCounters_( DEFAULT_VRF, model, counters )
         populateDpdkPoolCounters_( DEFAULT_VRF, model, counters )
         for vrf in allVrfStatusLocal.vrf:
            populateDpdkHostCounters_( vrf, model, counters )
            populateDpdkPoolCounters_( vrf, model, counters )
      else:
         populateKernelCounters_( DEFAULT_VRF, model )
         for vrf in allVrfStatusLocal.vrf:
            populateKernelCounters_( vrf, model )

   else:
      if vrf != DEFAULT_VRF:
         vrfStatus = allVrfStatusLocal.vrf.get( vrf )
         if not vrfStatus:
            mode.addError( "VRF %s not found" % vrf )
            return
      if natHwCapabilities.natDpdkProcessingSupported:
         populateDpdkHostCounters_( vrf, model, counters )
         populateDpdkPoolCounters_( vrf, model, counters )
      else:
         populateKernelCounters_( vrf, model )

   return model

#------------------------------------------------------------------------------------
# Warning
#------------------------------------------------------------------------------------

aclUsedByNatWarningMsg =\
      "NAT configurations associated with this ACL will become inactive."

def aclDeletionWarningHook( mode, aclName, *args, **kargs ):
   # Add a warning if the acl named aclName is used by a Nat rule
   for intfname in natConfig.intfConfig :
      intfConfig = natConfig.intfConfig.get( intfname )
      for staticKey in intfConfig.staticNat :
         if staticKey.acl == aclName :
            mode.addWarning( aclUsedByNatWarningMsg )
            return None
      for dynamicNat in intfConfig.dynamicNat :
         if dynamicNat.acl == aclName :
            mode.addWarning( aclUsedByNatWarningMsg )
            return None

aclUnconfReferencesHook.addExtension( aclDeletionWarningHook )

# -----------------------------------------------------------------------------------
# show configuration consistency nat access-list
# -----------------------------------------------------------------------------------
@configConsistencyCluster( cluster='nat',
                           desc='Check potentially bad configs for NAT' )
def cfCsstNatValid( mode, context=None ):
   return ( natHwCapabilities.natConfigConsistencySupported and
            _natSupported() )

@configConsistencyCategory( cluster='nat', category='access-list',
                            desc='Show potential bad access-list configs' )
def cfCsstNatACLOverlapValid( mode, context=None ):
   return natHwCapabilities.natConfigConsistencyAclCheckSupported

@configConsistencyCheck( cluster='nat', category='access-list' )
def natACLOverlapCheck( model, mode, args ):
   if ( not AgentDirectory.agent( mode.sysname, 'Nat' ) and
        not os.environ.get( 'SIMULATION_NAT' ) ):
      # Skip check if there is no NAT config
      return

   if natHwCapabilities.natTrapEgressFilteringSupported:
      # Skip check if the platform supports trapping egress filter
      return

   def sameIntfOrProfile( acl1, acl2, profileMode ):
      set1 = set( acl1.profiles ) if profileMode else set( acl1.interfaces )
      set2 = set( acl2.profiles ) if profileMode else set( acl2.interfaces )
      return set1 == set2

   def netOverlap( n1, n2 ):
      return IPv4Network( n1 ).overlaps( IPv4Network( n2 ) )

   def aclOverlap( acl1, acl2 ):
      rules1 = [ r for r in acl1.aclRules if r.valid ]
      rules2 = [ r for r in acl2.aclRules if r.valid ]
      for r1, r2 in itertools.product( rules1, rules2 ):
         src1, dst1 = r1.description.strip(' ()').split(', ')
         src2, dst2 = r2.description.strip(' ()').split(', ')
         if netOverlap( src1, src2 ) and netOverlap( dst1, dst2 ):
            return 'fail'
      return 'ok'

   builder = ConfigConsistencyMultiTableModelBuilder( model )
   tableName = 'ACL Overlap'
   builder.addTable(
      tableName,
      headings=[ 'Access List', 'Result', 'Description' ] )

   # Generate warn/fail items in the ACL Overlap table
   # 1. Get all ACLs configured on NAT
   # 2. Iterate all ACL pairs ( ACL1, ACL2 )
   # 3. Skip if ACL1 and ACL2 are configured on the same interface or profile
   # 4. Add or patch the description of an item ( keyed by ACL1 )
   #    if ACL1 and ACL2 overlaps
   acls = cmdShowIpNatAcl( mode, args ).aclList
   profileMode = natHwCapabilities.natProfileConfigSupported
   for i in range( len( acls ) - 1 ):
      for j in range( i + 1, len( acls ) ):
         if sameIntfOrProfile( acls[ i ], acls[ j ], profileMode ):
            continue
         st = aclOverlap( acls[ i ], acls[ j ] )  # status
         if builder.worse( st, 'ok' ):
            acl1, acl2 = acls[ i ].aclName, acls[ j ].aclName
            if not builder.hasItem( tableName, acl1 ):
               builder.addItem( tableName, acl1,
                                status=st, description=f"Overlaps with { acl2 }" )
            else:
               builder.patchItem( tableName, acl1,
                                  descriptionToAppend=f", { acl2 }" )

#------------------------------------------------------------------------------------
# show configuration consistency nat hw-tables
#
# Check functions decorated with @configConsistencyCheck are in
# Platform Dependent NAT agents' CLI source code directory
#------------------------------------------------------------------------------------
@configConsistencyCategory( cluster='nat', category='hw-tables',
                            desc='Show potential bad hardware table configs' )
def cfCsstNatHwTablesValid( mode, context=None ):
   return natHwCapabilities.natConfigConsistencyHwCheckSupported

#------------------------------------------------------------------------------------
# Plugin
#------------------------------------------------------------------------------------

def Plugin( entityManager ):
   global natConfig, natStatus, natHwStatus, natHwCapabilities, aclConfig
   global vxlanNatStatus
   global natSyncConfig, natSyncStatus, allVrfStatusLocal, mcastTranslationMode
   global platformWarnings
   global smashedAdvertisedStatus
   global discoveredStatus
   global dynamicConnectionStatus, dynamicConnectionHwStatus
   global natDropRulesStatus
   global vrfMap
   global natHwFlowStatus
   global vrfIdMapStatus
   global sfeLauncherConfig
   global routingHwRouteStatus

   IraVrfCli.deletedVrfHook.addExtension( natVrfCleanup )
   IraIpIntfCli.canSetVrfHook.addExtension( vrfChangeHook )

   natConfig = ConfigMount.mount(
      entityManager, 'ip/nat/config', 'Ip::Nat::Config', 'w' )
   natStatus = LazyMount.mount(
      entityManager, 'ip/nat/status', 'Ip::Nat::Status', 'r' )
   vxlanNatStatus = LazyMount.mount(
      entityManager, 'ip/nat/vxlanStatus', 'Vxlan::VxlanNatStatus', 'r' )
   natSyncConfig = ConfigMount.mount(
         entityManager, 'ip/nat/sync/config', 'Ip::Nat::Sync::Config', 'w' )
   aclConfig = LazyMount.mount(
         entityManager, 'acl/config/cli', 'Acl::Input::Config', 'r' )
   natSyncStatus = LazyMount.mount(
         entityManager, 'ip/nat/sync/status', 'Ip::Nat::Sync::Status', 'r' )
   natHwCapabilities = LazyMount.mount(
      entityManager, 'ip/nat/hwCapabilities', 'Ip::Nat::HwCapabilities', 'r' )
   natHwStatus = LazyMount.mount(
      entityManager, 'ip/nat/hwStatus', 'Ip::Nat::HwStatus', 'r' )
   allVrfStatusLocal = LazyMount.mount(
      entityManager, Cell.path( "ip/vrf/status/local" ),
      "Ip::AllVrfStatusLocal", "r" )
   mcastTranslationMode = LazyMount.mount(
      entityManager, 'ip/nat/mcastTranslationModeConfig',
      'Ip::Nat::McastTranslationMode', 'r' )
   platformWarnings = LazyMount.mount(
      entityManager, 'ip/nat/platformWarnings',
      'Ip::Nat::PlatformWarnings', 'r' )
   natHwFlowStatus = LazyMount.mount(
      entityManager, 'ip/nat/hwFlowStatus', 'Ip::Nat::HwNatFlowStatus', 'r' )
   routingHwRouteStatus = LazyMount.mount(
      entityManager, 'routing/hardware/route/status',
      'Routing::Hardware::RouteStatus', 'r' )

   shmemEm = SharedMem.entityManager( sysdbEm=entityManager )

   smashedAdvertisedStatus = shmemEm.doMount(
      'ip/nat/sync/advertisedStatus', 'Ip::Nat::Sync::Smash::AdvertisedStatus',
      Smash.mountInfo( 'keyshadow' ) )

   dynamicConnectionStatus = shmemEm.doMount( "ip/nat/status/dynamicConnection",
                                              "Ip::Nat::DynamicConnectionStatus",
                                              Smash.mountInfo( 'shadow' ) )
   dynamicConnectionHwStatus = shmemEm.doMount( "ip/nat/hwStatus/dynamicConnection",
                                                "Ip::Nat::DynamicConnectionHwStatus",
                                                Smash.mountInfo( 'shadow' ) )

   discoveredStatus = shmemEm.doMount( "ip/nat/sync/discoveredStatus",
                                       "Ip::Nat::Sync::DiscoveredStatus",
                                       Smash.mountInfo( 'shadow' ) )

   shmemMg = shmemEm.getMountGroup()
   path = "ip/nat/status/natDropRulesStatus"
   tacType = "Ip::Nat::NatDropRulesStatus"
   sharkMountInfo = Shark.mountInfo( 'shadow' )
   natDropRulesStatus = shmemMg.doMount( path, tacType, sharkMountInfo )
   shmemMg.doClose()

   IntfCli.Intf.registerDependentClass( NatIntf )
   vrfMap = CliVrfMapper( allVrfStatusLocal, os.getuid(), os.getgid() )

   # Mount the vrfIdToName map used by Sfe. This mapping may be different from
   # the vrfIdToName map written by Ira.
   vrfIdMapStatus = shmemEm.doMount( 'vrf/hardware/vrfIdMapStatus',
         'Vrf::VrfIdMap::Status', Smash.mountInfo( 'reader' ) )

   sfeLauncherConfig = LazyMount.mount( entityManager, 'hardware/sfe/launcherConfig',
         'Tac::Dir', 'ri' )
