# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCliModes
from CliCommand import CliCommandClass
import CliPlugin.NatCli as NatCli # pylint: disable=consider-using-from-import
from CliPlugin.NatCli import (
      nodeNat,
      nodeFlow,
      matcherFlowPolicyName,
      matcherFlowMatchName,
      matcherFlowActionName,
      matcherIpPrefix,
      matcherProto,
      matcherIpGlobal,
      matcherFlowIntf,
      matcherMacAddr,
)
from CliMatcher import IntegerMatcher
from CliToken.Ip import ipMatcherForConfig

#--------------------------------------------------------------------------------
# ip nat flow
#--------------------------------------------------------------------------------
class IpNatFlowCmd( CliCommandClass ):
   syntax = 'ip nat flow'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : ipMatcherForConfig,
      'nat' : nodeNat,
      'flow' : nodeFlow
   }

   handler = NatCli.gotoIpNatFlowConfigMode
   noOrDefaultHandler = NatCli.NatFlowConfigMode.clear

BasicCliModes.GlobalConfigMode.addCommandClass( IpNatFlowCmd )

#--------------------------------------------------------------------------------
# "[no|default] policy <policy-name>" mode command
#--------------------------------------------------------------------------------
class IpNatFlowPolicyCmd( CliCommandClass ):
   syntax = 'policy FLOW_POLICY_NAME'
   noOrDefaultSyntax = 'policy [ FLOW_POLICY_NAME ] ...'
   data = {
      'policy' : 'Configure NAT flow policy',
      'FLOW_POLICY_NAME' : matcherFlowPolicyName
   }

   handler = NatCli.gotoIpNatFlowPolicyConfigMode
   noOrDefaultHandler = NatCli.deleteIpNatFlowPolicyConfig

NatCli.NatFlowConfigMode.addCommandClass( IpNatFlowPolicyCmd )

#--------------------------------------------------------------------------------
# "[no|default] match <match-name>" mode command
#--------------------------------------------------------------------------------
class IpNatFlowMatchCmd( CliCommandClass ):
   syntax = 'match MATCH_NAME'
   noOrDefaultSyntax = 'match [ MATCH_NAME ] ...'
   data = {
      'match' : 'Configure a match+action set for this policy',
      'MATCH_NAME' : matcherFlowMatchName
   }

   handler = NatCli.gotoIpNatFlowMatchConfigMode
   noOrDefaultHandler = NatCli.deleteIpNatFlowMatchConfig

NatCli.NatFlowPolicyConfigMode.addCommandClass( IpNatFlowMatchCmd )

#--------------------------------------------------------------------------------
# "[no|default] source <prefix>" command
#--------------------------------------------------------------------------------
class IpNatFlowMatchSourcePrefixCmd( CliCommandClass ):
   syntax = 'source PREFIX'
   noOrDefaultSyntax = 'source ...'
   data = {
      'source' : 'Source address prefix to match',
      'PREFIX' : matcherIpPrefix
   }

   handler = NatCli.configNatFlowMatchSourcePrefix
   noOrDefaultHandler = NatCli.deleteNatFlowMatchSourcePrefix

NatCli.NatFlowMatchConfigMode.addCommandClass( IpNatFlowMatchSourcePrefixCmd )


#--------------------------------------------------------------------------------
# "[no|default] destination <prefix>" command
#--------------------------------------------------------------------------------
class IpNatFlowMatchDestinationPrefixCmd( CliCommandClass ):
   syntax = 'destination PREFIX'
   noOrDefaultSyntax = 'destination ...'
   data = {
      'destination' : 'Destination address prefix to match',
      'PREFIX' : matcherIpPrefix
   }

   handler = NatCli.configNatFlowMatchDestinationPrefix
   noOrDefaultHandler = NatCli.deleteNatFlowMatchDestinationPrefix

NatCli.NatFlowMatchConfigMode.addCommandClass( IpNatFlowMatchDestinationPrefixCmd )

#--------------------------------------------------------------------------------
# "[no|default] protocol <protocol> <source|destination> <port> [<portRangeEnd>]"
#--------------------------------------------------------------------------------
class IpNatFlowMatchPortCmd( CliCommandClass ):
   syntax = (
      'protocol PROTOCOL '
      '[ source SRC_PORT [ UPPER_SRC_PORT ] ]'
      '[ destination DST_PORT [ UPPER_DST_PORT ] ]'
   )
   noOrDefaultSyntax = (
      'protocol [ PROTOCOL [ ( source | destination ) ] ] ...'
   )
   data = {
      'protocol' : 'L4 protocol to match on',
      'PROTOCOL' : matcherProto,
      'source' : 'Match on source port or port range',
      'destination' : 'Match on destination port or port range',
      'SRC_PORT' : IntegerMatcher( 0, 65535,
                   helpdesc="Port or lower bound of port range to match on" ),
      'UPPER_SRC_PORT' : IntegerMatcher( 0, 65535,
                         helpdesc="Upper bound of port range to match on" ),
      'DST_PORT' : IntegerMatcher( 0, 65535,
                   helpdesc="Port or lower bound of port range to match on" ),
      'UPPER_DST_PORT' : IntegerMatcher( 0, 65535,
                         helpdesc="Upper bound of port range to match on" ),
   }

   handler = NatCli.configNatFlowMatchPort
   noOrDefaultHandler = NatCli.deleteNatFlowMatchPort

NatCli.NatFlowMatchConfigMode.addCommandClass( IpNatFlowMatchPortCmd )

#--------------------------------------------------------------------------------
# "[no|default] priority <priority>" command
#--------------------------------------------------------------------------------
class IpNatFlowMatchPriorityCmd( CliCommandClass ):
   syntax = "priority PRIORITY"
   noOrDefaultSyntax = "priority ..."
   data = {
      'priority': 'Match priority',
      'PRIORITY': IntegerMatcher( 1, 65535,
         helpdesc='Priority value (lower value is higher priority; ' + \
                  'default zero is lowest priority)' )
   }

   handler = NatCli.configNatFlowMatchPriority
   noOrDefaultHandler = handler

NatCli.NatFlowMatchConfigMode.addCommandClass( IpNatFlowMatchPriorityCmd )

#--------------------------------------------------------------------------------
# "[no|default] action <action-name>" mode command
#--------------------------------------------------------------------------------
class IpNatFlowActionCmd( CliCommandClass ):
   syntax = 'action ACTION_NAME'
   noOrDefaultSyntax = 'action [ ACTION_NAME ] ...'
   data = {
      'action' : 'Configure an action for this match',
      'ACTION_NAME' : matcherFlowActionName
   }

   handler = NatCli.gotoIpNatFlowActionConfigMode
   noOrDefaultHandler = NatCli.deleteIpNatFlowActionConfig

NatCli.NatFlowMatchConfigMode.addCommandClass( IpNatFlowActionCmd )

#--------------------------------------------------------------------------------
# "[no|default] nat <source|destination> <sip|dip> <sport|dport>" command
#--------------------------------------------------------------------------------
class IpNatFlowActionNatCmd( CliCommandClass ):
   syntax = 'nat ( source | destination ) ADDRESS [ port PORT ]'
   noOrDefaultSyntax = 'nat [ ( source | destination ) ] ...'
   data = {
      'nat' : 'Set address translations for outgoing packet',
      'source' : 'Set source address of outgoing packet',
      'destination' : 'Set destination address of outgoing packet',
      'ADDRESS' : matcherIpGlobal,
      'port' : 'Set source port of outgoing packet',
      'PORT' : IntegerMatcher( 1, 65535,
                  helpdesc='Translated port or translated start of port range' )
   }

   handler = NatCli.configNatFlowActionNat
   noOrDefaultHandler = NatCli.deleteNatFlowActionNat

NatCli.NatFlowActionConfigMode.addCommandClass( IpNatFlowActionNatCmd )

#--------------------------------------------------------------------------------
# "[no|default] output interface <intf> [mac <dmac>] [vlan <vlanId>]
#--------------------------------------------------------------------------------
class IpNatFlowActionOutputCmd( CliCommandClass ):
   syntax = 'output interface INTFS [ mac MAC ] [ vlan VLANID ]'
   noOrDefaultSyntax = 'output [ interface INTFS ] ...'
   data = {
      'output' : 'Set packet next-hop properties',
      'interface' : 'Interface to send the packet on',
      'INTFS' : matcherFlowIntf,
      'mac' : 'Set destination MAC address',
      'MAC' : matcherMacAddr,
      'vlan' : 'Set VLAN ID',
      'VLANID' : IntegerMatcher( 1, 4095, helpdesc='Identifier for a VLAN' )
   }

   handler = NatCli.configNatFlowActionOutput
   noOrDefaultHandler = NatCli.deleteNatFlowActionOutput

NatCli.NatFlowActionConfigMode.addCommandClass( IpNatFlowActionOutputCmd )
