# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCliModes
from CliCommand import CliCommandClass, Node, guardedKeyword
from CliMatcher import EnumMatcher, IntegerMatcher, KeywordMatcher
from CliPlugin.IpAddrMatcher import IpAddrMatcher
import CliPlugin.NatCli as NatCli # pylint: disable=consider-using-from-import
from CliPlugin.NatCli import (
      matcherTranslation,
      matcherPoolName,
      nodeNat,
      nodeNatForHardware,
)
from CliPlugin.NatCli import (
   natIntfDynamicSupported,
   natCountersSupported,
   natFragmentCommandSupported,
   natVrfLeakCommandSupported,
   natLoggingSupported,
   natConnectionLimitSupported,
   natImmediateTcpEstablishSupported,
   natDpdkSupported,
   natRandomAddressSelectionSupported
)
from CliToken.Ip import ipMatcherForConfig
import CliToken.Hardware

matcherHost = KeywordMatcher( 'host',
                              helpdesc='Configure connection limit for hosts' )
matcherPool = KeywordMatcher( 'pool',
                              helpdesc='Configure connection limit for pool' )
matcherSymmetric = KeywordMatcher( 'symmetric',
                                   helpdesc='Limit for symmetric NAT connections' )
matcherFullCone = KeywordMatcher( 'full-cone',
                                  helpdesc='Limit for full-cone NAT connections' )
nodeMaxEntries = guardedKeyword( 'max-entries', 'Configure connection limit',
                                 natConnectionLimitSupported )

nodeTcp = guardedKeyword( 'tcp', 'Configure NAT TCP flows',
                          natImmediateTcpEstablishSupported )

#--------------------------------------------------------------------------------
# [ no | default ]
# ip nat translation
# ( tcp-timeout [ unestablished ] ) | udp-timeout | icmp-timeout TIMEOUT'
#--------------------------------------------------------------------------------
class IpNatTranslationTimeoutCmd( CliCommandClass ):
   syntax = 'ip nat translation ' \
      '( tcp-timeout [ unestablished ] ) | udp-timeout | icmp-timeout TIMEOUT'
   noOrDefaultSyntax = 'ip nat translation ' \
      '( tcp-timeout [ unestablished ] ) | udp-timeout | icmp-timeout ...'
   data = {
      'ip' : ipMatcherForConfig,
      'nat' : nodeNat,
      'translation' : matcherTranslation,
      'tcp-timeout' : guardedKeyword( 'tcp-timeout',
                                      'TCP connection inactive timeout',
                                      natIntfDynamicSupported ),
      'unestablished' : guardedKeyword( 'unestablished',
                                        'Unestablished connection timeout',
                                        natDpdkSupported ),
      'udp-timeout' : guardedKeyword( 'udp-timeout',
                                      'UDP connection inactive timeout',
                                      natIntfDynamicSupported ),
      'icmp-timeout' : guardedKeyword( 'icmp-timeout',
                                       'ICMP connection inactive timeout',
                                       natDpdkSupported ),
      'TIMEOUT' : IntegerMatcher( 0, 2 ** 32 - 1, helpdesc='Seconds' ),
   }

   handler = NatCli.cmdIpNatTranslationTimeout
   noOrDefaultHandler = NatCli.cmdNoIpNatTranslationTimeout

BasicCliModes.GlobalConfigMode.addCommandClass( IpNatTranslationTimeoutCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip nat translation counters
#--------------------------------------------------------------------------------
class IpNatTranslationCountersCmd( CliCommandClass ):
   syntax = 'ip nat translation counters'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : ipMatcherForConfig,
      'nat' : nodeNat,
      'translation' : matcherTranslation,
      'counters' : guardedKeyword(
         'counters', 'Enable counters for Static and Twice NAT connections',
         natCountersSupported ),
   }

   handler = NatCli.cmdIpNatTranslationCounters
   noOrDefaultHandler = NatCli.cmdNoIpNatTranslationCounters

BasicCliModes.GlobalConfigMode.addCommandClass( IpNatTranslationCountersCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip nat logging
#--------------------------------------------------------------------------------
class IpNatLoggingCmd( CliCommandClass ):
   syntax = 'ip nat logging'
   noOrDefaultSyntax = syntax
   data = {
         'ip' : ipMatcherForConfig,
         'nat' : nodeNat,
         'logging' : guardedKeyword( 'logging', 'Enable NAT event logging',
                                     natLoggingSupported ),
   }

   handler = NatCli.cmdIpNatLogging
   noOrDefaultHandler = NatCli.cmdNoIpNatLogging

BasicCliModes.GlobalConfigMode.addCommandClass( IpNatLoggingCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip nat translation low-mark PERCENT
#      [ host | pool | symmetric | full-cone ]
#--------------------------------------------------------------------------------
class IpNatTranslationLowMarkCmd( CliCommandClass ):
   syntax = '''ip nat translation low-mark PERCENT
               [ host | pool | symmetric | full-cone ]'''
   noOrDefaultSyntax = '''ip nat translation low-mark [ PERCENT ]
                          [ host | pool | symmetric | full-cone ]'''
   data = {
      'ip' : ipMatcherForConfig,
      'nat' : nodeNat,
      'translation' : matcherTranslation,
      'low-mark' : guardedKeyword( 'low-mark',
                                   'Connection limit low mark percentage',
                                   natConnectionLimitSupported ),
      'PERCENT' : IntegerMatcher( 1, 99, helpdesc='Low mark percentage' ),
      'host' : matcherHost,
      'pool' : matcherPool,
      'symmetric' : matcherSymmetric,
      'full-cone' : matcherFullCone,
   }

   handler = NatCli.cmdIpNatTranslationLowMark
   noOrDefaultHandler = NatCli.cmdNoIpNatTranslationLowMark

BasicCliModes.GlobalConfigMode.addCommandClass( IpNatTranslationLowMarkCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip nat translation max-entries MAXENTRIES
#    [ host | IPADDR | (pool POOLNAME) | symmetric | full-cone ]
#--------------------------------------------------------------------------------
class IpNatTranslationMaxEntriesCmd( CliCommandClass ):
   syntax = '''ip nat translation max-entries MAXENTRIES
               [ host | IPADDR | ( pool POOLNAME ) | symmetric | full-cone ]'''
   noOrDefaultSyntax = '''ip nat translation max-entries [ MAXENTRIES ]
                          [ host | IPADDR | ( pool POOLNAME ) |
                            symmetric | full-cone ]'''
   data = {
      'ip' : ipMatcherForConfig,
      'nat' : nodeNat,
      'translation' : matcherTranslation,
      'max-entries' : nodeMaxEntries,
      'MAXENTRIES' : IntegerMatcher( 0, 2**32 - 1,
                                     helpdesc='Max connection entries' ),
      'host' : matcherHost,
      'IPADDR' : IpAddrMatcher( helpdesc='Host IP address' ),
      'pool' : matcherPool,
      'POOLNAME' : matcherPoolName,
      'symmetric' : matcherSymmetric,
      'full-cone' : matcherFullCone,
   }

   handler = NatCli.cmdIpNatTranslationMaxEntries
   noOrDefaultHandler = NatCli.cmdNoIpNatTranslationMaxEntries

BasicCliModes.GlobalConfigMode.addCommandClass( IpNatTranslationMaxEntriesCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip nat kernel buffer size VALUE
#--------------------------------------------------------------------------------

class IpNatKernelBufferSizeCmd( CliCommandClass ):
   syntax = 'ip nat kernel buffer size BUFSIZE'
   noOrDefaultSyntax = 'ip nat kernel buffer size ...'
   data = {
      'ip' : ipMatcherForConfig,
      'nat' : nodeNat,
      'kernel' : 'Configure NAT kernel parameters',
      'buffer' : 'Configure NAT netlink buffer',
      'size' : 'Configure NAT netlink buffer size',
      'BUFSIZE' : IntegerMatcher( 1, 64, helpdesc='Buffer size in MB' ),
   }

   handler = NatCli.cmdNatIpKernelBufferSize
   noOrDefaultHandler = NatCli.cmdNatIpKernelBufferSize

BasicCliModes.GlobalConfigMode.addCommandClass( IpNatKernelBufferSizeCmd )

#------------------------------------------------------------------------------------
# [no|default] ip nat tcp flow creation trigger <syn | syn-ack>
#------------------------------------------------------------------------------------

class IpNatTcpFlowCreationTriggerCmd( CliCommandClass ):
   syntax = ( 'ip nat tcp flow creation trigger TRIGGER' )
   noOrDefaultSyntax = ( 'ip nat tcp flow creation trigger ...' )
   data = {
      'ip' : ipMatcherForConfig,
      'nat' : nodeNat,
      'tcp' : nodeTcp,
      'flow' : 'Configure NAT TCP flows',
      'creation' : 'Configure NAT TCP flows creation parameters',
      'trigger' : 'Select the trigger for the creation of NAT TCP flows',
      'TRIGGER' : Node( matcher=EnumMatcher( {
         'syn' : 'NAT TCP flow created after receiving SYN packet',
         'syn-ack' : 'NAT TCP flow created after receiving SYN and SYN-ACK packets'
      } ) ),
   }
   handler = NatCli.setTcpFlowCreationTrigger
   noOrDefaultHandler = NatCli.setTcpFlowCreationTrigger

BasicCliModes.GlobalConfigMode.addCommandClass( IpNatTcpFlowCreationTriggerCmd )

#------------------------------------------------------------------------------------
# [no|default] ip nat translation address selection algorithm ( hash | random )
#------------------------------------------------------------------------------------

class IpNatTranslationAlgorithmCmd( CliCommandClass ):
   syntax = ( 'ip nat translation address selection algorithm ( hash | random )' )
   noOrDefaultSyntax = ( 'ip nat translation address selection algorithm ...' )
   data = {
      'ip' : ipMatcherForConfig,
      'nat' : nodeNat,
      'translation' : matcherTranslation,
      'address' : KeywordMatcher(
         'address', helpdesc='Configure NAT address translation parameters' ),
      'selection' : KeywordMatcher(
         'selection',
         helpdesc='Configure NAT address translation source selection' ),
      'algorithm' :  guardedKeyword(
         'algorithm', 'Configure NAT address selection algorithm',
         natRandomAddressSelectionSupported ),
      'hash' : KeywordMatcher(
         'hash', helpdesc='Configure NAT address selection based on hash' ),
      'random' : KeywordMatcher(
         'random', helpdesc='Configure random NAT address selection' ),
   }
   handler = NatCli.setIpNatTranslationAlgorithm
   noOrDefaultHandler = NatCli.setIpNatTranslationAlgorithmDefault

BasicCliModes.GlobalConfigMode.addCommandClass( IpNatTranslationAlgorithmCmd )

#------------------------------------------------------------------------------------
# [no|default] ip nat translation address selection hash field source-ip
#------------------------------------------------------------------------------------

class IpNatTranslationHashCmd( CliCommandClass ):
   syntax = ( 'ip nat translation address selection hash field source-ip' )
   noOrDefaultSyntax = ( 'ip nat translation address selection hash ...' )
   data = {
      'ip' : ipMatcherForConfig,
      'nat' : nodeNat,
      'translation' : matcherTranslation,
      'address' : KeywordMatcher(
         'address', helpdesc='Configure NAT address translation parameters' ),
      'selection' : KeywordMatcher(
         'selection',
         helpdesc='Configure NAT address translation source selection' ),
      'hash' : KeywordMatcher(
         'hash', helpdesc='Configure NAT address translation hash parameters' ),
      'field' : KeywordMatcher(
         'field', helpdesc='Configure NAT address translation hash key fields' ),
      'source-ip' : KeywordMatcher(
         'source-ip', helpdesc=
         'NAT address translation hash computation based on source IP only' )
   }
   handler = NatCli.setIpNatTranslationHash
   noOrDefaultHandler = NatCli.setIpNatTranslationHashDefault

BasicCliModes.GlobalConfigMode.addCommandClass( IpNatTranslationHashCmd )

#------------------------------------------------------------------------------------
# [no|default] ip nat translation address selection any
#------------------------------------------------------------------------------------

class IpNatTranslationAddressSelectionCmd( CliCommandClass ):
   syntax = ( 'ip nat translation address selection any' )
   noOrDefaultSyntax = syntax
   data = {
      'ip' : ipMatcherForConfig,
      'nat' : nodeNat,
      'translation' : matcherTranslation,
      'address' : KeywordMatcher(
         'address', helpdesc='Configure NAT address translation parameters' ),
      'selection' : KeywordMatcher(
         'selection',
         helpdesc='Configure NAT address translation source selection' ),
      'any' : guardedKeyword(
         'any', "Try any available IP address to find a valid NAT translation",
         natDpdkSupported ),
   }
   handler = NatCli.setIpNatTranslationAddressSelection
   noOrDefaultHandler = NatCli.setIpNatTranslationAddressSelectionDefault

BasicCliModes.GlobalConfigMode.addCommandClass( IpNatTranslationAddressSelectionCmd )

#------------------------------------------------------------------------------------
# The "[no | default] hardware nat static access-list resource sharing" command,
# in "config" mode.
#------------------------------------------------------------------------------------
class HardwareNatStaticAclResourceSharingCmd( CliCommandClass ):
   syntax = "hardware nat static access-list resource sharing"
   noOrDefaultSyntax = "hardware nat static access-list resource sharing"
   data = {
      'hardware': CliToken.Hardware.hardwareForConfigMatcher,
      'nat': nodeNatForHardware,
      'static': "Static NAT configuration",
      'access-list': "Static NAT with access-list filter configuration",
      'resource': "Static NAT access-list resource configuration",
      'sharing': ( "Sharing behavior of HW resources for Static NAT with "
                   "access-list filter" ),
   }

   handler = NatCli.cmdHardwareNatStaticAclResourceSharing
   noOrDefaultHandler = NatCli.cmdNoHardwareNatStaticAclResourceSharing


BasicCliModes.GlobalConfigMode.addCommandClass(
   HardwareNatStaticAclResourceSharingCmd )

#------------------------------------------------------------------------------------
# The "hardware nat [ static | dynamic ] fragment [ disabled ]" command,
# in "config" mode.
#------------------------------------------------------------------------------------
class HardwareNatStaticDynamicFragmentDisabled( CliCommandClass ):
   syntax = "hardware nat [ static | dynamic ] fragment [ disabled ]"
   noOrDefaultSyntax = "hardware nat [ static | dynamic ] fragment ..."
   data = {
      'hardware': CliToken.Hardware.hardwareForConfigMatcher,
      'nat': nodeNatForHardware,
      'static': "Static NAT configuration",
      'dynamic': "Dynamic NAT configuration",
      'fragment': guardedKeyword(
         'fragment', "Configuring fragment entries", natFragmentCommandSupported ),
      'disabled': "Disabling fragment entries",
   }

   handler = NatCli.cmdHardwareNatFragment
   noOrDefaultHandler = NatCli.cmdNoHardwareNatFragment

BasicCliModes.GlobalConfigMode.addCommandClass(
   HardwareNatStaticDynamicFragmentDisabled )

#------------------------------------------------------------------------------------
# The "[no | default] hardware nat vrf leak" command, in "config" mode.
#------------------------------------------------------------------------------------
class HardwareNatVrfLeakCmd( CliCommandClass ):
   syntax = "hardware nat vrf leak"
   noOrDefaultSyntax = syntax
   data = {
      'hardware': CliToken.Hardware.hardwareForConfigMatcher,
      'nat': nodeNatForHardware,
      'vrf': guardedKeyword(
         'vrf', "VRF-related NAT config",
         natVrfLeakCommandSupported ),
      'leak': "Enable NAT functionality on VRF leaked routes",
   }

   handler = NatCli.cmdHardwareNatVrfLeak
   noOrDefaultHandler = NatCli.cmdNoHardwareNatVrfLeak

BasicCliModes.GlobalConfigMode.addCommandClass( HardwareNatVrfLeakCmd )
