# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from BasicCliModes import (
   ConfigModeBase,
   GlobalConfigMode,
   )
import CliCommand
from CliPlugin.IntfCli import (
   Intf,
   )
from CliPlugin.IpGenAddrMatcher import (
   IpGenAddrMatcher,
   )
from CliPlugin.MacAddr import (
   macAddrMatcher,
   )
from CliToken.Neighbor import (
   neighborMatcherForConfig,
   )
from CliMode.NeighborResolution import (
   NeighborResolutionMode,
   NeighborResolutionIntfMode,
   )
import ConfigMount
from TypeFuture import (
   TacLazyType,
   )

neighborResolutionConfig = None
IntfId = TacLazyType( 'Arnet::IntfId' )
Neighbor = TacLazyType( 'Arp::Lib::NeighborResolution::Config::Neighbor' )

class NeighborResolutionConfigMode( NeighborResolutionMode, ConfigModeBase ):
   name = 'Neighbor resolution configuration'

   def __init__( self, parent, session ):
      NeighborResolutionMode.__init__( self )
      ConfigModeBase.__init__( self, parent, session )

class NeighborResolutionIntf:
   def __init__( self, intfId ):
      self._intfId = intfId
      self._interface = neighborResolutionConfig.interface.get( intfId, None )

   @property
   def longName( self ):
      return self._intfId.stringValue

   @property
   def intfId( self ):
      return self._intfId

   @property
   def interface( self ):
      return self._interface

   def noInterface( self ):
      if self.interface:
         del neighborResolutionConfig.interface[ self.intfId ]
         self._interface = None

   def addNeighbor( self, addr, mac ):
      if not self.interface:
         self._interface = neighborResolutionConfig.newInterface( self.intfId )
      neighbor = Neighbor( addr, mac )
      if addr.af == 'ipv4':
         self.interface.addV4Neighbor( neighbor )
      elif addr.af ==  'ipv6':
         self.interface.addV6Neighbor( neighbor )
      else:
         assert False, "invalid address family"

   def delNeighbor( self, addr ):
      if not self.interface:
         # if the interface isn't already present then there are no neighbor's
         # to delete
         return
      if addr.af == 'ipv4':
         del self.interface.v4Neighbor[ addr ]
      elif addr.af == 'ipv6':
         del self.interface.v6Neighbor[ addr ]
      else:
         assert False, "invalid address family"
      if not self.interface.v4Neighbor and not self.interface.v6Neighbor:
         del neighborResolutionConfig.interface[ self.intfId ]
         self._interface = None

class NeighborResolutionIntfConfigMode( NeighborResolutionIntfMode, ConfigModeBase ):
   name = 'Neighbor resolution interface configuration'

   def __init__( self, parent, session, nrIntf ):
      self._nrIntf = nrIntf
      NeighborResolutionIntfMode.__init__( self, self.nrIntf.intfId )
      ConfigModeBase.__init__( self, parent, session )

   @property
   def nrIntf( self ):
      return self._nrIntf

class NeighborResolutionConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'neighbor resolution'
   noOrDefaultSyntax = syntax
   data = {
      'neighbor': neighborMatcherForConfig,
      'resolution': 'Manage ARP and ND configuration'
      }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( NeighborResolutionConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      childMode = mode.childMode( NeighborResolutionConfigMode )
      childMode.removeComment()
      for intfId in neighborResolutionConfig.interface:
         nrIntf = NeighborResolutionIntf( IntfId( intfId ) )
         childMode = mode.childMode( NeighborResolutionIntfConfigMode,
               nrIntf=nrIntf )
         childMode.removeComment()
      neighborResolutionConfig.interface.clear()

GlobalConfigMode.addCommandClass( NeighborResolutionConfigCmd )

class NeighborResolutionIntfConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'interface INTF'
   noOrDefaultSyntax = syntax
   data = {
      'interface': 'Configure neighbor resolution parameters on an interface',
      'INTF': Intf.matcherWithArpSupport,
   }

   @staticmethod
   def handler( mode, args ):
      nrIntf = NeighborResolutionIntf( IntfId( args[ 'INTF' ].name ) )
      childMode = mode.childMode( NeighborResolutionIntfConfigMode, nrIntf=nrIntf )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      nrIntf = NeighborResolutionIntf( IntfId( args[ 'INTF' ].name ) )
      childMode = mode.childMode( NeighborResolutionIntfConfigMode, nrIntf=nrIntf )
      childMode.removeComment()
      nrIntf.noInterface()

NeighborResolutionConfigMode.addCommandClass( NeighborResolutionIntfConfigCmd )

class NeighborResolutionNeighborConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'neighbor ADDR mac-address MAC'
   noOrDefaultSyntax = 'neighbor ADDR ...'
   data = {
      'neighbor': 'Static neighbor entry',
      'ADDR': IpGenAddrMatcher( helpdesc='Address of neighbor',
                                helpdesc4='IPv4 address of neighbor',
                                helpdesc6='IPv6 address of neighbor' ),
      'mac-address': 'MAC address of static entry',
      'MAC': macAddrMatcher,
      }

   @staticmethod
   def handler( mode, args ):
      mode.nrIntf.addNeighbor( args[ 'ADDR' ], args[ 'MAC' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if mode.nrIntf:
         mode.nrIntf.delNeighbor( args[ 'ADDR' ] )

NeighborResolutionIntfConfigMode.addCommandClass(
      NeighborResolutionNeighborConfigCmd )

def Plugin( entityManager ):
   global neighborResolutionConfig
   neighborResolutionConfig = ConfigMount.mount( entityManager,
         'arp/neighborResolution/config/cli',
         'Arp::Lib::NeighborResolution::Config::SourceRoot',
         'w' )
