# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import sys

import BasicCli
from CliMode.Netconf import MgmtNetconfMode, NetconfTransportMode
from CliPlugin import OpenConfigCliLib
import ConfigMount
from IpLibConsts import DEFAULT_VRF
import Tac

netconfConfig = None

# ------------------------------------------------------
# NETCONF config commands
# ------------------------------------------------------

class MgmtNetconfConfigMode( MgmtNetconfMode, BasicCli.ConfigModeBase ):
   """CLI configuration mode 'management api netconf'."""

   name = "NETCONF configuration"

   def __init__( self, parent, session ):
      self.config_ = netconfConfig

      MgmtNetconfMode.__init__( self, "api-netconf" )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoMgmtNetconfConfigMode( mode, args ):
   childMode = mode.childMode( MgmtNetconfConfigMode )
   mode.session_.gotoChildMode( childMode )

def noMgmtNetconfConfigMode( mode, args ):
   """Resets NETCONF configuration to default."""
   netconfConfig.enabled = False
   for name in netconfConfig.endpoints:
      noNetconfTransportConfigMode( mode, { 'TRANSPORT_NAME': name } )

class NetconfTransportConfigMode( NetconfTransportMode, BasicCli.ConfigModeBase ):
   """CLI configuration submode 'transport <name>'."""

   name = 'Transport for NETCONF'

   def __init__( self, parent, session, name ):
      self.config_ = netconfConfig
      self.name = name

      NetconfTransportMode.__init__( self, name )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoNetconfTransportConfigMode( mode, args ):
   name = args[ 'TRANSPORT_NAME' ]
   if name not in netconfConfig.endpoints:
      if OpenConfigCliLib.otherEnabledTransportExists( mode, name ):
         return

      endpoint = netconfConfig.newEndpoints( name )
      endpoint.transport = 'ssh'
      endpoint.vrfName = DEFAULT_VRF
      # since 'initially' attributes don't get attrlogged, be explicit for now
      endpoint.port = endpoint.portDefault
      endpoint.enabled = True

      OpenConfigCliLib.updateLevelEnabledFlag( netconfConfig )

   childMode = mode.childMode( NetconfTransportConfigMode, name=name )
   mode.session_.gotoChildMode( childMode )

def noNetconfTransportConfigMode( mode, args ):
   name = args[ 'TRANSPORT_NAME' ]
   endpoint = netconfConfig.endpoints.get( name )
   if endpoint is not None:
      endpoint.enabled = False
      endpoint.port = endpoint.portDefault
      del netconfConfig.endpoints[ name ]

   OpenConfigCliLib.updateLevelEnabledFlag( netconfConfig )

def shutdown( mode, args ):
   mode.config_.endpoints[ mode.name ].enabled = False

   OpenConfigCliLib.updateLevelEnabledFlag( mode.config_ )

def noShutdown( mode, args ):
   # This stanza can only be enabled if there is no other
   # stanza with same 'transport' type already enabled
   endpoint = mode.config_.endpoints.get( mode.name )
   for e in mode.config_.endpoints.values():
      if e.name != endpoint.name and e.enabled and e.transport == endpoint.transport:
         mode.addError( "transport '%s' of type '%s' already "
            "enabled; can not enable another" % ( e.name, e.transport ) )
         return
   endpoint.enabled = True
   mode.config_.enabled = True

def setVrfName( mode, args ):
   vrfName = args.get( 'VRFNAME', DEFAULT_VRF )
   mode.config_.endpoints[ mode.name ].vrfName = vrfName

def setPort( mode, args ):
   port = args.get( 'PORT', mode.config_.endpoints[ mode.name ].portDefault )
   mode.config_.endpoints[ mode.name ].port = port

# ----------------------
# netconf start-client
# ----------------------
# This is a hidden command which starts OpenConfig with -ssh_subsystem_mode
# NETCONF client is run only as a SSH subsystem and by running
# OpenConfig with -ssh_subsystem_mode flag. This is done by adding
# 'Subsystem netconf /usr/bin/OpenConfig -ssh_subsystem_mode' to the sshd_config file
# See SysMgr/SuperServerPlugin/Ssh.py for the sshd_config file modifications
#
#
# However, the above only works for root as the above command needs to run in
# bash which is root's prompt but for other users the prompt is the EOS CLI
# so for this to work for non-root users, ssh Subsystem command needs to invoke
# a EOS CLI command. Hence the need for the hidden command
# and sshd_config file addition to be 'Subsystem netconf netconf start-client'

def doNetconfClient( mode, args ):
   args = [ '/usr/bin/OpenConfig', '-ssh_subsystem_mode' ]
   try:
      Tac.run( args, stdin=sys.stdin, stdout=sys.stdout,
               stderr=sys.stderr, env=os.environ )
   except Tac.SystemCommandError as e:
      mode.addError( e.output.strip() )

def Plugin( entityManager ):
   global netconfConfig

   netconfConfig = ConfigMount.mount( entityManager, "mgmt/netconf/config",
                                   "Netconf::Config", "w" )
