# Copyright (c) 2007-2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import BasicCli
import BasicCliSession
import Cell
import CliCommand
import CliMatcher
import CliParser
# pylint: disable-next=consider-using-from-import
import CliPlugin.IpAddrMatcher as IpAddrMatcher
# pylint: disable-next=consider-using-from-import
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
import CliPlugin.VrfCli as VrfCli # pylint: disable=consider-using-from-import
import CliToken.Ip
import CliToken.Ipv6
import ConfigMount
import DscpCliLib
import HostnameCli
# pylint: disable-next=consider-using-from-import,ungrouped-imports
import CliPlugin.IntfCli as IntfCli
import LazyMount
import Logging
import ShowCommand
# pylint: disable-next=consider-using-from-import,ungrouped-imports
import CliPlugin.NetConfigModels as NetConfigModels
import Tac
import itertools
from CliMode.VrfConfig import NameServerGroupMode
from NetConfigLib import createNameServer, createVrfIpPair, getRedundantPriorityInfo
from TypeFuture import TacLazyType
from Tracing import t0
from Toggles.NetConfigToggleLib import toggleNameServerGroupEnabled

#------------------------------------------------------------------------------------
# The "hostname" command, in config mode.
#
#  hostname <hostname>
#  no hostname
#------------------------------------------------------------------------------------

netConfig = None
netGroupConfigDir = None
internalNamespace = None
netStatus = None
dscpConfig = None

af = TacLazyType( "Arnet::AddressFamily" )
configType = TacLazyType( "System::NetConfig" )

SYS_CLI_HOSTNAME_INVALID = Logging.LogHandle(
   "SYS_CLI_HOSTNAME_INVALID",
   severity=Logging.logError,
   fmt="Configured hostname \'%s\' is invalid",
   explanation="The configured hostname is invalid and has been ignored. The "
               "hostname must contain only alphanumeric characters, '.' and '-'. "
               "It must begin and end with an alphanumeric character."
               "Maximum characters in hostname is 64.",
   recommendedAction="Configure a valid hostname." )

def doSetHostname( mode, hostname="" ):
   if hostname:
      cmd = [ 'getconf', 'HOST_NAME_MAX' ]
      output = Tac.run( cmd, stdout=Tac.CAPTURE, stderr=Tac.DISCARD )
      try:
         hostnameMaxLength = int( output )
      except ValueError:
         hostnameMaxLength = 64
      if len( hostname ) > hostnameMaxLength:
         mode.addError( "Hostname cannot exceed %d characters" % hostnameMaxLength )
         return

      if '_' in hostname:
         mode.addWarning( "The hostname contains '_'. This is not a valid Internet "
                          "hostname. The system might show '%s' instead of '%s' in "
                          "certain cases." % ( hostname.replace( '_', '-' ),
                                               hostname ) )
   netConfig.hostname = hostname
   if not ( mode.session_.inConfigSession() or mode.session_.startupConfig() or
            mode.session_.isStandalone() ):
      # Give the NetworkManager agent a chance to update the hostname according to
      # the new configured value, so that the next time we print a CLI prompt it
      # includes the updated hostname.  But don't wait too long, because if the agent
      # isn't running, or if two CLI processes set the hostname simultaneously, then
      # the hostname may never be updated.
      try:
         Tac.waitFor( lambda: netStatus.hostname == ( hostname or "localhost" ),
                      timeout=5, warnAfter=None, maxDelay=0.1, sleep=True )
      except Tac.Timeout:
         # This could happen if the NetworkManager/SuperServer agent is not
         # running (like when loading the startup config), is overwhelmed,
         # or is buggy. This isn't necessarily a bad thing, but we need to
         # make sure the user isn't confused when the prompt comes back with
         # the old hostname.
         mode.addWarning(
            "Timed out waiting for the hostname to be set. The prompt may "
            "not be updated\nimmediately. Check that the SuperServer agent "
            "is running, and if the problem\ndoes not resolve itself soon, "
            "contact customer support." )

def hostnameValueFunc( mode, match ):
   hostname = match.group()
   # Support _ even if it's not valid hostname
   if not HostnameCli.validateHostname( hostname.replace( '_', '-' ) ):
      mode.addError( HostnameCli.hostnameErrorMsg )
      if mode.session.startupConfig():
         Logging.log( SYS_CLI_HOSTNAME_INVALID, hostname )
      raise CliParser.AlreadyHandledError()

   return hostname

class HostnameCmd( CliCommand.CliCommandClass ):
   syntax = "hostname HOSTNAME"
   noOrDefaultSyntax = "hostname ..."
   data = {
      'hostname': 'Configure the system hostname',
      'HOSTNAME': HostnameCli.HostnameMatcher( helpname="WORD",
                                                helpdesc="The system's hostname",
                                                value=hostnameValueFunc )
      }

   @staticmethod
   def handler( mode, args ):
      doSetHostname( mode, args[ 'HOSTNAME' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      doSetHostname( mode )

BasicCli.GlobalConfigMode.addCommandClass( HostnameCmd )

#------------------------------------------------------------------------------------
# DNS config command, in config mode.
#
#  ip name-server <server-address> [ <server-address> ... ]
#  no|default ip name-server [ <server-address> ... ]
#
#    where <server-address> is the ipv4 or ipv6 address of a name server.
#
#  Some deprecation is happening
#  ip domain-name <domainname>    ->   dns domain <domainname>
#  no|default ip domain-name      ->   no|default dns domain
#
#  ip domain-list <domainname>
#  no|default ip domain-list <domainname>
#------------------------------------------------------------------------------------

def getConfPath( mode ):
   if isinstance( mode, NameServerGroupConfigMode ):
      return netGroupConfigDir.get( mode.groupName )
   return netConfig

def _getNameServerString( nameServer ):
   if isinstance( nameServer, str ):
      return nameServer
   else:
      return nameServer.stringValue

def _numConfigNameServers( conf ):
   return len( conf.nameServer ) + len( conf.v6NameServer )

def doAddNameServer( mode, args ):
   vrfName = args.get( 'VRF', VrfCli.DEFAULT_VRF )
   conf = getConfPath( mode )
   newNsIps = itertools.chain( args.get( 'V4', [] ), args.get( 'V6', [] ) )
   priority = args.get( 'PRIORITY', 0 )
   for ipAddr in newNsIps:
      ipAddrStr = _getNameServerString( ipAddr )
      nameServer = createNameServer( ipAddrStr, vrfName, priority )
      t0( 'doAddNameServer: trying to add', ipAddrStr )
      numNs = _numConfigNameServers( conf )
      if numNs >= conf.maxNameServers:
         mode.addError(
            "Maximum number of nameservers reached. '%s' not added." %
            ipAddrStr )
         continue
      if nameServer.ipAddr.af == af.ipv4:
         t0( 'adding v4 server ', ipAddrStr )
         conf.nameServer.addMember( nameServer )
      else:
         t0( 'adding v6 server ', ipAddrStr )
         conf.v6NameServer.addMember( nameServer )
   updateDscpRules( conf )

def doRemoveNameServer( mode, args ):
   conf = getConfPath( mode )
   # `no ip name-server` and `no ip name-server default` are distinct commands,
   # the former deleting all name servers and the latter deleting ones in the
   # default VRF.
   vrfName = args.get( 'VRF', None )
   delNsIps = args.get( 'V4', [] ) + args.get( 'V6', [] )
   allNameServers = itertools.chain( conf.nameServer.values(),
                                     conf.v6NameServer.values() )
   if vrfName is not None:
      # Proceed if there is at least 1 ns in the specified VRF, spitting out an
      # error otherwise. `no ip name-server` (vrfName is None) doesn't error.
      vrfs = { nameServer.vrfName for nameServer in allNameServers }
      if vrfName not in vrfs:
         mode.addErrorAndStop(
            "No 'name-server' configuration found in VRF '%s'." % vrfName )

   if delNsIps:
      # If name servers are specified but vrfName is not, we can assume default
      # so that `ip name-server A.B.C.D` `no ip name-server A.B.C.D` works
      if vrfName is None:
         vrfName = VrfCli.DEFAULT_VRF
      for ipAddr in delNsIps:
         nameServerString = _getNameServerString( ipAddr )
         t0( 'doRemoveNameServer: removing', nameServerString )
         vrfIpPair = createVrfIpPair( nameServerString, vrfName )
         if vrfIpPair not in conf.nameServer and \
               vrfIpPair not in conf.v6NameServer:
            mode.addError( "Nameserver '%s' is not configured" %
                           nameServerString )
         del conf.nameServer[ vrfIpPair ]
         del conf.v6NameServer[ vrfIpPair ]
   else:
      # If no nameservers are specified, remove those in the specified VRF
      # or all if the vrf is not specified
      for vrfIpPair in conf.nameServer:
         if vrfName is None or vrfIpPair.vrfName == vrfName:
            del conf.nameServer[ vrfIpPair ]
      for vrfIpPair in conf.v6NameServer:
         if vrfName is None or vrfIpPair.vrfName == vrfName:
            del conf.v6NameServer[ vrfIpPair ]
   updateDscpRules( conf )

vrfExprFactory = VrfCli.VrfExprFactory(
   helpdesc='Configure a nameserver in a VRF' )

class NameServerGroupConfigMode( NameServerGroupMode, BasicCli.ConfigModeBase ):
   name = "Name-Server Group Configuration"

   def __init__( self, parent, session, groupName ):
      self.groupName = groupName
      NameServerGroupMode.__init__( self, groupName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class NameServerGroupCmd( CliCommand.CliCommandClass ):
   syntax = "ip name-server group GROUP"
   noOrDefaultSyntax = syntax
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'name-server': 'Configure the name server address',
      'group': 'Configure the group name',
      'GROUP': CliMatcher.PatternMatcher( '[a-zA-Z.0-9_]+',
               helpname='WORD',
               helpdesc='Group name' )
      }

   @staticmethod
   def handler( mode, args ):
      groupName = args[ 'GROUP' ]
      if not internalNamespace.get( groupName ):
         internalNamespace.newEntity( "Tac::Dir", groupName )
      if not netGroupConfigDir.get( groupName ):
         netGroupConfigDir.newEntity( "System::NetConfig", groupName )
      childMode = mode.childMode( NameServerGroupConfigMode, groupName=groupName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      internalNamespace.deleteEntity( args[ 'GROUP' ] )
      netGroupConfigDir.deleteEntity( args[ 'GROUP' ] )

if toggleNameServerGroupEnabled():
   BasicCli.GlobalConfigMode.addCommandClass( NameServerGroupCmd )

class NameServerSubGroupConfigCmd( CliCommand.CliCommandClass ):
   matchObj_ = object()
   syntax = "name-server [ VRF ] { V4 | V6 } [ priority PRIORITY ]"
   maxServers_ = configType.maxNameServers
   data = {
      'name-server': 'Configure the name server address',
      'VRF': vrfExprFactory,
      'V4': CliCommand.Node(
         IpAddrMatcher.IpAddrMatcher(
               f"Domain name server IPv4 address (max {maxServers_})" ),
         maxMatches=maxServers_, sharedMatchObj=matchObj_ ),
      'V6': CliCommand.Node(
         Ip6AddrMatcher.Ip6AddrMatcher(
               f"Domain name server IPv6 address (max {maxServers_})" ),
         maxMatches=maxServers_, sharedMatchObj=matchObj_ ),
      # Priority values are clamped 0-5 due to a limitation in glibc.
      # See: BUG885454
      'priority': "Domain name server query priority",
      'PRIORITY': CliMatcher.IntegerMatcher( 0, maxServers_ - 1,
         helpdesc="Priority value (lower is first)" ),
      }
   noOrDefaultSyntax = "name-server [ VRF ] [ { V4 | V6 } ] ..."

   handler = doAddNameServer
   noOrDefaultHandler = doRemoveNameServer

if toggleNameServerGroupEnabled():
   NameServerGroupConfigMode.addCommandClass( NameServerSubGroupConfigCmd )

class NameServerCmd( CliCommand.CliCommandClass ):
   matchObj_ = object()
   syntax = "ip name-server [ VRF ] { V4 | V6 } [ priority PRIORITY ]"
   maxServers_ = configType.maxNameServers
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'name-server': 'Configure the name server address',
      'VRF': vrfExprFactory,
      'V4': CliCommand.Node(
         IpAddrMatcher.IpAddrMatcher(
               f"Domain name server IPv4 address (max {maxServers_})" ),
         maxMatches=maxServers_, sharedMatchObj=matchObj_ ),
      'V6': CliCommand.Node(
         Ip6AddrMatcher.Ip6AddrMatcher(
               f"Domain name server IPv6 address (max {maxServers_})" ),
         maxMatches=maxServers_, sharedMatchObj=matchObj_ ),
      'priority': "Domain name server query priority",
      'PRIORITY': CliMatcher.IntegerMatcher( 0, maxServers_ - 1,
         helpdesc="Priority value (lower is first)" ),
      }
   noOrDefaultSyntax = "ip name-server [ VRF ] [ { V4 | V6 } ] ..."

   handler = doAddNameServer
   noOrDefaultHandler = doRemoveNameServer

BasicCli.GlobalConfigMode.addCommandClass( NameServerCmd )

domainNameHelp = 'Configure the DNS domain name'

dnsConfigKw = CliMatcher.KeywordMatcher(
   'dns',
   helpdesc='Domain Name System configuration' )

domainNameMatcher = CliMatcher.PatternMatcher( r'\S+',
                                               helpname="WORD",
                                               helpdesc="DNS domain name" )

def doSetDomainName( mode, domainName="" ):
   # no/default form leads to non-deprecated cmd
   conf = getConfPath( mode )
   conf.domainName = domainName

class DomainNameDeprecated( CliCommand.CliCommandClass ):
   syntax = "ip domain-name DOMAIN_NAME"
   noOrDefaultSyntax = "ip domain-name ..."
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'domain-name': CliCommand.Node(
         CliMatcher.KeywordMatcher( 'domain-name',
                                    helpdesc='%s (deprecated)' % domainNameHelp ),
         deprecatedByCmd='dns domain' ),
      'DOMAIN_NAME': domainNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      doSetDomainName( mode, domainName=args.get( 'DOMAIN_NAME', '' ) )

   noOrDefaultHandler = handler

# (config)# ip domain-name <domain.name>
# deprecated
BasicCli.GlobalConfigMode.addCommandClass( DomainNameDeprecated )

class DnsDomainName( CliCommand.CliCommandClass ):
   syntax = "dns domain DOMAIN_NAME"
   noOrDefaultSyntax = "dns domain ..."
   data = {
      'dns': dnsConfigKw,
      'domain': domainNameHelp,
      'DOMAIN_NAME': domainNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      doSetDomainName( mode, domainName=args.get( 'DOMAIN_NAME', '' ) )

   noOrDefaultHandler = handler

# (config)# dns domain <domain.name>
# [legacy] `ip domain-name <domain.name>`
BasicCli.GlobalConfigMode.addCommandClass( DnsDomainName )
if toggleNameServerGroupEnabled():
   NameServerGroupConfigMode.addCommandClass( DnsDomainName )

def _numDomains( config ):
   if ( config.domainName
        and config.domainName not in config.domainList.values() ):
      return len( config.domainList ) + 1
   else:
      return len( config.domainList )

def _searchLineLen( config, newDomain ):
   searchDomains = [ newDomain ]
   if ( config.domainName and config.domainName not in config.domainList.values() ):
      searchDomains.append( config.domainName )
   searchDomains.extend( config.domainList.values() )
   return len( ' '.join( searchDomains ) )

def _domainListId( config, domainName ):
   for domainId, ds in config.domainList.items():
      if domainName == ds:
         return domainId
   return None

def _domainInDomainList( config, domainName ):
   return _domainListId( config, domainName ) is not None

def doAddDomainName( mode, domainName ):
   conf = getConfPath( mode )
   if not _domainInDomainList( conf, domainName ):
      if _numDomains( conf ) >= conf.maxSearchDomains:
         mode.addError( "Maximum number of domains (%d) exceeded. Domain %s not "
                        "added." % ( conf.maxSearchDomains, domainName ) )
      elif _searchLineLen( conf, domainName ) >= conf.maxSearchChars:
         # Search path character limit seems to be exclusive, not inclusive
         mode.addError( "Maximum number of characters (%d) exceeded. Domain %s "
                        "not added." % ( conf.maxSearchChars, domainName ) )
      else:
         conf.domainList.enq( domainName )

def doRemoveDomainName( mode, domainName ):
   conf = getConfPath( mode )
   if domainName:
      domainId = _domainListId( conf, domainName )
      if domainId is not None:
         del conf.domainList[ domainId ]
   else:
      conf.domainList.clear()

class DomainList( CliCommand.CliCommandClass ):
   syntax = "ip domain-list DOMAIN_NAME"
   noOrDefaultSyntax = "ip domain-list [ DOMAIN_NAME ]"
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'domain-list': 'Configure domain names to complete unqualified host names',
      'DOMAIN_NAME': CliMatcher.PatternMatcher( r'\S+',
                                                 helpname="WORD",
                                                 helpdesc="A domain name" )
      }

   @staticmethod
   def handler( mode, args ):
      doAddDomainName( mode, args[ 'DOMAIN_NAME' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      doRemoveDomainName( mode, args.get( 'DOMAIN_NAME' ) )

BasicCli.GlobalConfigMode.addCommandClass( DomainList )
if toggleNameServerGroupEnabled():
   NameServerGroupConfigMode.addCommandClass( DomainList )

def doSetLookupSourceIntf( mode, args ):
   vrfName = args.get( 'VRF', VrfCli.DEFAULT_VRF )
   intf = args.get( 'INTF' )
   t0( 'doSetLookupSourceIntf : vrfName=", vrfName, "intf=', intf )
   if intf:
      netConfig.sourceIntf[ vrfName ] = intf.name
   else:
      del netConfig.sourceIntf[ vrfName ]

domainConfigKw = CliMatcher.KeywordMatcher( 'domain',
                                            helpdesc='Domain configuration' )

class DomainSrcIntf( CliCommand.CliCommandClass ):
   syntax = "ip domain lookup [ VRF ] source-interface INTF"
   noOrDefaultSyntax = "ip domain lookup [ VRF ] source-interface ..."
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'domain': domainConfigKw,
      'lookup': 'Lookup a domain-name ip',
      'VRF': vrfExprFactory,
      'source-interface': 'Interface name to be looked up',
      'INTF': IntfCli.Intf.matcherWithIpSupport
      }

   handler = doSetLookupSourceIntf
   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( DomainSrcIntf )

class DomainProxy( CliCommand.CliCommandClass ):
   syntax = "ip domain proxy"
   noOrDefaultSyntax = syntax
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'domain': domainConfigKw,
      'proxy': 'Enable the Domain Name Service proxy for external hosts'
      }

   @staticmethod
   def handler( mode, args ):
      netConfig.externalDnsProxy = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      netConfig.externalDnsProxy = False

BasicCli.GlobalConfigMode.addCommandClass( DomainProxy )

class DomainProxyCacheSize( CliCommand.CliCommandClass ):
   syntax = 'dns cache size CACHE_SIZE entries'
   noOrDefaultSyntax = 'dns cache size ...'
   data = {
      'dns': dnsConfigKw,
      'cache': 'DNS cache configuration',
      'size': 'Configure maximum size for the DNS cache',
      'CACHE_SIZE': CliMatcher.IntegerMatcher( 1, 4096,
            helpdesc='Maximum number of entries that the DNS cache can hold' ),
      'entries': 'Entry count'
      }

   @staticmethod
   def handler( mode, args ):
      netConfig.dnsCacheSize = args[ 'CACHE_SIZE' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      netConfig.dnsCacheSize = netConfig.defaultDnsCacheSize

BasicCli.GlobalConfigMode.addCommandClass( DomainProxyCacheSize )

def updateDscpRules( conf=netConfig ):
   dscpValue = conf.dscpValue
   if not dscpValue:
      del dscpConfig.protoConfig[ 'dns' ]
      return

   protoConfig = dscpConfig.newProtoConfig( 'dns' )
   ruleColl = protoConfig.rule
   ruleColl.clear()

   port = 53
   proto = 'udp'
   v4Vrfs = []
   for nameServer in conf.nameServer.values():
      ipAddr = nameServer.ipAddr
      vrf = nameServer.vrfName
      DscpCliLib.addDscpRule( ruleColl, ipAddr.stringValue, port,
                              False, vrf, proto, dscpValue )
      if vrf not in v4Vrfs:
         DscpCliLib.addDscpRule( ruleColl, '0.0.0.0', port,
                                 True, vrf, proto, dscpValue )
         v4Vrfs.append( vrf )
   v6Vrfs = []
   for nameServer in conf.v6NameServer.values():
      ipAddr = nameServer.ipAddr
      vrf = nameServer.vrfName
      DscpCliLib.addDscpRule( ruleColl, ipAddr.stringValue, port,
                              False, vrf, proto, dscpValue, v6=True )
      if vrf not in v6Vrfs:
         DscpCliLib.addDscpRule( ruleColl, '::', port,
                                 True, vrf, proto, dscpValue, v6=True )
         v6Vrfs.append( vrf )


def setDscp( mode, args ):
   netConfig.dscpValue = args[ 'DSCP' ]
   updateDscpRules( netConfig )

def noDscp( mode, args ):
   netConfig.dscpValue = netConfig.dscpValueDefault
   updateDscpRules( netConfig )

# [no|default] dns qos dscp <dscpValue>
DscpCliLib.addQosDscpCommandClass( BasicCli.GlobalConfigMode, setDscp, noDscp,
                                   dnsConfigKw )

#------------------------------------------------------------------------------------
# hostname config command, in config mode.
#
#  ip host <hostname> A.B.C.D [ A.B.C.D ... ]
#  no ip host [<hostname>] [ A.B.C.D ... ]
#  no ip host
#  ipv6 host <hostname> A.B.C.D [ A.B.C.D ... ]
#  no ipv6 host [<hostname>] [ A.B.C.D ... ]
#  no ipv6 host
#------------------------------------------------------------------------------------
# This function adds one or more ipv4 mappings to a host
# Previous IPv4 mappings are cleared for hostname, if any
def doSetHostnameIp( mode, hostname, ipAddressList ):
   # create a new hostname entry if it doesn't exist already
   netConfig.newHostAddr( hostname )
   for address in netConfig.hostAddr[ hostname ].ipAddr:
      if address not in ipAddressList:
         del netConfig.hostAddr[ hostname ].ipAddr[ address ]
   for address in ipAddressList:
      netConfig.hostAddr[ hostname ].ipAddr[ address ] = True

# This function adds one or more ipv6 mappings to a host
# Previous IPv6 mappings are cleared for hostname, if any
def doSetHostnameIp6( mode, hostname, ip6AddressList ):
   # create a new hostname entry if doesn't exist already
   netConfig.newHostAddr( hostname )
   for address in netConfig.hostAddr[ hostname ].ip6Addr:
      if address not in ip6AddressList:
         del netConfig.hostAddr[ hostname ].ip6Addr[ address ]
   for address in ip6AddressList:
      netConfig.hostAddr[ hostname ].ip6Addr[ address ] = True

# This function removes ipv4 mapping from netconfig
def doRemoveMapping( mode, hostname=None, ipAddressList=None, ipv6=False ):
   if hostname:
      if not hostname in netConfig.hostAddr:
         return
      ipTable = ( netConfig.hostAddr[ hostname ].ip6Addr if ipv6
                  else netConfig.hostAddr[ hostname ].ipAddr )
      ipTable2 = ( netConfig.hostAddr[ hostname ].ipAddr if ipv6
                   else netConfig.hostAddr[ hostname ].ip6Addr )

      if ipAddressList:
         # removing specified ipv4/ipv6 mappings for the host
         for mappedAddress in ipAddressList:
            del ipTable[ mappedAddress ]
      else:
         # removing all ipv4/ipv6 mappings for the host
         ipTable.clear()
      # delete a host object from netconfig which has no mapping
      if not ( ipTable or ipTable2 ):
         del netConfig.hostAddr[ hostname ]

   else:
      # removing all the ipv4/ipv6 mappings( for all hosts)  from netconfig
      for name in netConfig.hostAddr:
         ipTable = ( netConfig.hostAddr[ name ].ip6Addr if ipv6
                     else netConfig.hostAddr[ name ].ipAddr )
         ipTable2 = ( netConfig.hostAddr[ name ].ipAddr if ipv6
                      else netConfig.hostAddr[ name ].ip6Addr )

         ipTable.clear()
         # delete a host object from netconfig which has no mapping
         if not ( ipTable or ipTable2 ):
            del netConfig.hostAddr[ name ]

hostKw = CliMatcher.KeywordMatcher( 'host',
                                    helpdesc='Configure the IP addresses for host' )
hostMatcher = HostnameCli.HostnameMatcher( helpname="WORD",
                                           helpdesc="Name of the host" )

class IpHostCmd( CliCommand.CliCommandClass ):
   syntax = "ip host HOSTNAME { IP }"
   noOrDefaultSyntax = "ip host [ HOSTNAME [ { IP } ] ]"
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'host': hostKw,
      'HOSTNAME': hostMatcher,
      'IP': IpAddrMatcher.IpAddrMatcher( helpdesc="IP address of the host" )
      }

   @staticmethod
   def handler( mode, args ):
      doSetHostnameIp( mode, args[ 'HOSTNAME' ], args[ 'IP' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      doRemoveMapping( mode, args.get( 'HOSTNAME' ), args.get( 'IP' ),
                       ipv6=False )

BasicCli.GlobalConfigMode.addCommandClass( IpHostCmd )

class Ip6HostCmd( CliCommand.CliCommandClass ):
   syntax = "ipv6 host HOSTNAME { IP }"
   noOrDefaultSyntax = "ipv6 host [ HOSTNAME [ { IP } ] ]"
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
      'host': hostKw,
      'HOSTNAME': hostMatcher,
      'IP': Ip6AddrMatcher.Ip6AddrMatcher( helpdesc="IPv6 address of the host" )
      }

   @staticmethod
   def handler( mode, args ):
      doSetHostnameIp6( mode, args[ 'HOSTNAME' ], args[ 'IP' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      doRemoveMapping( mode, args.get( 'HOSTNAME' ), args.get( 'IP' ),
                       ipv6=True )

BasicCli.GlobalConfigMode.addCommandClass( Ip6HostCmd )

#-----------------------------------------------------------------------------------
# show hostname
#-----------------------------------------------------------------------------------
class ShowHostname( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hostname'
   data = {
            'hostname': 'Show the system hostname',
          }
   cliModel = NetConfigModels.Hostname

   @staticmethod
   def handler( mode, args ):
      return NetConfigModels.Hostname( hostname=netStatus.hostname,
                                    fqdn=netStatus.fqdn )

BasicCli.addShowCommandClass( ShowHostname )

#------------------------------------------------------------------------------------
# Show hostname configuration commands, in unpriv / enable mode
#
#  show hosts
#------------------------------------------------------------------------------------
def doShowHosts( mode, args ):
   hostsInfo = NetConfigModels.SysMgrHostInfo()
   hostsInfo.domainName = netConfig.domainName
   hostsInfo.domainList = list( netConfig.domainList.values() )

   allNameServers = itertools.chain( netConfig.nameServer.values(),
                                     netConfig.v6NameServer.values() )
   for nameServer in sorted( allNameServers ):
      ipAddr = nameServer.ipAddr
      if ipAddr.af == af.ipv4:
         hostsInfo.nameServers.append( ipAddr.stringValue )
      nameServerConf = NetConfigModels.NameServerConfig(
            ipAddr=ipAddr, vrf=nameServer.vrfName, priority=nameServer.priority )
      hostsInfo.nameServerConfigs.append( nameServerConf )

   for host in netStatus.hostAddr:
      hostsInfo.hosts[ host ] = NetConfigModels.SysMgrHostInfo.IpAddresses()

      addresses = netStatus.hostAddr[ host ].ipAddr
      hostsInfo.hosts[ host ].ipv4Addresses.extend( addresses )

      addresses = netStatus.hostAddr[ host ].ip6Addr
      hostsInfo.hosts[ host ].ipv6Addresses.extend( addresses )

   return hostsInfo

class ShowHosts( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hosts'
   data = {
            'hosts': 'IP domain-name, lookup style, nameservers and host table',
          }
   cliModel = NetConfigModels.SysMgrHostInfo
   handler = doShowHosts

BasicCli.addShowCommandClass( ShowHosts )

#------------------------------------------------------------------------------------
# Show DNS configuration commands, in unpriv / enable mode
#
#  Deprecated command, but still valid
#  show ip domain-name    ->   show dns domain
#  show ip name-server
#------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------
# show ip domain-name
#-----------------------------------------------------------------------------------
showDomainNameHelp = 'Show the system domain name'

def doShowDomainName( mode, args ):
   groupName = args.get( 'GROUP' )
   if groupName:
      conf = netGroupConfigDir.get( groupName )
   else:
      conf = netConfig
   if conf:
      return NetConfigModels.DomainName( domainName=conf.domainName )
   return NetConfigModels.DomainName()

class ShowIpDomainName( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip domain-name'
   data = {
            'ip': 'Details related to IPv4',
            'domain-name': CliCommand.Node(
                              matcher=CliMatcher.KeywordMatcher( 'domain-name',
                                 helpdesc='%s (deprecated)' % showDomainNameHelp ),
                              deprecatedByCmd='show dns domain' )
          }
   cliModel = NetConfigModels.DomainName
   handler = doShowDomainName

BasicCli.addShowCommandClass( ShowIpDomainName )

#-----------------------------------------------------------------------------------
# show dns domain
#-----------------------------------------------------------------------------------
class ShowDnsDomain( ShowCommand.ShowCliCommandClass ):
   if toggleNameServerGroupEnabled():
      syntax = 'show dns domain [ group GROUP ]'
   else:
      syntax = 'show dns domain'
   data = {
            'dns': 'Domain Name System configuration',
            'domain': showDomainNameHelp,
            'group': 'Configure the group name',
            'GROUP': CliMatcher.PatternMatcher( '[a-zA-Z.0-9_]+',
                     helpname='WORD',
                     helpdesc='Group name' )
          }
   cliModel = NetConfigModels.DomainName
   handler = doShowDomainName

BasicCli.addShowCommandClass( ShowDnsDomain )

#-----------------------------------------------------------------------------------
# show ip name-server
#-----------------------------------------------------------------------------------
def doShowNameServer( mode, args ):
   groupName = args.get( 'GROUP' )
   conf = None
   if groupName:
      conf = netGroupConfigDir.get( groupName )
   else:
      conf = netConfig
   if not conf:
      return NetConfigModels.SysMgrNameServerInfo()

   allNameServers = list( itertools.chain( conf.nameServer.values(),
                                           conf.v6NameServer.values() ) )

   excessPriorities = None
   priorities = list( nameServer.priority for nameServer in allNameServers )
   largestValid, firstRedundant = getRedundantPriorityInfo( priorities )
   if largestValid and firstRedundant:
      excessPriorities = NetConfigModels.NameServerExcessPriorities(
            largestValidPriorityValue=largestValid,
            firstRedundantPriorityValue=firstRedundant )

   nameServerInfo = NetConfigModels.SysMgrNameServerInfo(
         _excessPriorities=excessPriorities )

   for nameServer in sorted( allNameServers ):
      ipAddr = nameServer.ipAddr
      if ipAddr.af == af.ipv4:
         nameServerInfo.v4NameServers.append( ipAddr.stringValue )
      else:
         nameServerInfo.v6NameServers.append( ipAddr.stringValue )
      nameServerConf = NetConfigModels.NameServerConfig(
            ipAddr=ipAddr, vrf=nameServer.vrfName, priority=nameServer.priority )
      nameServerInfo.nameServerConfigs.append( nameServerConf )

   return nameServerInfo

class ShowIpNameServer( ShowCommand.ShowCliCommandClass ):
   if toggleNameServerGroupEnabled():
      syntax = 'show ip name-server [ group GROUP ]'
   else:
      syntax = 'show ip name-server'
   data = {
            'ip': CliToken.Ip.ipMatcherForShow,
            'name-server': 'Show name-server configuration',
            'group': 'Configure the group name',
            'GROUP': CliMatcher.PatternMatcher( '[a-zA-Z.0-9_]+',
                     helpname='WORD',
                     helpdesc='Group name' )
          }
   cliModel = NetConfigModels.SysMgrNameServerInfo
   handler = doShowNameServer

BasicCli.addShowCommandClass( ShowIpNameServer )

# -----------------------------------------------------------------------------------
# show dns cache counters
# -----------------------------------------------------------------------------------
def doShowDnsCacheCounters( mode, args ):
   return NetConfigModels.DnsCacheCountersModel(
         size=netStatus.dnsCacheCounters.size,
         insertions=netStatus.dnsCacheCounters.insertions,
         evictions=netStatus.dnsCacheCounters.evictions,
         misses=netStatus.dnsCacheCounters.misses,
         hits=netStatus.dnsCacheCounters.hits )

class ShowDnsCacheCounters( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dns cache counters'
   data = {
            'dns': 'Domain Name System configuration',
            'cache': 'DNS cache',
            'counters': 'DNS cache counters',
          }
   cliModel = NetConfigModels.DnsCacheCountersModel
   handler = doShowDnsCacheCounters

BasicCli.addShowCommandClass( ShowDnsCacheCounters )

def _hostnameHandler():
   return netStatus.hostname

def Plugin( entityManager ):
   global netConfig
   global netGroupConfigDir
   global internalNamespace
   global netStatus
   global dscpConfig

   netConfig = ConfigMount.mount( entityManager,
                                  "sys/net/config", "System::NetConfig", "w" )
   netGroupConfigDir = ConfigMount.mount( entityManager,
                             "sys/net/groupConfigDir", "Tac::Dir", "w" )
   internalNamespace = ConfigMount.mount( entityManager,
                                        "sys/net/internalNamespace",
                                        "Tac::Dir", "w" )
   netStatus = LazyMount.mount( entityManager, Cell.path( "sys/net/status" ),
                                "System::NetStatus", "r" )
   dscpConfig = ConfigMount.mount( entityManager, "mgmt/dscp/config",
                                   "Mgmt::Dscp::Config", "w" )
   BasicCliSession.registerHostnameHandler( _hostnameHandler )
