#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Arnet import IpGenAddr
import BasicCliModes
import CliCommand
import CliMatcher
import CliParser
import CliPlugin.AclCli as AclCli # pylint: disable=consider-using-from-import
import CliPlugin.ClockCli as ClockCli # pylint: disable=consider-using-from-import
import CliPlugin.IntfCli as IntfCli # pylint: disable=consider-using-from-import
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
import CliPlugin.VrfCli as VrfCli # pylint: disable=consider-using-from-import
import CliPlugin.Ntp as Ntp # pylint: disable=consider-using-from-import
from CliPlugin.VrfCli import VrfExprFactory
import DscpCliLib
import HostnameCli
import MultiRangeRule
import ReversibleSecretCli
import Tac
from Toggles.NtpToggleLib import toggleNtpDnsRefreshEnabled

Config = Tac.Type( 'Ntp::Config' )
localIntfHelp = 'Configure the interface from which the IP source address is taken'
matcherNtp = CliMatcher.KeywordMatcher( 'ntp',
      helpdesc='Configure NTP' )
vrfExprFactory = VrfCli.VrfExprFactory(
      helpdesc='Use a specific VRF' )

#--------------------------------------------------------------------------------
# [ no | default ] ntp authenticate [servers]
#--------------------------------------------------------------------------------
class NtpAuthenticateCmd( CliCommand.CliCommandClass ):
   syntax = 'ntp authenticate [ servers ]'
   noOrDefaultSyntax = 'ntp authenticate ...'
   data = {
      'ntp': matcherNtp,
      'authenticate': 'Require authentication for NTP synchronization',
      'servers': 'Authentication required only for incoming NTP server responses',
   }
   handler = Ntp.doConfigAuth
   noOrDefaultHandler = Ntp.doDisableAuth

BasicCliModes.GlobalConfigMode.addCommandClass( NtpAuthenticateCmd )

#--------------------------------------------------------------------------------
# ( no | default ) ntp bind [ VRF ] INTFS
#--------------------------------------------------------------------------------
class NtpBindCmd( CliCommand.CliCommandClass ):
   syntax = 'ntp bind [ VRF ] INTFS'
   noOrDefaultSyntax = 'ntp bind [ VRF ] [ INTFS ] ...'
   data = {
      'ntp': matcherNtp,
      'bind': 'Configure the interfaces for NTP to listen on',
      'VRF': vrfExprFactory,
      'INTFS': IntfCli.Intf.rangeMatcherWithIpSupport
   }
   hidden = True
   handler = Ntp.doAddBoundIntfs
   noOrDefaultHandler = Ntp.doRemoveBoundIntfs

BasicCliModes.GlobalConfigMode.addCommandClass( NtpBindCmd )

#--------------------------------------------------------------------------------
# ( no | default ) ntp
#--------------------------------------------------------------------------------
class NtpCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'ntp'
   data = {
      'ntp': matcherNtp,
   }
   noOrDefaultHandler = Ntp.doDisableNtp

BasicCliModes.GlobalConfigMode.addCommandClass( NtpCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ntp force-restarts
#--------------------------------------------------------------------------------
class NtpForceRestartsCmd( CliCommand.CliCommandClass ):
   syntax = 'ntp force-restarts'
   noOrDefaultSyntax = 'ntp force-restarts ...'
   data = {
      'ntp': matcherNtp,
      'force-restarts': 'Force restarts on interface changes',
   }
   hidden = True
   handler = Ntp.enableRestarts
   noOrDefaultHandler = Ntp.disableRestarts

BasicCliModes.GlobalConfigMode.addCommandClass( NtpForceRestartsCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ntp local-interface [ VRF ] INTF
#--------------------------------------------------------------------------------
class NtpLocalInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'ntp ( source | local-interface ) [ VRF ] INTF'
   noOrDefaultSyntax = 'ntp ( source | local-interface ) ...'
   data = {
      'ntp': matcherNtp,
      'local-interface': localIntfHelp,
      'source': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'source',
            # pylint: disable-next=consider-using-f-string
            helpdesc='%s (deprecated)' % localIntfHelp ),
         deprecatedByCmd='ntp local-interface' ),
      'VRF': vrfExprFactory,
      'INTF': IntfCli.Intf.matcherWithIpSupport,
   }
   handler = Ntp.doConfigDefaultSourceIntf
   noOrDefaultHandler = Ntp.doRemoveDefaultSourceIntf

BasicCliModes.GlobalConfigMode.addCommandClass( NtpLocalInterfaceCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ntp qos dscp DSCP
#--------------------------------------------------------------------------------
DscpCliLib.addQosDscpCommandClass( BasicCliModes.GlobalConfigMode, Ntp.setDscp,
                                   Ntp.noDscp, tokenProto=matcherNtp )

#--------------------------------------------------------------------------------
# [ no | default ] ntp serve all [ vrf VRF ]
#--------------------------------------------------------------------------------
class NtpServeAllCmd( CliCommand.CliCommandClass ):

   syntax = 'ntp serve all [ VRF ]'
   noOrDefaultSyntax = 'ntp serve all [ VRF ] ...'
   data = {
      'ntp': matcherNtp,
      'serve': 'Configure the switch as an NTP server',
      'all': 'Service NTP requests received on any interface',
      'VRF': VrfExprFactory( helpdesc='Service NTP requests received on '
                                      'any interface in specified VRF' ),
   }
   handler = Ntp.doNtpServeAll
   noOrDefaultHandler = Ntp.doRemoveNtpServeAll

BasicCliModes.GlobalConfigMode.addCommandClass( NtpServeAllCmd )

#--------------------------------------------------------------------------------
# ( no | default ) ntp serve ip access-group IP_ACL [ vrf VRF ] in
#--------------------------------------------------------------------------------
class NtpServeIpAccessGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'ntp serve ip access-group IP_ACL [ VRF ] in'
   noOrDefaultSyntax = 'ntp serve ip access-group [ IP_ACL ] [ VRF ] ...'
   data = {
      'ntp': matcherNtp,
      'serve': 'Configure the switch as an NTP server',
      'ip': AclCli.ipKwForServiceAclMatcher,
      'IP_ACL': AclCli.standardIpAclNameMatcher,
      'access-group': AclCli.accessGroupKwMatcher,
      'VRF': AclCli.vrfExprFactoryForConfigAcl,
      'in': AclCli.inKwMatcher,
   }
   handler = Ntp.setNtpIpAcl
   noOrDefaultHandler = Ntp.noNtpIpAcl

BasicCliModes.GlobalConfigMode.addCommandClass( NtpServeIpAccessGroupCmd )

#--------------------------------------------------------------------------------
# ( no | default ) ntp serve ipv6 access-group IP6_ACL [ vrf VRF ] in
#--------------------------------------------------------------------------------
class NtpServeIpv6AccessGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'ntp serve ipv6 access-group IP6_ACL [ VRF ] in'
   noOrDefaultSyntax = 'ntp serve ipv6 access-group [ IP6_ACL ] [ VRF ] ...'
   data = {
      'ntp': matcherNtp,
      'serve': 'Configure the switch as an NTP server',
      'ipv6': AclCli.ipv6KwMatcherForServiceAcl,
      'IP6_ACL': AclCli.standardIp6AclNameMatcher,
      'access-group': AclCli.accessGroupKwMatcher,
      'VRF': AclCli.vrfExprFactoryForConfigAcl,
      'in': AclCli.inKwMatcher,
   }
   handler = Ntp.setNtpIpAcl
   noOrDefaultHandler = Ntp.noNtpIpAcl

BasicCliModes.GlobalConfigMode.addCommandClass( NtpServeIpv6AccessGroupCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ntp server [ vrf VRF ] HOST OPTIONS
#--------------------------------------------------------------------------------

NtpServerHostNode = CliCommand.Node(
   matcher=HostnameCli.IpAddrOrHostnameMatcher( ipv6=True ),
   storeSharedResult=True )

class NtpServerSourceAddrMatcher( IpGenAddrMatcher ):
   '''Matches an IPv4 or IPv6 address, or either, depending on the address family
   used to specify the host.
   '''
   def _hostAddressFamily( self, mode, context ):
      hostMatch = context.sharedResult[ 'HOST' ]
      host = HostnameCli.ipAddrOrHostnameValueFunc( mode, hostMatch )
      try:
         addr = IpGenAddr( host )
         return addr.af
      except ( IndexError, ValueError ):
         pass
      return 'ipunknown'

   def match( self, mode, context, token ):
      mr = super().match( mode, context, token )
      if mr == CliParser.noMatch:
         return mr

      hostAf = self._hostAddressFamily( mode, context )
      if hostAf not in ( 'ipv4', 'ipv6' ):
         # Hostname ==> don't constrain match
         return mr

      addr = mr.result
      if addr.af == hostAf:
         return mr
      else:
         # Returning noMatch here would not get the right behavior -- the parse tree
         # would try to use completions() to auto-complete.
         # But this error works (IpGenAddrMatcher uses it in a similar way).
         raise CliParser.InvalidInputError()

   def completions( self, mode, context, token ):
      genComps = super().completions( mode, context, token )
      hostAf = self._hostAddressFamily( mode, context )
      if hostAf == 'ipv4':
         return [ c for c in genComps if 'IPv4' in c.help ]
      if hostAf == 'ipv6':
         return [ c for c in genComps if 'IPv6' in c.help ]
      # Unconstrained match
      return genComps

sharedMatchObj = object()
class NtpServerSourceExpr( CliCommand.CliExpression ):
   expression = ( '( ( source | local-interface INTF ) | '
                   ' ( source-address ADDR ) )' )
   data = {
      'source': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'source', helpdesc=localIntfHelp ),
         maxMatches=1, sharedMatchObj=sharedMatchObj ),
      'local-interface': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'local-interface',
            helpdesc=localIntfHelp ),
         maxMatches=1, sharedMatchObj=sharedMatchObj ),
      'INTF': IntfCli.Intf.matcherWithIpSupport,
      'source-address': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'source-address',
                                            helpdesc="IPv4 or IPv6 source address" ),
         maxMatches=1, sharedMatchObj=sharedMatchObj ),
      'ADDR': NtpServerSourceAddrMatcher( 'IPv4 or IPv6 Address',
                                          helpdesc4='IPv4 address',
                                          helpdesc6='IPv6 address' ),
   }

class NtpServerOptionsExpr( CliCommand.CliExpression ):
   if toggleNtpDnsRefreshEnabled():
      expression = ( '{ ( version VERSION ) | prefer | refresh | '
                   ' ( minpoll MIN_POLL ) | ( maxpoll MAX_POLL ) | '
                   ' burst | iburst | ( key KEY_ID ) | '
                   ' SOURCE }' )
   else:
      expression = ( '{ ( version VERSION ) | prefer | '
               ' ( minpoll MIN_POLL ) | ( maxpoll MAX_POLL ) | '
               ' burst | iburst | ( key KEY_ID ) | '
               ' SOURCE }' )
   data = {
      'version': CliCommand.singleKeyword( 'version',
         helpdesc='Configure the NTP version' ),
      'VERSION': CliMatcher.IntegerMatcher( 1, 4,
         helpdesc='Version of the NTP protocol' ),
      'prefer': CliCommand.singleKeyword( 'prefer',
         helpdesc='Mark this server as preferred' ),
      'refresh': CliCommand.singleKeyword( 'refresh',
         helpdesc='Periodically refresh DNS resolution' ),
      'minpoll': CliCommand.singleKeyword( 'minpoll',
         helpdesc='Minimum poll interval' ),
      'MIN_POLL': CliMatcher.IntegerMatcher( 3, 17,
         helpdesc='Base-2 logarithm of minimum poll interval' ),
      'maxpoll': CliCommand.singleKeyword( 'maxpoll',
         helpdesc='Maximum poll interval' ),
      'MAX_POLL': CliMatcher.IntegerMatcher( 3, 17,
         helpdesc='Base-2 logarithm of maximum poll interval' ),
      'burst': CliCommand.singleKeyword( 'burst',
         helpdesc='Send a burst of packets instead of the usual one' ),
      'iburst': CliCommand.singleKeyword( 'iburst',
         helpdesc='Send bursts of packets until the server is reached' ),
      'key': CliCommand.singleKeyword( 'key',
         helpdesc='Set a key to use for authentication' ),
      'KEY_ID': CliMatcher.IntegerMatcher( 1, 65535, helpdesc='Key identifier' ),
      'SOURCE': NtpServerSourceExpr,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      options = {}
      if args.pop( 'key', None ):
         options[ 'keyId' ] = args.pop( 'KEY_ID' )[ 0 ]
      for i in ( ( 'version', 'VERSION' ), ( 'minpoll', 'MIN_POLL' ),
                 ( 'maxpoll', 'MAX_POLL' ) ):
         if args.pop( i[ 0 ], None ):
            options[ i[ 0 ] ] = args.pop( i[ 1 ] )[ 0 ]
      for i in ( 'prefer', 'refresh', 'burst', 'iburst' ):
         if args.pop( i, None ):
            options[ i ] = True
      if 'INTF' in args:
         options[ 'sourceIntf' ] = args.pop( 'INTF' )[ 0 ].name
         options[ 'sourceUsed' ] = 'source' in args
      if 'ADDR' in args:
         options[ 'sourceAddr' ] = args.pop( 'ADDR' )[ 0 ]

      args[ 'OPTIONS' ] = options

class NtpServerCmd( CliCommand.CliCommandClass ):
   syntax = 'ntp server [ VRF ] HOST [ OPTIONS ]'
   noOrDefaultSyntax = 'ntp server [ VRF ] HOST ...'
   data = {
      'ntp': matcherNtp,
      'server': 'Configure NTP server to synchronize to',
      'VRF': vrfExprFactory,
      'HOST': NtpServerHostNode,
      'OPTIONS': NtpServerOptionsExpr
   }
   handler = Ntp.doConfigNtpServer
   noOrDefaultHandler = Ntp.doDeleteNtpServer

BasicCliModes.GlobalConfigMode.addCommandClass( NtpServerCmd )

#--------------------------------------------------------------------------------
# [ ( no | default ) ] ntp trusted-key KEY_RANGE
#--------------------------------------------------------------------------------
class NtpTrustedKeyCmd( CliCommand.CliCommandClass ):
   syntax = 'ntp trusted-key KEY_RANGE'
   noOrDefaultSyntax = 'ntp trusted-key ...'
   data = {
      'ntp': matcherNtp,
      'trusted-key': ( 'Configure the set of keys that are accepted for incoming '
                       'messages' ),
      'KEY_RANGE': MultiRangeRule.MultiRangeMatcher( lambda: ( 1, 65535 ),
         noSingletons=False, helpdesc='Key identifier(s)' )
   }
   handler = Ntp.doConfigTrustedKey
   noOrDefaultHandler = Ntp.doNoTrustedKey

BasicCliModes.GlobalConfigMode.addCommandClass( NtpTrustedKeyCmd )

#--------------------------------------------------------------------------------
# [ ( no | default ) ] ntp authentication-key KEY_RANGE
#--------------------------------------------------------------------------------
class AuthKeyCommand( CliCommand.CliCommandClass ):
   syntax = 'ntp authentication-key KEY_ID md5 | sha1 KEY'
   noOrDefaultSyntax = 'ntp authentication-key KEY_ID ...'
   data = {
      'ntp': matcherNtp,
      'authentication-key': 'Define a key to use for authentication',
      'KEY_ID': CliMatcher.IntegerMatcher( 1, 65535, helpdesc='Key identifier' ),
      'md5': 'MD5 hash algorithm',
      'sha1': 'SHA-1 hash algorithm',
      'KEY': ReversibleSecretCli.defaultReversiblePwdCliExpr,
   }
   handler = Ntp.doConfigAuthenKey
   noOrDefaultHandler = Ntp.doDeleteAuthenKey

BasicCliModes.GlobalConfigMode.addCommandClass( AuthKeyCommand )

# --------------------------------------------------------------------------------
# [ ( no | default ) ] ntp local [ stratum STRATUM ]
# --------------------------------------------------------------------------------
class NtpLocalTimeReferenceCommand( CliCommand.CliCommandClass ):
   syntax = 'ntp local [ stratum STRATUM ]'
   noOrDefaultSyntax = 'ntp local ...'
   data = {
      'ntp': matcherNtp,
      'local': 'Local reference mode (can serve time without an upstream server)',
      'stratum': 'Stratum reported to clients',
      'STRATUM': CliMatcher.IntegerMatcher(
         Config.localStratumEnabledMin, Config.localStratumEnabledMax,
         helpdesc=f'Stratum (default {Config.localStratumEnabledDefault})'
      )
   }
   handler = Ntp.doEnableLocalReferenceMode
   noOrDefaultHandler = Ntp.doDisableLocalReferenceMode

BasicCliModes.GlobalConfigMode.addCommandClass( NtpLocalTimeReferenceCommand )

#--------------------------------------------------------------------------------
# clock source ntp
#--------------------------------------------------------------------------------
class ClockSourceNtpCommand( CliCommand.CliCommandClass ):
   syntax = 'clock source ntp'
   data = {
      'clock' : ClockCli.configClockMatcher,
      'source' : ClockCli.sourceMatcher,
      'ntp' : 'Network Time Protocol (default)',
   }
   handler = lambda mode, args: ClockCli.doSetClockSource( mode, 'ntp' )

BasicCliModes.GlobalConfigMode.addCommandClass( ClockSourceNtpCommand )
