#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
from CliMode.OcspProfile import OcspProfileMode
from CliPlugin import Security
from CliPlugin import Ssl
from CliPlugin import VrfCli
from Toggles import MgmtSecurityToggleLib

ocspChainCertRequirementMatcher = CliMatcher.EnumMatcher( {
   'all': 'Perform OCSP verification on the whole chain',
   'leaf': 'Only perform OCSP verification on the leaf',
   'none': 'Only perform OCSP verfication on certs which have a responder specified'
} )
ocspNonceMatcher = CliMatcher.EnumMatcher( {
   'disabled': 'Don\'t send a nonce in the OCSP request',
   'response': 'Require the OCSP nonce to be in the OCSP response'
} )
ocspUrlMatcher = CliMatcher.PatternMatcher(
   pattern=r'http://[^\s]+',
   helpname='URL',
   helpdesc='The HTTP URL of the overriding OCSP responder'
)

#----------------------------------------------------------------
# OcspProfileConfigMode
#----------------------------------------------------------------
class OcspProfileConfigMode( OcspProfileMode, BasicCli.ConfigModeBase ):
   name = "OCSP Profile Configuration"

   def __init__( self, parent, session, ocspProfileConfig ):
      OcspProfileMode.__init__( self, ocspProfileConfig.name )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.ocspProfileConfig_ = ocspProfileConfig

   def config( self ):
      return self.ocspProfileConfig_

#-----------------------------------------------------------------------------------
# [ no | default ] ocsp profile OCSP_PROFILE
#-----------------------------------------------------------------------------------
class GotoOcspProfileModeCmd( CliCommand.CliCommandClass ):
   syntax = 'ocsp profile OCSP_PROFILE'
   noOrDefaultSyntax = 'ocsp profile OCSP_PROFILE'
   data = {
      'ocsp': 'Configure OCSP',
      'profile': 'Configure OCSP profile',
      'OCSP_PROFILE': Ssl.ocspProfileNameMatcher
   }
   handler = "OcspProfileHandler.GotoOcspProfileModeCmd_handler"
   noOrDefaultHandler = (
      "OcspProfileHandler.GotoOcspProfileModeCmd_noOrDefaultHandler" )

Security.SecurityConfigMode.addCommandClass( GotoOcspProfileModeCmd )

#--------------------------------------------------
# [no|default] chain certificate requirement responder [ all | leaf | none ]
#--------------------------------------------------
class ChainCertRequirementResponderCmd( CliCommand.CliCommandClass ):
   syntax = 'chain certificate requirement responder SETTING'
   noOrDefaultSyntax = 'chain certificate requirement responder...'
   data = {
            'chain': Ssl.chainKwMatcher,
            'certificate': 'Certificate',
            'requirement': Ssl.requirementKwMatcher,
            'responder': 'Configure requirements for querying OCSP responder',
            'SETTING': ocspChainCertRequirementMatcher
          }
   handler = "OcspProfileHandler.OcspCertRequirementCmd_handler"
   noOrDefaultHandler = handler

OcspProfileConfigMode.addCommandClass( ChainCertRequirementResponderCmd )

#-----------------------------------------------------------------------------------
# [ no | default ] extension nonce request [ disabled | response ]
#-----------------------------------------------------------------------------------
class OcspNonceCmd( CliCommand.CliCommandClass ):
   syntax = 'extension nonce request [ SETTING ]'
   noOrDefaultSyntax = 'extension nonce request ...'
   data = {
      'extension': 'Configure OCSP extensions',
      'nonce': 'Configure the OCSP nonce settings',
      'request': 'Configure including a nonce in OCSP requests',
      'SETTING': ocspNonceMatcher,
   }
   handler = "OcspProfileHandler.OcspNonceCmd_handler"
   noOrDefaultHandler = handler

OcspProfileConfigMode.addCommandClass( OcspNonceCmd )

#-----------------------------------------------------------------------------------
# [ no | default ] timeout SECONDS
#-----------------------------------------------------------------------------------
class OcspTimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'timeout SECONDS'
   noOrDefaultSyntax = 'timeout ...'
   data = {
      'timeout': 'Configure OCSP queries timeout',
      'SECONDS': CliMatcher.IntegerMatcher(
         1, 600,
         helpdesc='OCSP queries timeout in seconds' ),
   }
   handler = "OcspProfileHandler.OcspTimeoutCmd_handler"
   noOrDefaultHandler = handler

OcspProfileConfigMode.addCommandClass( OcspTimeoutCmd )

#-----------------------------------------------------------------------------------
# [ no | default ] url URL
#-----------------------------------------------------------------------------------
class OcspUrlCmd( CliCommand.CliCommandClass ):
   syntax = 'url URL'
   noOrDefaultSyntax = 'url ...'
   data = {
      'url': 'Configure the overriding OCSP responder\'s URL',
      'URL': ocspUrlMatcher
   }
   handler = "OcspProfileHandler.OcspUrlCmd_handler"
   noOrDefaultHandler = handler

OcspProfileConfigMode.addCommandClass( OcspUrlCmd )

#-----------------------------------------------------------------------------------
# [ no | default ] vrf VRF
#-----------------------------------------------------------------------------------
class OcspVrfCmd( CliCommand.CliCommandClass ):
   syntax = 'vrf VRF'
   noOrDefaultSyntax = 'vrf ...'
   data = {
      'vrf': 'Configure the VRF which the OCSP queries should be perfomred in',
      'VRF': CliMatcher.DynamicNameMatcher( VrfCli.getVrfNames, 'VRF name' )
   }
   handler = "OcspProfileHandler.OcspVrfCmd_handler"
   noOrDefaultHandler = handler

if MgmtSecurityToggleLib.toggleOcspProfileVrfEnabled():
   OcspProfileConfigMode.addCommandClass( OcspVrfCmd )
