# Copyright (c) 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements IP DHCP Relay configuration.
# It contains both interface-specific and non-interface-specific commands.
#-------------------------------------------------------------------------------

import Tac
import CliCommand, CliMatcher, BasicCliModes
import ConfigMount
from CliToken.Ip import ipMatcherForConfig
from CliToken.Ip import ipMatcherForConfigIf
from CliToken.Dhcp import dhcpMatcherForConfig
from CliPlugin import IntfCli
from CliPlugin import IraIpIntfCli
from CliPlugin.DhcpRelayHelperCli import circuitIdValueMatcher
from CliPlugin.IpAddrMatcher import IpAddrMatcher

dhcpRelayConfig = None

matcherRelay = CliMatcher.KeywordMatcher( 'relay',
      helpdesc='DHCP Relay' )

# RoutingProtocolIntfConfigModelet omits management and loopback interfaces
modelet = IraIpIntfCli.RoutingProtocolIntfConfigModelet

#-------------------------------------------------------------------------------
# The "[no|default] ip dhcp relay [client] [server]" interface command
#-------------------------------------------------------------------------------

relayMatcher = CliMatcher.KeywordMatcher( 'relay', helpdesc='DHCP Relay' )

class IpDhcpRelayClientServerCmd( CliCommand.CliCommandClass ):
   syntax = 'ip dhcp relay [ client ] [ server ]'
   noOrDefaultSyntax = syntax

   data = {
      'ip' : ipMatcherForConfigIf,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : relayMatcher,
      'client' : 'Service DHCP clients on this interface',
      'server' : 'Process DHCP server responses on this interface',
   }
   
   hidden = True
   handler = "OldDhcpRelayHandler.setIntfDhcpRelay"
   noOrDefaultHandler = handler
   
modelet.addCommandClass( IpDhcpRelayClientServerCmd )

#-------------------------------------------------------------------------------
# The "[no|default] ip dhcp relay circuit-id <circuit>" interface command
#-------------------------------------------------------------------------------
circuitIdMatcher = CliMatcher.KeywordMatcher(
   'circuit-id', helpdesc='Relay Agent (Option 82) Circuit ID for this interface' )

class IpDhcpRelayCircuitIdCmd( CliCommand.CliCommandClass ):
   syntax = 'ip dhcp relay circuit-id ID'
   noOrDefaultSyntax = 'ip dhcp relay circuit-id ...'

   data = {
      'ip' : ipMatcherForConfigIf,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : relayMatcher,
      'circuit-id' : circuitIdMatcher,
      'ID' : circuitIdValueMatcher,
   }

   hidden = True
   handler = "OldDhcpRelayHandler.setIntfCircuitId"
   noOrDefaultHandler = "OldDhcpRelayHandler.noIntfCircuitId"
 
modelet.addCommandClass( IpDhcpRelayCircuitIdCmd )

class OldDhcpRelayIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      del dhcpRelayConfig.intfConfig[ self.intf_.name ]

#--------------------------------------------------------------------------------
# [ no | default ] ip dhcp relay log verbose
#--------------------------------------------------------------------------------
class IpDhcpRelayLogVerboseCmd( CliCommand.CliCommandClass ):
   syntax = 'ip dhcp relay log verbose'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : ipMatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'log' : 'Specify Logging Level',
      'verbose' : 'More Log Messages',
   }

   handler = "OldDhcpRelayHandler.setDhcpRelayLogLevel"
   noOrDefaultHandler = handler
   
BasicCliModes.GlobalConfigMode.addCommandClass( IpDhcpRelayLogVerboseCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip dhcp relay server ADDR
#--------------------------------------------------------------------------------
class IpDhcpRelayServerAddrCmd( CliCommand.CliCommandClass ):
   syntax = 'ip dhcp relay server ADDR'
   noOrDefaultSyntax = syntax
   # In the original code, there is some consideration to allowing ADDR to be
   # an interface name, but was disabled because it is low win and has some
   # problems (e.g. if the intf is not routable the configuration will not work
   # and is confusing)
   data = {
      'ip' : ipMatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'server' : 'Specify DHCP Server',
      'ADDR' : IpAddrMatcher( helpdesc="DHCP Server's address" ),
   }

   handler = "OldDhcpRelayHandler.setDhcpRelayServer"
   noOrDefaultHandler = handler
   
BasicCliModes.GlobalConfigMode.addCommandClass( IpDhcpRelayServerAddrCmd )

def Plugin( entityManager ):
   global dhcpRelayConfig
   dhcpRelayConfig = ConfigMount.mount( entityManager, "ip/dhcp/relay/config",
                                        "Ip::Dhcp::Relay::Config", "w" )

   IntfCli.Intf.registerDependentClass( OldDhcpRelayIntf )

