#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
import CliToken.OpenConfigTokens
from CliPlugin import AclCli
from CliPlugin import Gnmi
from CliPlugin import VrfCli
from CliPlugin import IpGenAddrMatcher
from CliPlugin import OpenConfigCliLib
from CliPlugin.AclCli import ( ipKwForServiceAclMatcher, 
                               inKwMatcher, 
                               accessGroupKwMatcher )
import AuthnUserPriorityCli

matcherPort = CliMatcher.KeywordMatcher( 'port', 
      helpdesc='The port number to listen on' )
matcherProfile = CliMatcher.KeywordMatcher( 'profile', 
      helpdesc='Configure SSL profile' )
matcherSsl = CliMatcher.KeywordMatcher( 'ssl', 
      helpdesc='Configure SSL related options' )
matcherAuthz = CliMatcher.KeywordMatcher( 'authorization', 
      helpdesc='Configure authorization' )
matcherNotification = CliMatcher.KeywordMatcher( 'notification',
      helpdesc='Configure subscription notification options' )
matcherTime = CliMatcher.KeywordMatcher( 'timestamp', 
      helpdesc='Configure timestamp options' )
matcherLastChangeTime = CliMatcher.KeywordMatcher( 'last-change-time', 
      helpdesc='Set timestamp to last change time' )
matcherSendTime = CliMatcher.KeywordMatcher( 'send-time', 
      helpdesc='Set timestamp to send time' )
matcherListenAddr = CliMatcher.KeywordMatcher( 'listen-addresses',
      helpdesc='Configure listen addresses' )
matcherReq = CliMatcher.KeywordMatcher( 'requests',
      helpdesc='Configure per-RPC authorization' )
matcherCert = CliMatcher.KeywordMatcher( 'certificate',
      helpdesc='Configure certificate options' )
matcherUsername = CliMatcher.KeywordMatcher( 'username',
      helpdesc='Configure certificate username options' )
matcherCertUsernameAuthn = CliMatcher.KeywordMatcher( 'authentication',
      helpdesc='Configure certificate username authentication' )
authenticationMatcher = CliCommand.Node( matcher=matcherCertUsernameAuthn,
     deprecatedByCmd='authentication username priority' )
matcherAccounting = CliMatcher.KeywordMatcher( 'accounting',
      helpdesc='Configure accounting' )
matcherAccountingReq = CliMatcher.KeywordMatcher( 'requests',
      helpdesc='Configure RPC requests accounting' )
matcherConnection = CliMatcher.KeywordMatcher( 'connection',
      helpdesc='Configure gRPC connection options' )
matcherLimit = CliMatcher.KeywordMatcher( 'limit',
      helpdesc='Configure maximum connection limit' )

#--------------------------------------------------------------------------------
# [ no | default ] vrf VRFNAME 
#--------------------------------------------------------------------------------
class Cmd( CliCommand.CliCommandClass ):
   syntax = ' vrf VRFNAME '
   noOrDefaultSyntax = 'vrf ...'
   data = {
      'vrf': 'Configure VRF',
      'VRFNAME': CliMatcher.DynamicNameMatcher( VrfCli.getVrfNames, 'VRF name' ),
   }
   handler = Gnmi.setVrfName
   noOrDefaultHandler = handler

Gnmi.GnmiTransportConfigMode.addCommandClass( Cmd )

#--------------------------------------------------------------------------------
# [ no | default ] authorization requests
#--------------------------------------------------------------------------------
class AuthzCmd( CliCommand.CliCommandClass ):
   syntax = ' authorization requests '
   noOrDefaultSyntax = 'authorization requests'
   data = {
      'authorization': matcherAuthz,
      'requests': matcherReq
   }
   handler = Gnmi.setAuthorization
   noOrDefaultHandler = Gnmi.noAuthorization

Gnmi.GnmiTransportConfigMode.addCommandClass( AuthzCmd )

#--------------------------------------------------------------------------------
# [ no | default ] accounting requests
#--------------------------------------------------------------------------------
class AccountingCmd( CliCommand.CliCommandClass ):
   syntax = 'accounting requests'
   noOrDefaultSyntax = 'accounting requests'
   data = {
      'accounting': matcherAccounting,
      'requests': matcherAccountingReq
   }

   @staticmethod
   def handler( mode, args ):
      mode.config_.endpoints[ mode.name ].accounting = True

   @staticmethod
   def noOrDefaultHandler ( mode, args ):
      mode.config_.endpoints[ mode.name ].accounting = False

Gnmi.GnmiTransportConfigMode.addCommandClass( AccountingCmd )

#--------------------------------------------------------------------------------
# [ no | default ] certificate username authentication
#--------------------------------------------------------------------------------
class CertUsernameAuthnCmd( CliCommand.CliCommandClass ):
   syntax = 'certificate username authentication'
   noOrDefaultSyntax = 'certificate username authentication'
   data = {
      'certificate': matcherCert,
      'username': matcherUsername,
      'authentication': authenticationMatcher,
   }
   handler = Gnmi.setCertificateUsernameAuthentication
   noOrDefaultHandler = Gnmi.noCertificateUsernameAuthentication

Gnmi.GnmiTransportConfigMode.addCommandClass( CertUsernameAuthnCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip access-group ACLNAME [ in ]
#--------------------------------------------------------------------------------
class IpAccessGroupAclnameCmd( CliCommand.CliCommandClass ):
   syntax = 'ip access-group ACLNAME [ in ]'
   noOrDefaultSyntax = 'ip access-group ...'
   data = {
      'ip': ipKwForServiceAclMatcher,
      'access-group': accessGroupKwMatcher,
      'ACLNAME': AclCli.standardIpAclNameMatcher,
      'in': inKwMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      endpoint = mode.config_.endpoints.get( mode.name )
      if ( endpoint.serviceAcl 
           or not OpenConfigCliLib.existsOtherTransportSameVrfWithServiceAcl(
              mode, endpoint.vrfName ) ):
         endpoint.serviceAcl = args[ 'ACLNAME' ]
         OpenConfigCliLib.setServiceAcl( mode, 'gnmi', endpoint )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      endpoint = mode.config_.endpoints.get( mode.name )
      if endpoint.serviceAcl:
         endpoint.serviceAcl = ""
         OpenConfigCliLib.noServiceAcl( mode, "gnmi", endpoint )

Gnmi.GnmiTransportConfigMode.addCommandClass( IpAccessGroupAclnameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] listen-addresses [{ ips }]
#--------------------------------------------------------------------------------
class IpListenAddressesCmd( CliCommand.CliCommandClass ):
   syntax = 'listen-addresses [ { IP } ]'
   noOrDefaultSyntax = 'listen-addresses ...'
   data = {
      'listen-addresses': matcherListenAddr,
      'IP': IpGenAddrMatcher.IpGenAddrMatcher( "IP addresses to listen on" ),
   }

   @staticmethod
   def handler( mode, args ):
      ipAddrs = set( args.get( 'IP', [] ) )
      listenAddrs = mode.config_.endpoints.get( mode.name ).listenAddrs
      listenAddrs.clear()
      for ip in ipAddrs:
         listenAddrs.add( ip )

   @staticmethod
   def noOrDefaultHandler ( mode, args ):
      mode.config_.endpoints.get( mode.name ).listenAddrs.clear()

Gnmi.GnmiTransportConfigMode.addCommandClass( IpListenAddressesCmd )

Gnmi.GnmiTransportConfigMode.addCommandClass(
      AuthnUserPriorityCli.authnUsernamePriorityCmd )

#--------------------------------------------------------------------------------
# [ no | default ] port PORT
#--------------------------------------------------------------------------------
class PortPortCmd( CliCommand.CliCommandClass ):
   syntax = 'port PORT'
   noOrDefaultSyntax = 'port ...'
   data = {
      'port': matcherPort,
      'PORT': CliToken.OpenConfigTokens.matcherPortValue,
   }
   handler = Gnmi.setPort
   noOrDefaultHandler = handler

Gnmi.GnmiTransportConfigMode.addCommandClass( PortPortCmd )

#--------------------------------------------------------------------------------
# [ no | default ] shutdown
#--------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Disable protocol',
   }
   handler = Gnmi.shutdown
   noHandler = Gnmi.noShutdown
   defaultHandler = Gnmi.shutdown

Gnmi.GnmiTransportConfigMode.addCommandClass( ShutdownCmd )

#--------------------------------------------------------------------------------
# [ no | default ] notification timestamp ( last-change-time | send-time )
#--------------------------------------------------------------------------------
class UseNotifSendTimeCmd( CliCommand.CliCommandClass ):
   syntax = 'notification timestamp ( last-change-time | send-time )'
   noOrDefaultSyntax = 'notification timestamp ...'
   data = {
      'notification': matcherNotification,
      'timestamp': matcherTime,
      'last-change-time': matcherLastChangeTime,
      'send-time': matcherSendTime,
   }

   @staticmethod
   def handler( mode, args ):
      mode.config_.endpoints[ mode.name ].useNotifSendTime = 'send-time' in args

   noOrDefaultHandler = handler

Gnmi.GnmiTransportConfigMode.addCommandClass( UseNotifSendTimeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ssl profile PROFILENAME
#--------------------------------------------------------------------------------
class SslProfileProfilenameCmd( CliCommand.CliCommandClass ):
   syntax = 'ssl profile PROFILENAME'
   noOrDefaultSyntax = 'ssl profile ...'
   data = {
      'ssl': matcherSsl,
      'profile': matcherProfile,
      'PROFILENAME': Gnmi.profileNameMatcher,
   }
   handler = Gnmi.setSslProfile
   noOrDefaultHandler = Gnmi.noSslProfile

Gnmi.GnmiTransportConfigMode.addCommandClass( SslProfileProfilenameCmd )

# --------------------------------------------------------------------------------
# [ no | default ] connection limit CONNECTION_LIMIT
# --------------------------------------------------------------------------------
class ConnectionLimitConnectionLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'connection limit CONNECTION_LIMIT'
   noOrDefaultSyntax = 'connection limit ...'
   data = {
      'connection': matcherConnection,
      'limit': matcherLimit,
      'CONNECTION_LIMIT': CliToken.OpenConfigTokens.matcherConnectionLimit,
   }
   handler = Gnmi.setConnectionLimit
   noOrDefaultHandler = handler

Gnmi.GnmiTransportConfigMode.addCommandClass( ConnectionLimitConnectionLimitCmd )
