#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import BasicCliModes
import CliCommand
from CliMatcher import DynamicNameMatcher, IntegerMatcher
import CliParserCommon
from CliMode.Gnsi import MgmtGnsiMode, MgmtGnsiAcctzMode
from CliPlugin.ConfigMgmtMode import managementKwMatcher, apiKwMatcher
import ConfigMount
import LazyMount
import Tac
from Toggles import GnsiToggleLib, OpenConfigToggleLib

gnsiConfig = None
gnmiConfig = None

gnsiConfigHelpStr = "Configure gNSI"

transportNameMatcher = DynamicNameMatcher(
      lambda mode: gnsiConfig.endpoints,
      'Transport name', helpname='WORD',
      extraEmptyTokenCompletionFn=lambda mode, context:[
            CliParserCommon.Completion( endpoint, endpoint )
            for endpoint in gnmiConfig.endpoints ] )

class MgmtGnsiConfigMode( MgmtGnsiMode, BasicCli.ConfigModeBase ):
   """CLI configuration mode 'management api gnsi'."""

   name = "GNSI configuration"

   def __init__( self, parent, session ):
      self.config_ = gnmiConfig
      MgmtGnsiMode.__init__( self, "api-gnsi" )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class MgmtGnsiAcctzConfigMode( MgmtGnsiAcctzMode, BasicCli.ConfigModeBase ):
   """CLI configuration mode 'service acctz'."""

   name = "GNSI acctz configuration"

   def __init__( self, parent, session ):
      self.config_ = gnsiConfig
      MgmtGnsiAcctzMode.__init__( self, "api-gnsi-acctz" )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#--------------------------------------------------------------------------------
# [ no | default ] management api gnsi
#--------------------------------------------------------------------------------
class ManagementApiGnsiCmd( CliCommand.CliCommandClass ):
   syntax = 'management api gnsi'
   noOrDefaultSyntax = syntax
   data = {
      'management': managementKwMatcher,
      'api': apiKwMatcher,
      'gnsi': gnsiConfigHelpStr,
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( MgmtGnsiConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # Clear all gnsi endpoints
      gnsiConfig.endpoints.clear()
      # Disable all gnsi sub-services
      gnsiConfig.service = Tac.Value( "Gnsi::Service" )
      # Clear the gNSIEnabled in all gNMI endpoints
      for name in gnmiConfig.endpoints:
         gnmiConfig.endpoints[ name ].gnsiEnabled = False
      # If a non-default history-limit is set for Acctz, default it
      if gnsiConfig.acctzConfig:
         gnsiConfig.acctzConfig.historyLimit = \
               gnsiConfig.acctzConfig.historyLimitDefault

BasicCliModes.GlobalConfigMode.addCommandClass( ManagementApiGnsiCmd )

class TransportGnsiCmd( CliCommand.CliCommandClass ):
   syntax = 'transport gnmi TRANSPORT_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'transport': 'Configure transport',
      'gnmi': 'Configure gNSI endpoint using gNMI\'s transport configuration',
      'TRANSPORT_NAME': transportNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      if name := args.get( 'TRANSPORT_NAME' ):
         if name in gnmiConfig.endpoints:
            gnmiConfig.endpoints[ name ].gnsiEnabled = True
         gnsiConfig.endpoints.add( name )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if name := args.get( 'TRANSPORT_NAME' ):
         gnsiConfig.endpoints.remove( name )
         if endpoint := gnmiConfig.endpoints.get( name ):
            endpoint.gnsiEnabled = False

MgmtGnsiConfigMode.addCommandClass( TransportGnsiCmd )

class AuthzGnsiCmd( CliCommand.CliCommandClass ):
   syntax = 'service authz'
   noOrDefaultSyntax = syntax
   data = {
      'service': gnsiConfigHelpStr,
      'authz': 'Configure gNSI authz service',
   }

   @staticmethod
   def handler( mode, args ):
      service = Tac.Value( "Gnsi::Service", gnsiConfig.service.value )
      service.authz = True
      gnsiConfig.service = service

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      service = Tac.Value( "Gnsi::Service", gnsiConfig.service.value )
      service.authz = False
      gnsiConfig.service = service

MgmtGnsiConfigMode.addCommandClass( AuthzGnsiCmd )

class CertzGnsiCmd( CliCommand.CliCommandClass ):
   syntax = 'service certz'
   noOrDefaultSyntax = syntax
   data = {
      'service': gnsiConfigHelpStr,
      'certz': 'Configure gNSI certz service',
   }

   @staticmethod
   def handler( mode, args ):
      service = Tac.Value( "Gnsi::Service", gnsiConfig.service.value )
      service.certz = True
      gnsiConfig.service = service

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      service = Tac.Value( "Gnsi::Service", gnsiConfig.service.value )
      service.certz = False
      gnsiConfig.service = service

MgmtGnsiConfigMode.addCommandClass( CertzGnsiCmd )

class CredentialzGnsiCmd( CliCommand.CliCommandClass ):
   syntax = 'service credentialz'
   noOrDefaultSyntax = syntax
   data = {
      'service': gnsiConfigHelpStr,
      'credentialz': 'Configure gNSI credentialz service',
   }

   @staticmethod
   def handler( mode, args ):
      service = Tac.Value( "Gnsi::Service", gnsiConfig.service.value )
      service.credentialz = True
      gnsiConfig.service = service

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      service = Tac.Value( "Gnsi::Service", gnsiConfig.service.value )
      service.credentialz = False
      gnsiConfig.service = service

MgmtGnsiConfigMode.addCommandClass( CredentialzGnsiCmd )

class PathzGnsiCmd( CliCommand.CliCommandClass ):
   syntax = 'service pathz'
   noOrDefaultSyntax = syntax
   data = {
      'service': gnsiConfigHelpStr,
      'pathz': 'Configure gNSI pathz service',
   }

   @staticmethod
   def handler( mode, args ):
      service = Tac.Value( "Gnsi::Service", gnsiConfig.service.value )
      service.pathz = True
      gnsiConfig.service = service

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      service = Tac.Value( "Gnsi::Service", gnsiConfig.service.value )
      service.pathz = False
      gnsiConfig.service = service

if OpenConfigToggleLib.toggleOCGNSIPathzToggleEnabled():
   MgmtGnsiConfigMode.addCommandClass( PathzGnsiCmd )

class AcctzGnsiCmd( CliCommand.CliCommandClass ):
   syntax = 'service acctz'
   noOrDefaultSyntax = syntax
   data = {
      'service': gnsiConfigHelpStr,
      'acctz': 'Configure gNSI acctz service',
   }

   @staticmethod
   def handler( mode, args ):
      service = Tac.Value( "Gnsi::Service", gnsiConfig.service.value )
      service.acctz = True
      gnsiConfig.service = service

      # if gnsiConfig.acctzConfig doesn't exist yet, create it
      if not gnsiConfig.acctzConfig:
         gnsiConfig.acctzConfig = ()

      childMode = mode.childMode( MgmtGnsiAcctzConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      service = Tac.Value( "Gnsi::Service", gnsiConfig.service.value )
      service.acctz = False
      gnsiConfig.service = service
      # Default the acctz config if present
      if gnsiConfig.acctzConfig:
         gnsiConfig.acctzConfig.historyLimit = \
            gnsiConfig.acctzConfig.historyLimitDefault

if GnsiToggleLib.toggleOCGNSIAcctzEnabled():
   MgmtGnsiConfigMode.addCommandClass( AcctzGnsiCmd )

class AcctzGnsiConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'records history limit RECORD_LIMIT'
   noOrDefaultSyntax = 'records history limit ...'
   data = {
      'records': 'GNSI accounting records',
      'history': 'Accounting history',
      'limit': 'History limit',
      'RECORD_LIMIT': IntegerMatcher( 100, 1000,
         helpdesc='Maximum accounting records' )
   }

   @staticmethod
   def handler( mode, args ):
      gnsiConfig.acctzConfig.historyLimit = args[ 'RECORD_LIMIT' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if gnsiConfig.acctzConfig:
         gnsiConfig.acctzConfig.historyLimit = \
            gnsiConfig.acctzConfig.historyLimitDefault

MgmtGnsiAcctzConfigMode.addCommandClass( AcctzGnsiConfigCmd )

def Plugin( entityManager ):
   global gnsiConfig, gnmiConfig

   gnmiConfig = LazyMount.mount( entityManager, "mgmt/gnmi/config",
                                "Gnmi::Config", "r" )
   gnsiConfig = ConfigMount.mount( entityManager, "mgmt/gnsi/config",
                                  "Gnsi::Config", "w" )
