#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=ungrouped-imports

import BasicCli
import CliCommand
import CliMatcher
from CliPlugin import ConfigMgmtMode
import ShowCommand
from CliPlugin import AclCli
from CliPlugin import OpenConfigShowMgmt
from CliPlugin import Gnmi
from CliPlugin.AclCliModel import AllAclList
from CliPlugin.OpenConfigShowMgmt import (
      GnmiAclStatus,
      GnmiAggregateStatus,
      GnsiStatus,
      ModelsStatus,
      NetconfEndpointStatus,
      RestconfEndpointStatus,
)

matcherAccessList = AclCli.accessListKwMatcherForServiceAcl
matcherManagement = ConfigMgmtMode.managementShowKwMatcher
matcherApiForShow = CliMatcher.KeywordMatcher( 'api',
      helpdesc='Show management APIs' )
matcherDynamic = CliMatcher.KeywordMatcher( 'dynamic',
      helpdesc='Dynamic (non-persistent) access-list' )
matcherSummary = CliMatcher.KeywordMatcher( 'summary',
      helpdesc='Access list summary' )

nodeDetail = CliCommand.guardedKeyword( 'detail',
      helpdesc='Access list detail',
      guard=AclCli.countersPerChipEnabledGuard )

# --------------------------------------------------------------------------------
# show management api gnmi access-list [ ACL ] [ summary ] [ dynamic ] [ detail ]
# --------------------------------------------------------------------------------
class ManagementApiGnmiAccessListCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show management api gnmi access-list
               [ ACL ] [ summary ] [ dynamic ] [ detail ]
            '''
   data = {
      'management': matcherManagement,
      'api': matcherApiForShow,
      'gnmi': CliCommand.guardedKeyword( 'gnmi',
                 helpdesc='Show status of gNMI endpoints',
                 guard=AclCli.serviceAclGuard ),
      'access-list': matcherAccessList,
      'summary': matcherSummary,
      'dynamic': matcherDynamic,
      'detail': nodeDetail,
      'ACL': AclCli.ipAclNameMatcher,
   }
   handler = OpenConfigShowMgmt.showAclStatusGnmi
   cliModel = GnmiAclStatus

BasicCli.addShowCommandClass( ManagementApiGnmiAccessListCmd )

# --------------------------------------------------------------------------------
# show management api gnmi
# --------------------------------------------------------------------------------
class ManagementApiGnmiCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management api gnmi'
   data = {
      'management': matcherManagement,
      'api': matcherApiForShow,
      'gnmi': 'Show status of gNMI endpoints',
   }
   handler = OpenConfigShowMgmt.showStatusGnmi
   cliModel = GnmiAggregateStatus

BasicCli.addShowCommandClass( ManagementApiGnmiCmd )

# --------------------------------------------------------------------------------
# show management api gnmi
# --------------------------------------------------------------------------------
class ManagementApiGnsiCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management api gnsi'
   data = {
      'management': matcherManagement,
      'api': matcherApiForShow,
      'gnsi': 'Show status of gNSI',
   }
   handler = OpenConfigShowMgmt.showStatusGnsi
   cliModel = GnsiStatus

BasicCli.addShowCommandClass( ManagementApiGnsiCmd )

# --------------------------------------------------------------------------------
# show management api netconf
# --------------------------------------------------------------------------------
class ManagementApiNetconfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management api netconf'
   data = {
      'management': matcherManagement,
      'api': matcherApiForShow,
      'netconf': 'Show status of NETCONF endpoints',
   }
   handler = OpenConfigShowMgmt.showStatusNetconf
   cliModel = NetconfEndpointStatus

BasicCli.addShowCommandClass( ManagementApiNetconfCmd )

# --------------------------------------------------------------------------------
# show management api restconf access-list [ ACL ] [ summary ] [ dynamic ] [ detail ]
# --------------------------------------------------------------------------------
class ManagementApiRestconfAccessListCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show management api restconf access-list
               [ ACL ] [ summary ] [ dynamic ] [ detail ]
            '''
   data = {
      'management': matcherManagement,
      'api': matcherApiForShow,
      'restconf': CliCommand.guardedKeyword( 'restconf',
                     helpdesc='Show status of RESTCONF endpoints',
                     guard=AclCli.serviceAclGuard ),
      'access-list': matcherAccessList,
      'summary': matcherSummary,
      'dynamic': matcherDynamic,
      'detail': nodeDetail,
      'ACL': AclCli.ipAclNameMatcher,
   }
   handler = OpenConfigShowMgmt.showAclStatusRestconf
   cliModel = AllAclList

BasicCli.addShowCommandClass( ManagementApiRestconfAccessListCmd )

# --------------------------------------------------------------------------------
# show management api restconf
# --------------------------------------------------------------------------------
class ManagementApiRestconfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management api restconf'
   data = {
      'management': matcherManagement,
      'api': matcherApiForShow,
      'restconf': 'Show status of RESTCONF endpoints',
   }
   handler = OpenConfigShowMgmt.showStatusRestconf
   cliModel = RestconfEndpointStatus

BasicCli.addShowCommandClass( ManagementApiRestconfCmd )

# --------------------------------------------------------------------------------
# show management api models
# --------------------------------------------------------------------------------
class ManagementApiModelsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management api models'
   data = {
      'management': matcherManagement,
      'api': matcherApiForShow,
      'models': 'Show status of models',
   }
   handler = OpenConfigShowMgmt.showStatusModels
   cliModel = ModelsStatus

BasicCli.addShowCommandClass( ManagementApiModelsCmd )

# --------------------------------------------------------------------------------
# show gnmi get { PATH }
# --------------------------------------------------------------------------------
class ShowGnmiGetCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show gnmi get { PATH }'
   data = {
      'gnmi': Gnmi.matcherGnmi,
      'get': 'Send a Get request to the gNMI server on this device',
      'PATH': CliMatcher.QuotedStringMatcher(
         helpname='PATH',
         helpdesc='Requested path' )
   }

   handler = Gnmi.showGnmiGet
   privileged = True

BasicCli.addShowCommandClass( ShowGnmiGetCmd )
