#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
# pylint: disable-next=consider-using-from-import
import CliPlugin.EbraEthIntfCli as EbraEthIntfCli
# pylint: disable-next=consider-using-from-import
import CliPlugin.OpenFlowCli as OpenFlowCli
import CliPlugin.VlanCli as VlanCli # pylint: disable=consider-using-from-import
import Intf.IntfRange as IntfRange # pylint: disable=consider-using-from-import
import Tac

matcherBind = CliMatcher.KeywordMatcher( 'bind',
      helpdesc='Change OpenFlow bind parameters' )
matcherController = CliMatcher.KeywordMatcher( 'controller',
      helpdesc='Add an OpenFlow controller address' )
matcherRouting = CliCommand.guardedKeyword(
      'routing', 'Configure IP routing to and from OpenFlow-bound VLANs',
      guard=OpenFlowCli.openFlowVeosGuard )

#--------------------------------------------------------------------------------
# [ no | default ] bind interface INTFS
#--------------------------------------------------------------------------------
class BindInterfaceIntf1Cmd( CliCommand.CliCommandClass ):
   syntax = 'bind interface INTFS'
   noOrDefaultSyntax = 'bind interface [ INTFS ]'
   data = {
      'bind': matcherBind,
      'interface': 'Bind an interface to OpenFlow',
      'INTFS': IntfRange.IntfRangeMatcher(
         explicitIntfTypes=EbraEthIntfCli.switchPortIntfTypes ),
   }
   handler = OpenFlowCli.setBindInterface
   noOrDefaultHandler = OpenFlowCli.noBindInterface

OpenFlowCli.OpenFlowConfigMode.addCommandClass( BindInterfaceIntf1Cmd )

#--------------------------------------------------------------------------------
# [ no | default ] bind mode BIND_MODE
#--------------------------------------------------------------------------------
class BindModeCmd( CliCommand.CliCommandClass ):
   syntax = 'bind mode BIND_MODE'
   noOrDefaultSyntax = 'bind mode ...'
   data = {
      'bind': CliCommand.Node( matcher=matcherBind,
                               guard=OpenFlowCli.multitableGuard ),
      'mode': 'Change OpenFlow bind mode',
      'BIND_MODE': CliMatcher.EnumMatcher( {
         'vlan': 'Process traffic on specific VLANs with OpenFlow',
         'interface': ( 'Process traffic from specific ingress interfaces with '
                                                                        'OpenFlow' ),
         'monitor': ( 'Process all traffic normally with OpenFlow-controlled '
                                                                      'monitoring' ),
      } )
   }
   handler = lambda mode, args: OpenFlowCli.setBindMode( mode, args[ 'BIND_MODE' ] )
   noOrDefaultHandler = lambda mode, args: OpenFlowCli.setBindMode( mode )

OpenFlowCli.OpenFlowConfigMode.addCommandClass( BindModeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] bind vlan VLAN_SET
#--------------------------------------------------------------------------------
class BindVlanVlansetCmd( CliCommand.CliCommandClass ):
   syntax = 'bind vlan VLAN_SET'
   noOrDefaultSyntax = 'bind vlan [ VLAN_SET ]'
   data = {
      'bind': matcherBind,
      'vlan': 'Bind a VLAN to OpenFlow',
      'VLAN_SET': VlanCli.vlanSetMatcher,
   }
   handler = lambda mode, args: OpenFlowCli.setBindVlan( mode, args[ 'VLAN_SET' ] )
   noOrDefaultHandler = lambda mode, args: OpenFlowCli.noBindVlan( mode,
                                                             args.get( 'VLAN_SET' ) )

OpenFlowCli.OpenFlowConfigMode.addCommandClass( BindVlanVlansetCmd )

#--------------------------------------------------------------------------------
# [ no | default ] controller CONTROLLER
#--------------------------------------------------------------------------------
class ControllerModeCmd( CliCommand.CliCommandClass ):
   syntax = 'controller CONTROLLER'
   noOrDefaultSyntax = 'controller [ CONTROLLER ]'
   data = {
      'controller': matcherController,
      'CONTROLLER': CliMatcher.PatternMatcher( pattern='tcp:[\\d\\.]+:\\d+',
         helpdesc='Controller IP address and TCP port', helpname='tcp:A.B.C.D:N' ),
   }
   handler = lambda mode, args: OpenFlowCli.gotoControllerMode( mode,
                                                               args[ 'CONTROLLER' ] )
   noOrDefaultHandler = lambda mode, args: OpenFlowCli.noController( mode,
                                                           args.get( 'CONTROLLER' ) )

OpenFlowCli.OpenFlowConfigMode.addCommandClass( ControllerModeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] controller version minimum 1.3
#--------------------------------------------------------------------------------
class MinimumVersionCmd( CliCommand.CliCommandClass ):
   syntax = 'controller version minimum 1.3'
   noOrDefaultSyntax = 'controller version minimum ...'
   data = {
      'controller': matcherController,
      'version': 'Set minimum OpenFlow version',
      'minimum': 'Set minimum OpenFlow protocol version',
      '1.3': 'Set minimum OpenFlow version to be 1.3',
   }

   hidden = True
   handler = lambda mode, args: OpenFlowCli.minimumVersion13( mode )
   noOrDefaultHandler = lambda mode, args: OpenFlowCli.noMinimumVersion13( mode )

OpenFlowCli.OpenFlowConfigMode.addCommandClass( MinimumVersionCmd )

#--------------------------------------------------------------------------------
# [ no | default ] default-action ( controller | drop )
#--------------------------------------------------------------------------------
class DefaultActionCmd( CliCommand.CliCommandClass ):
   syntax = 'default-action ( controller | drop )'
   noOrDefaultSyntax = 'default-action'
   data = {
      'default-action': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher(
            'default-action',
            helpdesc='Set the action of the default flow table entry' ),
         guard=OpenFlowCli.multitableGuard ),
      'controller': 'Output to the controller',
      'drop': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'drop', helpdesc='Drop' ),
         guard=OpenFlowCli.defaultDropGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      action = 'controller' if 'controller' in args else 'drop'
      assert action in args
      return OpenFlowCli.setDefaultAction( mode, action )

   noOrDefaultHandler = lambda mode, args: OpenFlowCli.setDefaultAction( mode )

OpenFlowCli.OpenFlowConfigMode.addCommandClass( DefaultActionCmd )

#--------------------------------------------------------------------------------
# [ no | default ] description DESCRIPTION
#--------------------------------------------------------------------------------
class DescriptionCmd( CliCommand.CliCommandClass ):
   syntax = 'description DESCRIPTION'
   noOrDefaultSyntax = 'description ...'
   data = {
      'description': 'Set switch description',
      'DESCRIPTION': CliMatcher.StringMatcher(
         helpdesc='Up to 256 characters describing this switch', helpname='LINE' ),
   }
   handler = lambda mode, args: OpenFlowCli.setDesc( mode, args[ 'DESCRIPTION' ] )
   noOrDefaultHandler = lambda mode, args: OpenFlowCli.setDesc( mode, '' )

OpenFlowCli.OpenFlowConfigMode.addCommandClass( DescriptionCmd )

#--------------------------------------------------------------------------------
# [ no | default ] datapath id ID
#--------------------------------------------------------------------------------
class DpidCmd( CliCommand.CliCommandClass ):
   syntax = 'datapath id ID'
   noOrDefaultSyntax = 'datapath id ...'
   data = {
      'datapath': 'Switch datapath identifier',
      'id': 'Switch datapath identifier',
      'ID': CliMatcher.IntegerMatcher( lbound=1, ubound=0x7FFFFFFFFFFFFFFF,
         helpdesc='U64 datapath identifier', helpname='ID' ),
   }
   handler = lambda mode, args: \
         OpenFlowCli.setDpid( mode,
            dpidType=Tac.Type( "OpenFlow::DpidType" ).userDefined,
            dpid=args[ 'ID' ] )
   noOrDefaultHandler = lambda mode, args: OpenFlowCli.setDpid( mode )

OpenFlowCli.OpenFlowConfigMode.addCommandClass( DpidCmd )

class DpidTypeIntfMa( CliCommand.CliCommandClass ):
   syntax = 'datapath id interface management1'
   noOrDefaultSyntax = 'datapath id interface ...'
   data = {
      'datapath': 'Switch datapath identifier',
      'id': 'Switch datapath identifier',
      'interface': 'use interface MAC address',
      'management1': 'management1 interface',
   }
   handler = lambda mode, args: OpenFlowCli.setDpid( mode,
         dpidType=Tac.Type( "OpenFlow::DpidType" ).managementMac )
   noOrDefaultHandler = lambda mode, args: OpenFlowCli.setDpid( mode )

OpenFlowCli.OpenFlowConfigMode.addCommandClass( DpidTypeIntfMa )

#--------------------------------------------------------------------------------
# [ no | default ] exception ttl expired action controller
#--------------------------------------------------------------------------------
class ExpiredTtlFlowCmd( CliCommand.CliCommandClass ):
   syntax = 'exception ttl expired action controller'
   noOrDefaultSyntax = syntax
   data = {
      'exception': 'Configure exception flow',
      'ttl': 'Time-To-Live',
      'expired': 'Expired Time-To-Live',
      'action': 'Action',
      'controller': 'Output to the controller',
   }

   hidden = True
   handler = lambda mode, args: OpenFlowCli.setExpiredTtlFlow( mode, no=None )
   noOrDefaultHandler = lambda mode, args: OpenFlowCli.setExpiredTtlFlow( mode,
                                                                          no=True )

OpenFlowCli.OpenFlowConfigMode.addCommandClass( ExpiredTtlFlowCmd )

#--------------------------------------------------------------------------------
# [ no | default ] extension tlv tracer
#--------------------------------------------------------------------------------
class ExtensionTlvTracerCmd( CliCommand.CliCommandClass ):
   syntax = 'extension tlv tracer'
   noOrDefaultSyntax = syntax
   data = {
      'extension': 'Set vendor specific extensions with PACKET-IN and PACKET-OUT',
      'tlv': 'Set vendor tlv extension',
      'tracer': 'Enable path tracer',
   }
   hidden = True
   handler = lambda mode, args: OpenFlowCli.setTracerTlv( mode )
   noOrDefaultHandler = lambda mode, args: OpenFlowCli.noTracerTlv( mode )

OpenFlowCli.OpenFlowConfigMode.addCommandClass( ExtensionTlvTracerCmd )

#--------------------------------------------------------------------------------
# [ no | default ] forwarding pipeline PIPELINE
#--------------------------------------------------------------------------------
class ForwardingPipelineCmd( CliCommand.CliCommandClass ):
   syntax = 'forwarding pipeline PIPELINE'
   noOrDefaultSyntax = 'forwarding pipeline ...'
   data = {
      'forwarding': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher(
            "forwarding",
            helpdesc='The pipeline to use for forwarding of OpenFlow packets.' ),
         guard=OpenFlowCli.forwardingPipelineGuard ),
      'pipeline': 'The pipeline to use for forwarding of OpenFlow packets.',
      'PIPELINE': CliMatcher.EnumMatcher( {
         'flow': 'Default OpenFlow mode',
         'flow-and-normal': 'OpenFlow mode with normal processing',
      } )
   }
   handler = lambda mode, args: OpenFlowCli.setForwardingPipeline( mode,
                                                                 args[ 'PIPELINE' ] )
   noOrDefaultHandler = lambda mode, args: OpenFlowCli.setForwardingPipeline( mode )

OpenFlowCli.OpenFlowConfigMode.addCommandClass( ForwardingPipelineCmd )

#--------------------------------------------------------------------------------
# [ no | default ] keepalive PERIOD
#--------------------------------------------------------------------------------
class KeepalivePeriodCmd( CliCommand.CliCommandClass ):
   syntax = 'keepalive PERIOD'
   noOrDefaultSyntax = 'keepalive ...'
   data = {
      'keepalive': 'Set OpenFlow controller keepalive period',
      'PERIOD': CliMatcher.IntegerMatcher( 1, 1000000,
         helpdesc='Keepalive period in sec' ),
   }
   handler = lambda mode, args: OpenFlowCli.setKeepalive( mode, args[ 'PERIOD' ] )
   noOrDefaultHandler = lambda mode, args: OpenFlowCli.noKeepalive( mode )

OpenFlowCli.OpenFlowConfigMode.addCommandClass( KeepalivePeriodCmd )

#--------------------------------------------------------------------------------
# [ no | default ] profile PROFILE
#--------------------------------------------------------------------------------
class ProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'profile PROFILE'
   noOrDefaultSyntax = 'profile ...'
   data = {
      'profile': 'Change the OpenFlow table profile',
      'PROFILE': CliMatcher.EnumMatcher( {
         'full-match': 'Match or wildcard any field',
         'l2-match': 'Match or wildcard L2 fields, wildcard others',
         'vxlan-match': ( 'Match or wildcard fields in VXLAN header or '
                          'encapsulated packet' ),
         'auto': 'Match or wildcard any field',
      } )
   }
   handler = lambda mode, args: OpenFlowCli.setTableProfile( mode,
                                                          profile=args[ 'PROFILE' ] )
   noOrDefaultHandler = lambda mode, args: OpenFlowCli.setTableProfile( mode )

OpenFlowCli.OpenFlowConfigMode.addCommandClass( ProfileCmd )

#--------------------------------------------------------------------------------
# ( no | default ) routing
#--------------------------------------------------------------------------------
class RoutingCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'routing'
   data = {
      'routing': matcherRouting,
   }
   noOrDefaultHandler = lambda mode, args: OpenFlowCli.noRouting( mode )

OpenFlowCli.OpenFlowConfigMode.addCommandClass( RoutingCmd )

#--------------------------------------------------------------------------------
# [ no | default ] routing recirculation-interface INTF
#--------------------------------------------------------------------------------
class RoutingRecircIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'routing recirculation-interface INTF'
   noOrDefaultSyntax = 'routing recirculation-interface ...'
   data = {
      'routing': matcherRouting,
      'recirculation-interface': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher(
            'recirculation-interface',
            helpdesc=( 'Set interface for recirculating packets '
                       'routed to/from OpenFlow VLANs' ) ),
         guard=OpenFlowCli.multitableGuard ),
      'INTF': EbraEthIntfCli.switchPortMatcher,
   }
   handler = lambda mode, args: OpenFlowCli.setRoutingRecircIntf( mode,
                                                                  args[ 'INTF' ] )
   noOrDefaultHandler = lambda mode, args: OpenFlowCli.noRoutingRecircIntf( mode )

OpenFlowCli.OpenFlowConfigMode.addCommandClass( RoutingRecircIntfCmd )

#--------------------------------------------------------------------------------
# [ no | default ] routing vlan ( untagged | VLAN_ID ) routed-vlan ROUTED_VLAN_ID
#--------------------------------------------------------------------------------
class RoutingVlanRoutedVlanVlanidCmd( CliCommand.CliCommandClass ):
   syntax = 'routing vlan ( untagged | VLAN_ID ) routed-vlan ROUTED_VLAN_ID'
   noOrDefaultSyntax = 'routing vlan [ ( untagged | VLAN_ID ) ... ]'
   data = {
      'routing': matcherRouting,
      'vlan': 'OpenFlow-bound VLAN',
      'untagged': 'No VLAN ID',
      'VLAN_ID': VlanCli.vlanIdMatcher,
      'routed-vlan': 'Routed transit VLAN',
      'ROUTED_VLAN_ID': VlanCli.vlanIdMatcher,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'untagged' in args:
         args[ 'VLAN_ID' ] = VlanCli.Vlan( 0 )

   handler = lambda mode, args: OpenFlowCli.setRoutingVlan( mode, args[ 'VLAN_ID' ],
                                                           args[ 'ROUTED_VLAN_ID' ] )
   noOrDefaultHandler = lambda mode, args: OpenFlowCli.noRoutingVlan( mode,
                                                              args.get( 'VLAN_ID' ) )

OpenFlowCli.OpenFlowConfigMode.addCommandClass( RoutingVlanRoutedVlanVlanidCmd )

#--------------------------------------------------------------------------------
# [ no | default ] shell-command allowed
#--------------------------------------------------------------------------------
class ShellCommandAllowedCmd( CliCommand.CliCommandClass ):
   syntax = 'shell-command allowed'
   noOrDefaultSyntax = syntax
   data = {
      'shell-command': 'Configure running of arbitrary shell commands',
      'allowed': 'Allow controller to run arbitrary shell commands',
   }
   handler = lambda mode, args: OpenFlowCli.setShellCommandAllowed( mode )
   noOrDefaultHandler = lambda mode, args: OpenFlowCli.noShellCommandAllowed( mode )

OpenFlowCli.OpenFlowConfigMode.addCommandClass( ShellCommandAllowedCmd )

#--------------------------------------------------------------------------------
# [ no | default ] shutdown
#--------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Enable or disable the OpenFlow feature',
   }

   handler = lambda mode, args: OpenFlowCli.setEnable( mode, no=None )
   noHandler = lambda mode, args: OpenFlowCli.setEnable( mode, no=True )
   defaultHandler = handler

OpenFlowCli.OpenFlowConfigMode.addCommandClass( ShutdownCmd )
