#!/usr/bin/env python3
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=bad-string-format-type
# pylint: disable=consider-using-f-string

# pylint: disable-msg=W0212

import Tac
from CliModel import Model
from CliModel import Dict, GeneratorList
from CliModel import Int, Float, Str, Enum, Bool, GeneratorDict, Submodel
from CliModel import List
from ArnetModel import Ip4Address, Ip6Address, IpGenericAddress
from HumanReadable import formatTimeInterval
from Ark import ReversibleDict, timestampToStr
from DeviceNameLib import intfLongName, kernelIntfFilter
import ConfigMount
from IpLibConsts import DEFAULT_VRF 
from CliPlugin.OspfCliModels import GR_HELPER_EXIT_REASON_MAP, grHlprExitReasonStr 
from CliPlugin.OspfCliModels import GR_SPEAKER_EXIT_REASON_MAP, grSpkrExitReasonStr 
from CliPlugin.OspfCliModels import GR_STATE_MAP, GR_RESTART_REASON_MAP
from CliPlugin.OspfCliModels import grRestartTypeStr
from CliPlugin.OspfCliModels import OspfShowInstGracefulRestartInfoModel
from RibCapiLib import timestampToLocalTime, timestampToLocalTimeMsec
import PyRibAmiClient
import Ospf3Consts
import LazyMount
import TableOutput

ospf3ConfigDir = None 
ospf3StatusDir = None

def ospf3Config():
   return ospf3ConfigDir

def ospf3Status():
   return ospf3StatusDir

def getInstanceConfigByProcessId( processId ):
   for vc in ospf3Config().vrfConfig.values():
      if vc.processId == processId:
         return vc.instanceConfig[ 0 ]
   return None

def getProcessIdFromVrfName( vrf=DEFAULT_VRF ):
   for vc in ospf3Config().vrfConfig.values():
      if vc.vrfName == vrf:
         return vc.processId
   return None

def getInstanceConfig( vrfName, instanceId ):
   for vc in ospf3Config().vrfConfig.values():
      if vc.vrfName == vrfName:
         return vc.instanceConfig[ instanceId ]
   return None

def getInstanceConfigFromKeys( vrfName, instanceId=None ):
   processId = getProcessIdFromVrfName( vrfName )
   if processId:
      return getInstanceConfigByProcessId( processId )
   else:
      assert instanceId in [ 0, 64, 255 ]
      return getInstanceConfig( vrfName, instanceId )

def getOspf3Interface( intfName ):
   # Sham links don't correspond to an actual EOS Interface,
   # so there is no IntfId
   if "OSPF3_SL" not in intfName:
      # Ensure the type check for real interfaces
      return intfLongName( intfName )
   return intfName

#-------------------------------------------------------------------------------
# "show ipv6 ospf [process-id] [vrf vrfname]
#-------------------------------------------------------------------------------

maxMetricUnsetReason = { 1 : 'TimerExpired',
                         2 : 'BgpConverged',
                         3 : 'BgpTimerExpired',
                         4 : 'UserConfig' }

areaType = { 0 : 'stubArea',
             1 : 'normalArea',
             2 : 'normalArea' }

hmacAlgo  = { 1 : 'HMAC-MD5',
              2 : 'HMAC-SHA1' }

proto = { 50 : 'IPPROTO_ESP',
          51 : 'IPPROTO_AH' }
              
NUM_MS_PER_SEC = 1000

class Ospf3ShowInstAreaRangeModelBase( Model ):
   maskLen = Int( help='Area range mask length' )
   cost = Int( help='Area range cost' )
   doNotAdvertise = Bool( default=False, optional=True,
                          help='Do not advertise the area range. The cost value '
                               'is invalid if this flag is set to true' )

   def getKey( self, data ):
      return data[ 'address' ]

   def processData( self, data ):
      if data[ 'cost' ] == -1:
         self.doNotAdvertise = True
      return data

   def render( self ):
      if self.doNotAdvertise:
         print( "      %s/%d DoNotAdvertise" % ( self.address, self.maskLen ) )
      else:
         print( "      %s/%d Cost %d Advertise" % (
               self.address, self.maskLen, self.cost ) )

class Ospf3ShowInstAreaRangeModel( Ospf3ShowInstAreaRangeModelBase ):
   address = Ip6Address( help='Area range address' )

   def processData( self, data ):
      return Ospf3ShowInstAreaRangeModelBase.processData( self, data )

   def render( self ):
      return Ospf3ShowInstAreaRangeModelBase.render( self )

class Ospf3MultiAfShowInstAreaRangeModel( Ospf3ShowInstAreaRangeModelBase ):
   address = IpGenericAddress( help='Area range address' )

   def processData( self, data ):
      return Ospf3ShowInstAreaRangeModelBase.processData( self, data )

   def render( self ):
      return Ospf3ShowInstAreaRangeModelBase.render( self )

class Ospf3AreaAuthInfoModel( Model ):
   areaAuthConfig = Bool( default=False, help='area authentication'
                             'is enabled or not' )
   authProtocol = Enum( optional=True, values=( 'IPPROTO_ESP', 'IPPROTO_AH' ),
                        help='OSPFv3 area authentication protocol' )
   hmacAlgorithm = Enum( optional=True, values=( 'HMAC-MD5', 'HMAC-SHA1' ),
                         help='HMAC algorithm being used' )
   secParamIndex = Int( optional=True, help='IPsec Security paramater index' )

class Ospf3ShowInstAreaModelBase( Model ):
   areaId = Ip4Address( help='OSPFv3 area ID' )
   numIntf = Int( help='Number of interfaces in the area' )
   areaType = Enum( values=( 'normalArea', 'stubArea', 'nssaArea' ),
                    help='Type of area(normal/stub/nssa)' )
   nssaTransState =  Bool( default=False,
                           help='Nssa translate always flag for area' )
   areaAuthTypeInformation = Submodel( valueType=Ospf3AreaAuthInfoModel, 
                              optional=True, help='Area authentication information' )
   spfCount = Int( help='Area SPF count' )
   areaFiltersConfigured = Bool( default=False,
                                 help='Area summary filters configured' )
   areaFilterPrefixList = Str( optional=True, help='Area Filter prefix-list' )

   def getKey( self, data ):
      return data[ 'areaId' ]

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF3_INSTANCE_SUMMARY_AREA

   def processData( self, data ):
      if 'stubArea' in data:
         data[ 'areaType' ] = 'stubArea'
      if 'normalArea' in data:
         data[ 'areaType' ] = 'normalArea'
      if 'nssaArea' in data:
         data[ 'areaType' ] = 'nssaArea'
      if 'translator' in data:
         data[ 'nssaTransState' ] = ord( data.pop( 'translator' )) != 0
      if 'security-enabled' in data:
         areaAuthTypeInfo = Ospf3AreaAuthInfoModel()
         areaAuthTypeInfo.areaAuthConfig = \
             ord( data.pop( 'security-enabled' ) ) != 0
         if 'security-proto' in data:
            areaAuthTypeInfo.authProtocol = proto[ data.pop( 'security-proto' ) ]
         if 'hmac-algo' in data:
            areaAuthTypeInfo.hmacAlgorithm = hmacAlgo[ data.pop( 'hmac-algo' ) ]
         if 'spi' in data:
            areaAuthTypeInfo.secParamIndex = data.pop( 'spi' ) 
         data[ 'areaAuthTypeInformation' ] = areaAuthTypeInfo
         
      if 'areaFiltersConfigured' in data:
         data[ 'areaFiltersConfigured' ] = \
               ord( data[ 'areaFiltersConfigured' ] ) != 0
      
      if 'areaFilterPrefixList' in data and \
         data[ 'areaFilterPrefixList' ].startswith( '.ipv6.' ):
         # For ipv6 prefix-list, gated appends .ipv6. before the name
         # remove this prefix when populating the model
         data[ 'areaFilterPrefixList' ] = \
               data[ 'areaFilterPrefixList' ].split( '.', 2 )[ 2 ]

      return data

   def render( self ):
      print( "  Area %s" % self.areaId )

      print( "    Number of interface in this area is %s" %
             self.numIntf )
      if self.areaType == 'normalArea':
         print( "    It is a normal area" )
      if self.areaType == 'nssaArea':
         print( "    It is a nssa area" )
      if self.areaType == 'stubArea':
         print( "    It is a stub area" )

      if self.nssaTransState:
         print( "    Perform type-7/type-5 LSA translation" )
         print( "    nssa-translator-stability-interval is 40" )

      if self.areaAuthTypeInformation:
         if self.areaAuthTypeInformation.authProtocol == \
                'IPPROTO_ESP':
            print( "    ESP", end=' ' )
         else:
            print( "    AH", end=' ' )
         print( "authentication is enabled with", end=' ' )
         if self.areaAuthTypeInformation.hmacAlgorithm == \
                'HMAC-MD5':
            print( "HMAC-MD5 and", end=' ' )
         else:
            print( "HMAC-SHA1 and", end=' ' )
         print( "SPI %s" % self.areaAuthTypeInformation.secParamIndex )

      print( "    SPF algorithm executed %d times" % self.spfCount )

      if self.areaFiltersConfigured:
         print( "    Area summary filters configured" )
      
      if self.areaFilterPrefixList:
         print( "    Area summary filter prefix-list %s" %
                self.areaFilterPrefixList )

      printRangeListHeader = True
      for _, rangeModel in self.rangeList:
         if printRangeListHeader is True:
            print( "    Area ranges are" )
            printRangeListHeader = False
         rangeModel.render()

class Ospf3ShowInstAreaModel( Ospf3ShowInstAreaModelBase ):
   rangeList = GeneratorDict( keyType=Ip6Address,
                              valueType=Ospf3ShowInstAreaRangeModel,
                              help='List of address ranges in this OSPFv3 area' )

   def processData( self, data ):
      return Ospf3ShowInstAreaModelBase.processData( self, data )

   def render( self ):
      return Ospf3ShowInstAreaModelBase.render( self )

class Ospf3MultiAfShowInstAreaModel( Ospf3ShowInstAreaModelBase ):
   rangeList = GeneratorDict( keyType=IpGenericAddress,
                              valueType=Ospf3MultiAfShowInstAreaRangeModel,
                              help='List of address ranges in this OSPFv3 area' )

   def processData( self, data ):
      return Ospf3ShowInstAreaModelBase.processData( self, data )

   def render( self ):
      return Ospf3ShowInstAreaModelBase.render( self )

class Ospf3ShowInstSpfThrottleTimersInfoModel( Model ):
   spfStartInterval = Int( help='SPF start interval in msec' )
   spfHoldInterval = Int( help='SPF hold interval in msecs' )
   spfCurrHoldInterval = Int( help='SPF current hold interval in msecs' )
   spfMaxWaitInterval = Int( help='SPF max wait interval in msecs' )
   lastSpf = Int( help='Last SPF run (secs)' )
   nextSpf = Int( help='Next SPF run (msecs)' )

class Ospf3ShowInstLsaThrottleTimersInfoModel( Model ):
   lsaArrivalInterval = Int( help='LSA arrival interval in msecs' )
   lsaStartInterval = Int( help='LSA start interval in msecs' )
   lsaHoldInterval = Int( help='LSA hold interval in msecs' )
   lsaMaxWaitInterval = Int( help='LSA max wait interval in msecs' )
   numLsa = Int( help='Number of LSAs' )

class Ospf3ShowInstMaxMetricTypeInfoModel( Model ):
   maxMetricCondition = Bool( default=False, help='Originating router LSA '
                              'with max metric, is configured or not' )
   maxMetricState = Bool( default=False, help = 'Max metric advertisements being '
                          'currently active or not')
   maxMetricWaitBgp = Bool( default=False,
                            help='Max metric while Bgp is converging' )
   maxMetricCfgDuration = Int( optional= True, help='Max metric config duration'
                               'in secs' )
   maxMetricExpiryTimestamp = Int( optional= True,
                                help='UTC time stamp of expiry of max metric'
                                   'advertisement' )
   maxMetricExternal = Int( optional=True, help='Max metric for external LSA' )
   maxMetricStub = Bool( optional=True, help='Max metric including stubs' )
   maxMetricSummary = Int( optional=True, help='Max metric for summary LSA' )

class Ospf3ShowInstMaxMetricUnsetInfoModel( Model ):
   maxMetricUnsetReason = Enum( optional=True, values=(
         'TimerExpired', 'BgpConverged',
         'BgpTimerExpired', 'UserConfig' ),
         help='Max metric unset reason' )
   maxMetricUnsetOrigDuration = Int( optional=True,
                                     help='Max metric originate duration (secs)' )
   maxMetricUnsetTimestamp = Int( optional=True,
                              help='Max metric unset timestamp' )

class Ospf3ShowInstLsaLimitInfoModel( Model ):
   lsaLimit = Int( help='Maximum number of LSAs allowed' )
   warningLimit = Int( help='LSA limit threshold for warning message' )
   exceedAction = Str( help='LSA limit exceed-action option' )
   disabledTime = Int( help='LSA limit disabled time (mins)' )
   clearTimeout = Int( help='LSA limit clear timeout (mins)' )
   allowedIncidentCount = Int( help='LSA limit allowed incident count' )
   currentIncidentCount = Int( help='LSA limit current incident count' )
   disabled = Bool( default=False, help='disabled due to LSA limit' )
   disabledTimeEnd = Float( optional=True, help='LSA limit disabled time end' )

   def processModel( self, data ):
      instanceId = data.get( 'instanceId' )      
      instanceConfig = getInstanceConfigFromKeys( data[ 'vrf' ], instanceId )
      globalInstanceConfig = getInstanceConfigFromKeys( data[ 'vrf' ], 255 ) 
      
      # configured either through the instance or global config
      if instanceConfig.lsaLimitConfigPresent:
         lsaLimitConfig = instanceConfig.lsaLimitConfig
      else:
         lsaLimitConfig = globalInstanceConfig.lsaLimitConfig

      self.lsaLimit = lsaLimitConfig.lsaLimit
      self.warningLimit = lsaLimitConfig.lsaLimitWarningLimit
      self.exceedAction = 'disable'
      if lsaLimitConfig.lsaLimitExceedAction == 'exceedActionWarn':
         self.exceedAction = 'warn'
      self.disabledTime = lsaLimitConfig.lsaLimitDisabledTime
      self.clearTimeout = lsaLimitConfig.lsaLimitClearTimeout
      self.allowedIncidentCount = lsaLimitConfig.lsaLimitIncidentCount
    
      status = ospf3Status()
      vrf = data.get( 'vrf', instanceConfig.vrfName )
      vrfStatus = status.get( vrf, None )
      if vrfStatus is None:
         return
      instanceStatus = vrfStatus.instanceStatus.get( instanceId, None )
      if instanceStatus is None:
         return
      self.currentIncidentCount = instanceStatus.lsaLimitIncidentCount
      if instanceStatus.lsaLimitDisabled:
         self.disabled = True
         if ( instanceStatus.lsaLimitIncidentCount <=
               lsaLimitConfig.lsaLimitIncidentCount ):
            self.disabledTimeEnd = \
               instanceStatus.lsaLimitDisabledTimeEnd - Tac.now()
      else:
         self.disabled = False

class Ospf3SecondaryDomainId( Model ):
   domainId = Str( help='Secondary domain identifier' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF3_INSTANCE_SUMMARY_VPN_SECONDARY_DOMAIN_ID

   def processData( self, data ):
      self.domainId = data.pop( 'vpnSecondaryDomainId' )

   def render( self ):
      print( "  Domain Identifier: %s" % self.domainId )

class Ospf3ShowInstModelBase( Model ):
   instanceId = Int( help='OSPFv3 instance ID', optional=True )
   lsaLimitInformation = Submodel( valueType=Ospf3ShowInstLsaLimitInfoModel, 
                                      help='LSA limit information' ) 
   routerId = Ip4Address( optional=True, help='OSPFv3 Router id' )
   referenceBandwidth = Int( optional=True, help='Auto-cost reference bandwidth '
                                                 'in Mbps' )
   asbrInstance = Bool( default=False,
                        help='This OSPFv3 instance is an AS Border Router' )
   abrInstance = Bool( default=False,
                       help='This OSPFv3 instance is an Area Border Router' )
   spfInformation = Submodel( valueType=Ospf3ShowInstSpfThrottleTimersInfoModel, 
                              optional=True, help='SPF information' )

   lsaInformation = Submodel( valueType=Ospf3ShowInstLsaThrottleTimersInfoModel,
                              optional=True, help='LSA information' )
   vrf = Str( optional=True, help='VRF name' )
   maxMetricTypeInformation = Submodel( valueType=
                              Ospf3ShowInstMaxMetricTypeInfoModel,
                              optional=True, help='Max metric type information' )

   maxMetricUnsetInformation = Submodel(
      valueType=Ospf3ShowInstMaxMetricUnsetInfoModel,
      optional=True, help='Max metric unset information' )

   numAreas = Int( default=0, help='Number of areas' )
   numNormalAreas = Int( default=0, help='Number of normal areas' )
   numStubAreas = Int( default=0, help='Number of stub areas' )
   numNssaAreas = Int( default=0, help='Number of nssa areas' )
   numNeighbors = Int( default=0, help='Number of neighbors' )
   adjacencyExchangeStartThreshold = Int( optional=True,
                                          help='Adjacency exchange-start threshold' )
   ecmpMaximumNexthops = Int( optional=True,
                              help='Maximum number of ECMP nexthops' )
   gracefulRestartInfo = Submodel( valueType=OspfShowInstGracefulRestartInfoModel,
                                   optional=True,
                                   help='Graceful restart information' )
   shutDown = Bool( default=False,
                    help='Indicates if OSPFv3 instance is currently shut down' )

   floodPacing = Int( default=0, help='Flood packet pacing timer in msecs' )
   fipsStatus = Bool( default=False,
                      help='Indicates if FIPS mode is successfully enabled' )
   numBackboneNeighbors = Int( default=0, help='Number of backbone neighbors' )
   primaryDomainId = Str( optional=True,
                          help='Primary domain identifier' )

   def processData( self, data ):
      lsaLimitInfo = Ospf3ShowInstLsaLimitInfoModel()
      lsaLimitInfo.processModel( data )
      data[ 'lsaLimitInformation' ] = lsaLimitInfo

      instanceId = data[ 'instanceId' ] if 'instanceId' in data else None
      instanceConfig = getInstanceConfigFromKeys( data[ 'vrf' ], instanceId )
      globalInstanceConfig = getInstanceConfigFromKeys( data[ 'vrf' ], 255 )
      status = ospf3Status()
      instanceStatus = status.get( data.get( 'vrf', instanceConfig.vrfName ), None ) 
 
      fipsSuccessful = instanceStatus.fipsSuccessful if instanceStatus else False
      self.fipsStatus = fipsSuccessful

      if 'routerId' in data:
         if 'vrf' in data:
            data[ 'vrf' ] = data.pop( 'vrf' )
         if 'asbr' in data:
            data[ 'asbrInstance' ] = ord( data[ 'asbr' ] ) != 0
         if 'abr' in data:
            data[ 'abrInstance' ] = ord( data[ 'abr' ] ) != 0

         spfInfo = Ospf3ShowInstSpfThrottleTimersInfoModel()
         spfInfo.lastSpf = data.pop( 'lastSpf' )
         spfInfo.nextSpf = data.pop( 'nextSpf' )
         spfInfo.spfStartInterval = data.pop( 'spfStartInterval' )
         spfInfo.spfHoldInterval = data.pop( 'spfHoldInterval' )
         spfInfo.spfCurrHoldInterval = data.pop( 'spfCurrentHoldInterval' )
         spfInfo.spfMaxWaitInterval = data.pop( 'spfMaxWaitInterval' )
         data[ 'spfInformation' ] = spfInfo

         lsaInfo = Ospf3ShowInstLsaThrottleTimersInfoModel()
         lsaInfo.lsaArrivalInterval = data.pop( 'lsaArrivalInterval' )
         lsaInfo.lsaStartInterval = data.pop( 'lsaStartInterval' )
         lsaInfo.lsaHoldInterval = data.pop( 'lsaHoldInterval' )
         lsaInfo.lsaMaxWaitInterval = data.pop( 'lsaMaxWaitInterval' )
         lsaInfo.numLsa = data.pop( 'numLsa' )
         data[ 'lsaInformation' ] = lsaInfo

         if 'maxMetricCondition' in data:
            maxMetricTypeInfo = Ospf3ShowInstMaxMetricTypeInfoModel()
            maxMetricTypeInfo.maxMetricCondition = \
                ord( data.pop( 'maxMetricCondition' )) != 0
            if 'maxMetricState' in data:
               maxMetricTypeInfo.maxMetricState = \
                   ord(data.pop( 'maxMetricState' )) != 0
            if 'maxMetricCfgDuration' in data:
               maxMetricTypeInfo.maxMetricCfgDuration = \
                   data.pop( 'maxMetricCfgDuration' )
            if 'maxMetricRemainingTime' in data:
               maxMetricTypeInfo.maxMetricExpiryTimestamp = \
                   data.pop( 'maxMetricRemainingTime' )
            if 'maxMetricWaitBgp' in data:
               maxMetricTypeInfo.maxMetricWaitBgp = \
                   ord(data.pop( 'maxMetricWaitBgp' ) ) != 0
            if 'maxMetricStub' in data:
               maxMetricTypeInfo.maxMetricStub = \
                   ord(data.pop( 'maxMetricStub' ) ) != 0
            if 'maxMetricExternal' in data:
               maxMetricTypeInfo.maxMetricExternal = \
                    data.pop( 'maxMetricExternal')
            if 'maxMetricSummary' in data:
               maxMetricTypeInfo.maxMetricSummary = \
                   data.pop( 'maxMetricSummary')
            data['maxMetricTypeInformation' ] = maxMetricTypeInfo

            if 'maxMetricUnsetTime' in data:
               maxMetricUnsetInfo = Ospf3ShowInstMaxMetricUnsetInfoModel()
               maxMetricUnsetInfo.maxMetricUnsetTimestamp = \
                   data.pop( 'maxMetricUnsetTime' )
               if 'maxMetricUnsetReason' in data:
                  maxMetricUnsetInfo.maxMetricUnsetReason = \
                      maxMetricUnsetReason[ ord(data.pop( 'maxMetricUnsetReason' )) ]
               if 'maxMetricOrigDuration' in data:
                  maxMetricUnsetInfo.maxMetricUnsetOrigDuration = \
                      data.pop( 'maxMetricOrigDuration' )
               data['maxMetricUnsetInformation'] = maxMetricUnsetInfo

         # For multiAf, graceful restart is configured in the
         # global instance OR in the af instance(s)
         gracefulRestartInfo = OspfShowInstGracefulRestartInfoModel()
         if instanceConfig.gracefulRestart:
            gracefulRestartInfo.gracefulRestart = instanceConfig.gracefulRestart
            gracefulRestartInfo.gracePeriod = instanceConfig.grGracePeriod
            gracefulRestartInfo.plannedOnly = instanceConfig.grPlannedOnly
         elif globalInstanceConfig and globalInstanceConfig.gracefulRestart:
            gracefulRestartInfo.gracefulRestart = \
               globalInstanceConfig.gracefulRestart
            gracefulRestartInfo.gracePeriod = \
               globalInstanceConfig.grGracePeriod
            gracefulRestartInfo.plannedOnly = \
               globalInstanceConfig.grPlannedOnly
         if instanceConfig.grHelper:
            gracefulRestartInfo.helperMode = instanceConfig.grHelper
            gracefulRestartInfo.helperLooseLsaCheck = \
               instanceConfig.grHelperLooseLsaChk
         elif globalInstanceConfig and globalInstanceConfig.grHelper:
            gracefulRestartInfo.helperMode = \
               globalInstanceConfig.grHelper
            gracefulRestartInfo.helperLooseLsaCheck = \
               globalInstanceConfig.grHelperLooseLsaChk

         if 'grState' in data:
            gracefulRestartInfo.state = GR_STATE_MAP[ data.pop( 'grState' ) ]
         if 'grTimeRemaining' in data:
            gracefulRestartInfo.restartExpirationTime  = \
               data.pop( 'grTimeRemaining' ) + Tac.utcNow() 
         if 'grLastExitReason' in data:
            gracefulRestartInfo.lastExitReason = \
               GR_SPEAKER_EXIT_REASON_MAP[ data.pop( 'grLastExitReason' ) ]
         if 'grLastExitTime' in data:
            # time reported is seconds since last exit
            gracefulRestartInfo.lastExitTime = Tac.utcNow() - \
               data.pop( 'grLastExitTime' )
         if 'grLastSpecificReason' in data:
            gracefulRestartInfo.lastRestartReason = \
               GR_RESTART_REASON_MAP[ data.pop( 'grLastSpecificReason' ) ]
         if 'grDuration' in data:
            gracefulRestartInfo.restartDuration = data.pop ('grDuration')

         data[ 'gracefulRestartInfo' ] = gracefulRestartInfo

         if 'numberAreas' in data:
            data['numAreas'] = data.pop( 'numberAreas')

      return data

   def render( self ):
      if self.shutDown:
         print( '  Process is disabled due to shutdown command' )
         return
      if self.fipsStatus:
         print( '  FIPS mode enabled' )
      else:
         print( '  FIPS mode disabled' )
      if self.lsaLimitInformation:
         print( '  Maximum number of LSAs allowed %s' % (
            self.lsaLimitInformation.lsaLimit ) )
         print( '    Exceed action %s ' %
                ( self.lsaLimitInformation.exceedAction ) )
         print( "    LSA limit for warning message %d%%" %
                self.lsaLimitInformation.warningLimit )
         print( '    Disabled-time %d minutes, clear timeout %d minutes' % (
               self.lsaLimitInformation.disabledTime,
               self.lsaLimitInformation.clearTimeout ) )
         print( '    Incident count {}, incident count limit {}'.format(
               self.lsaLimitInformation.currentIncidentCount,
               self.lsaLimitInformation.allowedIncidentCount ) )
         if self.lsaLimitInformation.disabled:
            if self.lsaLimitInformation.disabledTimeEnd is None:
               print( '    Permanently ignoring all neighbors due to LSA',
                      'limit' )
            else:
               print( '    Ignoring all neighbors due to LSA limit,',
                     'time remaining: %s' % formatTimeInterval(
                  self.lsaLimitInformation.disabledTimeEnd ) )

      if self.routerId:
         if self.asbrInstance:
            print( "  It is an autonomous system boundary router", end=' ' )
         else:
            print( "  It is not an autonomous system boundary router", end=' ' )

         if self.abrInstance:
            print( "and is an area border router" )
         else:
            print( "and is not an area border router" )

         print( "  Minimum LSA arrival interval %s msecs" %
                self.lsaInformation.lsaArrivalInterval )
         print( "  Initial LSA throttle delay %s msecs" %
                self.lsaInformation.lsaStartInterval )
         print( "  Minimum hold time for LSA throttle %s msecs" %
                self.lsaInformation.lsaHoldInterval )
         print( "  Maximum wait time for LSA throttle %s msecs" %
                self.lsaInformation.lsaMaxWaitInterval )

         if self.floodPacing:
            print( "  Interface flood pacing timer %d msecs" %
                   self.floodPacing )

         print( "  It has %s fully adjacent neighbors" %
                self.numNeighbors )
         print( "  Number of areas in this router is %d." %
                self.numAreas, end=' ' )
         print( "%d normal, %d stub, %d nssa" % (
            self.numNormalAreas, self.numStubAreas, self.numNssaAreas ) )

         print( "  Number of LSAs %d" % self.lsaInformation.numLsa )         
         if self.referenceBandwidth:
            print( "  Reference bandwidth is %d Mbps" % (
               self.referenceBandwidth ) )

         if self.maxMetricTypeInformation:
            print( "  Originating router-LSAs with maximum metric" )
            if self.maxMetricTypeInformation.maxMetricWaitBgp:
               if self.maxMetricTypeInformation.maxMetricState:
                  print( "     Condition: on startup while BGP is converging, "
                         "State: active" )
               else:
                  print( "     Condition: on startup while BGP is converging, "
                         "State: inactive" )
            elif self.maxMetricTypeInformation.maxMetricCfgDuration:
               if self.maxMetricTypeInformation.maxMetricState:
                  if self.maxMetricTypeInformation.maxMetricExpiryTimestamp:
                     print( "     Condition: on startup for %s seconds, "
                            "State: active" %
                            self.maxMetricTypeInformation.maxMetricCfgDuration )
                     print( "     , time left: %s secs" %
                            self.maxMetricTypeInformation.maxMetricExpiryTimestamp )
                  else:
                     print( "     Condition: on startup for %s seconds, "
                            "State: active" %
                            self.maxMetricTypeInformation.maxMetricCfgDuration )
               else:
                  print( "     Condition: on startup for %s seconds, "
                         "State: inactive" %
                         self.maxMetricTypeInformation.maxMetricCfgDuration )
            else:
               print( "     Condition: always, State: active" )

            if self.maxMetricTypeInformation.maxMetricStub:
               print( "     Advertise stub links with maximum metric"
                      " in router-LSAs" )
            if self.maxMetricTypeInformation.maxMetricSummary:
               print( "     Advertise summary-LSAs with metric %s" %
                      self.maxMetricTypeInformation.maxMetricSummary )

            if self.maxMetricTypeInformation.maxMetricExternal:
               print( "     Advertise external-LSAs with metric %s" %
                      self.maxMetricTypeInformation.maxMetricExternal )

            if self.maxMetricUnsetInformation:
               print( "     Unset reason:", end=' ' )
               if self.maxMetricUnsetInformation.maxMetricUnsetReason == \
                      'TimerExpired':
                  print( "timer expired,", end=' ' )
               elif self.maxMetricUnsetInformation.maxMetricUnsetReason == \
                      'BgpConverged':
                  print( "BGP Converged,", end=' ' )
               elif self.maxMetricUnsetInformation.maxMetricUnsetReason == \
                      'BgpTimerExpired':
                  print( "BGP default timer expired,", end=' ' )
               else:
                  print( "Unknown,", end=' ' )
               print( "Originated for: %d seconds" %
                      self.maxMetricUnsetInformation.maxMetricUnsetOrigDuration )
               print( "     Unset time: %s" % formatTimeInterval(
                  self. maxMetricUnsetInformation.maxMetricUnsetTimestamp ) )

         print( "  Initial SPF schedule delay %s msecs" %
                self.spfInformation.spfStartInterval )
         print( "  Minimum hold time between two consecutive SPFs %d msecs" %
                self.spfInformation.spfHoldInterval )
         print( "  Current hold time between two consecutive SPFs %d msecs" %
                self.spfInformation.spfCurrHoldInterval )
         print( "  Maximum wait time between two consecutive SPFs %d msecs" %
                self.spfInformation.spfMaxWaitInterval )
         print( "  SPF algorithm last executed %s ago" % formatTimeInterval(
            self.spfInformation.lastSpf ) )
         if self.spfInformation.nextSpf:
            if ( self.spfInformation.nextSpf // NUM_MS_PER_SEC ) > 0:
               print( "  Scheduled SPF in %d secs" % (
                  self.spfInformation.nextSpf // NUM_MS_PER_SEC ) )
            else:
               print( "  Scheduled SPF in %d msecs" %
                      self.spfInformation.nextSpf )
         else:
            print( "  No scheduled SPF" )
         if self.adjacencyExchangeStartThreshold:
            print( "  Adjacency exchange-start threshold is %d"
                   % self.adjacencyExchangeStartThreshold )
         if self.ecmpMaximumNexthops:
            print( "  Maximum number of next-hops supported in ECMP is %d"
                   % self.ecmpMaximumNexthops )

         print( "  Number of backbone neighbors is %d"
                % self.numBackboneNeighbors )

         if self.gracefulRestartInfo:
            if not self.gracefulRestartInfo.gracefulRestart:
               print( "  Graceful-restart is not configured" )
            else:
               print( "  Graceful-restart %sis configured, grace-period %d seconds"
                     % ( "(Planned-Only) " if self.gracefulRestartInfo.plannedOnly 
                     else "", self.gracefulRestartInfo.gracePeriod ) )
            # 'gii set ospf restart' command can be used to initiate planned 
            # graceful restart, even if gr is not configured.
            if self.gracefulRestartInfo.gracefulRestart or \
               ( self.gracefulRestartInfo.state != 'none' ):
               expires = ""
               if self.gracefulRestartInfo.restartExpirationTime is not None:
                  timeSecs = int( max( 0, 
                                  self.gracefulRestartInfo.restartExpirationTime - \
                                  Tac.utcNow() ) )
                  expires = "expires in %s seconds" % ( timeSecs )

               print( "    State:", end=' ' )
               if self.gracefulRestartInfo.state == 'graceful':
                  print( "In progress, " + expires )
               elif self.gracefulRestartInfo.state == 'pending':
                  print( "Planned pending" )
               elif self.gracefulRestartInfo.state == 'exiting':
                  print( "Exiting" )
               elif self.gracefulRestartInfo.state == 'signaled':
                  print( "In progress, signaled, " + expires )
               else:
                  print( "Inactive" )

            if self.gracefulRestartInfo.lastExitReason is not None and \
               self.gracefulRestartInfo.lastExitTime is not None:
               timeSecs = int( Tac.utcNow() - self.gracefulRestartInfo.lastExitTime )
               timeHMS = formatTimeInterval( timeSecs )
               print( "    Last graceful restart completed %s ago," %
                      ( timeHMS ), end=' ' )
               print( "status:", end=' ' )
               print( grSpkrExitReasonStr[
                     self.gracefulRestartInfo.lastExitReason ] )

               if self.gracefulRestartInfo.restartDuration is not None and \
                  self.gracefulRestartInfo.lastRestartReason is not None:
                  grPlanned = grRestartTypeStr[
                     self.gracefulRestartInfo.lastRestartReason ]
                  grReason = grPlanned + " restart, reason " + \
                     self.gracefulRestartInfo.lastRestartReason

                  timeSecs = self.gracefulRestartInfo.lastExitTime - \
                             ( self.gracefulRestartInfo.restartDuration // 1000000 )
                  timeAbsolute = timestampToStr( timeSecs, False, Tac.utcNow() )

                  print( "    %s, initiated at %s, duration %1.2f secs" %
                        ( grReason, timeAbsolute, ( float
                        ( self.gracefulRestartInfo.restartDuration ) //
                              1000000 ) ) )

            if self.gracefulRestartInfo.helperMode:
               print( "  Graceful-restart-helper mode is enabled%s" % (
                  " (strict LSA checking)"
                  if not self.gracefulRestartInfo.helperLooseLsaCheck else "" ) )
            else:
               print( "  Graceful-restart-helper mode is disabled" )

         if self.primaryDomainId:
            print( "  Domain Identifier (primary): %s" % self.primaryDomainId )
         for entry in self.secondaryDomainIds:
            entry.render()

         for _, areaModel in self.areaList:
            areaModel.render()
   
      print( '' )

class Ospf3ShowInstModel( Ospf3ShowInstModelBase ):
   processId = Int( help='OSPFv3 process ID' )
   secondaryDomainIds = GeneratorList( valueType=Ospf3SecondaryDomainId,
                                       optional=True,
                                       help='List of secondary domain identifiers' )
   areaList = GeneratorDict( optional=True, keyType=Ip4Address,
                             valueType=Ospf3ShowInstAreaModel,
                             help='List of areas in this OSPFv3 instance' )

   def processData( self, data ):
      data = Ospf3ShowInstModelBase.processData( self, data )
      if 'routerId' in data and 'vrf' in data:
         data[ 'processId' ] = getProcessIdFromVrfName( vrf=data[ 'vrf' ] )
      return data

   def render( self ):
      # If shutdown is enabled, it is possible that routerId will not be available
      # due to being neither explicitly configured nor available from gated,
      # so avoid printing it in this case.
      if self.shutDown:
         print( "Routing Process \"ospfv3 %s\" with Instance %d VRF %s" % (
            self.processId, self.instanceId, self.vrf ) )
      elif self.routerId:
         print( "Routing Process \"ospfv3 %s\" with ID %s "
             "and Instance %d VRF %s" % (
            self.processId, self.routerId, self.instanceId, self.vrf ) )
      Ospf3ShowInstModelBase.render( self )

class Ospf3MultiAfShowInstModel( Ospf3ShowInstModelBase ):
   secondaryDomainIds = GeneratorList( valueType=Ospf3SecondaryDomainId,
                                       optional=True,
                                       help='List of secondary domain identifiers' )
   areaList = GeneratorDict( optional=True, keyType=Ip4Address,
                             valueType=Ospf3MultiAfShowInstAreaModel,
                             help='List of areas in this OSPFv3 instance' )

   def processData( self, data ):
      return  Ospf3ShowInstModelBase.processData( self, data )

   def renderModel( self, af ):
      # see Ospf3ShowInstModel,render()
      if self.shutDown:
         print( "OSPFv3 address-family %s" % af )
         print( "Routing Process \"ospfv3\" Instance %d VRF %s" % (
                self.instanceId, self.vrf ) )
      elif self.routerId:
         print( "OSPFv3 address-family %s" % af )
         print( "Routing Process \"ospfv3\" with ID %s "
               "and Instance %d VRF %s" % (
                self.routerId, self.instanceId, self.vrf ) )
      Ospf3ShowInstModelBase.render( self )

#-------------------------------------------------------------------------------
# "show ipv6 ospf interface"
#-------------------------------------------------------------------------------

INTERFACE_STATE_MAP = ReversibleDict( { 'down' : "Down", 
                                        'loopback' : "Loopback",
                                        'waiting' : "Waiting",
                                        'p2p' : "P2P",
                                        'dr' : "DR",
                                        'backupDr' : "Backup DR",
                                        'drOther' : "DR Other" } )

BFD_STATE_MAP       = { 0 : 'adminDown',
                        1 : 'init',
                        2 : 'down',
                        3 : 'up' }

BFD_CONFIG_STATE_MAP = { 'process' : { 0 : 'default',
                                       1 : 'enabled',
                                       2 : 'disabled' },
                         'render' : { 'default' : '',
                                      'enabled' : ', BFD Enabled',
                                      'disabled' : ', BFD Disabled' } }

PACKET_OPTIONS_MAP = { 'V6' : "ipv6",
                       'E' : "nonStubArea",
                       'MC' : "multicastForwarding",
                       'N' : "nssa",
                       'R' : "activeRouter",
                       'DC' : "demandCircuit",
                       'AF' : "addressFamily" }

PACKET_OPTIONS_BITMASK_MAP = { 1 : "ipv6",
                               2 : "nonStubArea",
                               4 : "multicastForwarding",
                               8 : "nssa",
                               16 : "activeRouter",
                               32 : "demandCircuit",
                               256 : "addressFamily" }

INTERFACE_TYPE_MAP  = ReversibleDict( { 'broadcast' : "Broadcast",
                                        'p2p' : "Point-To-Point",
                                        'nbma' : "Non-Broadcast Multi-Access",
                                        'p2mp' : "Point-to-Multipoint",
                                        'virtual' : "Virtual" } )

INTERFACE_AUTH_PROTO_MAP = { 50 : "ESP",
                             51 : "AH" }

INTERFACE_AUTH_HASH_MAP = { 1 : "HMAC-MD5",
                            2 : "HMAC-SHA1" }

interfaceStateHelp = '''OSPFv3 interface state.
  down      - Initial interface state.
  loopback  - Interface is looped back.
  waiting   - Identity determination of designated routers underway.
  p2p       - Interface connects to a point-to-point network or virtual link.
  dr        - Router is the designated router on the attached network.
  backupDr  - Router is the backup designated router on the network.
  drOther   - Interface is neither DR or BDR, and connects to a DR or BDR.'''

interfaceTypeHelp = '''OSPFv3 interface type.
  broadcast   - Broadcast interface.
  nbma        - Non-Broadcast Multi-Access interface.
  virtual     - Virtual interface. 
  p2p         - Point-to-point interface.
  p2mp        - Point-to-multipoint interface.'''

authProtocolHelp = '''IP Security Protocol.
 AH           - Authentication Header
 ESP          - Encapsulating Security Payload.'''

class Ospf3IntfOptions( Model ):
   ipv6 = Bool( default=False, help='Include in IPV6 Route Calculations' )
   nonStubArea = Bool( default=False, help='Attached to a non-stub area' )
   multicastForwarding = Bool( default=False, 
                               help='IP Multicast datagram forwarding' )
   nssa = Bool( default=False, help='Attached to a Not-so-stubby-area' )
   activeRouter = Bool( default=False, help='Originator is an Active-router' )
   demandCircuit = Bool( default=False, help='Demand Circuit Handling' )
   addressFamily = Bool( default=False, optional=True,
                         help='Router supports address families' )

   def render( self ):
      optionsStr = ''
      for key, option in PACKET_OPTIONS_MAP.items():
         if getattr( self, option ):
            optionsStr +=  key + ' '
      print( "    Options: %s" % ( optionsStr.strip() if optionsStr else '(null)' ) )

   def processData( self, data ):
      if 'options' in data:
         optionsPresent = data.pop( 'options' ).split()
         for option in optionsPresent:
            setattr( self, PACKET_OPTIONS_MAP[ option ], True )

   def processBitmask( self, optionMask ):
      for bit, value in PACKET_OPTIONS_BITMASK_MAP.items():
         if bit & optionMask:
            setattr( self, value, True )

class Ospf3IntfAuthInfo( Model ):
   algorithm = Enum( values=( 'HMAC-MD5', 'HMAC-SHA1'), 
                     help='Authentication HMAC Algorithm' )
   protocol = Enum( values=( 'AH', 'ESP' ), help=authProtocolHelp )
   spi = Int( help='Security Parameter Index' )

class Ospf3InterfaceEntry( Model ):
   vrf = Str( help='VRF name' )
   state = Enum( values=( 'down', 'loopback', 'waiting', 'p2p', 'dr', 
                          'backupDr', 'drOther' ), help=interfaceStateHelp )
   priority = Int( help='Interface priority' )
   cost = Int( help='Interface metric' ) 
   neighborCount = Int( help='Neighbor count' ) 
   passive = Bool( default=False, help='Passive state' )
   designatedRouter = Ip4Address( help='Designated router ID' ) 
   backupDesignatedRouter = Ip4Address( help='Backup designated router ID' )
   helloInterval = Int( help='hello-interval in seconds' ) 
   deadInterval = Int( help='Router dead interval in seconds' ) 
   retransmitInterval = Int( help='Retransmit interval in seconds' ) 
   transmitDelay = Int( help='Transmit delay in seconds' )
   options = Submodel( help='Interface optional capabilities', 
                       valueType=Ospf3IntfOptions )
   area = Ip4Address( help='Area assigned to the interface' ) 
   interfaceAddress = IpGenericAddress( help='Interface ipv6 link-local address' )
   interfaceV4Address = Ip4Address( optional=True,
                                    help='Interface ipv4 primary address' ) 
   interfaceV4MaskLen = Int( optional=True,
                            help='Interface ipv4 primary address'
                            ' prefix len' ) 
   interfaceType = Enum( values=( 'broadcast', 'p2p', 'nbma', 'p2mp', 
                                  'virtual'), help=interfaceTypeHelp )
   authentication = Submodel( optional=True, 
                              help='Interface authentication details', 
                              valueType=Ospf3IntfAuthInfo )
   bfdState = Enum( values=( 'default', 'enabled', 'disabled' ),
                    help='BFD configuration state')
   remoteAddress = Ip6Address( optional=True,
                               help='Interface remote IPv6 address' )

   def getKey( self, data ):
      assert data[ 'interfaceName' ]
      ifName = getOspf3Interface( data[ 'interfaceName' ] )
      data.pop( 'interfaceName' )
      return ifName

   def processData( self, data ):
      self.options = Ospf3IntfOptions()
      self.options.processData( data )

      if getProcessIdFromVrfName( vrf=data[ 'vrf' ] ) :
         self.options.addressFamily = None

      if data[ 'state' ]:
         revIntfStateMap = INTERFACE_STATE_MAP.reverse()
         data[ 'state' ] = revIntfStateMap[ data[ 'state' ] ]
      
      if data[ 'interfaceType' ]:
         revIntfTypeMap = INTERFACE_TYPE_MAP.reverse()
         data[ 'interfaceType' ] = revIntfTypeMap[ data[ 'interfaceType' ] ]
      
      data [ 'bfdState' ] = BFD_CONFIG_STATE_MAP[ 'process' ][ data[ 'bfdState' ] ]

      if 'passive' in data:
         self.passive = ord( data.pop( 'passive' ) ) != 0

      if 'security-enabled' in data:
         authInfo = Ospf3IntfAuthInfo()
         authInfo.protocol = INTERFACE_AUTH_PROTO_MAP[ data[ 'security-proto' ] ]
         authInfo.algorithm = INTERFACE_AUTH_HASH_MAP[ data[ 'hmac-algo' ] ] 
         authInfo.spi = data[ 'spi' ]
         data[ 'authentication' ] = authInfo
      else:
         data[ 'authentication' ] = None

      return data

   def renderEntry( self, ifname ):
      options = getOptionsStr( self.options, PACKET_OPTIONS_MAP )
      if 'OSPF3_SL' in ifname:
         status = 'up' if INTERFACE_STATE_MAP[ self.state ] != 'Down' else 'down'
         print( 'Sham Link {} to address {} is {}'.format(
                ifname, self.remoteAddress, status ) )
      else:
         print( "%s is up" % ( ifname ) )
      if self.interfaceAddress:
         print( "  Interface Address {}, VRF {}, Area {} ".format(
            self.interfaceAddress, self.vrf, self.area ) )
      else:
         print( "  Interface Address unassigned, VRF {}, Area {}".format(
               self.vrf, self.area ) )
      if self.interfaceV4Address:
         print( "  Interface IPv4 Primary Address {}/{}".format(
            self.interfaceV4Address, self.interfaceV4MaskLen ) )
      print( "  Network Type %s, Cost %d " % ( 
            INTERFACE_TYPE_MAP[ self.interfaceType ], self.cost ) )
      bfdString = BFD_CONFIG_STATE_MAP[ 'render' ][ self.bfdState ]
      if self.designatedRouter is not None:
         print( "  Transmit Delay is %d sec, State %s, Priority %d%s"  % ( 
               self.transmitDelay, INTERFACE_STATE_MAP[ self.state ], self.priority, 
               bfdString ) )
         print( "  Designated Router is %s " % ( self.designatedRouter ) )
      else:
         print( "  Transmit Delay is %d sec, State %s%s"  % ( 
               self.transmitDelay, INTERFACE_STATE_MAP[ self.state ], bfdString ) )
         print( "  No Designated Router on this network" )
      if self.backupDesignatedRouter is not None:
         print( "  Backup Designated Router is %s " %
                ( self.backupDesignatedRouter ) )
      else:
         print( "  No Backup Designated Router on this network" )
      print( "  Timer intervals configured, Hello %d, Dead %d, Retransmit %d " % ( 
            self.helloInterval, self.deadInterval, self.retransmitInterval ) )
      print( "  Neighbor Count is %s%s" %
            ( self.neighborCount," (Passive Interface)"
            if self.passive else "" ) )
      if self.authentication:
         print( "  %s authentication is enabled with %s and SPI %d " % (
               self.authentication.protocol, self.authentication.algorithm,
               self.authentication.spi ) )
      if options != '0':
         print( "  Options are %s" % options )

class Ospf3Interface( Model ):
   indices = GeneratorDict( keyType=str, valueType=Ospf3InterfaceEntry,
                            help='Dictionary of OSPFv3 Interfaces indexed by the '
                            'interface name' )

   def render( self ):
      for ifName, intf in self.indices:
         intf.renderEntry( ifName )

class Ospf3MultiAfInterface( Model ):
   interfaces = GeneratorDict( keyType=str,
                               valueType=Ospf3InterfaceEntry,
                               help='Dictionary of OSPFv3 interfaces indexed '
                                    'by the interface name' )

   def renderModel( self, af ):
      firstIntf = True
      for ifName, intf in self.interfaces:
         if firstIntf:
            print( "OSPFv3 address-family %s" % af )
            firstIntf = False
         intf.renderEntry( ifName )
      if not firstIntf:
         print()

#-------------------------------------------------------------------------------
# "show ipv6 ospf [processId] neighbor [vrf vrf-name|default|all]"
# "show ipv6 ospf [processId] neighbor summary [vrf vrf-name|default|all]"
# "show ipv6 ospf [processId] neighbor state abc [vrf vrf-name|default|all]"
#-------------------------------------------------------------------------------

# The neighbor state dictionary has to be synced with gated/ospf3/ospf3_ngb.h
# and gated/ospf3/ospf3_ngb.c.
ADJACENCY_STATE_MAP = ReversibleDict( { 'down'      : 'Down',
                                        'attempt'   : 'Attempt',
                                        'init'      : 'Init',
                                        '2Ways'     : '2 Ways',
                                        'exchStart' : 'Exch Start',
                                        'exchange'  : 'Exchange',
                                        'loading'   : 'Loading',
                                        'full'      : 'Full',
                                        'restart'   : 'Graceful Restart' } )

NEIGHBOR_OPTIONS_MAP = { 'V6' : 'doNotUseInRouteCalc',
                         'E'  : 'externalRoutingCapability',
                         'MC' : 'multicastCapability',
                         'N'  : 'nssaCapability',
                         'R'  : 'activeRouter',
                         'DC' : 'demandCircuitsSupport',
                         'AF' : 'addressFamily' }

adjacencyStateHelp = '''OSPFv3 neighbor adjacency state.
  down         -   No hello received from neighbor
  attempt      -   Router attempting to establish connection 
  init         -   Hello received but two-way communication not established
  2-ways       -   Bi-directional communication established between two routers
  exch-start   -   Exchange of link state information between   
                    routers and their DR and BDR are started
  exchange     -   Routers exchanging database descriptor packets
  loading      -   Exchange of link state information occuring
  full         -   Routers are fully adjacent with each other
  restart      -   Neighbor is restarting'''

bfdStateHelp = '''OSPFv3 neighbor BFD state.
   adminDown - Session taken down for administrative purposes
   init - Waiting for session establishment
   up - Session established successfully
   down - Session is down or has just been created'''

def getOptionsStr( options, optionMap ):
   optionsStr = ''
   for key, option in sorted( optionMap.items() ):
      if getattr( options, option ):
         optionsStr +=  key + ' '
   if optionsStr != '':
      return optionsStr.strip()
   else:
      return '0'

class Ospf3NeighborDetails( Model ):
   bfdState = Enum( values=( 'adminDown', 'init', 'up', 'down'),
                    help=bfdStateHelp )
   bfdRequestSent = Bool( default=False,
                          help='OSPFv3 neighbor BFD request is sent' )
   grHelperTimer = Float( optional=True,
      help='Timestamp of OSPFv3 neighbor graceful period expiration' )
   stateTime = Float( help='OSPFv3 neighbor current state time' )
   grNumAttempts = Int( help='OSPFv3 neighbor graceful restart attempts' )
   grExitReason = Enum( optional=True,
                        values=tuple( GR_HELPER_EXIT_REASON_MAP.values() ),
      help='OSPFv3 neighbor graceful restart exit reason' )
   grLastRestartTime = Float( optional=True,
      help='Timestamp of the OSPFv3 neighbor last graceful restart time' )

class Ospf3NeighborOptions( Model ):
   doNotUseInRouteCalc = Bool( default=False,
                               help='A type 3,5 or 7 LSA is sent from'
                                    ' a PE router to a CE router' )
   externalRoutingCapability = Bool( default=False,
                                     help='Originating router is capable of'
                                          ' accepting AS External LSAs' )
   multicastCapability = Bool( default=False,
                               help='Originating router supports'
                                    ' multicast extensions to OSPFv3' )
   nssaCapability = Bool( default=False,
                          help='Originating router supports Type-7'
                               ' NSSA-External-LSAs' )
   activeRouter = Bool( default=False,
                        help='Originating router is active or not' )
   demandCircuitsSupport = Bool( default=False,
                                 help='Originating router supports OSPFv3' 
                                      ' over Demand Circuits' )
   addressFamily = Bool( default=False, optional=True,
                         help='Router supports address families' )


class Ospf3NeighborsEntry( Model ):
   routerId = Str( help='OSPFv3 neighbor router identifier' )
   priority = Int( help='OSPFv3 neighbor priority' )
   adjacencyState = Enum( values=( 'down', 'attempt', 'init', '2Ways',
                                   'exchStart', 'exchange', 'loading',
                                   'full', 'restart' ),
                          help=adjacencyStateHelp )
   designatedRouter = Ip4Address( help='OSPFv3 neighbor ID of '
                                             'designated router' )
   backupDesignatedRouter = Ip4Address( help='OSPFv3 neighbor ID of' 
                                              ' backup designated router' )
   options = Submodel( valueType=Ospf3NeighborOptions,
                       help='OSPFv3 neighbor options' )
   areaId = Ip4Address( help='OSPFv3 neighbor area identifier' )
   interfaceName = Str( help='OSPFv3 neighbor interface name' )
   inactivity = Float( help='Timestamp of OSPFv3 neighbor inactivity/dead timer' ) 
   _vrf = Str( help='VRF name' )
   details = Submodel( valueType=Ospf3NeighborDetails, optional=True,
                       help='OSPFv3 neighbor details' )

   def processData( self, data ):
      data[ 'interfaceName' ] = getOspf3Interface( data[ 'interfaceName' ] )

      state  = data.pop( 'state' )
      revdAdjStateMap = ADJACENCY_STATE_MAP.reverse()
      if state in revdAdjStateMap:
         data[ 'adjacencyState' ] = revdAdjStateMap[ state ]

      options = Ospf3NeighborOptions()
      if 'options' in data and data[ 'options' ]:
         optionsList = data[ 'options' ].split()
         for option in optionsList:
            setattr( options, NEIGHBOR_OPTIONS_MAP[ option ], True )         

      if getProcessIdFromVrfName( vrf=data[ 'vrf' ] ) :
         options.addressFamily = None

      data[ 'options' ] = options
      data[ 'inactivity' ] = Tac.utcNow() + data[ 'inactivity' ]
      self._vrf = data.pop( 'vrf' )
      details = Ospf3NeighborDetails()
      bfdState = data.pop( 'bfdState' )
      if bfdState in BFD_STATE_MAP:
         details.bfdState = BFD_STATE_MAP[ bfdState ]
      details.bfdRequestSent = ord( data.pop( 'bfdRequestSent' ) ) != 0
      details.stateTime = Tac.utcNow() - data.pop( 'stateTime' )

      if 'grHelperTimer' in data:
         details.grHelperTimer = data.pop( 'grHelperTimer' ) - Tac.utcNow()

      details.grNumAttempts = data.pop( 'grNumAttempts' )

      if 'grExitReason' in data:
         exitReason = data.pop( 'grExitReason' )
         if exitReason in GR_HELPER_EXIT_REASON_MAP:
            details.grExitReason = GR_HELPER_EXIT_REASON_MAP[ exitReason ]

      if 'grLastRestartTime' in data:
         lastRestartTime = data.pop( 'grLastRestartTime' )
         if lastRestartTime is not None:
            details.grLastRestartTime = Tac.utcNow() - lastRestartTime

      data[ 'details' ] = details

      return data

   def renderEntry( self ):
      state = ADJACENCY_STATE_MAP[ self.adjacencyState ]
      options = getOptionsStr( self.options, NEIGHBOR_OPTIONS_MAP )
      bfdState = getattr( self.details, 'bfdState' )
      bfdState = bfdState[ 0 ].upper() + bfdState[ 1: ]

      print( 'Neighbor %s VRF %s priority is %d, state is %s' % ( 
         self.routerId, self._vrf, self.priority, state ) )
      print( f'  In area {self.areaId} interface {self.interfaceName}' )
      
      stateTime = int( Tac.utcNow() - getattr( self.details, 'stateTime' ) )
      stateTime = formatTimeInterval( stateTime )

      if state == 'Full':
         print( '  Adjacency was established %s ago' % ( stateTime ) )
      print( '  Current state was established %s ago' % ( stateTime ) )
      
      print( '  DR is {} BDR is {}'.format( self.designatedRouter,
                                       self.backupDesignatedRouter ) )
      print( '  Options is %s' % options )
      inactivity = max( 0, int( self.inactivity - Tac.utcNow() ) )
      print( '  Dead timer is due in %s seconds' % inactivity )

      if getattr( self.details, 'bfdRequestSent' ):
         print( '  Bfd request is sent and the state is %s' % ( bfdState ) )

      if getattr( self.details, 'grHelperTimer' ):
         grHelperTimerSecs = max( 0, self.details.grHelperTimer + Tac.utcNow() )
         grHelperTimerHMS = formatTimeInterval( grHelperTimerSecs )
         print( '  Graceful-restart-helper mode is Active' )
         print( '  Graceful-restart-helper timer is due in %s' %
                ( grHelperTimerHMS ) )
      else:
         print( '  Graceful-restart-helper mode is Inactive' )

      print( '  Graceful-restart attempts: %s' % ( self.details.grNumAttempts ) )
      if getattr( self.details, 'grExitReason' ):
         print( "  Graceful-restart-helper last exit reason:", end=' ' )
         print( grHlprExitReasonStr[ self.details.grExitReason ] )

      if getattr( self.details, 'grLastRestartTime' ):
         grLastRestartTimeSecs = int( Tac.utcNow() \
                                      - self.details.grLastRestartTime )
         grLastRestartTimeHMS = formatTimeInterval( grLastRestartTimeSecs )
         print( '  Graceful-restart last attempt %s ago' %
                ( grLastRestartTimeHMS ) )

class Ospf3NeighborsInstance( Model ):
   ospf3NeighborEntries = GeneratorList( valueType=Ospf3NeighborsEntry,
                                         help='List of OSPFv3 neighbor entries' )
   _summary = Bool( default=False,
                    help='Private attribute to indicate whether summary oriented'
                         ' neighbor output is required or not' )
   _state = Str( default=None, 
                    help='Private attribute to indicate whether state oriented'
                         ' neighbor output is required or not' )
   _processId = Int( help='Private attribute for OSPFv3 Process ID' )

   def getKey( self, data ):
      if 'instanceId' in data and data[ 'instanceId' ]:
         return data.pop( 'instanceId' )
      return None

   def processData( self, data ):
      self._processId = getProcessIdFromVrfName( data[ 'vrf' ] )

   def render( self ):
      firstInst = True
      if self._summary:
         results = {}
         for value in ADJACENCY_STATE_MAP.values():
            results[ value ] = 0

      for entry in self.ospf3NeighborEntries:
         if firstInst:  
            print( 'Routing Process "ospf %s":' % self._processId )
            firstInst = False
         if not self._summary:
            if self._state is None or self._state == entry.adjacencyState:
               entry.renderEntry()
         elif self._summary:
            neighborState = ADJACENCY_STATE_MAP[ entry.adjacencyState ]
            for value in ADJACENCY_STATE_MAP.values():
               if neighborState == value:
                  results[ neighborState ] += 1
      if self._summary:
         for key, value in results.items():
            print( '%7d neighbors are in state %s' % ( value, key ) )

class Ospf3MultiAfNeighborsInstance( Model ):
   instanceId = Int( help='OSPFv3 instance ID' )
   ospf3NeighborEntries = GeneratorList( valueType=Ospf3NeighborsEntry,
                                         help='List of OSPFv3 neighbor entries' )
   _summary = Bool( default=False,
                    help='Private attribute to indicate whether summary oriented'
                         ' neighbor output is required or not' )
   _state = Str( default=None, 
                    help='Private attribute to indicate whether state oriented'
                         ' neighbor output is required or not' )

   def renderModel( self, af ):
      firstInst = True
      if self._summary:
         results = {}
         for value in ADJACENCY_STATE_MAP.values():
            results[ value ] = 0

      for entry in self.ospf3NeighborEntries:
         if firstInst:   
            print( "OSPFv3 address-family %s" % af )
            print( "Routing Process \"ospfv3\" Instance %d VRF %s" % (
                   self.instanceId, entry._vrf ) )
            firstInst = False
         if not self._summary:
            if self._state is None or self._state == entry.adjacencyState:
               entry.renderEntry()
         elif self._summary:
            neighborState = ADJACENCY_STATE_MAP[ entry.adjacencyState ]
            for value in ADJACENCY_STATE_MAP.values():
               if neighborState == value:
                  results[ neighborState ] += 1
      if self._summary:
         for key, value in results.items():
            print( '%7d neighbors are in state %s' % ( value, key ) )

#------------------------------------------------------------------------------
# "show ipv6 ospf [process-id] border-routers [vrf vrf-name|default|all]"
#------------------------------------------------------------------------------
class Ospf3BorderRoutersType( Model ):
   asbr = Bool( default=False,
                help='Router is an autonomous system boundary router' )
   abr = Bool( default=False,
               help='Router is an area border router' )

   def processData( self, data ):
      if 'asbr' in data:
         self.asbr = ord( data.pop( 'asbr' ) ) != 0
      if 'abr' in data:
         self.abr = ord( data.pop( 'abr' ) ) != 0

   def renderEntry( self, routerId, areaId ):
      asbr = 'ASBR' if self.asbr else ''
      abr = 'ABR' if self.abr else ''
      state = " ".join( [ asbr, abr ] ).lstrip()      

      print( f'  Router {routerId} area {areaId} {state}' )

class Ospf3BorderRoutersEntry( Model ):
   borderRouters = Dict( keyType=Ip4Address,
                         valueType=Ospf3BorderRoutersType,
                         help='Dictionary of OSPFv3 border routers type'
                              ' indexed by router-id' )
   _areaId = Ip4Address( help='area identifier' )

   def getKey( self, data ):
      assert 'areaId' in data and data[ 'areaId' ]
      return data[ 'areaId' ]

   def overrideHierarchy( self, data ):
      readNext = True
      if not self.borderRouters:
         self._areaId = data[ 'areaId' ]
         routerType = Ospf3BorderRoutersType()
         routerType.processData( data )
         self.borderRouters[ data[ 'routerId' ] ] = routerType
         return ( data, readNext )
      elif str( self._areaId ) == self.getKey( data ):
         routerType = Ospf3BorderRoutersType()
         routerType.processData( data )
         self.borderRouters[ data[ 'routerId' ] ] = routerType
         return ( data, readNext )
      else:
         return ( data, False )

   def renderEntry( self, areaId ):
      for routerId, routerType in self.borderRouters.items():
         routerType.renderEntry( routerId, areaId )

class Ospf3BorderRoutersInstance( Model ):
   processId = Int( help='OSPFv3 process identifier' )
   vrf = Str( help='VRF name' )
   ospf3BorderRouterEntries = GeneratorDict( keyType=Ip4Address,
                                             valueType=Ospf3BorderRoutersEntry,
                                             help='Dictionary of OSPFv3 border '
                                                  'router entries indexed by '
                                                  'area-id' )

   def processData( self, data ):
      data[ 'processId' ] = getProcessIdFromVrfName( vrf=data[ 'vrf' ] )
      return data
   
   def render( self ):
      print( 'Routing Process "ospf {}", VRF {}'.format( self.processId,
                                                    self.vrf ) )
      for areaId, entry in self.ospf3BorderRouterEntries:
         entry.renderEntry( areaId )     

class Ospf3MultiAfBorderRoutersInstance( Model ):
   instanceId = Int( help='OSPFv3 instance identifier' )
   vrf = Str( help='VRF name' )
   ospf3BorderRouterEntries = GeneratorDict( keyType=Ip4Address,
                                             valueType=Ospf3BorderRoutersEntry,
                                             help='Dictionary of OSPFv3 border '
                                                  'router entries indexed by '
                                                  'area-id' )
   
   def renderModel( self, af ):
      firstEntry = True
      for areaId, entry in self.ospf3BorderRouterEntries:
         if firstEntry:
            print( "OSPFv3 address-family %s" % af )
            print( "Routing Process \"ospfv3\" Instance %d, VRF %s" % (
               self.instanceId, self.vrf ) )
            firstEntry = False
         entry.renderEntry( areaId )     

# Models / changes to support "show ipv6 ospf database"
# Enums and string conversions for printing, values etc are derived from C headers
def printOspf3LegendInformation():
   print( "Codes: AEX - AS External, GRC - Grace," )
   print( "       IAP - Inter Area Prefix, IAR - Inter Area Router," )
   print( "       LNK - Link, NAP - Intra Area Prefix," )
   print( "       NSA - Not So Stubby Area, NTW - Network," )
   print( "       RTR - Router" )

def printOspf3LsaHeader():
   print( "%4s%15s%15s%5s%11s%11s" % ( "Type", "Link ID", "ADV Router",
                                      "Age", "Seq#", "Checksum" ) )

#Maps to o3r_database_get_lsa_type() & the corresponding definitions
LSA_TYPE_MAP = ReversibleDict( { 8         : 'linkLsa',
                                 11        : 'graceLsa',
                                 8193      : 'routerLsa',
                                 8194      : 'networkLsa',
                                 8195      : 'interAreaPrefixLsa',
                                 8196      : 'interAreaRouterLsa',
                                 8199      : 'nssaLsa',
                                 8201      : 'intraAreaPrefixLsa',
                                 16389     : 'asExternalLsa'
               } )
# Mapping for text print from python model
LSA_TEXT_TYPE_MAP = { 'linkLsa'                 : 'LNK',
                      'graceLsa'                : 'GRC',
                      'routerLsa'               : 'RTR',
                      'networkLsa'              : 'NTW',
                      'interAreaPrefixLsa'      : 'IAP',
                      'interAreaRouterLsa'      : 'IAR',
                      'nssaLsa'                 : 'NSA',
                      'intraAreaPrefixLsa'      : 'NAP',
                      'asExternalLsa'           : 'AEX' 
                    }

ROUTER_LSA_OPTIONS = { 'ABR' : 'areaBorderRouter',
                       'ASB' : 'asBoundaryRouter',
                       'VTL' : 'virtualLinkEndpoint',
                       'NT' : 'nssaTranslation' }

PREFIX_OPTIONS_MAP = { 'NU' : 'noUnicastCapability',
                       'LA' : 'localAddressCapability',
                       'MC' : 'multicastCapability',
                       'P' : 'propagateNssaPrefix',
                       'DN' : 'doNotUseInRouteCalc' }

#Maps to o3_db_vtx_exception_flag_t & corresponding strings in o3r_database_lsa_cb
EXCEPTION_TYPE_MAP = { 
         1 : 'announcingMaximumCost' , # DB_VTX_EXCEP_FLAG_MAX_LINK_COST = 1
         2 : 'announcingMaximumMetric' # DB_VTX_EXCEP_FLAG_MAX_METRIC = 2
}
EXCEPTION_TYPE_TEXT_MAP = { 
         'announcingMaximumCost'   : 'Announcing maximum link costs',
         'announcingMaximumMetric' : 'Announcing maximum metrics'
}

#Strings map to ar_ospf3_db_rtr_lnk_type (integers are protocol defined)
ROUTER_LSA_INTERFACE_MAP = { 1  : 'pointToPoint',
                             2  : 'transitNetwork',
                             3  : 'reserved',
                             4  : 'virtualLink' }
# Map to print the text from python model
ROUTER_LSA_INTERFACE_TEXT_MAP = { 'pointToPoint'  : 'P2P',
                                  'transitNetwork': 'NTW',
                                  'reserved'      : '---',
                                  'virtualLink'   : 'VRT' }

# Help strings for Ospf3RouterLsaInterfaceEntry Model
routerLsaInterfaceTypeHelp = '''Router LSA interface type.
  pointToPoint : Point-to-Point connected interface
  transitNetwork : Connected to transit network
  reserved : Reserved
  virtualLink : Virtual link
'''
class Ospf3RouterLsaInterfaceEntry ( Model ):
   interfaceType = Enum( values=( list( ROUTER_LSA_INTERFACE_MAP.values() ) ),
                         help=routerLsaInterfaceTypeHelp )
   metric = Int( help='Interface cost' )
   interfaceId = Ip4Address( help='Local interface id' )
   neighborInterfaceId = Ip4Address( help= 'Interface id of the neighbor router' )
   neighborRouterId = Ip4Address( help='Router id of the neighbor router' )

   def processData( self, data ):
      assert 'interfaceType' in data
      intfType = ord( data.pop( 'interfaceType' ) ) 
      assert intfType in ROUTER_LSA_INTERFACE_MAP
      self.interfaceType = ROUTER_LSA_INTERFACE_MAP[ intfType ]
      return data

   def renderEntry( self ):
      print( "    Intf Type: %s" % ROUTER_LSA_INTERFACE_TEXT_MAP[
      self.interfaceType ] )
      print( "    Intf Metric: %s" % self.metric )
      print( "    Intf ID: %s" % self.interfaceId )
      print( "    Neighbor Intf ID: %s" % self.neighborInterfaceId )
      print( "    Neighbor Router ID: %s" % self.neighborRouterId )

class Ospf3RouterLsaOptions( Model ):
   areaBorderRouter = Bool( default=False, help='Area border router' )
   asBoundaryRouter = Bool( default=False, help='Autonomous system boundary router' )
   virtualLinkEndpoint = Bool( default=False, 
                               help='Endpoint for virtual link(s) in the area' )
   nssaTranslation = Bool( default=False, 
                           help='Translate NSSA LSAs to AS external LSAs' )

   def render( self ):
      optionsStr = ''
      for key, option in ROUTER_LSA_OPTIONS.items():
         if getattr( self, option ):
            optionsStr +=  key + ' '
      print( "    Options: %s" % ( optionsStr.strip() if optionsStr else '(null)' ) )

   def processData( self, data ):
      if 'options' in data:
         optionsList = data.pop( 'options' ).split()
         for option in optionsList:
            setattr( self, ROUTER_LSA_OPTIONS[ option ], True )

class Ospf3RouterLsa( Model ):
   options = Submodel( valueType=Ospf3RouterLsaOptions, 
                       help='Optional capabilities supported by the router' )
   routerLsaLinks = List( valueType=Ospf3RouterLsaInterfaceEntry,
                          help='List of router links' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF3_RTR_LSA # 1

   def processData( self, data ):
      self.options = Ospf3RouterLsaOptions()
      self.options.processData( data )
      return data

   def renderEntry( self ):
      self.options.render()
      for link in self.routerLsaLinks:
         link.renderEntry()
      print()

class Ospf3NetworkLsa( Model ):
   options = Submodel( valueType=Ospf3IntfOptions, 
                       help='Optional capabilities supported by the router' )
   attachedRouters = List( valueType=Ip4Address,
                           help='List of routers attached to the link' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF3_NTW_LSA # 2

   def overrideHierarchy( self, data ):
      readNext = True
      if not self.options:
         self.options = Ospf3IntfOptions()
         self.options.processData( data )
         self.attachedRouters = []
         return ( data, readNext )
      elif 'attachedRouterId' in data:
         self.attachedRouters.append( data.pop( 'attachedRouterId' ) )
         return ( data, readNext )
      return ( data, False )

   def renderEntry( self ):
      self.options.render()
      print( "    Attach Router(s)" )

      for attachedRtr in self.attachedRouters:
         print( "        %s" %  attachedRtr )
      print()

class Ospf3LsaPrefixOptions( Model ):
   noUnicastCapability = Bool( default=False, 
                               help='Prefix excluded from IPv6 unicast'
                                    ' calculations' )
   localAddressCapability = Bool( default=False,
                                  help='Prefix scope is IPv6 link local' )
   multicastCapability = Bool( default=False,
                               help='Originating router supports'
                                    ' multicast extensions to OSPFv3' )
   propagateNssaPrefix = Bool( default=False,
                               help='Prefix propagated by NSSA area border routers' )
   doNotUseInRouteCalc = Bool( default=False,
                               help='A type 3,5 or 7 LSA is sent from'
                                    ' a PE router to a CE router' )

   def render( self ):
      optionsStr = ''
      for key, option in PREFIX_OPTIONS_MAP.items():
         if getattr( self, option ):
            optionsStr +=  key + ' '
      print( "      Options: %s" % (
            optionsStr.strip() if optionsStr else '(null)' ) )

   def processData( self, data ):
      if 'prefixOptions' in data:
         optionsPresent = data.pop( 'prefixOptions' ).split()
         for option in optionsPresent:
            setattr( self, PREFIX_OPTIONS_MAP[ option ], True )

class Ospf3LsaPrefix( Model ):
   metric = Int( optional=True, help='Prefix cost' )
   prefixLength = Int( help='Prefix length' )
   prefixOptions = Submodel( valueType=Ospf3LsaPrefixOptions,
                             help='Prefix options' )
   prefix = IpGenericAddress( help='Prefix advertised' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF3_PFX  # 1

   def processData( self, data ):
      self.prefixOptions = Ospf3LsaPrefixOptions()
      self.prefixOptions.processData( data )
      return data

   def renderEntry( self ):
      print( "  Prefix" )
      print( "      Prefix: %s" % self.prefix )
      print( "      Length: %s" % self.prefixLength )
      self.prefixOptions.render() 
      if self.metric is not None:
         print( "      Metric: %s" % self.metric )
      print()

class Ospf3InterAreaPrefixLsa ( Model ):
   metric = Int( help='Route cost' )
   prefix = Submodel( valueType=Ospf3LsaPrefix,
                      help='Prefix advertised in the LSA' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF3_IAP_LSA # 3

   def renderEntry( self ):
      print( "    Metric: %s" % self.metric )

      if self.prefix:
         self.prefix.renderEntry()

class Ospf3InterAreaRouterLsa( Model ):
   options = Submodel( valueType=Ospf3IntfOptions, 
                       help='Optional capabilities supported '
                            'by the destination router' )
   metric = Int( help='Route cost' )
   destinationRouterId = Ip4Address( 
                         help='Router id of the destination router' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF3_IAR_LSA # 4

   def processData( self, data ):
      self.options = Ospf3IntfOptions()
      self.options.processData( data )
      return data

   def renderEntry( self ):
      self.options.render()
      print( "    Metric: %s" % self.metric )
      print( "    Destination Router ID: %s" % self.destinationRouterId )
      print()

metricTypeHelp = '''External metric type (E bit)
  type1 : External metric Type 1
  type2 : External metric Type 2
'''
class Ospf3ExternalLsa ( Model ):
   metric = Int( help='Route cost' )
   externalRouteTag = Int( optional=True, 
                           help='External route tag' )
   forwardingAddress = IpGenericAddress( optional=True, 
                                         help='Forwarding address if available' )
   metricType  = Enum( values=( 'type1', 'type2' ), help=metricTypeHelp )
   prefix = Submodel( valueType=Ospf3LsaPrefix, 
                      help='Prefix adverstised in the LSA' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF3_ASE_LSA # 5

   def processData( self, data ):
      if ord( data[ 'metric_type2' ] ):
         self.metricType = 'type2'
         data.pop( 'metric_type2' )
      else:
         self.metricType = 'type1'
      return data

   def renderEntry( self ):
      print( "    Metric Type: %s" %
             ( "2" if self.metricType == 'type2' else "1" ) )
      print( "    Metric: %s" % self.metric )
      if self.forwardingAddress is not None:
         print( "    Forwarding Address: %s" % self.forwardingAddress )
      if self.externalRouteTag is not None:
         print( "    External Route Tag: %s" % self.externalRouteTag )

      if self.prefix:
         self.prefix.renderEntry()

class Ospf3IntraAreaPrefixLsa( Model ):
   referencedLsaType = Enum( values=( list( LSA_TYPE_MAP.values() ) ),
                             help='Referenced LSA type' )
   referencedLinkStateId = Ip4Address( help='Referenced link state id' )
   referencedAdvertisingRouter = Ip4Address( help='Referenced advertising router' )
   numPrefixes = Int( help='Number of prefixes in intra area prefix LSA' )
   prefixList = List( valueType=Ospf3LsaPrefix,
                      help='List of prefixes in intra area prefix LSA' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF3_NAP_LSA # 7

   def processData( self, data ):
      assert 'referencedLsaType' in data
      assert data[ 'referencedLsaType' ] in LSA_TYPE_MAP
      self.referencedLsaType = LSA_TYPE_MAP[ data.pop( 'referencedLsaType' ) ]
      return data

   def renderEntry( self ):
      print( "    Intra-Area-Prefix LSA" )
      print( "    Number of Prefixes: %s" % self.numPrefixes )
      print( "    Referenced LSA Type: %s" %
             LSA_TEXT_TYPE_MAP[ self.referencedLsaType ] )
      print( "    Referenced LS Id: %s" % self.referencedLinkStateId )
      print( "    Referenced Adv-Rtr: %s" % self.referencedAdvertisingRouter )

      for prefix in self.prefixList:
         prefix.renderEntry()

class Ospf3LinkLsa( Model ):
   _optionsAndPriority = Int( help='Link options and priority' )
   options = Submodel( valueType=Ospf3IntfOptions, 
                       help='Optional capabalities '
                            'supported by the originating router' )
   priority = Int( help='Router priority of the originating router' )
   linkLocalAddress = IpGenericAddress( help='Link local address of the '
                                       'originating router' )
   numPrefixes = Int( help='Number of prefixes in link LSA' )
   prefixList = List( valueType=Ospf3LsaPrefix,
                      help='List of prefixes in link LSA' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF3_LNK_LSA # 6

   def processData( self, data ):
      optionsAndPriority = data.pop( 'opt_prio' )
      self._optionsAndPriority = optionsAndPriority
      options = optionsAndPriority & 0x00FFFFFF
      self.options = Ospf3IntfOptions()
      self.options.processBitmask( options )
      self.priority = optionsAndPriority >> 24
      return data

   def renderEntry( self ):
      print( "    Option Priority: %s" % self._optionsAndPriority )
      print( "    Link Local Addr: %s" % self.linkLocalAddress )
      print( "    Number of Prefixes: %s" % self.numPrefixes )
      print()

      for prefix in self.prefixList:
         prefix.renderEntry()

# The LSA model defined here roughly follows
# https://tools.ietf.org/html/rfc5340#appendix-A.4.2
class Ospf3Lsa( Model ):
   lsaType = Enum( values=( list( LSA_TYPE_MAP.values() ) ), help='LSA type' )
   age = Int( help='LSA age in seconds' )
   linkStateId = Ip4Address( help='LSA link state id' )
   advertisingRouter = Ip4Address( help='LSA advertising router' )
   sequenceNumber = Int( help='LSA sequence number' )
   checksum = Int( help='LSA checksum' )
   length = Int( help='LSA length' )
   exceptionFlag = Enum( values=( list( EXCEPTION_TYPE_MAP.values() ) ),
                         optional=True, 
                         help='Indicates maximum cost being advertised for self'
                         ' originated LSAs' )
   # Specific LSA submodels
   ospf3RouterLsa = Submodel( valueType=Ospf3RouterLsa,
                              optional=True, help='Router LSA' )
   ospf3NetworkLsa = Submodel( valueType=Ospf3NetworkLsa,
                               optional=True, help='Network LSA' )
   ospf3InterAreaPrefixLsa = Submodel( valueType=Ospf3InterAreaPrefixLsa,
                                       optional=True, help='Inter area prefix LSA' )
   ospf3InterAreaRouterLsa = Submodel( valueType=Ospf3InterAreaRouterLsa,
                                       optional=True, help='Intra area router LSA' )
   ospf3ExternalLsa = Submodel( valueType=Ospf3ExternalLsa,
                               optional=True, help='AS external or NSSA LSA' )
   ospf3IntraAreaPrefixLsa = Submodel( valueType=Ospf3IntraAreaPrefixLsa,
                                       optional=True, help='Intra area prefix LSA' )
   ospf3LinkLsa = Submodel( valueType=Ospf3LinkLsa,
                            optional=True, help='Link LSA' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF3_DATABASE_LSA # 1

   def processData( self, data ):
      assert 'lsaType' in data
      assert data[ 'lsaType' ] in LSA_TYPE_MAP
      self.lsaType = LSA_TYPE_MAP[ data.pop( 'lsaType' ) ]
      if 'exceptionFlag' in data:
         assert data[ 'exceptionFlag' ] in EXCEPTION_TYPE_MAP
         self.exceptionFlag = EXCEPTION_TYPE_MAP[ data.pop( 'exceptionFlag' ) ]
      return data

   def renderEntry( self, detail ):
      if detail is False:
         print( "%4s%15s%15s%5s 0x%08x   0x%06x" %
               ( LSA_TEXT_TYPE_MAP[ self.lsaType ], self.linkStateId,
                 self.advertisingRouter, self.age, self.sequenceNumber, 
                 self.checksum ) )
      else:
         print( "LSA Type: %s" % LSA_TEXT_TYPE_MAP[ self.lsaType ] )
         print( "    Link State ID: %s" % self.linkStateId )
         print( "    Advertising Router: %s" % self.advertisingRouter )
         if self.exceptionFlag:
            print( "    Exception Flag:: %s" %
                   EXCEPTION_TYPE_TEXT_MAP[ self.exceptionFlag ] )
         print( "    Age: %s" % self.age )
         print( "    Sequence Number: 0x%08x" % self.sequenceNumber )
         print( "    Checksum: 0x%06x" % self.checksum )
         print( "    Length: %s" % self.length )

         if self.ospf3RouterLsa:
            self.ospf3RouterLsa.renderEntry()
         elif self.ospf3NetworkLsa:
            self.ospf3NetworkLsa.renderEntry()
         elif self.ospf3InterAreaPrefixLsa:
            self.ospf3InterAreaPrefixLsa.renderEntry()
         elif self.ospf3InterAreaRouterLsa:
            self.ospf3InterAreaRouterLsa.renderEntry()
         elif self.ospf3ExternalLsa:
            self.ospf3ExternalLsa.renderEntry()
         elif self.ospf3IntraAreaPrefixLsa:
            self.ospf3IntraAreaPrefixLsa.renderEntry()
         elif self.ospf3LinkLsa:
            self.ospf3LinkLsa.renderEntry()
         else:
            print()

class Ospf3AreaModel( Model ):
   ospf3AreaLsaList = GeneratorList( valueType=Ospf3Lsa,
                                     help='List of LSAs for the area' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF3_DATABASE_SCOPE_AREA_ENTRY # 2

   def getKey( self, data ):                              
      assert 'area' in data and data[ 'area' ]
      area = data.pop( 'area' )
      return area

   def renderEntry( self, area, detail ):
      print()
      print( "  Area %s LSDB" % area )
      print()

      if detail is False:
         printOspf3LsaHeader()

      for lsa in self.ospf3AreaLsaList:
         lsa.renderEntry( detail )

class Ospf3InterfaceModel( Model ):
   ospf3InterfaceLsaList = GeneratorList( valueType=Ospf3Lsa,
                                          help='List of LSAs for the interface' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF3_DATABASE_SCOPE_LINK_ENTRY # 3

   def getKey( self, data ):
      assert 'interfaceName' in data and data[ 'interfaceName' ]
      return getOspf3Interface( data.pop( 'interfaceName' ) )

   def renderEntry( self, interfaceName, detail ):
      print()
      print( "  Interface %s LSDB" % interfaceName )
      print()

      if detail is False:
         printOspf3LsaHeader()
      for lsa in self.ospf3InterfaceLsaList:
         lsa.renderEntry( detail )

class Ospf3DatabaseInstanceBase( Model ):
   _vrf = Str( help='Private attribute VRF name' ) 
   _renderAsHeaders = Bool( default=False, help='Private attribute to control '
                                                'rendering of AS scope headers' )
   _detailsPresent = Bool( default=False,
                           help='Private attribute to indicate that the '
                                'details submodel is present' )

   ospf3AsLsas = GeneratorList( valueType=Ospf3Lsa,
                                help='List of LSAs with AS scope' ) 
   ospf3AreaEntries = GeneratorDict( keyType=Ip4Address, valueType=Ospf3AreaModel,
                                     help='A mapping between an Area and its '
                                          'LSA list' )
   ospf3InterfaceEntries = GeneratorDict( keyType=str,
                                           valueType=Ospf3InterfaceModel,
                                           help='A mapping between an Interface and '
                                                'its LSA list' )

   def processData( self, data ):
      self._vrf = data.pop( 'vrf' )
      if 'printAsScope' in data:
         self._renderAsHeaders = ord( data.pop( 'printAsScope' ) ) != 0 
      return data 

   def renderAsScopeHeaders( self, detail ):
      print() 
      print( "  AS Scope LSDB" )
      print()

      if detail is False:
         printOspf3LsaHeader()

   def render( self ):
      if self._renderAsHeaders:
         self.renderAsScopeHeaders( self._detailsPresent )

      for lsa in self.ospf3AsLsas:
         lsa.renderEntry( self._detailsPresent )

      for areaId, areaLsas in self.ospf3AreaEntries:
         areaLsas.renderEntry( areaId, self._detailsPresent )

      for ifName, intfLsas in self.ospf3InterfaceEntries:
         intfLsas.renderEntry( ifName, self._detailsPresent )

class Ospf3DatabaseInstance( Ospf3DatabaseInstanceBase ):
   _instance = Int( help='Private attribute instance id' )

   def processData( self, data ):
      self._instance = data.pop( 'instance' )
      return Ospf3DatabaseInstanceBase.processData( self, data )

   def render( self ):
      print( f'Routing Process "ospf {self._instance}", VRF {self._vrf}' )
      Ospf3DatabaseInstanceBase.render( self )

class Ospf3MultiAfDatabaseInstance( Ospf3DatabaseInstanceBase ):
   instanceId = Int( help='OSPFv3 instance ID' )
   shutdown = Bool( default=False,
                    help='OSPFv3 instance is currently shut down' )

   def processData( self, data ):
      self.instanceId = data.pop( 'instance' )
      return Ospf3DatabaseInstanceBase.processData( self, data )

   def renderModel( self, af ):
      print( "OSPFv3 address-family %s" % af )
      print( "Routing Process \"ospfv3\" Instance %d VRF %s" %
             ( self.instanceId, self._vrf ) )
      if self.shutdown:
         print( '  Process is disabled due to shutdown command' )
         return
      Ospf3DatabaseInstanceBase.render( self )

#------------------------------------------------------------------------------
# "show ipv6 ospf [process-id] database database-summary"
#------------------------------------------------------------------------------
class Ospf3DatabaseSummaryBase( Model ):
   # elements to pull out of database_sum_entry_t part of database
   numRouter = Int( default=0, help='Number of Router LSAs' )
   numNetwork = Int( default=0, help='Number of Network LSAs' )
   numInterPfx = Int( default=0, help ='Number of Inter Area Prefix LSAs' )
   numInterRtr = Int( default=0, help ='Number of Inter Area Router LSAs' )
   numAsex = Int( default = 0, help ='Number of AS-external LSAs' )
   numNssa = Int( default = 0, help ='Number of type-7 NSSA  LSAs' )
   numLink = Int( default = 0, help ='Number of Link LSAs' )
   numIntraPfx = Int( default=0, help ='Number of Intra-Area Prefix LSAs' )
   numGrace = Int( default=0, help ='Number of Grace LSAs' )
   vrf = Str( help='VRF name' )
   instanceId = Int( help='Instance ID' )
   areaId = IpGenericAddress( help='Area identifier' )

   def processData( self, data ):
      self.vrf = data.pop( 'vrf' )
      self.instanceId = data.pop( 'instance' )
      return data 

   def vrfIs( self, vrf ):
      self.vrf = vrf

   def instanceIdIs( self, instanceId ):
      self.instanceId = instanceId

   def areaIdIs( self, areaId ):
      self.areaId = areaId

   def render ( self ):
      if self.areaId:
         print( "  Area %s LSDB" % self.areaId )
         print()
      
      # setup labels and values to be printed
      values = [ ( "Router",             self.numRouter ),
                 ( "Network",            self.numNetwork ),
                 ( "Inter Area Prefix",  self.numInterPfx ),
                 ( "Inter Area Router",  self.numInterRtr ),
                 ( "Summary Asex",       self.numAsex ),
                 ( "Nssa",               self.numNssa ),
                 ( "Link",               self.numLink ),
                 ( "Intra Area Prefix",  self.numIntraPfx),
                 ( "Grace",              self.numGrace ), ]

      #add the number values for a total (skipping header fields)
      total = sum( value[ 1 ] for value in values )

      values.append( ( "Total", total ) )

      # Print command output with (left justified) labels
      # followed by (right justified) associated count
      outputHeaderFormat = "{:<17} {:>10}"
      outputDataFormat = "{:<17} {:10d}"
      print( outputHeaderFormat.format( "LSA Type", "Count" ) ) 
      for label, value in values:
         if self.areaId:
            # skip the instance specific lsas if filter is 'area'
            if label in ( "Summary Asex", "Link", "Grace" ):
               continue
         print( outputDataFormat.format( label, value ) )
      print( " " )

class Ospf3DatabaseSummary( Ospf3DatabaseSummaryBase ):

   def processData( self, data ):
      return Ospf3DatabaseSummaryBase.processData( self, data )

   def render( self ):
      # print header 
      print( f'Routing Process "ospf {self.instanceId}"  VRF {self.vrf}' )
      print() 
      # call common render to print data
      Ospf3DatabaseSummaryBase.render( self )

class Ospf3MultiAfDatabaseSummary( Ospf3DatabaseSummaryBase ):

   def processData( self, data ):
      return Ospf3DatabaseSummaryBase.processData( self, data )

   def renderModel( self, af ):
      # print header 
      print( "OSPFv3 address-family %s" % af )
      print( "Routing Process \"ospfv3\" Instance %d VRF %s" %
             ( self.instanceId, self.vrf ) )
      print()
      # call common render to print data
      Ospf3DatabaseSummaryBase.render( self )

#----------------------------------------------------------------------------
# Models used by "show ipv6 lsa-log" and 
# "show ospfv3 [ipv4|ipv6] lsa-log vrf <vrf-name>"
#----------------------------------------------------------------------------

# Refer to ospf3_cli.c. Please keep the strings in sync with the ones in
# o3r_lsalog_entry_cb(). 

LSA_EVENT_TYPE_TEXT_MAP = {
      'backoffStarted'    : 'backoff started, new hold value',
      'backedOff'         : 'backed off, new hold value',
      'backoffRestarted'  : 'backoff restarted, new hold value',
      'outDelay'          : 'out-delay applied, delayed by',
      'discarded'         : 'discarded, was early by',
}

LSA_EVENT_TYPE_MAP = ReversibleDict( {'backoffStarted'    : 0,
                                      'backedOff'         : 1,
                                      'backoffRestarted'  : 2,
                                      'outDelay'          : 3,
                                      'discarded'         : 4 } )

lsaEventTypeHelp = '''LSA event types:
      backoffStarted     - lsa backoff started
      backedOff          - lsa backed off
      backoffRestarted   - lsa backoff restarted
      outDelay           - lsa delayed
      discarded          - lsa discarded '''

class Ospf3LsaLogEntry( Model ):
   linkStateId = Ip4Address( help='LSA link state identifier' )
   advertisingRouter = Ip4Address( help='LSA advertising router' )
   lsaEventType = Enum( values =( 'backoffStarted', 'backedOff', 
                                  'backoffRestarted', 
                                  'outDelay', 'discarded' ), 
                                  help=lsaEventTypeHelp )
   logTime = Float( help='LSA log generation time in seconds' )
   lsaType = Enum( values =( list( LSA_TYPE_MAP.values() ) ), help='LSA type' )
   sequenceNumber = Int( help='LSA sequence number' )
   lsaHoldInterval = Int( help='LSA Hold interval in milliseconds' )

   def processData( self, data ):
      if 'lsaEventType' in data:
         reverse_event_map = LSA_EVENT_TYPE_MAP.reverse()
         self.lsaEventType = reverse_event_map[ data.pop ( 'lsaEventType' ) ]
      if 'lsaType' in data:
         self.lsaType = LSA_TYPE_MAP[ data.pop( 'lsaType' ) ]
      if 'logTime' in data:
         self.logTime = float( data.pop( 'logTime' ) )
      return data

   def renderEntry( self ):
      print( "[%s] type %s: %s [%s], event %s, %s %d msecs" %
             ( timestampToLocalTime( self.logTime ), 
               LSA_TEXT_TYPE_MAP[ self.lsaType ], 
               self.linkStateId, 
               self.advertisingRouter, 
               LSA_EVENT_TYPE_MAP[ self.lsaEventType ],
               LSA_EVENT_TYPE_TEXT_MAP[ self.lsaEventType ], 
               self.lsaHoldInterval ) )

# Any model that is used for operational commands of old style V6 config, 
# should only use vrf and instance id names and processId starting with 
# 'underscore' which makes them hidden attributes.
# vrf, instanceId and processId should be hidden attributes in the model 
# since the JSON output will already include these in the outermost 
# dicts generated by Ospf3VrfModelDec/Ospf3MultiAfVrfModelDec. 
# Including them here would be duplication  of information. 
# We do need them here since the render function uses them.
class Ospf3LsaLogInstanceBase( Model ):
   routerId = Ip4Address( optional=True, help='OSPFv3 router identifier' )
   _vrf = Str( help='Private attribute VRF name' )
   ospf3LsaLogEntries = GeneratorList( valueType = Ospf3LsaLogEntry,
                                        help='List of LSA log entries' )
   def render( self ):
      if not self.routerId:
         return
      for entry in self.ospf3LsaLogEntries:
         entry.renderEntry()

class Ospf3LsaLogInst( Ospf3LsaLogInstanceBase ):
   def render( self ):
      print( "OSPF3 Process {}, VRF {}, LSA Throttling Log:".format(
             self.routerId, self._vrf ) )
      Ospf3LsaLogInstanceBase.render( self )

class Ospf3MultiAfLsaLogInst( Ospf3LsaLogInstanceBase ):
   instanceId = Int( help='Private attribute for instance identifier' )
   shutdown = Bool( default=False,
                    help='OSPFv3 instance is currently shut down' )
   def renderModel( self, af ):
      print( "OSPFv3 address-family %s" % af )
      if self.shutdown:
         print( "Routing Process \"ospfv3\" Instance %d VRF %s" % ( 
                     self.instanceId, self._vrf ) )
         print( '  Process is disabled due to shutdown command' )
      else:
         print( "Routing Process \"ospfv3\" with ID %s and "
               "Instance %d, VRF %s LSA Throttling Log:" % (
                   self.routerId, self.instanceId, self._vrf ) )
         Ospf3LsaLogInstanceBase.render( self )

#-------------------------------------------------------------------------
# Models used by "show ipv6 spf-log" and 
# "show ospfv3 [ipv4|ipv6] spf-log vrf <vrf-name>"
#-------------------------------------------------------------------------
# Refer to ospf3_cli.c. Please refer to o3r_spflog_entry_cb()

class Ospf3SpfLogEntry( Model ):
   spfStartTime = Float( help='SPF start time in UTC' )
   spfDuration = Float( help='SPF duration in milliseconds' )
   spfScheduleReason = Str( help='Reason for scheduling SPF' )

   def processData( self, data ):
      self.spfDuration = data[ 'spfDurationMsec' ] + ( \
                    float( data[ 'spfDurationUsec' ] ) / float( pow ( 10, 3 ) ) )
      if 'spfStartTimeSec' in data and 'spfStartTimeMsec' in data:
         self.spfStartTime =  \
                    data[ 'spfStartTimeSec' ] + ( \
                    float( data[ 'spfStartTimeMsec' ] ) / float( pow ( 10, 3 ) ) )

      return data

   def renderEntry( self ):
      if self.spfScheduleReason:
         scheduledBuf = "Scheduled after %.3f ms" % ( self.spfDuration )
         print( "%-13s %-24s %s" %
               ( timestampToLocalTimeMsec( self.spfStartTime ),  
                     scheduledBuf, self.spfScheduleReason ) )
      else:
         print( "%s  SPF ran for %.3f ms" %
               ( timestampToLocalTimeMsec( self.spfStartTime ), 
                 self.spfDuration ) )

class Ospf3SpfLogInstanceBase( Model ):
   routerId = Ip4Address( optional=True, help='OSPFv3 router identifier' )
   _vrf = Str( help='Private attribute for VRF name' )
   
   ospf3SpfLogEntries = GeneratorList( valueType = Ospf3SpfLogEntry,
                                        help='OSPFv3 SPF Log Entries' )
   def render( self ):
      if not self.routerId:
         return
      for entry in self.ospf3SpfLogEntries:
         entry.renderEntry()

class Ospf3SpfLogInst( Ospf3SpfLogInstanceBase ):

   def render( self ):
      print( "OSPF3 Process {}, VRF {}".format(
             self.routerId, self._vrf ) )
      print( "TIME          EVENT                    REASON" )
      Ospf3SpfLogInstanceBase.render( self )

class Ospf3MultiAfSpfLogInst( Ospf3SpfLogInstanceBase ):
   instanceId = Int( help='Private attribute for instance identifier' )
   shutdown = Bool( default=False,
                    help='OSPFv3 instance is currently shut down' )
   def renderModel( self, af ):
      print( "OSPFv3 address-family %s" % af )
      if self.shutdown:
         print( "Routing Process \"ospfv3\" Instance %d VRF %s" % (
                  self.instanceId, self._vrf ) )
         print( '  Process is disabled due to shutdown command' )
      else:
         print( "Routing Process \"ospfv3\" with ID %s and "
               "Instance %d, VRF %s" % (
                   self.routerId, self.instanceId, self._vrf ) )
         print( "TIME          EVENT                    REASON" )
         Ospf3SpfLogInstanceBase.render( self )

#------------------------------------------------------------------------------
# "show ipv6 ospf retransmission-list [vrf vrf-name|default|all]"
#------------------------------------------------------------------------------
class Ospf3LsaEntry( Model ):
   lsaType = Enum( values=tuple( LSA_TYPE_MAP.values() ), help='LSA type' )
   linkStateId = Ip4Address( help='LSA link state identifier' )
   advertisingRouter = Ip4Address( help='LSA advertising router' )
   age = Int( help='LSA age in seconds' )
   sequenceNumber = Int( help='LSA sequence number' )
   checksum = Int( help='LSA checksum' )

   def processData( self, data ):
      lsaType = int( data.pop( 'lsaType' ) )
      self.lsaType = LSA_TYPE_MAP.get( lsaType )
      assert self.lsaType is not None, 'Invalid value %s for lsaType' % lsaType
      return data

   def render( self ):
      lsaType = LSA_TEXT_TYPE_MAP[ self.lsaType ]
      print( '%4s%15s%15s%8u   %0#8lx   %0#8lx' % ( lsaType, self.linkStateId,
                                                   self.advertisingRouter, self.age,
                                                   self.sequenceNumber,
                                                   self.checksum ) )

class Ospf3LsaListBase( Model ):
   _vrf = Str( help='VRF name' )
   interfaceName = Str( help='OSPFv3 neighbor interface name' )
   interfaceAddress = IpGenericAddress( help='OSPFv3 neighbor interface address' )

   def getKey( self, data ):
      return data[ 'routerId' ]

   def processData( self, data ):
      self.interfaceName = getOspf3Interface( data.pop( 'interfaceName' ) )
      return data

   def renderModel( self, routerId ):
      print( 'Neighbor %s interface %s address %s' %
            ( routerId, self.interfaceName, self.interfaceAddress ) )

class Ospf3LsaRetransmissionInstance( Ospf3LsaListBase ):
   nextRetransmissionTime = Float( help='Next retransmission time in UTC' )
   retransmissions = GeneratorList( valueType=Ospf3LsaEntry,
                                    help='LSA retransmission list' )

   def processData( self, data ):
      data = Ospf3LsaListBase.processData( self, data )
      nextRetransmissionSec = data.pop( 'nextRetransmissionTime' ) / 1000
      self.nextRetransmissionTime = Tac.utcNow() + nextRetransmissionSec
      return data

   def renderModel( self, routerId ):
      Ospf3LsaListBase.renderModel( self, routerId )
      nextRetransmissionMsec = ( self.nextRetransmissionTime - Tac.utcNow() ) * 1000
      print( 'LSA retransmission due in %d msec.\n' % nextRetransmissionMsec )

      printOspf3LsaHeader()
      for retransmissionEntry in self.retransmissions:
         retransmissionEntry.render()

class Ospf3LsaRetransmissionList( Model ):
   neighbors = GeneratorDict( keyType=Ip4Address,
                              valueType=Ospf3LsaRetransmissionInstance,
                              help='Dictionary of retransmission list keyed by'
                                   ' neighbor router ID' )

   def render( self ):
      for routerId, neighborModel in self.neighbors:
         neighborModel.renderModel( routerId )

class Ospf3MultiAfLsaRetransmissionList( Model ):
   instanceId = Int( help='OSPFv3 instance identifier' )
   neighbors = GeneratorDict( keyType=Ip4Address,
                              valueType=Ospf3LsaRetransmissionInstance,
                              help='Dictionary of retransmission list keyed by'
                                   ' neighbor router ID' )

   def renderModel( self, af ):
      firstInst = True
      for routerId, neighborModel in self.neighbors:
         if firstInst:
            print( "Routing Process \"ospfv3\" Instance %d VRF %s" % (
                  self.instanceId, neighborModel._vrf ) )
            firstInst = False
         neighborModel.renderModel( routerId )

#------------------------------------------------------------------------------
# "show ipv6 ospf request-list [vrf vrf-name|default|all]"
#------------------------------------------------------------------------------
class Ospf3LsaRequestInstance( Ospf3LsaListBase ):
   requestQueueLength = Int( help='LSA request queue length' )
   requests = GeneratorList( valueType=Ospf3LsaEntry,
                             help='LSA request list' )

   def renderModel( self, routerId ):
      Ospf3LsaListBase.renderModel( self, routerId )
      printOspf3LsaHeader()
      for requestEntry in self.requests:
         requestEntry.render()

class Ospf3LsaRequestList( Model ):
   instanceId = Int( help='OSPFv3 instance identifier' )
   neighbors = GeneratorDict( keyType=Ip4Address,
                              valueType=Ospf3LsaRequestInstance,
                              help='Dictionary of request keyed by'
                                   ' neighbor router ID' )

   def render( self ):
      for routerId, neighborModel in self.neighbors:
         neighborModel.renderModel( routerId )

class Ospf3MultiAfLsaRequestList( Model ):
   instanceId = Int( help='OSPFv3 instance identifier' )
   neighbors = GeneratorDict( keyType=Ip4Address,
                              valueType=Ospf3LsaRequestInstance,
                              help='Dictionary of request keyed by'
                                   ' neighbor router ID' )

   def renderModel( self, af ):
      firstInst = True
      for routerId, neighborModel in self.neighbors:
         if firstInst:
            print( "OSPFv3 address-family %s" % af )
            print( "Routing Process \"ospfv3\" Instance %d VRF %s" % (
                  self.instanceId, neighborModel._vrf ) )
            firstInst = False
         neighborModel.renderModel( routerId )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global ospf3ConfigDir, ospf3StatusDir
   ospf3ConfigDir = ConfigMount.mount( entityManager, Ospf3Consts.configPath,
                                       Ospf3Consts.configType, 'w' )
   ospf3StatusDir = LazyMount.mount( entityManager, Ospf3Consts.statusPath,
                                    'Tac::Dir', 'ri' )

#------------------------------------------------------------------------------
# show ospfv3 [ipv4|ipv6] counters [vrf vrf-name|default|all]
#------------------------------------------------------------------------------
LSA_TYPE_COUNTERS_MAP = \
      { 'newLsaReceived'                : 'New LSAs received',
        'selfOriginatedLsaNew'          : 'New LSAs self-originated',
        'selfOriginatedLsaUpdated'      : 'Updated self-originated LSAs',
        'selfOriginatedLsaRefreshed'    : 'Refreshed self-originated LSAs',
        'lsaInLsdb'                     : 'LSAs in link state database' }

class Ospf3CountersLsaEntry( Model ):
   numRouter = Int( default=0, help="Router LSAs" )
   numNetwork = Int( default=0, help="Network LSAs" )
   numIap = Int( default=0, help="Inter-Area Prefix LSAs" )
   numIar = Int( default=0, help="Inter-Area Router LSAs" )
   numAsex = Int( default=0, help="AS-External LSAs" )
   numNssa = Int( default=0, help="NSSA LSAs" )
   numLink = Int( default=0, help="Link LSAs" )
   numNap = Int( default=0, help="Intra-Area Prefix LSAs" )
   numGrc = Int( default=0, help="Grace LSAs" )

   def getKey( self, data ):
      return str( data[ 'counterName' ] )

   def renderEntry( self, counterName, table ):
      table.newRow( LSA_TYPE_COUNTERS_MAP[ counterName ],
                    self.numRouter, self.numNetwork, 
                    self.numIap, self.numIar, self.numAsex, 
                    self.numNssa, self.numLink, self.numNap,
                    self.numGrc )

class Ospf3CountersInstance( Model ):
   _instanceId = Int( help='Private attribute for OSPFv3 instance ID' )
   _vrf = Str( help='Private attribute for VRF name' )
   lastResetTime = Float( optional=True,
                          help='UTC timestamp of last counters reset' )
   selfOriginatedLsaReceived = Int( help='Number of self-originated LSAs received' )
   selfOriginatedLsaReceivedUpdated = \
         Int( help='Number of self-originated LSAs received and updated' )
   selfOriginatedLsaReceivedDeleted = \
         Int( help='Number of self-originated LSAs received and deleted' )
   spfCount = Int( help='Number of SPF runs' )
   lsaTypeCounters = GeneratorDict( keyType=str, valueType=Ospf3CountersLsaEntry,
                                    help='Dictionary of LSA type counters keyed'
                                         ' by counter name' )
   def processData( self, data ):
      if 'lastResetTime' in data:
         self.lastResetTime = Tac.utcNow() - data.pop( 'lastResetTime' )
      return data

   def renderModel( self, af ):
      if self._instanceId is None:
         return

      print( "VRF: %s" % self._vrf )
      print( "OSPFv3 instance: %s" % self._instanceId )

      if self.lastResetTime is not None:
         timeSecs = int( Tac.utcNow() - self.lastResetTime )
         timeDHMS = formatTimeInterval( timeSecs )
         print( "The counters were last reset %s ago\n" % timeDHMS )
      else:
         print( "The counters have never been reset\n" )

      headings = ( "Counter Description", "Value" )
      table = TableOutput.createTable( headings,
                                       tableWidth=TableOutput.terminalWidth() )
      cf = TableOutput.Format( justify="left" )
      cf.noPadLeftIs( True )
      cf.padLimitIs( True )
      cf2 = TableOutput.Format( justify="center" )
      cf2.padLimitIs( True )
      table.formatColumns( cf, cf2 )
      table.newRow( "Self-originated LSAs received",
                    self.selfOriginatedLsaReceived )
      table.newRow( "   Updated", self.selfOriginatedLsaReceivedUpdated )
      table.newRow( "   Deleted", self.selfOriginatedLsaReceivedDeleted )
      table.newRow( "SPF runs", self.spfCount )
      print( table.output() )

      headings = ( "Counter Description", "RTR", "NTW", "IAP", "IAR", "AEX",
                   "NSA", "LNK", "NAP", "GRC" )
      table = TableOutput.createTable( headings,
                                       tableWidth=TableOutput.terminalWidth() )
      table.formatColumns( cf, cf2, cf2, cf2, cf2, cf2, cf2, cf2, cf2, cf2 )
      for counterName, counter in self.lsaTypeCounters:
         counter.renderEntry( counterName, table )
      print( table.output() )

class Ospf3CountersInterfaceEntry( Model ):
   lastResetTime = Float( optional=True,
                          help="UTC timestamp of the last time counters were reset" )
   multicastLsUpdatesSent = Int( default=0,
                                 help="Number of multicast LS updates sent" )
   delayedLsAcksSent = Int( default=0,
                            help="Number of delayed LS acknowledgements sent" )
   lsAcksReceived = Int( default=0, help="Number of LS acknowledgements received" )
   lsUpdatesInFloodQueue = Int( default=0,
                                help="Number of LS updates currently"
                                     " in the flood queue" )
   def getKey( self, data ):
      return getOspf3Interface( data.pop( 'ifname' ) )

   def processData( self, data ):
      if 'lastResetTime' in data:
         self.lastResetTime = Tac.utcNow() -  data.pop( 'lastResetTime' )
      return data

   def renderEntry( self, ifname ):
      print( "\nStatistics for interface %s:" % ifname )
      if self.lastResetTime is not None:
         timeSecs = int( Tac.utcNow() - self.lastResetTime )
         timeDHMS = formatTimeInterval( timeSecs )
         print( "The counters were last reset %s ago\n" % timeDHMS )
      else:
         print( "The counters have never been reset\n" )

      headings = ( "Counter Description", "Value" )
      table = TableOutput.createTable( headings,
                                       tableWidth=TableOutput.terminalWidth() )
      cf = TableOutput.Format( justify="left" )
      cf.noPadLeftIs( True )
      cf.padLimitIs( True )
      cf2 = TableOutput.Format( justify="center" )
      cf.padLimitIs( True )
      table.formatColumns( cf, cf2 )
      table.newRow( "Multicast LS updates sent", self.multicastLsUpdatesSent )
      table.newRow( "Delayed LS acknowledgements sent", self.delayedLsAcksSent )
      table.newRow( "LS acknowledgements received", self.lsAcksReceived )
      table.newRow( "LS updates currently in flood queue",
                    self.lsUpdatesInFloodQueue )
      print( table.output() )

class Ospf3CountersInterface( Model ):
   interfaces = GeneratorDict( keyType=str,
                               valueType=Ospf3CountersInterfaceEntry,
                               help="Dictionary of OSPFv3 interfaces indexed by "
                                    "the interface name" )
   _summary = Bool( default=False,
                    help="Private attribute to indicate that a consolidated"
                         " summary of all interfaces is to be displayed" )
   _instanceId = Int( help="Instance ID" )
   _vrf = Str( help="VRF name" )

   def renderModel( self, af ):
      if self._instanceId is None:
         return
      if self._summary:
         print( "VRF: %s" % self._vrf )
         print( "OSPFv3 instance: %d" % self._instanceId )
         
         sumMulticastLsUpdatesSent = 0
         sumDelayedLsAcksSent = 0
         sumLsAcksReceived = 0
         sumLsUpdatesInFloodQueue = 0
         for ifname, interface in self.interfaces:
            sumMulticastLsUpdatesSent += interface.multicastLsUpdatesSent
            sumDelayedLsAcksSent += interface.delayedLsAcksSent
            sumLsAcksReceived += interface.lsAcksReceived
            sumLsUpdatesInFloodQueue += interface.lsUpdatesInFloodQueue

         print( "\nStatistics summary for all interfaces:\n" )
         headings = ( "Counter Description", "Value" )
         table = TableOutput.createTable( headings,
                                          tableWidth=TableOutput.terminalWidth() )
         cf = TableOutput.Format( justify="left" )
         cf.noPadLeftIs( True )
         cf.padLimitIs( True )
         cf2 = TableOutput.Format( justify="center" )
         cf.padLimitIs( True )
         table.formatColumns( cf, cf2 )
         table.newRow( "Multicast LS updates sent", sumMulticastLsUpdatesSent )
         table.newRow( "Delayed LS acknowledgements sent", sumDelayedLsAcksSent )
         table.newRow( "LS acknowledgements received", sumLsAcksReceived )
         table.newRow( "LS updates currently in flood queue",
                       sumLsUpdatesInFloodQueue )
         print( table.output() )
      else:
         # Sometimes OSPF3 instances exist without being configured on any 
         # interface.
         # The following makes sure that the header is printed only if
         # at least one intf exists in the instance.
         firstIntf = True
         for ifname, interface in self.interfaces:
            if firstIntf:
               print( "VRF: %s" % self._vrf )
               print( "OSPFv3 instance: %d" % self._instanceId )
               firstIntf = False
            interface.renderEntry( ifname )

#------------------------------------------------------------------------------------
# "show ospfv3 [ipv6|ipv4] counters neighbor [vrf vrf-name|default|all]"
# "show ospfv3 [ipv6|ipv4] counters neighbor summary [vrf vrf-name|default|all]"
# "show ospfv3 [ipv6|ipv4] counters neighbor [<neighbor-id>]
#                                            [vrf vrf-name|default|all]"
# "show ospfv3 [ipv6|ipv4] counters neighbor [<interface-name>]
#                                            [<neighbor-id>]
#                                            [vrf vrf-name|default|all]"
#------------------------------------------------------------------------------------
class Ospf3CountersNeighborEntry( Model ):
   neighborRouterId = Ip4Address( help='OSPFv3 neighbor router identifier' )
   interfaceName = Str( help='Name of interface connected to neighbor' )
   neighborLinkLocalAddress = IpGenericAddress( help='OSPFv3 neighbor link local'
                                                     ' address' )
   lastResetTime = Float( optional=True,
                          help='UTC timestamp of the last time counters were reset' )
   databaseDescriptionSent = Int( default=0,
                                  help='Number of database description'
                                       ' packets sent' )
   lsAcknowledgementSent = Int( default=0,
                                help='Number of LS acknowledgement packets sent' )
   newLsRequestSent = Int( default=0,
                           help='Number of new LS request packets sent' )
   lsRequestSentDueToRetransmission = Int( default=0,
                                           help='Number of LS request packets '
                                                'sent due to retransmission' )
   lsUpdateSentDueToLsRequest = Int( default=0,
                                     help='Number of LS update packets sent due to'
                                          ' receiving LS requests' )
   lsUpdateSentDueToOldLsaReceived = Int( default=0,
                                          help='Number of LS update packets sent'
                                               ' due to receiving old LSAs' )
   lsUpdateSentDueToRetransmission = Int( default=0, 
                                          help='Number of LS update packets sent '
                                               'due to retransmission' )
   databaseDescriptionReceived = Int( default=0,
                                      help='Number of database description packets'
                                           ' received' )
   lsRequestReceived = Int( default=0,
                            help='Number of LS request packets received' )
   unicastLsUpdateReceived = Int( default=0,
                                  help='Number of unicast LS update packets'
                                       ' received' )
   multicastLsUpdateReceived = Int( default=0,
                                    help='Number of multicast LS update packets'
                                         ' received' )
   unicastLsAcknowledgementReceived = Int( default=0,
                                           help='Number of unicast LS'
                                                ' acknowledgment packets received' )
   multicastLsAcknowledgementReceived = Int( default=0,
                                             help='Number of multicast LS'
                                                  ' acknowledgement packets'
                                                  ' received' )
   unknownLsaReceived = Int( default=0, help='Number of unknown LSAs received' )
   neighborStateChanges = Int( default=0, help='Number of neighbor state changes' )
   lsUpdatesInRetransmissionList = Int( default=0,
                                        help='Number of LS updates in retransmission'
                                             ' list' )
   lsRequestsInRequestTree = Int( default=0,
                                  help='Number of LS requests in request tree' )
   newCopyTooQuick = Int( default=0,
                          help='Number of dropped LSAs due to LSA arriving '
                               'before its scheduled next receive time' )
   invalidChecksum = Int( default=0,
                          help='Number of dropped LSAs due to invalid checksum' )
   floodScopeMismatch = Int( default=0,
                             help='Number of dropped LSAs due to an AS-external '
                                  'type LSA being received in a stub area' )
   type7InNonNssa = Int( default=0,
                         help='Number of dropped LSAs due to an NSSA type LSA '
                              'being received in a non-NSSA area' )
   rtrIdZero = Int( default=0,
                    help='Number of dropped LSAs due to the advertising router '
                         'id of recieved LSA being zero' )
   unknownLsType = Int( default=0,
                        help='Number of dropped LSAs due to an unknown LSA with '
                             'U-bit set being received in a stub area' )
   prefixLenSmall = Int( default=0,
                         help='Number of dropped LSAs due to the prefix length '
                              'of received LSA being less than minimum '
                              'prefix length' )
   prefixScopeMismatch = Int( default=0,
                              help='Number of dropped LSAs due to the prefix '
                              'address scope of received LSA '
                              'not sitelocal or global (only for IPv6 '
                              'address family)' )
   prefixLenBig = Int( default=0,
                      help='Number of dropped LSAs due to the prefix length of '
                           'the received LSA being greater than AF mask' )
   prefixLenInvalid = Int( default=0,
                           help="Number of dropped LSAs due to the LSA "
                           " received whose length of the prefix"
                           " header was less than (packet's prefix length +"
                           " length of empty prefix packet)" )

   def processData( self, data ):
      self.interfaceName = getOspf3Interface(
                              kernelIntfFilter( data.pop( 'intfName' ) ) )
      time = data.pop( 'lastResetTime', None )
      if time is not None:
         self.lastResetTime = Tac.utcNow() - time
      return data
 
   def renderEntry( self, vrf ):
      formatStr = '  %s: %d'

      print( 'Neighbor {} VRF {} on interface {}'.format( self.neighborRouterId, vrf,
                                                     self.interfaceName ) )
      if self.lastResetTime is not None:
         timeSecs = int( Tac.utcNow() - self.lastResetTime )
         timeDHMS = formatTimeInterval( timeSecs )
         print( "The counters were last reset %s ago" % timeDHMS )
      else:
         print( "The counters have never been reset" )
      print( formatStr % ( 'DD packets sent', self.databaseDescriptionSent ) )
      print( formatStr % ( 'LS-Ack packets sent', self.lsAcknowledgementSent ) )
      print( formatStr % ( 'New LS-Request packets sent', self.newLsRequestSent ) )
      print( formatStr % ( 'LS-Request packets sent due to retransmission', 
                          self.lsRequestSentDueToRetransmission ) )
      print( formatStr % ( 'LS-Update packets sent due to receiving LS-Requests',
                          self.lsUpdateSentDueToLsRequest ) )
      print( formatStr % ( 'LS-Update packets sent due to receiving old LSAs',
                          self.lsUpdateSentDueToOldLsaReceived ) )
      print( formatStr % ( 'LS-Update packets sent due to retransmission', 
                          self.lsUpdateSentDueToRetransmission ) )
      print( formatStr % ( 'DD packets received',
            self.databaseDescriptionReceived ) )
      print( formatStr % ( 'LS-Request packets received', self.lsRequestReceived ) )
      print( formatStr % ( 'Unicast LS-Update packets received',
                          self.unicastLsUpdateReceived ) )
      print( formatStr % ( 'Multicast LS-Update packets received',
                          self.multicastLsUpdateReceived ) )
      print( formatStr % ( 'Unicast LS-Ack packets received',
                          self.unicastLsAcknowledgementReceived ) )
      print( formatStr % ( 'Multicast LS-Ack packets received',
                          self.multicastLsAcknowledgementReceived ) )
      print( formatStr % ( 'Unknown LSAs received', self.unknownLsaReceived ) )
      print( formatStr % ( 'Neighbor state changes', self.neighborStateChanges ) )
      print( formatStr % ( 'LS-Updates in retransmission list',
                          self.lsUpdatesInRetransmissionList ) )
      print( formatStr % ( 'LS-Requests in request tree',
                          self.lsRequestsInRequestTree ) )
      print( formatStr % ( 'Dropped LSAs due to prefix scope mismatch',
                          self.prefixScopeMismatch ) )
      print( formatStr % ( 'Dropped LSAs due to prefix length too small',
                          self.prefixLenSmall ) )
      print( formatStr % ( 'Dropped LSAs due to prefix length invalid',
                          self.prefixLenInvalid ) )
      print( formatStr % ( 'Dropped LSAs due to prefix length too big',
                          self.prefixLenBig ) )
      print( formatStr % ( 'Dropped LSAs due to flood scope mismatch',
                          self.floodScopeMismatch ) )
      print( formatStr % ( 'Dropped LSAs due to type 7 in non-NSSA',
                          self.type7InNonNssa ) )
      print( formatStr % ( 'Dropped LSAs due to unknown LS type with '
                          'U-bit set in stubby area',
                          self.unknownLsType ) )
      print( formatStr % ( 'Dropped LSAs due to invalid checksum',
                          self.invalidChecksum ) )
      print( formatStr % ( 'Dropped LSAs due to router ID zero',
                          self.rtrIdZero ) )
      print( formatStr % ( 'Dropped LSAs due to receiving new copy too soon',
                          self.newCopyTooQuick ) )

class Ospf3MultiAfCountersNeighor( Model ):
   vrf = Str( help='VRF name' )
   instanceId = Int( help='Instance ID' )
   neighborCounterEntries = GeneratorList( valueType=Ospf3CountersNeighborEntry,
                                           help='List of neighbor-level counter'
                                                ' entries' )
   _summary = Bool( default=False,
                    help='Private attribute to indicate that a consolidated'
                         ' summary of all neighbors is to be displayed' )

   def renderModel( self, af ):
      if self._summary:
         print( "OSPFv3 address-family %s" % af )
         print( "Routing Process \"ospfv3\" Instance %d VRF %s" % ( self.instanceId,
                                                                   self.vrf ) )
         print( "Statistics summary for all neighbors" )

         sumDDSent = 0
         sumLsAcksSent = 0
         sumNewLsRequestsSent = 0
         sumLsRequestSentDueToRetransmission = 0
         sumLsUpdatesSentDueLsRequests = 0
         sumLsUpdatesSentDueOldLsa = 0
         sumLsUpdateSentDueToRetransmission = 0
         sumDDReceived = 0
         sumLsRequestsReceived = 0
         sumUnicastLsUpdatesReceived = 0
         sumMulticastLsUpdatesReceived = 0
         sumUnicastLsAcksReceived = 0
         sumMulticastLsAcksReceived = 0
         sumUnknownLsasReceived = 0
         sumNeighborStateChanges = 0
         sumLsUpdatesInRetransmissionList = 0
         sumLsRequestsInRequestTree = 0
         sumPrefixScopeMismatch = 0
         sumPrefixLenSmall = 0
         sumPrefixLenInvalid = 0
         sumPrefixLenBig = 0
         sumFieldScopeMismatch = 0
         sumType7InNssa = 0
         sumUnknownLsType = 0
         sumInvalidChecksum = 0
         sumRtrIdZero = 0
         sumNewCopyTooQuick = 0

         for neighbor in self.neighborCounterEntries:
            sumDDSent += neighbor.databaseDescriptionSent
            sumLsAcksSent += neighbor.lsAcknowledgementSent
            sumNewLsRequestsSent += neighbor.newLsRequestSent
            sumLsRequestSentDueToRetransmission += \
                                    neighbor.lsRequestSentDueToRetransmission
            sumLsUpdatesSentDueLsRequests += neighbor.lsUpdateSentDueToLsRequest
            sumLsUpdatesSentDueOldLsa += neighbor.lsUpdateSentDueToOldLsaReceived
            sumLsUpdateSentDueToRetransmission += \
                                    neighbor.lsUpdateSentDueToRetransmission
            sumDDReceived += neighbor.databaseDescriptionReceived
            sumLsRequestsReceived += neighbor.lsRequestReceived
            sumUnicastLsUpdatesReceived += neighbor.unicastLsUpdateReceived
            sumMulticastLsUpdatesReceived += neighbor.multicastLsUpdateReceived
            sumUnicastLsAcksReceived += neighbor.unicastLsAcknowledgementReceived
            sumMulticastLsAcksReceived += \
                                       neighbor.multicastLsAcknowledgementReceived
            sumUnknownLsasReceived += neighbor.unknownLsaReceived
            sumNeighborStateChanges += neighbor.neighborStateChanges
            sumLsUpdatesInRetransmissionList += \
                                             neighbor.lsUpdatesInRetransmissionList
            sumLsRequestsInRequestTree += neighbor.lsRequestsInRequestTree
            sumPrefixScopeMismatch += neighbor.prefixScopeMismatch
            sumPrefixLenSmall += neighbor.prefixLenSmall
            sumPrefixLenInvalid += neighbor.prefixLenInvalid
            sumPrefixLenBig += neighbor.prefixLenBig
            sumFieldScopeMismatch += neighbor.floodScopeMismatch
            sumType7InNssa += neighbor.type7InNonNssa
            sumUnknownLsType += neighbor.unknownLsType
            sumInvalidChecksum += neighbor.invalidChecksum
            sumRtrIdZero += neighbor.rtrIdZero
            sumNewCopyTooQuick += neighbor.newCopyTooQuick

         formatStr = '%s: %d'
         print( formatStr % ( 'DD packets sent', sumDDSent ) )
         print( formatStr % ( 'LS-Ack packets sent', sumLsAcksSent ) )
         print( formatStr % ( 'New LS-Request packets sent',
               sumNewLsRequestsSent ) )
         print( formatStr % ( 'LS-Request packets sent due to retransmission', 
                             sumLsRequestSentDueToRetransmission ) )
         print( formatStr % ( 'LS-Update packets sent due to receiving LS-Requests',
                             sumLsUpdatesSentDueLsRequests ) )
         print( formatStr % ( 'LS-Update packets sent due to receiving old LSAs',
                             sumLsUpdatesSentDueOldLsa ) )
         print( formatStr % ( 'LS-Update packets sent due to retransmission', 
                             sumLsUpdateSentDueToRetransmission ) )
         print( formatStr % ( 'DD packets received', sumDDReceived ) )
         print( formatStr % ( 'LS-Request packets received',
               sumLsRequestsReceived ) )
         print( formatStr % ( 'Unicast LS-Update packets received',
                             sumUnicastLsUpdatesReceived ) )
         print( formatStr % ( 'Multicast LS-Update packets received',
                             sumMulticastLsUpdatesReceived ) )
         print( formatStr % ( 'Unicast LS-Ack packets received',
                             sumUnicastLsAcksReceived ) )
         print( formatStr % ( 'Multicast LS-Ack packets received',
                             sumMulticastLsAcksReceived ) )
         print( formatStr % ( 'Unknown LSAs received', sumUnknownLsasReceived ) )
         print( formatStr % ( 'Neighbor state changes', sumNeighborStateChanges ) )
         print( formatStr % ( 'LS-Updates in retransmission list',
                             sumLsUpdatesInRetransmissionList ) )
         print( formatStr % ( 'LS-Requests in request tree',
                             sumLsRequestsInRequestTree ) )
         print( formatStr % ( 'Dropped LSAs due to prefix scope mismatch',
                             sumPrefixScopeMismatch ) )
         print( formatStr % ( 'Dropped LSAs due to prefix length too small',
                             sumPrefixLenSmall ) )
         print( formatStr % ( 'Dropped LSAs due to prefix length invalid',
                             sumPrefixLenInvalid ) )
         print( formatStr % ( 'Dropped LSAs due to prefix length too big',
                             sumPrefixLenBig ) )
         print( formatStr % ( 'Dropped LSAs due to flood scope mismatch',
                             sumFieldScopeMismatch ) )
         print( formatStr % ( 'Dropped LSAs due to type 7 in non-NSSA',
                             sumType7InNssa ) )
         print( formatStr % ( 'Dropped LSAs due to unknown LS type with '
                             'U-bit set in stubby area',
                             sumUnknownLsType ) )
         print( formatStr % ( 'Dropped LSAs due to invalid checksum',
                             sumInvalidChecksum ) )
         print( formatStr % ( 'Dropped LSAs due to router ID zero',
                             sumRtrIdZero ) )
         print( formatStr % ( 'Dropped LSAs due to receiving new copy too soon',
                             sumNewCopyTooQuick ) )
      else:
         firstNgb = True

         for neighbor in self.neighborCounterEntries:
            if firstNgb:
               print( "OSPFv3 address-family %s" % af )
               print( "Routing Process \"ospfv3\" Instance %d VRF %s" % (
                  self.instanceId, self.vrf ) )
               firstNgb = False

            neighbor.renderEntry( self.vrf )
            print( '' )
