# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AuthnUserPriorityCli as aupCli
import AgentCommandRequest
import BasicCli
import ConfigMount
from io import StringIO
from ast import literal_eval
import LazyMount
from CliPlugin.P4RuntimeModel import ( P4RtStatus, P4RtDeviceStatusEntry,
                                       P4RtTransport, P4RuntimeElectionId,
                                       P4RtDeviceStatus,
                                       P4RtDeviceStatusEntryDetail,
                                       P4RtControllerPktMetadataEntry,
                                       P4RtDeviceCounters, P4RtDeviceCounterEntry )
import CliMatcher
import CliCommand
import CliParser
import ShowCommand
from Toggles.P4RuntimeConfigToggleLib import (
   toggleP4RuntimeAccountingRequestsEnabled,
   toggleP4RuntimeAUPEnabled )

p4RtSystemConfig = None
p4RtDeviceConfig = None
p4RtSystemStatus = None
p4RtDeviceStatus = None
p4RtHwStatus = None

def p4RuntimeSupportedGuard( mode, token ):
   if p4RtHwStatus.p4RuntimeSupported:
      return None
   return CliParser.guardNotThisPlatform

matcherP4Runtime = CliMatcher.KeywordMatcher( 'p4-runtime',
                                              helpdesc='Show P4Runtime state' )
nodeP4Runtime = CliCommand.Node( matcher=matcherP4Runtime,
                                 guard=p4RuntimeSupportedGuard )

def addDeviceStateToModelEntry( entry, deviceName, devId, devStatus ):
   entry.deviceName = deviceName
   entry.deviceId = devId
   if devStatus and devStatus.connection:
      conn = next( iter( devStatus.connection.values() ) )
      if conn.isPrimary:
         primaryElectionId = P4RuntimeElectionId()
         primaryElectionId.high = conn.electionId.high
         primaryElectionId.low = conn.electionId.low
         entry.primaryElectionId = primaryElectionId

def populateDeviceConnections( mode ):
   for deviceName, devId in sorted( p4RtDeviceConfig.fapNameToDeviceId.items() ):
      devStatus = p4RtDeviceStatus.deviceStatus.get( devId )
      if devStatus:
         entry = P4RtDeviceStatusEntry()
         addDeviceStateToModelEntry( entry, deviceName, devId, devStatus )
         yield deviceName, entry

def showP4RuntimeStatus( mode, args ):
   model = P4RtStatus()
   model.enabled = p4RtSystemStatus.enabled
   numClients = sum( len( devStatus.connection ) for
                     devStatus in p4RtDeviceStatus.deviceStatus.values() )
   model.numClients = numClients if p4RtSystemStatus.enabled else 0
   model.lastServiceStartTimeStamp = p4RtSystemStatus.lastServiceStartTimeStamp
   model.authzEnabled = p4RtSystemStatus.authzEnabled
   if not p4RtSystemStatus.enabled or not p4RtSystemStatus.serverReady:
      return model
   transport = P4RtTransport()
   transport.vrfName = p4RtSystemConfig.grpcServerVrfName
   transport.port = p4RtSystemConfig.grpcServerPort
   transport.sslProfile = None if not p4RtSystemStatus.sslProfile else \
            p4RtSystemStatus.sslProfile
   if toggleP4RuntimeAUPEnabled():
      transport.authnUsernamePriority = [
            aupCli.authnUsernamePriorityToCLI[ v.source ]
               for v in p4RtSystemStatus.authnUsernamePriority.values() ]
   else:
      transport.certUsernameAuthn = p4RtSystemStatus.certUsernameAuthn
   if toggleP4RuntimeAccountingRequestsEnabled():
      transport.accountingRequests = p4RtSystemStatus.accountingRequests
   model.transport = transport
   model.devices = populateDeviceConnections( mode )
   return model

def populateDevices( mode ):
   for deviceName, devId in sorted( p4RtDeviceConfig.fapNameToDeviceId.items() ):
      devStatus = p4RtDeviceStatus.deviceStatus.get( devId )
      if devStatus is None:
         continue
      entry = P4RtDeviceStatusEntryDetail()
      addDeviceStateToModelEntry( entry, deviceName, devId, devStatus )
      entry.writeRpcReceived = devStatus.tableWriteRequestDone
      if devStatus.p4ControllerPktMetadata is None:
         yield deviceName, entry
         continue
      # Add pkt metadata Entry
      for metadataEntry in devStatus.p4ControllerPktMetadata.metadataEntry.values():
         if metadataEntry.metadataName not in [ 'packet_in', 'packet_out' ]:
            # We only care about packetIn/packetOut metadata
            continue
         messageType = 'PacketIn' if \
               metadataEntry.metadataName == 'packet_in' else 'PacketOut'
         for metaId, field in sorted( metadataEntry.metadataField.items() ):
            listEntry = P4RtControllerPktMetadataEntry()
            listEntry.messageType = messageType
            listEntry.metadataId = metaId
            listEntry.name = field.name
            listEntry.bitWidth = field.bitwidth
            entry.packetMetadata.append( listEntry )
      yield deviceName, entry

def showP4RuntimeDeviceStatus( mode, args ):
   model = P4RtDeviceStatus()
   if not p4RtSystemConfig.enabled:
      return model
   model.devices = populateDevices( mode )
   return model

def showP4RuntimeDeviceCounters( mode, args ):
   cmd = "deviceCounters"
   buff = StringIO()
   AgentCommandRequest.runSocketCommand( mode.entityManager, "P4Runtime",
                                         "callback", cmd, stringBuff=buff )
   model = P4RtDeviceCounters()
   output = buff.getvalue()
   try:
      # pylint: disable-msg=eval-used
      outputDict = literal_eval( output )
   except SyntaxError:
      mode.addError( output )
      return model

   for deviceName, entry in outputDict[ 'p4RtDeviceCounters' ].items():
      modelEntry = P4RtDeviceCounterEntry()
      modelEntry.setAttrsFromDict( entry )
      model.devicesCounter[ deviceName ] = modelEntry
   return model

# --------------------------------------------------------------------------------
# show p4-runtime
# --------------------------------------------------------------------------------
class ShowP4RuntimeStatus( ShowCommand.ShowCliCommandClass ):
   syntax = 'show p4-runtime'
   data = {
      'p4-runtime' : nodeP4Runtime,
   }
   cliModel = P4RtStatus
   handler = showP4RuntimeStatus

BasicCli.addShowCommandClass( ShowP4RuntimeStatus )

# --------------------------------------------------------------------------------
# show p4-runtime device
# --------------------------------------------------------------------------------
class ShowP4RuntimeDeviceStatus( ShowCommand.ShowCliCommandClass ):
   syntax = 'show p4-runtime device'
   data = {
      'p4-runtime' : nodeP4Runtime,
      'device' : 'P4Runtime device status'
   }
   cliModel = P4RtDeviceStatus
   handler = showP4RuntimeDeviceStatus

BasicCli.addShowCommandClass( ShowP4RuntimeDeviceStatus )

# --------------------------------------------------------------------------------
# show p4-runtime device counters
# --------------------------------------------------------------------------------
class ShowP4RuntimeDeviceCounters( ShowCommand.ShowCliCommandClass ):
   syntax = 'show p4-runtime device counters packet-io'
   data = {
      'p4-runtime' : nodeP4Runtime,
      'device' : 'P4Runtime device status',
      'counters' : 'P4Runtime device counters',
      'packet-io' : 'P4Runtime packet IO device counters'
   }
   cliModel = P4RtDeviceCounters
   handler = showP4RuntimeDeviceCounters

BasicCli.addShowCommandClass( ShowP4RuntimeDeviceCounters )

def Plugin( entityManager ):
   global p4RtSystemStatus, p4RtDeviceStatus, p4RtSystemConfig, p4RtDeviceConfig
   global p4RtHwStatus
   p4RtSystemConfig = ConfigMount.mount( entityManager, "p4runtime/config/system",
                                         "P4RuntimeConfig::P4RtSystemConfig", "w" )
   p4RtDeviceConfig = ConfigMount.mount( entityManager,
                                         "p4runtime/config/deviceConfig",
                                         "P4RuntimeConfig::P4RtDeviceConfig", "w" )
   p4RtSystemStatus = LazyMount.mount( entityManager, "p4runtime/status/system",
                                       "P4RuntimeConfig::P4RtSystemStatus", "r" )
   p4RtDeviceStatus = LazyMount.mount( entityManager, "p4runtime/status/devices",
                                       "P4RuntimeConfig::P4RtDeviceStatusCol", "r" )
   p4RtHwStatus = LazyMount.mount( entityManager, "p4runtime/status/hardware/status",
                                   "P4Runtime::HwStatus", "r" )
