#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
from CliModel import Int, Str, Dict, Bool, Model, Enum, List
from ArnetModel import IpGenericAddr
from CliPlugin.PolicyMapModel import ActionModel, PolicyClassMapModel

class FwdActionModel( ActionModel ):
   def renderText( self ):
      raise NotImplementedError

class NoActionModel( FwdActionModel ):
   def renderText( self ):
      return ''

class DropActionModel( FwdActionModel ):
   def renderText( self ):
      return 'drop'

class NexthopModel( FwdActionModel ):
   nexthops = Dict( keyType=int,
                    valueType=IpGenericAddr,
                    help='Set of next hops' )
   recursive = Bool( help="recursive or not" )
   vrfName = Str( help="VRF name" )

   def renderText( self ):
      nexthopList = sorted( [ x.ip for x in self.nexthops.values() ] )
      nexthopStrList = [ x.stringValue for x in nexthopList ]
      nexthopStr = ' '.join( nexthopStrList )
      return 'set nexthop ' + \
            ( "recursive " if self.recursive else '' ) + \
            nexthopStr + ( ( ' vrf ' + self.vrfName ) \
            if self.vrfName != '' else '' ) 

class NexthopGroupModel( FwdActionModel ):
   nexthopGroup = Str( help='Next hop group name' )

   def renderText( self ):
      # pylint: disable-next=consider-using-f-string
      return 'set nexthop-group %s' % self.nexthopGroup

class SetDscpModel( ActionModel ):
   dscpValue = Int( help='DSCP value' )
   def renderText( self ):
      return 'set dscp %d' % self.dscpValue # pylint: disable=consider-using-f-string

class SetTtlModel( ActionModel ):
   ttlValue = Int( help='TTL value' )
   def renderText( self ):
      return 'set ttl %d' % self.ttlValue # pylint: disable=consider-using-f-string

pbrActionTypes = Tac.Type( 'PolicyMap::PbrActionType' ).attributes
printableActionType = {
   'noAction' : 'No PBR action',
   'dropAction' : 'Drop packet',
   'nexthopAction' : 'Route to nexthop',
   'nexthopGroupAction' : 'Route to nexthop group',
   'ttlAction' : 'Set TTL of outgoing packet to'
}

class PbrPolicyClassMapVrfModel( Model ):
   action = Enum( values=pbrActionTypes,
                  default='noAction',
                  help='Type of routing action programmed for the vrf' )
   nexthops = List( valueType=IpGenericAddr,
                    optional=True,
                    help='Nexthops programmed for the vrf' )
   ttl = Int( optional=True, help='TTL value' )
   nexthopGroup = Str( optional=True,
                       help ='Nexthop group programmed for the vrf' )
   actionVrf = Str( optional=True, 
                    help='Name for egress vrf' )

class PbrPolicyClassMapModel( PolicyClassMapModel ):
   __revision__ = 2
   vrfs = Dict( valueType=PbrPolicyClassMapVrfModel, optional=True,
                help="mapping of vrf name to policies" )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         if "default" in dictRepr[ "vrfs" ]:
            dictRepr[ "activeRoutingAction" ] = \
                  dictRepr[ "vrfs" ][ "default" ][ "action" ] 
            dictRepr[ "vrfName" ] = "default"

            if "nexthops" in dictRepr[ "vrfs" ][ "default" ]:
               dictRepr[ "activeNexthop" ] = \
                     dictRepr[ "vrfs" ][ "default" ][ "nexthops" ]

            if "nexthopGroup" in dictRepr[ "vrfs" ][ "default" ]:
               dictRepr[ "activeNexthopGroup" ] = \
                     dictRepr[ "vrfs" ][ "default" ][ "nexthopGroup" ]
         else:
            dictRepr[ "activeRoutingAction" ] = "noAction" 
         del dictRepr["vrfs"]

      return dictRepr

   def render( self ):
      PolicyClassMapModel.render( self )
      programmedStr = '    Active routing action: '

      for vrfName, vrfModel in self.vrfs.items(): 
         programmedStr = programmedStr + '\n' + '    VRF ' + vrfName + '\n'

         if vrfModel.action == 'nexthopAction':
            nexthopList = sorted( [ x.ip for x in vrfModel.nexthops ] )
            nexthopStrList = [ x.stringValue for x in nexthopList ]
            nexthopStrAddr = ' '.join( nexthopStrList )

            nexthopStr = '        ' + \
                         printableActionType[ 'nexthopAction' ]
            programmedStr = programmedStr + nexthopStr + " " + nexthopStrAddr + \
                            ' ' + vrfModel.actionVrf
            if vrfModel.ttl != 0:
               programmedStr = programmedStr + '\n' + '        ' + \
                               printableActionType[ 'ttlAction' ] + ' ' + \
                               str( vrfModel.ttl )

         elif vrfModel.action == 'nexthopGroupAction':
            programmedStr = programmedStr + '        ' + \
                            printableActionType[ 'nexthopGroupAction' ] + " " + \
                            vrfModel.nexthopGroup
            if vrfModel.ttl != 0:
               programmedStr = programmedStr + '\n' + '        ' + \
                               printableActionType[ 'ttlAction' ] + ' ' + \
                               str( vrfModel.ttl )
         else:
            programmedStr = programmedStr + '        ' + \
                            printableActionType[ vrfModel.action ]

      print( programmedStr )

