#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import BasicCliModes
import CliCommand
import CliMatcher
import ShowCommand
import HostnameCli
from CliPlugin.VrfCli import VrfNameExprFactory
from CliPlugin import IntfCli
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
from CliPlugin.ServerCliModel import TWServer, TWSummary
import LazyMount
import ConfigMount
import Tac
import Ark
from datetime import datetime
from ipaddress import IPv6Address
from IpLibConsts import DEFAULT_VRF
from PerfsonarTwampConsts import pingCommand, minPort, maxPort, defaultLocalAddress
from PerfsonarTwampConsts import defaultListenPort
from PerfsonarTwampConsts import defaultMaxControlSessions, maxControlSessions
from PerfsonarTwampConsts import defaultMinTestPort, defaultMaxTestPort
from PerfsonarTwampConsts import defaultUser, defaultGroup, defaultAuthMode
from PerfsonarTwampConsts import serverKeyword, clientKeyword
import re

twampConfig = None
twampStatus = None
twampTest = None
intfStatusAll = None

procResults = {}

# Populate dictionary to enable mapping of raw ping results to
# content for Sysdb and CLI output
def populateProcResults():

   procResults[ 'SID' ] = { 'cliDesc': 'Session ID',
                           'sysdbDesc': 'sessID',
                           'value': '', 'display': True }
   procResults[ 'FROM_HOST' ] = { 'cliDesc': 'Source hostname',
                                 'sysdbDesc': 'fromHost',
                                 'value': '', 'display': True }
   procResults[ 'FROM_ADDR' ] = { 'cliDesc': 'Source address',
                                 'sysdbDesc': 'fromAddr',
                                 'value': '', 'display': True }
   procResults[ 'FROM_PORT' ] = { 'cliDesc': 'Source test port',
                                 'sysdbDesc': 'fromPort',
                                 'value': '', 'display': True }
   procResults[ 'TO_HOST' ] = { 'cliDesc': 'Destination hostname',
                               'sysdbDesc': 'toHost',
                               'value': '', 'display': True }
   procResults[ 'TO_ADDR' ] = { 'cliDesc': 'Destination address',
                               'sysdbDesc': 'toAddr',
                               'value': '', 'display': True }
   procResults[ 'TO_PORT' ] = { 'cliDesc': 'Destination test port',
                               'sysdbDesc': 'toPort',
                               'value': '', 'display': True }
   procResults[ 'UNIX_START_TIME' ] = { 'cliDesc': 'Test start time',
                                        'sysdbDesc': 'startTime',
                                        'value': '', 'display': True }
   procResults[ 'UNIX_END_TIME' ] = { 'cliDesc': 'Test end time',
                                      'sysdbDesc': 'endTime',
                                      'value': '', 'display': True }
   procResults[ 'SENT' ] = { 'cliDesc': 'Packets sent',
                            'sysdbDesc': 'pktsSent',
                            'value': '', 'display': True }
   procResults[ 'LOST' ] = { 'cliDesc': 'Packets lost',
                            'sysdbDesc': 'pktsLost',
                            'value': '', 'display': True }
   procResults[ 'DUPS_FWD' ] = { 'cliDesc': 'Packets sent duplicates',
                                'sysdbDesc': 'pktsSendDup',
                                'value': '', 'display': True }
   procResults[ 'DUPS_BCK' ] = { 'cliDesc': 'Packets reflected duplicates',
                                'sysdbDesc': 'pktsReflDup',
                                'value': '', 'display': True }
   procResults[ 'MIN' ] = { 'cliDesc': 'Round-trip time min',
                           'sysdbDesc': 'roundTripTimeMin',
                           'value': '', 'display': True }
   procResults[ 'MAX' ] = { 'cliDesc': 'Round-trip time max',
                           'sysdbDesc': 'roundTripTimeMax',
                           'value': '', 'display': True }
   procResults[ 'MEDIAN' ] = { 'cliDesc': 'Round-trip time median',
                              'sysdbDesc': 'roundTripTimeMedian',
                              'value': '', 'display': True }
   procResults[ 'MAXERR' ] = { 'cliDesc': 'Round-trip time error',
                              'sysdbDesc': 'roundTripTimeErr',
                              'value': '', 'display': True }
   procResults[ 'MIN_FWD' ] = { 'cliDesc': 'Send time min',
                               'sysdbDesc': 'sendTimeMin',
                               'value': '', 'display': True }
   procResults[ 'MAX_FWD' ] = { 'cliDesc': 'Send time max',
                               'sysdbDesc': 'sendTimeMax',
                               'value': '', 'display': True }
   procResults[ 'MEDIAN_FWD' ] = { 'cliDesc': 'Send time median',
                                  'sysdbDesc': 'sendTimeMedian',
                                  'value': '', 'display': True }
   procResults[ 'MAXERR_FWD' ] = { 'cliDesc': 'Send time error',
                                  'sysdbDesc': 'sendTimeErr',
                                  'value': '', 'display': True }
   procResults[ 'MIN_BCK' ] = { 'cliDesc': 'Reflect time min',
                               'sysdbDesc': 'reflectTimeMin',
                               'value': '', 'display': True }
   procResults[ 'MAX_BCK' ] = { 'cliDesc': 'Reflect time max',
                               'sysdbDesc': 'reflectTimeMax',
                               'value': '', 'display': True }
   procResults[ 'MEDIAN_BCK' ] = { 'cliDesc': 'Reflect time median',
                                  'sysdbDesc': 'reflectTimeMedian',
                                  'value': '', 'display': True }
   procResults[ 'MAXERR_BCK' ] = { 'cliDesc': 'Reflect time error',
                                  'sysdbDesc': 'reflectTimeErr',
                                  'value': '', 'display': True }
   procResults[ 'MIN_PROC' ] = { 'cliDesc': 'Reflector processing time min',
                                'sysdbDesc': 'reflectorProcTimeMin',
                                'value': '', 'display': True }
   procResults[ 'MAX_PROC' ] = { 'cliDesc': 'Reflector processing time max',
                                'sysdbDesc': 'reflectorProcTimeMax',
                                'value': '', 'display': True }
   procResults[ 'PDV' ] = { 'cliDesc': 'Two-way jitter',
                           'sysdbDesc': 'twoWayJitter',
                           'value': '', 'display': True }
   procResults[ 'PDV_FWD' ] = { 'cliDesc': 'Send jitter',
                               'sysdbDesc': 'sendJitter',
                               'value': '', 'display': True }
   procResults[ 'PDV_BCK' ] = { 'cliDesc': 'Reflect jitter',
                               'sysdbDesc': 'reflectJitter',
                               'value': '', 'display': True }
   procResults[ 'MINTTL_FWD' ] = { 'cliDesc': 'Send hops max',
                                   'sysdbDesc': 'sendTtlMin',
                                   'value': '', 'display': True }
   procResults[ 'MAXTTL_FWD' ] = { 'cliDesc': 'Send hops min',
                                   'sysdbDesc': 'sendTtlMax',
                                   'value': '', 'display': True }
   procResults[ 'MINTTL_BCK' ] = { 'cliDesc': 'Reflect hops max',
                                   'sysdbDesc': 'reflectTtlMin',
                                   'value': '', 'display': True }
   procResults[ 'MAXTTL_BCK' ] = { 'cliDesc': 'Reflect hops min',
                                   'sysdbDesc': 'reflectTtlMax',
                                   'value': '', 'display': True }


populateProcResults()

# ------------------------------------------
# Command :
#     show twserver [vrf <vrfname>] [summary]
# ------------------------------------------

matcherShowServer = CliMatcher.KeywordMatcher( serverKeyword,
  helpdesc='Details about configured/running TWAMP server(s)' )

# Find VRFs where server processes potentially running
def findCurrentServers():
   # return a list of vrfs (including 'default') where attempt has been made
   # to start the server e.g. [ 'default', 'vrf1', 'vrf2', 'vrf-3' ]
   sysCtlStatus = Tac.run( [ 'systemctl', 'list-units', '--plain', 'twamp*' ],
                           stdout=Tac.CAPTURE, ignoreReturnCode=True )
   twampVrfs = re.findall( r'\s+twamp[^ ]+', sysCtlStatus )
   twampVrfs.sort()

   return [ re.split( r'@|\.', vrf )[ 1 ] if vrf.find( '@' ) > 0 else 'default'
      for vrf in twampVrfs ]

def opt( var, default ):
   return var if var is not None else default

def handleShow( mode, args ):
   # Find the VRFs with servers potentially running
   serverVrfs = findCurrentServers()
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   serverInfo = twampStatus.info.get( vrfName )
   result = TWServer()

   serverProcessRunning = vrfName in serverVrfs
   if serverInfo is None:
      return result
   if serverInfo.serverRunning and serverProcessRunning:
      result.startTime = Ark.switchTimeToUtc( serverInfo.serverStartTime )
      serverConfig = twampConfig.server.get( vrfName )
      if serverConfig:
         result.listenPort = opt( serverConfig.listenPort, defaultListenPort )
         result.localIpOrHost = opt( serverConfig.localIpOrHost, '0' )
         result.minTestPort = opt( serverConfig.minTestPort, 0 )
         result.maxTestPort = opt( serverConfig.maxTestPort, 0 )
         result.maxControlSessions = opt( serverConfig.maxControlSessions, 0 )
   return result

def handleShowSummary( mode, args ):
   serverVrfs = findCurrentServers()
   return TWSummary( configuredVrfs=list( twampConfig.server ),
                        activeVrfs=serverVrfs,
                        serverStartAttempts=twampStatus.serverStartAttempts,
                        serverStartSuccesses=twampStatus.serverStartSuccesses,
                        serverStopAttempts=twampStatus.serverStopAttempts,
                        serverStopSuccesses=twampStatus.serverStopSuccesses )

class ShowTwServer( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show %s [ (vrf VRF ) ]' % serverKeyword )
   data = {
      serverKeyword: matcherShowServer,
      'vrf': 'Server information in specific VRF',
      'VRF': VrfNameExprFactory( inclAllVrf=True )
   }
   cliModel = TWServer
   handler = handleShow

class ShowTwServerSummary( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show %s summary' % serverKeyword )
   data = {
      serverKeyword: matcherShowServer,
      'summary': 'Summary server information (across all VRFs)'
   }
   cliModel = TWSummary
   handler = handleShowSummary

BasicCli.addShowCommandClass( ShowTwServer )
BasicCli.addShowCommandClass( ShowTwServerSummary )

# ------------------------------------------
# Command:
#     twping
#             [vrf <vrf name>]
#             address <hostname or IP address>
#             port <port>
#             [auth-mode ( open | authenticated | encrypted | mixed )]
#             [count <count>]
#             [dscp-value <dscp-value>]
#             [end-delay <end-delay>]
#             [interface <interface>]
#             [max-test-port <max-test-port>]
#             [min-test-port <min-test-port>]
#             [padding <padding>]
#             [source-address <source-address>]
#             [spacing <spacing>]
#             [start-delay <start-delay>]
#             [timeout <timeout>]
#             [user <user>]
#
# Mode:
#     Enable
# ------------------------------------------

matcherClient = CliMatcher.KeywordMatcher( clientKeyword,
  helpdesc='Set up TWAMP session and send test messages to specified server' )

def parsePingResults( rawPingResults ):

   # Initially clear the any results from previous ping
   for val in procResults.values():
      val[ 'value' ] = ''

   # Now parse and store the latest results
   resultLines = rawPingResults.split( "\n" )
   for line in resultLines:
      metricValue = line.split()
      if len( metricValue ) != 2:
         continue

      metric = metricValue[ 0 ]
      value = metricValue[ 1 ]

      if metric not in procResults:
         continue
      procResults[ metric ][ 'value' ] = value

# Write ping results to Sysdb
def writePingResults( vrfName, ipAddrOrHostname, port ):

   # Get existing object using the appropriate key and
   # if object does not exist, create it
   vrfResults = twampTest.result.newMember( vrfName )
   vrfAddress = vrfResults.address.newMember( ipAddrOrHostname )
   result = vrfAddress.port.newMember( port )

   for metric in procResults.values():
      sysdbKey = metric[ 'sysdbDesc' ]
      value = metric[ 'value' ]
      setattr( result, sysdbKey, value )

# Convert TTL into number of hops traversed by packet
def ttlHopsConverter( ttl ):

   try:
      ttlVal = float( ttl )
   except ValueError:
      return None

   return 255 - int( ttlVal )

# Potentially convert time units from seconds to milliseconds, microseconds
# or nanoseconds, with a slight bias towards milliseconds
def timeFormatter( timeInput ):

   try:
      timeValue = float( timeInput )
   except ValueError:
      return None

   timeUnit = 's'

   if timeValue < 1:
      timeValue *= 1000
      timeUnit = 'ms'

      if timeValue < 0.01:
         timeValue *= 1000
         timeUnit = 'us'

         if timeValue < 0.1:
            timeValue *= 1000
            timeUnit = 'ns'

   return f'{timeValue:.3f} {timeUnit}'

# Write ping results to stdout
def printPingResults():

   printedLine = False

   # Print results, exploiting fact that procResults[ metric ][ 'value' ] != ''
   # only if metric updated in latest ping.  Try to be paranoid when processing
   # values in case results data from twping executable in unexpected format/type.

   # Print start/end time and session ID
   timeFmt = '%Y %b %d %H:%M:%S'
   startTime = procResults.get( 'UNIX_START_TIME' )
   try:
      startTimeVal = float( startTime[ 'value' ] )
      formatedTime = datetime.fromtimestamp( startTimeVal )
      print( '{}: {}'.format( startTime[ 'cliDesc' ],
            formatedTime.strftime( timeFmt ) ) )
      printedLine = True
   except ValueError:
      startTimeVal = 0

   endTime = procResults.get( 'UNIX_END_TIME' )
   try:
      endTimeVal = float( endTime[ 'value' ] )
      formatedTime = datetime.fromtimestamp( endTimeVal )
      print( '{}: {}'.format( endTime[ 'cliDesc' ],
            formatedTime.strftime( timeFmt ) ) )
      if startTimeVal:
         elapsedTime = endTimeVal - startTimeVal
         print( 'Test duration: %.3f s' % elapsedTime )
      printedLine = True
   except ValueError:
      pass

   sessionId = procResults.get( 'SID' )
   try:
      sessionIdVal = int( sessionId[ 'value' ], 16 )
      sessionIdFmt = IPv6Address( sessionIdVal )
      print( '{}: {}'.format( sessionId[ 'cliDesc' ], sessionIdFmt ) )
      printedLine = True
   except ValueError:
      pass

   if printedLine:
      print()
      printedLine = False

   # Print endpoints information
   keys = [ 'FROM_ADDR', 'FROM_HOST', 'FROM_PORT', 'TO_ADDR', 'TO_HOST', 'TO_PORT' ]
   printedLine = printMetrics( keys, False )

   if printedLine:
      print()
      printedLine = False

   # Print packet metrics
   packetsSent = procResults.get( 'SENT' )
   try:
      packetsSentVal = int( packetsSent[ 'value' ] )
      print( '%s: %d' % ( packetsSent[ 'cliDesc' ], packetsSentVal ) )
      printedLine = True
   except ValueError:
      packetsSentVal = 0

   packetsLost = procResults.get( 'LOST' )
   try:
      packetsLostVal = int( packetsLost[ 'value' ] )
      lostString = '%s: %d' % ( packetsLost[ 'cliDesc' ], packetsLostVal )
      if packetsSentVal > 0:
         lossPercent = 100 * float( packetsLostVal ) / float( packetsSentVal )
         lostString += ' (%.3f%%)' % lossPercent
      print( lostString )
      printedLine = True
   except ValueError:
      pass


   keys = [ 'DUPS_FWD', 'DUPS_BCK' ]
   printedLine |= printMetrics( keys, False )

   if printedLine:
      print()
      printedLine = False

   # Print time measurements
   prefixes = [ 'MIN', 'MEDIAN', 'MAX', 'MAXERR' ]
   suffixes = [ '', '_FWD', '_BCK' ]
   keys = []
   for suffix in suffixes:
      for prefix in prefixes:
         keys.append( prefix + suffix )
   printedLine = printMetrics( keys )

   if printedLine:
      print()
      printedLine = False

   # Print jitter results
   keys = [ 'PDV', 'PDV_FWD', 'PDV_BCK' ]
   printedLine = printMetrics( keys )

   if printedLine:
      print()
      printedLine = False

   # Print processing time results
   keys = [ 'MIN_PROC', 'MAX_PROC' ]
   printedLine = printMetrics( keys )

   if printedLine:
      print()
      printedLine = False

   # Print TTL metrics
   keys = [ 'MAXTTL_FWD', 'MINTTL_FWD', 'MAXTTL_BCK', 'MINTTL_BCK' ]
   printedLine = printMetrics( keys, False, True )

   if printedLine:
      print()

# Utility print routine.  Use to specified key(s) to retrieve metric
# and print out the retrieved info.
def printMetrics( keys=tuple(), timeFormat=True, ttlFormat=False ):

   printed = False

   for key in keys:
      metric = procResults.get( key )
      value = metric[ 'value' ]
      if value:

         if timeFormat:
            # Convert time values to appropriate SI units for consumption
            value = timeFormatter( value )
            if value is None:
               continue

         if ttlFormat:
            # Convert TTL values to hops as arguably more intuitive
            value = ttlHopsConverter( value )
            if value is None:
               continue

         print( '{}: {}'.format( metric[ 'cliDesc' ], value ) )
         printed = True

   return printed

# Validate destination for ping before calling underlying
# TWAMP ping executable to send the appropriate packets.
def handleClient( mode, args ):

   vrfName = args.get( 'VRF', DEFAULT_VRF )

   # Address and port are mandatory parameters
   ipAddrOrHostname = args[ 'ADDRESS' ]
   port = args[ 'PORT' ]

   if ( HostnameCli.resolveHostname( mode, ipAddrOrHostname, doWarn=False ) ==
        HostnameCli.Resolution.UNRESOLVED ):
      mode.addError( "%s is invalid server address/hostname" % ipAddrOrHostname )
      return

   # Use IP address and port to start composing command
   # If non-default VRF specified, prepend appropriate prefix to
   # ensure ping executed in correct VRF, i.e. namespace

   # Compulsory options to use when executing twping command
   # -M: Print results in 'machine-readble' format, i.e. as key-value pairs.
   #       + These will be stored in procResults[ key ] = {'value' = value} )
   # -U: Print timestamps in Unix timestamps
   #       + Allows easy conversion to day/time format
   # -b: Resolution of bins in histogram of packet delays (default = 0.0001 s)
   #       + Produces greater precision in median delay and jitter calculations
   options = '-U -M -b 0.00001'

   if vrfName != DEFAULT_VRF:
      pre = 'ip netns exec ns-%s' % vrfName
      cmd = "%s %s %s [%s]:%d" % ( pre, pingCommand, options,
                                   ipAddrOrHostname, port )
   else:
      cmd = "%s %s [%s]:%d" % ( pingCommand, options,
                                ipAddrOrHostname, port )

   # If count was specified, append to command
   pktCount = args.get( 'COUNT' )
   if pktCount:
      cmd += " -c %d" % pktCount[ 0 ]

   # If spacing was specified, append to command.  "f" used to specify fixed
   # intervals (as opposed to default of exponentially-distributed intervals).
   pktSpacing = args.get( 'SPACING' )
   if pktSpacing:
      cmd += " -i %ff" % pktSpacing[ 0 ]

   # If start delay was specified, append to command
   startDelay = args.get( 'START_DELAY' )
   if startDelay:
      cmd += " -z %d" % startDelay[ 0 ]

   # If end delay was specified, append to command
   endDelay = args.get( 'END_DELAY' )
   if endDelay:
      cmd += " -E %d" % endDelay[ 0 ]

  # If timeout was specified, append to command
   timeout = args.get( 'TIMEOUT' )
   if timeout:
      cmd += " -L %d" % timeout[ 0 ]

   # If padding was specified, append to command
   padding = args.get( 'PADDING' )
   if padding:
      cmd += " -s %d" % padding[ 0 ]

   minTestPort = args.get( 'MIN_TEST_PORT', minPort )
   if minTestPort != minPort:
      minTestPort = minTestPort[ 0 ]

   maxTestPort = args.get( 'MAX_TEST_PORT', maxPort )
   if maxTestPort != maxPort:
      maxTestPort = maxTestPort[ 0 ]

   if minTestPort == maxTestPort:
      mode.addError( "Invalid test ports specified ( i.e. min (%d) equal"
                     " to max (%d))" % ( minTestPort, maxTestPort ) )
      return

   if minTestPort > maxTestPort:
      mode.addError( "Invalid test ports specified ( i.e. min (%d) > "
                     " max (%d))" % ( minTestPort, maxTestPort ) )
      return

   # Either min or max set; update ping command accordingly
   if minTestPort != minPort or maxTestPort != maxPort:
      cmd += " -P %d-%d" % ( minTestPort, maxTestPort )

   # If DSCP value was specified, append to command
   dscpValue = args.get( 'DSCP_VALUE' )
   if dscpValue:
      cmd += " -D %d" % dscpValue[ 0 ]

   # If source address was specified, append to command
   sourceAddress = args.get( 'SOURCE_ADDRESS' )
   if sourceAddress:
      cmd += " -S %s" % str( sourceAddress[ 0 ] )

   # If interface specified, find associated device & append to command appropriately
   interface = args.get( 'INTERFACE' )
   if interface:

      intfName = str( interface[ 0 ] )
      try:
         devName = intfStatusAll.intfStatus[ intfName ].deviceName
      except KeyError:
         mode.addError( "Cannot find device name for interface %s" % intfName )
         return

      cmd += " -B %s" % devName

   # If user was specified, append to command
   user = args.get( 'USER' )
   if user:
      cmd += " -u %s" % user[ 0 ]

   # Handle authentication mode setting
   # Default is 'mixed', i.e. 'AEO'
   authMode = args.get( 'AUTH_MODE' )
   if authMode:
      authMode = authModesMap[ authMode[ 0 ] ]
      cmd += ' -A %s' % authMode

   try:
      pingOutput = Tac.run( cmd.split(), asRoot=True,
                            stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )

   except Tac.SystemCommandError as e:
      errString = f'Error running "{cmd}": {e.output}'
      mode.addError( errString )
      return
   except OSError as e:
      errString = f'Error executing "{cmd}": {e.strerror}'
      mode.addError( errString )
      return

   # Now parse response, and write to Sysdb and stdout
   parsePingResults( pingOutput )
   writePingResults( vrfName, ipAddrOrHostname, port )
   printPingResults()

authModes = {
   'open': 'No encryption carried out',
   'authenticated': 'Authenticated authentication mode',
   'encrypted': 'Encrypted authentication mode',
   'mixed': 'Use most strict mutually supported authentication mode',
   }

class TwPingOptionsExpr( CliCommand.CliExpression ):
   expression = ( '{ ( count COUNT ) | '
                    '( start-delay START_DELAY ) | '
                    '( end-delay END_DELAY ) | '
                    '( spacing SPACING ) | '
                    '( padding PADDING ) | '
                    '( timeout TIMEOUT ) | '
                    '( min-test-port MIN_TEST_PORT ) | '
                    '( max-test-port MAX_TEST_PORT ) | '
                    '( dscp-value DSCP_VALUE ) | '
                    '( source-address SOURCE_ADDRESS ) | '
                    '( auth-mode AUTH_MODE ) | '
                    '( user USER ) | '
                    '( interface INTERFACE ) } ' )

   data = {
      'count': CliCommand.singleKeyword( 'count',
         helpdesc='Number of test packets' ),
      'start-delay': CliCommand.singleKeyword( 'start-delay',
         helpdesc='Time to wait before starting test (secs)' ),
      'end-delay': CliCommand.singleKeyword( 'end-delay',
         helpdesc='Time to wait before sending stop session message (secs)' ),
      'spacing': CliCommand.singleKeyword( 'spacing',
         helpdesc='Average inter-packet time (secs)' ),
      'timeout': CliCommand.singleKeyword( 'timeout',
         helpdesc='Time to wait for test packet before declaring it lost (secs)' ),
      'padding': CliCommand.singleKeyword( 'padding',
         helpdesc='Size of padding to be added to each test packet (bytes)' ),
      'min-test-port': CliCommand.singleKeyword( 'min-test-port',
         helpdesc='Minimum UDP port to be used for tests' ),
      'max-test-port': CliCommand.singleKeyword( 'max-test-port',
         helpdesc='Maximum UDP port to be used for tests' ),
      'dscp-value': CliCommand.singleKeyword( 'dscp-value',
         helpdesc='DSCP value in TOS byte in IP header of test packets' ),
      'source-address': CliCommand.singleKeyword( 'source-address',
         helpdesc='Source address for control and test packets' ),
      'auth-mode': CliCommand.singleKeyword( 'auth-mode',
         helpdesc='Authentication mode' ),
      'user': CliCommand.singleKeyword( 'user',
         helpdesc='User name used for deriving authentication session keys' ),
      'interface': CliCommand.singleKeyword( 'interface',
         helpdesc='Interface to use for control and test packets' ),
      'COUNT': CliMatcher.IntegerMatcher( 10, 10000,
         helpdesc='Number of test packets' ),
      'START_DELAY': CliMatcher.IntegerMatcher( 1, 60,
         helpdesc='Time to wait before starting test (secs)' ),
      'END_DELAY': CliMatcher.IntegerMatcher( 1, 60,
         helpdesc='Time to wait before sending stop session message (secs)' ),
      'TIMEOUT': CliMatcher.IntegerMatcher( 1, 60,
         helpdesc='Time to wait for test packet before declaring it lost (secs)' ),
      'SPACING': CliMatcher.FloatMatcher( 0, 60,
         helpdesc='Average inter-packet time (secs)', precisionString='%.2f' ),
      'PADDING': CliMatcher.IntegerMatcher( 1, 100000,
         helpdesc='Size of padding to be added to each test packet (bytes)' ),
      'MIN_TEST_PORT': CliMatcher.IntegerMatcher( minPort,
         maxPort, helpdesc='Minimum UDP port to be used for tests' ),
      'MAX_TEST_PORT': CliMatcher.IntegerMatcher( minPort,
         maxPort, helpdesc='Maximum UDP port to be used for tests' ),
      'DSCP_VALUE': CliMatcher.IntegerMatcher( 0, 63,
         helpdesc='DSCP value in TOS byte in IP header of test packets (decimal)' ),
      'SOURCE_ADDRESS': IpGenAddrMatcher(
         helpdesc='Source address for control and test packets' ),
      'AUTH_MODE': CliMatcher.EnumMatcher( authModes ),
      'USER': CliMatcher.PatternMatcher( '.+', helpname='WORD',
         helpdesc='User name used for deriving authentication session keys' ),
      'INTERFACE': IntfCli.Intf.matcher }

class TwPingCmd( CliCommand.CliCommandClass ):
   syntax = '%s [ vrf VRF ] address ADDRESS port PORT [ OPTIONS ]' % clientKeyword
   data = {
      clientKeyword: matcherClient,
      'address': 'Hostname or IP address of destination TWAMP server',
      'ADDRESS': HostnameCli.IpAddrOrHostnameMatcher( ipv6=True ),
      'port': 'TCP port on which server listens',
      'PORT': CliMatcher.IntegerMatcher( minPort, maxPort,
         helpdesc='TCP port on which server listens' ),
      'vrf': 'VRF in which TWAMP tests should be be run',
      'VRF': VrfNameExprFactory( inclAllVrf=True ),
      'OPTIONS': TwPingOptionsExpr
   }
   handler = handleClient

BasicCliModes.EnableMode.addCommandClass( TwPingCmd )

# ------------------------------------------
# Command :
#     [no] twserver
#             [vrf <vrf name>]
#             [listen-port <port>]
#             [local-address <hostname or IP address> ]
#             [min-test-port <port>]
#             [max-test-port <port>]
#             [max-control-sessions <count>]
#             [auth-mode ( open | authenticated | encrypted | mixed )]
#             [user <user>]
#             [group <group>]
# Mode:
#     Config
# ------------------------------------------

authModesMap = { None: defaultAuthMode,
                 'authenticated': 'A',
                 'encrypted': 'E',
                 'open': 'O',
                 'mixed': defaultAuthMode }

# Server has apparently been configured.  Add server to
# Sysdb, after appropriate verifications, using defaults
# where necessary.  The corresponding reactor in the
# Twamp agent  will ultimately stop/start the server
# process appropriately.
def handleServer( mode, args ):

   vrfName = args.get( 'VRF', DEFAULT_VRF )

   # Extract any options set and if none found, use defaults appropriately.
   localIpOrHost = args.get( 'ADDRESS', defaultLocalAddress )
   if localIpOrHost != defaultLocalAddress:
      localIpOrHost = localIpOrHost[ 0 ]

      # Sanity check input hostname or IP address
      if ( HostnameCli.resolveHostname( mode, localIpOrHost, doWarn=False ) ==
           HostnameCli.Resolution.UNRESOLVED ):
         mode.addError( "%s is invalid local address/hostname" % localIpOrHost )
         return

   # pylint: disable-msg=W0621
   listenPort = args.get( 'PORT', defaultListenPort )
   if listenPort != defaultListenPort:
      listenPort = listenPort[ 0 ]

   minTestPort = args.get( 'MIN_TEST_PORT', minPort )
   if minTestPort != minPort:
      minTestPort = minTestPort[ 0 ]

   maxTestPort = args.get( 'MAX_TEST_PORT', maxPort )
   if maxTestPort != maxPort:
      maxTestPort = maxTestPort[ 0 ]

   if minTestPort == maxTestPort:
      mode.addError( "Invalid test ports specified ( i.e. min (%d) equal"
                     " to max (%d))" % ( minTestPort, maxTestPort ) )
      return

   if minTestPort > maxTestPort:
      mode.addError( "Invalid test ports specified ( i.e. min (%d) > "
                     " max (%d))" % ( minTestPort, maxTestPort ) )
      return

   if minTestPort == minPort and maxTestPort == maxPort:
      # Neither min nor max set; set both to defaults
      minTestPort = defaultMinTestPort
      maxTestPort = defaultMaxTestPort

   maxControlSessions = args.get( 'MAX_CONTROL_SESSIONS',
                                  defaultMaxControlSessions )
   if maxControlSessions != defaultMaxControlSessions:
      maxControlSessions = maxControlSessions[ 0 ]

   # Perhaps verify user's existence using 'getent passwd user'?
   user = args.get( 'USER', defaultUser )
   if user != defaultUser:
      user = user[ 0 ]

   group = args.get( 'GROUP', defaultGroup )
   if group != defaultGroup:
      group = group[ 0 ]

   authMode = args.get( 'AUTH_MODE' )
   if authMode:
      authMode = authMode[ 0 ]
   authMode = authModesMap[ authMode ]

   # Since new server being created if any of the parameters
   # change, delete old server if it exists and then create new
   # server.  Must explicitly delete old server here rather than
   # overwriting so can actually kill old server and carry out
   # any other necessary tidy ups
   del twampConfig.server[ vrfName ]

   twampConfig.newServer( vrfName, localIpOrHost, listenPort,
                          minTestPort, maxTestPort, maxControlSessions,
                          user, group, authMode )

# Server has apparently been unconfigured.  Delete server in
# Sysdb, after appropriate verifications.
def handleNoServer( mode, args ):

   vrfName = args.get( 'VRF', DEFAULT_VRF )

   if vrfName not in twampConfig.server:
      mode.addWarning( "TWAMP server not previously configured in VRF %s" % vrfName )
      return

   # Now delete server config and status in Sysdb
   del twampConfig.server[ vrfName ]

# ------------------------------------------
# Command :
#     [no] twserver
#             [vrf <vrf name>]
#             [listen-port <port>]
#             [local-address <hostname or IP address> ]
#             [min-test-port <port>]
#             [max-test-port <port>]
#             [max-control-sessions <count>]
#             [auth-mode ( open | authenticated | encrypted | mixed )]
#             [user <user>]
#             [group <group>]
# Mode:
#     Config
# ------------------------------------------

matcherConfServer = CliMatcher.KeywordMatcher( serverKeyword,
  helpdesc='Configure TWAMP server' )

class TwserverOptionsExpr( CliCommand.CliExpression ):
   expression = ( '{ ( listen-port PORT ) | '
                    '( local-address ADDRESS ) | '
                    '( min-test-port MIN_TEST_PORT ) | '
                    '( max-test-port MAX_TEST_PORT ) | '
                    '( max-control-sessions MAX_CONTROL_SESSIONS ) | '
                    '( auth-mode AUTH_MODE ) | '
                    '( group GROUP ) | '
                    '( user USER ) } ' )

   data = {
      'listen-port': CliCommand.singleKeyword( 'listen-port',
         helpdesc='TCP port on which server listens' ),
      'local-address': CliCommand.singleKeyword( 'local-address',
         helpdesc='Local address(es) on which server may be reached' ),
      'min-test-port': CliCommand.singleKeyword( 'min-test-port',
         helpdesc='Minimum UDP port to be used for tests' ),
      'max-test-port': CliCommand.singleKeyword( 'max-test-port',
         helpdesc='Maximum UDP port to be used for tests' ),
      'max-control-sessions': CliCommand.singleKeyword( 'max-control-sessions',
         helpdesc='Maximum number of control sessions (0 means unbounded)' ),
      'auth-mode': CliCommand.singleKeyword( 'auth-mode',
         helpdesc='Authentication mode' ),
      'user': CliCommand.singleKeyword( 'user',
         helpdesc='User ID for the server process' ),
      'group': CliCommand.singleKeyword( 'group',
         helpdesc='Group ID for the server process' ),
      'PORT': CliMatcher.IntegerMatcher( minPort,
         maxPort, helpdesc='TCP port on which server listens' ),
      'ADDRESS': HostnameCli.IpAddrOrHostnameMatcher( ipv6=True,
         helpdesc='Local address(es) on which server may be reached' ),
      'MIN_TEST_PORT': CliMatcher.IntegerMatcher( minPort,
         maxPort, helpdesc='Minimum UDP port to be used for tests' ),
      'MAX_TEST_PORT': CliMatcher.IntegerMatcher( minPort,
         maxPort, helpdesc='Maximum UDP port to be used for tests' ),
      'MAX_CONTROL_SESSIONS': CliMatcher.IntegerMatcher( defaultMaxControlSessions,
        maxControlSessions, helpdesc='Maximum number of control sessions' ),
      'AUTH_MODE': CliMatcher.EnumMatcher( authModes ),
      'USER': CliMatcher.PatternMatcher( '.+', helpname='WORD',
         helpdesc='POPULATE DESCRIPTION' ),
      'GROUP': CliMatcher.PatternMatcher( '.+', helpname='WORD',
         helpdesc='IS THIS NEEDED?' ),
   }

class TwServerCmd( CliCommand.CliCommandClass ):
   syntax = '%s [ vrf VRF ] [ OPTIONS ]' % serverKeyword
   noOrDefaultSyntax = '%s [ vrf VRF ] ...' % serverKeyword
   data = {
      serverKeyword: matcherConfServer,
      'vrf': 'Start/stop server in a specific VRF',
      'VRF': VrfNameExprFactory( inclAllVrf=True ),
      'OPTIONS': TwserverOptionsExpr
   }
   handler = handleServer
   noOrDefaultHandler = handleNoServer

BasicCliModes.GlobalConfigMode.addCommandClass( TwServerCmd )

def Plugin( entityManager ):
   global twampConfig
   twampConfig = ConfigMount.mount( entityManager, "perfsonartwamp/config",
                                    "PerfsonarTwamp::Config", "w" )
   global twampStatus
   twampStatus = LazyMount.mount( entityManager, "perfsonartwamp/status",
                                  "PerfsonarTwamp::Status", "w" )
   global twampTest
   twampTest = LazyMount.mount( entityManager, "perfsonartwamp/test",
                                  "PerfsonarTwamp::Test", "w" )

   global intfStatusAll
   intfStatusAll = LazyMount.mount( entityManager, "interface/status/all",
                                    "Interface::AllIntfStatusDir", "r" )
