#!/usr/bin/env python3
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-from-import
# pylint: disable=consider-using-f-string
# pylint: disable=unnecessary-comprehension
# pylint: disable=bad-option-value

import BasicCli
import CliCommand
import CliMatcher
import CliParser
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.PhyCli as PhyCli
import CliPlugin.PhyConfigCli as PhyConfigCli
import CommonGuards
import Intf
import Tac
import XcvrLib

coherentDspNode = CliCommand.guardedKeyword( 'coherent-dsp',
      helpdesc='Set coherent DSP parameters',
      guard=CommonGuards.standbyGuard )

#--------------------------------------------------------------------------------
# [ no | default ] phy coherent-dsp transmitter roll-off ROLL_OFF
#--------------------------------------------------------------------------------
class RollOffCmd( CliCommand.CliCommandClass ):
   syntax = 'phy coherent-dsp transmitter roll-off ROLL_OFF'
   noOrDefaultSyntax = 'phy coherent-dsp transmitter roll-off ...'
   data = {
      'phy' : PhyCli.phyNode,
      'coherent-dsp' : coherentDspNode,
      'transmitter' : 'Set transmitter parameters',
      'roll-off' : 'Set transmitter parameters',
      'ROLL_OFF' : CliMatcher.FloatMatcher( 0, 1,
         helpdesc='The roll-off factor',
         precisionString='%%.%dg' % PhyConfigCli.rollOffPrecision ),
   }

   handler = "PhyCoherentCliHandler.rollOffCmdHandler"
   noOrDefaultHandler = "PhyCoherentCliHandler.rollOffCmdNoHandler"

PhyCli.PhyConfigModelet.addCommandClass( RollOffCmd )

#--------------------------------------------------------------------------------
# [ no | default ] phy coherent-dsp receiver state-of-polarization tracking fast
#--------------------------------------------------------------------------------
def coherentSopGuard( mode, token ):
   xcvrName = XcvrLib.getXcvrSlotName( mode.intf.name )
   xcvrConfig = PhyCli.xcvrConfigDir.xcvrConfig.get( xcvrName, None )
   if hasattr( xcvrConfig, 'cfp2SlotCapabilities' ) and \
      xcvrConfig.cfp2SlotCapabilities.cfp2DcoSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

class SopFastTrackingCmd( CliCommand.CliCommandClass ):
   syntax = 'phy coherent-dsp receiver state-of-polarization tracking fast'
   noOrDefaultSyntax = syntax
   data = {
      'phy' : PhyCli.phyNode,
      'coherent-dsp' : coherentDspNode,
      'receiver' : 'Set receiver parameters',
      'state-of-polarization' : CliCommand.guardedKeyword( 'state-of-polarization',
         helpdesc='Set state-of-polarization', guard=coherentSopGuard ),
      'tracking' : 'Set tracking mode',
      'fast' : 'Set fast tracking mode',
   }

   handler = "PhyCoherentCliHandler.sopFastTrackingCmdHandler"
   noOrDefaultHandler = "PhyCoherentCliHandler.sopFastTrackingCmdNoHandler"

PhyCli.PhyConfigModelet.addCommandClass( SopFastTrackingCmd )

#--------------------------------------------------------------------------------
# [ no | default ] phy coherent-dsp modulation MODE [rate RATE]
#--------------------------------------------------------------------------------
class ModulationCmd( CliCommand.CliCommandClass ):
   syntax = 'phy coherent-dsp modulation MODE [rate RATE]'
   noOrDefaultSyntax = 'phy coherent-dsp modulation ...'
   data = {
      'phy' : PhyCli.phyNode,
      'coherent-dsp' : coherentDspNode,
      'modulation' : 'Set modulation method',
      'MODE' : CliMatcher.EnumMatcher( {
         '8qam' : 'Set modulation method to 8QAM',
         '16qam' : 'Set modulation method to 16QAM',
         'dp-qpsk' : 'Set modulation method to DP-QPSK'
         } ),
      'rate' : 'Set optical data rate',
      'RATE' : CliMatcher.EnumMatcher( {
         '100g' : 'Set line rate to 100G',
         '200g' : 'Set line rate to 200G',
         '300g' : 'Set line rate to 300G',
         '400g' : 'Set line rate to 400G'
         } )
   }

   handler = "PhyCoherentCliHandler.modulationCmdHandler"
   noOrDefaultHandler = "PhyCoherentCliHandler.modulationCmdNoHandler"

PhyCli.PhyConfigModelet.addCommandClass( ModulationCmd )

class SkewCmd( CliCommand.CliCommandClass ):
   syntax = '''phy coherent-dsp
               ( ( transmitter skew TX_INTF ( ( compute peer-interface PEER_INTF )
                                            | activate
                                            | deactivate
                                            | clear ) )
               | ( receiver skew RX_INTFS ( compute
                                          | activate
                                          | deactivate
                                          | clear ) ) )'''
   data = {
      'phy': PhyCli.phyNode,
      'coherent-dsp': coherentDspNode,
      'skew': 'Modify skew configuration',
      'transmitter': CliCommand.guardedKeyword( 'transmitter',
                        helpdesc='Modify transmitter skew configuration',
                        guard=PhyCli.coherentStatusGuard ),
      'receiver': CliCommand.guardedKeyword( 'receiver',
                     helpdesc='Modify receiver skew configuration',
                     guard=PhyCli.coherentStatusGuard ),
      'compute': 'Compute and store new skew values',
      'activate': 'Activate stored skew values',
      'deactivate': 'Deactivate stored skew values and revert to defaults',
      'clear': 'Delete stored skew values and revert to defaults',
      'peer-interface': 'Peer interfaces to use for computing new skew values',
      'TX_INTF': EthIntfCli.EthPhyIntf.ethMatcher,
      'RX_INTFS': Intf.IntfRange.IntfRangeMatcher(
                    explicitIntfTypes=( EthIntfCli.EthPhyAutoIntfType, ) ),
      'PEER_INTF': EthIntfCli.EthPhyIntf.ethMatcher,
   }

   handler = "PhyCoherentCliHandler.skewCmdHandler"

BasicCli.GlobalConfigMode.addCommandClass( SkewCmd )
