# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-enumerate

import Tac, re

from CliPlugin.EventsModel import CountAndChangedStat
from CliPlugin.PhyFecModel import FireCodeModel
from CliPlugin.PhyFecModel import InterfaceFecInfo
from CliPlugin.PhyFecModel import RsFec
from CliPlugin.PhyFecModel import RsFecAlignmentLock
from CliPlugin.PhyFecModel import RsFecCorrectedCodewords
from CliPlugin.PhyFecModel import RsFecCorrectedSymbols
from CliPlugin.PhyFecModel import RsFecUnCorrectedCodewords
from CliPlugin.PhyFecModel import RsGearboxModel
from Ark import switchTimeToUtc

def parseRsFecLaneMapping( laneMapping ):
   # input from lane mapping register (register 1.206)
   mapping = {}
   for i in range( 4 ):
      mapping[ i ] = laneMapping >> 4 * i & 0x0f
   return mapping

def populateConditionModel( model, original, checkpoint ):
   model.value = bool( original.current )
   model.changes = original.changes - checkpoint.changes
   if original.lastChange == checkpoint.lastChange:
      model.lastChange = 0.0
   else:
      model.lastChange = switchTimeToUtc( original.lastChange )
   return model

def populateCountModel( model, original, checkpoint ):
   model.changes = original.changes - checkpoint.changes
   if original.lastChange == checkpoint.lastChange:
      model.value = 0
      model.lastChange = 0.0
   else:
      model.value = original.count
      model.lastChange = switchTimeToUtc( original.lastChange )
   return model

def populateRsFecModel( rsOrig, rsCkpt, fecEncoding ):
   model = RsFec()
   EthFecEncoding = Tac.Type( 'Interface::EthFecEncoding' )
   if fecEncoding == EthFecEncoding.fecEncodingReedSolomon544:
      model.codewordSize = '544'
   else:
      assert fecEncoding == EthFecEncoding.fecEncodingReedSolomon
      model.codewordSize = '528'

   if rsCkpt is not None:
      model.alignmentLock = \
            populateConditionModel( RsFecAlignmentLock(),
                                    rsOrig.fecAlignStatus,
                                    rsCkpt.fecAlignStatus )
      model.correctedCodewords = \
            populateCountModel( RsFecCorrectedCodewords(),
                                rsOrig.fecCorrectedCodewords,
                                rsCkpt.fecCorrectedCodewords )
      model.uncorrectedCodewords = \
            populateCountModel( RsFecUnCorrectedCodewords(),
                                rsOrig.fecUncorrectedCodewords,
                                rsCkpt.fecUncorrectedCodewords )
      if rsCkpt.fecSer:
         # pylint: disable-msg=protected-access
         if rsCkpt.fecSer.lastUpdateTime == rsOrig.fecSer.lastUpdateTime:
            model._totalSymbols = rsCkpt.fecSer.totalSymbols
            model.correctedSymbolRate = 0.0
         else:
            model._totalSymbols =  rsOrig.fecSer.totalSymbols
            model.correctedSymbolRate = rsOrig.fecSer.ser
      for laneId in range( 0, len( rsOrig.fecSymbolErrors ) ):
         # The key checks here are in case there is a problem with how we add
         # the counters to Sysdb (such as a crashing agent) which prevents some
         # of the counters from being created.
         #
         # We should probably issue a warning for this as well.
         if rsCkpt.fecSymbolErrors.get( laneId ) is not None:
            model.correctedSymbols[ laneId ] = \
                  populateCountModel( RsFecCorrectedSymbols(),
                                      rsOrig.fecSymbolErrors[ laneId ],
                                      rsCkpt.fecSymbolErrors[ laneId ] )
   else:
      model.alignmentLock = RsFecAlignmentLock().toModel(
            rsOrig.fecAlignStatus )
      model.correctedCodewords = RsFecCorrectedCodewords().toModel(
            rsOrig.fecCorrectedCodewords )
      model.uncorrectedCodewords = RsFecUnCorrectedCodewords().toModel(
            rsOrig.fecUncorrectedCodewords )
      if rsOrig.fecSer:
         model.correctedSymbolRate = rsOrig.fecSer.ser
         # pylint: disable-msg=protected-access
         model._totalSymbols = rsOrig.fecSer.totalSymbols
      for laneId in range( 0, len( rsOrig.fecSymbolErrors ) ):
         model.correctedSymbols[ laneId ] = RsFecCorrectedSymbols().toModel(
               rsOrig.fecSymbolErrors[ laneId ] )

   model.laneMap = parseRsFecLaneMapping( rsOrig.fecLaneMapping )

   return model

def populateRsGearboxModel( rsGearboxOrig, rsGearboxCkpt ):
   model = RsGearboxModel()

   if rsGearboxCkpt is not None:
      pcsAlignStatusModel = CountAndChangedStat()
      pcsAlignStatusModel = populateCountModel( pcsAlignStatusModel,
                                                rsGearboxOrig.pcsAlignStatus,
                                                rsGearboxCkpt.pcsAlignStatus )
      ( model.pcsAlignStatus,
        model.pcsAlignStatusChanges,
        model.pcsAlignStatusLastChange ) = ( pcsAlignStatusModel.value,
                                             pcsAlignStatusModel.changes,
                                             pcsAlignStatusModel.lastChange )

      pcsBlockLockStatusModel = CountAndChangedStat()
      pcsBlockLockStatusModel = populateCountModel(
                                             pcsBlockLockStatusModel,
                                             rsGearboxOrig.pcsBlockLockStatus,
                                             rsGearboxCkpt.pcsBlockLockStatus )
      ( model.pcsBlockLockStatus,
        model.pcsBlockLockChanges,
        model.pcsBlockLockLastChange ) = ( pcsBlockLockStatusModel.value,
                                           pcsBlockLockStatusModel.changes,
                                           pcsBlockLockStatusModel.lastChange )

      for laneId in range( 0, len( rsGearboxOrig.pcsLaneMapping ) ):
         bipHostCount = CountAndChangedStat()
         bipModuleCount = CountAndChangedStat()

         # The key checks here are in case there is a problem with how we add
         # the counters to Sysdb (such as a crashing agent) which prevents some
         # of the counters from being created.
         #
         # We should probably issue a warning for this as well.
         if ( laneId in rsGearboxOrig.pcsHostBipErrors and
              laneId in rsGearboxCkpt.pcsHostBipErrors ):
            bipHostCount = \
                  populateCountModel( bipHostCount,
                                      rsGearboxOrig.pcsHostBipErrors[ laneId ],
                                      rsGearboxCkpt.pcsHostBipErrors[ laneId ] )
         model.pcsHostBipErrors[ laneId ] = bipHostCount

         if ( laneId in rsGearboxOrig.pcsModuleBipErrors and
              laneId in rsGearboxCkpt.pcsModuleBipErrors ):
            bipModuleCount = \
                  populateCountModel( bipModuleCount,
                                      rsGearboxOrig.pcsModuleBipErrors[ laneId ],
                                      rsGearboxCkpt.pcsModuleBipErrors[ laneId ] )
         model.pcsModuleBipErrors[ laneId ] = bipModuleCount

         model.pcsLaneMapping[ laneId ] = rsGearboxOrig.pcsLaneMapping[ laneId ]
   else:
      model.pcsAlignStatus = rsGearboxOrig.pcsAlignStatus.count
      model.pcsAlignStatusChanges = rsGearboxOrig.pcsAlignStatus.changes
      model.pcsAlignStatusLastChange = switchTimeToUtc(
            rsGearboxOrig.pcsAlignStatus.lastChange )

      model.pcsBlockLockStatus = rsGearboxOrig.pcsBlockLockStatus.count
      model.pcsBlockLockChanges = rsGearboxOrig.pcsBlockLockStatus.changes
      model.pcsBlockLockLastChange = switchTimeToUtc(
            rsGearboxOrig.pcsBlockLockStatus.lastChange )

      for laneId in range( 0, len( rsGearboxOrig.pcsLaneMapping ) ):
         bipHostCount = CountAndChangedStat()
         bipModuleCount = CountAndChangedStat()

         # The key checks here are in case there is a problem with how we add
         # the counters to Sysdb (such as a crashing agent) which prevents some
         # of the counters from being created.
         #
         # We should probably issue a warning for this as well.
         if laneId in rsGearboxOrig.pcsHostBipErrors:
            bipHostCount.value = rsGearboxOrig.pcsHostBipErrors[ laneId ].count
            bipHostCount.changes = rsGearboxOrig.pcsHostBipErrors[ laneId ].changes
            bipHostCount.lastChange = switchTimeToUtc(
                  rsGearboxOrig.pcsHostBipErrors[ laneId ].lastChange )
         model.pcsHostBipErrors[ laneId ] = bipHostCount

         if laneId in rsGearboxOrig.pcsModuleBipErrors:
            bipModuleCount.value = rsGearboxOrig.pcsModuleBipErrors[ laneId ].count
            bipModuleCount.changes = \
                  rsGearboxOrig.pcsModuleBipErrors[ laneId ].changes
            bipModuleCount.lastChange = switchTimeToUtc(
                        rsGearboxOrig.pcsModuleBipErrors[ laneId ].lastChange )
         model.pcsModuleBipErrors[ laneId ] = bipModuleCount

         model.pcsLaneMapping[ laneId ] = rsGearboxOrig.pcsLaneMapping[ laneId ]
   return model

def populateFireCodeModel( fcOrig, fcCkpt ):
   model = FireCodeModel()

   if fcCkpt is not None:
      for laneId in range( 0, len( fcOrig.laneFecCorrectedBlocks ) ):
         corrected = CountAndChangedStat()

         # Again, the checks here are to protect against cases where the agent
         # fails to provide a key value. We should probably also give a
         # warning in this case
         if ( laneId in fcOrig.laneFecCorrectedBlocks and
              laneId in fcCkpt.laneFecCorrectedBlocks ):
            corrected = populateCountModel( corrected,
                                            fcOrig.laneFecCorrectedBlocks[ laneId ],
                                            fcCkpt.laneFecCorrectedBlocks[ laneId ] )
         model.perLaneCorrectedFecBlocks[ laneId ] = corrected

         uncorrected = CountAndChangedStat()
         if ( laneId in fcOrig.laneFecUncorrectedBlocks and
              laneId in fcCkpt.laneFecUncorrectedBlocks ):
            uncorrected = \
                  populateCountModel( uncorrected,
                                      fcOrig.laneFecUncorrectedBlocks[ laneId ],
                                      fcCkpt.laneFecUncorrectedBlocks[ laneId ] )
         model.perLaneUncorrectedFecBlocks[ laneId ] = uncorrected
   else:
      for laneId in range( 0, len( fcOrig.laneFecCorrectedBlocks ) ):
         corrected = CountAndChangedStat()
         # Again, the checks here are to protect against cases where the agent
         # fails to provide a key value. We should probably also give a
         # warning in this case
         if laneId in fcOrig.laneFecCorrectedBlocks:
            corrected.value = fcOrig.laneFecCorrectedBlocks[ laneId ].count
            corrected.changes = fcOrig.laneFecCorrectedBlocks[ laneId ].changes
            corrected.lastChange = switchTimeToUtc(
                  fcOrig.laneFecCorrectedBlocks[ laneId ].lastChange )
         model.perLaneCorrectedFecBlocks[ laneId ] = corrected

         uncorrected = CountAndChangedStat()
         if laneId in fcOrig.laneFecUncorrectedBlocks:
            uncorrected.value = fcOrig.laneFecUncorrectedBlocks[ laneId ].count
            uncorrected.changes = fcOrig.laneFecUncorrectedBlocks[ laneId ].changes
            uncorrected.lastChange = switchTimeToUtc(
                  fcOrig.laneFecUncorrectedBlocks[ laneId ].lastChange )
         model.perLaneUncorrectedFecBlocks[ laneId ] = uncorrected
   return model

# This is where we actually populate the model
def showOneInterfacePhyFecDetail( fecEncoding, reedSolomon, rsGearbox, fireCode,
                                  reedSolomonCkpt, rsGearboxCkpt, fireCodeCkpt ):
   interfaceInfo = InterfaceFecInfo()
   interfaceInfo.fecType = 'None'
   EthFecEncoding = Tac.Type( 'Interface::EthFecEncoding' )

   # Unknown case for auto negotiation
   if fecEncoding == EthFecEncoding.fecEncodingUnknown:
      interfaceInfo.fecType = 'Unknown'
      return interfaceInfo

   # Actually populate the model
   if reedSolomon is not None:
      interfaceInfo.fecType = 'Reed-Solomon'
      interfaceInfo.reedSolomon = populateRsFecModel( reedSolomon,
                                                      reedSolomonCkpt,
                                                      fecEncoding )
   if rsGearbox is not None:
      interfaceInfo.rsGearbox = populateRsGearboxModel( rsGearbox,
                                                        rsGearboxCkpt )
   if fireCode is not None:
      interfaceInfo.fecType = 'Fire-Code'
      interfaceInfo.fireCode = populateFireCodeModel( fireCode,
                                                      fireCodeCkpt )

   return interfaceInfo

# Get the linecard number from the interface name
def getLinecard( intfName ):
   match = re.search( r'(\d+)/', intfName )
   assert match
   # This method should only be called if there is a linecard
   return match.group( 1 )

# In the future, we may choose to integrate this in more fully with
# showInterfacesPhy in PhyCli.py
fecStsDirGlobal = None
fecCfgDirGlobal = None
fecCntrCkptDirGlobal = None
def showInterfacesPhyFecDetail( em, intfs=None ):
   global fecStsDirGlobal, fecCfgDirGlobal, fecCntrCkptDirGlobal

   if fecStsDirGlobal is None:
      # Mount the directories we care about
      mg = em.mountGroup()
      fecStsDir = mg.mount( 'hardware/phy/status/errorCorrection/slice',
                            'Tac::Dir', 'ri' )
      fecCfgDir = mg.mount( 'hardware/phy/config/errorCorrection/slice',
                            'Tac::Dir', 'ri' )
      fecCntrCkptDir = mg.mount( 'hardware/phy/counter/checkpoint/fec',
                                 'Hardware::Phy::FecStatusDir', 'ri' )
      mg.close( blocking=True )
      fecStsDirGlobal = fecStsDir
      fecCfgDirGlobal = fecCfgDir
      fecCntrCkptDirGlobal = fecCntrCkptDir
   else:
      fecStsDir = fecStsDirGlobal
      fecCfgDir = fecCfgDirGlobal
      fecCntrCkptDir = fecCntrCkptDirGlobal

   intfNames = intfs
   fecStatuses = dict() # pylint: disable=use-dict-literal
   EthFecEncoding = Tac.Type( 'Interface::EthFecEncoding' )

   # Because this is not a Tac::Dir and its parent is, we have to
   # mount it explicitly in order to correctly identify its type,
   # but we don't know where it is until we have completed the mount

   agents = []

   for name, sliceDir in fecStsDir.items():
      cfg = fecCfgDir.get( name )
      if not cfg or cfg and cfg.genId != sliceDir.genId:
         continue
      agents.append( sliceDir )

   for intfName in intfNames:
      # Note that these directories may not be present
      rsInfo = None
      rsGearboxInfo = None
      fireCodeInfo = None
      rsCkptInfo = None
      rsGearboxCkptInfo = None
      fireCodeCkptInfo = None
      originalGenerationId = None
      checkpointGenerationId = None

      intfId = Tac.Value( "Arnet::IntfId", intfName )

      # We don't know which agent has the interface, so we have to
      # iterate through all of them. Luckily there shouldn't be too
      # many (at least for now).

      for agent in agents:
         fecEncoding = agent.ethFecEncoding.get( intfId )
         if fecEncoding:
            originalGenerationId = agent.generationId.get( intfId )
            checkpointGenerationId = fecCntrCkptDir.generationId.get( intfId )
            if fecEncoding in [ EthFecEncoding.fecEncodingReedSolomon544,
                                EthFecEncoding.fecEncodingReedSolomon ]:
               rsInfo = agent.rsFecStatus.get( intfId )
               rsGearboxInfo = agent.rsGearbox.get( intfId )
               rsCkptInfo = fecCntrCkptDir.rsFecStatus.get( intfId )
               rsGearboxCkptInfo = fecCntrCkptDir.rsGearbox.get( intfId )
               if rsGearboxInfo:
                  assert fecEncoding == EthFecEncoding.fecEncodingReedSolomon
            elif fecEncoding == EthFecEncoding.fecEncodingFireCode:
               fireCodeInfo = agent.fireCodeFecStatus.get( intfId )
               fireCodeCkptInfo = fecCntrCkptDir.fireCodeFecStatus.get( intfId )

            if ( originalGenerationId is not None and
                 checkpointGenerationId is not None and
                 originalGenerationId.id == checkpointGenerationId.id ):
               model = showOneInterfacePhyFecDetail( fecEncoding, rsInfo,
                                                     rsGearboxInfo, fireCodeInfo,
                                                     rsCkptInfo, rsGearboxCkptInfo,
                                                     fireCodeCkptInfo )
            else:
               model = showOneInterfacePhyFecDetail( fecEncoding, rsInfo,
                                                     rsGearboxInfo, fireCodeInfo,
                                                     None, None, None )

            fecStatuses[ intfName ] = model
            break

   # Return models
   return fecStatuses
