# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from IntfModels import Interface
from CliPlugin.PhyModulationModel import InterfacePhyCoherentBase
from CliPlugin.PhyModulationModel import InterfacesPhyCoherentBase
from CliModel import Enum
from CliModel import Bool
from CliModel import Dict
from CliModel import Model
from CliModel import Submodel
from Arnet import sortIntf

class InterfaceAvailableLineRates( Model ):
   lr100g = Bool( help="100g line rate", optional=True )
   lr200g = Bool( help="200g line rate", optional=True )
   lr300g = Bool( help="300g line rate", optional=True )
   lr400g = Bool( help="400g line rate", optional=True )

   def availableRates( self ) :
      for attrName in self.__attributes__:
         if getattr( self, attrName ):
            return True
      return False

class InterfacePhyLineRate( InterfacePhyCoherentBase ):
   # The user applied configuration.
   _noConfiguration = Bool( help="True if user hasn't applied any configuration" )
   lineRateConfigured = Enum(
         values=( 'none', '100', '200', '300', '400' ),
         help="Configured line rate. 'none' implies that no configuration has " + \
               "been applied on this interface." )
   # The line rate mode in operation
   lineRateStatus = Enum(
         values=( 'none', '100', '200', '300', '400' ),
         help="Line rate in operation. 'none' implies that this interface " + \
               "is inactive" )
   availableLineRates = \
         Submodel( valueType=InterfaceAvailableLineRates,
                   help="Line rate choices available for " + \
                        "configuration on this interface" )

   def lineRateConfiguredString( self ):
      """Returns a string representing the modulation configuration"""
      if self._noConfiguration:
         return 'default'
      valueToStr = {
         'none' : 'none',
         '100' : '100G',
         '200' : '200G',
         '300' : '300G',
         '400' : '400G' }
      return valueToStr.get( self.lineRateConfigured, 'default' )

   def lineRateStatusString( self ):
      """Returns a string representing the modulation status"""
      valueToStr = {
         'none' : 'none',
         '100' : '100G',
         '200' : '200G',
         '300' : '300G',
         '400' : '400G' }
      return valueToStr.get( self.lineRateStatus, 'none' )

   def lineRateAvailableString( self ):
      """Returns a string representing the available modulations"""
      available = []
      if self.availableLineRates.lr100g:
         available.append( '100G' )
      if self.availableLineRates.lr200g:
         available.append( '200G' )
      if self.availableLineRates.lr300g:
         available.append( '300G' )
      if self.availableLineRates.lr400g:
         available.append( '400G' )

     # Add ', '
      if available:
         return ', '.join( available )
      return 'none'

class InterfacesPhyLineRate( InterfacesPhyCoherentBase ):
   interfaceLineRate = Dict(
         keyType=Interface,
         valueType=InterfacePhyLineRate,
         help="Mapping between an interface and phy line rate information" )

   def render( self ):
      if not self.interfaceLineRate:
         return

      # Make sure that there is at least one interface with available line rates
      # to print the display
      display = False
      for intf in self.interfaceLineRate:
         status = self.interfaceLineRate[ intf ]
         if status.availableLineRates.availableRates():
            display = True
      if not display:
         return

      hdrFmt = '%-18s  %-16s %-16s %-24s %-15s'
      print( hdrFmt % ( 'Interface', 'Modulation', 'Configured',
                       'Available', 'Operational' ) )
      print( hdrFmt % ( '-' * 16, '-' * 13, '-' * 12, '-' * 20, '-' * 11 ) )

      for intf in sortIntf( self.interfaceLineRate ):
         status = self.interfaceLineRate[ intf ]
         print( hdrFmt % ( intf,
                          status.modulationStatusString(),
                          status.lineRateConfiguredString(),
                          status.lineRateAvailableString(),
                          status.lineRateStatusString() ) )

