# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

# pylint: disable-msg=W0212

import CliModel
from CliPlugin import EventsModel
from CliPlugin import PhyStatusModel
from CliPlugin.PhyModel import ethModulationToStr
from CliPlugin.PhySerdesModelCommon import ( linktrnStsToStr, rxPamModeToStr,
                                             brPdEnToStr, txPiEnToStr,
                                             SerdesLaneStatus )
from CliPlugin.PhyStatusModel import phyDetailStatFmt
import Ark

class BlackhawkPhySerdesLaneModelBase( SerdesLaneStatus ):
   eyeLeft = CliModel.Int( optional=True,
                           help="Horizontal left eye margin @ 1e-5 as seen"
                           " by internal diagnostic slicer in mUI" )
   eyeRight = CliModel.Int( optional=True,
                            help="Horizontal right eye margin @ 1e-5 as seen"
                            " by internal diagnostic slicer in mUI" )
   eyeUpper = CliModel.Int( optional=True,
                            help="Vertical upper eye margin @ 1e-5 as seen"
                            " by internal diagnostic slicer in mV" )
   eyeLower = CliModel.Int( optional=True,
                            help="Vertical lower eye margin @ 1e-5 as seen"
                            " by internal diagnostic slicer in mV" )
   brPdEnable = CliModel.Enum( values=list( brPdEnToStr.values() ),
                               help="Baud rate phase detector enable" )
   brPdForced = CliModel.Bool( optional=True, help="Whether the BR CDR is forced" )
   osPdForced = CliModel.Bool( optional=True, help="Whether the OS CDR is forced" )
   mainPeakingFilter = CliModel.Int( help="Peaking filter main settings" )
   lowFrequncyPeakingFilter = CliModel.Int( help="Low frequency peaking"
                                            " filter control" )
   highFrequencyPeakingFilter = CliModel.Int( help="High frequency peaking"
                                              " filter control" )
   m1SlicerThreshold = CliModel.Int( help="Vertical threshold voltage of"
                                     " m1 slicer (mV), 'channel loss hint'"
                                     " in PAM4 mode should always be 0" )
   p1SlicerThreshold = CliModel.Int( help="Vertical threshold voltage of"
                                     " p1 slicer (mV)" )
   _channelLossEstimate = CliModel.Int( help="Initial channel loss estimate" )
   _afeBw = CliModel.Int( help="Arista internal use only" )
   _usrStatusEqDebug1 = CliModel.Int( help="Arista internal use only" )
   rxPamMode = CliModel.Enum( values=list( rxPamModeToStr.values() ),
                              help=" Receiver PAM4 mode" )

   def toModel( self, serdes ):
      super().toModel( serdes )
      if serdes.rxPamMode == 0 or serdes.pam4EyeSupported:
         self.eyeLeft = serdes.heyeLeft
         self.eyeRight = serdes.heyeRight
         self.eyeUpper = serdes.veyeUpper
         self.eyeLower = serdes.veyeLower
      self.brPdEnable = brPdEnToStr[ serdes.brPdEn ]
      self.brPdForced = serdes.brPdForced
      self.osPdForced = serdes.osPdForced
      self.mainPeakingFilter = serdes.pfMain
      self.lowFrequncyPeakingFilter = serdes.pf2Ctrl
      self.highFrequencyPeakingFilter = serdes.pf3Ctrl
      self.m1SlicerThreshold = serdes.m1Lvl
      self.p1SlicerThreshold = serdes.p1Lvl
      self._channelLossEstimate = serdes.blindCtleCtrl
      self._afeBw = serdes.afeBw
      self._usrStatusEqDebug1 = serdes.usrStatusEqDebug1
      self.rxPamMode = rxPamModeToStr[ serdes.rxPamMode ]
      return self

class BlackhawkPhySerdesLaneModel( BlackhawkPhySerdesLaneModelBase ):
   txPrecoderEnable = CliModel.Bool( help="Tx precoder enable" )
   rxDecoderEnable = CliModel.Bool( help="Rx decode enable" )
   linkTrainingEnable = CliModel.Bool( help="Tx link training enable" )
   linkTrainingStatus = CliModel.Enum( values=list( linktrnStsToStr.values() ),
                                       help="Link training status" )
   def toModel( self, serdes ):
      super().toModel( serdes )
      self.txPrecoderEnable = serdes.txPrecEn
      self.rxDecoderEnable = serdes.rxDecEn
      self.linkTrainingEnable = serdes.linktrnEn
      self.linkTrainingStatus = linktrnStsToStr[ serdes.linktrnSts ]
      return self

class BlackhawkLtOptionalPhySerdesLaneModel( BlackhawkPhySerdesLaneModelBase ):
   txPrecoderEnable = CliModel.Bool( optional=True,
                                     help="Tx precoder enable" )
   rxDecoderEnable = CliModel.Bool( optional=True,
                                    help="Rx decode enable" )
   linkTrainingEnable = CliModel.Bool( optional=True,
                                       help="Tx link training enable" )
   linkTrainingStatus = CliModel.Enum( optional=True,
                                       values=list( linktrnStsToStr.values() ),
                                       help="Link training status" )
   def toModel( self, serdes ):
      super().toModel( serdes )
      if serdes.linktrnValid:
         self.linkTrainingEnable = serdes.linktrnEn
         self.linkTrainingStatus = linktrnStsToStr[ serdes.linktrnSts ]
      return self

class BlackhawkPhySerdesModelBase( CliModel.Model ):
   serdesStats = CliModel.Dict( keyType=int,
                                valueType=BlackhawkPhySerdesLaneModelBase,
                                help="The serdes statistics" )
   sortedSerdesStats = CliModel.List( valueType=int,
                                      help="Sorted Serdes statistics" )
   def toModel( self, serdes ):
      for lane, stats in serdes.items():
         serdesLaneModel = self.getSerdesLaneModel()
         laneStats = serdesLaneModel().toModel( stats )
         # XXX kewei: Remove serdesId once BUG336826 is resolved.
         self.serdesStats[ getattr( lane, 'laneId', lane ) ] = laneStats
      return self

   def printLinkTrainingInfo( self ):
      headerPrinted = False
      for lane in self.sortedSerdesStats:
         info = self.serdesStats[ lane ]
         if info.linkTrainingEnable is None:
            continue
         if not headerPrinted:
            print( PhyStatusModel.phyDetailLongHeaderFmt % ( "Link Training" ) )
            headerPrinted = True
         if info.linkTrainingEnable:
            print( PhyStatusModel.phyDetailInfoFmt % ( '  Lane %d' % lane,
                  info.linkTrainingStatus ) )
         else:
            print( PhyStatusModel.phyDetailInfoFmt % ( '  Lane %d' % lane, 'off' ) )

   def printTxEqInfo( self ):
      print( PhyStatusModel.phyDetailLongHeaderFmt % ( 'Tx Equalization' ) )
      colWidths = ( 8, 5, 5, 5, 6, 6, 6 )
      header = '    ' + ' '.join( f'{{:{w}}}' for w in colWidths )
      row = header.replace( ':', ':<' )
      print( header.format( 'Lane', 'pre2', 'pre1', 'main', 'post1', 'post2',
                           'post3' ) )

      for lane in self.sortedSerdesStats:
         info = self.serdesStats[ lane ]
         # In NRZ mode, only 3 txTaps are programmable ( pre, main, post )
         if len( info.txTaps ) == 3:
            print( row.format( lane,
                               'x',
                               info.txTaps.get( -1, 'x' ),
                               info.txTaps.get( 0, 'x' ),
                               info.txTaps.get( 1, 'x' ),
                               'x',
                               'x' ) )
         else:
            print( row.format( lane,
                               info.txTaps.get( -2, 'x' ),
                               info.txTaps.get( -1, 'x' ),
                               info.txTaps.get( 0, 'x' ),
                               info.txTaps.get( 1, 'x' ),
                               info.txTaps.get( 2, 'x' ),
                               info.txTaps.get( 3, 'x' ) ) )

   def printSerdesStats( self ):
      print( PhyStatusModel.phyDetailLongHeaderFmt % ( 'SerDes Statistics' ) )
      colWidths = ( 5, 6, 6, 12, 6, 6, 8, 15, 10 )
      header = '    ' + ' '.join( f'{{:{w}}}' for w in colWidths )
      row = header.replace( ':', ':<' )
      print( header.format( "Lane", "RXPPM", "CLK90", "PF(M,L,H)", "VGA", "DCO",
                           "P1mV", "TP(0,1,2,3)", "TXPPM" ) )

      for lane in self.sortedSerdesStats:
         info = self.serdesStats[ lane ]
         pfTaps = [ info.mainPeakingFilter, info.lowFrequncyPeakingFilter,
                    info.highFrequencyPeakingFilter ]

         tpTaps = [ info.m1SlicerThreshold, info._channelLossEstimate, info._afeBw,
                    info._usrStatusEqDebug1 ]
         print( row.format( lane, info.rxPpm, info._clk90,
                           ( ",".join( str( x ) for x in pfTaps ) ), info.vga,
                           info._dcOffset, info.p1SlicerThreshold,
                           ( ",".join( str( x ) for x in tpTaps ) ), info.txPpm ) )
      # Eye values are printed only when at least one lane has eye values set.
      displayEyeValues = any( info.eyeLeft is not None or info.eyeRight is not None
            or info.eyeUpper is not None or info.eyeLower is not None
                                for info in self.serdesStats.values() )
      eyeValuesHeader = "EYE(L,R,U,D)" if displayEyeValues else ""
      print( "    {:5} {:20} {:10} {:20}".format( "Lane", "DFE(1,2,3,4,5,6)",
                                                      "CDR", eyeValuesHeader ) )
      for lane in self.sortedSerdesStats:
         info = self.serdesStats[ lane ]
         if info.rxPamMode == rxPamModeToStr[ 0 ]:
            dfeTaps = [ info.dfeTaps.get( 1, "x" ), info.dfeTaps.get( 2, "x" ),
                        info.dfeTaps.get( 3, "x" ), info.dfeTaps.get( 4, "x" ),
                        info.dfeTaps.get( 5, "x" ), info.dfeTaps.get( 6, "x" ) ]
            # NRZ is assumed to support eye values
            eyeValues = [ info.eyeLeft, info.eyeRight, info.eyeUpper, info.eyeLower ]
         else:
            dfeTaps = [ "x", info.dfeTaps.get( 2, "x" ),
                        info.dfeTaps.get( 3, "x" ), info.dfeTaps.get( 4, "x" ),
                        info.dfeTaps.get( 5, "x" ), info.dfeTaps.get( 6, "x" ) ]
            if displayEyeValues:
               getEyeOrX = lambda val : 'x' if val is None else val
               eyeValues = [ getEyeOrX( info.eyeLeft ),
                             getEyeOrX( info.eyeRight ),
                             getEyeOrX( info.eyeUpper ),
                             getEyeOrX( info.eyeLower ) ]
            else:
               eyeValues = []
         dfeTapsStr = ",".join( str ( x ) for x in dfeTaps )
         eyeValuesStr = ",".join( str ( x ) for x in eyeValues )
         cdrModeStrPrefix = 'auto-'
         if info.brPdForced or info.osPdForced:
            cdrModeStrPrefix = 'forced-'
         cdrModeStr = cdrModeStrPrefix + info.brPdEnable
         print( "    {:<5} {:<20} {:<10} {:<20}".format( lane,
               dfeTapsStr, cdrModeStr, eyeValuesStr ) )

   def renderSerdesDetail( self ):
      self.sortedSerdesStats = sorted( self.serdesStats )
      # print precoding info once http://reviewboard/r/153440/ merges
      self.printLinkTrainingInfo()
      self.printTxEqInfo()
      self.printSerdesStats()

   def getSerdesLaneModel( self ):
      return BlackhawkPhySerdesLaneModelBase

class BlackhawkPhySerdesModel( BlackhawkPhySerdesModelBase ):
   def getSerdesLaneModel( self ):
      return BlackhawkPhySerdesLaneModel

class BlackhawkLtOptionalPhySerdesModel( BlackhawkPhySerdesModelBase ):
   def getSerdesLaneModel( self ):
      return BlackhawkLtOptionalPhySerdesLaneModel

class FalconPhySerdesLaneModel( SerdesLaneStatus ):
   # Protected fields(fields with preceding underscore) are not included in JSON
   # output.
   _clkp1 = CliModel.Int( help="Delay of diagnostic LMS p1 slicer" )
   mainPeakingFilter = CliModel.Int( help="Peaking filter main settings" )
   lowFrequncyPeakingFilter = CliModel.Int( help="Low frequency peaking"
                                            " filter control" )
   eyeLeft = CliModel.Int( help="Horizontal left eye margin @ 1e-5 as seen"
                           " by internal diagnostic slicer in mUI" )
   eyeRight = CliModel.Int( help="Horizontal right eye margin @ 1e-5 as seen"
                            " by internal diagnostic slicer in mUI" )
   eyeUpper = CliModel.Int( help="Vertical upper eye margin @ 1e-5 as seen"
                            " by internal diagnostic slicer in mV" )
   eyeLower = CliModel.Int( help="Vertical lower eye margin @ 1e-5 as seen"
                            " by internal diagnostic slicer in mV" )

   def toModel( self, serdes ):
      super().toModel( serdes )
      self._clkp1 = serdes.clkp1
      self.mainPeakingFilter = serdes.pfMain
      self.lowFrequncyPeakingFilter = serdes.pf2Ctrl
      self.eyeLeft = serdes.heyeLeft
      self.eyeRight = serdes.heyeRight
      self.eyeUpper = serdes.veyeUpper
      self.eyeLower = serdes.veyeLower
      return self

class FalconPhySerdesModel( CliModel.Model ):
   serdesStats = CliModel.Dict( keyType=int,
                                valueType=FalconPhySerdesLaneModel,
                                help="A mapping of SerDes lane to"
                                " SerDes statistics" )

   def toModel( self, serdes ):
      for lane, stats in serdes.items():
         laneStats = FalconPhySerdesLaneModel().toModel( stats )
         # XXX kewei: Remove serdesId once BUG336826 is resolved.
         self.serdesStats[ getattr( lane, 'laneId', lane ) ] = laneStats
      return self

   def printTxEqInfo( self ):
      print( PhyStatusModel.phyDetailLongHeaderFmt % ( 'Tx Equalization' ) )
      colWidths = ( 5, 5, 5, 6, 6, 6 )
      header = "    " + " ".join( f"{{:{w}}}" for w in colWidths )
      row = header.replace( ":", ":<" )
      print( header.format( "Lane", "pre1", "main", "post1", "post2", "post3" ) )

      for lane in sorted( self.serdesStats ):
         info = self.serdesStats[ lane ]
         print( row.format( lane, info.txTaps.get( -1, "x" ),
                           info.txTaps.get( 0, "x" ),
                           info.txTaps.get( 1, "x" ),
                           info.txTaps.get( 2, "x" ),
                           info.txTaps.get( 3, "x" ) ) )

   def printSerdesStats( self ):
      print( PhyStatusModel.phyDetailLongHeaderFmt % ( 'SerDes Statistics' ) )
      colWidths = ( 5, 6, 6, 6, 9, 6, 6, 6 )
      header = "    " + " ".join( f"{{:{w}}}" for w in colWidths )
      row = header.replace( ":", ":<" )
      print( header.format( "Lane", "RXPPM", "CLK90", "CLKP1", "PF(M,L)", "VGA",
                           "DCO", "TXPPM" ) )

      for lane in sorted( self.serdesStats ):
         info = self.serdesStats[ lane ]
         pfStr = "%d,%d" % ( info.mainPeakingFilter,
                             info.lowFrequncyPeakingFilter )
         print( row.format( lane, info.rxPpm, info._clk90, info._clkp1,
                           pfStr, info.vga, info._dcOffset, info.txPpm ) )
      # Eye values are printed only when at least one lane runs in NRZ mode.
      # Since all lane in Falcon runs in NRZ mode, Eye values are always printed.
      eyeValuesHeader = "EYE(L,R,U,D)"
      print( "    {:5} {:20} {:20}".format( "Lane", "DFE(1,2,3,4,5,6)",
                                              eyeValuesHeader ) )

      for lane in sorted( self.serdesStats ):
         info = self.serdesStats[ lane ]
         dfeTaps = \
            [ info.dfeTaps.get( 1, "x" ), info.dfeTaps.get( 2, "x" ),
              info.dfeTaps.get( 3, "x" ), info.dfeTaps.get( 4, "x" ),
              info.dfeTaps.get( 5, "x" ), info.dfeTaps.get( 6, "x" ) ]
         eyeValues = [ info.eyeLeft, info.eyeRight, info.eyeUpper, info.eyeLower ]
         dfeTapsStr = ",".join( str( x ) for x in dfeTaps )
         eyeValuesStr = ",".join( str ( x ) for x in eyeValues )
         print( "    {:<5} {:<20} {:<20}".format( lane,
               dfeTapsStr, eyeValuesStr ) )

   def renderSerdesDetail( self ):
      self.printTxEqInfo()
      self.printSerdesStats()

headerIndent = '    '
txEqColWidths = ( 8, 5, 5, 5, 6, 6 )
txEqHeaderFmt = headerIndent + ' '.join(
                f'{{:{w}}}' for w in txEqColWidths )
txEqHeader = txEqHeaderFmt.format( 'Lane', 'pre2', 'pre1', 'main', 'post1', 'post2' )
serdesColWidths1 = ( 5, 6, 19, 2, 2, 12, 4 )
serdesHeaderFmt1 = headerIndent + ' '.join(
                   f'{{:{w}}}' for w in serdesColWidths1 )
serdesHeader1 = serdesHeaderFmt1.format( 'Lane', 'RXPPM', 'CTLE Peaking(S1,S2)',
                                         'G1', 'G2', 'ChEst(OF/HF)', 'DAC' )
serdesColWidths2 = ( 5, 21, 16, 10 )
serdesHeaderFmt2 = headerIndent + ' '.join(
                   f'{{:{w}}}' for w in serdesColWidths2 )
serdesPam4Header = serdesHeaderFmt2.format( 'Lane', 'DFE(F0,F1,F1/F0,F13)',
                                            'Pll Vco({},{})', 'Eye(U,C,D)' )
serdesNrzHeader = serdesHeaderFmt2.format( 'Lane', 'DFE(F1,F2,F3)',
                                           'Pll Vco({},{})', 'Eye' )

class CredoPhyAdaptCnt( EventsModel.CountAndChangedStat ):
   def renderPhyDetail( self, lane ):
      print( phyDetailStatFmt % ( '  Lane %d' % ( lane ), self.value, self.changes,
            Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )

class CredoPhyReAdaptCnt( EventsModel.CountAndChangedStat ):
   def renderPhyDetail( self, lane ):
      print( phyDetailStatFmt % ( '  Lane %d' % ( lane ), self.value, self.changes,
            Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )

class CredoPhyLinkLostCnt( EventsModel.CountAndChangedStat ):
   def renderPhyDetail( self, lane ):
      print( phyDetailStatFmt % ( '  Lane %d' % ( lane ), self.value, self.changes,
            Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )

class CredoPhyLossOfSignalCnt( EventsModel.CountAndChangedStat ):
   def renderPhyDetail( self, lane ):
      print( phyDetailStatFmt % ( '  Lane %d' % ( lane ), self.value, self.changes,
            Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )

class CredoPhyAdaptCompleted( EventsModel.ConditionAndChangedStat ):
   def renderPhyDetail( self, lane ):
      print( phyDetailStatFmt % ( '  Lane %d' % ( lane ),
            'ok' if self.value else 'not completed', self.changes,
            Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )

class CredoPhyPllVcoCap( CliModel.Model ):
   txPllCap = CliModel.Int( optional=True,
         help='Tx VcoCap of Pll, encoded as 0 to 127' )
   rxPllCap = CliModel.Int( optional=True,
         help='Rx VcoCap of Pll, encoded as 0 to 127' )

   def toModel( self, p ):
      for attr in ( 'txPllCap', 'rxPllCap' ):
         setattr( self, attr, getattr( p, attr ) )
      return self

class CredoPhyTxTaps( CliModel.Model ):
   pre2 = CliModel.Int( help=' The second pre cursor tap' )
   pre1 = CliModel.Int( help=' The first pre cursor tap' )
   main = CliModel.Int( help=' The main cursor tap' )
   post1 = CliModel.Int( help=' The first post cursor tap' )
   post2 = CliModel.Int( help=' The second post cursor tap' )

   def toModel( self, laneStatus ):
      for attr in ( 'pre2', 'pre1', 'main', 'post1', 'post2' ):
         setattr( self, attr, getattr( laneStatus, attr ) )
      return self

   def renderTxEqInfo( self, laneId ):
      row = txEqHeaderFmt.replace( ':', ':<' )
      print( row.format( laneId, self.pre2, self.pre1,
                        self.main, self.post1, self.post2 ) )

class CredoPhySerdes( CliModel.Model ):
   _modulation = CliModel.Enum( values=list( ethModulationToStr.values() ),
                                help='Modulation' )
   dfeTaps = CliModel.List( valueType=float, optional=True,
         help='SerDes lane Decision Feedback Equalizer taps, ordered as main tap '
         'first tap, second tap, ... , last tap and initial tap' )
   txTaps = CliModel.Submodel( valueType=CredoPhyTxTaps, optional=True,
         help='SerDes lane TX pre-emphasis taps' )
   adaptCnt = CliModel.Submodel( valueType=CredoPhyAdaptCnt, optional=True,
         help='Firmware adaptation counter' )
   reAdaptCnt = CliModel.Submodel( valueType=CredoPhyReAdaptCnt, optional=True,
         help='Firmware re-adaptation counter' )
   linkLostCnt = CliModel.Submodel( valueType=CredoPhyLinkLostCnt, optional=True,
         help='Firmware link lost counter' )
   losCnt = CliModel.Submodel( valueType=CredoPhyLossOfSignalCnt, optional=True,
         help='Firmware lost of signal counter' )
   adaptCompleted = CliModel.Submodel( valueType=CredoPhyAdaptCompleted,
         optional=True, help='Firmware adaptation completed' )
   pllVcoCap = CliModel.Submodel( valueType=CredoPhyPllVcoCap, optional=True,
         help='Voltage control oscillator capacitor of phase locked loop' )
   rxFreqPPM = CliModel.Int( help='Frequency offset of local reference clock with '
                                  'respect to the frequency of received data in '
                                  'parts per million of Hz' )
   ctlePeaking = CliModel.List( valueType=int, optional=True,
         help='SerDes lane Continuous Time Linear Equalizer peaking values' )
   ctleGain = CliModel.List( valueType=int, optional=True,
         help='SerDes lane Continuous Time Linear Equalizer gain values' )
   channelEst = CliModel.Float( optional=True,
         help='Channel estimation' )
   signalAmplitude = CliModel.Int( optional=True,
         help='Signal envelope to represent the '
         'amplitude of signal after equalization, encoded as 0 to 63' )
   highFreqTrans = CliModel.Int( optional=True,
         help='High frequency transition count after equalization' )
   rxDacGain = CliModel.Int( optional=True, help='RX digital sampler gain value' )
   eyeMargins = CliModel.List( valueType=float, optional=True,
         help='Eye margin in mV, in case of multi-bit encodings, '
         'ordered by top to bottom' )
   linkTrainingEnable = CliModel.Bool( help="Tx link training enable" )
   linkTrainingStatus = CliModel.Enum( values=list( linktrnStsToStr.values() ),
                                       help="Link training Status" )

   def isPam4( self ):
      return self._modulation == 'PAM-4'

   def renderStatistics1( self, laneId ):
      # pylint: disable-next=bad-string-format-type
      chestStr = '%.2f(%02d/%02d)' % ( self.channelEst, self.signalAmplitude,
            self.highFreqTrans )
      ctleStr = '%d(%d,%d)' % ( self.ctlePeaking[ 0 ], self.ctlePeaking[ 1 ],
                                self.ctlePeaking[ 2 ] )
      row = serdesHeaderFmt1.replace( ':', ':<' )
      print( row.format( laneId, self.rxFreqPPM, ctleStr, self.ctleGain[ 0 ],
                        self.ctleGain[ 1 ], chestStr, self.rxDacGain ) )

   def renderStatistics2( self, laneId ):
      row = serdesHeaderFmt2.replace( ':', ':<' )
      if self._modulation == 'PAM-4':
         tap0 = self.dfeTaps[ 0 ]
         tap1 = self.dfeTaps[ 1 ]
         ratio = tap1 / tap0 if tap0 else 0.0
         taps = '%.2f,%.2f,%.2f,%.2f'
         taps %= ( tap0, tap1, ratio, self.dfeTaps[ -1 ] )
         eyes = ','.join( '%.0f' % v for v in self.eyeMargins )
      else:
         taps = '%.0f,%.0f,%.0f'
         taps %= ( self.dfeTaps[ 1 ], self.dfeTaps[ 2 ], self.dfeTaps[ 3 ] )
         eyes = int( self.eyeMargins[ 0 ] )
      # pylint: disable-next=bad-string-format-type
      caps = '%d,%d' % ( self.pllVcoCap.txPllCap, self.pllVcoCap.rxPllCap )
      print( row.format( laneId, taps, caps, eyes ) )

   def renderLinkTrainingInfo( self, laneId ):
      if self.linkTrainingEnable:
         print( PhyStatusModel.phyDetailInfoFmt % ( '  Lane %d' % laneId,
                                                   self.linkTrainingStatus ) )
      else:
         print( PhyStatusModel.phyDetailInfoFmt % ( '  Lane %d' % laneId, 'off' ) )

   def toModel( self, laneStatus ):
      self.adaptCnt = CredoPhyAdaptCnt().toModel( laneStatus.adaptCnt )
      self.reAdaptCnt = CredoPhyReAdaptCnt().toModel( laneStatus.reAdaptCnt )
      self.linkLostCnt = CredoPhyLinkLostCnt().toModel( laneStatus.linkLostCnt )
      self.adaptCompleted = CredoPhyAdaptCompleted().toModel(
            laneStatus.adaptCompleted )
      self.losCnt = CredoPhyLossOfSignalCnt().toModel( laneStatus.losCnt ) 
      self.linkTrainingEnable = laneStatus.linkTrainingEnable
      self.linkTrainingStatus = linktrnStsToStr[ laneStatus.linkTrainingStatus ]

      self.txTaps = CredoPhyTxTaps().toModel( laneStatus )
      self.pllVcoCap = CredoPhyPllVcoCap().toModel( laneStatus.pllVcoCap )
      if laneStatus.modulation == 'modulationUnknown':
         # No valid LaneStatus could be printed
         return None
      self._modulation = ethModulationToStr[ laneStatus.modulation ]
      for attr in ( 'rxFreqPPM', 'channelEst', 'signalAmplitude',
                    'highFreqTrans', 'rxDacGain' ):
         setattr( self, attr, getattr( laneStatus, attr ) )
      for attr in ( 'ctlePeaking', 'ctleGain', 'dfeTaps', 'eyeMargins' ):
         data = getattr( laneStatus, attr )
         sortedKey = sorted( data.keys() )
         setattr( self, attr, list( data[ k ] for k in sortedKey ) )
      return self


class OspreyPhySerdesLaneModel( SerdesLaneStatus ):
   rxPamMode = CliModel.Enum( values=list( rxPamModeToStr.values() ),
                              help="Receiver PAM4 mode" )
   brPdEnable = CliModel.Enum( values=list( brPdEnToStr.values() ),
                               help="Baud rate phase detector enable" )
   brPdForced = CliModel.Bool( optional=True, help="Whether the BR CDR is forced" )
   osPdForced = CliModel.Bool( optional=True, help="Whether the OS CDR is forced" )
   enable6Taps = CliModel.Bool( help="6 tap TX equalization enabled" )
   disableEyeDisplay = CliModel.Bool( help="Whether eye values are to be displayed" )
   mid1PeakingFilter = CliModel.Int( help="Peaking filter mid1 settings" )
   mid2PeakingFilter = CliModel.Int( help="Peaking filter mid2 settings" )
   lowFrequencyPeakingFilter = CliModel.Int( help="Low frequency peaking"
                                             " filter control" )
   highFrequencyPeakingFilter = CliModel.Int( help="High frequency peaking"
                                              " filter control" )
   fga = CliModel.Int( help="Fixed Gain Amplifier gain" )
   p1mV = CliModel.Int( help="Vertical threshold voltage of p1 slicer (mV)" )
   tp0 = CliModel.Int( help="Tuning parameter 0. Vertical threshold voltage of m1"
                       " slicer (mV), 'channel loss hint' in PAM4 mode should always"
                       " be 0" )
   # Source: TSC-Osprey-AN100.pdf in go/th4-sw -> broadcom docs -> TH4-100G
   tp1 = CliModel.Int( help="AFE Initial Channel Loss Estimate from 0 to 31,"
                       " a negative value is an indicator of over-equalization" )
   tp2 = CliModel.Int( help="Feedback to optimize TX pre-emphasis tap pre2,"
                       " suboptimal outside +/- 20" )
   tp3 = CliModel.Int( help="Feedback to optimize TX pre-emphasis tap pre3,"
                       " suboptimal outside +/- 20" )
   rxFfeEnabled = CliModel.Bool( help="RX feed-forward equalizer enabled" )
   rxFfeGain = CliModel.Int( help="RX feed-forward equalizer gain" )
   rxFfeN1 = CliModel.Int( help="RX feed-forward equalizer pre1" )
   rxFfeP1 = CliModel.Int( help="RX feed-forward equalizer post1" )
   dfeSIdx = CliModel.Int( help="Total DFE activity Index" )
   linkTrainingEnable = CliModel.Bool( help="TX link training enable" )
   linkTrainingStatus = CliModel.Enum( values=list( linktrnStsToStr.values() ),
                                       help="Link training Status" )
   txPrecoderEnable = CliModel.Bool( help="TX precoder enable" )
   eyeLeft = CliModel.Int( help="Horizontal left eye margin @ 1e-5 as seen"
                           " by internal diagnostic slicer in mUI" )
   eyeRight = CliModel.Int( help="Horizontal right eye margin @ 1e-5 as seen"
                            " by internal diagnostic slicer in mUI" )
   eyeUpper = CliModel.Int( help="Vertical upper eye margin @ 1e-5 as seen"
                            " by internal diagnostic slicer in mV" )
   eyeLower = CliModel.Int( help="Vertical lower eye margin @ 1e-5 as seen"
                            " by internal diagnostic slicer in mV" )
   txPiEn = CliModel.Enum( values=list( txPiEnToStr.values() ),
                           help="TX phase interpolator enable" )

   def toModel( self, serdes ):
      super().toModel( serdes )
      self.rxPamMode = rxPamModeToStr[ serdes.rxPamMode ]
      self.brPdEnable = brPdEnToStr[ serdes.brPdEn ]
      self.brPdForced = serdes.brPdForced
      self.osPdForced = serdes.osPdForced
      self.enable6Taps = serdes.enable6Taps
      self.disableEyeDisplay = serdes.disableEyeDisplay
      self.mid1PeakingFilter = serdes.pfMid1
      self.mid2PeakingFilter = serdes.pfMid2
      self.lowFrequencyPeakingFilter = serdes.pfLow
      self.highFrequencyPeakingFilter = serdes.pfHigh
      self.fga = serdes.fga
      self.p1mV = serdes.p1mV
      self.tp0 = serdes.tp0
      self.tp1 = serdes.tp1
      self.tp2 = serdes.tp2
      self.tp3 = serdes.tp3
      self.rxFfeEnabled = serdes.rxFfeEnabled
      self.rxFfeGain = serdes.rxFfeGain
      self.rxFfeN1 = serdes.rxFfeN1
      self.rxFfeP1 = serdes.rxFfeP1
      self.dfeSIdx = serdes.dfeSIdx
      self.linkTrainingEnable = serdes.linktrnEn
      self.linkTrainingStatus = linktrnStsToStr[ serdes.linktrnSts ]
      self.txPrecoderEnable = serdes.txPrecEn
      self.eyeLeft = serdes.heyeLeft
      self.eyeRight = serdes.heyeRight
      self.eyeUpper = serdes.veyeUpper
      self.eyeLower = serdes.veyeLower
      self.txPiEn = txPiEnToStr[ serdes.txPiEn ]
      return self

# Example output
# Lane 0 showing max length values for comma separated
# -----------------------------------------------------------------------------
# Link Training
#   Lane 0                    off
#   Lane 1                    off
#   Lane 2                    off
#   Lane 3                    off
# Tx Equalization
#   Lane     pre3  pre2  pre1  main  post1  post2
#   0        -...  -...  -...  -...  -...   -...
#   1        0     0     -12   88    -26    0
#   2        0     0     -12   88    -26    0
#   3        0     0     -12   88    -26    0
# SerDes Statistics
#   Lane  RXPPM  CLK90  PF(M1,M2,L,H)    VGA(1,2)   DCO    P1mV     TXPPM
#   0     -..... -...   ...,...,...,...  -...,-...  -...   -.....   -.....*
#   1     0      84     0,5,2,45         20         -5     148      0
#   2     0      82     0,5,2,45         24         11     147      0
#   3     0      78     0,5,2,45         21         -10    148      0
#   Lane  TP(0,1,2,3)                  RXFFE(G,1,2)
#   0     -.....,-.....,-...,-...      ...,-...,-...
#   1     86,3,10,0
#   2     85,246,9,0
#   3     86,252,10,
#   Lane  DFE(1,2,3,4,5,S)             EYE(L,R,U,D)
#   0     -...,-...,-...,-...,-...,... .....,.....,.....,.....
#   1     73,7,1,1,1,0                 267,472,161,106
#   2     72,6,1,1,0,0                 285,478,172,107
#   3     73,6,1,1,1,0                 308,453,186,105
class OspreyPhySerdesModel( CliModel.Model ):
   serdesStats = CliModel.Dict( keyType=int,
                                valueType=OspreyPhySerdesLaneModel,
                                help="The SerDes statistics" )

   _sortedSerdesStats = CliModel.List( valueType=int,
                                       help="Sorted SerDes statistics")

   def toModel( self, serdes ):
      for lane, stats in serdes.items():
         laneStats = OspreyPhySerdesLaneModel().toModel( stats )
         # XXX kewei: Remove serdesId once BUG336826 is resolved.
         self.serdesStats[ getattr( lane, 'laneId', lane ) ] = laneStats
      return self

   def printLinkTrainingInfo( self ):
      print( PhyStatusModel.phyDetailLongHeaderFmt % ( "Link Training" ) )
      for lane in self._sortedSerdesStats:
         info = self.serdesStats[ lane ]
         if info.linkTrainingEnable:
            print( PhyStatusModel.phyDetailInfoFmt % ( '  Lane %d' % lane,
                  info.linkTrainingStatus ) )
         else:
            print( PhyStatusModel.phyDetailInfoFmt % ( '  Lane %d' % lane, 'off' ) )

   def printTxEqInfo( self ):
      print( PhyStatusModel.phyDetailLongHeaderFmt % ( 'Tx Equalization' ) )
      colWidths = ( 8, 5, 5, 5, 5, 6, 6 )
      header = '    ' + ' '.join( f'{{:{w}}}' for w in colWidths )
      row = header.replace( ':', ':<' )
      print( header.format( 'Lane', 'pre3', 'pre2', 'pre1', 'main', 'post1',
                            'post2' ) )

      for lane in self._sortedSerdesStats:
         info = self.serdesStats[ lane ]
         if info.enable6Taps:
            print( row.format( lane,
                               info.txTaps.get( -3, 'x' ),
                               info.txTaps.get( -2, 'x' ),
                               info.txTaps.get( -1, 'x' ),
                               info.txTaps.get( 0, 'x' ),
                               info.txTaps.get( 1, 'x' ),
                               info.txTaps.get( 2, 'x' ) ) )
         else:
            print( row.format( lane,
                               "x",
                               "x",
                               info.txTaps.get( -1, 'x' ),
                               info.txTaps.get( 0, 'x' ),
                               info.txTaps.get( 1, 'x' ),
                               "x" ) )

   def printSerdesStats( self ):
      #   Lane  RXPPM  CLK90  PF(M1,M2,L,H)   VGA(1,2)   DCO    P1mV     TXPPM
      #   0     -..... -...   ...,...,...,... -...,-...  -...   -.....   -.....*
      print( PhyStatusModel.phyDetailLongHeaderFmt % ( 'SerDes Statistics' ) )
      colWidths = ( 5, 6, 6, 16, 10, 6, 8, 7 )
      header = '    ' + ' '.join( f'{{:{w}}}' for w in colWidths )
      row = header.replace( ':', ':<' )
      print( header.format( "Lane", "RXPPM", "CLK90", "PF(M1,M2,L,H)",
                            "VGA(1,2)", "DCO", "P1mV", "TXPPM" ) )

      for lane in self._sortedSerdesStats:
         info = self.serdesStats[ lane ]
         pfTaps = [ info.mid1PeakingFilter, info.mid2PeakingFilter,
                    info.lowFrequencyPeakingFilter,
                    info.highFrequencyPeakingFilter ]
         gainAmps = [ info.vga, info.fga ]
         txPiMark = "*" if info.txPiEn == txPiEnToStr[ 1 ] else ""
         print( row.format( lane, info.rxPpm, info._clk90,
                            ( ",".join( str( x ) for x in pfTaps ) ),
                            ( ",".join( str( x ) for x in gainAmps ) ),
                            info._dcOffset, info.p1mV,
                            str( info.txPpm ) + txPiMark ) )

      #   Lane  TP(0,1,2,3)                  RXFFE(G,1,2)
      #   0     -.....,-.....,-...,-...      ...,-...,-...
      colWidths = ( 5, 28, 14, 8 )
      header = '    ' + ' '.join( f'{{:{w}}}' for w in colWidths )
      row = header.replace( ':', ':<' )
      print( header.format( "Lane", "TP(0,1,2,3)", "RXFFE(G,1,2)", "CDR" ) )
      for lane in self._sortedSerdesStats:
         info = self.serdesStats[ lane ]
         tpTaps = [ info.tp0, info.tp1, info.tp2,
                    info.tp3 ]
         rxFfe = [ info.rxFfeGain, info.rxFfeN1, info.rxFfeP1 ]
         if info.rxFfeEnabled:
            rxFfe = [ info.rxFfeGain, info.rxFfeN1, info.rxFfeP1 ]
         else:
            rxFfe = [ info.rxFfeGain, "x", "x" ]
         tpTapsStr = ",".join( str ( x ) for x in tpTaps )
         rxFfeStr = ",".join( str ( x ) for x in rxFfe )
         cdrModeStrPrefix = 'auto-'
         if info.brPdForced or info.osPdForced:
            cdrModeStrPrefix = 'forced-'
         cdrModeStr = cdrModeStrPrefix + info.brPdEnable
         print( row.format( lane, tpTapsStr, rxFfeStr, cdrModeStr ) )

      #   Lane  DFE(1,2,3,4,5,S)             EYE(L,R,U,D)
      #   0     -...,-...,-...,-...,-...,... .....,.....,.....,.....
      colWidths = ( 5, 28, 24 )
      header = '    ' + ' '.join( f'{{:{w}}}' for w in colWidths )
      row = header.replace( ':', ':<' )
      displayEyeValues = not all( info.disableEyeDisplay
                                  for info in self.serdesStats.values() )
      eyeValuesHeader = "EYE(L,R,U,D)" if displayEyeValues else ""
      print( header.format( "Lane", "DFE(1,2,3,4,5,S)", eyeValuesHeader ) )
      for lane in self._sortedSerdesStats:
         info = self.serdesStats[ lane ]
         dfeTaps = []
         eyeValues = []
         if info.disableEyeDisplay:
            # Don't fetch if there are no values
            eyeValues = [ 'x', 'x', 'x', 'x' ] if displayEyeValues else []
         elif info.rxFfeEnabled:
            eyeValues = [ info.eyeLeft, info.eyeRight,
                          info.eyeUpper, info.eyeLower ]
         else:
            eyeValues = [ "x", "x", info.eyeUpper, info.eyeLower ]

         if info.rxPamMode == rxPamModeToStr[ 0 ]:
            dfeTaps = [ info.dfeTaps.get( 1, "x" ), info.dfeTaps.get( 2, "x" ),
                        info.dfeTaps.get( 3, "x" ), info.dfeTaps.get( 4, "x" ),
                        info.dfeTaps.get( 5, "x" ), info.dfeSIdx ]
         else:
            dfeTaps = [ "x", info.dfeTaps.get( 2, "x" ),
                        info.dfeTaps.get( 3, "x" ), info.dfeTaps.get( 4, "x" ),
                        info.dfeTaps.get( 5, "x" ), info.dfeSIdx ]


         dfeTapsStr = ",".join( str ( x ) for x in dfeTaps )
         eyeValuesStr = ",".join( str ( x ) for x in eyeValues )
         print( row.format( lane, dfeTapsStr, eyeValuesStr ) )

   def renderSerdesDetail( self ):
      self._sortedSerdesStats = sorted( self.serdesStats )
      self.printLinkTrainingInfo()
      self.printTxEqInfo()
      self.printSerdesStats()
