# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliGlobal
import CliMatcher
import CliParser
import ConfigMount
from CliMode.PhyTxEqProfilesMode import (
   ProfileMode, 
   ProfilesMode, 
   TapsMode 
)
from Toggles.PhyEeeToggleLib import togglePhyTxEqProfilesConfigCliCmdEnabled
import Tracing
ProfilesConfigCliEnabled = togglePhyTxEqProfilesConfigCliCmdEnabled()
supportedTapNames = [ 'pre1', 'pre2', 'pre3', 'main', 'post1', 'post2', 'post3' ]
traceHandle = Tracing.Handle( 'PhyTxEqProfilesConfigCli' )
t0 = traceHandle.trace0
gv = CliGlobal.CliGlobal( phyTxEqProfiles=None )

#-------------------------------------------------------------------------------
# The "phy transmitter equalization profiles" mode command.
#-------------------------------------------------------------------------------
class ConfigPhyTxEqProfiles( CliCommand.CliCommandClass ):
   syntax = 'phy transmitter equalization profiles'
   data = {
      'phy': 'Configure phy-specific parameters',
      'transmitter': 'Configure transmitter parameters',
      'equalization': 'Configure equalization parameters',
      'profiles': 'Configure profiles parameters'
   }
   @staticmethod
   def handler( mode, args ):
      t0( 'phy transmitter equalization profiles handler' )
      childMode = mode.childMode( ProfilesConfigMode )
      mode.session_.gotoChildMode( childMode )

if ProfilesConfigCliEnabled:
   BasicCli.GlobalConfigMode.addCommandClass( ConfigPhyTxEqProfiles )

#
# Management config mode for health monitor.
#
class ProfilesConfigMode( ProfilesMode, BasicCli.ConfigModeBase ):
   name = "phy-tx-eq-profile"
   modeParseTree = CliParser.ModeParseTree()
   def __init__( self, parent, session ):
      ProfilesMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-------------------------------------------------------------------------------
# The "[no|default] profile PROFILE_NAME" mode command.
#-------------------------------------------------------------------------------
class AddProfile( CliCommand.CliCommandClass ):
   syntax = 'profile PROFILE_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'profile': 'Phy transmitter equalization profile',
      'PROFILE_NAME': CliMatcher.DynamicNameMatcher(
            lambda mode: gv.phyTxEqProfiles.customProfiles,
            helpname='WORD',
            helpdesc='Profile name' )
   }

   @staticmethod
   def handler( mode, args ):
      profileName = args[ 'PROFILE_NAME' ]
      t0( 'profile', profileName )
      childMode = mode.childMode( ProfileConfigMode,
                                  profileName=profileName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profileName = args[ 'PROFILE_NAME' ]
      # now remove profile-name from gv.phyTxEqProfiles.customProfiles collection
      if profileName in gv.phyTxEqProfiles.customProfiles:
         removeProfile( profileName )
      else:
         mode.addWarning( f"Profile '{profileName}' not registered." )
         
def removeProfile( profileName ):
   del gv.phyTxEqProfiles.customProfiles[ profileName ]

if ProfilesConfigCliEnabled:
   ProfilesConfigMode.addCommandClass( AddProfile )

#
# Management config mode for profile.
#
class ProfileConfigMode( ProfileMode, BasicCli.ConfigModeBase ):
   name = "profile configuration"
   modeParseTree = CliParser.ModeParseTree()
   def __init__( self, parent, session, profileName ):
      self.profileName = profileName
      ProfileMode.__init__( self, profileName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.addProfile()
   def addProfile( self ):
      # now add profile-name to gv.phyTxEqProfiles.customProfiles collection
      if self.profileName not in gv.phyTxEqProfiles.customProfiles:
         gv.phyTxEqProfiles.customProfiles.newMember( self.profileName )
      else:
         self.addWarning( f"Profile '{self.profileName}' already registered." )

#-------------------------------------------------------------------------------
# The "taps" mode command.
#-------------------------------------------------------------------------------
class AddTaps( CliCommand.CliCommandClass ):
   syntax = 'taps'
   data = {
      'taps': 'Configure taps',
   }
 
   @staticmethod
   def handler( mode, args ):
      t0( 'taps handler' )
      childMode = mode.childMode( TapsConfigMode, profileName=mode.profileName )
      mode.session_.gotoChildMode( childMode )
      childMode.action = 'add'

if ProfilesConfigCliEnabled:
   ProfileConfigMode.addCommandClass( AddTaps )

#-------------------------------------------------------------------------------
# The "[no|default] TAP_NAME TAP_VALUE" command.
#-------------------------------------------------------------------------------
class AddTap( CliCommand.CliCommandClass ):
   syntax = 'TAP_NAME TAP_VALUE'
   noOrDefaultSyntax = syntax
   data = {
      'TAP_NAME': CliMatcher.PatternMatcher(
         pattern=r'.+',
         helpname='STRING',
         helpdesc='Transmit tuning parameter' ),
      'TAP_VALUE': CliMatcher.IntegerMatcher( 
         -150, 
         150, 
         helpdesc='Transmit tuning parameter value' )
   }

   @staticmethod
   def handler( mode, args ):
      tapName = args[ 'TAP_NAME' ]
      tapValue = args[ 'TAP_VALUE' ]
      t0( 'tap:', tapName, 'value:', tapValue )
      
      if tapName in supportedTapNames:
         mode.tapsValues[ tapName ] = tapValue
      else:
         supportedNamesStr = ', '.join( supportedTapNames )
         mode.addWarning( f"Unsupported parameter name: {tapName}. "
                          f"Supported names are: {supportedNamesStr}." )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      tapName = args[ 'TAP_NAME' ]
      if tapName in gv.phyTxEqProfiles.customProfiles[ 
         mode.profileName ].customTxTaps:
         mode.tapsValues[ tapName ] = None
      elif tapName in mode.tapsValues:
         del mode.tapsValues[ tapName ]

#
# Management config mode for profile taps.
#
class TapsConfigMode( TapsMode, BasicCli.ConfigModeBase ):
   name = "profile taps configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, profileName ):
      self.profileName = profileName
      self.action = None
      # tapsValues = { 'tapName' : tapValue }
      self.tapsValues = {}
      TapsMode.__init__( self, profileName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def addTap( self, tapName ):
      # add tap name and value to profile's customTxTaps collection
      tapValue = self.tapsValues[ tapName ]
      t0( 'adding tap', tapName, 'with value', tapValue )
      newTap = gv.phyTxEqProfiles.customProfiles[ 
         self.profileName ].customTxTaps.newMember( tapName )
      newTap.value = tapValue

   def removeTap( self, tapName ):
      t0( 'removing tap', tapName )
      # remove tap from profile's customTxTaps collection
      if tapName in gv.phyTxEqProfiles.customProfiles[ 
            self.profileName ].customTxTaps:
         removeTap( self.profileName, tapName )
      else:
         self.addWarning( f"Tap '{tapName}' not registered." )

   def handleTaps( self ):
      for tapName, tapValue in self.tapsValues.items():
         if tapValue is None:
            self.removeTap( tapName )
         else:
            self.addTap( tapName )

   def onExit( self ):
      t0( 'starting commit' )
      # update Taps
      self.handleTaps()
      t0( 'finished commiting, trying to exit' )
      BasicCli.ConfigModeBase.onExit( self )

def removeTap( profileName, tapName ):
   del gv.phyTxEqProfiles.customProfiles[ profileName ].customTxTaps[ tapName ]

if ProfilesConfigCliEnabled:
   TapsConfigMode.addCommandClass( AddTap )

def Plugin( em ):
   gv.phyTxEqProfiles = ConfigMount.mount( 
      em,
      'hardware/archer/phy/config/cli/phyTxEqProfiles',
      'Hardware::Phy::PhyTxEqProfiles',
      'w' )
