#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import Cell
import CliGlobal
import CliMatcher
from CliPlugin import EthIntfCli, IntfCli
from CliPlugin.PhyTxEqModel import (
   PhyTxEqProfilesModel,
   PhyTxEqSlot,
   PhyTxEqSlotLane
)
from CliPlugin.PhyTxEqLaneConfigCli import intfHelper
import LazyMount
import MultiRangeRule
from PhyFeatureCliLib import getPhyFeatureStatusNested
import re
import ShowCommand
from Toggles.PhyEeeToggleLib import togglePhyTxEqProfilesConfigCliCmdEnabled
import Tracing

gv = CliGlobal.CliGlobal(
   entityManager=None, phyFeatureStatusSliceDir=None, phyFeatureConfigSliceDir=None,
   xcvrConfigDir=None )

traceHandle = Tracing.Handle( 'PhyTxEqLaneShowCli' )
t0 = traceHandle.trace0

# --------------------------------------------------------------------------
# show phy transmitter equalization profile [
# ( slot SLOTID ) | ( slot SLOTID lanes LANES ) ]
# --------------------------------------------------------------------------

slotMatcher = CliMatcher.PatternMatcher(
   pattern=r"\d+(/\d+)?", helpname="WORD", helpdesc="Slot name" )

laneRangeMatcher = MultiRangeRule.MultiRangeMatcher(
   rangeFn=lambda: ( 0, 7 ), noSingletons=False, helpdesc='Lane range' )

intfTypes = ( EthIntfCli.EthIntf, ) # TODO: BUG945364 - Impl fabric intfs

def showTxEq( mode, args ):
   model = PhyTxEqProfilesModel()
   intfs = IntfCli.Intf.getAll( mode, intfType=intfTypes )
   fixed = Cell.cellType() == "fixed"

   for intf in intfs:
      slotName = intfHelper( intf.name, fixed, "slotName" )
      if not slotName: # filters non-Ethernet/Fabric intfs
         continue
      if 'slot' in args:
         if slotName != args.get( 'SLOTID' ):
            continue
      xcvrName = gv.xcvrConfigDir.intfToXcvrName.get( intf.name )
      if not intfHelper( intf.name, fixed, "primaryIntf" ):
         continue

      slotStr = re.sub( r'[^0-9/]', '', xcvrName )
      phyFeatureConfigDir = gv.phyFeatureConfigSliceDir[ intf.slotId() ]
      phyFeatureConfig = phyFeatureConfigDir.config.get( intf.name )
      phyTxEqProfilesCfg = phyFeatureConfig.phyTxEqProfilesConfig
      phyFeatureStat = getPhyFeatureStatusNested(
         gv.phyFeatureStatusSliceDir, intf.name )
      if not phyFeatureStat:
         continue
      slotModel = PhyTxEqSlot()
      for lane, status in sorted( phyFeatureStat.phyTxEqProfilesStatus.items() ):
         if 'lanes' not in args or lane - 1 in args.get( 'LANES' ).values():
            slotLaneModel = PhyTxEqSlotLane(
               operational=status.operationalPhyTxEqProfile )
            configuredProf = phyTxEqProfilesCfg.get( lane, "default" )
            slotLaneModel.configured.append( configuredProf )
            slotModel.lanes[ lane - 1 ] = slotLaneModel
      model.slot[ slotStr ] = slotModel
   return model

class PhyTxEqShowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show phy transmitter equalization profile [ ( slot SLOTID ) | \
      ( slot SLOTID lanes LANES ) ]'
   data = {
      'phy': 'Show PHY-specific information',
      'transmitter': 'Transmitter information',
      'equalization': 'Equalization information',
      'profile': 'Profile information',
      'slot': 'Slot',
      'SLOTID': slotMatcher,
      'lanes': 'Lanes',
      'LANES': laneRangeMatcher,
   }
   cliModel = PhyTxEqProfilesModel
   handler = showTxEq

if togglePhyTxEqProfilesConfigCliCmdEnabled():
   BasicCli.addShowCommandClass( PhyTxEqShowCmd )

def Plugin( em ):
   gv.entityManager = em
   gv.phyFeatureStatusSliceDir = LazyMount.mount( em,
      'hardware/archer/phy/status/feature/slice', 'Tac::Dir', 'r' )
   gv.phyFeatureConfigSliceDir = LazyMount.mount( em,
      'hardware/archer/phy/config/cli/feature/slice', 'Tac::Dir', 'r' )
   gv.xcvrConfigDir = LazyMount.mount( em,
      'hardware/xcvr/config/all/', 'Xcvr::AllConfigDir', 'r' )
