# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import ConfigMount
import BasicCli, CliToken
import CliCommand
from IpLibConsts import DEFAULT_VRF
from CliPlugin.PimBidirCliLib import RouterPimBidirBaseMode, RouterPimBidirVrfMode, \
     RouterPimBidirAfCommonSharedModelet
from CliPlugin import PimBidirCliToken
from CliPlugin import PimCliLib
import Tracing

traceHandle = Tracing.Handle( "PimBidirConfigCli" )
t0 = traceHandle.trace0

# Globals
_pimBidirConfig = None

def _pimBidirConfigCreation( vrfName ):
   if vrfName not in _pimBidirConfig.vrfConfig:
      vc = _pimBidirConfig.newVrfConfig( vrfName )
      assert vc is not None

def _pimBidirConfigDeletion( vrfName ):
   # pylint: disable-next=consider-using-f-string
   t0( "_pimBidirConfigDeletion : %s"  % vrfName )
   if vrfName in _pimBidirConfig.vrfConfig:
      pimBidirConfig = _pimBidirConfig.vrfConfig[ vrfName ]
      if vrfName != DEFAULT_VRF and \
            pimBidirConfig.groupExpiryInterval == \
            pimBidirConfig.groupExpiryIntervalDefault:
         # only delete if there is no non-default config and the VRF is not defined
         # pylint: disable-next=consider-using-f-string
         t0( "Deleting vrfConfig : %s" % vrfName )
         del _pimBidirConfig.vrfConfig[ vrfName ]

def _canDeletePimBidirVrfConfig( vrfName ):
   canDelete = True
   if vrfName in _pimBidirConfig.vrfConfig:
      config = _pimBidirConfig.vrfConfig[ vrfName ]
      if config.groupExpiryInterval != \
            config.groupExpiryIntervalDefault:
         canDelete = False
   return canDelete

def _getVrfNameFromMode( mode ):
   if hasattr( mode, 'vrfName' ):
      return mode.vrfName
   return DEFAULT_VRF

def _getPimBidirVrfConfig( vrfName ):
   # Note: v6 config is not yet supported
   pimBidirVrfConfig = None
   if vrfName in _pimBidirConfig.vrfConfig:
      pimBidirVrfConfig = _pimBidirConfig.vrfConfig[ vrfName ]
   return pimBidirVrfConfig

def _cleanupPimBidirVrfConfig( vrfName ):
   # pylint: disable-next=consider-using-f-string
   t0( "_cleanupPimBidirVrfConfig : %s " % vrfName )
   if vrfName in _pimBidirConfig.vrfConfig:
      pimBidirVrfConfig = _getPimBidirVrfConfig( vrfName )
      pimBidirVrfConfig.groupExpiryInterval = \
            pimBidirVrfConfig.groupExpiryIntervalDefault

def setGroupExpiryIntervalRange( mode, args ):
   groupExpiryInterval = args.get( 'EXPIRY_TIMER' )
   _pimBidirVrfConfig = _getPimBidirVrfConfig( _getVrfNameFromMode( mode ) )
   if _pimBidirVrfConfig:
      _pimBidirVrfConfig.groupExpiryInterval = groupExpiryInterval

def noGroupExpiryIntervalRange( mode, args ):
   _pimBidirVrfConfig = _getPimBidirVrfConfig( _getVrfNameFromMode( mode ) )
   if _pimBidirVrfConfig:
      _pimBidirVrfConfig.groupExpiryInterval = \
            _pimBidirVrfConfig.groupExpiryIntervalDefault

#----------------------------------------------------------------------
# switch(config-router-pim-bidir-*)# group-expiry-timer <value>
#----------------------------------------------------------------------
class GroupExpiryTimerCmd( CliCommand.CliCommandClass ):
   syntax = 'group-expiry-timer EXPIRY_TIMER'
   noOrDefaultSyntax = 'group-expiry-timer ...'
   data = {
         'group-expiry-timer' : PimBidirCliToken.groupExpiryTimerMatcher,
         'EXPIRY_TIMER' : PimBidirCliToken.groupExpiryTimerValueMatcher
   }
   handler = setGroupExpiryIntervalRange
   noOrDefaultHandler = noGroupExpiryIntervalRange

RouterPimBidirAfCommonSharedModelet.addCommandClass( GroupExpiryTimerCmd )

#----------------------------------------------------------------------
# Legacy:
# switch(config)# ip pim group-expiry-timer <value>
#----------------------------------------------------------------------
class IpPimGroupExpiryTimerLegacyCmd( CliCommand.CliCommandClass ):
   syntax = 'ip pim group-expiry-timer EXPIRY_TIMER'
   noOrDefaultSyntax = 'ip pim group-expiry-timer ...'
   data = {
         'ip' : CliToken.Ip.ipMatcherForConfig,
         'pim' : CliToken.Pim.pimNode,
         'group-expiry-timer' : PimBidirCliToken.groupExpiryTimerMatcher,
         'EXPIRY_TIMER' : PimBidirCliToken.groupExpiryTimerValueMatcher
   }
   handler = setGroupExpiryIntervalRange
   noOrDefaultHandler = noGroupExpiryIntervalRange

RouterPimBidirBaseMode.addCommandClass( IpPimGroupExpiryTimerLegacyCmd )
BasicCli.GlobalConfigMode.addCommandClass( IpPimGroupExpiryTimerLegacyCmd )
RouterPimBidirVrfMode.addCommandClass( IpPimGroupExpiryTimerLegacyCmd )

def Plugin( entityManager ):
   global _pimBidirConfig

   _pimBidirConfig = ConfigMount.mount( entityManager,
         'routing/pim/bidir/config',
         'Routing::Pim::Bidir::ConfigColl', 'w' )

   PimCliLib.pimBidirVrfConfiguredHook.addExtension( _pimBidirConfigCreation )
   PimCliLib.pimBidirVrfDeletedHook.addExtension( _pimBidirConfigDeletion )
   PimCliLib.canDeletePimBidirModeVrfHook.addExtension( _canDeletePimBidirVrfConfig )
   PimCliLib.pimBidirVrfConfigCleanupHook.addExtension( _cleanupPimBidirVrfConfig )
