#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import socket
import BasicCli
import CliCommand
import CliMatcher
import Arnet
from McastCommonCliLib import mcastGenRoutingSupportedIntfGuard, defaultMcastPrefix
from CliPlugin.AclCli import getAclConfig
from CliPlugin.IraIpRouteCliLib import isValidPrefixWithError as validIpv4Prefix
from CliPlugin.IraIp6RouteCliLib import isValidIpv6PrefixWithError as validIpv6Prefix
from CliPlugin.RouterMulticastCliLib import AddressFamily
from CliPlugin.IraIpIntfCli import RoutingProtocolIntfConfigModelet
# pylint: disable-next=consider-using-from-import
import CliPlugin.PimBsrCliLib as PimBsrCliLib
# pylint: disable-next=consider-using-from-import
import CliPlugin.PimBsrConfigCli as PimBsrConfigCli
from CliPlugin.IpAddrMatcher import IpAddrMatcher, IpPrefixMatcher
from CliPlugin.Ip6AddrMatcher import ( Ip6AddrMatcher,
                                       Ip6PrefixMatcher )
from CliPlugin.IntfCli import Intf
from CliPlugin.PimCliLib import ( RouterPimSparseIpv4Modelet,
                                  RouterPimSparseIpv6Modelet,
                                  RouterPimSparseSharedModelet )
from CliPlugin.PimBidirCliLib import ( RouterPimBidirIpv4Mode,
                                       RouterPimBidirIpv6Mode,
                                       RouterPimBidirSharedModelet )
import CliPlugin.VrfCli as VrfCli # pylint: disable=consider-using-from-import
from CliToken.Ip import ipMatcherForConfig, ipMatcherForConfigIf
from CliToken.Pim import ( rpMatcher,
                           accessListMatcher,
                           aclNameMatcher,
                           priorityMatcher,
                           pimNode )
import Tracing

__defaultTraceHandle__ = Tracing.Handle( 'PimBsr::Cli' )
t0 = Tracing.trace0

candidateMatcher = CliMatcher.KeywordMatcher(
      'candidate',
      helpdesc='Rendezvous point candidate ( to advertise to bootstrap router )' )
priorityNode = CliCommand.Node( matcher=priorityMatcher, maxMatches=1 )

intervalMatcher = CliMatcher.KeywordMatcher(
      'interval', helpdesc='Interval between candidate RP advertisements' )
intervalNode = CliCommand.Node( matcher=intervalMatcher, maxMatches=1 )

candidateBsrMatcher = CliMatcher.KeywordMatcher(
      'candidate', helpdesc='Become a candidate bootstrap router' )

hashmaskMatcher = CliMatcher.KeywordMatcher(
      'hashmask', helpdesc='Hash Mask value' )
hashmaskNode = CliCommand.Node( matcher=hashmaskMatcher, maxMatches=1 )

intervalBsrMatcher = CliMatcher.KeywordMatcher(
      'interval', helpdesc='Interval between bootstrap router message origination' )
intervalBsrNode = CliCommand.Node( matcher=intervalBsrMatcher, maxMatches=1 )

priorityBsrMatcher = CliMatcher.KeywordMatcher(
      'priority', helpdesc='Priority of this candidate bootstrap router' )
priorityBsrNode = CliCommand.Node( matcher=priorityBsrMatcher, maxMatches=1 )

pimMatcher = CliMatcher.KeywordMatcher(
      'pim', helpdesc='Protocol Independent Multicast (PIM)' )
pimIntfNode = CliCommand.Node(
      matcher=pimMatcher, guard=mcastGenRoutingSupportedIntfGuard )

bsrMatcher = CliMatcher.KeywordMatcher(
      'bsr', helpdesc='PIM-BSR' )

ipv4Matcher = CliMatcher.KeywordMatcher(
      'ipv4', helpdesc='IPv4 version' )

ipv6Matcher = CliMatcher.KeywordMatcher(
      'ipv6', helpdesc='IPv6 version' )

borderMatcher = CliMatcher.KeywordMatcher(
      'border', helpdesc='Enable/disable sending/receiving PIM bootstrap '
                         'router messages' )

bsrBorderMatcher = CliMatcher.KeywordMatcher(
      'bsr-border',
      helpdesc='Enable/disable sending/receiving PIM bootstrap router messages' )
bsrBorderNodeDeprecated = CliCommand.Node(
      matcher = bsrBorderMatcher,
      deprecatedByCmd='pim bsr [ ipv4|ipv6 ] border' )

rpCandidateMatcher = CliMatcher.KeywordMatcher(
      'rp-candidate',
      helpdesc='Rendezvous point candidate (to advertise to bootstrap router)' )

advertisementFilterMatcher = CliMatcher.KeywordMatcher(
      'advertisement-filter',
      helpdesc='Rendezvous point candidate advertisement filter' )

groupRangeOrAclMatchObj = object()

class Ipv4GroupRangeOrAclExpr( CliCommand.CliExpression ):
   expression = '( ADDR MASK ) | ( ADDRPREFIX ) | ( access-list ACL )'
   data = {
      'ADDR' : CliCommand.Node(
                     matcher=IpAddrMatcher( helpdesc='Group range prefix' ),
                     maxMatches=1,
                     sharedMatchObj=groupRangeOrAclMatchObj ),
      'MASK' : CliCommand.Node(
                     matcher=IpAddrMatcher( helpdesc='Group range prefix mask',
                              checkContinuous=True ),
                     maxMatches=1 ),
      'ADDRPREFIX' : CliCommand.Node(
                           matcher=IpPrefixMatcher( 
                                 helpdesc= 'Group address range with prefix' ),
                           maxMatches=1,
                           sharedMatchObj=groupRangeOrAclMatchObj ),
      'access-list' : CliCommand.Node(
                            matcher=accessListMatcher,
                            maxMatches=1,
                            sharedMatchObj=groupRangeOrAclMatchObj ),
      'ACL' : CliCommand.Node( matcher=aclNameMatcher, maxMatches=1 )
   }
   @staticmethod
   def adapter( mode, args, argsList ):
      if 'ACL' in args:
         gAcl = args[ 'ACL' ]
         args[ 'DEFAULTPREFIX' ] = defaultMcastPrefix( AddressFamily.ipv4 )
         acl = getAclConfig( 'ip', cliConf=True ).get( gAcl )
         if acl and not acl.standard:
            # pylint: disable-next=consider-using-f-string
            mode.addError( '%s is not a standard acl' % gAcl )

      else:
         if 'ADDRPREFIX' in args:
            prefixStr = args.pop( 'ADDRPREFIX' )
         elif 'ADDR' in args:
            mask = Arnet.Mask( args.pop( 'MASK' ), addrFamily=socket.AF_INET )
            # pylint: disable-next=consider-using-f-string
            prefixStr = '%s/%d' % ( args.pop( 'ADDR' ), mask.maskLen )
         else:
            args[ 'DEFAULTPREFIX' ] = defaultMcastPrefix( AddressFamily.ipv4 )
            return

         if not validIpv4Prefix( mode, prefixStr ):
            mode.addError( 'Invalid IPv4 prefix' )
            return

         args[ 'ADDRPREFIX' ] = Arnet.IpGenPrefix( prefixStr )

class Ipv6GroupRangeOrAclExpr( CliCommand.CliExpression ):
   expression = '( ADDR MASK ) | ( ADDRPREFIX ) | ( access-list ACL )'
   data = {
      'ADDR' : CliCommand.Node(
                     matcher=Ip6AddrMatcher( helpdesc='Group range prefix' ),
                     maxMatches=1,
                     sharedMatchObj=groupRangeOrAclMatchObj ),
      'MASK' : CliCommand.Node(
                     matcher=Ip6AddrMatcher( helpdesc='Group range prefix mask',
                                             checkContinuous=True ),
                     maxMatches=1 ),
      'ADDRPREFIX' : CliCommand.Node(
                           matcher=Ip6PrefixMatcher( 
                                 helpdesc= 'Group address range with prefix' ),
                           maxMatches=1,
                           sharedMatchObj=groupRangeOrAclMatchObj ),
      'access-list' : CliCommand.Node(
                            matcher=accessListMatcher,
                            maxMatches=1,
                            sharedMatchObj=groupRangeOrAclMatchObj ),
      'ACL' : CliCommand.Node(
                    matcher=aclNameMatcher,
                    maxMatches=1 )
   }
   @staticmethod
   def adapter( mode, args, argsList ):
      if 'ACL' in args:
         gAcl = args[ 'ACL' ]
         args[ 'DEFAULTPREFIX' ] = defaultMcastPrefix( AddressFamily.ipv6 )
         acl = getAclConfig( AddressFamily.ipv6, cliConf=True ).get( gAcl )
         if acl and not acl.standard:
            # pylint: disable-next=consider-using-f-string
            mode.addError( '%s is not a standard acl' % gAcl )

      else:
         if 'ADDRPREFIX' in args:
            ip6Prefix = args[ 'ADDRPREFIX' ]
            if not validIpv6Prefix( mode, ip6Prefix ):
               mode.addError( 'Invalid IPv6 prefix' )
               return
            args[ 'ADDRPREFIX' ] = Arnet.IpGenPrefix( ip6Prefix.stringValue )
         elif 'ADDR' in args:
            mask = Arnet.Mask( args.pop( 'MASK' ), addrFamily=socket.AF_INET6 )
            # pylint: disable-next=consider-using-f-string
            prefixStr = '%s/%d' % ( args.pop( 'ADDR' ), mask.maskLen )
            gPrefix = Arnet.IpGenPrefix( prefixStr )
            if not validIpv6Prefix( mode, gPrefix ):
               mode.addError( 'Invalid IPv6 prefix' )
               return
            args[ 'ADDRPREFIX' ] = gPrefix
         else:
            args[ 'DEFAULTPREFIX' ] = defaultMcastPrefix( AddressFamily.ipv6 )

class RpCandidateOptionsExpr( CliCommand.CliExpression ):
   expression = '( priority PRIORITY ) | ( interval INTERVAL )'
   data = {
      'priority' : priorityNode,
      'PRIORITY' : CliCommand.Node(
                         matcher=CliMatcher.IntegerMatcher( 0, 255, helpdesc=
                         'Priority value of RP (0 is the highest priority )' ),
                         maxMatches=1 ),
      'interval' : intervalNode,
      'INTERVAL' : CliCommand.Node(
                         matcher=CliMatcher.IntegerMatcher( 1, 21845, helpdesc=
                         'Interval between candidate RP advertisements in seconds' ),
                         maxMatches=1 ),
   }

class NoRpCandidateOptionsExpr( CliCommand.CliExpression ):
   expression = 'priority | interval'
   data = {
      'priority' : priorityNode,
      'interval' : intervalNode,
   }

class Ipv4CandidateOptionsExpr( CliCommand.CliExpression ):
   expression = '( priority PRIORITY ) | ( hashmask HASHMASK ) | '\
                '( interval INTERVAL )'
   data = {
      'priority' : priorityBsrNode,
      'PRIORITY' : CliCommand.Node(
                         matcher=CliMatcher.IntegerMatcher( 0, 255, helpdesc=
                         'Priority value this candidate bootstrap router '
                         '(255 is the highest priority )' ),
                         maxMatches=1 ),
      'hashmask' : hashmaskNode,
      'HASHMASK' : CliCommand.Node(
                         matcher=CliMatcher.IntegerMatcher( 0, 32, helpdesc=
                         'Hash Mask value used for choice resolution' ),
                         maxMatches=1 ),
      'interval' : intervalBsrNode,
      'INTERVAL' : CliCommand.Node(
                         matcher=CliMatcher.IntegerMatcher( 10, 536870906, helpdesc=
                         'Interval between bootstrap router message origination'
                         ' in seconds' ),
                         maxMatches=1 )
   }

class Ipv6CandidateOptionsExpr( CliCommand.CliExpression ):
   expression = '( priority PRIORITY ) | ( hashmask HASHMASK ) | '\
                '( interval INTERVAL )'
   data = {
      'priority' : priorityBsrNode,
      'PRIORITY' : CliCommand.Node(
                         matcher=CliMatcher.IntegerMatcher( 0, 255, helpdesc=
                         'Priority value this candidate bootstrap router '
                         '(255 is the highest priority )' ),
                         maxMatches=1 ),
      'hashmask' : hashmaskNode,
      'HASHMASK' : CliCommand.Node(
                         matcher=CliMatcher.IntegerMatcher( 0, 128, helpdesc=
                         'Hash Mask value used for choice resolution' ),
                         maxMatches=1 ),
      'interval' : intervalBsrNode,
      'INTERVAL' : CliCommand.Node(
                         matcher=CliMatcher.IntegerMatcher( 10, 536870906, helpdesc=
                         'Interval between bootstrap router message origination'
                         ' in seconds' ),
                         maxMatches=1 )
   }

class NoCandidateOptionsExpr( CliCommand.CliExpression ):
   expression = 'priority | hashmask | interval'
   data = {
      'priority' : priorityBsrNode,
      'hashmask' : hashmaskNode,
      'interval' : intervalBsrNode
   }

#--------------------------------------------------------------------------------
# (config-router-pim-sparse[-vrf]-ipv4)
# (config-router-pim-bidir[-vrf]-ipv4)
# rp candidate INTF [ { ( ( ( ADDR MASK ) |
#                           ( ADDRPREFIX ) |
#                           ( access-list ACL ) ) |
#                         ( priority PRIORITY ) |
#                         ( interval INTERVAL ) ) } ]
# ( no | default ) rp candidate [ INTF ]
#                               [ { ( ( ( ADDR MASK ) |
#                                       ( ADDRPREFIX ) |
#                                       ( access-list ACL ) ) |
#                                     ( priority | interval ) ) }
#--------------------------------------------------------------------------------
class Ipv4RpCandidateCmd( CliCommand.CliCommandClass ):
   syntax = 'rp candidate INTF [ { GROUP_RANGE_OR_ACL | RP_CANDIDATE_OPTIONS } ]'
   noOrDefaultSyntax = \
            'rp candidate [ INTF ] ' \
            '[ GROUP_RANGE_OR_ACL ] [ NO_RP_CANDIDATE_OPTIONS ] ...'
   data = {
      'rp' : rpMatcher,
      'candidate' : candidateMatcher,
      'INTF' : Intf.matcher,
      'GROUP_RANGE_OR_ACL' : Ipv4GroupRangeOrAclExpr,
      'RP_CANDIDATE_OPTIONS' : RpCandidateOptionsExpr,
      'NO_RP_CANDIDATE_OPTIONS' : NoRpCandidateOptionsExpr
   }
   handler = PimBsrConfigCli.setIpPimRpCandidate
   noOrDefaultHandler = PimBsrConfigCli.noIpPimRpCandidate

RouterPimSparseIpv4Modelet.addCommandClass( Ipv4RpCandidateCmd )
RouterPimBidirIpv4Mode.addCommandClass( Ipv4RpCandidateCmd )

#--------------------------------------------------------------------------------
# (config-router-pim-sparse[-vrf]-ipv6)
# (config-router-pim-bidir[-vrf]-ipv6)
# rp candidate INTF [ { ( ( ( ADDR MASK ) |
#                           ( ADDRPREFIX ) |
#                           ( access-list ACL ) ) |
#                         ( priority PRIORITY ) |
#                         ( interval INTERVAL ) ) } ]
# ( no | default ) rp candidate [ INTF ]
#                               [ { ( ( ( ADDR MASK ) |
#                                       ( ADDRPREFIX ) |
#                                       ( access-list ACL ) ) |
#                                     ( priority | interval ) ) }
#--------------------------------------------------------------------------------
class Ipv6RpCandidateCmd( CliCommand.CliCommandClass ):
   syntax = 'rp candidate INTF [ { GROUP_RANGE_OR_ACL | RP_CANDIDATE_OPTIONS } ]'
   noOrDefaultSyntax = \
            'rp candidate [ INTF ] ' \
            '[ GROUP_RANGE_OR_ACL ] [ NO_RP_CANDIDATE_OPTIONS ] ...'
   data = {
      'rp' : rpMatcher,
      'candidate' : candidateMatcher,
      'INTF' : Intf.matcher,
      'GROUP_RANGE_OR_ACL' : Ipv6GroupRangeOrAclExpr,
      'RP_CANDIDATE_OPTIONS' : RpCandidateOptionsExpr,
      'NO_RP_CANDIDATE_OPTIONS' : NoRpCandidateOptionsExpr
   }
   handler = PimBsrConfigCli.setIpPimRpCandidate
   noOrDefaultHandler = PimBsrConfigCli.noIpPimRpCandidate

RouterPimSparseIpv6Modelet.addCommandClass( Ipv6RpCandidateCmd )
RouterPimBidirIpv6Mode.addCommandClass( Ipv6RpCandidateCmd )


#--------------------------------------------------------------------------------
# Legacy:
# (config-router-pim-sparse[-vrf])
# (config-router-pim-bidir[-vrf])
# ip pim rp-candidate INTF [ { ( ( ( ADDR MASK ) |
#                                  ( ADDRPREFIX ) |
#                                  ( access-list ACL ) ) |
#                                ( priority PRIORITY ) |
#                                ( interval INTERVAL ) ) } ]
# ( no | default ) ip pim rp-candidate [ INTF ]
#                                      [ { ( ( ( ADDR MASK ) |
#                                              ( ADDRPREFIX ) |
#                                              ( access-list ACL ) ) |
#                                            ( priority | interval ) ) }
#--------------------------------------------------------------------------------
class IpPimRpCandidateLegacyCmd( CliCommand.CliCommandClass ):
   syntax = 'ip pim rp-candidate INTF [ { GROUP_RANGE_OR_ACL ' \
            '| RP_CANDIDATE_OPTIONS } ]'
   noOrDefaultSyntax = \
            'ip pim rp-candidate [ INTF ] [ GROUP_RANGE_OR_ACL ]' \
            '[ NO_RP_CANDIDATE_OPTIONS ] ...'
   data = {
      'ip' : ipMatcherForConfig,
      'pim' : pimNode,
      'rp-candidate' : rpCandidateMatcher,
      'INTF' : Intf.matcher,
      'GROUP_RANGE_OR_ACL' : Ipv4GroupRangeOrAclExpr,
      'RP_CANDIDATE_OPTIONS' : RpCandidateOptionsExpr,
      'NO_RP_CANDIDATE_OPTIONS' : NoRpCandidateOptionsExpr
   }
   handler = PimBsrConfigCli.setIpPimRpCandidate
   noOrDefaultHandler = PimBsrConfigCli.noIpPimRpCandidate

RouterPimSparseSharedModelet.addCommandClass( IpPimRpCandidateLegacyCmd )
RouterPimBidirSharedModelet.addCommandClass( IpPimRpCandidateLegacyCmd )
BasicCli.GlobalConfigMode.addCommandClass( IpPimRpCandidateLegacyCmd )

#--------------------------------------------------------------------------------
# (router-pim-bsr[-vrf]-ipv4)
# candidate INTF [ { ( ( ( ADDR MASK ) |
#                        ( ADDRPREFIX ) |
#                        ( access-list ACL ) ) |
#                      ( priority PRIORITY ) |
#                      ( hashmask HASHMASK ) |
#                      ( interval INTERVAL ) ) } ]
# ( no | default ) candidate [ INTF ]
#                            [ ( ( ADDR MASK ) |
#                                ( ADDRPREFIX ) |
#                                ( access-list ACL ) ) ]
#                            [ ( priority | hashmask | interval ) ]
#--------------------------------------------------------------------------------
class Ipv4CandidateCmd( CliCommand.CliCommandClass ):
   syntax = 'candidate INTF [ { ( GROUP_RANGE_OR_ACL ) | CANDIDATE_OPTIONS } ]'
   noOrDefaultSyntax = 'candidate [ INTF ] ' \
                       '[ GROUP_RANGE_OR_ACL ] [ NO_CANDIDATE_OPTIONS ] ...'
   data = {
      'candidate' : candidateBsrMatcher,
      'INTF' : Intf.matcherWithIpSupport,
      'GROUP_RANGE_OR_ACL' : Ipv4GroupRangeOrAclExpr,
      'CANDIDATE_OPTIONS' : Ipv4CandidateOptionsExpr,
      'NO_CANDIDATE_OPTIONS' : NoCandidateOptionsExpr
   }
   handler = PimBsrConfigCli.setPimBsrCandidate
   noOrDefaultHandler = PimBsrConfigCli.noPimBsrCandidate

PimBsrCliLib.RouterPimBsrIpv4Mode.addCommandClass( Ipv4CandidateCmd )

#--------------------------------------------------------------------------------
# (router-pim-bsr[-vrf]-ipv6)
# candidate INTF [ { ( ( ( ADDR MASK ) |
#                        ( ADDRPREFIX ) |
#                        ( access-list ACL ) ) |
#                      ( priority PRIORITY ) |
#                      ( hashmask HASHMASK ) |
#                      ( interval INTERVAL ) ) } ]
# ( no | default ) candidate [ INTF ]
#                            [ ( ( ADDR MASK ) |
#                                ( ADDRPREFIX ) |
#                                ( access-list ACL ) ) ]
#                            [ ( priority | hashmask | interval ) ]
#--------------------------------------------------------------------------------
class Ipv6CandidateCmd( CliCommand.CliCommandClass ):
   syntax = 'candidate INTF [ { GROUP_RANGE_OR_ACL | CANDIDATE_OPTIONS } ]'
   noOrDefaultSyntax = 'candidate [ INTF ] ' \
                       '[ GROUP_RANGE_OR_ACL ] [ NO_CANDIDATE_OPTIONS ] ...'
   data = {
      'candidate' : candidateBsrMatcher,
      'INTF' : Intf.matcherWithIpSupport,
      'GROUP_RANGE_OR_ACL' : Ipv6GroupRangeOrAclExpr,
      'CANDIDATE_OPTIONS' : Ipv6CandidateOptionsExpr,
      'NO_CANDIDATE_OPTIONS' : NoCandidateOptionsExpr
   }
   handler = PimBsrConfigCli.setPimBsrCandidate
   noOrDefaultHandler = PimBsrConfigCli.noPimBsrCandidate

PimBsrCliLib.RouterPimBsrIpv6Mode.addCommandClass( Ipv6CandidateCmd )


#--------------------------------------------------------------------------------
# Legacy:
# (router-router-pim-bsr[-vrf])#
# ip pim bsr-candidate INTF [ { ( ( ( ADDR MASK ) |
#                                   ( ADDRPREFIX ) |
#                                   ( access-list ACL ) ) |
#                                 ( priority PRIORITY ) |
#                                 ( hashmask HASHMASK ) |
#                                 ( interval INTERVAL ) ) } ]
# ( no | default ) ip pim bsr-candidate candidate [ INTF ]
#                            [ ( ( ADDR MASK ) |
#                                ( ADDRPREFIX ) |
#                                ( access-list ACL ) ) ]
#                            [ ( priority | hashmask | interval ) ]
#--------------------------------------------------------------------------------
class IpPimBsrCandidateLegacyCmd( CliCommand.CliCommandClass ):
   syntax = 'ip pim bsr-candidate INTF ' \
            '[ { ( GROUP_RANGE_OR_ACL ) | CANDIDATE_OPTIONS } ]'
   noOrDefaultSyntax = 'ip pim bsr-candidate [ INTF ] ' \
                       '[ GROUP_RANGE_OR_ACL ] [ NO_CANDIDATE_OPTIONS ] ...'
   data = {
      'ip' : ipMatcherForConfig,
      'pim' : pimNode,
      'bsr-candidate' : 'Become a candidate bootstrap router',
      'INTF' : Intf.matcherWithIpSupport,
      'GROUP_RANGE_OR_ACL' : Ipv4GroupRangeOrAclExpr,
      'CANDIDATE_OPTIONS' : Ipv4CandidateOptionsExpr,
      'NO_CANDIDATE_OPTIONS' : NoCandidateOptionsExpr
   }
   handler = PimBsrConfigCli.setPimBsrCandidate
   noOrDefaultHandler = PimBsrConfigCli.noPimBsrCandidate

PimBsrCliLib.RouterPimBsrSharedModelet.addCommandClass( IpPimBsrCandidateLegacyCmd )
BasicCli.GlobalConfigMode.addCommandClass( IpPimBsrCandidateLegacyCmd )

#--------------------------------------------------------------------------------
# [ no | default ] vrf VRF
#--------------------------------------------------------------------------------
class VrfCmd( CliCommand.CliCommandClass ):
   syntax = 'vrf VRF'
   noOrDefaultSyntax = 'vrf VRF ...'
   data = {
      'vrf': 'Configure PIM BSR in a VRF',
      'VRF': CliMatcher.DynamicNameMatcher( VrfCli.getVrfNames, 'VRF name' ),
   }
   handler = PimBsrConfigCli.gotoRouterPimBsrVrfMode
   noOrDefaultHandler = PimBsrConfigCli.deleteRouterPimBsrVrfMode

PimBsrCliLib.RouterPimBsrMode.addCommandClass( VrfCmd )

#----------------------------------------------------------------------
# (config-if)#
# [ no | default ] pim bsr ipv4 border [ ( ( ADDR MASK ) |
#                                          ( ADDRPREFIX ) |
#                                          ( access-list ACL ) ) ]
#----------------------------------------------------------------------
class Ipv4PimBsrBorderCmd( CliCommand.CliCommandClass ):
   syntax = 'pim bsr ipv4 border [ GROUP_RANGE_OR_ACL ]'
   noOrDefaultSyntax = syntax
   data = {
         'pim' : pimIntfNode,
         'bsr' : bsrMatcher,
         'ipv4' : ipv4Matcher,
         'border' : borderMatcher,
         'GROUP_RANGE_OR_ACL' : Ipv4GroupRangeOrAclExpr
   }
   handler = PimBsrConfigCli.setPimBsrBorder
   noOrDefaultHandler = PimBsrConfigCli.noPimBsrBorder

RoutingProtocolIntfConfigModelet.addCommandClass( Ipv4PimBsrBorderCmd )

#----------------------------------------------------------------------
# (config-if)#
# [ no | default ] pim bsr ipv6 border [ ( ( ADDR MASK ) |
#                                          ( ADDRPREFIX ) |
#                                          ( access-list ACL ) ) ]
#----------------------------------------------------------------------
class Ipv6PimBsrBorderCmd( CliCommand.CliCommandClass ):
   syntax = 'pim bsr ipv6 border [ GROUP_RANGE_OR_ACL ]'
   noOrDefaultSyntax = syntax
   data = {
         'pim' : pimIntfNode,
         'bsr' : bsrMatcher,
         'ipv6' : ipv6Matcher,
         'border' : borderMatcher,
         'GROUP_RANGE_OR_ACL' : Ipv6GroupRangeOrAclExpr
   }
   handler = PimBsrConfigCli.setPimBsrBorder
   noOrDefaultHandler = PimBsrConfigCli.noPimBsrBorder

RoutingProtocolIntfConfigModelet.addCommandClass( Ipv6PimBsrBorderCmd )

#----------------------------------------------------------------------
# Legacy
# (config-if)#
# [ no | default ] ip pim bsr-border [ ( ( ADDR MASK ) |
#                                        ( ADDRPREFIX ) |
#                                        ( access-list ACL ) ) ]
#----------------------------------------------------------------------
class IpPimBsrBorderLegacyCmd( CliCommand.CliCommandClass ):
   syntax = 'ip pim bsr-border [ GROUP_RANGE_OR_ACL ]'
   noOrDefaultSyntax = syntax
   data = {
         'ip' : ipMatcherForConfigIf,
         'pim' : pimIntfNode,
         'bsr-border' : bsrBorderNodeDeprecated,
         'GROUP_RANGE_OR_ACL' : Ipv4GroupRangeOrAclExpr
   }
   handler = PimBsrConfigCli.setPimBsrBorder
   noOrDefaultHandler = PimBsrConfigCli.noPimBsrBorder

RoutingProtocolIntfConfigModelet.addCommandClass( IpPimBsrBorderLegacyCmd )

#------------------------------------------------------------------------------------
# (router-pim-bsr[-vrf]-[ipv4|ipv6])
# [no | default ] rp-candidate advertisement-filter access-list ACL
#------------------------------------------------------------------------------------
class RpCandidateAdsFilterCmd( CliCommand.CliCommandClass ):
   syntax = "rp-candidate advertisement-filter access-list ACL"
   noOrDefaultSyntax = "rp-candidate advertisement-filter [ access-list ACL ] ..."
   data = {
         'rp-candidate' : rpCandidateMatcher,
         'advertisement-filter' : advertisementFilterMatcher,
         'access-list' : accessListMatcher,
         'ACL' : aclNameMatcher,
   }
   handler = PimBsrConfigCli.setIpPimBsrRpCandidateAdsFilter
   noOrDefaultHandler = PimBsrConfigCli.noIpPimBsrRpCandidateAdsFilter

PimBsrCliLib.RouterPimBsrAfCommonSharedModelet.addCommandClass(
      RpCandidateAdsFilterCmd )
     
#------------------------------------------------------------------------------------
# legacy: 
# switch( config-router-pim-bsr-*)#
# [ no ] ip pim bsr rp-candidate advertisement-filter access-list ACL
#------------------------------------------------------------------------------------
class IpPimBsrRpCandidateAdsFilterLegacyCmd( CliCommand.CliCommandClass ):
   syntax = "ip pim bsr rp-candidate advertisement-filter access-list ACL"
   noOrDefaultSyntax = "ip pim bsr rp-candidate advertisement-filter " \
                       "[ access-list ACL ] ..."
   data = {
         'ip' : ipMatcherForConfig,
         'pim' : pimNode,
         'bsr' : bsrMatcher,
         'rp-candidate' : rpCandidateMatcher,
         'advertisement-filter' : advertisementFilterMatcher,
         'access-list' : accessListMatcher,
         'ACL' : aclNameMatcher,
   }
   handler = PimBsrConfigCli.setIpPimBsrRpCandidateAdsFilter
   noOrDefaultHandler = PimBsrConfigCli.noIpPimBsrRpCandidateAdsFilter

PimBsrCliLib.RouterPimBsrSharedModelet.addCommandClass(
      IpPimBsrRpCandidateAdsFilterLegacyCmd )
BasicCli.GlobalConfigMode.addCommandClass( IpPimBsrRpCandidateAdsFilterLegacyCmd )
