#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import BasicCli
import CliCommand, CliMatcher, Tac
from CliToken.Ip import ipMatcherForConfig
from CliToken.Pim import pimNode, rpMatcher, aclNameMatcher
from CliPlugin import PimCliLib
from CliPlugin.IntfCli import Intf
from CliPlugin.Ip6AddrMatcher import Ip6AddrMatcher
from CliPlugin.IpAddrMatcher import IpAddrMatcher
from CliPlugin.IraCommonCli import AddressFamily
from CliPlugin.PimCliLib import RouterPimSparseSharedModelet, \
      RouterPimSparseIpv4Modelet, RouterPimSparseIpv6Modelet, \
      RouterPimSparseAfCommonSharedModelet
from CliPlugin.RouterMulticastCliLib import doConfigMounts, configGetters, \
      getAddressFamilyFromMode, legacyCliCallback
from IpLibConsts import DEFAULT_VRF

# Af Independent Config Types
PimRegConfigColl = "Routing::PimReg::ConfigColl"

( pimRegConfigColl,
  pimRegConfigCollFromMode,
  pimRegConfigVrf,
  pimRegConfigVrfFromMode ) = configGetters( PimRegConfigColl )

PimRegClearConfigColl = "Routing::PimReg::ClearConfigColl"

( pimRegClearConfigColl,
  pimRegClearConfigCollFromMode,
  pimRegClearConfigVrf,
  pimRegClearConfigVrfFromMode) = configGetters( PimRegClearConfigColl,
        collectionName='vrfClearConfig' )

def getVrfName( mode ):
   if hasattr( mode, 'vrfName' ):
      return mode.vrfName
   return DEFAULT_VRF

def getPimRegVrfConfig( vrfName, af ):
   pimRegVrfConfig = None
   if vrfName in pimRegConfigColl( af ).vrfConfig:
      pimRegVrfConfig = pimRegConfigColl( af ).vrfConfig[ vrfName ]
   return pimRegVrfConfig

def _pimRegVrfDefinitionHook( vrfName ):
   for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      pimRegConfigVrf( af, vrfName=vrfName )
      configRoot = pimRegConfigColl( af )
      assert vrfName in configRoot.vrfConfig
      pimRegClearConfigVrf( af, vrfName=vrfName )
      clearConfigRoot = pimRegClearConfigColl( af )
      assert vrfName in clearConfigRoot.vrfClearConfig

def _pimRegVrfDeletionHook( vrfName ):
   for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      if vrfName in pimRegConfigColl( af ).vrfConfig:
         config = getPimRegVrfConfig( vrfName, af )
         if vrfName != DEFAULT_VRF and config.isDefault:
            # only delete if there is no non-default config
            # and the VRF is not defined
            del pimRegConfigColl( af ).vrfConfig[ vrfName ]

      if vrfName in pimRegClearConfigColl( af ).vrfClearConfig and \
            vrfName != DEFAULT_VRF:
         del pimRegClearConfigColl( af ).vrfClearConfig[ vrfName ]

def canDeletePimRegVrfConfig( vrfName ):
   canDelete = True
   for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      if vrfName in pimRegConfigColl( af ).vrfConfig:
         config = pimRegConfigColl( af ).vrfConfig[ vrfName ]
         if config.registerSrc != config.registerSrcDefault or \
               config.registerSrcIntf != config.registerSrcIntfDefault or \
               config.registerSuppressionTimeout != \
               config.registerSuppressionTimeoutDefault or \
               config.probePeriod != config.probePeriodDefault or \
               config.registerDrFilter != config.registerFilterDefault or \
               config.registerSrcFilter != config.registerFilterDefault or \
               config.anycastRpConfig:
            canDelete = False
   return canDelete

def clearPimMessageCounters( vrfName, af=AddressFamily.ipv4 ):
   if vrfName in pimRegClearConfigColl( af ).vrfClearConfig:
      pimRegClearConfigVrf( af, vrfName ).countersCount += 1

def cleanupPimRegConfig( vrfName ):
   for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      if vrfName in pimRegConfigColl( af ).vrfConfig:
         config = getPimRegVrfConfig( vrfName, af )
         config.anycastRpConfig.clear()
         config.registerSrc = config.registerSrcDefault
         config.registerSrcIntf = config.registerSrcIntfDefault
         config.registerDrFilter = config.registerFilterDefault
         config.registerSrcFilter = config.registerFilterDefault

#------------------------------------------------------------------------------------
# legacy: switch(config)# ip pim register-source <interface-name>
# switch(config)# register local-interface <interface-name>
#------------------------------------------------------------------------------------

registerSourceKwMatcher = CliMatcher.KeywordMatcher(
   'register-source',
   helpdesc='Specify the source interface to use for PIM register messages' )
registerSourceNode = CliCommand.Node(
   matcher=registerSourceKwMatcher,
   deprecatedByCmd='register local-interface in ipv4 submode' )

iifKwMatcher = CliMatcher.KeywordMatcher(
   'iif',
   helpdesc='Use the IIF address of the (S,G) entry' )

registerKwMatcher = CliMatcher.KeywordMatcher(
   'register',
   helpdesc='PIM register commands' )

localInterfaceKwMatcher = CliMatcher.KeywordMatcher(
   'local-interface',
   helpdesc='Specify the local interface to use for PIM register messages' )

def registerSrcIntf( mode, args, legacy=False ):
   intf = args.get( 'INTF' )
   iif = args.get( 'iif' )
   vrf = getVrfName( mode )
   registerSrcEnum = Tac.Type( "Routing::PimReg::RegisterSrc" )
   config = getPimRegVrfConfig( vrf,
         getAddressFamilyFromMode( mode, legacy=legacy ) )
   if config is None:
      return
   if iif:
      config.registerSrc = registerSrcEnum.iif
      config.registerSrcIntf = config.registerSrcIntfDefault
   else:
      config.registerSrc = registerSrcEnum.interface
      config.registerSrcIntf = intf.name

def noRegisterSrcIntf( mode, args, legacy=False ):
   vrf = getVrfName( mode )
   config = getPimRegVrfConfig( vrf,
         getAddressFamilyFromMode( mode, legacy=legacy ) )
   if config is None:
      return
   config.registerSrc = config.registerSrcDefault
   config.registerSrcIntf = config.registerSrcIntfDefault

registerSrcIntfLegacy = legacyCliCallback( registerSrcIntf )
noRegisterSrcIntfLegacy = legacyCliCallback( noRegisterSrcIntf )

# legacy commands
class IpPimRegisterSourceCmdBase( CliCommand.CliCommandClass ):
   syntax = 'ip pim register-source ( iif | INTF )'
   noOrDefaultSyntax = 'ip pim register-source ...'
   data = {
      'pim': pimNode,
      'register-source': registerSourceNode,
      'iif': iifKwMatcher,
      'INTF': Intf.matcherWithIpSupport
   }
   handler = registerSrcIntfLegacy
   noOrDefaultHandler = noRegisterSrcIntfLegacy

class IpPimRegisterSourceCmd( IpPimRegisterSourceCmdBase ):
   data = {
      'ip': ipMatcherForConfig,
   }
   data.update( IpPimRegisterSourceCmdBase.data )

BasicCli.GlobalConfigMode.addCommandClass( IpPimRegisterSourceCmd )

class IpPimRegisterSourceIfCmd( IpPimRegisterSourceCmdBase ):
   data = {
      'ip': ipMatcherForConfig,
   }
   data.update( IpPimRegisterSourceCmdBase.data )

RouterPimSparseSharedModelet.addCommandClass( IpPimRegisterSourceIfCmd )

# new commands
class RegisterLocalInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'register local-interface ( iif | INTF )'
   noOrDefaultSyntax = 'register local-interface ...'
   data = {
      'register': registerKwMatcher,
      'local-interface': localInterfaceKwMatcher,
      'iif': iifKwMatcher,
      'INTF': Intf.matcherWithIpSupport
   }
   handler = registerSrcIntf
   noOrDefaultHandler = noRegisterSrcIntf

RouterPimSparseAfCommonSharedModelet.addCommandClass( RegisterLocalInterfaceCmd )

#------------------------------------------------------------------------------------
# Legacy:
# switch(config)# ip pim anycast-rp <anycast-rp-address> <other-rp-address>
#                                      [ register-count <regcount> ]
#------------------------------------------------------------------------------------
# switch(config)# anycast-rp <anycast-rp-address> <other-rp-address>
#                                      [ register-count <regcount> ]
#------------------------------------------------------------------------------------

anycastRpKwMatcher = CliMatcher.KeywordMatcher(
   'anycast-rp',
   helpdesc='Anycast rendezvous point' )
anycastRpDeprecatedNode = CliCommand.Node(
   matcher=anycastRpKwMatcher,
   deprecatedByCmd='anycast-rp in ipv4 submode' )

anycastRpAddrMatcher = IpAddrMatcher(
   helpdesc='anycast rendezvous point IP address' )
anycastRpAddr6Matcher = Ip6AddrMatcher(
   helpdesc='anycast rendezvous point IPv6 address' )
otherRpAddrMatcher = IpAddrMatcher(
   helpdesc='other anycast rendezvous point IP address' )
otherRpAddr6Matcher = Ip6AddrMatcher(
   helpdesc='other anycast rendezvous point IPv6 address' )

registerCountKwMatcher = CliMatcher.KeywordMatcher(
   'register-count',
   helpdesc='Number of registers to forward before pausing' )

infinityKwMatcher = CliMatcher.KeywordMatcher(
   'infinity',
   helpdesc='Forward all registers' )

registerCountMatcher = CliMatcher.IntegerMatcher(
   0, 2**32 - 1,
   helpdesc='Number of registers to forward' )

def setAnycastRpAddress( mode, args, legacy=False ):
   vrf = getVrfName( mode )
   config = getPimRegVrfConfig( vrf,
         getAddressFamilyFromMode( mode, legacy=legacy ) )
   if config is None:
      return

   anycastRp = args.get( 'ANYCAST_RP' )
   anycastRpAddr = Arnet.IpGenAddr( str( anycastRp ) )
   otherRp = args.get( 'OTHER_RP' )
   otherRpAddr = Arnet.IpGenAddr( str( otherRp ) )
   arp = config.anycastRpConfig.newMember( anycastRpAddr )

   infinity = args.get( 'infinity' )
   registerCount = args.get( 'REG_COUNT' )
   if infinity:
      registerCount = 0xFFFFFFFF
   elif registerCount is None:
      registerCount = config.registerCountDefault

   arp.peer.newMember( otherRpAddr, registerCount )

def noAnycastRpAddress( mode, args, legacy=False ):
   vrf = getVrfName( mode )
   if vrf in pimRegConfigCollFromMode( mode, legacy=legacy ).vrfConfig:
      anycastRp = args.get( 'ANYCAST_RP' )
      anycastRpAddr = Arnet.IpGenAddr( str( anycastRp ) )
      config = getPimRegVrfConfig( vrf, getAddressFamilyFromMode( mode,
                                                                  legacy=legacy ) )
      arp = config.anycastRpConfig.get( anycastRpAddr )
      if arp:
         otherRp = args.get( 'OTHER_RP' )
         if otherRp:
            otherRpAddr = Arnet.IpGenAddr( str( otherRp ) )
            del arp.peer[ otherRpAddr ]
         if not arp.peer or not otherRp:
            del config.anycastRpConfig[ anycastRpAddr ]

setAnycastRpAddressLegacy = legacyCliCallback( setAnycastRpAddress )
noAnycastRpAddressLegacy = legacyCliCallback( noAnycastRpAddress )

# legacy commands
class IpPimAnycastRpCmdBase( CliCommand.CliCommandClass ):
   syntax = ( 'ip pim anycast-rp ANYCAST_RP OTHER_RP '
      '[ register-count ( infinity | REG_COUNT ) ]' )
   noOrDefaultSyntax = 'ip pim anycast-rp ANYCAST_RP [ OTHER_RP ] ...'
   data = {
      'pim': pimNode,
      'anycast-rp': anycastRpDeprecatedNode,
      'ANYCAST_RP': anycastRpAddrMatcher,
      'OTHER_RP': otherRpAddrMatcher,
      'register-count': registerCountKwMatcher,
      'infinity': infinityKwMatcher,
      'REG_COUNT': registerCountMatcher
   }
   handler = setAnycastRpAddressLegacy
   noOrDefaultHandler = noAnycastRpAddressLegacy

class IpPimAnycastRpCmd( IpPimAnycastRpCmdBase ):
   data = {
      'ip': ipMatcherForConfig,
   }
   data.update( IpPimAnycastRpCmdBase.data )

BasicCli.GlobalConfigMode.addCommandClass( IpPimAnycastRpCmd )

class IpPimAnycastRpIfCmd( IpPimAnycastRpCmdBase ):
   data = {
      'ip': ipMatcherForConfig,
   }
   data.update( IpPimAnycastRpCmdBase.data )

RouterPimSparseSharedModelet.addCommandClass( IpPimAnycastRpIfCmd )

# new commands
class AnycastRpCmdBase( CliCommand.CliCommandClass ):
   syntax = ( 'anycast-rp ANYCAST_RP OTHER_RP '
      '[ register-count ( infinity | REG_COUNT ) ]' )
   noOrDefaultSyntax = 'anycast-rp ANYCAST_RP [ OTHER_RP ] ...'
   data = {
      'anycast-rp': anycastRpKwMatcher,
      'register-count': registerCountKwMatcher,
      'infinity': infinityKwMatcher,
      'REG_COUNT': registerCountMatcher
   }
   handler = setAnycastRpAddress
   noOrDefaultHandler = noAnycastRpAddress

class AnycastRpCmd( AnycastRpCmdBase ):
   data = {
      'ANYCAST_RP': anycastRpAddrMatcher,
      'OTHER_RP': otherRpAddrMatcher
   }
   data.update( AnycastRpCmdBase.data )

RouterPimSparseIpv4Modelet.addCommandClass( AnycastRpCmd )

class AnycastRp6Cmd( AnycastRpCmdBase ):
   data = {
      'ANYCAST_RP': anycastRpAddr6Matcher,
      'OTHER_RP': otherRpAddr6Matcher
   }
   data.update( AnycastRpCmdBase.data )

RouterPimSparseIpv6Modelet.addCommandClass( AnycastRp6Cmd )


#-------------------------------------------------------------------------------
# (config-router-pim-sparse-*ipv?)
# [no|default] rp register dr filter access-list <ACL_NAME>
#-------------------------------------------------------------------------------

def setIpPimRpDRFilter( mode, args ):

   config = pimRegConfigVrfFromMode( mode, legacy=False )
   if CliCommand.isNoOrDefaultCmd( args ):
      config.registerDrFilter = config.registerFilterDefault
      return

   acl = args.get( 'ACL_NAME', config.registerFilterDefault )

   config.registerDrFilter = acl
   return

class RouterPimSparseRpRegisterDrFilterCmd( CliCommand.CliCommandClass ):
   syntax = '''rp register dr filter access-list ACL_NAME'''
   noOrDefaultSyntax = 'rp register dr filter ...'
   data = {
      'rp' : rpMatcher,
      'register' : 'Pim Register packet',
      'dr': 'DR source specified by access list',
      'filter': 'Filtering based on DR source',
      'access-list': 'Access List',
      'ACL_NAME': aclNameMatcher,
   }
   handler = setIpPimRpDRFilter
   noOrDefaultHandler = handler

RouterPimSparseAfCommonSharedModelet.addCommandClass(
   RouterPimSparseRpRegisterDrFilterCmd )

# -------------------------------------------------------------------------------
# (config-router-pim-sparse-*ipv?)
# [no|default] rp register source filter access-list <ACL_NAME>
# -------------------------------------------------------------------------------

def setIpPimRpSourceFilter( mode, args ):

   config = pimRegConfigVrfFromMode( mode, legacy=False )
   if CliCommand.isNoOrDefaultCmd( args ):
      config.registerSrcFilter = config.registerFilterDefault
      return

   acl = args.get( 'ACL_NAME', config.registerFilterDefault )

   config.registerSrcFilter = acl
   return

class RouterPimSparseRpRegisterSourceFilterCmd( CliCommand.CliCommandClass ):
   syntax = '''rp register source filter access-list ACL_NAME'''
   noOrDefaultSyntax = 'rp register source filter ...'
   data = {
      'rp': rpMatcher,
      'register': 'Pim Register packet',
      'source': 'Multicast source specified by access list',
      'filter': 'Filtering based on multicast source',
      'access-list': 'Access List',
      'ACL_NAME': aclNameMatcher,
   }
   handler = setIpPimRpSourceFilter
   noOrDefaultHandler = handler

RouterPimSparseAfCommonSharedModelet.addCommandClass(
   RouterPimSparseRpRegisterSourceFilterCmd )

def Plugin( entityManager ):
   configTypes = [ PimRegConfigColl, PimRegClearConfigColl ]
   doConfigMounts( entityManager, configTypes )

   PimCliLib.pimClearMessageCountersHook.addExtension( clearPimMessageCounters )
   PimCliLib.pimSparseModeVrfConfiguredHook.addExtension( _pimRegVrfDefinitionHook )
   PimCliLib.pimSparseModeVrfDeletedHook.addExtension( _pimRegVrfDeletionHook )
   PimCliLib.canDeletePimSparseModeVrfHook.addExtension( canDeletePimRegVrfConfig )
   PimCliLib.pimSparseModeCleanupHook.addExtension( cleanupPimRegConfig )
