#!/usr/bin/env python3
# Copyright (c) 2007-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

#-------------------------------------------------------------------------------
# This module implements Pimsm show commands
#
# (legacy) show ip mroute sparse-mode [ <sourceOrGroup> ] [ <sourceOrGroup> ]
#                                     [ detail ]
# (legacy) show ip mroute sparse-mode [ vrf <vrf-name> ] <sourceOrGroup>
#                                     <sourceOrGroup> interface [ <interface> ]
#                                     [ detail ]
#
# show pim { ipv4 | ipv6 } [ vrf < vrf-name > ] sparse-mode route
#                          [ < sourceOrGroup > ] [ < sourceOrGroup > ] [ detail ]
#
# show pim { ipv4 | ipv6 } [ vrf < vrf-name > ] sparse-mode route <sourceOrGroup>
#                          <sourceOrGroup> interface <interface> [ detail ]
#
# (legacy) show ip mroute sparse-mode [ vrf < vrf-name > ] count
# show pim { ipv4 | ipv6 } [ vrf < vrf-name > ] sparse-mode route count
#
# (legacy) show ip pim upstream joins [ <sourceOrGroup> ] [ <sourceOrGroup> ]
#                                     [ neighbor <address> ]
# show pim { ipv4 | ipv6 } [ vrf <vrf-name> ] sparse-mode join [ <sourceOrGroup> ]
#                          [ <sourceOrGroup> ] [ neighbor <address> ]
#
# (legacy) show ip pim sparse-mode rp [ <groupPrefix> ] [ detail ]
# show pim { ipv4 | ipv6 } sparse-mode rp [ <groupPrefix> ] [ detail ]
#
# (legacy) show ip pim sparse-mode rp-hash <group> [ detail ]
# show pim { ipv4 | ipv6 } sparse-mode rp hash <group> [ detail ]
#-------------------------------------------------------------------------------
'''Show commands supported for Sparse Mode Pim'''

import sys

import Arnet
import CliPlugin.TechSupportCli
from CliPlugin import PimCliLib
from CliPlugin import PimConfigCheckModel
from CliPlugin.IraIp6RouteCliLib import isValidIpv6PrefixWithError
from CliPlugin.IraIpRouteCliLib import isValidPrefixWithError
from CliPlugin.McastCommonCli import mcastv6RoutingSupportedGuard
from CliPlugin.PimCliLib import pimsmConfigCheck, igmpConfigCheck, validVrfName
from CliPlugin.PimCliLib import getUribStatus
from CliPlugin import VrfCli
from CliPlugin.GmpShowTechCli import registerShowTechMulticast, ShowTechMulticastType
from CliPlugin.MrouteCli import mcastFibIpv4SupportedGuard
import CliGlobal
import SharkLazyMount

import LazyMount
import PimLib
from PimLib import getPath
import PimModel
import PimModelLib
import PimSmashHelper
import PimsmModel
import PimsmSmashHelper
import SmashLazyMount
import Tac
import Tracing
from IpLibConsts import DEFAULT_VRF as vrfDefault
from McastCommonCliLib import mcastRoutingSupported
from TypeFuture import TacLazyType
import Smash
import Cell
import Toggles.GmpToggleLib

__defaultTraceHandle__ = Tracing.Handle( 'PimsmCli' )
t0 = Tracing.trace0

RpCandidateStatusColl = TacLazyType( 'Routing::Pim::RpCandidateStatusColl' )
SmashStatus = TacLazyType( 'Routing::Pim::Smash::Status' )

_pimsmStatusColl = CliGlobal.CliGlobal( pimsmStatusColl={} )
_pimsmStatusReaderSmColl = {}
_pimsmStatusRestartSmColl = {}
_msdpDiscoveredSgColl = {}

for afi in [ "ipv4", "ipv6" ]:
   if not _pimsmStatusColl.pimsmStatusColl.get( afi ):
      _pimsmStatusColl.pimsmStatusColl[ afi ] = {}
   _pimsmStatusReaderSmColl[ afi ] = {}
   _pimsmStatusRestartSmColl[ afi ] = {}

_pimStatusColl = {}
_pimGlobalStatus = {}

_entityManager = None

_groupSourceMapColl = {}
_joinPruneStatusColl = {}
_staticRpColl = {}
_rpConfigColl = {}
_registerConfigColl = {}
_registerStatusColl = None
_pimBsrCrpStatusColl = {}
_pimsmRpHash = None
_pimsm6Status = None
_pim6Status = None
_staticRpComputeObjColl = {}
_staticTrieBuilderColl = {}
_bsrRpComputeObjColl = {}
_bsrTrieBuilderColl = {}
_ipConfig = None
_pimConfigRoot = None
_staticRpConfigColl = {}
_sourceNumSm = {}
_srcCollDict = {}
_grpCollDict = {}
_mcastBoundaryConfig = None
_aclConfigCli = None
_routingVrfInfoDir = {}
_fastFailoverStatusColl = {}
_mfibVrfConfig = {}
_l3IntfStatusDir = None
_enabledStatus = None
_pegDrStatus4 = None
_pegDrStatus6 = None
_ipTunnelVlanGroupStatuses = {}

_ipTunnelStatusTypes = \
   Tac.Type( "Routing::Multicast::IpTunnelGroupStatusType" ).attributes

_mvpnIntfStatus = None
_vrfNameStatus = None
_l3Config = None
_fecModeSm = None
_fecModeStatus = None

afMounts = {
   "registerStatusColl": None
}
cliGlob = {
   "ipv4": CliGlobal.CliGlobal( dict( afMounts ) ),
   "ipv6": CliGlobal.CliGlobal( dict( afMounts ) ),
}

# Add to message counters extension
def getPimMessageCounters( vrfName, af="ipv4", **kwargs ):
   pimGlobalStatus = _pimGlobalStatus[ af ]
   if vrfName in pimGlobalStatus.pimEnabledSparseModeVrf:
      path = getPath( 'Routing::Pim::Smash::MessageCounters', af, vrfName,
                      "sparsemode" )
      return SmashLazyMount.mount( _entityManager, path,
                                   'Routing::Pim::Smash::MessageCounters',
                                   SmashLazyMount.mountInfo( 'reader' ) )
   return None

def getPimStatus( vrfName, af="ipv4" ):
   pimGlobalStatus = _pimGlobalStatus[ af ]
   pimStatusColl = _pimStatusColl[ af ]
   if ( vrfName in pimGlobalStatus.pimEnabledSparseModeVrf and
        vrfName in pimStatusColl.vrfStatus ):
      return pimStatusColl.vrfStatus[ vrfName ]
   return None

def getPimSmashStatus( vrfName, af="ipv4" ):
   pimGlobalStatus = _pimGlobalStatus[ af ]
   if vrfName in pimGlobalStatus.pimEnabledSparseModeVrf:
      path = getPath( SmashStatus.tacTypeName, af, vrfName )
      return PimSmashHelper.mountInDependencyOrder( _entityManager, path,
             'keyshadow', SmashStatus )
   return None

def waitForPimStatusSms( mode, af="ipv4" ):
   try:
      readerSm = PimLib.getPimGlobalStatusReaderSm( _entityManager, af )
      description = ' Pim table to be populated'
      Tac.waitFor( lambda: readerSm and readerSm.initialized, timeout=10,
                   warnAfter=None, sleep=not Tac.activityManager.inExecTime.isZero,
                   maxDelay=0.1, description=description )
   except Tac.Timeout:
      mode.addError( "Command timed out" )
      return False

   return True

def getPimUpstreamJoins( vrfName, af="ipv4" ):
   status = _joinPruneStatusColl[ af ].vrfStatus.get( vrfName )
   return status

def doMaybeInitPimsmStatusSms( vrfName, af ):

   pimsmStatusColl = _pimsmStatusColl.pimsmStatusColl[ af ]

   if vrfName not in pimsmStatusColl:
      pimsmStatusColl[ vrfName ] = Tac.newInstance(
         "Routing::Pim::SparseMode::Status", "status-%s" % vrfName )

   pimsmStatus = pimsmStatusColl[ vrfName ]

   path = getPath( "Routing::Pim::SparseMode::Smash::Status", af, vrfName )
   pimsmSmashStatus = PimsmSmashHelper.mountInDependencyOrder(
      _entityManager, path, 'keyshadow' )

   pimStatus = _pimStatusColl[ af ].vrfStatus.get( vrfName )

   pimsmStatus.pimStatus = pimStatus

   readerSmColl = _pimsmStatusReaderSmColl[ af ]
   if not readerSmColl.get( vrfName ):
      readerSmColl[ vrfName ] = Tac.newInstance(
         "Routing::Pim::SparseMode::Smash::RouteSmashReaderSm",
         pimsmStatus, pimsmSmashStatus )

   restartSmColl = _pimsmStatusRestartSmColl[ af ]
   if not restartSmColl.get( vrfName ):
      restartSmColl[ vrfName ] = Tac.newInstance(
         "Routing::Pim::SparseMode::Smash::RouteSmashRestartSm",
         pimsmStatus, pimsmSmashStatus, 2, Tac.activityManager.clock )

   # Add some defensive code for multi-threaded issues as seen in
   # SR 573175, BUG592746 and BUG991515
   restartSm = restartSmColl[ vrfName ]
   readerSm = readerSmColl[ vrfName ]
   if pimsmStatus != restartSm.status or pimsmStatus != readerSm.status:
      del pimsmStatusColl[ vrfName ]
      del readerSmColl[ vrfName ]
      del restartSmColl[ vrfName ]
      # This method is called again from Tac.waitFor and
      # the above buggy condition will then get fixed.
      return False

   return pimsmStatus.initialized

def waitForPimsmStatusSms( mode, vrfName, af ):
   try:
      Tac.waitFor( lambda: doMaybeInitPimsmStatusSms( vrfName, af ),
                   timeout=60, warnAfter=None,
                   sleep=not Tac.activityManager.inExecTime.isZero, maxDelay=0.1,
                   description=' mroute table to be created' )
   except Tac.Timeout:
      mode.addError( "Command timed out" )
      return False

   return True

#------------------------------------------------------------------------------------
# show ip mroute sparse-mode [ <sourceOrGroup> ] [ <sourceOrGroup> ] [ detail ]
# show pim { ipv4 | ipv6 } [ vrf < vrf name > ] sparse-mode route
#                          [ <sourceOrGroup> ] [ <sourceOrGroup> ] [ detail ]
#------------------------------------------------------------------------------------
def getIpMrouteSparseMode( mode, vrfName=vrfDefault, source=None,
                           group=None, detail=False, submodel=False, af="ipv4" ):

   if not validVrfName( mode, vrfName, _mfibVrfConfig[ af ], af ):
      return None, None
   if not waitForPimStatusSms( mode, af ):
      return None, None
   if not waitForPimsmStatusSms( mode, vrfName, af ):
      return None, None

   path = getPath( "Routing::Pim::RpHashStatus", af, vrfName, "sparsemode" )
   pimsmRpHash = SmashLazyMount.mount( _entityManager, path,
                                       "Routing::Pim::RpHashStatus",
                                       SmashLazyMount.mountInfo( 'reader' ) )

   registerConfig = _registerConfigColl[ af ].vrfConfig.get( vrfName )
   pimsmStatus = _pimsmStatusColl.pimsmStatusColl[ af ][ vrfName ]
   groupSourceMap = _groupSourceMapColl[ af ].vrfGroupSourceMap.get( vrfName )

   if not groupSourceMap:
      return None, None

   if not source:
      addr = PimCliLib.defaultAddr( af )
      srcAddr = Tac.Value( "PimsmCliCapi::SourceAdd", addr, True )
   else:
      srcAddr = Tac.Value( "PimsmCliCapi::SourceAdd", source, False )

   if not group:
      addr = PimCliLib.defaultAddr( af )
      grpAddr = Tac.Value( "PimsmCliCapi::SourceAdd", addr, True )
   else:
      grpAddr = Tac.Value( "PimsmCliCapi::SourceAdd", group, False )

   showDetail = bool( detail )

   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()

   pmsiIntfId = Arnet.IntfId( "" )
   if _vrfNameStatus and _vrfNameStatus.vrfIdExists( vrfName ) :
      vrfId = _vrfNameStatus.nameToIdMap.vrfNameToId[ vrfName ]
      if _mvpnIntfStatus.vrfIntfId.get( vrfId ):
         pmsiIntfId = _mvpnIntfStatus.vrfIntfId[ vrfId ].pmsiIntfId

   if af == "ipv4":
      LazyMount.force( _l3IntfStatusDir )
      LazyMount.force( _enabledStatus )
      LazyMount.force( _ipTunnelVlanGroupStatuses[ af ][ "irbDecapStatus" ] )
      _pegDrStatus4.mount_()
      showMroute = Tac.newInstance( "PimsmCliCapi::ShowMroute", pimsmStatus,
                                    pimsmRpHash, registerConfig, groupSourceMap,
                                    _enabledStatus, _pegDrStatus4, _l3IntfStatusDir,
                            _ipTunnelVlanGroupStatuses[ af ][ "irbDecapStatus" ],
                            pmsiIntfId )
   else:
      _pegDrStatus6.mount_()
      showMroute = Tac.newInstance( "PimsmCliCapi::ShowMroute", pimsmStatus,
                                    pimsmRpHash, registerConfig, groupSourceMap,
                                    None, _pegDrStatus6, None, None, pmsiIntfId )
   showMroute.render( fd, fmt, srcAddr, grpAddr, showDetail, submodel )

   return 'modePimSm', PimModel.GroupSms

def showIpMroute( mode, args ):
   '''Handler for CLI commands:
   `show ip mroute sparse-mode [ vrf VRF ] [ IP1 [ IP2 ] ] [ detail ]`
   `show pim ipv4 [ vrf VRF ] sparse-mode route [ IP1 [ IP2 ] ] [ detail ]`
   `show pim ipv6 [ vrf VRF ] sparse-mode route [ IP1 [ IP2 ] ] [ detail ]`
   '''
   vrfName = args.get( 'VRF', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   groupOrSource1 = args.get( 'IP1' )
   if groupOrSource1 is not None:
      groupOrSource1 = str( groupOrSource1 )
   groupOrSource2 = args.get( 'IP2' )
   if groupOrSource2 is not None:
      groupOrSource2 = str( groupOrSource2 )
   detail = 'detail' in args
   af = 'ipv6' if 'ipv6' in args else 'ipv4'

   if not validVrfName( mode, vrfName, _mfibVrfConfig[ af ], af ):
      return None

   try:
      ( source, group ) = PimCliLib.ipPimParseSg( groupOrSource1, groupOrSource2 )
   except ValueError as e:
      mode.addError( "Must enter a multicast group and/or unicast source %s %s, %s"
                     % ( groupOrSource1, groupOrSource2, e ) )
      return None

   _, model = getIpMrouteSparseMode( mode, vrfName, source, group, detail, af=af )
   return model

#---------------------------------------------------------------------
# show ip mroute sparse-mode sources/groups
#---------------------------------------------------------------------
def showIpMrouteSourcesGroups( mode, args ):
   '''Handler for CLI command:
   `show ip mroute sparse-mode [ vrf VRF ] ( sources | groups )`
   '''
   vrfName = args.get( 'VRF', vrfDefault )
   af = "ipv4"

   if not validVrfName( mode, vrfName, _mfibVrfConfig[ af ], af ):
      return None

   if vrfName not in _srcCollDict:
      _srcCollDict[ vrfName ] = Tac.newInstance(
                                            'Routing::Pim::SparseMode::SourceCount',
                                            'srcColl')
   if vrfName not in _grpCollDict :
      _grpCollDict[ vrfName ] = Tac.newInstance(
                                            'Routing::Pim::SparseMode::GroupCount',
                                            'grpColl' )

   groupSourceMapColl = _groupSourceMapColl[ af ]
   pimsmStatusColl = _pimsmStatusColl.pimsmStatusColl[ af ]

   if vrfName not in _sourceNumSm:

      if not pimsmStatusColl.get( vrfName ):
         pimsmStatusColl[ vrfName ] = Tac.newInstance(
            "Routing::Pim::SparseMode::Status",
            "status-%s" % vrfName )

      pimsmStatus = pimsmStatusColl[ vrfName ]
      groupSourceMap = groupSourceMapColl.vrfGroupSourceMap.get( vrfName )
      _sourceNumSm[ vrfName ] = Tac.newInstance(
                                      'Routing::Pim::SparseMode::SgCountSm',
                                      pimsmStatus, _srcCollDict[ vrfName ],
                                      groupSourceMap, _grpCollDict[ vrfName ] )

   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()
   srcGrps = Tac.newInstance( "PimsmCliCapi::SourcesGroups",
                              _srcCollDict[ vrfName ],
                              _grpCollDict[ vrfName ] )
   srcGrps.render( fd, fmt, 'sources' in args )

   return PimsmModel.SourcesGroups

#------------------------------------------------------------------------------------
# show ip mroute sparse-mode [ vrf < vrf-name > ] <sourceOrGroup> <sourceOrGroup>
#                            interface [ <interface> ] [ detail ]
# show pim { ipv4 | ipv6 } [ vrf < vrf-name ] sparse-mode route  <sourceOrGroup>
#                          <sourceOrGroup> interface <interface> [ detail ]
#------------------------------------------------------------------------------------
def getIpMrouteIntfSparseMode( mode , vrfName=vrfDefault, source=None,
                               group=None, intf = None, detail=False,
                               submodel=False, af="ipv4" ):

   if not validVrfName( mode, vrfName, _mfibVrfConfig[ af ], af ):
      return None, None
   if not waitForPimStatusSms( mode, af ):
      return None, None
   if not waitForPimsmStatusSms( mode, vrfName, af ):
      return None, None

   pimsmStatus = _pimsmStatusColl.pimsmStatusColl[ af ][ vrfName ]
   showDetail = bool( detail )
   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()

   if intf:
      interface = str( intf )
   else:
      interface = ""
   if detail and af == "ipv4":
      LazyMount.force( _l3IntfStatusDir )
      LazyMount.force( _enabledStatus )
      LazyMount.force( _ipTunnelVlanGroupStatuses[ af ][ "irbEncapStatus" ] )
      _pegDrStatus4.mount_()
      showIntf = Tac.newInstance( "PimsmCliCapi::ShowMrouteInterface",
                                  pimsmStatus,
                                  _enabledStatus, _pegDrStatus4, _l3IntfStatusDir,
                             _ipTunnelVlanGroupStatuses[ af ][ "irbEncapStatus" ] )
   else:
      _pegDrStatus6.mount_()
      showIntf = Tac.newInstance( "PimsmCliCapi::ShowMrouteInterface",
                                  pimsmStatus, None, _pegDrStatus6, None, None )
   showIntf.render( fd, fmt, source, group, interface, showDetail, submodel )

   return 'modePimSm', PimModel.MrouteSmInterfaces

def showIpMrouteInterface( mode, args ):
   '''Handler for CLI commands:
   `show ip mroute sparse-mode [ vrf VRF ] IP1 IP2 interface [ INTF ] [ detail ]`
   `show pim ipv4 [ vrf VRF ] sparse-mode route IP1 IP2 interface [ INTF ] [ detail]`
   `show pim ipv6 [ vrf VRF ] sparse-mode route IP1 IP2 interface [ INTF ] [ detail]`
   '''
   vrfName = args.get( 'VRF', vrfDefault )
   groupOrSource1 = str( args[ 'IP1' ] )
   groupOrSource2 = str( args[ 'IP2' ] )
   intf = args.get( 'INTF' )
   detail = 'detail' in args
   af = 'ipv6' if 'ipv6' in args else 'ipv4'
   try:
      ( source, group ) = PimCliLib.ipPimParseSg( groupOrSource1, groupOrSource2 )
   except ValueError:
      mode.addError( "Must enter a multicast group and/or unicast source" )
      return None

   _, model = getIpMrouteIntfSparseMode( mode, vrfName, source, group, intf,
                                         detail, af=af )
   return model

#------------------------------------------------------------------------------------
# show ip mroute sparse-mode [ vrf < vrf-name > ] count
# show pim { ipv4 | ipv6 } [ vrf < vrf-name > ] sparse-mode route count
#------------------------------------------------------------------------------------
def getIpMrouteCountSparseMode( mode, vrfName=vrfDefault, submodel=False,
                                af="ipv4" ):

   if not waitForPimStatusSms( mode, af ):
      return None, PimModel.MrouteCount
   if not waitForPimsmStatusSms( mode, vrfName, af ):
      return None, PimModel.MrouteCount

   pimsmStatus = _pimsmStatusColl.pimsmStatusColl[ af ][ vrfName ]

   groupSourceMap = _groupSourceMapColl[ af ].vrfGroupSourceMap.get( vrfName )
   if not groupSourceMap:
      mode.addError( "Multicast group to source mapping unavailable for VRF %s" %
                     vrfName )
      return None, PimModel.MrouteCount

   registerConfig = _registerConfigColl[ af ].vrfConfig.get( vrfName )
   if not registerConfig:
      mode.addError( "PIM register information unavailable for VRF %s" % vrfName )
      return None, PimModel.MrouteCount

   registerStatus = cliGlob[ af ].registerStatusColl.vrfStatus.get(
      vrfName,
      Tac.newInstance( "Routing::Pim::SparseMode::RegisterStatus", "vrf" ) )

   msdpDiscoveredSg = _msdpDiscoveredSgColl[ af ].vrfMsdpDiscoveredSg.get( vrfName )

   path = getPath( "Routing::Pim::SparseMode::Smash::Status", af, vrfName )
   pimsmSmashStatus = PimsmSmashHelper.mountInDependencyOrder( _entityManager,
                                                               path, 'keyshadow' )

   count = Tac.newInstance( "PimsmCliCapi::ShowMrouteCount", pimsmStatus,
                            pimsmSmashStatus, groupSourceMap,
                            registerConfig, registerStatus,
                            msdpDiscoveredSg )

   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()
   count.render( fd, fmt, submodel )

   return 'modePimSm', PimModel.MrouteCount


def showPimRouteCount( mode, args ):
   '''Handler for CLI commands:
   `show ip mroute sparse-mode [ vrf VRF ] count`
   `show pim ipv4 [ vrf VRF ] sparse-mode route count`
   `show pim ipv6 [ vrf VRF ] sparse-mode route count`
   '''
   vrfName = args.get( 'VRF', vrfDefault )
   af = 'ipv6' if 'ipv6' in args else 'ipv4'
   if validVrfName( mode, vrfName, _mfibVrfConfig[ af ], af ):
      return getIpMrouteCountSparseMode( mode, vrfName, af=af )[ 1 ]
   return None

def showIpPimUpstreamJoinsSm( mode, args ):
   '''Handler for CLI commands:
   `show ip pim [ vrf VRF ] sparse-mode upstream joins
         [ IP1 [ IP2 ] ] neighbor NEIGHBOR ]`
   `show pim ipv4 [ vrf VRF ] sparse-mode join [ IP1 [ IP2 ] ] [ neighbor NEIGHBOR ]`
   `show pim ipv6 [ vrf VRF ] sparse-mode join [ IP1 [ IP2 ] ] [ neighbor NEIGHBOR ]`
   '''
   vrfName = args.get( 'VRF', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   af = 'ipv6' if 'ipv6' in args else 'ipv4'
   if not validVrfName( mode, vrfName, _mfibVrfConfig[ af ], af ):
      return None

   groupOrSource1 = args.get( 'IP1' )
   if groupOrSource1 is not None:
      groupOrSource1 = str( groupOrSource1 )
   groupOrSource2 = args.get( 'IP2' )
   if groupOrSource2 is not None:
      groupOrSource2 = str( groupOrSource2 )

   model = PimModelLib.PimUpstreamJoins()

   try:
      ( source, group ) = PimCliLib.ipPimParseSg( groupOrSource1, groupOrSource2 )
   except ValueError:
      mode.addError( "Must enter a multicast group and/or unicast source" )
      return None

   if not waitForPimStatusSms( mode, af ):
      return None

   neighborAddr = args.get( 'NEIGHBOR' )
   if neighborAddr is not None:
      neighborAddr = Arnet.IpGenAddr( str( neighborAddr ) )

   pimStatus = getPimStatus( vrfName, af )
   status = getPimUpstreamJoins( vrfName, af )

   joinPruneStatus = [ status ]
   model.initialize( pimStatus, joinPruneStatus, source, group, neighborAddr )

   return model

def getPimRpCandidatesSm( mode, vrfName=vrfDefault, groupRange=None, detail=None,
                          af='ipv4' ):
   model = PimModelLib.PimRpCandidates()
   pimBsrRpCandidateStatus = _pimBsrCrpStatusColl[ af ].vrfStatus.get( vrfName )
   staticRp = _staticRpColl[ af ].vrfStatus.get( vrfName )

   model.generate( staticRp, pimBsrRpCandidateStatus, pimMode='Sparse',
                   groupPrefix=Arnet.IpGenPrefix( groupRange ) \
                      if groupRange else groupRange,
                   detail=detail )
   return 'modePimSm', model

def showIpPimRpSparseMode( mode, args ):
   '''Handler for CLI commands:
   `show ip pim [ vrf VRF ] sparse-mode rp [ PREFIX ] [ detail ]`
   `show pim ipv4 [ vrf VRF ] sparse-mode rp [ PREFIX ] [ detail ]`
   `show pim ipv6 [ vrf VRF ] sparse-mode rp [ PREFIX ] [ detail ]`
   '''
   vrfName = args.get( 'VRF', vrfDefault )
   groupRange = args.get( 'PREFIX' )
   if 'ipv6' in args:
      af = 'ipv6'
      validatePrefix = isValidIpv6PrefixWithError
   else:
      af = 'ipv4'
      validatePrefix = isValidPrefixWithError
      if groupRange:
         groupRange = str( groupRange )

   if not validVrfName( mode, vrfName, _mfibVrfConfig[ af ], af ):
      return None

   if groupRange:
      if not validatePrefix( mode, groupRange ):
         return None
      groupRange = str( groupRange )

   detail = 'detail' in args
   return getPimRpCandidatesSm( mode, vrfName, groupRange, detail, af )[ 1 ]

def getIpPimRpHashSm( mode, vrfName, group, detail=None, af="ipv4" ):
   model = PimModelLib.PimRpHash()

   if not validVrfName( mode, vrfName, _mfibVrfConfig[ af ], af ):
      return None, None

   rpConfig = _rpConfigColl[ af ].vrfConfig[ vrfName ]
   if vrfName not in _staticRpComputeObjColl:
      staticRp = _staticRpColl[ af ].vrfStatus.get( vrfName )
      if staticRp:
         _staticRpComputeObjColl[ vrfName ] = Tac.newInstance(
                                                "Routing::Pim::RpCompute",
                                                staticRp,
                                                rpConfig )
         _staticTrieBuilderColl[ vrfName ] = \
             Tac.newInstance( "Routing::Pim::GrpPrefixTrieBuilderSm",
                              staticRp,
                              _staticRpComputeObjColl[ vrfName ] )

   if vrfName not in _bsrRpComputeObjColl:
      if not _pimBsrCrpStatusColl[ af ].mounted():
         # pylint: disable-msg=W0212
         _pimBsrCrpStatusColl[ af ]._mount()

      pimBsrRpCandidateStatus = _pimBsrCrpStatusColl[ af ].vrfStatus.get( vrfName )
      if pimBsrRpCandidateStatus:
         _bsrRpComputeObjColl[ vrfName ] = \
               Tac.newInstance( "Routing::Pim::RpCompute",
                                pimBsrRpCandidateStatus,
                                rpConfig )
         _bsrTrieBuilderColl[ vrfName ] = \
               Tac.newInstance( "Routing::Pim::GrpPrefixTrieBuilderSm",
                                pimBsrRpCandidateStatus,
                                _bsrRpComputeObjColl[ vrfName ] )

   pimsmRpHash = SmashLazyMount.mount( _entityManager,
                                   "routing/pim/rphash/sparsemode/" + vrfName,
                                   "Routing::Pim::RpHashStatus",
                                    SmashLazyMount.mountInfo('reader' ) )

   staticCompute = _staticRpComputeObjColl.get( vrfName )
   bsrCompute = _bsrRpComputeObjColl.get( vrfName )

   rpHashAlgorithm = rpConfig.rpHashAlgorithm

   model.generate( pimsmRpHash, staticCompute, bsrCompute, group, rpHashAlgorithm )

   return 'modePimSm', model


def showIpPimRpHash( mode, args ):
   '''Handler for CLI commands:
   `show ip pim [ vrf VRF ] sparse-mode rp-hash GROUP [ detail ]`
   `show pim ipv4 [ vrf VRF ] sparse-mode rp hash GROUP [ detail ]`
   `show pim ipv6 [ vrf VRF ] sparse-mode rp hash GROUP [ detail ]`
   '''
   vrfName = args.get( 'VRF', vrfDefault )
   group = args[ 'GROUP' ]
   detail = 'detail' in args
   if 'ipv6' in args:
      af = 'ipv6'
      if group:
         group = str( group )
   else:
      af = 'ipv4'
   return getIpPimRpHashSm( mode, vrfName, group, detail, af )[ 1 ]

def showPimSparseModeNonDrDropRules( mode, args ):
   '''Handler for CLI command:
   `show pim ipv4 [ vrf VRF ] sparse-mode non-dr drop-rules`
   '''
   vrfName = args.get( 'VRF', vrfDefault )
   af = 'ipv4'

   if not validVrfName( mode, vrfName, _mfibVrfConfig[ af ], af ):
      return None

   fastFailoverStatus = _fastFailoverStatusColl[ af ].vrfStatus.get( vrfName )

   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()
   showFailover = Tac.newInstance( "PimsmCliCapi::ShowNonDrDropRules",
                                   fastFailoverStatus )
   showFailover.render( fd, fmt )

   return PimsmModel.NonDrDropRuleInterfaces

#------------------------------------------------------------------------------------
# show tech-support
#------------------------------------------------------------------------------------
# Timestamps are made up to maintain historical order within show tech-support

if Toggles.GmpToggleLib.toggleMulticastShowTechEnabled():
   showTechPimsmv4Cmds = [ 'show pim ipv4 vrf {} sparse-mode route detail',
                           'show pim ipv4 vrf {} sparse-mode route count',
                           'show pim ipv4 vrf {} sparse-mode non-dr drop-rules' ]

   showTechPimsmv6Cmds = [ 'show pim ipv6 vrf {} sparse-mode route detail',
                           'show pim ipv6 vrf {} sparse-mode route count' ]

   registerShowTechMulticast( showTechPimsmv4Cmds,
                              showTechMcastType=ShowTechMulticastType.vrf,
                              guard=mcastFibIpv4SupportedGuard )

   registerShowTechMulticast( showTechPimsmv6Cmds,
                              showTechMcastType=ShowTechMulticastType.vrf,
                              guard=mcastv6RoutingSupportedGuard )
else:
   CliPlugin.TechSupportCli.registerShowTechSupportCmd(
      '2010-06-11 02:00:00',
      cmds=[ 'show pim ipv4 sparse-mode route detail',
             'show pim ipv4 sparse-mode route count',
             'show pim ipv4 sparse-mode non-dr drop-rules' ],
      cmdsGuard=lambda: mcastRoutingSupported( _entityManager.root(),
               routingHardwareStatus=None ),
      summaryCmds=[ 'show pim ipv4 sparse-mode route count' ],
      summaryCmdsGuard=lambda: mcastRoutingSupported( _entityManager.root(),
               routingHardwareStatus=None ) )

   CliPlugin.TechSupportCli.registerShowTechSupportCmd(
      '2010-06-11 02:00:00',
      cmds=[ 'show pim ipv6 sparse-mode route detail',
             'show pim ipv6 sparse-mode route count' ],
      cmdsGuard=lambda: mcastv6RoutingSupportedGuard() is None,
      summaryCmds=[ 'show pim ipv6 sparse-mode route count' ],
      summaryCmdsGuard=lambda: mcastv6RoutingSupportedGuard() is None )

def allowed( vrfName, af="ipv4" ):
   return PimCliLib.allowed( vrfName, _routingVrfInfoDir[ af ],
                             _mfibVrfConfig[ af ] )

def pimsmConfigSanity( mode, args ):
   '''Handler for CLI command:
   `show ip pim [ vrf VRF ] config-sanity sparse-mode`
   '''
   vrfName = args.get( 'VRF', vrfDefault )
   af = "ipv4"

   model = PimConfigCheckModel.PimConfigCheck()

   if not allowed( vrfName ):
      mode.addWarning( "PIM sparse-mode is not running for vrf %s." % vrfName )
      return model

   if not waitForPimStatusSms( mode, af ):
      return None
   if not waitForPimsmStatusSms( mode, vrfName, af ):
      return None

   print( "\nBelow are hints of potential PIM sparse-mode misconfiguration:\n" )

   pimStatus = _pimStatusColl[ af ].vrfStatus.get( vrfName )

   pimsmStatus = _pimsmStatusColl.pimsmStatusColl[ af ][ vrfName ]

   staticRpConfig = _staticRpConfigColl[ af ].vrfConfig.get( vrfName )

   ipRib = _fecModeStatus.fecMode == Tac.Type( 'Smash::Fib::FecMode' ).fecModeUnified
   routingStatus, forwardingStatus = getUribStatus( vrfName, _entityManager, ipRib )

   return pimsmConfigCheck( mode, vrfName,
                           _ipConfig, _pimConfigRoot, _aclConfigCli,
                           staticRpConfig,
                           pimStatus, pimsmStatus,
                           routingStatus, forwardingStatus)

def igmpConfigSanity( mode, args ):
   '''Handler for CLI command:
   `show ip pim [ vrf VRF ] config-sanity igmp`
   '''
   vrfName = args.get( 'VRF', vrfDefault )
   af = "ipv4"

   model = PimConfigCheckModel.IgmpConfigCheck()
   model.hintsFound = 0

   if not validVrfName( mode, vrfName, _mfibVrfConfig[ af ], af ):
      return model

   if not allowed( vrfName ):
      mode.addWarning( "Igmp is not running for vrf %s." % vrfName )
      return model

   if not waitForPimStatusSms( mode ):
      return model

   pimStatus = _pimStatusColl[ af ].vrfStatus.get( vrfName )
   if not pimStatus:
      return model

   staticRpConfig = _staticRpConfigColl[ af ].vrfConfig.get( vrfName )

   ipRib = _fecModeStatus.fecMode == Tac.Type( 'Smash::Fib::FecMode' ).fecModeUnified
   ( routingStatus, forwardingStatus ) = getUribStatus(
      vrfName, _entityManager, ipRib )

   return igmpConfigCheck( mode, vrfName, _ipConfig, _aclConfigCli,
                           _mcastBoundaryConfig, _pimConfigRoot, pimStatus,
                           staticRpConfig, routingStatus, forwardingStatus )

PimCliLib.registerShowPimConfigSanityCmdCallback(
   [ 'show ip pim config-sanity sparse-mode',
     'show ip pim config-sanity igmp' ] )

def Plugin( entityManager ):

   global _entityManager
   global _ipConfig
   global _pimConfigRoot
   global _mcastBoundaryConfig
   global _aclConfigCli
   global _l3IntfStatusDir
   global _enabledStatus
   global _pegDrStatus4
   global _pegDrStatus6
   global _mvpnIntfStatus
   global _vrfNameStatus
   global _l3Config

   _entityManager = entityManager

   for af in [ 'ipv4', 'ipv6' ]:
      path = getPath( 'Routing::Multicast::Fib::VrfConfig', af )
      _mfibVrfConfig[ af ] = LazyMount.mount( entityManager, path,
                                              'Routing::Multicast::Fib::VrfConfig',
                                              'r' )

      _pimStatusColl[ af ] = PimLib.getPimStatusColl( af, "modePimSm" )

      _pimGlobalStatus[ af ] = PimLib.getPimGlobalStatus( entityManager, af )

      _ = PimLib.getPimStatusModeFilterSm( af, "modePimSm" )

      typeStr = "Routing::Pim::SparseMode::RegisterConfigColl"
      path = getPath( typeStr, af )
      _registerConfigColl[ af ] = LazyMount.mount( entityManager, path, typeStr,
                                                   "r" )

      typeStr = "Routing::Pim::SparseMode::GroupSourceMapColl"
      path = getPath( typeStr, af )
      _groupSourceMapColl[ af ] = LazyMount.mount( entityManager, path, typeStr,
                                                   "r" )

      typeStr = "Routing::Pim::Packet::JoinPruneStatusColl"
      path = getPath( typeStr, af, "sparsemode" )
      _joinPruneStatusColl[ af ] = LazyMount.mount( entityManager, path, typeStr,
                                                    'r' )

      typeStr = "Routing::Pim::SparseMode::MsdpDiscoveredSgColl"
      path = getPath( typeStr, af )
      _msdpDiscoveredSgColl[ af ] = LazyMount.mount( entityManager, path, typeStr,
                                                     'r' )

      typeStr = RpCandidateStatusColl.tacType.fullTypeName
      path = RpCandidateStatusColl.staticMountPath( af, "sparsemode" )
      _staticRpColl[ af ] = LazyMount.mount( entityManager, path, typeStr, 'r' )

      path = RpCandidateStatusColl.bsrMountPath( af )
      _pimBsrCrpStatusColl[ af ] = LazyMount.mount( entityManager, path, typeStr,
                                                    'r' )
      # static RP config
      typeStr = 'Routing::Pim::Static::RpConfigColl'
      path = getPath( typeStr, af, "sparsemode" )
      _staticRpConfigColl[ af ] = LazyMount.mount( entityManager, path, typeStr,
                                                   'r' )
      typeStr = 'Routing::Pim::RpConfigColl'
      path = getPath( typeStr, af, "sparsemode" )
      _rpConfigColl[ af ] = LazyMount.mount( entityManager, path, typeStr, 'r' )

      # Sparsemode Fast Failover Status
      typeStr = 'Routing::Pim::SparseMode::FastFailoverStatusColl'
      path = getPath( typeStr, af )
      _fastFailoverStatusColl[ af ] =\
          LazyMount.mount( entityManager, path, typeStr, 'r' )

      typeStr = "Tac::Dir"
      if af == 'ipv4':
         path = "routing/vrf/routingInfo/status"
      else:
         path = "routing6/vrf/routingInfo/status"
      _routingVrfInfoDir[ af ] = LazyMount.mount( entityManager, path, typeStr, 'r' )

      gv = cliGlob[ af ]
      # Mounting RegisterStatusColl
      RegisterStatusColl = Tac.Type(
         "Routing::Pim::SparseMode::RegisterStatusColl" )
      path = RegisterStatusColl.mountPath( af )
      gv.registerStatusColl = SharkLazyMount.mount(
         entityManager, path, RegisterStatusColl.tacType.fullTypeName,
         SharkLazyMount.mountInfo( 'shadow' ), autoUnmount=True )

   _aclConfigCli = LazyMount.mount( entityManager, 'acl/config/cli',
                                    'Acl::Input::Config', 'r' )

   # Note: add ip6/config and others below to enable config check command for Ipv6
   _ipConfig = LazyMount.mount( entityManager, 'ip/config', 'Ip::Config', 'r' )

   def doMountsComplete():
      global _fecModeStatus
      global _fecModeSm
      _fecModeStatus = Tac.newInstance( 'Smash::Fib::FecModeStatus', 'fms' )
      _fecModeSm = Tac.newInstance( 'Ira::FecModeSm', _l3Config, _fecModeStatus )

   _pimConfigRoot = LazyMount.mount( entityManager, 'routing/pim/config',
                                 'Routing::Pim::ConfigColl', 'r' )
   _mcastBoundaryConfig = LazyMount.mount( entityManager,
                                          'routing/mcastboundary/config',
                                          'Routing::McastBoundary::Config', 'r' )
   _l3IntfStatusDir = LazyMount.mount( entityManager, 'l3/intf/status',
                                       'L3::Intf::StatusDir', 'r' )
   _enabledStatus = LazyMount.mount( entityManager, 'vxlan/enabledStatus',
                                     'Vxlan::EnabledStatus', 'r' )
   pegDrStatusPath = \
                 Tac.Type( 'Routing::Multicast::PegDrStatus' ).mountPath( "ipv4" )
   _pegDrStatus4 = SmashLazyMount.mount( entityManager, pegDrStatusPath,
           'Routing::Multicast::PegDrStatus', SmashLazyMount.mountInfo( 'reader' ) )
   pegDrStatusPath = \
                 Tac.Type( 'Routing::Multicast::PegDrStatus' ).mountPath( "ipv6" )
   _pegDrStatus6 = SmashLazyMount.mount( entityManager, pegDrStatusPath,
           'Routing::Multicast::PegDrStatus', SmashLazyMount.mountInfo( 'reader' ) )

   mvpnIntfStatusType = 'Routing::Multicast::MvpnIntfStatus' 
   mvpnIntfStatusPath = Tac.Type( mvpnIntfStatusType ).mountPath()
   _mvpnIntfStatus = SmashLazyMount.mount( entityManager,
         mvpnIntfStatusPath, mvpnIntfStatusType, Smash.mountInfo( 'reader' ) )
   _vrfNameStatus = LazyMount.mount( entityManager,
         Cell.path( 'vrf/vrfNameStatus' ), 'Vrf::VrfIdMap::NameToIdMapWrapper', 'r' )

   for af in [ "ipv4", "ipv6" ]:
      _ipTunnelVlanGroupStatuses[ af ] = {}
      tacType = "Routing::Multicast::IpTunnelVlanGroupStatus"
      for statusType in _ipTunnelStatusTypes:
         path = Tac.Type( tacType ).mountPath( af, statusType )
         _ipTunnelVlanGroupStatuses[ af ][ statusType ] = \
                LazyMount.mount( entityManager, path, tacType, 'r' )

   mg = entityManager.mountGroup()
   _l3Config = mg.mount( "l3/config", "L3::Config", "ri" )
   mg.close( doMountsComplete )

   PimCliLib.pimMessageCountersHook.addExtension( getPimMessageCounters )
   PimCliLib.pimUpstreamJoinsHook.addExtension ( getPimUpstreamJoins )
   PimCliLib.pimShowIpPimRpHook.addExtension(
         ( 'sparseMode', getPimRpCandidatesSm ) )
   PimCliLib.pimShowIpPimRpHashHook.addExtension( getIpPimRpHashSm )
   PimCliLib.pimShowIpMrouteHook.addExtension(
         ('sparseMode', getIpMrouteSparseMode ) )
   PimCliLib.pimShowIpMrouteIntfHook.addExtension(
         ('sparseMode', getIpMrouteIntfSparseMode ) )
   PimCliLib.pimShowIpMrouteCountHook.addExtension(
         ('sparseMode', getIpMrouteCountSparseMode ) )
