# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re

import AgentCommandRequest
import AgentDirectory
import ArPyUtils
import BasicCli
import Cell
import CliCommand
import CliGlobal
import CliMatcher
import CliMode.Poe
from CliModel import Model, Submodel, Str, Bool, Float, Enum, Dict, List
from IntfModels import Interface
import CliParser
import CliPlugin.FruCli as FruCli # pylint: disable=consider-using-from-import
import CliPlugin.ModuleCli as ModuleCli # pylint: disable=consider-using-from-import
# pylint: disable-next=consider-using-from-import
import CliPlugin.ModuleIntfCli as ModuleIntfCli
import CliPlugin.TechSupportCli
from CliPlugin import EthIntfCli
from CliPlugin import IntfCli
from CliPlugin import ReloadCli
import ConfigMount
import Intf.IntfRange as IntfRange # pylint: disable=consider-using-from-import
from IntfRangePlugin.PoeIntf import PoeAutoIntfType
import LazyMount
import PLDeviceAgent
import PLSystemAgent
import ShowCommand
import Tac
from CliToken.Platform import platformMatcherForShow
from TableOutput import Format, createTable
from io import StringIO

DEFAULT_REBOOT_ACTION = 'powerOff'
DEFAULT_INTF_POE_REBOOT_ACTION = 'unset'
DEFAULT_INTF_POE_LINK_DOWN_ACTION = 'unset'
DEFAULT_INTF_POE_LINK_DOWN_POWER_OFF_TIME = 5
DEFAULT_INTERFACE_SHUTDOWN_ACTION = 'maintain'
DEFAULT_INTF_POE_SHUTDOWN_ACTION = 'unset'
DEFAULT_POE_PRIORITY = 'defaultPriority'
DEFAULT_POE_POWER_LIMIT = Tac.Value( "Units::Watts" ).value
MAX_POE_POWER_LIMIT = 90
MIN_POE_POWER_LIMIT = 1

gv = CliGlobal.CliGlobal( sliceDir=None, poeTecConfig=None, globalPoeStatus=None,
                          poeStatus=None, poeCliConfig=None, sysdbRedSupStatus=None,
                          sysdbRedSupConfig=None, hwEntMib=None )

def printTable( table ):
   # Strip whitespace from the end of the lines, because that's how it was
   # before and it's what some parsers expect. Also remove the last newline.
   print( '\n'.join( map( str.rstrip, table.output().split( '\n' )[ : -1 ] ) ) )

# -----------------------------------------------------------------------------------
# Adds PoE related CLI commands to the "config-if" mode for PoE ports.
# -----------------------------------------------------------------------------------
class PoePortModelet( CliParser.Modelet ):

   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( isinstance( mode.intf, EthIntfCli.EthPhyIntf ) and
               mode.intf.name.startswith( "Ethernet" ) )

# -----------------------------------------------------------------------------------
# Associate the PoePortModelet with the "config-if" mode.
# -----------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( PoePortModelet )

def poeGuard( mode, token ):
   if not gv.poeStatus or not gv.poeStatus.entryState:
      return CliParser.guardNotThisPlatform
   if isinstance( mode, IntfCli.IntfConfigMode ):
      if mode.intf.name not in gv.poeStatus:
         return "not supported on this interface"
   return None

def poeConfigGuard( mode, token ):
   poeConfig = gv.sliceDir.get( 'PoeDevice' )
   platformConfig = gv.sliceDir.get( 'Platform' )
   if isinstance( mode, BasicCli.GlobalConfigMode ):
      if ( ( poeConfig and poeConfig.poePort ) or
           ( platformConfig and platformConfig.poePort ) or
           ( gv.poeTecConfig and gv.poeTecConfig.poeCard ) ):
         return None
      return CliParser.guardNotThisPlatform
   if not isinstance( mode, IntfCli.IntfConfigMode ):
      return CliParser.guardNotThisPlatform
   if poeConfig and poeConfig.poePort:
      if mode.intf.name in poeConfig.poePort:
         return None
   if platformConfig and platformConfig.poePort:
      if mode.intf.name in platformConfig.poePort:
         return None
   if gv.poeTecConfig and gv.poeTecConfig.poeCard:
      for poeCard in gv.poeTecConfig.poeCard.values():
         for poeChip in poeCard.poeChip.values():
            for poePort in poeChip.poePort:
               if mode.intf.name == poePort:
                  return None
   return CliParser.guardNotThisPlatform

def poeTecGuard( mode, token ):
   poeConfig = gv.sliceDir.get( 'PoeDevice', gv.poeTecConfig )
   if poeConfig is gv.poeTecConfig:
      platformConfig = gv.sliceDir.get( 'Platform' )
      # For 64 bit platforms, Pluto PoE is part of Platform group
      # Check for whether or not poePort attribute is empty
      if not ( platformConfig and platformConfig.poePort ):
         return None
   return CliParser.guardNotThisPlatform

matcherPoe = CliMatcher.KeywordMatcher(
   'poe',
   helpdesc='PoE configuration' )
nodePoe = CliCommand.Node(
   matcher=matcherPoe,
   guard=poeGuard )

nodePoeConfig = CliCommand.Node(
   matcher=matcherPoe,
   guard=poeConfigGuard )

matcherClassNumber = CliMatcher.PatternMatcher(
   pattern='class[0-8]',
   helpname='class<0-8>',
   helpdesc='Class for setting the power limit' )
matcherPowerLimit = CliMatcher.FloatMatcher(
   MIN_POE_POWER_LIMIT,
   MAX_POE_POWER_LIMIT,
   helpdesc='Number in watts',
   precisionString='%.25g' )

# # _cliIntfClazz is the class of physical interface to check after parsing, e.g.
# # "Ethernet3/1" gets turned into EthPhyIntf( "Ethernet3/1" )
PoeAutoIntfType.cliIntfClazzIs( EthIntfCli.EthPhyIntf )

poeRangeMatcher = IntfRange.IntfRangeMatcher(
   explicitIntfTypes=( PoeAutoIntfType, ) )

# -----------------------------------------------------------------------------------
# [no|default] poe [disabled]
# -----------------------------------------------------------------------------------
class PoePortDisabled( CliCommand.CliCommandClass ):
   syntax = 'poe [disabled]'
   noOrDefaultSyntax = 'poe disabled ...'
   data = {
      'poe': nodePoeConfig,
      'disabled': 'Disable PoE related functionality',
   }

   handler = "PlutoCliHandler.doPoePortDisabled"

   noOrDefaultHandler = "PlutoCliHandler.noPoePortDisabled"

PoePortModelet.addCommandClass( PoePortDisabled )

# -----------------------------------------------------------------------------------
# [no|default] poe priority (low|medium|high|critical) command, in "config-if" mode
# -----------------------------------------------------------------------------------
matcherPriorityEnum = CliMatcher.EnumMatcher( {
   'low': 'Low priority (default)',
   'medium': 'Medium priority',
   'high': 'High priority',
   'critical': 'Critical priority',
} )

class PoePortPriorityCmd( CliCommand.CliCommandClass ):
   syntax = 'poe priority PRIORITY'
   noOrDefaultSyntax = 'poe priority ...'
   data = {
      'poe': nodePoeConfig,
      'priority': 'Port priority',
      'PRIORITY': matcherPriorityEnum,
   }

   handler = "PlutoCliHandler.doPoePortPriorityCmd"

   noOrDefaultHandler = "PlutoCliHandler.noPoePortPriorityCmd"

PoePortModelet.addCommandClass( PoePortPriorityCmd )

# -----------------------------------------------------------------------------------
# [no|default] poe limit ((class CLASS_NUM)|(NUM_WATTS watts)) [fixed]
# -----------------------------------------------------------------------------------
class PoeLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'poe limit ( ( class CLASS_NUM ) | ( NUM_WATTS watts ) ) [ fixed ]'
   noOrDefaultSyntax = 'poe limit ...'
   data = {
      'poe': nodePoeConfig,
      'limit': 'Set power limit',
      'class': 'Powered device classification',
      'CLASS_NUM': matcherClassNumber,
      'NUM_WATTS': matcherPowerLimit,
      'watts': 'Unit watts',
      'fixed': 'Ignore hardware classification',
   }

   handler = "PlutoCliHandler.doPoeLimitCmd"

   noOrDefaultHandler = "PlutoCliHandler.noPoeLimitCmd"

PoePortModelet.addCommandClass( PoeLimitCmd )

# -----------------------------------------------------------------------------------
# [no|default] poe negotiation lldp [disabled]
# -----------------------------------------------------------------------------------
class PoePortLldpDisabled( CliCommand.CliCommandClass ):
   syntax = 'poe negotiation lldp [disabled]'
   noOrDefaultSyntax = 'poe negotiation lldp disabled ...'
   data = {
      'poe': nodePoeConfig,
      'negotiation': 'Power limit negotiation',
      'lldp': 'LLDP',
      'disabled': 'Disable PoE negotiation over LLDP',
   }

   handler = "PlutoCliHandler.doPoePortLldpDisabled"

   noOrDefaultHandler = "PlutoCliHandler.noPoePortLldpDisabled"

PoePortModelet.addCommandClass( PoePortLldpDisabled )

# -----------------------------------------------------------------------------------
# [no|default] poe legacy detect
# -----------------------------------------------------------------------------------
def guardLegacy( mode, token ):
   if ( mode.intf.name in gv.poeStatus and
        gv.poeStatus[ mode.intf.name ].legacyDetectSupported ):
      return None
   return "not supported on this interface"

nodeLegacy = CliCommand.guardedKeyword( 'legacy', helpdesc='Legacy detection',
                                        guard=guardLegacy )

class PoeLegacyDetectCmd( CliCommand.CliCommandClass ):
   syntax = 'poe legacy detect'
   noOrDefaultSyntax = 'poe legacy detect ...'
   data = {
      'poe': nodePoeConfig,
      'legacy': nodeLegacy,
      'detect': 'Detection type',
   }

   handler = "PlutoCliHandler.doPoeLegacyDetectCmd"

   noOrDefaultHandler = "PlutoCliHandler.noPoeLegacyDetectCmd"

PoePortModelet.addCommandClass( PoeLegacyDetectCmd )

# -----------------------------------------------------------------------------------
# [no|default] poe pairset ( ( alternative ( a | b ) ) | 4-pair )
# -----------------------------------------------------------------------------------
def guardPairset( mode, token ):
   if ( poeTecGuard( mode, token ) != CliParser.guardNotThisPlatform and
        mode.intf.name in gv.poeStatus and
        gv.poeStatus[ mode.intf.name ].capacity > 30.0 ):
      return None
   return "not supported on this interface"

nodePairset = CliCommand.guardedKeyword( 'pairset',
                                         helpdesc='Configure pairset(s) to enable',
                                         guard=guardPairset )

class PoePairsetCmd( CliCommand.CliCommandClass ):
   syntax = 'poe pairset ( ( alternative ( a | b ) ) | 4-pair )'
   noOrDefaultSyntax = 'poe pairset ...'
   data = {
      'poe': nodePoeConfig,
      'pairset': nodePairset,
      'alternative': 'Enable one of the pairsets',
      'a': 'Enable alternative A',
      'b': 'Enable alternative B',
      '4-pair': 'Enable both pairsets',
   }

   handler = "PlutoCliHandler.doPoePairsetCmd"

   noOrDefaultHandler = "PlutoCliHandler.noPoePairsetCmd"

PoePortModelet.addCommandClass( PoePairsetCmd )

poeStates = ( 'unknown', 'disabled', 'detecting', 'classified', 'powered', 'failed',
              'overloaded', 'linkDown', 'adminShutdown', 'invalid', 'fault',
              'mismatch' )

def valueStr( value, deci, unit ):
   if value is None:
      return 'N/A'
   if round( value, deci ) == round( value, 0 ):
      return ( "%.0f" % value ) + unit # pylint: disable=consider-using-f-string
   else:
      # pylint: disable-next=consider-using-f-string
      return ( "%.*f" % ( deci, value ) ) + unit

def stateStr( state ):
   camelCaseMap = { 'linkDown': 'link down', 'adminShutdown': 'admin down' }
   return camelCaseMap.get( state, state )

def trueFalseStr( value ):
   if value:
      return "true"
   else:
      return "false"

class PoePort( Model ):
   portName = Str( help='PoE port name' )
   portPresent = Bool( help='PoE port present' )
   portCapacity = Float( help='PoE port capacity (watts)', optional=True )
   pseEnabled = Bool( help='PSE functionalities enabled', optional=True )
   portPriority = Enum( values=( 'low', 'medium', 'high', 'critical' ),
                                 help='PoE port priority. Available power is given '
                                      'to higher priority ports. A power supply '
                                      'failure can reduce the available power.',
                                 optional=True )
   portPairset = Enum( values=( 'fourPair', 'altA', 'altB' ),
                                help='Enabled pairset(s)', optional=True )
   lldpEnabled = Bool( help='LLDP enabled', optional=True )
   legacyDetectEnabled = Bool( help='Legacy detect enabled', optional=True )
   powerLimit = Float( help='PoE port power limit (watts)', optional=True )
   grantedPower = Float( help='PoE port granted power (watts)',
                                  optional=True )
   portState = Enum( values=poeStates,
                              help='PoE port state', optional=True )
   pdClass = Str( help='Powered device class', optional=True )
   power = Float( help='Output power (watts)', optional=True )
   current = Float( help='Output current (milliamps)', optional=True )
   voltage = Float( help='Output voltage (volts)', optional=True )
   temperature = Float( help='Temperature (celsius)', optional=True )
   misconfigured = Bool( help='Legacy detect and port mode misconfiguraton',
                                  optional=True )

   class PoeChannel( Model ):
      pairSet = Str( help='Pairset of the channel', optional=True )
      portState = Enum( values=poeStates,
                                 help='PoE channel state', optional=True )
      pdClass = Str( help='Powered device class', optional=True )
      pseLimit = Float( help='PoE port power limit (watts)', optional=True )
      power = Float( help='Output power (watts)', optional=True )
      current = Float( help='Output current (milliamps)', optional=True )
      voltage = Float( help='Output voltage (volts)', optional=True )

      def getRow( self ):
         return [ self.pairSet,
                  stateStr( self.portState ),
                  self.pdClass,
                  valueStr( self.pseLimit, 0, 'W' ),
                  valueStr( self.power, 1, 'W' ),
                  valueStr( self.current, 0, 'mA' ),
                  valueStr( self.voltage, 1, 'V' ),
                 ]

   primary = Submodel( valueType=PoeChannel,
                          help='Details on primary channel',
                          optional=True )
   alternative = Submodel( valueType=PoeChannel,
                           help='Details on alternative channel',
                           optional=True )

   def getRow( self ):
      def portPairsetStr( value ):
         if value == 'fourPair':
            value = '4-pair'
         elif value == 'altA':
            value = 'Alt A'
         else:
            value = 'Alt B'
         if self.misconfigured:
            value = value + '*'
         return value

      return [ "Et" + re.split( '([0-9/]+)', self.portName )[ 1 ],
               valueStr( self.portCapacity, 0, 'W' ),
               trueFalseStr( self.pseEnabled ),
               portPairsetStr( self.portPairset ),
               self.portPriority,
               trueFalseStr( self.lldpEnabled ),
               trueFalseStr( self.legacyDetectEnabled ),
               valueStr( self.powerLimit, 0, 'W' ),
               valueStr( self.grantedPower, 2, 'W' ),
               stateStr( self.portState ),
               self.pdClass,
               valueStr( self.power, 1, 'W' ),
               valueStr( self.current, 0, 'mA' ),
               valueStr( self.voltage, 1, 'V' ),
               valueStr( self.temperature, 0, 'C' )
              ]

class PoePorts( Model ):
   poePorts = Dict( keyType=str, valueType=PoePort, help='PoE ports' )
   detail = Bool( help='Show detailed status of each 2-pairs' )

   def render( self ):
      if not self.poePorts:
         return

      poeTableHeader = [
         [ "", "hl", [ "Interface" ], ],
         [ "", "hr", [ "Capacity" ], ],
         [ "PSE", "hl", [ "Enabled" ], ],
         [ "", "hl", [ "Pairset" ] ],
         [ "", "hl", [ "Priority" ], ],
         [ "LLDP", "hl", [ "Enabled" ], ],
         [ "Legacy", "hl", [ "Detect" ], ],
         [ "Power", "hr", [ "Max" ], ],
         [ "Power", "hr", [ "Granted" ], ],
         [ "Port", "hl", [ "State" ], ],
         [ "", "hl", [ "Class" ], ],
         [ "", "hr", [ "Power" ], ],
         [ "", "hr", [ "Current" ], ],
         [ "", "hr", [ "Voltage" ], ],
         [ "", "hr", [ "Temp" ], ]
      ]

      poeDetailTableHeader = [
         [ "", "hl", [ "Interface" ], ],
         [ "", "hl", [ "Pairset" ], ],
         [ "Port", "hl", [ "State" ], ],
         [ "", "hl", [ "Class" ], ],
         [ "PSE", "hr", [ "Limit" ], ],
         [ "", "hr", [ "Power" ], ],
         [ "", "hr", [ "Current" ], ],
         [ "", "hr", [ "Voltage" ], ],
      ]

      def makePoeTable():
         t = createTable( poeTableHeader )
         fl = Format( justify="left" )
         fl.noPadLeftIs( True )
         fl.padLimitIs( True )
         fr = Format( justify="right" )
         fr.noPadLeftIs( True )
         fr.padLimitIs( True )
         t.formatColumns( fl, fr, fl, fl, fl, fl, fl, fr,
                          fr, fl, fl, fr, fr, fr, fr )
         return t

      def makePoeDetailTable():
         t = createTable( poeDetailTableHeader )
         fl = Format( justify="left" )
         fl.noPadLeftIs( True )
         fl.padLimitIs( True )
         fr = Format( justify="right" )
         fr.noPadLeftIs( True )
         fr.padLimitIs( True )
         t.formatColumns( fl, fl, fl, fl, fr, fr, fr, fr )
         return t

      if self.detail:
         outputTable = makePoeDetailTable()
      else:
         outputTable = makePoeTable()

      misconfigured = False
      rebootAction = gv.poeCliConfig.rebootAction == 'maintain'
      for portName in ArPyUtils.naturalsorted( self.poePorts ):
         port = self.poePorts[ portName ]
         portConfig = gv.poeCliConfig.newPoeCliConfig( portName )
         rebootAction = rebootAction or portConfig.rebootAction == 'maintain'
         if self.detail:
            portName = "Ethernet" + re.split( '([0-9/]+)', portName )[ 1 ]
            if port.primary:
               outputTable.newRow( portName, *port.primary.getRow() )
            if port.alternative:
               outputTable.newRow( portName, *port.alternative.getRow() )
         else:
            if port.misconfigured:
               misconfigured = True
            outputTable.newRow( *port.getRow() )
      if ( ( not self.detail ) and rebootAction and
           gv.sysdbRedSupStatus.protocol == 'rpr' ):
         print( "The reboot action is overridden from maintain to "
                "power-off due to the incompatible RPR redundancy mode.\n" )
      printTable( outputTable )
      if misconfigured:
         print( "* The pairset configuration is overridden "
                "from 4-pair to Alternative A due to the "
                "incompatible legacy detect configuration." )

class PoePortNbrInfo( Model ):
   portState = Enum( values=poeStates,
                              help='PoE port state', optional=True )
   power = Float( help='Output power (watts)', optional=True )
   lldpSystemDesc = List( help="Neighbor system description",
                                   valueType=str, optional=True )

   def getRow( self ):
      def lldpSystemDescStr( lldpSystemDesc ):
         if not lldpSystemDesc:
            return 'N/A'
         else:
            systemDesc = ''
            for desc in lldpSystemDesc:
               lines = [ desc[ i : i + 50 ] for i in range( 0, len( desc ), 50 ) ]
               systemDesc += '\n'.join( lines )
               systemDesc += '\n'
            return systemDesc[ : -1 ]
      return [ stateStr( self.portState ),
               valueStr( self.power, 1, 'W' ),
               lldpSystemDescStr( self.lldpSystemDesc ),
             ]

class PoePortNeighbors( Model ):
   poePorts = Dict( keyType=Interface, valueType=PoePortNbrInfo,
                             help='A mapping of PoE port to its information' )
   powerBudget = Float( help='Total power budget in watts' )
   usedPoePower = Float( help='Total used PoE output power in watts' )
   remainingPower = Float( help='Remaining power in watts' )

   def render( self ):
      if not self.poePorts:
         return

      poeTableHeader = [
         [ "", "hl", [ "Interface" ], ],
         [ "Port", "hl", [ "State" ], ],
         [ "", "hr", [ "Power" ], ],
         [ "", "hl", [ "Neighbor Description" ], ],
      ]

      def makePoeTable():
         t = createTable( poeTableHeader )
         fl = Format( justify="left" )
         fl.noPadLeftIs( True )
         fl.padLimitIs( True )
         fr = Format( justify="right" )
         fr.noPadLeftIs( True )
         fr.padLimitIs( True )
         t.formatColumns( fl, fl, fr, fl )
         return t

      # pylint: disable-next=consider-using-f-string
      print( "Power budget allocation: {}".format(
         "usage-based" if gv.poeCliConfig.telemetryBasedPoeEnabled
         else "class-based" ) )
      print( f"Total power budget: {self.powerBudget:.1f}W" )
      print( f"Power used: {self.usedPoePower:.1f}W" )
      print( f"Remaining power: {self.remainingPower:.1f}W" )
      print()
      outputTable = makePoeTable()
      for portName in ArPyUtils.naturalsorted( self.poePorts ):
         outputTable.newRow( "Et" + re.split( '([0-9/]+)', portName )[ 1 ],
                             *self.poePorts[ portName ].getRow() )
      printTable( outputTable )

# -----------------------------------------------------------------------------------
# show poe [ ( interface INTF ) | ( module MOD ) ] [ detail ]
# -----------------------------------------------------------------------------------
class ShowPoe( ShowCommand.ShowCliCommandClass ):
   syntax = 'show poe [ ( interface INTF ) | ( MOD ) ] [ detail ]'
   data = {
      'poe' : nodePoe,
      'interface' : 'Interface',
      'INTF' : poeRangeMatcher,
      'MOD' : ModuleIntfCli.ModuleExpressionFactory(),
      'detail' : 'Show detailed status of each 2-pairs',
   }
   privileged = True
   cliModel = PoePorts

   handler = "PlutoCliHandler.doShowPoe"

BasicCli.addShowCommandClass( ShowPoe )

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2019-09-19 16:10:00',
   cmds=[ 'show poe' ],
   cmdsGuard=lambda: not poeGuard( None, None ) )

# -----------------------------------------------------------------------------------
# show poe neighbor [ ( interface INTF ) | ( module MOD ) ]
# -----------------------------------------------------------------------------------
class ShowPoeNeighbor( ShowCommand.ShowCliCommandClass ):
   syntax = 'show poe neighbor [ ( interface INTF ) | ( MOD ) ]'
   data = {
      'poe': nodePoe,
      'neighbor': 'Show LLDP neighbor system information',
      'interface': 'Interface',
      'INTF': IntfCli.Intf.rangeMatcher,
      'MOD': ModuleIntfCli.ModuleExpressionFactory(),
   }
   privileged = True
   cliModel = PoePortNeighbors

   handler = "PlutoCliHandler.doShowPoeNeighbor"

BasicCli.addShowCommandClass( ShowPoeNeighbor )

# -----------------------------------------------------------------------------------
# Common
# -----------------------------------------------------------------------------------
matcherPluto = CliMatcher.KeywordMatcher(
   'pluto',
   helpdesc='Components managed with Pluto agents' )

def queryDevice( mode, command ):
   def f( name, dirName ):
      outputBuff = StringIO()
      AgentCommandRequest.runSocketCommand( mode.entityManager,
                                            dirName, name,
                                            command, stringBuff=outputBuff )
      return PlutoDevice( debugInfo=outputBuff.getvalue() )
   return f

def dictMap( f, names ):
   container = PlutoDeviceContainer()
   devices = container.devices
   sysname = names.entityManager_.sysname()
   for name in names:
      if AgentDirectory.agentIsRunning( sysname, 'PLDevice-' + name ):
         devices[ name ] = f( name, PLDeviceAgent.name + '-' + name )
   if AgentDirectory.agentIsRunning( sysname, 'PLSystem' ):
      devices[ 'system' ] = f( 'system', PLSystemAgent.name )

   return container

# ----------------------------------------------------------------------------------
class PlutoDevice( Model ):
   __public__ = False
   debugInfo = Str( help='Debug information from device' )

class PlutoDeviceContainer( Model ):
   __public__ = False
   devices = Dict( valueType=PlutoDevice,
                            help='Maps a device name to debug information about'
                            ' its components' )

   def render( self ):
      for deviceName in sorted( self.devices ):
         device = self.devices[ deviceName ]
         if device.debugInfo:
            print( deviceName )
            print( device.debugInfo, )

class ShowPlatformPluto( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform pluto COMPONENT'
   data = {
      'platform': platformMatcherForShow,
      'pluto': matcherPluto,
      'COMPONENT': CliMatcher.EnumMatcher( {
         'all': 'Information about all Pluto components',
         'fans': 'Fan debug information',
         'power-supplies': 'Power supply debug information',
         'system': 'System debug information',
         'temperature-sensors': 'Temperature sensor debug information'
      } )
   }
   privileged = True
   cliModel = PlutoDeviceContainer
   tokenMap_ = {
      'all': 'all',
      'fans': 'fan',
      'power-supplies': 'power',
      'system': 'system',
      'temperature-sensors': 'tempSensor',
   }

   @staticmethod
   def handler( mode, args ):
      command = ShowPlatformPluto.tokenMap_[ args[ 'COMPONENT' ] ]
      return dictMap( queryDevice( mode, command ), gv.sliceDir )

BasicCli.addShowCommandClass( ShowPlatformPluto )

# -----------------------------------------------------------------------------------
# show platform pluto poe-ports [ raw ]
# -----------------------------------------------------------------------------------
class ShowPlutoPoeRegisters( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform pluto poe-ports [ raw ]'
   data = {
      'platform': platformMatcherForShow,
      'pluto': matcherPluto,
      'poe-ports': 'PoE debug information',
      'raw': 'PoE raw debug information'
   }
   privileged = True
   cliModel = PlutoDeviceContainer

   @staticmethod
   def handler( mode, args ):
      poeCommand = 'poe raw' if 'raw' in args else 'poe'
      return dictMap( queryDevice( mode, poeCommand ), gv.sliceDir )

BasicCli.addShowCommandClass( ShowPlutoPoeRegisters )

# -----------------------------------------------------------------------------------
# poe config mode
# -----------------------------------------------------------------------------------
class PoeMode( CliMode.Poe.PoeMode, BasicCli.ConfigModeBase ):
   name = 'PoE configuration'

   def __init__( self, parent, session ):
      CliMode.Poe.PoeMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# -----------------------------------------------------------------------------------
# poe linecard config mode
# -----------------------------------------------------------------------------------
class PoeLinecardMode( CliMode.Poe.PoeLinecardMode, BasicCli.ConfigModeBase ):
   name = 'PoE linecard configuration'

   def __init__( self, parent, session, param ):
      CliMode.Poe.PoeLinecardMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# -----------------------------------------------------------------------------------
# [ no | default ] poe
# in global config mode
# -----------------------------------------------------------------------------------
class Poe( CliCommand.CliCommandClass ):
   syntax = 'poe'
   noOrDefaultSyntax = 'poe'
   data = {
      'poe': nodePoeConfig,
   }

   handler = "PlutoCliHandler.doPoe"

   noOrDefaultHandler = "PlutoCliHandler.noPoe"

BasicCli.GlobalConfigMode.addCommandClass( Poe )

# -----------------------------------------------------------------------------------
# [ no | default ] reboot action ( maintain | power-off )
# in poe config mode
# -----------------------------------------------------------------------------------
def guardRebootAction( mode, token ):
   poeConfig = gv.sliceDir.get( 'PoeDevice', gv.poeTecConfig )
   if poeConfig is not gv.poeTecConfig:
      if not poeConfig.continuousPoeSupported:
         return CliParser.guardNotThisPlatform
   platformConfig = gv.sliceDir.get( "Platform" )
   if platformConfig and platformConfig.poePort:
      # if PoE is part of PLDevice-Platform agent and not PLDevice-PoeDevice
      if not platformConfig.continuousPoeSupported:
         return CliParser.guardNotThisPlatform
   return None

nodeReboot = CliCommand.guardedKeyword(
   'reboot',
   helpdesc='Configure behavior on reboot',
   guard=guardRebootAction )

class RebootActionCmd( CliCommand.CliCommandClass ):
   syntax = 'reboot action ( maintain | power-off )'
   noOrDefaultSyntax = 'reboot action ...'
   data = {
      'reboot': nodeReboot,
      'action': 'Configure the action to take',
      'maintain': 'Maintain PoE power during reboot',
      'power-off': 'Power off PoE power during reboot (default)',
   }

   handler = "PlutoCliHandler.doRebootActionCmd"

   noOrDefaultHandler = "PlutoCliHandler.noRebootActionCmd"

PoeMode.addCommandClass( RebootActionCmd )

# -----------------------------------------------------------------------------------
# [ no | default ] poe reboot action ( maintain | power-off )
# in config-if mode
# -----------------------------------------------------------------------------------
class PoeRebootActionConfigIfCmd( CliCommand.CliCommandClass ):
   syntax = 'poe reboot action ( maintain | power-off )'
   noOrDefaultSyntax = 'poe reboot action ...'
   data = {
         'poe': nodePoeConfig,
         'reboot': nodeReboot,
         'action': 'Configure the action to take',
         'maintain': 'Maintain PoE power during reboot',
         'power-off': 'Power off PoE power during reboot',
   }

   handler = "PlutoCliHandler.doPoeRebootActionConfigIfCmd"

   noOrDefaultHandler = "PlutoCliHandler.noPoeRebootActionConfigIfCmd"

PoePortModelet.addCommandClass( PoeRebootActionConfigIfCmd )

# -----------------------------------------------------------------------------------
# [ no | default ] interface shutdown action ( maintain | power-off )
# in poe config mode
# -----------------------------------------------------------------------------------
class InterfaceShutdownActionCmd( CliCommand.CliCommandClass ):
   syntax = 'interface shutdown action ( maintain | power-off )'
   noOrDefaultSyntax = 'interface shutdown action ...'
   data = {
      'interface': 'Interface behavior',
      'shutdown': 'Configure behavior on admin shutdown',
      'action': 'Configure the action to take',
      'maintain': 'Maintain PoE power on interface shutdown',
      'power-off': 'Power off PoE power on interface shutdown',
      }

   handler = "PlutoCliHandler.doInterfaceShutdownActionCmd"

   noOrDefaultHandler = "PlutoCliHandler.noInterfaceShutdownActionCmd"

PoeMode.addCommandClass( InterfaceShutdownActionCmd )

# -----------------------------------------------------------------------------------
# [ no | default ] poe shutdown action ( maintain | power-off )
# in config-if mode
# -----------------------------------------------------------------------------------
class PoeShutdownActionConfigIfCmd( CliCommand.CliCommandClass ):
   syntax = 'poe shutdown action ( maintain | power-off )'
   noOrDefaultSyntax = 'poe shutdown action ...'
   data = {
         'poe': nodePoeConfig,
         'shutdown': 'Configure behavior on admin shutdown',
         'action': 'Configure the action to take',
         'maintain': 'Maintain PoE power on interface shutdown',
         'power-off': 'Power off PoE power on interface shutdown',
         }

   handler = "PlutoCliHandler.doPoeShutdownActionConfigIfCmd"

   noOrDefaultHandler = "PlutoCliHandler.noPoeShutdownActionConfigIfCmd"

PoePortModelet.addCommandClass( PoeShutdownActionConfigIfCmd )

# -----------------------------------------------------------------------------------
# [ no | default ] poe link down action ( maintain |
#                                         ( power-off [ NUM_SECONDS seconds ] ) )
# in config-if mode
# -----------------------------------------------------------------------------------
class PoeLinkDownActionConfigIfCmd( CliCommand.CliCommandClass ):
   syntax = ( 'poe link down action ( maintain | '
              '( power-off [ NUM_SECONDS seconds ] ) )' )
   noOrDefaultSyntax = 'poe link down action ...'
   data = {
         'poe': nodePoeConfig,
         'link': 'Configure link behavior',
         'down': 'Configure behavior on link down',
         'action': 'Configure the action to take',
         'maintain': 'Maintain PoE power during link down',
         'power-off': 'Power off PoE power during link down',
         'NUM_SECONDS': CliMatcher.IntegerMatcher(
                           1, 86400,
                           # pylint: disable-next=consider-using-f-string
                           helpdesc=( 'Number in seconds (default: %d)' %
                                      DEFAULT_INTF_POE_LINK_DOWN_POWER_OFF_TIME ) ),
         'seconds': 'Unit seconds',
   }

   handler = "PlutoCliHandler.doPoeLinkDownActionConfigIfCmd"

   noOrDefaultHandler = "PlutoCliHandler.noPoeLinkDownActionConfigIfCmd"

PoePortModelet.addCommandClass( PoeLinkDownActionConfigIfCmd )

# -----------------------------------------------------------------------------------
# [ no | default ] power budget allocation ( class-based | usage-based )
# -----------------------------------------------------------------------------------
def guardDynamicPoe( mode, token ):
   # On modular systems, all connected linecards need to support dynamic PoE for
   # it to work properly
   if ( poeTecGuard( mode, token ) == CliParser.guardNotThisPlatform or
        not gv.poeTecConfig.poeCard ):
      return CliParser.guardNotThisPlatform
   for card in gv.poeTecConfig.poeCard.values():
      if not card.isDynamicPoeSupported:
         return CliParser.guardNotThisPlatform
   return None

nodePower = CliCommand.guardedKeyword(
                  'power',
                  helpdesc='Configure PoE power behavior',
                  guard=guardDynamicPoe )

class PowerBudgetAllocationCmd( CliCommand.CliCommandClass ):
   syntax = 'power budget allocation ( class-based | usage-based )'
   noOrDefaultSyntax = 'power budget allocation ...'
   data = {
         'power': nodePower,
         'budget': 'Configure PoE power budget',
         'allocation': 'Configure how PoE power is allocated',
         'class-based': 'Use classified power limit to determine available power',
         'usage-based': 'Use real-time power usage to determine available power',
   }

   handler = "PlutoCliHandler.doPowerBudgetAllocationCmd"

   noOrDefaultHandler = "PlutoCliHandler.noPowerBudgetAllocationCmd"

PoeMode.addCommandClass( PowerBudgetAllocationCmd )

# -----------------------------------------------------------------------------------
# [ no | default ] module MODNAME
# in poe config mode
# -----------------------------------------------------------------------------------
class LinecardExpressionFactory( FruCli.SlotExpressionFactory ):

   def generate( self, name ):
      expression_ = name
      data_ = { name: self.slotMatcher_ }
      names = []
      k = 'Linecard'
      kwName = name + '_' + k.lower()
      slotName = name + '_' + k.upper()
      expression_ += f'| ( {kwName} {slotName} )'
      data_[ kwName ] = CliCommand.guardedKeyword(
         # pylint: disable-next=consider-using-f-string
         k, "%s modules" % k, FruCli.cardTypeGuard )
      data_[ slotName ] = FruCli.SlotMatcher( forceTags=[ k ] )
      names.append( slotName )

      class _SlotExpression( CliCommand.CliExpression ):
         expression = expression_
         data = data_
         slotNames_ = names

         @staticmethod
         def adapter( mode, args, argsList ):
            for n in _SlotExpression.slotNames_:
               val = args.pop( n, None )
               if val is not None:
                  if isinstance( val, list ):
                     # the expression is used in a list
                     args.setdefault( name, [] )
                     args[ name ].extend( val )
                  else:
                     args[ name ] = val
                     # only one arg is possible, just stop
                     break
      return _SlotExpression

class EnterModuleLinecardCmd( CliCommand.CliCommandClass ):
   syntax = 'module MODNAME'
   noOrDefaultSyntax = 'module MODNAME'
   data = {
      'module': ModuleCli.moduleNode,
      'MODNAME': LinecardExpressionFactory(),
   }

   handler = "PlutoCliHandler.doEnterModuleLinecardCmd"

   noOrDefaultHandler = "PlutoCliHandler.noEnterModuleLinecardCmd"

PoeMode.addCommandClass( EnterModuleLinecardCmd )

# -----------------------------------------------------------------------------------
# [no|default] priority (low|medium|high|critical) command,
# in "config-poelinecard" mode
# -----------------------------------------------------------------------------------
class PoeLinecardPriorityCmd( CliCommand.CliCommandClass ):
   syntax = 'priority PRIORITY'
   noOrDefaultSyntax = 'priority ...'
   data = {
      'priority': 'Linecard priority',
      'PRIORITY': matcherPriorityEnum,
   }

   handler = "PlutoCliHandler.doPoeLinecardPriorityCmd"

   noOrDefaultHandler = "PlutoCliHandler.noPoeLinecardPriorityCmd"

PoeLinecardMode.addCommandClass( PoeLinecardPriorityCmd )

# -----------------------------------------------------------------------------------
# Support for default interface command
# -----------------------------------------------------------------------------------
class PoeIntfCleaner( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      if not gv.poeCliConfig or not gv.poeCliConfig.poeCliConfig:
         return
      if self.intf_.name in gv.poeCliConfig.poeCliConfig:
         del gv.poeCliConfig.poeCliConfig[ self.intf_.name ]

IntfCli.Intf.registerDependentClass( PoeIntfCleaner )

# -----------------------------------------------------------------------------------
# Reload Cli hook for poe reboot action processing before reboot
# -----------------------------------------------------------------------------------
def _verifyPoePortStateStatus( intfs, setting ):
   def _checkPoePortState( intfs, setting ):
      for intf in intfs:
         if intf in gv.poeStatus and gv.poeStatus[ intf ]:
            portState = gv.poeStatus[ intf ].portStatus.poePortState
            if portState != setting:
               return False
      return True

   try:
      Tac.waitFor( lambda: _checkPoePortState( intfs, setting ),
                   timeout=30, warnAfter=5.0, maxDelay=5,
                   sleep=not Tac.activityManager.inExecTime.isZero,
                   description='Processing poe reboot action configurations' )
   except Tac.Timeout:
      return False
   return True

def _verifyGlobalPoeRebootActionStatus( setting ):
   if "PoeStatus" in gv.globalPoeStatus and gv.globalPoeStatus[ 'PoeStatus' ]:
      try:
         Tac.waitFor( lambda: gv.globalPoeStatus[ 'PoeStatus' ]
                      .rebootAction == setting,
                      timeout=30, warnAfter=5.0, maxDelay=5,
                      sleep=not Tac.activityManager.inExecTime.isZero,
                      description='Processing poe reboot action configurations' )
      except Tac.Timeout:
         return False
   return True

def doRebootActionBeforeReload( mode, power, now ):
   # If there are no poe ports, don't need to do anything
   if not gv.poeStatus or not gv.poeStatus.entryState:
      return True

   # sso is not considered as a reboot, never power off PoE ports
   if gv.sysdbRedSupStatus.protocol == 'sso':
      return True

   if not _verifyGlobalPoeRebootActionStatus( gv.poeCliConfig.rebootAction ):
      return False

   intfsDisabled = []
   # Make a copy of the global config because it may be modified in
   # the loop
   globalRebootActionConfig = gv.poeCliConfig.rebootAction
   for intf in gv.poeStatus:
      config = gv.poeCliConfig.newPoeCliConfig( intf )
      if globalRebootActionConfig == 'maintain':
         if config.rebootAction == 'powerOff':
            config.poeEnabled = False
            intfsDisabled.append( intf )
      elif globalRebootActionConfig == 'powerOff':
         # pylint: disable-next=consider-using-in
         if config.rebootAction == 'unset' or config.rebootAction == 'powerOff':
            config.poeEnabled = False
            intfsDisabled.append( intf )
         elif config.rebootAction == 'maintain':
            # Set the global register to maintain power, so that ports set to
            # "maintain" stay powered on, this is temporary and will be set to
            # "powerOff" after reload
            if gv.poeCliConfig.rebootAction != 'maintain':
               gv.poeCliConfig.rebootAction = 'maintain'
               if not _verifyGlobalPoeRebootActionStatus( 'maintain' ):
                  return False

   # Only need to check on maintain, since if global config is powerOff then the
   # hardware register is set to power off, and will power off regardless of software
   # settings
   if gv.poeCliConfig.rebootAction == 'maintain':
      if not _verifyPoePortStateStatus( intfsDisabled, 'disabled' ):
         return False
   return True

ReloadCli.registerReloadHook( doRebootActionBeforeReload, 'ReloadPoeRebootActionCli',
                              'RUN_AFTER_CONFIRMATION', always=True )

# -----------------------------------------------------------------------------------
# Register debug cmd into 'show tech-support'
# -----------------------------------------------------------------------------------
CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2014-10-31 14:14:00',
   cmds=[ 'show platform pluto all' ] )

# -----------------------------------------------------------------------------------
# Register Cli Plugin
# -----------------------------------------------------------------------------------
def Plugin( em ):
   gv.sliceDir = LazyMount.mount( em,
         'hardware/platform/device/config/slice', 'Tac::Dir', 'ri' )
   gv.poeTecConfig = LazyMount.mount( em,
         'hardware/poe/config/poetec', 'Hardware::Poe::PoeConfig', 'r' )
   gv.globalPoeStatus = LazyMount.mount( em,
         'environment/power/status/poe', 'Tac::Dir', 'ri' )
   gv.poeStatus = LazyMount.mount( em,
         'environment/power/status/poePort', 'Tac::Dir', 'ri' )
   gv.poeCliConfig = ConfigMount.mount( em,
         'hardware/poe/config/cli', 'Hardware::Poe::CliConfig', 'w' )
   gv.sysdbRedSupStatus = LazyMount.mount( em,
         Cell.path( "redundancy/status" ), "Redundancy::RedundancyStatus", "r" )
   gv.sysdbRedSupConfig = LazyMount.mount( em,
         "redundancy/config", "Redundancy::RedundancyConfig", "r" )
   gv.hwEntMib = LazyMount.mount( em, "hardware/entmib", "EntityMib::Status", "r" )
