#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import CliGlobal
import LazyMount
import Cell
import TableOutput
import Tac
import os
from CliModel import Dict, Enum, Model
import FruPlugin.PLSlot # pylint: disable=unused-import

gv = CliGlobal.CliGlobal(
   cardCliConfigMount=None,
   cardSlotStatusMount=None,
   cardSlotConfigMount=None,
   powerSlotCliConfigMount=None,
   powerSlotStatusMount=None,
   fanSlotConfigMount=None,
   fanSlotStatusMount=None,
   platformCliSlotConfigMount=None,
   platformSlotStatusMount=None )

class SlotStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Hardware::ModularSystem::FruStatus'

   def __init__( self, slotStatus ):
      Tac.Notifiee.__init__( self, slotStatus )
      self.handleState()

   @Tac.handler( 'state' )
   def handleState( self ):
      slot = Slot( self.notifier_.name )
      if not slot.inserted:
         slot.config.powerCycle = False

# Pylint can't handle property decorators
# pylint: disable-msg=E0102,E0202
class Slot:
   def __init__( self, _name ):
      name = _name.replace( 'SwitchcardCes', 'Switchcard' )
      self.name = name
      self.config = None
      self.status = None
      self.slotConfig = None
      self.powerSupply = None
      self.powerCycleEnableChecks = False
      if name.startswith( 'PowerSupply' ):
         shortName = name[ len( 'PowerSupply' ) : ]
         if shortName:
            self.status = gv.powerSlotStatusMount.slotStatus.get( shortName )
            if self.status:
               self.config = gv.powerSlotCliConfigMount.newCliSlotConfig( shortName )
      elif name.startswith( 'Fan' ):
         self.config = gv.fanSlotConfigMount.newSlotConfig( name )
         self.status = gv.fanSlotStatusMount.slotStatus.get( name )
      elif name.startswith( ( 'Linecard', 'Fabric', 'Switchcard' ) ):
         # System may not support these modules
         if gv.cardSlotStatusMount is None:
            return
         if _name.startswith( 'Switchcard' ):
            sysdbSwitchcard = gv.cardSlotStatusMount.slotStatus.get( name )
            if _name.startswith( 'SwitchcardCes' ) or sysdbSwitchcard is None:
               name = name.replace( 'Switchcard', 'SwitchcardCes' )
         self.status = gv.cardSlotStatusMount.slotStatus.get( name )
         self.config = gv.cardCliConfigMount.slotConfig.get( name )
         if not self.config:
            self.config = gv.cardCliConfigMount.newSlotConfig( name )
         self.slotConfig = gv.cardSlotConfigMount.slotConfig.get( name )
      if not self.status and gv.platformSlotStatusMount is not None:
         self.status = gv.platformSlotStatusMount.slotStatus.get( name )
         if not self.status:
            if name.startswith( 'Fan' ):
               name = name.replace( 'Fan', 'FanTray' )
               self.status = gv.platformSlotStatusMount.slotStatus.get( name )
         if self.status:
            self.config = gv.platformCliSlotConfigMount.newCliSlotConfig( name )

   @property
   def inserted( self ):
      return self.status and self.status.state not in [ 'unknown', 'notPresent',
                                                        'notOwned' ]

   @inserted.setter
   def inserted( self, inserted ):
      if inserted:
         try:
            self.config.enterDiagsModeOnFruRemoval = False
         except AttributeError:
            pass
         self.config.diagsMode = False
      else:
         originalValue = False
         try:
            originalValue = self.config.enterDiagsModeOnFruRemoval
            self.config.enterDiagsModeOnFruRemoval = True
         except AttributeError as e:
            raise NotImplementedError from e

         try:
            self.powerCycle()
         except NotImplementedError:
            self.config.enterDiagsModeOnFruRemoval = originalValue
            raise
      if 'SIMULATION_NORCALCARD' not in os.environ:
         Tac.waitFor( lambda: self.inserted == inserted, sleep=True, warnAfter=None,
                      timeout=60 )

   @property
   def diagsMode( self ):
      enterDiagsModeOnFruRemoval = False
      try:
         enterDiagsModeOnFruRemoval = self.config.enterDiagsModeOnFruRemoval
      except AttributeError:
         pass
      return ( not self.inserted and ( self.config.diagsMode or
                                       enterDiagsModeOnFruRemoval ) )

   @diagsMode.setter
   def diagsMode( self, diagsMode ):
      self.config.diagsMode = diagsMode

   @property
   def empty( self ):
      return not self.inserted and not self.diagsMode

   def powerCycle( self ):
      try:
         if self.inserted:
            try:
               self.config.enableChecks = self.powerCycleEnableChecks
               self.config.powerCycle = True
               if 'SIMULATION_NORCALCARD' not in os.environ:
                  _slotStatusReactor = SlotStatusReactor( self.status )
                  Tac.waitFor( lambda: not self.config.powerCycle,
                               sleep=True,
                               warnAfter=None,
                               maxDelay=1,
                               timeout=30 )
            finally:
               if 'SIMULATION_NORCALCARD' not in os.environ:
                  self.config.powerCycle = False
         else:
            raise NotImplementedError
      except AttributeError as e:
         raise NotImplementedError from e

   @property
   def statusString( self ):
      if self.inserted:
         return 'inserted'
      if self.diagsMode:
         return 'diags'
      if self.empty:
         return 'empty'
      return 'unknown'

   def validName( self, mode ):
      if self.config is None:
         mode.addError( f'"{self.name }" is not a valid module name' )
         return False
      return True

   def validNameNotEmpty( self, mode ):
      if not self.validName( mode ):
         return False
      if self.empty:
         mode.addError( f'"{self.name}" is empty' )
         return False
      return True
# pylint: enable-msg=E0102,E0202

# -------------------------------------------------------------------------------
# command to power cycle a module
#    requires "enable" mode, hidden
#
# full syntax:
#    platform module <slot> power-cycle [ enable checks ]
# -------------------------------------------------------------------------------
def doPowercycleModule( mode, args ):
   slot = Slot( args.get( "SLOTNAME" ) )
   slot.powerCycleEnableChecks = 'checks' in args
   if slot.validNameNotEmpty( mode ):
      if slot.diagsMode:
         mode.addError( f'"{slot.name}" '
                       'cannot be power cycled, already in diags mode' )
         return
      try:
         slot.powerCycle()
      except NotImplementedError:
         mode.addError( f'"{slot.name}" not available for power cycle' )

# -------------------------------------------------------------------------------
# command to remove a module
#    simulates physical removal by power cycling and
#        immediately placing in to diags mode ( to prevent rediscovery )
#    requires "enable" mode, hidden
#
# full syntax:
#    platform module <slot> remove [ enable checks ]
# -------------------------------------------------------------------------------
def doRemoveModule( mode, args ):
   slot = Slot( args.get( "SLOTNAME" ) )
   slot.powerCycleEnableChecks = 'checks' in args
   if slot.validNameNotEmpty( mode ):
      if slot.diagsMode:
         mode.addError( f'"{slot.name}" cannot be removed, already in diags mode' )
         return
      try:
         slot.inserted = False
      except NotImplementedError:
         mode.addError( f'"{slot.name}" not available for removal' )
      except Tac.Timeout:
         mode.addError( f'Timed out waiting for "{slot.name}" to be removed' )

# -------------------------------------------------------------------------------
# command to place module in to diags mode
#    requires "enable" mode, hidden
#
# full syntax:
#    platform module <slot> diags
# -------------------------------------------------------------------------------
def doDiagModule( mode, args ):
   slot = Slot( args.get( "SLOTNAME" ) )
   if slot.validName( mode ):
      slot.diagsMode = True

# -------------------------------------------------------------------------------
# command to insert a module, take out of diags
#    does not matter if the module reached diags mode
#        being removed or diagsed ( #stillaword )
#    requires "enable" mode, hidden
#
# full syntax:
#    platform module <slot> insert
# -------------------------------------------------------------------------------
def doInsertModule( mode, args ):
   slot = Slot( args.get( "SLOTNAME" ) )
   if slot.validNameNotEmpty( mode ):
      try:
         slot.inserted = True
      except Tac.Timeout:
         mode.addError( f'Timed out waiting for "{slot.name}" to be inserted' )

# -------------------------------------------------------------------------------
# command to show module(s)
#    requires "enable" mode, hidden
#    displays status of module or that the module's slot is empty
#
# full syntax:
#    show platform module [ <slot> | all ]
# -------------------------------------------------------------------------------
class PlatformModuleStatus( Model ):
   status = Enum( values=( "inserted", "diags", "empty", "unknown" ),
         help='Status of module' )

class PlatformModules( Model ):
   modules = Dict( keyType=str, valueType=PlatformModuleStatus,
         help='Mapping between module name and module status' )

   def render( self ):
      if self.modules:
         tableHeadings = ( "Name", "Status" )
         t = TableOutput.createTable( tableHeadings )
         f = TableOutput.Format( justify="left" )
         f.padLimitIs( True )
         t.formatColumns( *( [ f ] * len( tableHeadings ) ) )
         for label, module in sorted( self.modules.items() ):
            t.newRow( label, module.status )
         print( t.output() )

def doShowPlatformModule( mode, args ):
   modules = {}
   slotName = args.get( "SLOTNAME" )
   if slotName is None or slotName == "all":
      mounts = [ gv.cardSlotStatusMount, gv.fanSlotStatusMount,
                 gv.powerSlotStatusMount, gv.platformSlotStatusMount ]
      slotGroups = [ list( m.slotStatus ) for m in mounts if m is not None ]
      slotKeys = [ key for group in slotGroups for key in group ]
      for slotKey in slotKeys:
         if slotKey in gv.powerSlotStatusMount.slotStatus:
            slotKey = f"PowerSupply{slotKey}"
         slot = Slot( slotKey )
         modules[ slot.name ] = PlatformModuleStatus( status=slot.statusString )
   else:
      slot = Slot( slotName )
      if slot.validName( mode ):
         modules[ slot.name ] = PlatformModuleStatus( status=slot.statusString )

   return PlatformModules( modules=modules )

def Plugin( entityManager ):
   gv.powerSlotCliConfigMount = LazyMount.mount(
      entityManager,
      "hardware/powerSupply/slot/cli/config",
      "Hardware::PowerSupplySlot::CliConfig", "w" )
   gv.powerSlotStatusMount = LazyMount.mount(
      entityManager,
      "hardware/powerSupply/slot/status",
      "Hardware::PowerSupplySlot::Status", "r" )
   gv.fanSlotConfigMount = LazyMount.mount(
      entityManager,
      "hardware/fan/cliconfig",
      "Hardware::Fan::CliConfig", "w" )
   gv.fanSlotStatusMount = LazyMount.mount(
      entityManager,
      "hardware/fan/status",
      "Hardware::Fan::Status", "r" )
   if Cell.cellType() == "supervisor":
      gv.cardCliConfigMount = LazyMount.mount(
         entityManager,
         "hardware/modularSystem/config/cli",
         "Hardware::ModularSystem::CliConfig", "w" )
      gv.cardSlotStatusMount = LazyMount.mount(
         entityManager,
         "hardware/modularSystem/status",
         "Hardware::ModularSystem::Status", "r" )
      gv.cardSlotConfigMount = LazyMount.mount(
         entityManager,
         "hardware/modularSystem/config/slot",
         "Hardware::ModularSystem::Config", "r" )
   elif Cell.cellType() == "fixed":
      gv.platformCliSlotConfigMount = LazyMount.mount(
         entityManager,
         "hardware/platform/slot/cli/config",
         "Hardware::PLSlot::CliConfig", "w" )
      gv.platformSlotStatusMount = LazyMount.mount(
         entityManager,
         "hardware/platform/slot/status",
         "Hardware::PLSlot::Status", "r" )
