# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import BasicCliUtil
import ConfigMount
import LazyMount
import Tac
from CliPlugin import PowerCli
from CliPlugin.PowerCli import formatU16
from CliPlugin.PowerCli import formatU8

DEFAULT_LOW_VOLTAGE_WARNING_THRESHOLD = 0
DEFAULT_HIGH_VOLTAGE_WARNING_THRESHOLD = Tac.Value( "Units::Volts" ).value
DEFAULT_MAX_FAULT = 10
INVALID_COUNT = -1

class PmbusDetail:
   def __init__( self, status, formatStr ):
      self.status = status
      self.detailsDictRep = {}
      self.detailsArrRep = []

      for mfrData in sorted( self.status.mfrData.values(), key=lambda x: x.addr ):
         name = mfrData.name
         value = mfrData.value
         if value:
            try:
               value = value.decode( 'UTF-8' )
            except UnicodeDecodeError:
               first = value[ 0 ]
               if first <= 32: # First byte looks like length.
                  value = value[ 1 : 1 + first ]
               value = value.decode( 'UTF-8', 'ignore' )

            if value:
               self.detailsDictRep[ name ] = value
               self.detailsArrRep.append(
                  formatStr.format(
                     f'{name} ({formatU8( mfrData.addr )})', value ) )

      self.detailsDictRep[ 'OPERATION' ] = formatU8( status.operation )
      self.detailsDictRep[ 'STATUS_WORD' ] = formatU16( status.statusWord )
      if status.dualOutput:
         self.detailsDictRep[ 'STATUS_VOUT_A' ] = formatU8( status.statusVout )
         self.detailsDictRep[ 'STATUS_IOUT_A' ] = formatU8( status.statusIout )
         self.detailsDictRep[ 'STATUS_VOUT_B' ] = formatU8( status.statusVoutB )
         self.detailsDictRep[ 'STATUS_IOUT_B' ] = formatU8( status.statusIoutB )
      else:
         self.detailsDictRep[ 'STATUS_VOUT' ] = formatU8( status.statusVout )
         self.detailsDictRep[ 'STATUS_IOUT' ] = formatU8( status.statusIout )
      if status.dualInput:
         self.detailsDictRep[ 'STATUS_INPUT_A' ] = formatU8( status.statusInput )
         self.detailsDictRep[ 'STATUS_INPUT_B' ] = formatU8( status.statusInputB )
      else:
         self.detailsDictRep[ 'STATUS_INPUT' ] = formatU8( status.statusInput )
      self.detailsDictRep[ 'STATUS_TEMP' ] = formatU8( status.statusTemp )
      self.detailsDictRep[ 'STATUS_CML' ] = formatU8( status.statusCml )
      self.detailsDictRep[ 'STATUS_OTHER' ] = formatU8( status.statusOther )
      self.detailsDictRep[ 'STATUS_MFR' ] = formatU8( status.statusMfr )
      self.detailsDictRep[ 'STATUS_FANS_1_2' ] = formatU8( status.statusFans1_2 )
      self.detailsArrRep.append( formatStr.format( 'OPERATION (0x01)',
                                                   formatU8( status.operation ) ) )
      self.detailsArrRep.append( formatStr.format( 'STATUS_WORD (0x79)',
                                            formatU16( status.statusWord ) ) )
      if status.dualOutput:
         self.detailsArrRep.append( formatStr.format( 'STATUS_VOUT_A (0x7a, Page 0)',
                                               formatU8( status.statusVout ) ) )
         self.detailsArrRep.append( formatStr.format( 'STATUS_IOUT_A (0x7b, Page 0)',
                                               formatU8( status.statusIout ) ) )
         self.detailsArrRep.append( formatStr.format( 'STATUS_VOUT_B (0x7a, Page 1)',
                                               formatU8( status.statusVoutB ) ) )
         self.detailsArrRep.append( formatStr.format( 'STATUS_IOUT_B (0x7b, Page 1)',
                                               formatU8( status.statusIoutB ) ) )
      else:
         self.detailsArrRep.append( formatStr.format( 'STATUS_VOUT (0x7a)',
                                               formatU8( status.statusVout ) ) )
         self.detailsArrRep.append( formatStr.format( 'STATUS_IOUT (0x7b)',
                                               formatU8( status.statusIout ) ) )
      if status.dualInput:
         self.detailsArrRep.append( formatStr.format(
            'STATUS_INPUT_A (0x7c, Page 0)', formatU8( status.statusInput ) ) )
         self.detailsArrRep.append( formatStr.format(
            'STATUS_INPUT_B (0x7c, Page 1)', formatU8( status.statusInputB ) ) )
      else:
         self.detailsArrRep.append( formatStr.format(
            'STATUS_INPUT (0x7c)', formatU8( status.statusInput ) ) )
      self.detailsArrRep.append( formatStr.format( 'STATUS_TEMP (0x7d)',
                                            formatU8( status.statusTemp ) ) )
      self.detailsArrRep.append( formatStr.format( 'STATUS_CML (0x7e)',
                                            formatU8( status.statusCml ) ) )
      self.detailsArrRep.append( formatStr.format( 'STATUS_OTHER (0x7f)',
                                            formatU8( status.statusOther ) ) )
      self.detailsArrRep.append( formatStr.format( 'STATUS_MFR (0x80)',
                                            formatU8( status.statusMfr ) ) )
      self.detailsArrRep.append( formatStr.format( 'STATUS_FANS_1_2 (0x81)',
                                            formatU8( status.statusFans1_2 ) ) )
      for slot, device in pmbusConfig.powerSupply.items():
         if "PowerSupply%s" % slot == status.name and len( device.fan ) >= 3:
            self.detailsDictRep[ 'STATUS_FANS_3_4' ] = \
                                          formatU8( status.statusFans3_4 )
            self.detailsArrRep.append( formatStr.format( 'STATUS_FANS_3_4 (0x82)',
                                      formatU8( status.statusFans3_4 ) ) )
   def getDictRep( self ):
      return self.detailsDictRep

   def getStrRep( self ):
      return '\n'.join( self.detailsArrRep )

def pmbusDetail( status, formatStr ):
   return PmbusDetail( status, formatStr )

PowerCli.registerPowerSpecificInfoFn( 'Hardware::PowerSupply::HwPmbusPowerStatus',
                                      pmbusDetail )

#----------------------------------------------------------------------------
# 'power [PowerSupply#] input voltage warning low|high <volts> [readings <count>]'
#----------------------------------------------------------------------------
def powerSupplyVoltageWarning( mode, args ):
   if args.get( 'POWER_SUPPLY' ):
      slots = args[ 'POWER_SUPPLY' ].values()
      for slot in slots:
         args[ 'POWER_SUPPLY' ] = slot
         powerSupplyVoltageWarningEachPowerSupply( mode, args )
   else:
      for slot in slotConfig.slotConfig:
         powerSupplyName = "PowerSupply%s" % slot
         args[ 'POWER_SUPPLY' ] = powerSupplyName
         powerSupplyVoltageWarningEachPowerSupply( mode, args )

def powerSupplyVoltageWarningEachPowerSupply( mode, args ):
   voltageEnd = args.get( 'HIGH_LOW' )
   powerSupplyName = args.get( 'POWER_SUPPLY' )
   threshold = args.get( 'THRESHOLD' )
   count = args.get( 'COUNT', INVALID_COUNT )
   if powerSupplyName.startswith( 'PowerSupply' ):
      shortName = powerSupplyName[ len( 'PowerSupply' ): ]
      if shortName:
         slotId = int( shortName )
         # if the entity pmbusConfig.powerSupply is empty,
         # command is run as start up config which should always be valid
         if ( shortName in slotConfig.slotConfig ) or slotConfig.slotConfig:
            config = warningCliConfig.newWarningPmbusCliConfig( slotId )
            if ( voltageEnd == 'low' and
                 ( threshold != DEFAULT_LOW_VOLTAGE_WARNING_THRESHOLD or
                   count != DEFAULT_MAX_FAULT ) ):
               config.inputUnderVoltageWarningThreshold = threshold
               if count != INVALID_COUNT:
                  config.maxConsecutiveUnderVinFaults = count
            elif ( voltageEnd == 'high' and
                   ( threshold != DEFAULT_HIGH_VOLTAGE_WARNING_THRESHOLD or
                     count != DEFAULT_MAX_FAULT ) ):
               config.inputOverVoltageWarningThreshold = threshold
               if count != INVALID_COUNT:
                  config.maxConsecutiveOverVinFaults = count
            else:
               del warningCliConfig.warningPmbusCliConfig[ slotId ]
         else:
            mode.addWarning( "PowerSupply%d is invalid" % slotId )

#----------------------------------------------------------------------------
# 'no|default power [PowerSupply#] input voltage warning low|high'
#----------------------------------------------------------------------------
def noPowerSupplyVoltageWarning( mode, args ):
   for slot in slotConfig.slotConfig:
      powerSupplyName = "PowerSupply%s" % slot
      args[ 'POWER_SUPPLY' ] = powerSupplyName
      noPowerSupplyVoltageWarningEachPowerSupply( mode, args )

def noPowerSupplyVoltageWarningEachPowerSupply( mode, args ):
   voltageEnd = args.get( 'HIGH_LOW' )
   args[ 'COUNT' ] = DEFAULT_MAX_FAULT
   if voltageEnd == 'low':
      args[ 'THRESHOLD' ] = DEFAULT_LOW_VOLTAGE_WARNING_THRESHOLD
      powerSupplyVoltageWarningEachPowerSupply( mode, args )
   else:
      args[ 'THRESHOLD' ] = DEFAULT_HIGH_VOLTAGE_WARNING_THRESHOLD
      powerSupplyVoltageWarningEachPowerSupply( mode, args )

def doPowerEnable( mode, args ):
   for slot in args[ 'POWER_SUPPLY' ].values():
      psu = 'PowerSupply%d' % slot
      del powerFuseConfig.powerOffRequested[ psu ]

def doNoPowerEnable( mode, args ):
   activePsus = set()
   for psu in powerSupplyConfig.values():
      if psu.state == 'ok':
         activePsus.add( psu.name )
   psuArgs = set( 'PowerSupply%s' % slot 
                  for slot in args[ 'POWER_SUPPLY' ].values() )
   if activePsus.issubset( psuArgs ):
      warningStr = ( '! The switch will be powered off and require\r\n'
                     'manual recovery. Would you like to proceed? [Y/n] ' )
      confirm = BasicCliUtil.confirm( mode, prompt=warningStr, 
                                      answerForReturn=True )
      if not confirm:
         return

   for psuName in psuArgs:
      powerFuseConfig.powerOffRequested[ psuName ] = 'user configuration'

#--------------------------------------------------
# Plugin method - Mount the objects we need from Sysdb
#--------------------------------------------------
slotConfig = None
pmbusConfig = None
warningCliConfig = None
powerFuseConfig = None
powerSupplyConfig = None

def Plugin( entityManager ):
   global slotConfig
   global pmbusConfig
   global warningCliConfig
   global powerFuseConfig
   global powerSupplyConfig
   slotConfig = LazyMount.mount(
      entityManager,
      'hardware/powerSupply/slot/config',
      'Hardware::PowerSupplySlot::Config', "r" )
   pmbusConfig = LazyMount.mount(
      entityManager,
      'hardware/powerSupply/pmbus11',
      'Hardware::PowerSupply::Pmbus::Config', "r" )
   warningCliConfig = ConfigMount.mount(
      entityManager,
      'hardware/powerSupply/cli/warning',
      'Hardware::PowerSupply::Pmbus::WarningCliConfig', "w" )
   powerFuseConfig = ConfigMount.mount(
      entityManager,
      "power/fuse/config/admin",
      "Power::SoftwareFuse", "w" )
   powerSupplyConfig = LazyMount.mount(
      entityManager,
      'environment/archer/power/status/powerSupply',
      'Tac::Dir', "ri" )
