#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCliModes
import CliCommand
import CliMatcher
import CliPlugin.FruCli as FruCli # pylint: disable=consider-using-from-import
import CliPlugin.PmbusCli as PmbusCli # pylint: disable=consider-using-from-import
import MultiRangeRule

matcherHighLow = CliMatcher.EnumMatcher( {
      'high': 'Power supply high threshold',
      'low': 'Power supply low threshold',
} )
matcherThreshold = CliMatcher.FloatMatcher( 0, float( 'inf' ),
      helpdesc='Threshold for voltage',
      precisionString='%.25g' )
matcherCount = CliMatcher.IntegerMatcher( 0, 100,
      helpdesc='Max number of fault readings' )

class PowerSupplyExpression( CliCommand.CliExpression ):
   expression = 'POWER_SUPPLY'
   data = {}
   _matcher = MultiRangeRule.MultiRangeMatcher(
      noSingletons=False,
      helpdesc='Power Supplies',
      tagLong='PowerSupply',
      rangeFn=lambda: FruCli.rangeFn( 'PowerSupply' ),
   )
   _node = CliCommand.Node( _matcher,
                            guard=FruCli.powerSupplyGuard )
   data[ 'POWER_SUPPLY' ] = _node

class PowerSupplyVoltageWarningCmd( CliCommand.CliCommandClass ):
   syntax = ( 'power [ POWER_SUPPLY ] input voltage warning HIGH_LOW THRESHOLD '
              '[ readings COUNT ]' )
   noOrDefaultSyntax = 'power [ POWER_SUPPLY ] input voltage warning HIGH_LOW ...'
   data = {
      'power': 'Configure power supplies',
      'POWER_SUPPLY': PowerSupplyExpression,
      'input': 'Power supply input',
      'voltage': 'Power supply voltage',
      'warning': 'Power supply warning',
      'HIGH_LOW': matcherHighLow,
      'THRESHOLD': matcherThreshold,
      'readings': 'Fault readings',
      'COUNT': matcherCount,
   }
   handler = PmbusCli.powerSupplyVoltageWarning
   noOrDefaultHandler = PmbusCli.noPowerSupplyVoltageWarning

BasicCliModes.GlobalConfigMode.addCommandClass( PowerSupplyVoltageWarningCmd )

moduleMatcher = CliMatcher.KeywordMatcher( 'module',
                                           helpdesc='Configure module power' )
powerModuleNode = CliCommand.Node( moduleMatcher,
                                   guard=FruCli.powerSupplyGuard )

class PowerModuleCmd( CliCommand.CliCommandClass ):
   syntax = 'power enable module POWER_SUPPLY'
   noOrDefaultSyntax = 'power enable module POWER_SUPPLY'
   data = {
      'power' : 'Configure power supplies',
      'enable' : 'Configure power',
      'module' : powerModuleNode,
      'POWER_SUPPLY' : PowerSupplyExpression,
   }
   handler = PmbusCli.doPowerEnable
   noHandler = PmbusCli.doNoPowerEnable
   defaultHandler = PmbusCli.doPowerEnable

BasicCliModes.GlobalConfigMode.addCommandClass( PowerModuleCmd )
