#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliPlugin import IntfCli
from CliPlugin.QosCliIntfTypes import ethOrLagIntfPrefixes
from CliPlugin.QosCliModel import ( ModePolicingModel, InterfacePolicerAllModel,
                                    InterfacePolicingCounterAllModel )
from CliPlugin.QosCliCommon import ( matcherSize, matcherAdjustment, QosModelet )
# pylint: disable-next=consider-using-from-import
import CliPlugin.QosShowCommands as QosShowCommands
from CliToken.Clear import clearKwNode

import BasicCli
import BasicCliModes
import CliCommand
import CliMatcher
import CliParser
import ConfigMount
import LazyMount
import Plugins
import ShowCommand
import Tracing

__defaultTraceHandle__ = Tracing.Handle( 'PolicingCli' )
t0 = Tracing.trace0

# -----------------------------------------------------------------------------------
# Mount path holders ( Define all mount path holders here )
# -----------------------------------------------------------------------------------
qosInputConfig = None
qosAclHwStatus = None
qosHwStatus = None
bridgingHwCapabilities = None

# -----------------------------------------------------------------------------------
# Name Rules for Interface Policing
# -----------------------------------------------------------------------------------
def getProfileNameRule( mode ):
   if qosInputConfig.policingConfig is None:
      return []
   suggestedCompletions = list( qosInputConfig.policingConfig.profile )
   suggestedCompletions = suggestedCompletions + \
      list( qosInputConfig.policingConfig.policerPktSizeAdjProfile )
   return suggestedCompletions

def getPolicerInstanceNameRule( mode ):
   if qosInputConfig.policingConfig is None:
      return []
   suggestedCompletions = qosInputConfig.policingConfig.policerInstance
   return suggestedCompletions

# -----------------------------------------------------------------------------------
# Guards for Interface Policing
# -----------------------------------------------------------------------------------
def guardPolicingMode( mode, token ):
   if qosAclHwStatus.interfacePolicingSupported or \
      qosHwStatus.perPortPolicerPacketSizeAdjSupported or \
      qosAclHwStatus.interfaceControlPlanePolicingDedicatedPolicerSupported or \
      qosAclHwStatus.interfaceEgressPolicingDedicatedPolicerSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardIntfDedicatedPolicer( mode, token ):
   if mode.intf.isSubIntf() and \
      qosAclHwStatus.interfacePolicingDedicatedPolicerSupported and \
      ( qosAclHwStatus.interfacePolicingSupported or
        qosAclHwStatus.interfaceControlPlanePolicingDedicatedPolicerSupported or
        qosAclHwStatus.interfaceEgressPolicingDedicatedPolicerSupported ):
      return None
   elif ( not mode.intf.isSubIntf() and
          qosAclHwStatus.interfacePolicingDedicatedPolicerSupported and
          qosAclHwStatus.interfacePolicingSupported and
          qosAclHwStatus.interfacePolicingOnFppSupported ):
      return None
   else:
      return CliParser.guardNotThisPlatform

def guardIntfGroupPolicer( mode, token ):
   if mode.intf.isSubIntf() and \
      qosAclHwStatus.interfacePolicingGroupPolicerSupported and \
      qosAclHwStatus.interfacePolicingSupported:
      return None
   elif ( not mode.intf.isSubIntf() and
         qosAclHwStatus.interfacePolicingGroupPolicerSupported and
         qosAclHwStatus.interfacePolicingSupported and
         qosAclHwStatus.interfacePolicingGroupPolicerOnFppSupported ):
      return None
   else:
      return CliParser.guardNotThisPlatform

def guardIntfPolicing( mode, token ):
   if guardIntfDedicatedPolicer( mode, token ) and \
         guardIntfGroupPolicer( mode, token ):
      return CliParser.guardNotThisPlatform
   return None

def guardIntfCoppPolicing( mode, token ):
   if not qosAclHwStatus.interfaceControlPlanePolicingDedicatedPolicerSupported:
      return CliParser.guardNotThisPlatform
   return None

def guardIntfPolicerInputDir( mode, token ):
   if not qosAclHwStatus.interfacePolicingDedicatedPolicerSupported:
      return CliParser.guardNotThisPlatform
   return None

def guardIntfPolicerOutputDir( mode, token ):
   if not qosAclHwStatus.interfaceEgressPolicingDedicatedPolicerSupported:
      return CliParser.guardNotThisPlatform
   return None

# Show counters guard
def guardIntfPolicingCounters( mode, token ):
   if ( guardPolicingMode( mode, token ) or
        not ( qosAclHwStatus.interfacePolicingCountersSupported or
              qosAclHwStatus.interfaceEgressPolicingCountersSupported or
              qosAclHwStatus.interfaceControlPlanePolicingCountersSupported ) ):
      return CliParser.guardNotThisPlatform
   return None

def guardIntfPolicerPacketSizeAdj( mode, token ):
   if qosHwStatus.perPortPolicerPacketSizeAdjSupported and \
      not mode.intf.isSubIntf():
      return None
   return CliParser.guardNotThisPlatform

def guardShowPolicingDirection( mode, token ):
   if token == 'input' and qosAclHwStatus.interfacePolicingDedicatedPolicerSupported:
      return None
   if token == 'output' and \
      qosAclHwStatus.interfaceEgressPolicingDedicatedPolicerSupported:
      return None
   if token == 'cpu' and \
      qosAclHwStatus.interfaceControlPlanePolicingDedicatedPolicerSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardPolicingCountersDirection( mode, token ):
   if token == 'input' and qosAclHwStatus.interfacePolicingCountersSupported:
      return None
   if token == 'output' and qosAclHwStatus.interfaceEgressPolicingCountersSupported:
      return None
   if token == 'cpu' and \
      qosAclHwStatus.interfaceControlPlanePolicingCountersSupported:
      return None
   return CliParser.guardNotThisPlatform

# -------------------------------------------------------------------------------
# Utility Functions
# -------------------------------------------------------------------------------
def policerPktSizeAdjRangeFn( mode, context ):
   return ( 1, qosHwStatus.maxPolicerPktSizeAdjBytes )


# -------------------------------------------------------------------------------
# Tokens for Policing Mode, Interface-Policers in SubIntf Mode and Show Policing
# -------------------------------------------------------------------------------
nodePolicingMode = CliCommand.guardedKeyword( 'policing',
      helpdesc='Policing Configuration', guard=guardPolicingMode )
nodeProfileName = CliMatcher.DynamicNameMatcher( getProfileNameRule,
      helpdesc="Policer profile name",
   pattern=r'(?!counters$|interface$|summary$|group$|input$|'
           r'default-pkt-size-adj-profile$)[A-Za-z0-9_:{}\[\]-]+' )
nodePolicerName = CliMatcher.DynamicNameMatcher( getPolicerInstanceNameRule,
      helpdesc="Policer name",
   pattern=r'(?!counters$|interface$|summary$|group$|input$)[A-Za-z0-9_:{}\[\]-]+' )
nodeIntfPolicer = CliCommand.guardedKeyword( 'policer',
      helpdesc="Policer configuration on the interface", guard=guardIntfPolicing )
nodeIntfPolicerTypeCopp = CliCommand.guardedKeyword( 'cpu',
      helpdesc="Apply policing to CPU-bound packets",
      guard=guardIntfCoppPolicing,
      alias="TYPE" )
nodeIntfProfile = CliCommand.guardedKeyword( 'profile',
      helpdesc="Associate interface with a policer profile",
      guard=guardIntfDedicatedPolicer )
nodeIntfGroupPolicer = CliCommand.guardedKeyword( 'group',
      helpdesc="Associate interface with a group policer",
      guard=guardIntfGroupPolicer )
nodeInput = CliCommand.Node(
      CliMatcher.KeywordMatcher( 'input',
      helpdesc="Apply policing to inbound packets" ),
      alias="TYPE" )
nodeOutput = CliCommand.guardedKeyword( 'output',
      helpdesc="Apply policing to output packets",
      guard=guardIntfPolicerOutputDir,
      alias="TYPE" )
nodeShowPolicing = CliCommand.guardedKeyword( 'policing',
      helpdesc='Show policing configuration', guard=guardPolicingMode )
nodeShowPolicingDirection = CliCommand.Node(
      matcher=CliMatcher.EnumMatcher( {
         'input': 'For ingress direction',
         'output': 'For egress direction',
         'cpu': 'For CPU-bound direction', } ),
      guard=guardShowPolicingDirection )
nodeClearPolicing = CliCommand.guardedKeyword( 'policing',
      helpdesc='Clear policing counters', guard=guardIntfPolicingCounters )
nodePolicingCounters = CliCommand.guardedKeyword( 'counters',
      helpdesc='Counters', guard=guardIntfPolicingCounters )
nodePolicingCountersDirection = CliCommand.Node(
      matcher=CliMatcher.EnumMatcher( {
         'input': 'For ingress direction',
         'output': 'For egress direction',
         'cpu': 'For CPU-bound direction', } ),
      guard=guardPolicingCountersDirection )
nodeIntfPktSizeAdjPolicer = CliCommand.guardedKeyword( 'policer',
      helpdesc="Policer configuration on the interface",
      guard=guardIntfPolicerPacketSizeAdj )
nodeIntfPktSizeAdjProfile = CliCommand.guardedKeyword( 'profile',
      helpdesc="Associate interface with a policer packet size adjustment profile",
      guard=guardIntfPolicerPacketSizeAdj )
nodeIntfPacket = CliCommand.guardedKeyword( 'packet',
      helpdesc='Configure packet size parameters',
      guard=guardIntfPolicerPacketSizeAdj )

# -----------------------------------------------------------------------------------
# Interface Policer Modelet
# -----------------------------------------------------------------------------------
class IntfPolicerModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.name.startswith( ethOrLagIntfPrefixes )


IntfCli.IntfConfigMode.addModelet( IntfPolicerModelet )
# -----------------------------------------------------------------------------------
# Cli commands for Interface Policing
# -----------------------------------------------------------------------------------

# -----------------------------------------------------------------------------------
# [ no | default ] policer profile PROFILE group POLICER input | output | cpu
# -----------------------------------------------------------------------------------
class InterfacePolicerConfig( CliCommand.CliCommandClass ):
   syntax = ( 'policer (([profile PROFILE] group POLICER) | (profile PROFILE)) '
            '(input | output | cpu)' )
   noOrDefaultSyntax = syntax
   data = {
      'policer': nodeIntfPolicer,
      'profile': nodeIntfProfile,
      'PROFILE': nodeProfileName,
      'group': nodeIntfGroupPolicer,
      'POLICER': nodePolicerName,
      'input': nodeInput,
      'output': nodeOutput,
      'cpu': nodeIntfPolicerTypeCopp,
   }

   handler = "PolicingCli.doInterfacePolicerConfig"

   noOrDefaultHandler = "PolicingCli.noInterfacePolicerConfig"


IntfPolicerModelet.addCommandClass( InterfacePolicerConfig )

# -----------------------------------------------------------------------------------
# Show CLI commands
# -----------------------------------------------------------------------------------

# -----------------------------------------------------------------------------------
# show policing [ ( profile [ profileName ] ) | ( group [ groupName ] ) |
# ( packet size adjustment profile [ profileName ] ) ]
# -----------------------------------------------------------------------------------
class ShowPolicingPolicerProfile( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show policing [ ( profile [ PROFILE ] ) | ( group [ POLICER ] ) | '
              '( packet size adjustment profile [ PROFILE ] ) ]' )
   data = {
      'policing': nodeShowPolicing,
      'profile': 'Show policer profile',
      'PROFILE': nodeProfileName,
      'group': 'Show group policer',
      'POLICER': nodePolicerName,
      'packet': 'Show packet size adjustment profiles',
      'size': 'Show packet size',
      'adjustment': 'Show packet size adjustment'
   }

   handler = 'PolicingCli.showPolicingHandler'
   cliModel = ModePolicingModel


BasicCli.addShowCommandClass( ShowPolicingPolicerProfile )

# -----------------------------------------------------------------------------------
# [ no | default ] policer packet size adjustment profile PROFILE input
# -----------------------------------------------------------------------------------
class InterfacePolicerPacketSizeAdjConfig( CliCommand.CliCommandClass ):
   syntax = 'policer packet size adjustment profile PROFILE input'
   noOrDefaultSyntax = 'policer packet size adjustment ...'
   data = {
      'policer': nodeIntfPktSizeAdjPolicer,
      'packet': nodeIntfPacket,
      'size': matcherSize,
      'adjustment': matcherAdjustment,
      'profile': nodeIntfPktSizeAdjProfile,
      'PROFILE': nodeProfileName,
      'input': nodeInput,
   }

   handler = 'PolicingCli.setIntfPolicerPacketSizeAdj'
   noOrDefaultHandler = 'PolicingCli.noIntfPolicerPacketSizeAdj'


QosModelet.addCommandClass( InterfacePolicerPacketSizeAdjConfig )
# -----------------------------------------------------------------------------------
# show policing interfaces [ INTFNAME ] [ ( input | output | cpu ) ] [ detail ]
# -----------------------------------------------------------------------------------

class ShowPolicingIntfPolicer( ShowCommand.ShowCliCommandClass ):
   syntax = 'show policing interfaces [ INTF ] [ TYPE ] [ detail ]'
   data = {
      'policing': nodeShowPolicing,
      'interfaces': QosShowCommands.matcherInterfaces,
      'INTF': QosShowCommands.matcherIntf,
      'TYPE': nodeShowPolicingDirection,
      'detail': 'More comprehensive output'
   }

   handler = 'PolicingCli.showPolicingIntfHandler'
   cliModel = InterfacePolicerAllModel


BasicCli.addShowCommandClass( ShowPolicingIntfPolicer )

# -----------------------------------------------------------------------------------
# show policing interfaces [ INTFNAME ] counters [ ( input | output | cpu ) ]
# -----------------------------------------------------------------------------------
class ShowPolicingIntfPolicerCounters( ShowCommand.ShowCliCommandClass ):
   syntax = 'show policing interfaces [ INTF ] counters [ TYPE ]'
   data = {
      'policing': nodeShowPolicing,
      'interfaces': QosShowCommands.matcherInterfaces,
      'INTF': QosShowCommands.matcherIntf,
      'counters': nodePolicingCounters,
      'TYPE': nodePolicingCountersDirection
   }

   handler = 'PolicingCli.showPolicingIntfCountersHandler'
   cliModel = InterfacePolicingCounterAllModel


BasicCli.addShowCommandClass( ShowPolicingIntfPolicerCounters )

# -----------------------------------------------------------------------------------
# Clear CLI commands
# -----------------------------------------------------------------------------------

# -----------------------------------------------------------------------------------
# clear policing interfaces [ INTFNAME ] counters [ ( input | output | cpu ) ]
# -----------------------------------------------------------------------------------
class ClearPolicingIntfPolicerCounters( CliCommand.CliCommandClass ):
   syntax = 'clear policing interfaces [ INTF ] counters [ TYPE ]'
   data = {
      'clear': clearKwNode,
      'policing': nodeClearPolicing,
      'interfaces': QosShowCommands.matcherInterfaces,
      'INTF': QosShowCommands.matcherIntf,
      'counters': nodePolicingCounters,
      'TYPE': nodePolicingCountersDirection
   }

   handler = 'PolicingCli.clearPolicingIntfCountersHandler'


BasicCliModes.EnableMode.addCommandClass( ClearPolicingIntfPolicerCounters )

# -----------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
# -----------------------------------------------------------------------------------
@Plugins.plugin( provides=( "QosPolicingCli", ) )
def Plugin( entityManager ):
   global qosInputConfig, qosAclHwStatus, qosHwStatus, bridgingHwCapabilities

   qosInputConfig = ConfigMount.mount( entityManager, "qos/input/config/cli",
                                       "Qos::Input::Config", "w" )
   qosAclHwStatus = LazyMount.mount( entityManager,
                        "qos/hardware/acl/status/global", "Qos::AclHwStatus", "r" )
   qosHwStatus = LazyMount.mount( entityManager,
                        "qos/hardware/status/global", "Qos::HwStatus", "r" )
   bridgingHwCapabilities = LazyMount.mount( entityManager,
                                             "bridging/hwcapabilities",
                                             "Bridging::HwCapabilities", "r" )
