#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliPlugin.RoutingLibCliModels import (
   NlriModel,
   createNlriModel,
)
from CliPlugin.RcfDebugModels import (
   RcfDebugEvaluation, RcfCodeUnitText, RcfDebugResult
)
from CliPlugin.RouteMapCliModels import PolicyEvalRouteMap
from CliPlugin.RouteMapCli import createPolicyEvalRouteMapModel
from ArnetModel import IpGenericAddress
from CliModel import (
   Dict,
   Enum,
   Model,
   Str,
   Submodel,
   List,
)
from operator import attrgetter

class RcfDebugStateOutOfSync( Exception ):
   '''
   Used when constructing the CAPI model if the following data points
   are not in sync:
   - RCF text
   - Evaluated AET
   - Debug Symbols
   '''
   pass

# The clients do not need to depend on the rendering code.
# This is provided by the Rcf package which will hook in the rendering
# implementation here. Only the types will be stored in RcfLib.
def rcfDebugRenderHook( rcfEval, rcfCodeUnits ):
   print( "RCF debug rendering hook" )

def routeMapDebugRenderHook( routeMapEval, peerApplication ):
   routeMapEval.renderMap( peerApplication=peerApplication )

# The clients should not depend on RCF and its CliPlugin is where the DebugSymbols
# are mounted.
# This is provided by the Rcf package which will hook in the model construction
# implementation here.
def rcfDebugModelHook( evaluationLog, codeUnitsModel ):
   # Return a tuple to satisfy pylint.
   return( "RCF debug evaluation model hook", 0 )

def isPeerPointOfApplication( name ):
   return name in [ "inbound", "outbound" ]

def isImportExportPointOfApplication( name ):
   return name in [ "import", "export" ]

#------------------------------------------------------------------------------------
# 'show bgp debug policy {inbound|outbound}'
#
# Example CAPI format
# {
#     "evalResults": [
#         {
#             "nlri": {
#                "ipPrefix": "10.1.2.3/32"
#             },
#             "pointOfApplication": {
#                 "name": "inbound",
#                 "peerAddr": "1.1.1.1",
#             },
#             "routeMapEval|rcfEval": { ... },
#         },
#         ...
#     ],
#     "rcfCodeUnits": { ... },
# }
#
# 'show bgp debug policy redistribute'
#
# Example CAPI format
# {
#     "evalResults": [
#         {
#             "nlri": {
#                "ipPrefix": "10.1.2.3/32"
#             },
#             "pointOfApplication": {
#                 "name": "redistribute",
#                 "redistributionSource": "Connected",
#             },
#             "routeMapEval|rcfEval": { ... },
#         },
#         ...
#     ]
#     "rcfCodeUnits": { ... },
# }
#
# 'show bgp debug policy import'
#
# Example CAPI format
# {
#     "evalResults": [
#         {
#             "nlri": {
#                "ipPrefix": "10.1.2.3/32"
#             },
#             "pointOfApplication": {
#                 "name": "import",
#                 "vpnAfiSafi": "ipv4MplsVpn",
#             },
#             "routeMapEval|rcfEval": { ... },
#         },
#         ...
#     ]
#     "rcfCodeUnits": { ... },
# }
#
# 'show bgp debug policy export'
#
# Example CAPI format
# {
#     "evalResults": [
#         {
#             "nlri": {
#                "ipPrefix": "10.1.2.3/32"
#             },
#             "pointOfApplication": {
#                 "name": "export",
#                 "vpnAfiSafi": "ipv4MplsVpn",
#             },
#             "routeMapEval|rcfEval": { ... },
#         },
#         ...
#     ]
#     "rcfCodeUnits": { ... },
# }

# 'show bgp debug policy network'
#
# Example CAPI format
# {
#     "evalResults": [
#         {
#             "nlri": {
#                "ipPrefix": "10.1.2.3/32"
#             },
#             "pointOfApplication": {
#                 "name": "network",
#                 "networkSource": "connected",
#             },
#             "rcfEval": { ... },
#         },
#         ...
#     ]
#     "rcfCodeUnits": { ... },
# }
# -----------------------------------------------------------------------------------
class PolicyEvalPointOfApplication( Model ):
   """
   Store the point of application information for a particular policy evaluation
   """
   name = Enum( values=( "inbound", "outbound", "redistribute", "import", "export",
                         "network" ), help="Name of the point of application" )
   peerAddr = IpGenericAddress( optional=True, help="Peer address of application" )
   redistributionSource = Str( optional=True, help="Source of redistributed routes" )
   networkSource = Str( optional=True,
                        help="Source of routes injected with Network statement" )
   vpnAfiSafi = Enum( values=( "ipv4MplsVpn", "ipv6MplsVpn", "evpn", ),
                      optional=True, help="AfiSafi of VPN routes" )

   def renderPointOfApplication( self ):
      if self.name == "inbound":
         assert self.peerAddr is not None
         return "received from %s" % self.peerAddr
      elif self.name == "outbound":
         assert self.peerAddr is not None
         return "to %s" % self.peerAddr
      elif self.name == "redistribute":
         assert self.redistributionSource is not None
         return "redistributed from %s" % self.redistributionSource
      elif isImportExportPointOfApplication( self.name ):
         assert self.vpnAfiSafi is not None
         if self.vpnAfiSafi == 'ipv4MplsVpn':
            vpnAfiSafi = 'VPN-IPv4'
         elif self.vpnAfiSafi == 'ipv6MplsVpn':
            vpnAfiSafi = 'VPN-IPv6'
         else:
            vpnAfiSafi = 'EVPN'
         if self.name == "import":
            return "imported from %s" % vpnAfiSafi
         else:
            return "exported to %s" % vpnAfiSafi
      elif self.name == "network":
         assert self.networkSource is not None
         return f"network statement ({self.networkSource})"
      else:
         assert False, "Unexpected point of application"
         return "" # make pylint happy

class PolicyEvalResult( Model ):
   """
   Stores the policy evaluation results for a particular peer, redist source or
   imported/exported route
   """

   nlri = Submodel( NlriModel, help="NLRI of the evaluated path" )
   pointOfApplication = Submodel( PolicyEvalPointOfApplication,
                           help="Point of application evaluated" )
   routeMapEval = Submodel( PolicyEvalRouteMap, optional=True,
                            help="Route map evaluated" )
   rcfEval = Submodel( valueType=RcfDebugEvaluation, optional=True,
                       help="Debug information for an RCF evaluation" )
   messages = List( valueType=str, optional=True,
                   help="Message describing why policy evaluation "
                   "was not returned" )

   def renderResult( self, rcfCodeUnits=None ):
      print( "Path with NLRI %s, %s\n" %
                ( self.nlri.renderNlri(),
                  self.pointOfApplication.renderPointOfApplication() ) )
      if self.routeMapEval:
         peerApplication = isPeerPointOfApplication( self.pointOfApplication.name )
         routeMapDebugRenderHook( self.routeMapEval, peerApplication )
      elif self.rcfEval:
         rcfDebugRenderHook( self.rcfEval, rcfCodeUnits )
      elif self.messages:
         for message in self.messages:
            print( message )

class PolicyEvalResults( Model ):
   """
   Stores the policy evaluation results for a given NLRI across all peers,
   redistribution, originated and import sources, export destinations
   """
   __revision__ = 2

   evalResults = List( valueType=PolicyEvalResult,
                 help="Evaluation result for each NLRI" )
   rcfCodeUnits = Dict( keyType=str, valueType=RcfCodeUnitText, optional=True,
                        help="Reference to the text of RCF functions where "
                        "evaluation occurred in each code unit, keyed by code "
                        "unit name" )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         # Grab the NLRI and point of application information from the first entry
         if dictRepr[ 'evalResults' ]:
            firstEvalResult = dictRepr[ 'evalResults' ][ 0 ]
            dictRepr[ 'nlri' ] = firstEvalResult[ 'nlri' ].get( 'ipPrefix', '' )
            dictRepr[ 'application' ] = (
               firstEvalResult[ 'pointOfApplication' ][ 'name' ] )
         else:
            # This should never happen but we don't want to assert during the degrade
            dictRepr[ 'nlri' ] = ""
            dictRepr[ 'application' ] = "redistribute" # arbitrary enum choice

         peers = {}
         sources = {}
         # For each eval result:
         # - Determine if it is a peer eval or a redist eval
         # - Get the eval result key from the pointOfApplication info
         # - Prune the 'nlri' and 'pointOfApplication' fields, they are already in
         #      the top level of the degraded model
         # - Insert the pruned eval result in the peer or sources (redist) dict
         for evalResult in dictRepr[ 'evalResults' ]:
            if isPeerPointOfApplication(
                  evalResult[ 'pointOfApplication' ][ 'name' ] ):
               peerAddr = evalResult[ 'pointOfApplication' ].get( 'peerAddr',
                                                                  None )
               if peerAddr is not None:
                  del evalResult[ 'nlri' ]
                  del evalResult[ 'pointOfApplication' ]
                  peers[ peerAddr ] = evalResult
            elif evalResult[ 'pointOfApplication' ][ 'name' ] == 'redistribute':
               redistributionSource = ( evalResult[ 'pointOfApplication' ]
                                           .get( 'redistributionSource', None ) )
               if redistributionSource is not None:
                  del evalResult[ 'nlri' ]
                  del evalResult[ 'pointOfApplication' ]
                  sources[ redistributionSource ] = evalResult
            else:
               # We don't know how to degrade points of application other than
               # 'inbound', 'outbound' and 'redistribute'
               continue
         del dictRepr[ 'evalResults' ]
         dictRepr[ 'peers' ] = peers
         dictRepr[ 'sources' ] = sources
      return dictRepr

   def render( self ):
      first = True

      for result in sorted( self.evalResults,
                            key=attrgetter( 'pointOfApplication.peerAddr',
                                          'pointOfApplication.redistributionSource',
                                          'pointOfApplication.networkSource',
                                          'pointOfApplication.vpnAfiSafi' ) ):
         if not first:
            # Add two lines before printing a new peer/source/vpnAfiSafi
            print( "\n" )
         first = False
         result.renderResult( self.rcfCodeUnits )

def createPointOfApplicationModel( name, result ):
   attrs = { 'name': name }
   if isPeerPointOfApplication( name ):
      attrs[ 'peerAddr' ] = result[ 'peerAddr' ]
   elif name == 'redistribute':
      attrs[ 'redistributionSource' ] = result[ "source" ]
   elif name == 'network':
      attrs[ 'networkSource' ] = result[ "source" ]
   elif isImportExportPointOfApplication( name ):
      attrs[ 'vpnAfiSafi' ] = result[ "vpnAfiSafi" ]
   return PolicyEvalPointOfApplication( **attrs )

def createPolicyEvalModel( results, nlriType, name, nlriAttrs ):
   perEntryResults = []
   codeUnitsModel = None
   for result in results:
      routeMapEval = None
      rcfEval = None
      messages = result.get( 'message', None )
      if messages:
         messages = [ messages ]

      if result.get( 'policyType' ) == 'rcf':
         if 'log' in result:
            rcfEval, codeUnitsModel = rcfDebugModelHook( result[ 'log' ],
                                                         codeUnitsModel )
         else:
            # The RCF function must be undefined or referencing undefined policy.
            funcName = result[ 'entryPoint' ]
            reason = f'RCF function {funcName} is not available.'
            rcfEval = RcfDebugEvaluation(
               entryPoint=funcName,
               result=RcfDebugResult( value='false', reason=reason ) )
      else:
         routeMapEval = createPolicyEvalRouteMapModel( result )

      # If there is an rd value in results, we need to update the export nlriAttrs
      # This is because we do not pass through a rd value in the command as we get
      # the rd from the vrfConfig.
      # We do this per results as there is a chance that two paths can have different
      # rd values. This only happens for the export main function POA.
      if name == 'export' and 'rd' in result:
         nlriAttrs[ 'rd' ] = result[ 'rd' ]
      attrs = {
         'nlri': createNlriModel( nlriType, nlriAttrs ),
         'pointOfApplication': createPointOfApplicationModel( name, result ),
         'routeMapEval': routeMapEval,
         'rcfEval': rcfEval,
         'messages': messages,
      }
      perEntryResults.append( PolicyEvalResult( **attrs ) )

   return PolicyEvalResults( evalResults=perEntryResults,
                             rcfCodeUnits=codeUnitsModel )
