#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import os
from CliPlugin.PolicyMapModel import ( ActionBase,
                                       ActionSetModel,
                                       ClassMapCounter,
                                       ClassMapModel,
                                       ClassMatchAclModel,
                                       ClassMatchModel,
                                       PolicyClassMapModel,
                                       PolicyMapModel,
                                       )
from CliPlugin import AclCliModelImpl
from CliPlugin.PolicyMapModel import ClassMapMatchOption
from CliPlugin.PolicyMapModel import MplsRule
import Tac
import TacSigint
import collections
import AclLib

tacClassMapStatusType = Tac.Type( "PolicyMap::ClassMapStatus" )
tacActionType = Tac.Type( "PolicyMap::ActionType" )
ArnetVlanIntfId = Tac.Type( "Arnet::VlanIntfId" )
PolicyMapCountersKey = Tac.Type( "PolicyMap::Counters::PolicyMapKey" )
AceCheckpointStatus = Tac.Type( "PolicyMap::AceCheckpointStatus" )
UniqueId = Tac.Type( "Ark::UniqueId" )

counterUpdateTimeout = 10
mapTypeToStatusFilter_ = {}

class PolicyClassMapBase:
   statusClasses_ = {}
   @staticmethod
   def addStatusType( statusType, clazz ):
      PolicyClassMapBase.statusClasses_[ statusType ] = clazz

   @classmethod
   def createModel( cls, name, statuses ):
      clsType = None
      if statuses is not None:
         if hasattr( statuses, '__iter__' ):
            clsType = statuses[0].__class__ if len( statuses ) else None
         else:
            clsType = statuses.__class__

      clazz = PolicyClassMapBase.statusClasses_.get( clsType )
      if not clazz:
         # There is no class registered for this type. Just use the default.
         model = PolicyClassMapModel()
      else:
         model = clazz.createModel( statuses )
      model.name = name
      return model

def registerCmapStatusFilter( mapType, filterFunc ):
   mapTypeToStatusFilter_[ mapType ] = filterFunc

def getCmapStatusFilter( mapType ):
   statusFilter = mapTypeToStatusFilter_.get( mapType, None )
   if not statusFilter:
      # pylint: disable-next=unidiomatic-typecheck
      statusFilter = ( lambda x: type( x ) != tacClassMapStatusType )
      registerCmapStatusFilter( mapType, statusFilter )

   return statusFilter

def aclTypetoOption( aclType ): # pylint: disable=inconsistent-return-statements
   if aclType == AclCliModelImpl.AclType.ip:
      return ClassMapMatchOption.matchIpAccessGroup
   elif aclType == AclCliModelImpl.AclType.ipv6:
      return ClassMapMatchOption.matchIpv6AccessGroup
   elif aclType == AclCliModelImpl.AclType.mac:
      return ClassMapMatchOption.matchMacAccessGroup
   else:
      assert True

# clear checkpoint for a policymap in every agentCheckPoint
def clearPolicyMapCheckpoint( policyName, checkpoint ):
   if checkpoint is None or not checkpoint.checkpointAgentStatus:
      return 
   for agentCheckpoint in checkpoint.checkpointAgentStatus.values():
      if policyName in agentCheckpoint.pmapCkptStatus:
         del agentCheckpoint.pmapCkptStatus[ policyName ]


# pylint: disable-next=inconsistent-return-statements
def clearPolicyMapCounters( mode, policyName, cliConfig, mergedConfig,
                            status, checkpoint, counterInShark=False ):

   pmapCfg = mergedConfig.pmapType.pmap.get( policyName )
   if not pmapCfg:
      return False
   pmap = pmapCfg.currCfg
   if not pmap:
      return False

   clearPolicyMapCheckpoint( policyName, checkpoint )
   if counterInShark:
      # status from Shark is a dictionary keyed by platform
      counterStatus = status.values()
      countersAreStale = not waitForPmapCounters( policyName,
                                                  cliConfig.pmapType,
                                                  counterStatus,
                                                  sharkEnabled=True )
   else:
      pmapStatuses = [ pmapTypeStatus.status[ policyName ] for pmapTypeStatus in
                       status.values() if policyName in pmapTypeStatus.status ]

      countersAreStale = not waitForPmapCounters( policyName, cliConfig.pmapType,
                                                pmapStatuses )

   if countersAreStale:
      mode.addWarning( 'Using stale counters' )

   # create a new checkpoint for the policy in each agentCheckpoint
   # agentName same as pmapTypeStatus.name
   for s in status:
      checkpoint.checkpointAgentStatus.newMember( s )

   def updatePktHitCounterFromShark( subclassStatus, ckptClassMapStatus ):
      # subclassStatus is CountersData from one classCounter element
      aclTypeCkptStatus = ckptClassMapStatus.aclTypeCkptStatus
      classMapCountData = subclassStatus.countData
      ckptClassMapStatus.pktMatch += classMapCountData.pktHits
      ckptClassMapStatus.byteMatch += classMapCountData.byteHits
      ckptClassMapStatus.version = subclassStatus.counterVersion
      aclTypeCkptStatus = ckptClassMapStatus.aclTypeCkptStatus
      for aclName, aclCounter in classStatus.aclCounterData.items():
         aclType = aclCounter.aclType
         ckpt = aclTypeCkptStatus[ aclType ].acl
         dest = ckpt[ aclName ]
         dest.version = aclCounter.version
         for r, v in aclCounter.ace.items():
            if r not in dest.aceStatus:
               value = AceCheckpointStatus( v.packetCount, v.byteCount,
                                            v.version )
               dest.aceStatus[ r ] = value
            else:
               pkts = dest.aceStatus[ r ].pkts + v.packetCount
               pbytes = dest.aceStatus[ r ].bytes + v.byteCount
               value = AceCheckpointStatus( pkts, pbytes, v.version )
               dest.aceStatus[ r ] = value

   def updatePktHitCounter( subclassStatus, ckptClassMapStatus ):
      aclTypeCkptStatus = ckptClassMapStatus.aclTypeCkptStatus
      ruleStatus = subclassStatus.ruleStatus
      ckptClassMapStatus.pktMatch += ruleStatus.pkts
      ckptClassMapStatus.byteMatch += ruleStatus.bytes

      # copy the id also to checkpoint so that it can be used
      # to identify stale class-map counters in the checkpoint
      ckptClassMapStatus.version = subclassStatus.id

      # Iterate through each acl and copy the packet hit counter to 
      # checkpoint
      for aclType in AclLib.aclTypes:
         option = aclTypetoOption( aclType )
         cmapMatchStatus = subclassStatus.match.get( option )
         ckpt = aclTypeCkptStatus[ aclType ].acl
         if not cmapMatchStatus:
            continue
         for acl in cmapMatchStatus.acl.values():
            dest = ckpt[ acl.name ]
            dest.version = acl.version
            # copy counters into status
            for r, v in acl.ruleStatus.items():
               if r not in dest.ruleStatus:
                  value = Tac.Value( "Acl::RuleCheckpointStatus", pkts=v.pkts,
                                      bytes=v.bytes )
                  dest.ruleStatus[ r ] = value
               else:
                  # Individual attributes of value types can not be modified
                  pkts = dest.ruleStatus[ r ].pkts + v.pkts
                  pbytes = dest.ruleStatus[ r ].bytes + v.bytes
                  value = Tac.Value( "Acl::RuleCheckpointStatus", pkts=pkts,
                                      bytes=pbytes )
                  dest.ruleStatus[ r ] = value


   # Update every agents checkpoint independently.
   # pylint: disable=too-many-nested-blocks
   for agentCheckpoint in checkpoint.checkpointAgentStatus.values():
      agentName = agentCheckpoint.name
 
      #it's possible that a certain line card is pulled out. In that case we will
      # have checkpoint for it, but we won't have a status entity 
      if agentName not in status:
         continue
      
      # A policy may not be attached  to an interafce in all the slices. Checkpoint
      # is created for a slice only if it has corresponding policy attached
      if ( not counterInShark and policyName not in status[ agentName ].status ) or (
           counterInShark and status[ agentName ].policyKey.name != policyName
      ):
         continue
      ckptStatus = agentCheckpoint.pmapCkptStatus.newMember( policyName )
      for className in pmap.classPrio.values():
         ckptClassMapStatus = ckptStatus.cmapCkptStatus.newMember( className )
         ckptClassMapStatus.pktMatch = 0
         for aclType in AclLib.aclTypes:
            ckptClassMapStatus.newAclTypeCkptStatus( aclType )
         if counterInShark:
            pmapStatus = status[ agentName ]
            classStatus = pmapStatus.classCounters.get( className, None )
         else:
            pmapStatus = status[ agentName ].status[ policyName ]
            classStatus = pmapStatus.cmap.get( className )
         if not classStatus:
            continue

         if counterInShark:
            aclTypeCkptStatus = ckptClassMapStatus.aclTypeCkptStatus
            # create a member for each ACL
            for aclName, aclCounter in classStatus.aclCounterData.items():
               aclType = aclCounter.aclType
               ckpt = aclTypeCkptStatus[ aclType ].acl
               if aclName not in ckpt:
                  ckpt.newMember( aclName )
            # update ClassMap/ACL/ACE counter checkpoint
            updatePktHitCounterFromShark( classStatus, ckptClassMapStatus )

         else:
            # create a newMember for each aclRule in class
            aclTypeCkptStatus = ckptClassMapStatus.aclTypeCkptStatus
            for aclType in AclLib.aclTypes:
               # do we have aclStatus update in any one agent?
               option = aclTypetoOption( aclType )
               cmapMatchStatus = classStatus.match.get( option )
               if not cmapMatchStatus:
                  continue
               ckpt = aclTypeCkptStatus[ aclType ].acl
               for acl in cmapMatchStatus.acl.values():
                  if acl.name not in ckpt:
                     ckpt.newMember( acl.name )
            # update subclass status
            updatePktHitCounter( classStatus, ckptClassMapStatus )

      ckptStatus.ckptTimestamp = Tac.now()

def _getCheckpointAclStats( aclTypeCkptStatus, aclName, aclType, ruleid, version,
                            aclStatus=None,
                            counterInShark=False ):
   pktHits = 0
   byteHits = 0
   if aclTypeCkptStatus is not None:
      aclCkptStatus = aclTypeCkptStatus.acl.get( aclName )
      if aclCkptStatus is not None:
          # compare ACL's version
         if aclCkptStatus.version == version:
            if counterInShark:
               # ruleid from shark is the ACE name string
               ruleStatus = aclCkptStatus.aceStatus.get( ruleid )
               aceStatus = aclStatus.ace.get( ruleid )
               # compare ACE's version
               if ruleStatus is not None and aceStatus is not None and \
                  ruleStatus.version == aceStatus.version:
                  pktHits = ruleStatus.pkts
                  byteHits = ruleStatus.bytes
            else:
               ruleStatus = aclCkptStatus.ruleStatus.get( ruleid )
               if ruleStatus is not None:
                  pktHits = ruleStatus.pkts
                  byteHits = ruleStatus.bytes

   return pktHits, byteHits

def _getAclRuleStatsCheckpoint( checkpoint, policyName, className, aclName,
                                ruleid, aclType, aclStatus, counterInShark=False,
                                agentName=None ):
   pktHits = 0
   byteHits = 0
   ckptTime = 0

   if checkpoint is None or not checkpoint.checkpointAgentStatus:
      return ckptTime, pktHits, byteHits

   # Below check is only to get btests passing. We need to fix btests. See bug
   # 124637 for details.
   if not aclStatus.parent.parent.parent:
      return ckptTime, pktHits, byteHits

   # Only the checkpoint of the slice whose policymap/classmap/acl we are cosnidering
   # now is relevant
   # Get the agentName from function parameter when reading from Shark
   agentName = aclStatus.parent.parent.parent.name if not agentName else agentName
   agentCheckpoint = checkpoint.checkpointAgentStatus.get( agentName, None )
   if not agentCheckpoint: 
      return ckptTime, pktHits, byteHits
   
   if agentCheckpoint.pmapCkptStatus is not None: 
      pmapCkptStatus = agentCheckpoint.pmapCkptStatus.get( policyName )
      if pmapCkptStatus is not None:
         cmapCkptStatus = pmapCkptStatus.cmapCkptStatus.get( className )
         if cmapCkptStatus is not None:
            aclTypeCkptStatus = cmapCkptStatus.aclTypeCkptStatus[aclType]
            version = aclStatus.version
            pktHits, byteHits = _getCheckpointAclStats( aclTypeCkptStatus, aclName,
                                                        aclType, ruleid, version,
                                                        aclStatus, counterInShark=\
                                                        counterInShark )
            ckptTime = pmapCkptStatus.ckptTimestamp

   return ckptTime, pktHits, byteHits

def _getAclRuleStatsCkptLatest( policyName, className, aclName, ruleid, aclType,
                             checkpoints, aclStatus, counterInShark=False,
                             agentName=None ):
   checkpoint, sessionCheckpoint = checkpoints

   globalTime, globalPkts, globalBytes = _getAclRuleStatsCheckpoint(
                                             sessionCheckpoint,
                                             policyName, className, aclName,
                                             ruleid, aclType, aclStatus,
                                             counterInShark,
                                             agentName )
   sessTime, sessPkts, sessBytes = _getAclRuleStatsCheckpoint( checkpoint,
                                                               policyName, className,
                                                               aclName, ruleid,
                                                               aclType, aclStatus,
                                                               counterInShark,
                                                               agentName )
   # Use the latest checkpoint
   return ( sessPkts, sessBytes ) if sessTime >= globalTime else ( globalPkts,
                                                                   globalBytes )

def _getRuleStatsCheckpoint( checkpoint, policyName, className, status,
                             counterInShark=False,
                             agentName=None ):
   ckptTime = 0
   pktHits = 0
   bytesHit = 0
 
   if checkpoint is None or not checkpoint.checkpointAgentStatus:
      return ckptTime, pktHits, bytesHit

   # Below check is only to get btests passing. We need to fix btests. See bug
   # 124637 for details.
   if not status.parent:
      return ckptTime, pktHits, bytesHit

   # Only the checkpoint of the slice whose policymap/classmap we are cosnidering
   # now is relevant
   agentName = status.parent.name if not agentName else agentName
   agentCheckpoint = checkpoint.checkpointAgentStatus.get( agentName, None )
   if not agentCheckpoint: 
      return ckptTime, pktHits, bytesHit
 
   if agentCheckpoint.pmapCkptStatus is not None: 
      pmapCkptStatus = agentCheckpoint.pmapCkptStatus.get( policyName )
      if pmapCkptStatus is not None:
         cmapCkptStatus = pmapCkptStatus.cmapCkptStatus.get( className )
         if cmapCkptStatus is not None:
            # consider the check point pkt count only if version number match. This
            # is to ignore  stale check points that can exist after restart or
            # config replace
            if counterInShark:
               if cmapCkptStatus.version == status.counterVersion:
                  pktHits = cmapCkptStatus.pktMatch
                  bytesHit = cmapCkptStatus.byteMatch
                  ckptTime = pmapCkptStatus.ckptTimestamp
               return ckptTime, pktHits, bytesHit

            if cmapCkptStatus.version == status.id:
               pktHits = cmapCkptStatus.pktMatch
               bytesHit = cmapCkptStatus.byteMatch
               ckptTime = pmapCkptStatus.ckptTimestamp

   return ckptTime, pktHits, bytesHit

def _getRuleStatsCkptLatest( policyName, className, checkpoints, status,
                             counterInShark=False,
                             agentName=None ):
   checkpoint, sessionCheckpoint = checkpoints
   globalTime, globalPkts, globalBytes = _getRuleStatsCheckpoint( checkpoint,
                                                                  policyName,
                                                                  className,
                                                                  status,
                                                                  counterInShark,
                                                                  agentName )
   sessTime, sessPkts, sessBytes = _getRuleStatsCheckpoint( sessionCheckpoint,
                                                            policyName,
                                                            className,
                                                            status,
                                                            counterInShark,
                                                            agentName )
   # Use the latest checkpoint
   return ( sessPkts, sessBytes ) if sessTime >= globalTime else \
          ( globalPkts, globalBytes )

def _getAclRuleStatsWrapper( policyName, className, checkpoints,
                             counterInShark=False, agentName=None ):

   def _getAclRuleStatsFromShark( aclStatus, ruleConfig, ruleid, aclType,
                                  chipName=None ):
      if not aclStatus or not all( aclStatus ) \
         or ruleConfig.action == 'remark':
         return ( None, None, None, None, None, None, None )

      if not hasattr( aclStatus, '__iter__' ):
         aclStatus = [ aclStatus ]

      pkts = 0
      pbytes = 0
      ckptPkts = 0
      ckptBytes = 0
      lastChangedTime = 0.0

      # AclStatus here is one AclCounterData
      for a in aclStatus:
         # ruleid here is the ace name string
         ruleid = ruleConfig.ruleStr( False, False )
         ruleStatus = a.ace.get( ruleid )
         if ruleStatus is None:
            continue
         pkts += ruleStatus.packetCount
         pbytes += ruleStatus.byteCount
         if ruleStatus.countLastUpdateTime:
            if not lastChangedTime or \
               lastChangedTime > ruleStatus.countLastUpdateTime:
               lastChangedTime = ruleStatus.countLastUpdateTime
         newCkptPkts, newCkptBytes = _getAclRuleStatsCkptLatest( policyName,
                                                                 className,
                                                                 a.aclName, ruleid,
                                                                 aclType,
                                                                 checkpoints, a,
                                                                 counterInShark,
                                                                 agentName )
         ckptPkts += newCkptPkts
         ckptBytes += newCkptBytes
      return ( pkts, pbytes, 0, ckptPkts, ckptBytes, 0, float( lastChangedTime ) )

   def _getAclRuleStats( aclStatus, ruleConfig, ruleid, aclType, chipName=None ):
      if not aclStatus or not all( aclStatus ) \
         or ruleConfig.action == 'remark':
         return ( None, None, None, None, None, None, None )
      
      if not hasattr( aclStatus, '__iter__' ):
         aclStatus = [ aclStatus ]

      pkts = 0
      pbytes = 0
      ckptPkts = 0
      ckptBytes = 0
      lastChangedTime = 0.0
      for a in aclStatus:
         ruleStatus = a.ruleStatus.get( ruleid )
         if ruleStatus is None:
            continue
         pkts += ruleStatus.pkts
         pbytes += ruleStatus.bytes
         # lastChangedTime lets get the lowest value
         if ruleStatus.lastChangedTime:
            if not lastChangedTime or \
               lastChangedTime > ruleStatus.lastChangedTime:
               lastChangedTime = ruleStatus.lastChangedTime
         newCkptPkts, newCkptBytes = _getAclRuleStatsCkptLatest( policyName,
                                                                 className,
                                                                 a.name,
                                                                 ruleid,
                                                                 aclType,
                                                                 checkpoints,
                                                                 a )
         ckptPkts += newCkptPkts
         ckptBytes += newCkptBytes
      return ( pkts, pbytes, 0, ckptPkts, ckptBytes, 0, float( lastChangedTime ) )

   return _getAclRuleStats if not counterInShark else _getAclRuleStatsFromShark

def _getRuleStatsWrapper( policyName, className, checkpoints, counterInShark=False,
                                                              agentName=None ):

   def _getRuleStatsFromShark( status, config, ruleid, aclType, chipName=None ):
      if status is None:
         return ( None, None, None, None, None, None, None )
      if not hasattr( status, '__iter__' ):
         status = [ status ]

      pktHits = 0
      byteHits = 0
      ckptPkts = 0
      ckptBytes = 0
      lastChangedTime = 0.0
      for s in status:
         if not s:
            continue
         ruleStatus = s.countData
         pktHits += ruleStatus.pktHits
         byteHits += ruleStatus.byteHits
         if ruleStatus.lastUpdateTime:
            if not lastChangedTime or lastChangedTime > ruleStatus.lastUpdateTime:
               lastChangedTime = ruleStatus.lastUpdateTime
         newckptPkts, newckptBytes = _getRuleStatsCkptLatest( policyName, className,
                                                              checkpoints, s,
                                                              counterInShark,
                                                              agentName )
         ckptPkts += newckptPkts
         ckptBytes += newckptBytes
      return ( pktHits, byteHits, 0,
               ckptPkts, ckptBytes, 0,
               float( lastChangedTime ) )

   def _getRuleStats( status, config, ruleid, aclType, chipName=None ):
      # pylint: disable-next=use-implicit-booleaness-not-len
      if status is None or not len( status ):
         return ( None, None, None, None, None, None, None )
      if not hasattr( status, '__iter__' ):
         status = [ status ]
      
      pktHits = 0
      byteHits = 0
      ckptPkts = 0
      ckptBytes = 0
      lastChangedTime = 0.0
      for s in status:
         if not s:
            continue
         ruleStatus = s.ruleStatus
         pktHits += ruleStatus.pkts
         byteHits += ruleStatus.bytes
         if ruleStatus.lastChangedTime:
            if not lastChangedTime or \
               lastChangedTime > ruleStatus.lastChangedTime:
               lastChangedTime = ruleStatus.lastChangedTime
         newckptPkts, newckptBytes = _getRuleStatsCkptLatest( policyName, className,
                                                              checkpoints, s )
         ckptPkts += newckptPkts
         ckptBytes += newckptBytes
      return ( pktHits, byteHits, 0,
               ckptPkts, ckptBytes, 0,
               float( lastChangedTime ) )

   return _getRuleStats if not counterInShark else _getRuleStatsFromShark

def matchOptionToAclType( option ): # pylint: disable=inconsistent-return-statements
   if option == ClassMapMatchOption.matchIpAccessGroup:
      return AclCliModelImpl.AclType.ip
   elif option == ClassMapMatchOption.matchIpv6AccessGroup:
      return AclCliModelImpl.AclType.ipv6
   elif option == ClassMapMatchOption.matchMacAccessGroup:
      return AclCliModelImpl.AclType.mac
   else:
      assert True

# pylint: disable-next=inconsistent-return-statements
def populateAclListModel( policyName, className, aclConfig, prio, status, option,
                          aclRule, aclIndex, checkpoints, counterInShark=False,
                          agentName=None ):
   # for Sysdb, status is ClassMapStatus
   # for Shark, status  is CounterData
   aclType = matchOptionToAclType( option )
   aclTypeConfig = aclConfig.config[ aclType ]
   aclListConfig = aclTypeConfig.acl.get( aclRule )
   aclListStatus = []
   if not hasattr( status, '__iter__' ):
      # change status from shark to a list
      status = [ status ]
   if aclListConfig is None:
      return
   if status is not None:
      for s in status:
         if counterInShark:
            aclTypeStatus = s.aclCounterData
            if aclTypeStatus is not None:
               aclListStatus.append( aclTypeStatus[ aclRule ] )
               continue
         aclTypeStatus = s.match.get( option )
         if aclTypeStatus is not None:
            aclListStatus.append( aclTypeStatus.acl.get( prio ) )

      getAclRuleStatsHandler = _getAclRuleStatsWrapper( policyName, className,
                                                        checkpoints,
                                                        counterInShark,
                                                        agentName )
      return AclCliModelImpl.getAclListModel( aclListConfig, aclListStatus,
                                              aclType, aclRule,
                                              getAclRuleStatsHandler,
                                              policyMap=True )

def populateClassMapModel( cmap, policyName=None, status=None, aclConfig=None,
                           checkpoints=None, classMapModelType=ClassMapModel,
                           counterInShark=False, agentName=None ):
   className = cmap.className
   cmapModel = classMapModelType()
   cmapModel.matchCondition = cmap.matchCondition
   cmapModel.name = cmap.className
   cmapCounter = ClassMapCounter()
   getRuleStatsHandler = _getRuleStatsWrapper( policyName, className,
                                               checkpoints, counterInShark,
                                               agentName )
   ( cmapCounter.packetCount, cmapCounter.byteCount, _,
     cmapCounter.checkpointPacketCount, cmapCounter.checkpointByteCount, _,
     cmapCounter.lastChangedTime ) = getRuleStatsHandler( status, None,
                                                          None, None )
   cmapModel.counterData = cmapCounter
   
   for option, clMatch in cmap.match.items():
      cmatchModel = ClassMatchModel()
      cmatchModel.option = option

      if option == 'matchMplsAccessGroup':
         mplsRule = MplsRule()
         mplsRule.counterData = cmapModel.counterData
         cmatchModel.mplsRule = mplsRule

      for aclIndex, item in enumerate( clMatch.acl.items() ):
         prio, aclRule = item
         prio = int( prio )
         cMatchAclModel = ClassMatchAclModel()
         cMatchAclModel.name = aclRule
         cmatchModel.acl[ prio ] = cMatchAclModel
         if aclConfig is not None:
            aclListModel = populateAclListModel( policyName,
                                                 cmap.className,
                                                 aclConfig, prio,
                                                 status, option,
                                                 aclRule, aclIndex,
                                                 checkpoints,
                                                 counterInShark,
                                                 agentName )
            if option == 'matchIpAccessGroup':
               cMatchAclModel.ipAclList = aclListModel
            elif option == 'matchIpv6AccessGroup':
               cMatchAclModel.ipv6AclList = aclListModel
            elif option == 'matchMacAccessGroup':
               cMatchAclModel.macAclList = aclListModel
      for prio, ipRuleCfg in clMatch.ipRule.items():
         prio = int( prio )
         getRuleStatsHandler = _getRuleStatsWrapper( policyName,
                                                     cmap.className,
                                                     checkpoints,
                                                     counterInShark,
                                                     agentName )
         rule = AclCliModelImpl.getIp4Rule( prio, ipRuleCfg,
                                            status, prio, False,
                                            'ip', getRuleStatsHandler )
         cmatchModel.ipRule[ prio ] = rule

      for prio, ip6RuleCfg in clMatch.ip6Rule.items():
         prio = int( prio )
         getRuleStatsHandler = _getRuleStatsWrapper( policyName,
                                                     cmap.className,
                                                     checkpoints,
                                                     counterInShark,
                                                     agentName )
         rule = AclCliModelImpl.getIp6Rule( prio, ip6RuleCfg,
                                            status, prio, False,
                                            'ipv6', getRuleStatsHandler )
         cmatchModel.ip6Rule[ prio ] = rule

      for prio, macRuleCfg in clMatch.macRule.items():
         prio = int( prio )
         getRuleStatsHandler = _getRuleStatsWrapper( policyName,
                                                     cmap.className,
                                                     checkpoints,
                                                     counterInShark,
                                                     agentName )
         rule = AclCliModelImpl.getMacRule( prio, macRuleCfg,
                                            status, prio, False,
                                            'mac', getRuleStatsHandler )
         cmatchModel.macRule[ prio ] = rule
      cmapModel.match[ option ] = cmatchModel

   return cmapModel

def kickPmapCounterPoll( policyName, pmapType ):
   pmapType.counterUpdateRequestTime[ policyName ] = Tac.now()

def waitForPmapCounters( policyName, pmapType, pmapStatuses, sharkEnabled=False,
                         kickPoll=True ):
   if not pmapStatuses:
      return True
   if kickPoll:
      kickPmapCounterPoll( policyName, pmapType )
   try:
      statusAttr = 'countLastUpdateTime' if sharkEnabled else 'counterUpdateTime'
      for pmapStatus in pmapStatuses:
         TacSigint.check()
         Tac.waitFor(
            lambda: pmapType.counterUpdateRequestTime.get( policyName, 0 ) <=
            getattr( pmapStatus, statusAttr ), # pylint: disable=cell-var-from-loop
            description='counters to be available',
            warnAfter=None,
            sleep='POLICY_MAP_COHAB_TEST' not in os.environ,
            maxDelay=0.1,
            timeout=counterUpdateTimeout )
   except Tac.Timeout:
      return False
   finally:
      del pmapType.counterUpdateRequestTime[ policyName ]
   return True

def waitForAllPmapCounters( pmapNameAndCounterStatuses, pmapType,
                            sharkEnabled=False ):
   if not pmapNameAndCounterStatuses:
      return True
   def allStatusCompleted():
      statusAttr = 'countLastUpdateTime' if sharkEnabled else 'counterUpdateTime'
      for pmapName, pmapCounterStatuses in pmapNameAndCounterStatuses:
         for counterStatus in pmapCounterStatuses:
            if ( pmapType.counterUpdateRequestTime.get( pmapName, 0 ) >
                 getattr( counterStatus, statusAttr ) ):
               return False
      return True
   for pmapName, _ in pmapNameAndCounterStatuses:
      kickPmapCounterPoll( pmapName, pmapType )
   try:
      TacSigint.check()
      Tac.waitFor( allStatusCompleted,
         description='counters to be available',
         warnAfter=None,
         sleep='POLICY_MAP_COHAB_TEST' not in os.environ,
         maxDelay=0.1,
         timeout=counterUpdateTimeout )
   except Tac.Timeout:
      return False
   finally:
      for pmapName, _ in pmapNameAndCounterStatuses:
         del pmapType.counterUpdateRequestTime[ pmapName ]
   return True

class PolicyMapModelContainer:
   def __init__( self, cliConfig, cliIntfConfig, intfConfig, config, status,
                 aclConfig, hwStatus, sliceHwStatus, vtiStatusDir, dynVlanIntfDir,
                 pmapType, direction, policyMapAllModel=None,
                 counterStatusDir=None ):
      self.cliIntfConfig = cliIntfConfig
      self.cliConfig = cliConfig
      self.intfConfig = intfConfig
      self.config = config
      self.status = status
      self.aclConfig = aclConfig
      self.hwStatus = status
      self.sliceHwStatus = sliceHwStatus
      self.vtiStatusDir = vtiStatusDir
      self.dynVlanIntfDir = dynVlanIntfDir
      self.mapType = pmapType
      self.direction = direction
      self.policyMapAllModel = policyMapAllModel
      self.configuredPolicyIntf = None
      self.configuredFallbackPolicyIntf = None
      self.appliedPolicyIntf = None
      self.counterStatusDir = counterStatusDir

   def isDynamicSviIntf( self, intf ):
      if not self.dynVlanIntfDir:
         return False

      if ArnetVlanIntfId.isVlanIntfId( intf ):
         for dynVlanIntfSet in self.dynVlanIntfDir.values():
            if intf in dynVlanIntfSet.intfConfig:
               return True

      return False

   def getVniFromVlan( self, vlan ):
      for vti in self.vtiStatusDir.vtiStatus:
         extVlanToVniMap = self.vtiStatusDir.vtiStatus[ vti ].extendedVlanToVniMap
         if vlan in extVlanToVniMap:
            return extVlanToVniMap[ vlan ].vni

      return None

   def getInstalledIntfsStr( self, intf ):
      if not self.isDynamicSviIntf( intf ):
         return intf

      intfStr = intf
      vlanId = ArnetVlanIntfId.vlanId( intf )
      vni = self.getVniFromVlan( vlanId )
      if vni:
         intfStr = "VNI-" + str( vni ) + "(" + intf + ")"

      return intfStr

   def readPolicyIntf( self ):
      if self.configuredPolicyIntf is not None:
         return
      if self.configuredFallbackPolicyIntf is not None:
         return
      self.configuredPolicyIntf = collections.defaultdict( set )
      self.configuredFallbackPolicyIntf = collections.defaultdict( set )
      self.appliedPolicyIntf = collections.defaultdict( set )

      for intf, policy in self.intfConfig.intf.items():
         if not self.isDynamicSviIntf( intf ):
            self.configuredPolicyIntf[ policy ].add( intf )

      for vni, policy in self.intfConfig.vni.items():
         self.configuredPolicyIntf[ policy ].add( "VNI-" + str( vni ) )

      for intf, policy in self.intfConfig.intfFallback.items():
         self.configuredFallbackPolicyIntf[ policy ].add( intf )

      failedIntfs = collections.defaultdict( set )
      for sliceStatus in self.status.values():
         for status in sliceStatus.status.values():
            installedIntfs = []
            for intf, intfStatus in status.intfStatus.items():
               installed = all( x.installed == 'success'
                                for x in intfStatus.installed.values() )
               intfStr = self.getInstalledIntfsStr( intf )
               if installed:
                  if intf in failedIntfs[ status.name ]:
                     continue
                  installedIntfs.append( intfStr )
               else:
                  failedIntfs[ status.name ].add( intf )
                  self.appliedPolicyIntf[ status.name ].discard( intfStr )

            self.appliedPolicyIntf[ status.name ].update( installedIntfs )

   def populateAll( self, mode, summary, checkpoints ):
      for name in sorted( self.config.pmapType.pmap ):
         self.populatePolicyMap( mode, name, summary, checkpoints )
         TacSigint.check()

   def populatePolicyMap( self, mode, name, summary, checkpoints ):
      self.readPolicyIntf()
      pmapCfg = self.config.pmapType.pmap.get( name )
      if not pmapCfg:
         return False
      pmap = pmapCfg.currCfg
      if not pmap:
         return False

      pmapModel = PolicyMapModel()
      pmapModel.name = pmap.policyName
      pmapModel.mapType = pmapCfg.type
      pmapModel.configuredIngressIntfs = list( self.configuredPolicyIntf[ name ] )
      pmapModel.configuredIngressIntfsAsFallback = list( 
                   self.configuredFallbackPolicyIntf[ name ] )
      pmapModel.appliedIngressIntfs = list( self.appliedPolicyIntf[ name ] )
      agentName = ""
      if summary:
         # For a summary model, this is all we return.
         self.policyMapAllModel.append( name, pmapModel )
         return True

      # Get all the status objects from SysDb.
      pmapStatuses = [ pmapTypeStatus.status[ name ] for pmapTypeStatus in
                          self.status.values() if name in pmapTypeStatus.status ]

      counterInShark = self.counterStatusDir is not None
      if counterInShark:
         # Get all the status objects from Shark.
         for pmapTypeStatus in self.status.values():
            if name in pmapTypeStatus.status:
               agentName = pmapTypeStatus.name
               break
               # if policy name is not found or status is None, continue
         pmapKey = PolicyMapCountersKey( name, "" )
         pmapCounters = self.counterStatusDir.policyMapCounters.get( pmapKey )
         pmapCounterStatuses = []
         if pmapCounters is not None:
            pmapCounterStatuses.append( pmapCounters )
         countersAreStale = not waitForPmapCounters( name,
                                                     self.cliConfig.pmapType,
                                                     pmapCounterStatuses,
                                                     sharkEnabled=True )
      else:
         countersAreStale = not waitForPmapCounters( name,
                                                     self.cliConfig.pmapType,
                                                     pmapStatuses )
      if countersAreStale:
         # somehow we timed out, but we'll show whatever we have
         mode.addWarning( 'Displaying stale counters' )

      # XXX This code does not currently look at the AclStatus objects.
      # Once it does, it will need to account for the keys in the collection
      # of AclStatus objects perhaps not matching up with the configured sequence
      # numbers, if the user has run a resequence operation. Since the values should
      # still be in the same order though, we can use the keys from the configuration
      # for entering into the model, and just pair up those values with the list of
      # AclStatus objects.
      for prio, className in pmap.classPrio.items():
         if className in pmap.rawClassMap:
            cmapModel = ClassMapModel()
            cmap = pmap.rawClassMap[ className ]
            cmapModel.name = className
            cmapModel.matchCondition = cmap.matchCondition
            pmapModel.rawClassMap[ className ] = cmapModel
         else:
            classConfig = self.config.cmapType.cmap.get( className )
            if not classConfig:
               continue
            cmap = classConfig.currCfg
            if not cmap:
               continue
         # still need vrf from classStatus
         classStatuses = [ pmapStatus.cmap[ className ] for pmapStatus in
                              pmapStatuses if className in pmapStatus.cmap ]
         # Find the status objects that are a subclass of the base status, if any.
         subclassStatusList = list( filter( getCmapStatusFilter( self.mapType ),
                                       classStatuses ) )
         classMapModel = PolicyClassMapBase.createModel(
                                                className, subclassStatusList )
         if counterInShark:
            # pmapCounter could be none if the status is not published
            # pass an empty list for classmap model to
            # render the class-map's matching criteria
            classCounter = pmapCounters.classCounters.get( className ) if \
               pmapCounters else None
            classCounterList = [ classCounter ] if classCounter else []
            classMapModel.classMap = populateClassMapModel( cmap, name,
                              classCounterList, self.aclConfig, checkpoints,
                              counterInShark=counterInShark, agentName=agentName )
         else:
            classMapModel.classMap = populateClassMapModel( cmap, name,
                              subclassStatusList, self.aclConfig, checkpoints )
         classAction = pmap.classAction[ className ]
         for actType, action in classAction.policyAction.items():
            actionModel = ActionBase.createModel( action )
            classMapModel.configuredAction[ actType ] = actionModel
            if actType == tacActionType.actionSet:
               classMapModel.actionSetConfig = ActionSetModel.createModel( action )
         pmapModel.classMap[ int( prio ) ] = classMapModel
      self.policyMapAllModel.append( name, pmapModel )
      return True

class ClassMapModelContainer:
   def __init__( self, config, hwStatus, cmapType, classMapAllModel ):
      self.config = config
      self.hwStatus = hwStatus
      self.mapType = cmapType
      self.classMapAllModel = classMapAllModel
      self.classMapAllModel.mapType = cmapType

   def populateAll( self ):
      for name in sorted( self.config.cmapType.cmap ):
         self.populateClassMap( name )
         TacSigint.check()

   def populateClassMap( self, name ):
      cmapCfg = self.config.cmapType.cmap.get( name, None )
      if not cmapCfg:
         return False
      cmap = cmapCfg.currCfg
      if not cmap:
         return False

      cmapModel = populateClassMapModel( cmap )
      self.classMapAllModel.append( name, cmapModel )
      return True
